#include "token.h"
#include "token_type.h"

#include <library/cpp/telfinder/phone.h>

#include <util/charset/wide.h>
#include <util/digest/murmur.h>
#include <util/generic/string.h>
#include <util/string/cast.h>
#include <util/system/compiler.h>
#include <util/system/types.h>

NUnperson::TToken::~TToken() = default;

static void AppendPlaceholderPrefixTo(
    TUtf16String& out,
    NUnperson::ETokenType type)
{
    out += u'%';
    out.AppendAscii(ToString(type));
    out += u'_';
}

static void AppendPlaceholderSuffixTo(TUtf16String& out) {
    out += u'%';
}

static void AppendPlaceholderMurmurTo(
    TUtf16String& out,
    TWtringBuf value,
    bool erasePersonalInfo)
{
    if (erasePersonalInfo) {
        auto hash = MurmurHash<ui32>(value.data(), value.size() * sizeof(*value.data()));
        out.AppendAscii(ToString(hash));
    } else {
        out += value;
    }
}

static void AppendMurmurPlaceholderTo(
    TUtf16String& out,
    NUnperson::ETokenType type,
    TWtringBuf value,
    bool erasePersonalInfo)
{
    AppendPlaceholderPrefixTo(out, type);
    AppendPlaceholderMurmurTo(out, value, erasePersonalInfo);
    AppendPlaceholderSuffixTo(out);
}

static void AppendStringCharsSumPlaceholderTo(
    TUtf16String& out,
    TWtringBuf value,
    bool erasePersonalInfo)
{
    if (erasePersonalInfo) {
        ui64 sum = 0;
        for (auto c: value) {
            sum += c;
        }
        out.AppendAscii(ToString(sum));
    } else {
        out += value;
    }
}

void NUnperson::TNameToken::AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const {
    AppendMurmurPlaceholderTo(out, Type, Str, erasePersonalInfo);
}

void NUnperson::TPhoneToken::AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const {
    AppendMurmurPlaceholderTo(out, Type, Str, erasePersonalInfo);
}

void NUnperson::TPasswordToken::AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const {
    AppendMurmurPlaceholderTo(out, Type, Str, erasePersonalInfo);
}

void NUnperson::TShortPasswordToken::AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const {
    AppendPlaceholderPrefixTo(out, Type);
    AppendStringCharsSumPlaceholderTo(out, Str, erasePersonalInfo);
    AppendPlaceholderSuffixTo(out);
}

void NUnperson::TNumberToken::AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const {
    AppendMurmurPlaceholderTo(out, Type, Str, erasePersonalInfo);
}

void NUnperson::TShortNumberToken::AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const {
    AppendPlaceholderPrefixTo(out, Type);
    AppendStringCharsSumPlaceholderTo(out, Str, erasePersonalInfo);
    AppendPlaceholderSuffixTo(out);
}

void NUnperson::TUriToken::AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const {
    AppendPlaceholderPrefixTo(out, Type);
    out += Scheme;
    out += u'_';
    out += Domain;
    out += u'_';
    AppendPlaceholderMurmurTo(out, Str, erasePersonalInfo);
    AppendPlaceholderSuffixTo(out);
}

void NUnperson::TTimestampToken::AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const {
    AppendMurmurPlaceholderTo(out, Type, Str, erasePersonalInfo);
}

