#pragma once

#include "token_type.h"

#include <util/generic/noncopyable.h>
#include <util/generic/strbuf.h>
#include <util/generic/string.h>

class TPhone;

namespace NUnperson {
    static const size_t MIN_PASSWORD_LEN = 3;

    class TToken: TNonCopyable {
    public:
        const TWtringBuf Str;
        const ETokenType Type;

        TToken(TWtringBuf str, ETokenType type)
            : Str(std::move(str))
            , Type(type)
        {
        }

        virtual ~TToken();

        virtual void AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const = 0;
    };

    class TNameToken: public TToken {
    public:
        using TToken::TToken;

        void AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const override;
    };

    class TPhoneToken: public TToken {
    public:
        TPhoneToken(const TUtf16String& str)
            : TToken(TWtringBuf{str.Data(), str.Size()}, ETokenType::PhoneNumber)
        {
        }

        void AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const override;
    };

    class TPasswordToken: public TToken {
    public:
        TPasswordToken(TWtringBuf str)
            : TToken(str, ETokenType::Password)
        {
        }

        void AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const override;
    };

    class TShortPasswordToken: public TToken {
    public:
        TShortPasswordToken(TWtringBuf str)
            : TToken(str, ETokenType::ShortPassword)
        {
        }

        void AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const override;
    };

    class TNumberToken: public TToken {
    public:
        TNumberToken(TWtringBuf str)
            : TToken(str, ETokenType::Number)
        {
        }

        void AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const override;
    };

    class TShortNumberToken: public TToken {
    public:
        TShortNumberToken(TWtringBuf str)
            : TToken(str, ETokenType::ShortNumber)
        {
        }

        void AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const override;
    };

    class TUriToken: public TToken {
    private:
        const TWtringBuf Scheme;
        const TWtringBuf Domain;

    public:
        TUriToken(TWtringBuf str, TWtringBuf scheme, TWtringBuf domain)
            : TToken(str, ETokenType::Uri)
            , Scheme(scheme)
            , Domain(domain)
        {
        }

        void AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const override;
    };

    class TTimestampToken: public TToken {
    public:
        TTimestampToken(TWtringBuf str, ETokenType type)
            : TToken(str, type)
        {
        }

        void AppendPlaceholderTo(TUtf16String& out, bool erasePersonalInfo) const override;
    };
}

