#include <mail/so/libs/unperson/erasure_type.h>
#include <mail/so/libs/unperson/lemmercache.h>
#include <mail/so/libs/unperson/unperson.h>

#include <mail/so/libs/jniwrapper_base/jniwrapper_base.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/charset/wide.h>

static const char* DeobfuscateConfig =
    "<Unperson>\n"
        "ErasureType: TokenTypeOnly\n"
        "FusedTokensSeparator: _\n"
        "IgnoreListPath: ignore-list.txt\n"
    "</Unperson>\n"
    "<Deobfuscator>\n"
        "RemapPath: rus.dict/deobfuscator/remap.json\n"
        "TriePath: rus.dict/deobfuscator/dict.trie\n"
    "</Deobfuscator>\n"
    "<Deobfuscator>\n"
        "RemapPath: en.dict/deobfuscator/remap.json\n"
        "TriePath: en.dict/deobfuscator/dict.trie\n"
    "</Deobfuscator>\n";

static const char* DeobfuscateConfigNoUnpersonUri =
    "<Unperson>\n"
        "ErasureType: KeepAll\n"
        "FusedTokensSeparator: <space>\n"
        "MaxTokens: 14\n"
        "IgnoreListPath: ignore-list.txt\n"
        "UnpersonUri: false\n"
        "EraseSingleLetterWords: true\n"
        "EraseStopWords: true\n"
        "FixShortI: true\n"
        "OneSentencePerLine: true\n"
        "BestEffortSentenceTokenizer: true\n"
    "</Unperson>\n"
    "<Deobfuscator>\n"
        "RemapPath: hard.rus.dict/deobfuscator/remap.json\n"
        "TriePath: hard.rus.dict/deobfuscator/dict.trie\n"
    "</Deobfuscator>\n"
    "<Deobfuscator>\n"
        "RemapPath: en.dict/deobfuscator/remap.json\n"
        "TriePath: en.dict/deobfuscator/dict.trie\n"
    "</Deobfuscator>\n";

static TString Unperson(const TString& str, NUnperson::EErasureType erasureType = NUnperson::EErasureType::KeepAll) {
    NUnperson::TLemmerCache lemmerCache(2);
    NUnperson::TUnperson unperson(erasureType);
    return WideToUTF8(unperson.UnpersonText(TUtf16String::FromUtf8(str), lemmerCache));
}

static TString JniWrapperUnperson(const TString& str, const char* config = nullptr) {
    void* instance;
    UNIT_ASSERT_C(JniWrapperCreateUnperson(config, &instance) == 0, (const char*) instance);
    wchar16* text;
    size_t textLen;
    TUtf16String input{TUtf16String::FromUtf8(str)};
    UNIT_ASSERT_VALUES_EQUAL(
        JniWrapperUnpersonText(
            instance,
            input.data(),
            input.length(),
            nullptr,
            0,
            nullptr,
            0,
            &text,
            &textLen),
        0);
    TUtf16String result{static_cast<const wchar16*>(text), textLen};
    JniWrapperFree(text);
    JniWrapperDestroyUnperson(instance);
    return WideToUTF8(result);
}

Y_UNIT_TEST_SUITE(TestUnperson) {
    Y_UNIT_TEST(NonPersonalTest) {
        UNIT_ASSERT_STRINGS_EQUAL(Unperson("ничего"), "ничего");
    }

    Y_UNIT_TEST(AdminTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Admin admin Цена цена Ди\u200Bма дима"),
            "%FirstName_Admin% admin %FirstName_Цена% цена "
            "%FirstName_Дима% %FirstName_дима%");
    }

    Y_UNIT_TEST(PhoneTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("47-77-08 — это не мой телефон, в отличие от +7(903)059-23-38"),
            "%PhoneNumber_477708% — это не мой телефон, в отличие от %PhoneNumber_+7(903)0592338%");
    }

    Y_UNIT_TEST(PhoneTest2) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson(
                "+7 (495) 230 10 30\n"
                "+7 495 221 08 40\n"
                "+7 495 777 777 5\n"
                "+7 495 777-777-5\n"
                "+7(495)777-777-5\n"
                "+7(495)777-7775\n"
                "+7(32)333 222\n"
                "8 495 510-18-31\n"
                "8 495 777-777-5\n"
                "8 800 700 00 88\n"),
            "%PhoneNumber_+7(495)2301030%\n"
            "%PhoneNumber_+7(495)2210840%\n"
            "%PhoneNumber_+7(495)7777775%\n"
            "%PhoneNumber_+7(495)7777775%\n"
            "%PhoneNumber_+7(495)7777775%\n"
            "%PhoneNumber_+7(495)7777775%\n"
            "%PhoneNumber_+7(32)333222%\n"
            "%PhoneNumber_8(495)5101831%\n"
            "%PhoneNumber_8(495)7777775%\n"
            "%PhoneNumber_8(800)7000088%\n");
    }

    Y_UNIT_TEST(PhonePunctuationTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Мой телефон: +7(903)059 2338, а вот этот не мой: 47-77-08."),
            "Мой телефон: %PhoneNumber_+7(903)0592338%, а вот этот не мой: %PhoneNumber_477708%.");
    }

    Y_UNIT_TEST(UriTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Для регистрации пройдите по ссылке: HTTPs://Yandex.ru"),
            "Для регистрации пройдите по ссылке: %Uri_https_yandex.ru_HTTPs://Yandex.ru%");
    }

    Y_UNIT_TEST(UriParenthesisTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Disambiguation: https://ru.wikipedia.org/wiki/Kiss_(значения)"),
            "Disambiguation: %Uri_https_ru.wikipedia.org_https://ru.wikipedia.org/wiki/Kiss_%(значения)");
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "Disambiguation: https://ru.wikipedia.org/wiki/Kiss_(значения)",
                DeobfuscateConfig),
            "disambiguation %Uri% значения");
    }

    Y_UNIT_TEST(SchemelessUriTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Ссылка в скобках (www.yandex.ru/mail)"),
            "Ссылка в скобках (%Uri_http_www.yandex.ru_www.yandex.ru/mail%)");
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "Ссылка в скобках (www.yandex.ru/mail)",
                DeobfuscateConfig),
            "ссылка в скобках %Uri%");
    }

    Y_UNIT_TEST(DiacriticUriTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("з&#225;пи&#269;и cool.nastq@yandex.ua.К&#243;н&#283;чн&#243; вы м&#243;ж&#283;т&#283;"),
            "%Password_з&#225;пи&#269;и% %Uri_mailto_yandex.ua_cool.nastq@yandex.ua%.К%Password_&#243;н&#283;чн&#243%; вы %Password_м&#243;ж&#283;т&#283%;");
    }

    Y_UNIT_TEST(WwwlessUriTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Ссылка podari-zhizn.ru t.co yandex.com.tr/map t.b.com"),
            "Ссылка %Uri_http_podari-zhizn.ru_podari-zhizn.ru% "
            "%Uri_http_t.co_t.co% %Uri_http_yandex.com.tr_yandex.com.tr/map% "
            "%Uri_http_t.b.com_t.b.com%");
    }

    Y_UNIT_TEST(SchemelessRussianUriTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Ссылка в скобках (www.почта.рф/mail)"),
            "Ссылка в скобках (%Uri_http_www.почта.рф_www.почта.рф/mail%)");
    }

    Y_UNIT_TEST(SchemelessNotMatchedUriTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("www.-почта.рф/mail"),
            "%Password_www.-почта.рф/mail%");
    }

    Y_UNIT_TEST(MailtoTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Напиши мне на mailto:analizer@yandex.ru"),
            "Напиши мне на %Uri_mailto_yandex.ru_mailto:analizer@yandex.ru%");
    }

    Y_UNIT_TEST(SchemelessMailtoTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Напиши мне на analizer@yandex.ru"),
            "Напиши мне на %Uri_mailto_yandex.ru_analizer@yandex.ru%");
    }

    Y_UNIT_TEST(MalformedFtpUriTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Битая ссылка ftp:/file.storage.sh"),
            "Битая ссылка %Uri_ftp_file.storage.sh_ftp:/file.storage.sh%");
    }

    Y_UNIT_TEST(MalformedWrappedFtpUriTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("В скобках (irc:/file.storage.sh)"),
            "В скобках (%Uri_irc_file.storage.sh_irc:/file.storage.sh%)");
    }

    Y_UNIT_TEST(WrappedUriTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson(
                "Далее идёт ссылка в скобках: (www.yandex.ru), и текст.\n"
                "Иван Петров [[mailto:ivan.petrov@gmail.com]]\nSent:\n"
                "Иван Петров ((mailto:ivan.petrov@gmail.com))\nSent:\n"
                "Иван Петров mailto:ivan.petrov@gmail.com,\nSent:\n"
                "Wednesday, January 2, 2019 3:23 PM "
                "02/мая/2019:14:16:05.123456Z"),
            "Далее идет ссылка в скобках: "
            "(%Uri_http_www.yandex.ru_www.yandex.ru%), и текст.\n"
            "%FirstName_Иван% %LastName_Петров% "
            "[[%Uri_mailto_gmail.com_mailto:ivan.petrov@gmail.com%]]\nSent:\n"
            "%FirstName_Иван% %LastName_Петров% "
            "((%Uri_mailto_gmail.com_mailto:ivan.petrov@gmail.com%))\nSent:\n"
            "%FirstName_Иван% %LastName_Петров% "
            "%Uri_mailto_gmail.com_mailto:ivan.petrov@gmail.com%,\nSent:\n"
            "Wednesday, January 2, %ShortNumber_2019% %Time_3:23% PM "
            "%Timestamp_02/мая/2019:14:16:05.123456Z%");
    }

    Y_UNIT_TEST(UriIDNTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson(
                "https://ru.wikipedia.org/wiki/%D0%A2%D0%B5%D1%81%D1%82\n"
                "https://ru.wikipedia.org/wiki/Google\n"
                "https://яндекс.рф"),
            "%Uri_https_ru.wikipedia.org_https://ru.wikipedia.org/wiki/%D0%A2%D0%B5%D1%81%D1%82%\n"
            "%Uri_https_ru.wikipedia.org_https://ru.wikipedia.org/wiki/Google%\n"
            "%Uri_https_яндекс.рф_https://яндекс.рф%");
    }

    Y_UNIT_TEST(ConcatenatedLinksTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Напиши мне на (http://ya.ru/uri)<http://ya.ru/uri2>"),
            "Напиши мне на (%Uri_http_ya.ru_http://ya.ru/uri%)<%Uri_http_ya.ru_http://ya.ru/uri2%>");
    }

    Y_UNIT_TEST(ConcatenatedAuthLinksTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Напиши мне на (http://ya.ru/uri)<http://login@ya.ru/uri2>"),
            "Напиши мне на (%Uri_http_ya.ru_http://ya.ru/uri%)<%Uri_http_ya.ru_http://login@ya.ru/uri2%>");
    }

    Y_UNIT_TEST(ConcatenatedAuthLinks2Test) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Напиши мне на (http://login@ya.ru/uri)<http://ya.ru/uri2>"),
            "Напиши мне на (%Uri_http_ya.ru_http://login@ya.ru/uri%)<%Uri_http_ya.ru_http://ya.ru/uri2%>");
    }

    Y_UNIT_TEST(ConcatenatedSkypeLinksTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Позвони мне на (skype:dpotapov2)<skype:dpotapov2>"),
            "Позвони мне на (%Uri_skype_skype_skype:dpotapov2%)<%Uri_skype_skype_skype:dpotapov2%>");
    }

    Y_UNIT_TEST(ConcatenatedSlashlessLinksTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Напиши мне на (an2lizer@yandex2.ru)<an2lizer@ya2.ru>"),
            "Напиши мне на (%Uri_mailto_yandex2.ru_an2lizer@yandex2.ru%)<%Uri_mailto_ya2.ru_an2lizer@ya2.ru%>");
    }

    Y_UNIT_TEST(PortUriTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Ссылка: http://ya.ru:80 https://ya.ru:443/uri (http://ya.ru:80) [https://ya.ru:443/] <mailto:potapov.d@gmail.com:543>"),
            "Ссылка: %Uri_http_ya.ru_http://ya.ru:80% %Uri_https_ya.ru_https://ya.ru:443/uri% (%Uri_http_ya.ru_http://ya.ru:80%) [%Uri_https_ya.ru_https://ya.ru:443/%] <%Uri_mailto_gmail.com_mailto:potapov.d@gmail.com:543%>");
    }

    Y_UNIT_TEST(FioTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Здравствуйте, ПОТАПОВ Дмитрий Александрович, вам письмо от ЦАМУТАЛИ Александра Сергеевича"),
            "Здравствуйте, %LastName_ПОТАПОВ% %FirstName_Дмитрий% %SecondName_Александрович%, вам письмо от %MayBeName_ЦАМУТАЛИ% %FirstName_Александра% %SecondName_Сергеевича%");
    }

    Y_UNIT_TEST(EnFioTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Hello Samuel where is Kate Anderson?"),
            "Hello %FirstName_Samuel% where is %FirstName_Kate% %FirstName_Anderson%?");
    }

    Y_UNIT_TEST(EnArticleTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("The next generation Services and Protocols. Please refer to the Protocol standard"),
            "The next generation Services and Protocols. Please refer to the %FirstName_Protocol% standard");
    }

    Y_UNIT_TEST(MayBeFioTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Мы вчера два часа цамутали алкомотолодку, да так и не выцумуталили с потаповым (провал)"),
            "Мы вчера два часа цамутали алкомотолодку, да так и не выцумуталили с %LastName_потаповым% (провал)");
    }

    Y_UNIT_TEST(NumberTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Совершён перевод на сумму 100500 рублей 1595 копеек"),
            "Совершен перевод на сумму %Number_100500% рублей %ShortNumber_1595% копеек");
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Совершен перевод на сумму 100500 рублей 1595 копеек", NUnperson::EErasureType::ErasePersonalInfo),
            "Совершен перевод на сумму %Number_475298793% рублей %ShortNumber_212% копеек");
    }

    Y_UNIT_TEST(PasswordTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson(
                "Вы зарегистрировались под именем l33t. "
                "Ваш пароль: ..,HackmeIfY0uCan. "
                "Токен сессии: J48K-DS8R."),
            "Вы зарегистрировались под именем %ShortPassword_l33t%. "
            "Ваш пароль: ..,%Password_HackmeIfY0uCan%. "
            "Токен сессии: %ShortPassword_J48K%-%ShortPassword_DS8R%.");
    }

    Y_UNIT_TEST(PunctPasswordTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Используйте пароль (df0d2&d0) или ..,,.,,."),
            "Используйте пароль (%Password_df0d2%&d0) или %Password_..,,.,,%.");
    }

    Y_UNIT_TEST(DashSeparatedNameTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Композитор Римский-Корсаков более-менее хорош туда--сюда"),
            "Композитор %LastName_Римский%-%LastName_Корсаков% более-менее хорош туда--сюда");
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "Композитор Римский-Корсаков более-менее хорош туда--сюда",
                DeobfuscateConfig),
            "композитор %LastName% %LastName% более менее хорош туда сюда");
    }

    Y_UNIT_TEST(Apollo13Test) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson(
                "Запускаем Apollo-13, авось найдёт Apollo-12 или ещё что, "
                "нужно успеть до 2019-Apr-24 или 45-апреля-19 или 24-june-99. "
                "2232-4432-2314-"),
            "Запускаем Apollo-%ShortNumber_13%, авось найдет "
            "Apollo-%ShortNumber_12% или еще что, нужно "
            "успеть до %Date_2019-Apr-24% или %Date_45-апреля-19% или "
            "%Date_24-june-99%. "
            "%ShortNumber_2232%-%ShortNumber_4432%-%ShortNumber_2314%-");
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "Запускаем Apollo-13, авось найдёт Apollo-12 или ещё что, "
                "нужно успеть до 2019-Apr-24 или 45-апреля-19 или 24-june-99. "
                "2232-4432-2314-",
                DeobfuscateConfig),
            "запускаем apollo %ShortNumber% авось найдет apollo %ShortNumber% "
            "или еще что нужно успеть до %Date% или %Date% или %Date% "
            "%ShortNumber% %ShortNumber% %ShortNumber%");
    }

    Y_UNIT_TEST(BracketsTest) {
        TString text =
            "(тест) [тест] <тест> {тест} \"тест\" "
            "«тест» /тест/ 'тест' “тест”";
        UNIT_ASSERT_STRINGS_EQUAL(Unperson(text), text);
    }

    Y_UNIT_TEST(BracketsMultiWordTest) {
        TString text{"«тест тест__тест»"};
        UNIT_ASSERT_STRINGS_EQUAL(Unperson(text), text);
    }

    Y_UNIT_TEST(TimeTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("(12:31 30-04-2019)"),
            "(%Time_12:31% %Date_30-04-2019%)");
    }

    Y_UNIT_TEST(SmileTest) {
        TString text{"чибарем😃"};
        UNIT_ASSERT_STRINGS_EQUAL(Unperson(text), text);
    }

    Y_UNIT_TEST(TokenTypeOnlyTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("(12:31 30-04-2019)", NUnperson::EErasureType::TokenTypeOnly),
            "(%Time% %Date%)");
    }

    Y_UNIT_TEST(UriFailTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("[mailto:odava@\ninlight.ru\n]"),
            "[%Uri_mailto_odava_mailto:odava%@\n%Uri_http_inlight.ru_inlight.ru%\n]");
    }

    Y_UNIT_TEST(JniWrapperTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "Гибралтар–Лабрадор, начинается пожар, Гибралтар, 100500"),
            "Гибралтар–Лабрадор, начинается пожар, Гибралтар, %Number_475298793%");
    }

    Y_UNIT_TEST(JniWrapperTokenTypeOnlyTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "Гибралтар–Лабрадор, начинается пожар, Гибралтар, 100500",
                "<Unperson>\nErasureType: TokenTypeOnly\n</Unperson>"),
            "Гибралтар–Лабрадор, начинается пожар, Гибралтар, %Number%");
    }

    Y_UNIT_TEST(TokenSplitTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "ВΑМ ДОСТУПЕН БОНУС 87000РУБ, ДЛЯ ЗAПОЛНEНИЯ РЕКВИЗИТОВ ПЕРЕХОДИТЕ",
                "<Unperson>\nErasureType: TokenTypeOnly\n</Unperson>"),
            "%MayBeName% ДОСТУПЕН БОНУС %Number%%FirstName%, ДЛЯ ЗAПОЛНEНИЯ %LastName% ПЕРЕХОДИТЕ");
    }

    Y_UNIT_TEST(DeobfuscatorTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "ВΑМ ДОСТУПЕН БОНУС 87000РУБ, ДЛЯ ЗAПОЛНEНИЯ РЕКВИЗИТОВ ПЕРЕХОДИТЕ",
                DeobfuscateConfig),
            "вам доступен бонус %Number% руб для заполнения реквизитов переходите");
    }

    Y_UNIT_TEST(BonusTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "$1000_руб. 5000$ (500 БОНУСОВ) ((ДВОЙНЫЕ БОНУСЫ)) <(ВЫПЛАТЫ)> СЧЕТОВ СЧЕТА ВЫПЛАТА БОНУСЫ",
                DeobfuscateConfig),
            "%ShortNumber% руб %ShortNumber% %ShortNumber% бонусов двойные бонусы выплаты счетов счета выплата бонусы");
    }

    Y_UNIT_TEST(MatchMathSymbolsTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            // TODO: Add 'Ɓ' to deobfuscator
            JniWrapperUnperson(
                "Ҫ∏АСИБʘ ЗA BAШ ЛAЙК, ƁАШ ∏РИЗ 197000RUB 50000RUR ∏ҎИЗ "
                "СОСТАВИТ ДО 182000rub ∏ЕΡЕХΌДИТЕ рубл рубль рубли рублей "
                "рублёв рубликов бакс баксов баксы доллар долларов доллары евро",
                DeobfuscateConfig),
            "спасибо за ваш лайк %MayBeName% приз %Number% rub %Number% rur приз "
            "составит до %Number% rub переходите рубл рубль рубли рублей "
            "рублев рубликов бакс баксов баксы доллар долларов доллары "
            "евро");
    }

    Y_UNIT_TEST(NameHomonymsTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "Промокодов промокодов Коротких коротких Примените Промокоды Промокод",
                DeobfuscateConfig),
            "%LastName% промокодов коротких коротких примените промокоды промокод");
    }

    Y_UNIT_TEST(PasswordDeobfuscatorTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "∏рuвет, П0ДAPOK, П0К0КККААСВВА",
                DeobfuscateConfig),
            "привет подарок %Password%");
    }

    Y_UNIT_TEST(CollapseNumberTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "$1,030.49\n2022.49руб\n200 000долларов",
                DeobfuscateConfig),
            "1 %ShortNumber% %ShortNumber%\n%ShortNumber% %ShortNumber% руб\n%ShortNumber% 000 долларов");
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "$1,030.49\n2022.49руб\n200 000долларов",
                DeobfuscateConfigNoUnpersonUri),
            "%Number_1,030.49%\n%Number_2022.49% руб\n%Number_200 000% долларов");
    }

    Y_UNIT_TEST(ShortDateTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            Unperson("Начинаем 20 ноя 1999, заканчиваем 30 ноября"),
            "Начинаем %Date_20 ноя 1999%, заканчиваем %Date_30 ноября%");
    }

    Y_UNIT_TEST(ReplacementsTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                // English `o', `b' and `c'
                "O ВЫПЛАТАХ B ЭТОМ МЕСЯЦЕ C ПРОЦЕНТАМИ СВОЮ ПРИБЫЛЬ ПОЛУЧИТЕ",
                DeobfuscateConfigNoUnpersonUri),
            "выплатах месяце процентами прибыль получите");
    }

    Y_UNIT_TEST(AccentsTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "Литва́ официальное название — Лито́вская Респу́блика ЛИТОВСКИЙ",
                DeobfuscateConfigNoUnpersonUri),
        "%LastName_Литва% официальное название литовская республика литовский");
    }

    Y_UNIT_TEST(NamesSplitTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "Про ГУЛАГ написал А.И.Солженицын, известный русский писатель. "
                "А про войну — Л. Н. Толстой, известный дворянин. "
                "Александр Ф. Скляр рассказал нам про огуречную улицу.",
                DeobfuscateConfigNoUnpersonUri),
        // Truncated due to MaxTokens
        "гулаг написал %LastName_Солженицын% известный русский писатель\n"
        "войну %LastName_Толстой% известный дворянин\n"
        "%FirstName_Александр% %LastName_Скляр% рассказал огуречную");
    }

    Y_UNIT_TEST(DeobfuscatorLfTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "ещё один тeст ջվգե afфыва ջգվ ı İ ß ẞ\n.",
                DeobfuscateConfig),
            "еще один тест ջվգե afфыва ջգվ i i ß ß\n");
    }

    Y_UNIT_TEST(DeobfuscatorHardTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                // The very last character is И\u0301\u0306, so simple NFC
                // doesn't fold it to Й\u0301, we fix this in EraseMarks
                "∏рuвет, П0ДAPOK, П0К0КККААСВВА, 0XУGЕNS посмоmpeла CBЁKЛA 3eлёный ЗАГЛАВНЫИ́̆ 1 000 000РУБ",
                DeobfuscateConfigNoUnpersonUri),
            "привет подарок %Password_П0к0кккаасвва% oxygens посмотрела свекла зеленый заглавный %Number_1 000 000% руб");
    }

    Y_UNIT_TEST(FusedTokensTest) {
        UNIT_ASSERT_STRINGS_EQUAL(
            JniWrapperUnperson(
                "+$GNGGA,103454",
                DeobfuscateConfigNoUnpersonUri),
            "%FirstName_ондда% %Number_103454%");
    }

    Y_UNIT_TEST(LongStringTest) {
        constexpr size_t size = 1 << 20;
        TString str{size, 'A'};
        UNIT_ASSERT_STRINGS_EQUAL(Unperson(str), "%MayBeName_" + str + '%');

        TString str2{TString::Join(str, '@', str, ".ru")};
        UNIT_ASSERT_STRINGS_EQUAL(Unperson(str2), "%Password_" + str2 + '%');
    }
}

