#include <mail/so/libs/wmd_fast_text/wmd.h>

#include <mail/so/libs/fast_text/fasttext.h>

#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/stream/output.h>
#include <util/string/cast.h>

struct TWordMatch {
    TString Word{};
    float Distance{5.0f};
};

struct TWordInfo {
    TString Word;
    fasttext::TDoc Doc;
    size_t GroupId;

    TWordMatch BestSyn;
    TWordMatch WorstSyn;
    TWordMatch BestOther;

    TWordInfo(const TString& word, const fasttext::FastText& model, size_t groupId)
        : Word(word)
        , Doc(word, model)
        , GroupId(groupId)
    {
        WorstSyn.Distance = -5.0f;
    }

    TWordInfo(TWordInfo&& other) = default;
};

int main(int argc, char** argv) {
    if (argc < 2) {
        Cerr << "Usage: " << argv[0] << " <models...>" << Endl;
        return 1;
    }
    TVector<TVector<TString>> words = {
        {"моря", "приморский", "морской"},
        {"достоинством", "достойно", "достойный"},
        {"беленький", "белый", "светлый", "светленький"},
        {"красиво", "красивый", "прелестный", "великолепный"},
        {"приз", "выигрыш", "подарок", "презент"},
        {"безотлагательно", "срочно", "немедля", "немедленно"},
        {"брать", "забирать", "получать"},
        {"мамы", "мамина"},
        {"черный", "темный", "мрачный"},
        {"лисья", "лисы"},
        {"лесом", "лесу"},
        {"ночи", "ночная", "ночной"},
        {"каменная", "камня"},
        {"шерстяные", "шерсти"},
        {"острый", "заостренный"},
        {"мокрый", "влажный"},
        {"главный", "основной"},
        {"стул", "табурет"},
        {"электронный", "электрический"},
        {"письмо", "сообщение", "емейл", "извещение", "уведомление"},
        {"наличные", "деньги", "средства", "денежные", "рубли"},
        {"постановлению", "распоряжению", "указу"},
        {"помеха", "препятствие", "преграда"},
        {"непонятный", "неясный", "необъяснимый"},
        {"сотрудник", "работник", "служащий"},
        {"выплата", "начисление", "начисление", "компенсация", "перевод", "транзакция"},
        {"обработан", "посчитан", "подготовлен"},
        {"привет", "приветствую", "здравствуйте", "здрасти"},
        {"последнее", "окончательное", "крайнее"},
        {"коронавирус", "инфекция", "пандемия", "эпидемия"},
        {"весьма", "очень", "вполне"},
        {"заморозить", "удержать", "отменить"},
        {"появитесь", "объявитесь"},
        {"дней", "суток"},
        {"подробности", "детали", "разъяснения"},
        {"нужно", "необходимо", "требуется"},
        {"%Date%", "%Time%", "%Timestamp%"},
        {"%FirstName%", "%LastName%", "%MayBeName%"}
    };
    for (int i = 1; i < argc; ++i) {
        fasttext::FastText model;
        model.loadModel(argv[i]);

        TVector<TWordInfo> infos;
        for (size_t j = 0; j < words.size(); ++j) {
            for (auto x: words[j]) {
                infos.emplace_back(x, model, j);
            }
        }
        for (auto& left: infos) {
            for (auto& right: infos) {
                if (&left != &right) {
                    float distance = fasttext::MakeDistances(left.Doc, right.Doc).Data[0];
                    if (left.GroupId == right.GroupId) {
                        if (distance < left.BestSyn.Distance) {
                            left.BestSyn.Distance = distance;
                            left.BestSyn.Word = right.Word;
                        }
                        if (distance > left.WorstSyn.Distance) {
                            left.WorstSyn.Distance = distance;
                            left.WorstSyn.Word = right.Word;
                        }
                    } else if (distance < left.BestOther.Distance) {
                        left.BestOther.Distance = distance;
                        left.BestOther.Word = right.Word;
                    }
                }
            }
        }
        size_t good = 0;
        size_t bad = 0;
        size_t warn = 0;
        Cout << "\nModel: " << argv[i] << Endl << Endl;
        Cout << "Word\tBest syn\tBest syn dist\tWorst syn\tWorst syn dist\tBad match\tBad match dist\tStatus" << Endl;
        Cout << "----\t--------\t-------------\t---------\t--------------\t---------\t--------------\t------" << Endl;
        for (const auto& info: infos) {
            Cout << info.Word << "\t" << info.BestSyn.Word << "\t" << FloatToString(info.BestSyn.Distance, PREC_POINT_DIGITS_STRIP_ZEROES, 3)
                << "\t" << info.WorstSyn.Word << "\t" << FloatToString(info.WorstSyn.Distance, PREC_POINT_DIGITS_STRIP_ZEROES, 3)
                << "\t" << info.BestOther.Word << "\t" << FloatToString(info.BestOther.Distance, PREC_POINT_DIGITS_STRIP_ZEROES, 3) << "\t";
            if (info.WorstSyn.Distance < info.BestOther.Distance) {
                ++good;
                Cout << "good";
            } else if (info.BestSyn.Distance < info.BestOther.Distance) {
                ++warn;
                Cout << "warn";
            } else {
                ++bad;
                Cout << "bad";
            }
            Cout << Endl;
        }
        Cout << "Score:\t" << good << "\t" << warn << "\t" << bad << Endl;
    }
    return 0;
}

