#!/usr/bin/env python2.7
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
# Description:  Various auxiliary functions for using in other modules.
#
from __future__ import print_function
import os, os.path, sys, re, json, urllib2
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, 'WORKING_DIR')
from socket import _GLOBAL_DEFAULT_TIMEOUT
from random import randrange
from urllib import urlopen
from time import strftime
from smtplib import SMTP
from traceback import format_exception

__author__ = "Yaroslav Klimik <klimiky@yandex-team.ru>"
__version__ = "1.0"

CFG = {
    "retry_cnt":    3,
    "home_dir":     os.environ["HOME"] if "HOME" in os.environ else "/root",
    # Some predefined emails
    'robot':        'Robot Mailspam <robot-mailspam@yandex-team.ru>',
    'report':       'so-report@yandex-team.ru',
    # SMTP settings
    'smtp_host':    "outbound-relay.yandex.net",
    'smtp_port':    25,
    'smtp_timeout': 5.0
}

TVM = {
    "API": {
        "host":           "localhost",
        "port":           1111,
        "url":            "http://{0}:{1}/tvm",
        "URL":            "http://localhost:1111/tvm"
    },
    "BB": {
        "host":           "blackbox-mail.yandex.net",
        "url":            "https://{0}/blackbox?",
        "URL":            "https://blackbox-mail.yandex.net/blackbox?",
        "client_id":      2016577,
        "prod_client_id": 222,
        "timeout":        3.0
    },
    "BBcorp": {
        "host":           "blackbox.yandex-team.ru",
        "url":            "https://{0}/blackbox?",
        "URL":            "https://blackbox.yandex-team.ru/blackbox?",
        "client_id":      2016577,
        "prod_client_id": 223,
        "timeout":        3.0
    }
}


def get_traceback():
    exc_type, exc_value, exc_traceback = sys.exc_info()
    tb = ''
    for step in format_exception(exc_type, exc_value, exc_traceback):
        try:
            tb += "\t" + step.strip() + "\n"
        except:
            pass
    return tb


def writelog(msg, isTB=False, fh=None, prefix='', isAddTS=True):
    if not msg:
        return
    if not fh:
        fh = sys.stderr
    try:
        tb = "\n"
        if isTB:
            tb += get_traceback()
        s = prefix + msg + tb
        if isAddTS:
            s = strftime("[%Y-%m-%d %H:%M:%S]: ") + s
        os.write(fh.fileno(), s)
    except Exception, e:
        print("Writelog error: %s.%s" % (str(e), get_traceback()), file=sys.stderr)
        sys.stderr.flush()


def getUUID(sep='-'):
    if not sep:
        sep = '-'
    chars, s = ['a', 'b', 'c', 'd', 'e', 'f', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'], []
    for i in range(32):
        s.append(chars[int(randrange(16))])
    s.insert(8,  sep)
    s.insert(13, sep)
    s.insert(18, sep)
    s.insert(23, sep)
    return ''.join(s)


def doRequest(url, prompt="doRequest"):
    try:
        f = urlopen(url)
        if f.getcode() == 200:
            return f.read()
        else:
            writelog('{0} response HTTP code: {1}, body: {2}'.format(prompt, f.getcode(), f.info()))
    except Exception, e:
        writelog('%s HTTP request failed: %s.\n' % (prompt, str(e)), True)
    return ""


def getDC4Host(host):
    rec = doRequest('https://ro.admin.yandex-team.ru/api/host_query.sbml?hostname=%s&columns=short_dc,short_line,dc' % host, 'getDC4Host')
    if rec:
        return rec.split("\t")[0]
    return ""


def getGroup4Host(host):
    rec = doRequest('https://c.yandex-team.ru/api-cached/hosts2groups/%s' % host, 'getGroup4Host')
    if rec:
        return rec.split()[0]
    return ""


def getHosts4Group(conductor_group, default_hosts=[]):
    if conductor_group:
        for i in range(CFG["retry_cnt"]):
            r = doRequest("https://c.yandex-team.ru/api-cached/groups2hosts/%s" % conductor_group, "Get DB cluster hosts for group %s" % conductor_group)
            if r:
                hosts = map(str.strip, r.splitlines())
                return hosts if len(hosts) > 0 else default_hosts
            else:
                continue
    return default_hosts


def getCType():
    ctype = 'production'
    with open('/etc/yandex/environment.type') as f:
        ctype = f.readline().strip()
    return ctype


def getRoute():
    env_name_file = '/etc/yandex/environment.name'
    route = ""
    if not os.path.exists(env_name_file):
        env_name_file = './environment.name'
    if os.path.exists(env_name_file):
        with open(env_name_file) as f:
            route = f.readline().strip()
            if route.startswith("so"):
                route = route[2:]
    return route


def requestService(url, headers={}, data=None, timeout=_GLOBAL_DEFAULT_TIMEOUT, retry_cnt=CFG['retry_cnt'], log_fh=None):
    code = -1
    for i in range(retry_cnt):
        try:
            r = f = None
            r = urllib2.Request(url=url, data=data, headers=headers)
            f = urllib2.urlopen(r, timeout=timeout)
            if f:
                return f.read(), f.getcode()
            else:
                writelog('requestService request #%d response is empty!' % (i + 1), fh=log_fh)
                continue
        except Exception, e:
            serr = info = method = ''
            if hasattr(e, 'code'):
                code = e.code
                serr = ' (code=%s)' % code
            info = ' Info: {0}. '.format(f.info()) if f else ''
            method = "{0} ".format(r.get_method()) if r else ''
            writelog("requestService HTTP %srequest (attempt #%d) failed%s: '%s'. URL: %s.%s" % (method, i + 1, serr, str(e), url, info), True, log_fh)
            continue
    return "", code


def loadTVMConfig(TVM, log_fh=None):
    f, CURDIR, tvm_port = None, CFG["home_dir"], TVM["API"]["port"]
    try:
        with open("/etc/tvmtool/tvmtool.conf") as f:
            try:
                tvm_config = json.loads(f.read().strip())
                if 'port' in tvm_config:
                    tvm_port = int(tvm_config['port'])
            except Exception as e:
                errorlog("Failed to parse tvmtool's config!")
        TVM['API']['URL'] = TVM['API']['url'].format(TVM['API']['host'], tvm_port)
        with open("/var/lib/tvmtool/local.auth") as f:
            TVM['API']['Auth'] = f.read().strip()
    except Exception as e:
        writelog("loadTVMConfig failed: %s" % str(e), True, log_fh)


def getTVM2ticket(service, log_fh=None):
    resp, svc = {}, service.lower()
    try:
        answer, code = requestService('%s/tickets?dsts=%s' % (TVM['API']['URL'], svc), headers={"Authorization": TVM['API']['Auth']}, log_fh=log_fh)
        if code != 200:
            writelog("getTVM2ticket HTTPS error for service '%s' (status=%s): %s" % (service, code, str(answer)), False, log_fh)
        try:
            if not answer or len(answer) < 1:
                writelog("getTVM2ticket error answer for service '%s' (code=%s): %s" % (service, code, str(answer)), False, log_fh)
            else:
                resp = json.loads(answer)
        except Exception as e:
            writelog("getTVM2ticket decoding of JSON from TVM ticket query's answer failed for service '%s': %s" % (service, str(e)), True, log_fh)
    except Exception as e:
        writelog("getTVM2ticket HTTPS request failed for service '%s': %s." % (service, str(e)), True, log_fh)
    return resp[svc]['ticket'] if svc in resp and resp[svc] else None


def requestServiceByTVM(url, service, headers={}, data=None, log_fh=None):
    resp, code = '', 200
    ticket = getTVM2ticket(service, log_fh)
    if ticket:
        n = TVM[service].get("retry_cnt", 1)
        if not headers:
            headers = {}
        headers['X-Ya-Service-Ticket'] = ticket
        for i in range(n):
            try:
                resp, code = requestService(url, headers=headers, data=data, timeout=TVM[service].get("timeout", _GLOBAL_DEFAULT_TIMEOUT), retry_cnt=1, log_fh=log_fh)
            except Exception as e:
                writelog('requestServiceByTVM failed: %s. TicketAnswer: %s' % (str(e), str(ticket)), True, log_fh)
            if code == 200:
                break
            else:
                writelog('requestServiceByTVM failed (attempt #%s from %s, status=%s). Response: "%s"' % (i + 1, n, code, resp), False, log_fh)
                continue
    return resp, code


def requestBlackBox(params, bb_type='BB', log_fh=None):
    url = params if isinstance(params, basestring) else ('&'.join(map(lambda it: "{0}={1}".format(*it), params.items())) if isinstance(params, dict) else '')
    return requestServiceByTVM(TVM[bb_type]['URL'] + url, bb_type, log_fh=log_fh)


def getUID(login, bb_type='BB', log_fh=None):
    params = "method=userinfo&login=%s&userip=127.0.0.1" % login
    try:
        content, code = requestBlackBox("method=userinfo&login=%s&userip=127.0.0.1" % login, bb_type, log_fh)
        if code == 200:
            m, uid = re.search(r'<uid\b[^<>]*?>(\d+)<\/uid>', content.strip(), re.M), ''
            if m:
                uid = m.group(1)
            else:
                writelog("Failed to parse uid from BlackBox answer: %s" % content, False, log_fh)
            return uid
        else:
            writelog("Retriving UID by login from BB failed! Code: %s. Answer: %s." % (code, content), False, log_fh)
    except Exception, e:
        writelog("getUID exception: %s" % str(e), True, log_fh)
    return ''


def getUserInfo(filter_str, bb_type='BB', info_type="brief", log_fh=None):
    uid, login, suid, karma_status, reg_date, country, add_info_params = '', '', '', '', '', '', ''
    if info_type and info_type != "brief" and info_type != "uid":
        add_info_params = ",accounts.login.uid,subscription.suid.2"
        if info_type == "full":
            add_info_params += ",userinfo.reg_date.uid,userinfo.country.uid"
    params = "method=userinfo&%s&userip=127.0.0.1&dbfields=subscription.suid.2%s" % (filter_str, add_info_params)   # "&format=json" - for JSON answer
    info, code = requestBlackBox(params, bb_type, log_fh)
    if code == 200:
        m = re.search(r'<uid\b[^<>]*?>(\d+)<\/uid>', info)
        uid,= m.group(1) if m and m.group(1) else ''
        if info_type and info_type != "uid":
            m = re.search(r'<dbfield id="accounts.login.uid">(.*?)<\/dbfield>', info)
            if m and m.group(1):
                login = m.group(1)
            else:
                m = re.search(r'<login>([^<]+)', info)
                if m and m.group(1):
                    login = m.group(1)
            if info_type != "brief":
                m = re.search(r'^<dbfield id="subscription.suid.2">(\d+)</dbfield>', info, re.M)
                if m and m.group(1):
                    suid = m.group(1)
                if info_type == "full":
                    m = re.search(r'^<karma_status>(\d+)', info, re.M)
                    if m and m.group(1):
                        karma_status = m.group(1)
                    m = re.search(r'^<dbfield id="userinfo.reg_date.uid">([^<>]*?)</dbfield>', info, re.M)
                    if m and m.group(1):
                        reg_date = m.group(1)
                    m = re.search(r'^<dbfield id="userinfo.country.uid">([^<>]*?)</dbfield>', info, re.M)
                    if m and m.group(1):
                        country = m.group(1)
    else:
        writelog("getUserInfo failed to obtain user info from BlackBox (code=%s): %s" % (code, info), False, log_fh)
        if filter_str.startswith("uid="):
            uid = filter_str[4:]
        elif filter_str.startswith("login="):
            login = filter_str[6:]
        elif filter_str.startswith("suid="):
            suid = filter_str[5:]
    return uid, login, suid, karma_status, reg_date, country


def sendEmail(msg, fromaddr=CFG['robot'], toaddr=CFG['report'], log_fh=None):
    errorStr = ''
    try:
        if toaddr:
            server = SMTP(CFG['smtp_host'], CFG['smtp_port'], timeout=CFG['smtp_timeout'])
            server.sendmail(fromaddr, toaddr, msg)
            server.quit()
        else:
            errorStr = "Sending email failed: to-addr must be set!"
            writelog(errorStr, fh=log_fh)
    except Exception, e:
        errorStr = "Error in sendEmail: %s" % str(e)
        writelog(errorStr, True, fh=log_fh)
    return errorStr
