#!/usr/bin/env python2.7
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
from __future__ import print_function
import os
import os.path
import sys
import fcntl as fcntl
import argparse
from ConfigParser import SafeConfigParser
from socket import getfqdn
from signal import signal, SIG_IGN, SIGUSR1, SIGINT, SIGHUP, SIGCHLD, SIGPIPE, SIGQUIT, SIGABRT, SIGTERM
from hashlib import md5
from common import writelog

LOGS_FOLDER = '/logs'
STATE_FOLDER = '/state'

class LogTail:
    def __init__(self, config_path=None, **kwargs):
        self.LOCK = None
        self.offsetSize = 0
        self.offsetSizeNew = 0
        self.cfg = SafeConfigParser()
        self.cfg.optionxform = str
        self.cur_dir = os.path.dirname(os.path.abspath(__file__))
        if not self.cur_dir.endswith('/'):
            self.cur_dir = self.cur_dir + '/'
        self.conf_path = config_path if config_path else ''
        if not os.path.exists(self.conf_path):
            conf_filename = self.conf_path[self.conf_path.rfind('/') + 1:]
            self.conf_path = self.cur_dir + conf_filename
            if not os.path.exists(self.conf_path):
                self.conf_path = self.cur_dir + '../{0}'.format(conf_filename)
                if not os.path.exists(self.conf_path):
                    self.conf_path = ""
        if self.conf_path and os.path.isfile(self.conf_path):
            self.cfg.read(self.conf_path)
            if not self.cfg.has_section('log-tail'):
                writelog("Error: config file '%s' must have section 'log-tail'" % self.conf_path)
                sys.exit(1)
        self.log_file = kwargs['log'] if 'log' in kwargs and kwargs['log'] else (self.cfg.get('log-tail', 'log') if self.cfg.has_option('log-tail', 'log') else '')
        if not self.log_file:
            writelog("Error: log path must be specified!")
            sys.exit(1)
        if self.log_file.find('$h') > -1:
            self.log_file = self.log_file.replace('$h', getfqdn())
        self.log_id_text = kwargs['id'] if 'id' in kwargs and kwargs['id'] else (self.cfg.get('log-tail', 'id') if self.cfg.has_option('log-tail', 'id') else '')
        self.log_id = md5("%s %s" % (self.log_file, self.log_id_text)).hexdigest()
        self.log_err = kwargs['errorlog'] if 'errorlog' in kwargs and kwargs['errorlog'] else (self.cfg.get('log-tail', 'errorlog') if self.cfg.has_option('log-tail', 'errorlog') else '%s/log-tail.log' % LOGS_FOLDER)
        self.ERRORLOG = open(self.log_err, 'a+t')
        self.state_dir = kwargs['tmpdir'] if 'tmpdir' in kwargs and kwargs['tmpdir'] else (self.cfg.get('log-tail', 'tmpdir') if self.cfg.has_option('log-tail', 'tmpdir') else '%s/so-log-tail' % STATE_FOLDER)
        if self.state_dir.endswith('/'):
            self.state_dir = self.state_dir[: len(self.state_dir) - 1]
        if not os.path.isdir(self.state_dir):
            os.makedirs(self.state_dir, 0755)
        self.log_sep = kwargs['sep'] if 'sep' in kwargs and kwargs['sep'] else (self.cfg.get('log-tail', 'sep') if self.cfg.has_option('log-tail', 'sep') else '\n')
        self.max_records = kwargs['max'] if 'max' in kwargs and kwargs['max'] else (self.cfg.getint('log-tail', 'max_records') if self.cfg.has_option('log-tail', 'max_records') else 0)
        self.lock_file = "{0}/{1}".format(self.state_dir, self.log_id)
        self.offsetSize, self.offsetSizeNew, self.l, self.count = 0, os.stat(self.log_file).st_size, len(self.log_sep), 0
        if os.path.isfile(self.lock_file):
            try:
                self.LOCK = open(self.lock_file, 'rt')
            except Exception, e:
                self.error("Error while opening file '%s' for reading: %s" % (self.lock_file, str(e)), True)
                sys.exit(1)
            try:
                fcntl.flock(self.LOCK, fcntl.LOCK_EX | fcntl.LOCK_NB)
            except Exception, e:
                self.error("Unable to lock file '%s': %s" % (self.lock_file, str(e)), True)
                sys.exit(1)
            s = ''
            try:
                s = self.LOCK.read()
                self.offsetSize = int(s.strip()) if s.strip() else 0
            except Exception, e:
                self.error("Invalid content of file '%s': %s" % (self.lock_file, s), True)
                self.offsetSize = 0
            fcntl.flock(self.LOCK, fcntl.LOCK_UN | fcntl.LOCK_NB)
            self.LOCK.close()
        else:
            try:
                self.LOCK = open(self.lock_file, 'wt')
            except Exception, e:
                self.error("Error while opening file '%s' for writing: %s" % (self.lock_file, str(e)), True)
                sys.exit(1)
            self.LOCK.close()
        try:
            self.LOCK = open(self.lock_file, 'r+')
        except Exception, e:
            self.error("Error while opening file '%s' for reading and writing: %s" % (self.lock_file, str(e)), True)
            sys.exit(1)
        try:
            fcntl.flock(self.LOCK, fcntl.LOCK_EX | fcntl.LOCK_NB)
        except Exception, e:
            self.error("Unable to lock file '%s': %s. Most likely process with ID '%s' already launched." % (self.lock_file, str(e), self.log_id), True)
            sys.exit(1)
        try:
            self.LOG = open(self.log_file, 'rt')
        except Exception, e:
            self.error("Error while opening input log-file '%s' for reading: %s" % (self.log_file, str(e)), True)
            sys.exit(1)
        while self.offsetSizeNew > 0:
            self.offsetSizeNew -= 1
            self.LOG.seek(self.offsetSizeNew, 0)
            s = self.LOG.read(self.l)
            if s == self.log_sep:
                self.offsetSizeNew += self.l
                break
        if self.offsetSizeNew < self.offsetSize:
            self.offsetSize = 0
        self.LOG.seek(self.offsetSize, 0)

    def finalizing(self, offset):
        try:
            if not self.LOCK.closed and (self.LOCK.mode.startswith('r+') or self.LOCK.mode.startswith('w')):
                self.LOCK.seek(0, 0)
                self.LOCK.write("%s" % offset)
                self.LOCK.truncate(len(str(offset)))
                #fcntl.flock(self.LOCK, fcntl.LOCK_UN)
                self.LOCK.close()
            if hasattr(self, 'LOG') and not self.LOG.closed:
                self.LOG.close()
        except Exception, e:
            self.error("Finalizing error: %s" % str(e), True)

    def error(self, msg, isTB=False):
        writelog(msg, isTB, self.ERRORLOG, "{0} ".format(self.log_id_text.upper()))

    def __del__(self):
        self.finalizing(self.offsetSizeNew)

    def __call__(self):
        tail = ''
        while self.offsetSize < self.offsetSizeNew:
            row = self.LOG.readline()
            if not row:
                self.offsetSizeNew = self.offsetSize
                break
            if self.log_sep == "\n":
                buf = row
                self.count += 1
            else:
                buf = tail
                while row and row.find(self.log_sep) == -1:
                    buf += row
                    row = self.LOG.readline()
                if buf and tail:
                    self.count += 1
                if row and row.find(self.log_sep) > -1:
                    buf += row[:row.find(self.log_sep)]
                    tail = row[row.find(self.log_sep):]
            self.offsetSize += len(buf)
            yield buf
            if self.max_records and self.max_records <= self.count:
                self.offsetSizeNew = self.offsetSize

global LOG_TAIL

def gotSignal(signum, frame):
    global LOG_TAIL
    if LOG_TAIL:
        LOG_TAIL.finalizing(LOG_TAIL.offsetSize)
    sys.exit(1)

if __name__ == "__main__":
    signal(SIGCHLD, SIG_IGN)
    signal(SIGPIPE, gotSignal)
    signal(SIGQUIT, gotSignal)
    signal(SIGABRT, gotSignal)
    signal(SIGTERM, gotSignal)
    signal(SIGINT,  gotSignal)
    signal(SIGHUP,  gotSignal)

    parser = argparse.ArgumentParser()
    parser.add_argument('-c', '--conf',   type=str, help="Path to configuration file")
    parser.add_argument('-l', '--log',    type=str, help="Path to input log")
    parser.add_argument('-e', '--errorlog', type=str, help="Path to error log")
    parser.add_argument('-i', '--id',     type=str, help="Unique string for identification current log parsing process")
    parser.add_argument('-m', '--max',    type=str, help="Max records number for one call")
    parser.add_argument('-s', '--sep',    type=str, help="Record separator", default="\n")
    parser.add_argument('-t', '--tmpdir', type=str, help="Temporary directory for saving current file positions", default='%s/so-log-tail' % STATE_FOLDER)
    args = parser.parse_known_args()[0]
    if not args.conf and not(args.log and args.id):
        parser.print_help()
        exit(0)
    LOG_TAIL = LogTail(args.conf) if 'conf' in args else LogTail(**vars(args))
    for rec in LOG_TAIL():
        print(rec, end=''); sys.stdout.flush()

