#!/usr/bin/env python
#-*- coding: utf-8 -*-
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import os
import re
import time
import lzo
import base64
import socket
import zmq
import json
from bson.int64 import Int64
from pymongo import UpdateOne, DeleteMany
from pymongo.errors import ConnectionFailure
from collections import defaultdict, OrderedDict
from tornado.gen import coroutine, sleep, multi, Return
from tornado.ioloop import IOLoop, PeriodicCallback
from tornado.queues import Queue
from multiprocessing import Process
from threading import Thread
from db import loadMongoDbCredentials, getMongoDB
from logger import *

MIN_BLOCK_SIZE = 1000
MAX_BLOCK_SIZE = 10000
YEAR = time.strftime("%Y")
MESS_RE = re.compile(r'^(?:mess:)?\s*([A-Za-z]+\s+\d+\s+\d\d\:\d\d\:\d\d)', re.M)
RE = {
    "mess":        re.compile(r"^mess:[^-]+-\s*(\d+)", re.M),
    "from_addr":   re.compile(r"^From:addr:\s*(\S+)", re.I|re.M),
    "messageid":   re.compile(r"^MESSAGEID:\s*(\S+)", re.I|re.M),
    "rcvd":        re.compile(r"^rcvd: source ip = (\S+)", re.M),
    "queueid":     re.compile(r"^X-Yandex-QueueID:\s*(\S+)", re.I|re.M),
    "locl":        re.compile(r"^locl: ([^.]+)", re.M),
    "front":       re.compile(r"^X-Yandex-Front:\s*mx([^\.]+)", re.I|re.M),
    "sender_suid": re.compile(r'^mfrm: [^\n]+\bid=(\d+)\s', re.M),
    "sender_uid":  re.compile(r'^mfrm: [^\n]+\buid=(\d+)\s', re.M),
    "uid":         re.compile(r'\buid=(\d+)\s'),
    "rcpt_uid":    re.compile(r'^rcpt: [^\n]+\b(uid=\d+\s.*?)$', re.M),
    "code":        re.compile(r"^r_nl: [^\n]+, R(\d+),\s*", re.M),
    "pers_corr":   re.compile(r"^r_nl:.*\bPERSONAL_CORRECT\b", re.M)
}


def decompress(s):
    try:
        len_s = len(s)
        s += r"=" * (int((len_s + 3) / 4) * 4)
        return lzo.decompress(base64.urlsafe_b64decode(s))
    except Exception, e:
        error("Error while decompressing message: %s" % str(e))


def ip_to_integer(ipstr):
    try:
        return socket.inet_ntop(socket.AF_INET, socket.inet_pton(socket.AF_INET, ipstr))
    except socket.error, e:
        pass
    try:
        return socket.inet_ntop(socket.AF_INET6, socket.inet_pton(socket.AF_INET6, ipstr))
    except socket.error, e:
        pass
    return ""


def search_group(regexp, data, default=''):
    try:
        match = RE[regexp].search(data)
        if match:
            return match.group(1)
    except Exception, e:
        error("Error '%s' while parsing regexp: '%s'" % (str(e), regexp))
    return default


def parse_log(log):
    ipstr = search_group("rcvd", log)
    code = search_group("code", log, 0)
    match = RE["pers_corr"].search(log)
    if match:
        if code == '1':
            code = '8'
        elif code == '4':
            code = '127'
    return {
        "ts":        int(search_group("mess", log, str(int(time.time())))),
        "fromaddr":  search_group("from_addr", log).lower(),
        "mid":       search_group("messageid", log).strip("<>"),
        "ip":        ip_to_integer(ipstr),
        "qid":       search_group("queueid", log),
        "locl":      search_group("locl", log),
        "mx":        search_group("front", log),
        "suid":      search_group('sender_suid', log),
        "uid":       search_group('sender_uid', log),
        "rcpt_uids": RE["uid"].findall(search_group('rcpt_uid', log)),
        "code":      code,
        "log":       log.encode("koi8-r")
    }


class LocalDBClient(object):
    def __int__(self):
        LocalDBClient.init(self)

    def init(self):
        self.index_db = self.queue_db = None
        self.db_ready = False
        self.useParsing = True
        self.rulesStat = defaultdict(lambda: defaultdict(lambda: defaultdict(Int64)))
        self.rulesDaily = defaultdict(lambda: defaultdict(lambda: defaultdict(Int64)))

    def connectLocalDB(self, useIndexDB=True, useQueueDB=True):
        self.db_ready = False
        while not self.db_ready:
            try:
                if useIndexDB:
                    self.index_db = getMongoDB({'db': "solog"}, {"maxPoolSize": 256, "socketTimeoutMS": 300000})
                if self.useParsing:
                    self.queue_db = getMongoDB({'db': "queue"}, {"maxPoolSize": 256, "socketTimeoutMS": 300000})
                self.db_ready = True
            except Exception, e:
                error("LocalDBClient.connectLocalDB failed: %s" % str(e))
                time.sleep(5)

    @coroutine
    def insertDataToQueue(self, queue_name, data):
        attempt = 1
        while attempt < 4:
            try:
                collection = self.queue_db[queue_name]
                collection.ensure_index("ts")
                collection.ensure_index("time")
                collection.insert_many(data)
                break
            except ConnectionFailure, e:
                error("LocalDBClient.insertDataToQueue failed to insert data to queue DB collection '%s' because of DB connection's loss: %s. Attempt %d" % (queue_name, str(e), attempt))
                self.connectLocalDB()
                attempt += 1
                yield sleep(0.1)
            except Exception, e:
                error("LocalDBClient.insertDataToQueue failed to insert data to queue DB collection '%s': %s" % (queue_name, str(e)))
                break

    def loadDataFromQueue(self, queue_name, ts):
        attempt, success = 1, False
        while attempt < 4:
            try:
                collection = self.queue_db[queue_name]
                for record in collection.find({"ts": {"$lte": ts}}):
                    d = time.strftime("%Y-%m-%d", time.localtime(record["time"]))
                    for par, val in record["stat"].iteritems():
                        self.rulesStat[record["time"]][record["rule"]][par] += val
                        self.rulesDaily[d][record["rule"]][par] += val
                success = True
                break
            except ConnectionFailure, e:
                success = False
                error("LocalDBClient.loadDataFromQueue failed to insert data to queue DB collection '%s' because of DB connection's loss: %s. Attempt %d" % (queue_name, str(e), attempt))
                self.connectLocalDB(useIndexDB=False)
                attempt += 1
                self.rulesStat.clear()
                self.rulesDaily.clear()
                time.sleep(0.1)
            except Exception, e:
                error("LocalDBClient.loadDataFromQueue failed to insert data to queue DB collection '%s': %s" % (queue_name, str(e)))
                success = False
                break
        return success

    def deleteDataFromQueue(self, queue_name, ts):
        attempt, success = 1, False
        while attempt < 4:
            try:
                collection = self.queue_db[queue_name]
                collection.delete_many({"ts": {"$lte": ts}})
                success = True
                break
            except ConnectionFailure, e:
                success = False
                error("LocalDBClient.deleteDataFromQueue failed to delete data from queue DB collection '%s' because of DB connection's loss: %s. Attempt %d" % (queue_name, str(e), attempt))
                self.connectLocalDB(useIndexDB=False)
                attempt += 1
                time.sleep(0.1)
            except Exception, e:
                err_text, success, t0 = str(e), False, ts - 30 * 86400
                error("LocalDBClient.deleteDataFromQueue failed to delete data from queue DB collection '%s': %s" % (queue_name, err_text))
                if err_text.find("timed out") > -1:
                    success = True
                    while t0 <= ts:
                        try:
                            collection = self.queue_db[queue_name]
                            collection.delete_many({"ts": {"$lte": t0}})
                        except Exception, e:
                            error("LocalDBClient.deleteDataFromQueue failed to delete data from queue DB collection '%s': %s" % (queue_name, str(e)))
                            success = False
                            break
                        t0 += 86400
                if not success:
                    break
        return success

    @coroutine
    def insertDataToIndex(self, table_name, data):
        attempt = 1
        while attempt < 4:
            try:
                collection = self.index_db[table_name]
                index_info = collection.index_information()
                if "ip4_1_ip3_1_ip2_1_ip_1" not in index_info:
                    collection.ensure_index("ip")
                if "code_1" not in index_info:
                    collection.ensure_index("fr")
                    collection.ensure_index("ts")
                collection.ensure_index("code")
                collection.ensure_index("uid")
                collection.ensure_index("suid")
                collection.ensure_index("rcpt_uid")
                collection.ensure_index("mid")
                collection.ensure_index("qid")
                collection.ensure_index("offset")
                collection.insert_many(data)
                break
            except ConnectionFailure, e:
                error("LocalDBClient.insertDataToIndex failed to insert data to queue DB collection '%s' because of DB connection's loss: %s. Attempt %d" % (table_name, str(e), attempt))
                self.connectLocalDB(useQueueDB=self.useParsing)
                attempt += 1
                yield sleep(0.1)
            except Exception, e:
                error("LocalDBClient.insertDataToIndex failed to insert data to queue DB collection '%s': %s" % (table_name, str(e)))
                break


class RulesStatsSender(Thread, LocalDBClient):
    def __int__(self):
        Thread.__init__(self)
        LocalDBClient.__init__(self)

    def init(self, route, rules_db_cfg, queue_batch_size):
        LocalDBClient.init(self)
        self.daemon = False
        self.route = route
        self.rules_db_cfg = rules_db_cfg
        self.queue_batch_size = queue_batch_size

    def sendRulesStats2DB(self):
        t0, cnt, cntDaily = time.time(), 0, 0
        try:
            self.getDataFromQueue()
            requests = []
            for (t, rule_stat) in self.rulesStat.iteritems():
                for (rule, stat) in rule_stat.iteritems():
                    requests.append(({'rule': rule, 'time': t}, {'$inc': dict(stat)},))
            cnt = len(requests)
            if cnt > 0:
                self.sendBatchUpdates('detailed_Rules_%s' % self.route, requests)
            requests = []
            for (d, rules_stat) in self.rulesDaily.iteritems():
                for (rule, stat) in rules_stat.iteritems():
                    requests.append(({'rule': rule, 'date': d}, {'$inc': dict(stat)},))
            cntDaily = len(requests)
            if cntDaily > 0:
                self.sendBatchUpdates('Rules_%s' % self.route, requests)
        except Exception, e:
            error("RulesStatsSender.sendRulesStats2DB exception while DB operations: %s" % str(e))
        trace("RulesStatsSender.sendRulesStats2DB done in %f sec. Requests count: %d + %d" % (time.time() - t0, cnt, cntDaily))

    def sendBatchUpdates(self, table_name, queries):
        attempt, cnt = 1, len(queries)
        while attempt < 3 and len(queries) > 0:
            try:
                self.rules_db[table_name].bulk_write(map(lambda query: UpdateOne(*query, upsert=True), queries))
                queries = []
                break
            except Exception, e:
                err_text = str(e)
                error("RulesStatsSender.sendBatchUpdates failed in bulk request to DB collection '%s' (batch_size=%d): %s. Attempt: %d" % (table_name, len(queries), err_text, attempt))
                if err_text.find("timed out") > -1:
                    bad_queries = []
                    for query in queries:
                        try:
                            self.rules_db[table_name].update_one(*query, upsert=True)
                        except Exception, e:
                            error("RulesStatsSender.sendBatchUpdates failed in request to DB collection '%s': %s. Query: %s. Attempt: %d" % (table_name, str(e), str(query), attempt))
                            bad_queries.append(query)
                    queries = bad_queries
            attempt += 1
        if len(queries) > 0:
            trace("RulesStatsSender.sendBatchUpdates failed to send updates into table '%s' %d queries from %d" % (table_name, len(queries), cnt))
        else:
            trace("RulesStatsSender.sendBatchUpdates successfully sent updates into table '%s' (queries count = %d)" % (table_name, cnt))

    def getDataFromQueue(self):
        ts, success = time.time(), False
        self.rulesStat = defaultdict(lambda: defaultdict(lambda: defaultdict(Int64)))
        self.rulesDaily = defaultdict(lambda: defaultdict(lambda: defaultdict(Int64)))
        try:
            table_name = 'so_%s' % self.route.lower()
            if self.loadDataFromQueue(table_name, ts):
                if self.deleteDataFromQueue(table_name, ts):
                    success = True
                else:
                    self.rulesStat.clear()
                    self.rulesDaily.clear()
            trace("RulesStatsSender.getDataFromQueue: time=%f sec, success=%s" % (time.time() - ts, success))
        except Exception, e:
            error("RulesStatsSender.getDataFromQueue exception: %s" % str(e))

    def run(self):
        self.connectLocalDB(useIndexDB=False)
        try:
            self.rules_db = getMongoDB(self.rules_db_cfg, {"maxPoolSize": 256, "socketTimeoutMS": 300000})
        except Exception, e:
            error("RulesStatsSender.run DB exception: %s" % str(e))
        while True:
            try:
                self.sendRulesStats2DB()
                time.sleep(1)
            except Exception, e:
                error("RulesStatsSender.run exception: %s" % str(e))


class ZMQueue(object):
    def __init__(self):
        self.zmq_queue_ready = self.zmq_client_ready = False
        self.zmq_client_context = self.zmq_client = self.zmq_queue_context = self.zmq_queue = None

    def reconnectZMQueue(self):
        self.zmq_client_ready = False
        while not self.zmq_client_ready:
            try:
                if not self.zmq_client_context or self.zmq_client_context and self.zmq_client_context.closed:
                    self.zmq_client_context = zmq.Context().instance()
                if not self.zmq_client or self.zmq_client and self.zmq_client.closed:
                    self.zmq_client = self.zmq_client_context.socket(zmq.PUSH)
                    self.zmq_client.setsockopt(zmq.SNDHWM, self.max_queue_size)
                    self.zmq_client.setsockopt(zmq.RCVHWM, self.max_queue_size)
                    self.zmq_client.connect(self.queue_socket)
                self.zmq_client_ready = True
            except Exception, e:
                error("ZMQueue.reconnectZMQueue failed: %s" % str(e))
                time.sleep(1)

    def rebindZMQueue(self):
        self.zmq_queue_ready = False
        while not self.zmq_queue_ready:
            try:
                if not self.zmq_queue_context or self.zmq_queue_context and self.zmq_queue_context.closed:
                    self.zmq_queue_context = zmq.Context().instance()
                if not self.zmq_queue or self.zmq_queue and self.zmq_queue.closed:
                    self.zmq_queue = self.zmq_queue_context.socket(zmq.PULL)
                    self.zmq_queue.bind(self.queue_socket)
                self.zmq_queue_ready = True
            except Exception, e:
                error("ZMQueue.rebindZMQueue failed: %s" % str(e))
                time.sleep(1)

    def pushDataToZMQueue(self, obj):
        success = False
        while True:
            try:
                self.zmq_client.send_pyobj(obj, zmq.NOBLOCK)
                self.zmq_client_ready = success = True
                break
            except zmq.ZMQError, e:
                self.zmq_client_ready = success = False
                if e.errno == 11:
                    error("ZMQueue.pushDataToQueue 0MQ error: Queue is full")
                else:
                    error("ZMQueue.pushDataToQueue 0MQ error: %s" % str(e))
                self.reconnectZMQueue()
            except Exception, e:
                error("ZMQueue.pushDataToQueue failed to add message to queue: %s" % str(e))
                success = False
                break
            time.sleep(0.1)
        return success

    @coroutine
    def pullDataFromZMQueue(self):
        queries = []
        for i in xrange(MAX_BLOCK_SIZE):
            try:
                params = self.zmq_queue.recv_pyobj(zmq.NOBLOCK)
                queries.append(params)
            except zmq.ZMQError, e:
                if e.errno == 11:
                    if i < MIN_BLOCK_SIZE:
                        yield sleep(0.1)
                        break
                else:
                    error("ZMQueue.pullDataFromQueue 0MQ recv failed: %s" % str(e))
            except Exception, e:
                error("ZMQueue.pullDataFromQueue failed to extract message from 0MQ queue: %s" % str(e))
        raise Return(queries)


class LogDumper(Process, LocalDBClient, ZMQueue):
    def __init__(self, sotype, cfg, rules_db_cfg):
        Process.__init__(self)
        LocalDBClient.__init__(self)
        ZMQueue.__init__(self)
        self.daemon = True
        self.sotype = sotype.lower()
        self.path = cfg["path"]
        self.queue_socket = cfg["queue_socket"]
        self.max_queue_size = cfg["queue_size"]
        self.useParsing = self.do_parsing = True if cfg["parsing"] == "on" else False
        self.indexers_concurrency = cfg["indexers_concurrency"]
        self.send_time_period = cfg["send_time_period"]
        self.data_time_step = cfg["data_time_step"]
        if self.do_parsing:
            loadMongoDbCredentials(rules_db_cfg)
            self.parsing_queue_size = cfg["parsing_queue_size"]
            self.parsing_queue_warning_size = cfg["parsing_queue_warning_size"]
            self.parsers_concurrency = cfg["parsers_concurrency"]
            self.rules_stats_sender = RulesStatsSender()
            self.rules_stats_sender.init(sotype.capitalize(), rules_db_cfg, cfg["parsing_queue_batch_size"])
        self.start()

    def add(self, data):
        params = {}
        try:
            if "mail" in data:
                log = decompress(data["mail"]).decode("koi8-r", "ignore").strip()
                if log:
                    params.update(parse_log(log + "\n\n"))
            elif "json" in data:
                log = decompress(data["json"]).strip()
                if log:
                    if log[0] != '[' and log[0] != '{':
                        log += "\n"
                    params.update({
                        "ts":        data["ts"],
                        "fromaddr":  data["fromaddr"],
                        "mid":       data["msgid"].strip("<>"),
                        "ip":        ip_to_integer(data["source_ip"]),
                        "qid":       data["queueid"],
                        "locl":      data["locl"],
                        "mx":        data["mx"],
                        "uid":       data["uid"],
                        "suid":      data["suid"],
                        "rcpt_uids": data["rcpts_uids"].split(','),
                        "code":      data["code"],
                        "log":       log + "\n"
                    })
        except Exception, e:
            error("LogDumper.add failed to extract data from imcoming query: %s" % str(e))
        self.reconnectZMQueue()
        if len(params) > 0:
            if "rcpt_uids" not in params or not isinstance(params["rcpt_uids"], list) or len(params["rcpt_uids"]) < 1:
                params["rcpt_uids"] = ['']
            return self.pushDataToZMQueue(params)
        return True

    def getFile(self):
        datestr = time.strftime("%Y%m%d")
        filename = "so_%s%s.log" % (self.sotype, datestr)
        if not hasattr(self, "logfile") or not self.logfile.name.endswith(filename):
            self.logfile = open(os.path.join(self.path, filename), "at")
        return self.logfile

    def getOffset(self):
        return self.getFile().tell()

    @coroutine
    def writeIndex(self, indexes):
        datestr = time.strftime("%Y%m%d")
        yield self.insertDataToIndex("so_%s%s" % (self.sotype, datestr), indexes)

    @coroutine
    def writeLog(self, data):
        try:
            logfile = self.getFile()
            logfile.write(data)
            logfile.flush()
        except Exception, e:
            error("LogDumper.writeLog failed: %s" % str(e))

    @coroutine
    def handleQueries(self, queries):
        indexes, logs = [], []
        offset = self.getOffset()
        for params in queries:
            log = params["log"]
            for rcpt_uid in params["rcpt_uids"]:
                indexes.append({
                    "offset":   offset,
                    "size":     len(log),
                    "ts":       params["ts"],
                    "fr":       params["fromaddr"],
                    "mid":      params["mid"],
                    "qid":      params["qid"],
                    "locl":     params["locl"],
                    "mx":       params["mx"],
                    "ip":       params["ip"],
                    "uid":      params["uid"],
                    "suid":     params["suid"],
                    "rcpt_uid": rcpt_uid,
                    "code":     params["code"]
                })
            logs.append(log)
            offset += len(log)
        t1 = time.time()
        yield self.writeIndex(indexes)
        t2 = time.time()
        yield self.writeLog(''.join(logs))
        t3 = time.time()
        if self.do_parsing:
            yield self.parsing_queue.put(logs)
        trace("Count: %d. WriteIndex: %f sec. WriteLog: %f sec." % (len(queries), t2 - t1, t3 - t2))

    @coroutine
    def processLogItems(self):
        parsers = []
        if self.do_parsing:
            self.parsing_queue = Queue(self.parsing_queue_size)
            parsers = [self.parseLog() for _ in range(self.parsers_concurrency)]
        indexers = [self.processQueries() for _ in range(self.indexers_concurrency)]
        #trace("LogDumper.processLogItems")
        yield multi(indexers + parsers)

    @coroutine
    def processQueries(self):
        while True:
            queries = yield self.pullDataFromZMQueue()
            if queries:
                try:
                    yield self.handleQueries(queries)
                    #trace("LogDumper.processQueries (added: %s, total: %s)" % (len(queries), self.parsing_queue.qsize()))
                except Exception, e:
                    error("LogDumper.processQueries failed to process queries: %s" % str(e))

    @coroutine
    def parseLog(self):
        last_warning_ts = int(time.time())
        while True:
            try:
                logs = yield self.parsing_queue.get()
                if logs == None:
                    break
                ts = time.time()
                if self.parsing_queue.qsize() > self.parsing_queue_warning_size and last_warning_ts + 1 < ts:
                    trace("LogDumper.parseLog: in parsing's queue remains %s records" % self.parsing_queue.qsize())
                    last_warning_ts = ts
                queue_data, data = [], defaultdict(lambda: defaultdict(lambda: defaultdict(Int64)))
                for log in logs:
                    if log is None:
                        trace("LogDumper.parseLog: parsing's queue has exhausted")
                        break
                    try:
                        valid, t, rules, rc, banst = yield self.prepareRecord(log)
                        if not valid:
                            error("LogDumper.parseLog failed to parse delivery-log's record: '%s'" % log)
                            continue
                        t = int(t / 60) * 60
                        t0 = int(t / self.data_time_step) * self.data_time_step
                        if re.search(r'\b(PERSONAL_CORRECT)\b', rules):
                            if rc == 4:
                                rc = 127
                            elif rc == 1:
                                rc = 8
                        if banst:
                            data[t0][banst]['R%s' % rc] += 1
                        for rule in re.split(r',\s*', rules):
                            if not rule:
                                continue
                            m = re.match(r'^(\w+)', rule)
                            if m:
                                data[t0][m.group(1)]["R%s" % rc] += 1
                            else:
                                error("LogDumper.parseLog failed to parse rule '%s' in log's fragment: %s" % (rule, log))
                    except Exception, e:
                        error("LogDumper.parseLog failed: %s" % str(e))
                for (t0, rule_stat) in data.iteritems():
                    for (rule, stat) in rule_stat.iteritems():
                        queue_data.append({"ts": ts, "time": t0, "rule": rule, "stat": dict(stat)})
                if len(queue_data):
                    yield self.insertDataToQueue('so_%s' % self.sotype, queue_data)
                trace("LogDumper.parseLog done in %f sec. QueueDataCount: %d" % (time.time() - ts, len(queue_data)))
            except Exception, e:
                error("LogDumper.parseLog exception: %s" % str(e))

    @coroutine
    def prepareRecord(self, record):
        valid, t, rules, rc, banst, data = False, time.time(), '', '', '', {}
        mess_header = record
        if record[0] == '[' or record[0] == '{':
            try:
                data = OrderedDict(json.loads(record.decode("koi8-r")))
            except Exception, e:
                error("LogDumper.prepareRecord failed to parse JSON: %s. InputData: %s." % (str(e), record))
            mess_header = data["mess"] if "mess" in data else ""
            rules = ' '.join(map(lambda k: data["r_{}".format(k)] if "r_{}".format(k) in data else '', ['sp', 'dl', 'nl']))
        else:
            rules = ' '.join(re.findall(r'^r_(?:sp|dl|nl): (.+)', record, re.M))
        m = MESS_RE.match(mess_header)
        if m:
            try:
                t = int(time.mktime(time.strptime("{0} {1}".format(YEAR, m.group(1)), "%Y %b %d %H:%M:%S")))
                if t >= time.time() + 86400:
                    t = time.time()
                m = re.search(r',\s*R(\d+),\s*$', rules)
                if m:
                    rc = int(m.group(1))
                    if rc:
                        valid = True
                else:
                    m = re.search(r' (BANST_\d+)(?:_M)?R(\d+),', rules)
                    if m:
                        rc, banst = int(m.group(2)), m.group(1)
                        rules = rules[:m.start()]
                        valid = True
                    else:
                        error("LogDumper.prepareRecord failed: wrong format of r_nl header in log record!")
            except Exception, e:
                error("LogDumper.prepareRecord exception '%s' while checking log record: %s" % (str(e), record))
        raise Return((valid, t, rules, rc, banst))

    def run(self):
        self.rebindZMQueue()
        self.connectLocalDB(useQueueDB=self.do_parsing)
        IOLoop.current().add_callback(self.processLogItems)
        if self.do_parsing:
            self.rules_stats_sender.start()
        IOLoop.current().start()
