#!/usr/bin/python2
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import os, os.path, sys, argparse, json, urllib2
from urllib import urlopen
import yt.wrapper as ytw
from traceback import format_exception

RETRY_COUNT = 3
NIRVANA_API_HOST = 'https://nirvana.yandex-team.ru'
SANDBOX_URL = "https://proxy.sandbox.yandex-team.ru/%s/%s"
SO_ML_BACKEND_URL = "https://so-web.n.yandex-team.ru/ml"

def get_traceback():
    exc_type, exc_value, exc_traceback = sys.exc_info()
    tb = ''
    for step in format_exception(exc_type, exc_value, exc_traceback):
        try:
            tb += "\t" + step.strip() + "\n"
        except:
            pass
    return tb

def writelog(msg, isTB = False):
    if not msg: return
    try:
        tb = "\n"
        if isTB:
            tb = get_traceback()
        print >>sys.stderr, msg, tb
    except Exception, e:
        print >>sys.stderr, "Writelog error: %s" % str(e)

def doRequest(url, prompt):
    try:
        f = urlopen(url)
        if f.getcode() == 200:
            return f.read()
        else:
            print >>sys.stderr, '{0} response HTTP code: {1}, body: {2}'.format(prompt, f.getcode(), f.info())
    except Exception, e:
        writelog('%s HTTP request failed: %s.%s' % (prompt, str(e), get_traceback()))
    return ""

def downloadFileFromSandBox(resource_id, resource_file, output_file):
    try:
        f = open(output_file, 'w')
        f.write(doRequest(SANDBOX_URL % (resource_id, resource_file), "Retrieving formula's description file '%s' from SandBox" % resource_file))
        f.close()
    except Exception, e:
        writelog('Downloading file from SandBox failed: %s.%s' % (str(e), get_traceback()))

def getMLInfo(params = {}, prompt = 'getMLInfo'):
    info = {}
    try:
        url = "{0}/ui/".format(SO_ML_BACKEND_URL) + ('?{0}'.format('&'.join(map(lambda it: "%s=%s" % (it[0], it[1]), params.items()))) if params else '')
        info = json.loads(doRequest(url, prompt))
    except Exception, e:
        writelog("%s failed: %s.%s" % (prompt, str(e), get_traceback()))
    return info

def getModelInfo(formula_id, route = 'in'):
    return getMLInfo({"action": "get_model", "formula_id": formula_id}, 'getModelInfo')

def getProdModelsInfo(route = 'in'):
    return getMLInfo({"action": "get_prod_models", "route": route}, 'getProdModelsInfo')

def loadInfo(info_source, file_type = 'json'):
    info = {}
    try:
        if file_type == 'json':
            f = open(info_source)
            info = json.loads(f.read())
            f.close()
        elif file_type == 'text':
            f = open(info_source)
            info = f.read().strip()
            f.close()
        elif file_type == 'json_url':
            info = json.loads(doRequest(info_source, "Downloading Nirvana workflow block's result"))
        elif file_type == 'text_url':
            info = doRequest(info_source, "Downloading Nirvana workflow block's result")
    except Exception, e:
        writelog("Loading models info file failed: %s.%s" % (str(e), get_traceback()))
    return info

def requestNirvana(method, params, token):
    for i in range(RETRY_COUNT):
        try:
            r = urllib2.Request(url = '%s/api/public/v1/%s' % (NIRVANA_API_HOST, method),
                data = json.dumps({"jsonrpc": "2.0", "method": method, "id": params['workflowId'], "params": params}),
                headers = {'Content-Type': 'application/json; charset=utf-8', 'Authorization': 'OAuth %s' % token})
            f = urllib2.urlopen(r)
            if f:
                return f.read()
            else:
                writelog('requestNirvana request #%d response is empty!' % (i + 1))
                continue
        except urllib2.URLError, e:
            writelog('requestNirvana HTTP request (attempt #%d) failed: %s\n' % (i + 1, e.reason), True)
            continue
        except urllib2.HTTPError, e:
            writelog('requestNirvana HTTP request (attempt #%d) failed (code=%s): %s\n' % (i + 1, e.code, e.reason), True)
            continue
        except Exception, e:
            writelog('requestNirvana HTTP request #%d failed: %s\n' % (i + 1, str(e)), True)
            continue
    return ""

def getMLblockOutput(block_info):
    output = ''
    try:
        workflow = json.loads(requestNirvana('getBlockResults', {'workflowId': block_info['workflow_id'], 'workflowInstanceId': block_info['workflow_instance_id'], 'blocks': [{'code': block_info['local_id']}], 'outputs': [block_info['output']]}))
        if 'error' in workflow:
            writelog('NIRVANA getBlockResults error: code=%s, message="%s", data="%s"' % (workflow['error']['code'], workflow['error']['message'], workflow['error']['data'] if 'data' in workflow['error'] else ''))
        else:
            writelog("getMLblockOutput Workflow info: %s" % str(workflow))
            output = workflow['result'][0]['results'][0]['directStoragePath']
    except Exception, e:
        writelog("Exception while retrieving Nirvana workflow block's result: %s" % str(e), True)
    return output

def getModelResourceID(model_info):
    resource_id = 0
    try:
        resource_id = int(loadInfo(getMLblockOutput({
            'workflowId':         model_info['workflow_id'],
            'workflowInstanceId': model_info['workflow_instance_id'],
            'local_id':           'operation-1516696928277-121$773',
            'output':             'resource_id'
        }), 'text_url'))
        writelog("For model %s resource_id=%s" % (model_info['formula_id'], resource_id))
    except Exception, e:
        writelog("Exception while retrieving Nirvana workflow block's result: %s" % str(e), True)
    return resource_id

def vw_thresholds(model_info):
    threshold = 0
    FILE = "./vw_%s.json" % model_info["vw_model_resource_id"]
    downloadFileFromSandBox(model_info["vw_model_resource_id"], 'vw.json', FILE)
    if os.stat(FILE).st_size > 10:
        vw_model_info = loadInfo(FILE)
        threshold = vw_model_info['best_f1']['threshold']
    if not threshold:
        vw_model_info = {}
        try:
            vw_model_info = loadInfo(getMLblockOutput({
                'workflowId':         model_info['workflow_id'],
                'workflowInstanceId': model_info['workflow_instance_id'],
                'local_id':           'operation-1525784969401-134$193',
                'output':             'stats'
            }), 'json_url')
            writelog("For model %s VW model info: %s" % (model_info['formula_id'], str(vw_model_info)))
            threshold = float(vw_model_info['best_f1']['threshold'])
        except Exception, e:
            writelog("Exception while retrieving Nirvana workflow block's result: %s" % str(e), True)
    return threshold

def writeJSON2file(file_path, data):
    try:
        f = open(file_path, 'w')
        f.write(json.dumps(data))
        f.close()
    except Exception, e:
        writelog("Writing file '%s' failed: %s.%s" % (file_path, str(e), get_traceback()))

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument('-r', '--route',       type = str, help = "The type of mail for which the models are calculated")
    parser.add_argument('-t', '--token',       type = str, help = "Application token for get access to Nirvana HTTP API for current user")
    parser.add_argument('-o', '--output_json', type = str, help = "Output JSON with prod SO models info")
    args = parser.parse_known_args()[0]
    ROUTE = args.route if args.route else 'in'
    OUTPUT_JSON = {'formula_ids': [], 'slots': [], 'sandbox_resource_ids': [], 'thresholds': [], 'vw_resource_ids': [], 'vw_thresholds': []}

    for model in getProdModelsInfo(ROUTE):
        model_info = getModelInfo(model["formula"], ROUTE)
        OUTPUT_JSON['formula_ids'].append(model["formula"])
        OUTPUT_JSON['slots'].append(model["slot"])
        OUTPUT_JSON['thresholds'].append(model["threshold"])
        OUTPUT_JSON['sandbox_resource_ids'].append(getModelResourceID(model_info))
        OUTPUT_JSON['vw_resource_ids'].append(model_info["vw_model_resource_id"])
        OUTPUT_JSON['vw_thresholds'].append(vw_thresholds(model_info))
    writeJSON2file(args.output_json, OUTPUT_JSON)
