#!/usr/bin/python2
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import os
import os.path
import sys
import time
import re
import argparse
import logging
import tempfile
import shutil
import tarfile
import contextlib
from traceback import format_exception
from cStringIO import StringIO
from urllib import urlopen
import sandbox.common as sbcommon
import sandbox.common.upload as sbupload

CURDIR = os.path.dirname(os.path.abspath(__file__))
SANDBOX_PROXY_URL = "http://proxy.sandbox.yandex-team.ru"
SANDBOX_URL = "http://sandbox.yandex-team.ru"

logging.basicConfig(level=logging.INFO, format="%(asctime)-15s %(levelname)8s %(process)6d %(message)s")

global subs, is_meta, task_id, resource_id


def get_traceback():
    exc_type, exc_value, exc_traceback = sys.exc_info()
    tb = ''
    for step in format_exception(exc_type, exc_value, exc_traceback):
        try:
            tb += "\t" + step.strip() + "\n"
        except:
            pass
    return tb


def writelog(msg, isTB=False, logf=logging.info):
    if not msg:
        return
    try:
        tb = "\n"
        if isTB:
            tb = get_traceback()
        logf(msg + tb)
    except Exception, e:
        print >>sys.stderr, "Writelog error: %s" % str(e), get_traceback()


def doRequest(url, prompt):
    try:
        f = urlopen(url)
        if f.getcode() == 200:
            return f.read()
        else:
            print >>sys.stderr, '{0} response HTTP code: {1}, body: {2}'.format(prompt, f.getcode(), f.info())
    except Exception, e:
        print >>sys.stderr, '%s HTTP request failed: %s' % (prompt, str(e))
    return ""


def substitution(matchobj):
    global subs, is_meta, task_id, resource_id
    ctx = meta = None
    if is_meta:
        import nirvana.job_context as nv
        ctx = nv.context()
        meta = ctx.get_meta()
    try:
        subs.append(eval(matchobj.group(1), {'ctx': ctx, 'meta': meta, 'task_id': task_id, 'resource_id': resource_id}))
    except Exception, e:
        writelog("Substitution error: %s" % str(e), True)
    return "{%s}" % (len(subs) - 1)


def substitute(s):
    global subs
    subs = []
    return re.sub(r'\\*\$\{([^{}]+)\}', substitution, s).format(*subs)


@contextlib.contextmanager
def make_file_metas(root):
    (parent, d) = os.path.split(os.path.abspath(root).rstrip('/'))
    os.chdir(parent)
    result = []

    def addfile(path, fake=False):
        result.append(sbupload.HTTPHandle.FileMeta(StringIO() if fake else open(path), 0 if fake else os.stat(path).st_size, path))
    try:
        if os.path.isdir(d):
            for (path, subdirs, files) in os.walk(d):
                if len(files):
                    for f in files:
                        addfile(os.path.join(path, f))
                else:
                    addfile(os.path.join(path, '.keep'), fake=True)
        else:
            addfile(d)
        yield result
    finally:
        for r in result:
            r.handle.close()


def do_upload(filename, resource_meta, token):
    writelog('Uploading {}'.format(filename))
    upload_states = sbcommon.types.misc.Upload
    result = (None, None)
    with make_file_metas(filename) as fmetas:
        uploader = sbupload.HTTPHandle(resource_meta, token, SANDBOX_URL, SANDBOX_PROXY_URL, None, *fmetas)
        for state in uploader():
            if isinstance(state, upload_states.Prepare):
                result = (state.task_id, state.resource_id)
    return result


def main():
    global subs, is_meta, task_id, resource_id
    parser = argparse.ArgumentParser()
    parser.add_argument('-b', '--input_files',             nargs='*', help="Input bundle, usually tar.gz-file")
    parser.add_argument('-s', '--sandbox_token',           type=str,  help="SandBox OAuth token")
    parser.add_argument('-n', '--untar',        action='store_true',  help="Whether to unarchive file or not")
    parser.add_argument('-t', '--sb_resource_type',        type=str,  help="Sandbox resource's type")
    parser.add_argument('-d', '--sb_resource_description', type=str,  help="SandBox resource description")
    parser.add_argument('-l', '--sb_resource_ttl',         type=int,  help="SandBox resource TTL (in days)")
    parser.add_argument('-a', '--sb_resource_attrs',       nargs='*', help="SandBox resource additional attributes")
    parser.add_argument('-c', '--sb_resource_arch',        type=str,  help="SandBox resource arch")
    parser.add_argument('-o', '--sb_resource_owner',       type=str,  help="SandBox resource owner")
    parser.add_argument('-p', '--sb_release_to',           type=str,  help="Release resource in SandBox as")
    parser.add_argument('-q', '--sb_release_description',  type=str,  help="Release resource with this description")
    parser.add_argument('-w', '--web_hook_url',            type=str,  help="Web-hook URL which to be called after finishing all job")
    parser.add_argument('-j', '--use_nirvana_job_context', action='store_true', help="Whether to use Nirvana job's context or not")
    parser.add_argument('-m', '--temp_dir',                type=str,  help="Folder for temporary files")
    parser.add_argument('-g', '--sub_dir',                 type=str,  help="Subfolder for include all files in resource")
    parser.add_argument('-x', '--clean_temp_files', action='store_true', help="Whether to clean temporary files")
    parser.add_argument('-i', '--output_task_id',          type=str,  help="SandBox task ID for the new task")
    parser.add_argument('-r', '--output_resource_id',      type=str,  help="SandBox resource ID for the new resource")
    parser.add_argument('-u', '--output_resource_url',     type=str,  help="SandBox resource URL for the new resource")
    args, subs = parser.parse_known_args()[0], []
    is_meta = True if args.use_nirvana_job_context else False
    RESOURCE_TYPE = args.sb_resource_type if args.sb_resource_type else 'OTHER_RESOURCE'
    RESOURCE_DESC = args.sb_resource_description if args.sb_resource_description else ''
    RELEASE_DESC = args.sb_release_description if args.sb_release_description else 'Autorelease {}'.format(RESOURCE_TYPE)
    WEB_HOOK_URL = args.web_hook_url if args.web_hook_url else ''
    RESOURCE_ATTRS = args.sb_resource_attrs if args.sb_resource_attrs else []
    if args.sb_resource_ttl:
        RESOURCE_ATTRS.append('ttl={}'.format(args.sb_resource_ttl))
    TEMP_DIR = args.temp_dir if args.temp_dir else CURDIR
    task_id, resource_id = None, None
    writelog("Meta: {}, Desc: {}, ResourceDesc: {}, {}.".format(is_meta, RELEASE_DESC, RESOURCE_DESC, RESOURCE_ATTRS))

    RESOURCE_DESC = substitute(RESOURCE_DESC)
    writelog("Resource description (substitution result): %s." % RESOURCE_DESC)
    RELEASE_DESC = substitute(RELEASE_DESC)
    writelog("Release description (substitution result): %s." % RELEASE_DESC)
    WEB_HOOK_URL = substitute(WEB_HOOK_URL)
    writelog("Web-hook URL (substitution result): %s." % WEB_HOOK_URL)

    try:
        if not os.path.exists(TEMP_DIR):
            os.makedirs(TEMP_DIR)
    except Exception, e:
        writelog("Creation of folder for temporary files failed: %s" % str(e), True)
        TEMP_DIR = CURDIR

    rmeta = sbupload.HTTPHandle.ResourceMeta(type=RESOURCE_TYPE, arch=args.sb_resource_arch if args.sb_resource_arch else 'any', owner=args.sb_resource_owner,
                                             description=RESOURCE_DESC, attributes=','.join(RESOURCE_ATTRS), release_to_yd=False)

    writelog("Processing files: {}".format(str(args.input_files)))
    if len(args.input_files) > 1 or len(args.input_files) == 1 and os.path.isdir(args.input_files[0]):
        td = tempfile.mkdtemp(dir=TEMP_DIR)
        try:
            d = os.path.join(td, args.sub_dir if args.sub_dir else 'files')
            if len(args.input_files) > 1:
                os.makedirs(d)
                for fn in args.input_files:
                    shutil.copy(fn, d)
            else:
                shutil.copytree(args.input_files[0], d)
            (task_id, resource_id) = do_upload(d, rmeta, args.sandbox_token)
        except Exception, e:
            writelog("Error while moving input files to temp folder '{0}' and uploading this folder to SandBox: {1}".format(td, str(e)), True)
        finally:
            if args.clean_temp_files:
                shutil.rmtree(td)
    elif args.untar:
        td = tempfile.mkdtemp(dir=TEMP_DIR)
        try:
            d = os.path.join(td, args.sub_dir if args.sub_dir else 'files')
            os.makedirs(d)
            writelog('Unpacking to {}'.format(d))
            with tarfile.open(args.input_files[0]) as tf:
                tf.extractall(d)
            (task_id, resource_id) = do_upload(d, rmeta, args.sandbox_token)
        except Exception, e:
            writelog("Error while moving input files to temp folder '{0}' and uploading this folder to SandBox: {1}".format(d, str(e)), True)
        finally:
            if args.clean_temp_files:
                shutil.rmtree(td)
    else:
        (task_id, resource_id) = do_upload(args.input_files[0], rmeta, args.sandbox_token)
    writelog('Task id: {task}, resource id: {res}'.format(task=task_id, res=resource_id))
    if args.sb_release_to is not None:
        writelog('Releasing task to {}'.format(args.sb_release_to))
        client = sbcommon.rest.Client(base_url=SANDBOX_URL + '/api/v1.0', auth=args.sandbox_token)
        client.release(task_id=task_id, subject=RELEASE_DESC, type=args.sb_release_to)
        status = None
        fail_states = frozenset(['NOT_RELEASED', 'DELETING', 'DELETED'])
        while status not in fail_states:
            try:
                status = client.task[task_id][:]['status']
                if status == 'RELEASED':
                    writelog('Resource released.')
                    break
                writelog('Current resource status is {}'.format(status))
            except Exception, e:
                writelog('Sandbox request for resource\'s releasing failed: {}'.format(str(e)), True, logging.error)
            time.sleep(1)
        writelog("Releasing DONE.")
    writelog('Writing result files')
    with open(args.output_resource_url, 'w') as fd:
        fd.write('{}/resource/{}/view'.format(SANDBOX_URL, resource_id))
    with open(args.output_resource_id, 'w') as fd:
        fd.write(str(resource_id))
    with open(args.output_task_id, 'w') as fd:
        fd.write(str(task_id))
    writelog('Writing result files DONE.')
    if WEB_HOOK_URL:
        doRequest(WEB_HOOK_URL, 'Call web-hook URL after finishing uploading process')


if __name__ == "__main__":
    main()
