#!/usr/bin/env python
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
import os
import os.path
import sys
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, 'WORKING_DIR')
import stat
import fcntl
import psycopg2
from subprocess import Popen, PIPE
from plotnik_common import CFG, PG_PLOTNIK, writeLog, getPGCredentials, getPGdb, get_metric_hash, get_table

LOGFILE = "%s/plotnik_collector.log" % CFG['log_dir']

global writelog

writelog = writeLog(LOGFILE)


def check_pid(filename):
    pid_fh = open(filename, "w")
    try:
        fcntl.lockf(pid_fh, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except IOError:
        return False
    print >>pid_fh, os.getpid()
    return True


def create_table(db, table):
    try:
        cursor = db.cursor()
        cursor.execute("""CREATE TABLE %s (
                                timestamp INTEGER NOT NULL,
                                metric BIGINT NOT NULL,
                                value REAL DEFAULT 0,
                                PRIMARY KEY(metric, timestamp)
                        )""" % table)
        writelog("DB notices: %s" % db.notices)
        cursor.close()
        db.commit()
        return 1
    except psycopg2.DatabaseError, e:
        writelog("psycopg2 Error of creating table '%s' (code: %s): %s. %s" % (table, e.pgcode, str(e), e.args), True, 'ERR')
    except Exception, e:
        writelog("Exception: %s" % str(e), True, 'ERR')
    db.rollback()
    return -1


def create_metric_table(db):
    try:
        cursor = db.cursor()
        cursor.execute("""CREATE TABLE metric (
                                metric VARCHAR(100) NOT NULL,
                                hash BIGINT NOT NULL,
                                PRIMARY KEY(metric)
                        )""")
        cursor.close()
        db.commit()
        return 1
    except psycopg2.DatabaseError, e:
        writelog("psycopg2 Error (code: %s): %s" % (e.pgcode, str(e)), True, 'ERR')
    except Exception, e:
        writelog("Exception: %s" % str(e), True, 'ERR')
    db.rollback()
    return -1


def update_metrics(db, metrics):
    hashes = []
    for metric in metrics:
        hashes.append((metric, get_metric_hash(metric)))

    cursor = db.cursor()
    for i in xrange(2):
        try:
            cursor.executemany("INSERT INTO metric (metric, hash) VALUES (%s, %s) ON CONFLICT DO NOTHING", hashes)
            db.commit()
            break
        except psycopg2.DatabaseError, e:
            db.rollback()
            if e.pgcode == '42P01':     # undefined_table
                create_metric_table(db)
            else:
                writelog("psycopg2 Error (code: %s): %s" % (e.pgcode, str(e)), True, 'ERR')
                break
        except Exception, e:
            writelog("Exception: %s" % str(e), True, 'ERR')
            break

    cursor.close()


def update_db(db, cursor, metric, value, timestamp):
    table = get_table(metric)
    metric_hash = get_metric_hash(metric)

    for i in xrange(2):
        try:
            value = value.lower()
            if value.find("_") >= 0:
                if value.startswith("set_"):
                    update_op = "value = %s" % value[4:]
                elif value.startswith("max_"):
                    update_op = "value = greatest(%s.value, %s)" % (table, value[4:])
                elif value.startswith("min_"):
                    update_op = "value = least(%s.value, %s)" % (table, value[4:])
                value = value.split("_")[-1]
            else:
                update_op = "value = %s.value + %s" % (table, value)

            cursor.execute("""INSERT INTO %(table)s (metric, value, timestamp)
                              VALUES('%(metric_hash)s', %(value)s, %(timestamp)s)
                              ON CONFLICT (metric, timestamp) DO UPDATE SET %(update_op)s""" % locals())
            break
        except psycopg2.DatabaseError, e:
            db.rollback()
            if e.pgcode == '42P01':     # undefined_table
                return create_table(db, table)
            else:
                writelog("psycopg2 error (code: %s): %s" % (e.pgcode, str(e)), True, 'ERR')
                return -1
        except Exception, e:
            db.rollback()
            writelog("Exception: %s" % str(e), True, 'ERR')
            return -1
    return 1


def upload_file(f, fid=None):
    values, res = [], ''
    try:
        for line in f:
            parts = line.lower().split()
            if len(parts) == 3:
                values.append(parts)
            elif len(line[:-1]) > 0:
                writelog("Process '%s': wrong line format '%s'" % (f.name if hasattr(f, 'name') else '<input list>', line), False, "ERR")
        writelog("Parsed lines for '%s': %s" % (fid, len(values)))
        getPGCredentials(PG_PLOTNIK)
        db = getPGdb(PG_PLOTNIK)
        update_metrics(db, set(map(lambda parts: parts[0], values)))
        cursor = db.cursor()
        for (metric, value, timestamp) in sorted(values):
            if update_db(db, cursor, metric.lower(), value, timestamp) < 0:
                cursor = db.cursor()
        db.commit()
    except Exception, e:
        res = str(e)
        writelog('Error while uploading data file to Plotnik: %s' % str(e), True, "ERR")
    return res


def run_script(filename, file_id=None):
    if not os.access(filename, os.X_OK):
        return
    writelog("Process '%s': start" % filename)
    res = ''
    try:
        data, err_str = Popen(filename, shell=True, stdout=PIPE, stderr=PIPE).communicate()
        if err_str:
            writelog("Process '%s' returns: %s" % (filename, err_str), False, "ERR")
        if not file_id:
            file_id = filename[filename.rfind('/')+1:filename.rfind('.')]
        res = upload_file(data.splitlines(), file_id)
    except Exception, e:
        writelog('Error while processing file %s: %s' % (filename, str(e)), True, "ERR")
    writelog("Process '%s': done%s" % (filename, ' with error "{0}"'.format(res) if res else ''))


def upload_filename(filename):
    if os.path.exists(filename) and os.path.isfile(filename):
        res = upload_file(open(filename))
        writelog("Process {0}: done{1}".format(filename, ' with error "{0}"'.format(res) if res else ''))
    else:
        writelog("Process %s: file does not exist!" % filename, False, "ERR")


if __name__ == '__main__':
    if len(sys.argv) < 2:
        print "Usage: plotnik_collector.py <directory_with_collectors>|-f <txt_filename>"
        sys.exit(1)

    topdir = sys.argv[1]

    if topdir == "-f":
        upload_filename(sys.argv[2])
    else:
        child_pids = []
        for root, dirs, files in os.walk(topdir):
            for name in files:
                file_path, file_id = os.path.join(root, name), name[:name.rfind('.')]
                pid_file = "{0}/plotnik_collector_{1}".format(CFG['pidfile_dir'], file_id)
                if not check_pid(pid_file):
                    writelog("Plotnik Collector is already running for %s" % name, False, "WARN")
                    continue
                if bool(os.stat(file_path).st_mode & stat.S_IXUSR):
                    try:
                        child_pid = os.fork()
                        if child_pid:
                            child_pids.append(child_pid)
                            continue
                        else:
                            run_script(file_path, file_id)
                            os.unlink(pid_file)
                            sys.exit(0)
                    except Exception, e:
                        writelog("Running of Plotnik Collector's child process failed: %s" % str(e), True, "ERR")
        for pid in child_pids:
            os.waitpid(pid, 0)
