#!/usr/bin/python2
# encoding: utf-8
# kate: space-indent on; indent-width 4; replace-tabs on;
#
from __future__ import print_function
import os
import os.path
import sys
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, 'WORKING_DIR')
import psycopg2
import pymongo
import redis
from binascii import crc32
from urllib import urlopen
from time import strftime
from datetime import datetime, timedelta
from traceback import format_exception

CFG = {
    'pidfile_dir':        'WORKING_DIR/pids',
    'log_dir':            '/logs',
    'plugins_folder':     'plugins',
    'retry_count':        3,
    'metric_table_parts': 2,
    'datetime_start':     datetime(1970, 1, 5)
}

PG_PLOTNIK = {
    "host": "plotnikdb01e.db.yandex.net,plotnikdb01f.db.yandex.net,sas-dwhnh2qpvlt2krvg.db.yandex.net",
    "port": 6432,
    "db":   "plotnikdb",
    "user": "plotnik"
}

PG_FRODO = {
    "host": "so-frodo-statdb01e.db.yandex.net,so-frodo-statdb01f.db.yandex.net,sas-vknw0n81pw74pdkv.db.yandex.net",
    "port": 6432,
    "db":   "so_frodo_statdb",
    "user": "so_frodo"
}

MONGO_RULES = {
    'hosts':   ["sas-agixxin7fbr78u0o.db.yandex.net", "vla-hwmeehtmq450wvke.db.yandex.net", "vlx-g9q00jkxh959zk6m.db.yandex.net"],
    'port':    27018,
    'db':      'rules',
    'user':    'solog'
}

REDIS = {
    "cluster_name":  "users_stat",
    "hosts":         ['sas-667541avf1c1ontn.db.yandex.net', 'vla-9cp0yx8ud9yrt90n.db.yandex.net'],
    "port":          6379,
    "sentinel_port": 26379,
    "db":            0,
    "timeout":       3.0,
    "auth":          True
}

REDIS_STAT = REDIS.copy()
REDIS_STAT["db"] = 1

global writelog


def get_traceback():
    exc_type, exc_value, exc_traceback = sys.exc_info()
    tb = ''
    for step in format_exception(exc_type, exc_value, exc_traceback):
        try:
            tb += "\t" + step.strip() + "\n"
        except:
            pass
    return tb


class writeLog():
    def __init__(self, log_file=None):
        self.fh = open(log_file, "a+t") if log_file and os.path.exists(log_file) else sys.stderr

    def __call__(self, msg, isTB=False, prefix='', isAddTS=True):
        if not msg:
            return
        try:
            tb = "\n"
            if isTB:
                tb += get_traceback()
            s = ("{0} ".format(prefix) if prefix else '') + msg + tb
            if isAddTS:
                s = strftime("[%Y-%m-%d %H:%M:%S]: ") + s
            os.write(self.fh.fileno(), s)
        except Exception, e:
            print("Writelog error: %s.%s" % (str(e), get_traceback()), file=sys.stderr)
            sys.stderr.flush()


writelog = writeLog()


def doRequest(url, prompt):
    try:
        f = urlopen(url)
        if f.getcode() == 200:
            return f.read()
        else:
            writelog('{0} response HTTP code: {1}, body: {2}'.format(prompt, f.getcode(), f.info()))
    except Exception, e:
        writelog('%s HTTP request failed: %s.' % (prompt, str(e)), True)
    return ""


def cluster_hosts(conductor_group, default_hosts=[]):
    if conductor_group:
        for i in range(CFG['retry_count']):
            r = doRequest("https://c.yandex-team.ru/api-cached/groups2hosts/%s" % conductor_group, "Get DB cluster hosts for group %s" % conductor_group)
            if r:
                hosts = map(str.strip, r.splitlines())
                return hosts if len(hosts) > 0 else default_hosts
            else:
                continue
    return default_hosts


def mongo_conn_str(cfg):
    prfx = ("%s:%s@" % (cfg['user'], cfg['passwd'])) if 'user' in cfg and cfg['user'] else ''
    return "mongodb://%s%s/%s" % (prfx, ','.join(cluster_hosts(cfg.get('cluster', ''), cfg['hosts'])), cfg['db'])


def getDbCredentials(cfg, db_type, db_name_field='db'):
    f, CURDIR, dbname = None, 'WORKING_DIR', cfg[db_name_field][:len(cfg[db_name_field]) - 2] if db_name_field in cfg and cfg[db_name_field].endswith('db') else cfg.get(db_name_field, '')
    try:
        if not os.path.exists('{0}/.{1}.{2}'.format(CURDIR, db_type, dbname)):
            CURDIR = os.path.dirname(os.path.abspath(__file__))
            if not os.path.exists('{0}/.{1}.{2}'.format(CURDIR, db_type, dbname)):
                CURDIR = os.environ['HOME'] if 'HOME' in os.environ else '/root'
        f = open('{0}/.{1}.{2}'.format(CURDIR, db_type, dbname))
        for line in f:
            sf = line.split(':')
            if len(sf) == 1:
                cfg['passwd'] = sf[0].strip()
                break
            elif len(sf) == 2 and sf[0] == cfg[db_name_field if db_type.startswith('redis') else 'user']:
                cfg['passwd'] = sf[1].strip()
                break
        f.close()
    except Exception, e:
        writelog("get%sCredentials exception: %s" % (db_type.capitalize(), str(e)), True)


def loadMongoDbCredentials(cfg):
    getDbCredentials(cfg, 'mongodb')


def getPGCredentials(cfg):
    getDbCredentials(cfg, 'pgpass')


def getRedisCredentials(cfg):
    getDbCredentials(cfg, 'redis', 'cluster_name')


def getPGdb(cfg, mode='read-write'):
    if not hasattr(getPGdb, "%s_connection" % cfg['db']) or hasattr(getattr(getPGdb, "%s_connection" % cfg['db']), 'closed') and getattr(getattr(getPGdb, "%s_connection" % cfg['db']), 'closed'):
        CURDIR = 'WORKING_DIR'
        if not os.path.exists('{0}/.pgsql/root.crt'.format(CURDIR)):
            CURDIR = os.path.dirname(os.path.abspath(__file__))
            if not os.path.exists('{0}/.pgsql/root.crt'.format(CURDIR)) and 'HOME' in os.environ:
                CURDIR = os.environ['HOME']
        if hasattr(psycopg2, '__libpq_version__') and psycopg2.__libpq_version__ < 100000:
            setattr(getPGdb, "%s_connection" % cfg['db'], psycopg2.connect(dbname=cfg['db'], user=cfg['user'], password=cfg['passwd'], host=cfg['host'],
                                                                           port=cfg['port'], sslmode='verify-full', sslrootcert='%s/.pgsql/root.crt' % CURDIR))
        else:
            if ',' in cfg['host']:
                for host in cfg['host'].split(','):
                    try:
                        pg = psycopg2.connect(dbname=cfg['db'], user=cfg['user'], password=cfg['passwd'], host=host, port=cfg['port'],
                                              sslmode='verify-full', sslrootcert='%s/.pgsql/root.crt' % CURDIR)
                        pg_cursor = pg.cursor()
                        pg_cursor.execute("SELECT pg_is_in_recovery()")
                        res = pg_cursor.fetchone()
                        pg_cursor.close()
                        if res and (mode == 'read-write' and not res[0] or mode != 'read-write' and res[0]):
                            return pg
                    except Exception, e:
                        writelog("getPGdb exception: %s" % str(e), True)
                        continue
            else:
                setattr(getPGdb, "%s_connection" % cfg['db'], psycopg2.connect(dbname=cfg['db'], user=cfg['user'], password=cfg['passwd'], host=cfg['host'], port=cfg['port'],
                                                                               sslmode='verify-full', sslrootcert='%s/.pgsql/root.crt' % CURDIR))
    return getattr(getPGdb, "%s_connection" % cfg['db'])


def getMongoDB(cfg):
    if not hasattr(getMongoDB, "%s_connection" % cfg['db']):
        setattr(getMongoDB, "%s_connection" % cfg['db'],
                pymongo.MongoClient(host=mongo_conn_str(cfg), port=cfg['port'], connectTimeoutMS=cfg.get('timeout', 10000),
                                    socketTimeoutMS=cfg.get('timeout', 10000))[cfg['db']])
    return getattr(getMongoDB, "%s_connection" % cfg['db'])


def redis_connect(host, port, cfg):
    if 'passwd' in cfg and cfg['passwd']:
        return redis.Redis(host=host, port=port, db=cfg['db'], password=cfg['passwd'], socket_timeout=cfg.get('timeout', None))
    else:
        return redis.Redis(host=host, port=port, db=cfg['db'], socket_timeout=cfg.get('timeout', None))


def redis_reconnect(cfg):
    rediscli = None
    try:
        if 'auth' in cfg and cfg['auth']:
            getRedisCredentials(cfg)
        else:
            cfg['passwd'] = None
        if 'host' in cfg and cfg['host']:
            rediscli = redis.Redis(host=cfg['host'], port=cfg['port'], db=cfg['db'], passport=cfg['passwd'], socket_timeout=cfg.get('timeout', None))
        elif 'hosts' in cfg and cfg['hosts']:
            if 'cluster_name' in cfg and cfg['cluster_name']:
                try:
                    redis_sentinel = redis.Redis(host=cfg['hosts'][0], port=cfg.get('sentinel_port', 26379), socket_timeout=cfg.get('timeout', None))
                    host = redis_sentinel.sentinel_get_master_addr_by_name(cfg['cluster_name'])
                    rediscli = redis_connect(host[0], host[1], cfg)
                except Exception, e:
                    writelog("Exception in redis_reconnect for cluster '%s': %s" % (cfg['cluster_name'], str(e)), True)
            else:
                for host in cfg['hosts']:
                    try:
                        rediscli = redis_connect(host, cfg['port'], cfg)
                        if rediscli.info()["role"] == "master":
                            return rediscli
                    except Exception, e:
                        writelog("Exception in redis_reconnect: %s" % str(e), True)
    except Exception, e:
        writelog("Exception while redis_reconnect: %s" % str(e), True)
    return rediscli


def get_metric_value(metric):
    return metric.split(".")[2].upper()


def get_metric_parameter(metric):
    return metric.split(".")[-1].upper()


def get_table(metric):
    parts = metric.split(".")
    if len(parts) >= CFG['metric_table_parts']:
        return "_".join(parts[0:CFG['metric_table_parts']])
    return "common"


def get_metric_hash(metric):
    return crc32(metric) & 0xFFFFFFFF


def get_weekstart(timestamp):
    dt = datetime.fromtimestamp(timestamp)
    weeks = (dt - CFG['datetime_start']).days / 7
    return CFG['datetime_start'] + timedelta(days=weeks * 7)


def get_field_cluster(data, parameter):
    R1, R2, R4, R8, R127, R256, cmpl_spam_nopf, cmpl_ham_nopf, cmpl_spam, cmpl_ham = data
    if parameter == "HAM":
        return R1 + R2
    elif parameter == "HAMPF":
        return R8
    elif parameter == "HAMALL":
        return R1 + R2 + R8
    elif parameter == "SPAM":
        return R4
    elif parameter == "SPAMPF":
        return R127
    elif parameter == "SPAMALL":
        return R4 + R127
    elif parameter == "MALIC":
        return R256
    elif parameter == "CMPLSPAM" or parameter == "CMPL_SPAM_NOPF":
        return cmpl_spam_nopf
    elif parameter == "CMPLHAM" or parameter == "CMPL_HAM_NOPF":
        return cmpl_ham_nopf
    elif parameter == "CMPLSPAMPF" or parameter == "CMPL_SPAM":
        return cmpl_spam
    elif parameter == "CMPLHAMPF" or parameter == "CMPL_HAM":
        return cmpl_ham
    elif parameter == "TOTAL":
        return R1 + R2 + R4 + R8 + R127 + R256


def get_field_shingler(data, parameter):
    R1, R4, R256 = data
    if parameter == "HAM":
        return R1
    elif parameter == "SPAM":
        return R4
    elif parameter == "MALIC":
        return R256
    elif parameter == "TOTAL":
        return R1 + R4


def get_field_passport(data, parameter):
    R1, R4, R256 = data
    if parameter == "HAM":
        return R1
    elif parameter == "SPAM":
        return R4 - R256
    elif parameter == "MALIC":
        return R256
    elif parameter == "TOTAL":
        return R1 + R4


def getField(table, data, parameter):
    tbl = table.lower()
    if tbl == "rules_sopassport":
        return get_field_passport(data, parameter)
    elif tbl.startswith("rules_so") or tbl.startswith("detailed_rules_so"):
        return get_field_shingler(data, parameter)
    else:
        return get_field_cluster(data, parameter)


def get_table_fields_cluster(aggregation):
    return ("R1", "R2", "R4", "R8", "R127", "R256", "cmpl_spam_nopf", "cmpl_ham_nopf", "cmpl_spam", "cmpl_ham")


def get_table_fields_shingler(aggregation):
    return ("R1", "R4", "R256")


def get_table_fields(table, aggregation):
    tbl = table.lower()
    if tbl.startswith("rules_so") or tbl.startswith("detailed_rules_so"):
        return get_table_fields_shingler(aggregation)
    else:
        return get_table_fields_cluster(aggregation)
