#!/usr/bin/env python
# encoding

import os, os.path, sys
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))
sys.path.insert(0, 'WORKING_DIR')
import re, json, cgi
import glob
import uuid
import time
import math
import base64
import importlib
import psycopg2
import numpy as np
from scipy.signal import savgol_filter, medfilt, wiener
from urllib import unquote
from collections import defaultdict
from datetime import datetime, timedelta
from jinja2 import Environment, FileSystemLoader
from plotnik_common import *

global plugins, writelog

GRAPH_POINTS = 300
ERROR_LOG = "%s/plotnik-ui.log" % CFG['log_dir']

plugins, writelog = {}, writeLog(ERROR_LOG)

def jsonify(obj):
    return json.dumps(obj).replace("'", "\\'")

def timestamp(s):
    try:
        return int(time.mktime(datetime.strptime(s, "%Y-%m-%d %H:%M").timetuple()))
    except:
        return int(time.time())

def todayMidnight():
    return timestamp(datetime.today().strftime("%Y-%m-%d 00:00"))

def now():
    return timestamp(datetime.today().strftime("%Y-%m-%d %H:%M"))

def parse_time(timestr):
    if is_number(timestr):
        if str(timestr).startswith("0") and timestr != 0:
            return timestamp((datetime.today() - timedelta(days = int(timestr))).strftime("%Y-%m-%d 00:00"))
        else:
            return timestamp((datetime.today() - timedelta(days = int(timestr))).strftime("%Y-%m-%d %H:%M"))
    else:
        parts = timestr.split()
        parts_time = filter(lambda s: s.find(":") >= 0, parts)
        parts_date = filter(lambda s: s.find("-") >= 0, parts)
        if not parts_time:
            parts_time = ["00:00"]
        if not parts_date:
            parts_date = [datetime.today().strftime("%Y-%m-%d")]
        return timestamp(" ".join(parts_date + parts_time))

def parameters(form, dashboard):
    if "endtime" in form:
        endtime = timestamp(form["endtime"][0].strip())
    elif "endtime" in dashboard:
        endtime = parse_time(dashboard["endtime"])
    else:
        endtime = now()
    if "starttime" in form:
        starttime = timestamp(form["starttime"][0].strip())
    elif "starttime" in dashboard:
        starttime = parse_time(dashboard["starttime"])
    else:
        starttime = todayMidnight()
    follow = "follow" in form
    return starttime, endtime, follow

def load_plugins(folder, functionname):
    filenames = map(os.path.basename, glob.glob("%s/%s/*.py" % (os.path.dirname(__file__), folder)))
    plugin_functions = {}
    for filename in filenames:
        if filename.startswith("_"):
            continue
        modulename = filename.replace(".py", "")
        metric = modulename.replace("_", ".")
        module = importlib.import_module("%s.%s" % (folder, modulename))
        plugin_functions[metric] = getattr(module, functionname)
    return plugin_functions

def create_tables():
    db = getPGdb(PG_PLOTNIK)
    cursor = db.cursor()
    try:
        cursor.execute("""CREATE TABLE dashboard (
                                name varchar(200) NOT NULL,
                                info TEXT,
                                PRIMARY KEY(name)
                           ) %s""" % ("ENGINE=INNODB" if db == "mysql" else ""))
    except psycopg2.DatabaseError, e:
        cursor.close()
        db.rollback()
        if e.pgcode != '42P07': # duplicate_table
            raise
    else:
        cursor.close()
        db.commit()

def get_monthstart(timestamp):
    dt = datetime.fromtimestamp(timestamp)
    return (dt - timedelta(days = dt.day - 1)).replace(hour = 0, minute = 0, second = 0, microsecond = 0)

def get_timestamp_range(starttime, endtime, follow, graph_points, min_period):
    if follow:
        endtime = int(time.time())
    if min_period == 2592000:
        date_start = get_monthstart(starttime)
        date_end = get_monthstart(endtime)
        while date_start <= date_end:
            if int(date_start.strftime("%s")) >= starttime:
                yield date_start
            date_start += timedelta(days = 31)
            date_start = get_monthstart(int(date_start.strftime("%s")))
    elif min_period == 604800:
        date_start = get_weekstart(starttime)
        date_end = get_weekstart(endtime)
        while date_start <= date_end:
            if int(date_start.strftime("%s")) >= starttime:
                yield date_start
            date_start += timedelta(days = 7)
    elif min_period == 86400:
        date_start = datetime.fromtimestamp(starttime).replace(hour = 0, minute = 0, second = 0, microsecond = 0)
        date_end = datetime.fromtimestamp(endtime).replace(hour = 0, minute = 0, second = 0, microsecond = 0)
        for days in xrange((date_end - date_start).days + 1):
            yield (date_start + timedelta(days = days))
    else:
        coeff = max(min_period, int((endtime - starttime) / graph_points))
        for timestamp in xrange(int(starttime / coeff) * coeff, endtime + 1, coeff):
            yield datetime.fromtimestamp(timestamp)

def get_graph_data(metric, graph):
    global writelog
    metric = metric.lower()
    db = getPGdb(PG_PLOTNIK)
    graph['table'] = get_table(metric)
    graph['metric_hash'] = get_metric_hash(metric)
    if 'follow' not in graph or graph['follow']:
        graph['endtime'] = int(time.time())
    coeff = max(graph['min_period'], int((graph['endtime'] - graph['starttime']) / graph['graph_points'])) if graph['graph_points'] > 0 else graph['min_period']
    data_db = {}
    try:
        cursor = db.cursor()
        if graph['min_period'] == 86400:
            graph['groupby'] = "EXTRACT(EPOCH FROM DATE_TRUNC('DAY', TO_TIMESTAMP(timestamp)))"
        elif graph['min_period'] == 604800:
            graph['groupby'] = "EXTRACT(EPOCH FROM DATE_TRUNC('WEEK', TO_TIMESTAMP(timestamp)))"
        elif graph['min_period'] == 2592000:
            graph['groupby'] = "EXTRACT(EPOCH FROM DATE_TRUNC('MONTH', TO_TIMESTAMP(timestamp)))"
        else:
            graph['groupby'] = "(floor(timestamp / {0}) * {0})".format(coeff)
        query = """SELECT %(groupby)s AS ts, %(aggregation)s(value) FROM %(table)s
                   WHERE timestamp >= %(starttime)s AND timestamp <= %(endtime)s AND metric = %(metric_hash)s GROUP BY %(groupby)s""" % graph
        cursor.execute(query)
        for rec in cursor:
            data_db[str(int(rec[0]))] = rec[1]
    except psycopg2.DatabaseError, e:
        cursor.close()
        if e.pgcode != '42P01': # undefined_table
            writelog("PGaaS exception (code=%s): %s.\nQuery: %s" % (e.pgcode, str(e), query), True)
            raise
    except Exception, e:
        writelog("Error for metric %s: %s. Graph: %s. Query: %s" % (metric, str(e), str(graph), query), True)
    else:
        cursor.close()
    data = {}
    for dt in get_timestamp_range(graph['starttime'], graph['endtime'], graph['follow'], graph['graph_points'], graph['min_period']):
        ts = str(int(dt.strftime("%s")))
        if ts in data_db:
            data[datetime.fromtimestamp(int(ts))] = data_db[ts]
        elif ts not in data_db and graph['zerofill']:
            data[datetime.fromtimestamp(int(ts))] = 0
    return data

def is_number(value):
    try:
        float(value)
        return True
    except:
        return False

def get_number_data(value, graph):
    data = {}
    for dt in get_timestamp_range(graph['starttime'], graph['endtime'], graph['follow'], graph['graph_points'], graph['min_period']):
        data[dt] = float(value)
    return data

def convert_dict_to_graphlist(all_data):
    ret = []
    for (date, value) in sorted(all_data.iteritems()):
        value["date"] = date
        ret.append(dict(value))
    return ret

def find_metric_plugin(metric):
    global plugins
    for metricbase, function in plugins.iteritems():
        if metric.startswith(metricbase):
            return function
    return None

def saveop(obj1, obj2, op):
    try:
        return getattr(float(obj1), op)(obj2)
    except:
        return 0.0

def saveop(obj1, obj2, op):
    try:
        return getattr(float(obj1), op)(obj2)
    except:
        return 0.0

def combine(dict1, dict2):
    res = {}
    for key in set(dict1.keys()) & set(dict2.keys()):
        res[key] = (dict1[key], dict2[key])
    return res

class MathDict(dict):
    def mathop(self, obj, op):
        if isinstance(obj, (int, long, float, complex)):
            return MathDict(dict(map(lambda (key, elem): (key, saveop(elem, obj, op)), self.iteritems())))
        else:
            return MathDict(map(lambda (key, (elem1, elem2)): (key, saveop(elem1, elem2, op)), combine(self, obj).iteritems()))

    def __div__(self, obj):
        return self.mathop(obj, "__div__")

    def __mul__(self, obj):
        return self.mathop(obj, "__mul__")

    def __add__(self, obj):
        return self.mathop(obj, "__add__")

    def __sub__(self, obj):
        return self.mathop(obj, "__sub__")

    def __floordiv__(self, obj):
        return self.mathop(obj, "__floordiv__")

    def __mod__(self, obj):
        return self.mathop(obj, "__mod__")

    def __pow__(self, obj):
        return self.mathop(obj, "__pow__")

    def __neg__(self):
        return self.mathop(-1.0, "__mul__")

    def __pos__(self):
        return self.mathop(1.0, "__mul__")

    def filter(self, boundary):
        return dict(map(lambda (key, elem): (key, min(boundary, elem)), self.iteritems()))

def round_sigdigits(num, digits, maxdigits):
    if num == 0:
        return 0
    digits_round = -int(math.floor(math.log10(abs(num))) - (digits - 1))
    return round(num, min(digits_round, maxdigits))

def smart_round(num, digits, maxdigits = 6):
    absnum = abs(num)
    sign = num / absnum if absnum != 0 else 1
    intnum = int(absnum)
    fraqnum = absnum - intnum
    ret = sign * (intnum + round_sigdigits(fraqnum, digits, maxdigits))
    return ret

def do_regression(data, regression_type):
    global writelog
    N, rdata, x, y, smoothing_type, smoothing_window = 1, {}, [], [], '', -1
    if regression_type.startswith('constant') or regression_type.startswith('zero_degree'):
        N = 0
    elif regression_type.startswith('linear') or regression_type.startswith('1st_degree'):
        N = 1
    elif regression_type.startswith('quadratic') or regression_type.startswith('2nd_degree'):
        N = 2
    elif regression_type.startswith('cubic') or regression_type.startswith('3rd_degree'):
        N = 3
    else:
        m = re.match(r'(\d+)(?:th)?_degree', regression_type)
        if m:
            try:
                N = int(m.group(1))
            except:
                pass
    m = re.search(r'_(savgol|medfilt|wiener)_?(\d+)', regression_type)
    if m:
        try:
            smoothing_type = m.group(1)
            smoothing_window = ((int(m.group(2)) if int(m.group(2)) > 0 else 15) // 2) * 2 + 1
        except:
            pass
    writelog("Regression: degree=%s, smoothType='%s', smoothWindow=%s" % (str(N), smoothing_type, smoothing_window))
    miny = maxy = 0.0
    for (xi, yi) in sorted(data.iteritems(), key=lambda el: el[0]):
        if len(x) == 0:
            miny = maxy = yi
        if miny > yi: miny = yi
        if maxy < yi: maxy = yi
        x.append(xi)
        y.append(yi)
    n, mx, my = len(x), (x[0] + x[-1]) * 1.0 / 2, (miny + maxy) * 1.0 / 2
    for i in range(n):
        x[i] -= mx
        y[i] -= my
    try:
        if smoothing_type == 'savgol':
            y = list(savgol_filter(np.array(y), smoothing_window, N))
        elif smoothing_type == 'medfilt':
            y = list(medfilt(np.array(y), smoothing_window))
        elif smoothing_type == 'wiener':
            y = list(wiener(np.array(y), smoothing_window))
        else:
            a = []
            for i in range(n):
                if len(a) < i + 1:
                    a.append([])
                a[i].append(1)
                for j in range(1, N + 1):
                    a[i].append(a[i][j - 1] * x[i])
            A = np.array(a)
            Y = np.array(y)
            result = np.linalg.lstsq(A, y)
            b = result[0]
            writelog("Regression of %d degree: rank of matrix = %s" % (N, result[2]))
            if len(b) > 0:
                writelog("Regression output coefficients: %s" % map(lambda j: b[j], range(N+1)))
                for i in range(n):
                    yi = 0
                    for j in range(N + 1):
                        yi += a[i][j] * b[j]
                    y[i] = yi
    except Exception, e:
        writelog("Error while solving regression equation: %s" % str(e), True)
    for i in range(n):
        rdata[datetime.fromtimestamp(mx + x[i])] = my + y[i]
    return rdata

def get_graphs_data(metric_expressions, graph):
    all_data = defaultdict(lambda: defaultdict(float))
    for i, metric_expression in enumerate(metric_expressions):
        metrics = re.findall("[\w\.]+", metric_expression)
        m = {}
        eval_expression = metric_expression
        for metric in set(metrics):
            function = find_metric_plugin(metric)
            if function:
                data = function(metric, graph)
            elif is_number(metric):
                data = get_number_data(metric, graph)
            else:
                data = get_graph_data(metric, graph)
            m[metric] = MathDict(data)
            eval_expression = re.sub("(%s)([^\w\.]|$)" % re.escape(metric), "m[ '%s' ]\\2" % metric, eval_expression)
        data = eval("%s" % eval_expression)
        if graph['regression'] and i in graph['regression']:
            data = do_regression(dict(map(lambda (d, v): (int(time.mktime(d.timetuple())), v), data.iteritems())), graph['regression'][i])
        for timestamp, value in data.iteritems():
            if value is not None:
                all_data[timestamp.strftime("%Y/%m/%d %H:%M:%S")]["graph%s" % (i + 1)] = smart_round(value, graph['sign_digits'], graph['max_frac_digits'])
    return convert_dict_to_graphlist(all_data)

def get_dashboards_names(db, showhidden):
    cursor = db.cursor()
    cursor.execute("SELECT name, info FROM dashboard")
    data = cursor.fetchall()
    cursor.close()
    res = []
    for elem in data:
        if (not showhidden) and elem[0][:5] in (u"temp_", u"edit_"):
            continue
        try:
            title = json.loads(elem[1])["title"]
        except:
            title = ""
        res.append((elem[0], title))
    return res

def get_dashboard_info_raw(db, name):
    cursor = db.cursor()
    cursor.execute("""SELECT info FROM dashboard WHERE name = '%(name)s'""" % locals())
    data = cursor.fetchone()
    cursor.close()
    return data[0] if data else ""

def get_dashboard_info(db, name):
    info = get_dashboard_info_raw(db, name)
    return json.loads(info) if info else {}

def set_dashboard_info(db, name, info):
    if not json.loads(info):
        return
    cursor = db.cursor()
    cursor.execute("INSERT INTO dashboard (name, info) VALUES(%s, %s) ON CONFLICT (name) DO UPDATE SET info = EXCLUDED.info", (name, info))
    cursor.close()
    db.commit()

def dashboard_copy(db, name, newname):
    info = get_dashboard_info_raw(db, name)
    if info:
        set_dashboard_info(db, newname, info)

def dashboard_delete(db, name):
    cursor = db.cursor()
    cursor.execute("""DELETE FROM dashboard WHERE name='%(name)s'""" % locals())
    cursor.close()
    db.commit()

def dashboard_original_name(name):
    if not name.startswith("edit_"):
        return name
    return name[5:name.rfind("_")]

def generate_dashboard_name():
    dt = datetime.today()
    return "temp_%d%02d%02d_%s" % (dt.year, dt.month, dt.day, uuid.uuid4())

def generate_dashboard_editname(name):
    dt = datetime.today()
    return "edit_%s_%s" % (name, uuid.uuid4())

def parse_minperiod(period_str):
    period_str = str(period_str).lower()
    if period_str == "minute":
        return 60
    elif period_str == "hour":
        return 3600
    elif period_str == "day":
        return 86400
    elif period_str == "week":
        return 604800
    elif period_str == "month":
        return 2592000
    else:
        return int(period_str)

def base64_urldecode(s):
    return base64.urlsafe_b64decode(s.ljust(((len(s) + 3) / 4) * 4, "="))

def dashboard(params):
    global writelog
    if "name" not in params and "content" not in params:
        return "No dashboard name specified"
    output, dashboard_info = "", None
    readonly = "readonly" in params
    hideTimeCtrls = "hideTimeCtrls" in params and params["hideTimeCtrls"][0] == "1"
    try:
        if "content" in params:
            dashboard_content = params["content"][0]
            dashboard_info = json.loads(base64_urldecode(str(dashboard_content)))
            readonly = True
        else:
            dashboard_name = params["name"][0]
            dashboard_info = get_dashboard_info(getPGdb(PG_PLOTNIK), dashboard_name)
        starttime, endtime, follow = parameters(params, dashboard_info)
        charts, regression = [], {}
        if "charts" in dashboard_info:
            for chart in dashboard_info["charts"]:
                if "graphs" not in chart:
                    continue
                for i, r in enumerate(chart["graphs"]):
                    if 'regression' in r:
                        regression[i] = r['regression']
                    r['graph_name'] = "graph%s" % (i + 1)
                graph_params = {
                    'starttime':       starttime,
                    'endtime':         endtime,
                    'follow':          follow,
                    'graph_points':    int(chart["max_graph_points"]) if "max_graph_points" in chart else GRAPH_POINTS,
                    'min_period':      parse_minperiod(chart["minperiod"]) if "minperiod" in chart else 60,
                    'aggregation':     chart["aggregation"] if "aggregation" in chart else "AVG",
                    'sign_digits':     int(chart["signdigits"]) if "signdigits" in chart else 3,
                    'max_frac_digits': int(chart["maxfracdigits"]) if "maxfracdigits" in chart else 6,
                    'zerofill':        False if "zerofill" in chart and not chart["zerofill"] else True,
                    'regression':      regression if regression else None
                }
                chart["data"] = get_graphs_data(map(lambda d: d["metric"], chart["graphs"]), graph_params)
                charts.append(chart)
    except Exception, e:
        writelog("Error while retrieving dashboard data: %s" % str(e), True)
    try:
        env = Environment(loader = FileSystemLoader("WORKING_DIR/so-plotnik-ui/"))
        env.filters["jsonify"] = jsonify
        template = env.get_template("chart.template.html")
        output = template.render(**locals()).encode("utf-8")
    except Exception, e:
        writelog("Exception while rendering html-page: %s" % str(e), True)
    return output

def application(environ, start_response):
    global writelog
    if not hasattr(application, "init"):
        application.init = True
        getPGCredentials(PG_PLOTNIK)
        getPGCredentials(PG_FRODO)
        loadMongoDbCredentials(MONGO_RULES)
        create_tables()
        plugins.update(load_plugins(CFG['plugins_folder'], "get_graph_data"))
    request_method, post_params = environ.get("REQUEST_METHOD", ""), {}
    params = dict(cgi.parse_qs(environ['QUERY_STRING'], keep_blank_values = False))
    query = re.sub(r'^/plotnik/?([^\s\?]+)?.*$', r'\1', environ['PATH_INFO'])
    #if request_method == "POST" and environ['wsgi.input']:
    #    post_params = cgi.parse_multipart(environ['wsgi.input'], {'Content-Length': int(environ.get('CONTENT_LENGTH', 0))})

    if query == "dashboard":
        start_response("200 OK", [("Content-type", "text/html")])
        return [dashboard(params)]

    elif query == "dashboard/new" or not query:
        params['name'] = [generate_dashboard_name()]
        start_response("200 OK", [("Content-type", "text/html")])
        return [dashboard(params)]

    elif query == "dashboard/saveas":
        if "name" not in params or "newname" not in params:
            start_response("400 Bad Request", [])
        dashboard_name = params["name"][0]
        dashboard_newname = params["newname"][0]
        db = getPGdb(PG_PLOTNIK)
        dashboard_copy(db, dashboard_name, dashboard_newname)
        if dashboard_name.startswith("edit_") or dashboard_name.startswith("temp_"):
            dashboard_delete(db, dashboard_name)
        start_response("200 OK", [("Content-type", "text/html")])
        return [dashboard_newname]

    elif query == "dashboard/save":
        if "name" not in params:
            start_response("400 Bad Request", [])
        dashboard_name = params["name"][0]
        original_name = dashboard_original_name(dashboard_name)
        if original_name and dashboard_name.startswith("edit_"):
            db = getPGdb(PG_PLOTNIK)
            dashboard_copy(db, dashboard_name, original_name)
            dashboard_delete(db, dashboard_name)
        start_response("200 OK", [("Content-type", "text/html")])
        return original_name

    elif query == "dashboard/saveedit" and request_method == "POST" and environ['wsgi.input']:
        if "name" not in params:
            start_response("400 Bad Request", [])
        dashboard_name = params["name"][0]
        try:
            if not dashboard_name.startswith("edit_") and not dashboard_name.startswith("temp_"):
                dashboard_name = generate_dashboard_editname(dashboard_name)
            info = unquote(environ['wsgi.input'].read())
            if info.startswith("info="):
                info = info[5:]
            db = getPGdb(PG_PLOTNIK)
            set_dashboard_info(db, dashboard_name, info)
        except Exception, e:
            writelog("Error in dashboard saving: %s" % str(e), True)
        start_response("200 OK", [("Content-type", "text/html")])
        return [dashboard_name]

    elif query == "dashboard/getdashboard":
        if "name" not in params:
            start_response("400 Bad Request", [])
        start_response("200 OK", [("Content-type", "text/html")])
        return [get_dashboard_info_raw(getPGdb(PG_PLOTNIK), params["name"][0])]

    elif query == "dashboard/remove":
        try:
            if "name" not in params:
                start_response("400 Bad Request", [])
            dashboard_delete(getPGdb(PG_PLOTNIK), params["name"][0])
        except Exception, e:
            writelog("Error while removing dashboard: %s" % str(e), True)
        start_response("200 OK", [("Content-type", "text/html")])
        return [""]

    elif query == "getdashboards" or query == "dashboard/getdashboards":
        showhidden = "showhidden" in params and params["showhidden"][0] == "1"
        start_response("200 OK", [("Content-type", "application/json")])
        return [json.dumps(get_dashboards_names(getPGdb(PG_PLOTNIK), showhidden))]

    else:
        s, sm = '', []
        for h, v in environ.items():
            if re.match(r'[A-Z_]+', h):
                s += "%s: %s\n" % (h, v)
            else: sm.append(h)
        start_response("200 OK", [("Content-type", "text/plain")])
        return ["Env: %sOther keys: %s\nQuery: %s\n" % (s, ', '.join(sm), query)]
