
#include <array>
#include <library/cpp/charset/doccodes.h>
#include <library/cpp/charset/codepage.h>
#include "context_symbols.h"

using TContextSymbolsTable = std::array<TSpSymbols, 256>;

static const auto codePage = CodePageByCharset(CODES_KOI8);


static TContextSymbolsTable HTMLSymbolsTable = [] {
    TContextSymbolsTable table{};
    table.fill(SP_NULL);

    for (size_t i = 0; i < 256; i++) {
        if (codePage->IsAlpha(i)) {
            table[i] = SP_LETTER;
        } else if (codePage->IsDigit(i)) {
            table[i] = SP_DIGIT;
        }
    }

    table[0x00] = SP_NULL;
    table[0x09] = SP_SPACE;
    table[0x0A] = SP_SPACE;
    table[0x0B] = SP_SPACE;
    table[0x0D] = SP_SPACE;
    table[0x20] = SP_SPACE;
    table[0xA0] = SP_SPACE;
    table[(ui8)('&')] = SP_AMPERSAND;
    table[(ui8)('<')] = SP_LESS;
    table[(ui8)('>')] = SP_GREAT;
    table[(ui8)('!')] = SP_EXCLAM;
    table[(ui8)('.')] = SP_DOT;
    table[(ui8)('@')] = SP_AT;
    table[(ui8)('$')] = SP_DOLLAR;
    table[(ui8)('\'')] = SP_QUOTE;
    table[(ui8)('\"')] = SP_QUOTE;
    table[(ui8)('`')] = SP_QUOTE;
    table[(ui8)('-')] = SP_SIGN;
    table[(ui8)('+')] = SP_SIGN;

    return table;
}();

static const TContextSymbolsTable CaseSymbolsTable = [] {
    TContextSymbolsTable table;
    table.fill(SP_NULL);

    for (size_t i = 0; i < 256; i++) {
        if (codePage->IsUpper(i)) {
            table[i] = SP_UPPER;
        } else if (codePage->IsLower(i)) {
            table[i] = SP_LOWER;
        } else if (codePage->IsDigit(i)) {
            table[i] = SP_DIGIT;
        } else if (i <= 32 && HTMLSymbolsTable[i] == SP_SPACE)
            table[i] = SP_SPACE;
    }
    return table;
}();

static const TContextSymbolsTable TelSymbolsTable = [] {
    TContextSymbolsTable table;
    table.fill(SP_NULL);

    table[0x20] = SP_SPACE;
    table[(ui8)('-')] = SP_SPACE;
    table[(ui8)('*')] = SP_SPACE;
    table[(ui8)('.')] = SP_SPACE;
    table[(ui8)('(')] = SP_SPACE;
    table[(ui8)(')')] = SP_SPACE;
    table[(ui8)('[')] = SP_SPACE;
    table[(ui8)(']')] = SP_SPACE;
    table[(ui8)('+')] = SP_SPACE;
    table[(ui8)(':')] = SP_SPACE;


    {
#ifdef WIN32 // cp1251
        ui8 Z_R = 0xc7;
        ui8 O_R = 0xce;
        ui8 CHE_R = 0xd7;
        ui8 SOFT_SIGN_R = 0xdc;
        ui8 B_R = 0xe1;
#else // koi8-r
        ui8 Z_R = 0xfa;
        ui8 O_R = 0xef;
        ui8 CHE_R = 0xfe;
        ui8 SOFT_SIGN_R = 0xf8;
        ui8 B_R = 0xc2;
#endif
        table[Z_R] = SP_DIGIT_FORGED;
        table[O_R] = SP_DIGIT_FORGED;
        table[CHE_R] = SP_DIGIT_FORGED;
        table[SOFT_SIGN_R] = SP_DIGIT_FORGED;
        table[B_R] = SP_DIGIT_FORGED;
    }

    table[(ui8)('O')] = SP_DIGIT_FORGED;
    table[(ui8)('I')] = SP_DIGIT_FORGED;
    table[(ui8)('l')] = SP_DIGIT_FORGED;
    table[(ui8)('Z')] = SP_DIGIT_FORGED;
    table[(ui8)('S')] = SP_DIGIT_FORGED;
    table[(ui8)('b')] = SP_DIGIT_FORGED;

    for (size_t i = 0; i < 256; i++) {
        if (codePage->IsDigit(i))
            table[i] = SP_DIGIT;
    }

    const char* forged_spaces = "=?|_~'#^$`!\"%&@\xB7"; // (B7 = '�')

    for (; *forged_spaces; forged_spaces++)
        table[(ui8)(*forged_spaces)] = SP_SPACE_FORGED;

    return table;
}();

static const TContextSymbolsTable UrlSymbolsTable = []{

    TContextSymbolsTable table;
    table.fill(SP_NULL);

    for (size_t i = 0; i < 256; i++) {
        if (codePage->IsAlpha(i)) {
            table[i] = SP_LAT_LETTER;
        } else if (codePage->IsDigit(i)) {
            table[i] = SP_DIGIT;
        }
    }

    return table;
}();

TSpSymbols GetHTMLSymbols(char c) {
    return HTMLSymbolsTable[ui8(c)];
}

TSpSymbols GetCaseSymbols(char c) {
    return CaseSymbolsTable[ui8(c)];
}

TSpSymbols GetTelSymbols(char c) {
    return TelSymbolsTable[ui8(c)];
}

TSpSymbols GetUrlSymbols(char c) {
    return UrlSymbolsTable[ui8(c)];
}


bool TestHtmlSymbol(char c, TSpSymbols s) {
    return GetHTMLSymbols(c) == s;
}

bool TestCaseSymbol(char c, TSpSymbols s) {
    return GetCaseSymbols(c) == s;
}

bool TestTelSymbol(char c, TSpSymbols s) {
    return GetTelSymbols(c) == s;
}

bool TestUrlSymbol(char c, TSpSymbols s) {
    return GetUrlSymbols(c) == s;
}
