#pragma once

#include <util/generic/fwd.h>
#include <util/stream/fwd.h>
#include <util/generic/string.h>
#include <util/generic/maybe.h>
#include <util/generic/hash_set.h>
#include <util/generic/vector.h>
#include <library/cpp/logger/log.h>
#include <mail/so/spamstop/tools/so-common/exp_boxes.h>

namespace NConfig{
    class TConfig;
    struct TDict;
}

class TFsPath;

struct TModelConfig{
    enum EModelType{
        MATRIXNET /* "matrixnet" */,
        CATBOOST /* "catboost" */,
    };

    friend IOutputStream& operator<<(IOutputStream& stream, const TModelConfig & model);
    operator size_t() const;
    const TMaybe<ui64>& GetFormulaId() const;
    bool operator==(const TModelConfig& config) const;
    explicit TModelConfig(const NConfig::TConfig& config);

    bool AcceptExperimentId(const TExpBoxes& expBoxes) const;

    class TMapper {
    public:
        TVector<std::pair<TString, double>> Map(const TLog& logger, const TVector<double>& values) const;

        static TMapper FromConfig(const NConfig::TConfig& config);

    private:
        TVector<std::pair<size_t, TString>> IndexToFeatureMap;
    };

    TString slot;
    TString briefName;
    TString binaryFile;
    TString featuresFile;
    ui64 resourceId{};
    double threshold{};
    TMaybe<ui64> formulaId;
    EModelType modelType;

    THashSet<ui64> slaveModelsIds;

    TMaybe<TMapper> OneVsAllMapper;

    size_t Priority{};

private:
    THashSet<uint64_t> AvailableExperiments;
};

class IModelApplier{
public:
    using TFeaturesMap = THashMap<TString, float>;

    struct TDict {
        friend IOutputStream& operator<<(IOutputStream& stream, const TDict& dict);

        /**
         * @brief: Apply maps in theirs order: if some feature was found in a map, search of the feature stops
         */
        TVector<float> MakeFeatures(std::initializer_list<std::reference_wrapper<const TFeaturesMap>> featuresMaps) const;

        size_t maxIndex{};
        THashMap<TString, size_t> featuresIndexesMap;
    };

    explicit IModelApplier(TDict dict) noexcept : dict(std::move(dict)) {}

    virtual TVector<double> Apply(const TVector<float>& features) const = 0;
    virtual ~IModelApplier() = default;

    const TDict dict;
};

class TFullModel;
IModelApplier::TDict LoadCatboostDict(IInputStream& stream, size_t numFloatFeatures);
IModelApplier::TDict LoadMatrixnetDict(IInputStream& stream);

using TAppliersMap = TVector<std::pair<TModelConfig, TSimpleSharedPtr <IModelApplier>>>;
TAppliersMap LoadAppliers(const TFsPath& workingDir, const NConfig::TConfig& config);

