#include <vector>
#include <algorithm>
#include <utility>

#include <util/generic/scope.h>
#include <util/generic/iterator_range.h>
#include <util/stream/file.h>
#include <util/string/cast.h>
#include <util/stream/tee.h>
#include <util/folder/filelist.h>
#include <util/folder/path.h>
#include <util/string/subst.h>
#include <util/string/join.h>
#include <util/string/split.h>
#include <util/system/hostname.h>
#include <util/string/strip.h>
#include <util/charset/utf8.h>

#include <dict/dictutil/streams.h>

#include <kernel/lemmer/core/lemmer.h>
#include <kernel/lemmer/core/language.h>
#include <library/cpp/unicode/normalization/normalization.h>
#include <library/cpp/threading/future/async.h>
#include <library/cpp/iterator/functools.h>
#include <library/cpp/shingles/shingler.h>
#include <library/cpp/digest/lower_case/lchash.h>
#include <library/cpp/charset/recyr.hh>

#include <mail/so/spamstop/tools/so-clients/cmpl_flags.h>
#include <mail/so/spamstop/tools/so-common/parsers.h>
#include <mail/so/spamstop/tools/so-common/sputil.h>
#include <mail/so/spamstop/tools/so-common/get_zone_time.h>
#include <mail/so/spamstop/tools/so-common/so_log.h>
#include <mail/so/spamstop/tools/so-common/receipt.h>
#include <mail/so/spamstop/tools/so-common/multicounter.h>
#include <mail/so/spamstop/tools/text2shingles/lib/text2shingles.h>
#include <mail/so/spamstop/tools/so-common/safe_recode.h>
#include <mail/so/spamstop/tools/so-common/so_answer.h>
#include <mail/so/spamstop/tools/so-common/exp_boxes.h>

#include <mail/so/libs/html_sanitizer_misc/html_sanitizer_misc.h>
#include <mail/so/libs/protect/protect.h>

#include <contrib/libs/cld2/compact_lang_det.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/config/config.h>
#include <library/cpp/uri/parse.h>
#include <kernel/dssm_applier/nn_applier/lib/states.h>
#include <mlp/mail/aspam/user_factors/lib/user_factors.h>
#include <util/datetime/cputimer.h>
#include <cstdlib>
#include <bitset>

#include "spamstop.h"
#include "sptypes.h"
#include "arform.h"
#include "setlistrule.h"
#include "setrules.h"
#include "sphtml.h"
#include "spalg.h"
#include "spbody.h"
#include "spstat.h"
#include "spruler.h"
#include "dkim_signature.h"
#include "chars.h"
#include "shingles.h"
#include "rengine.h"
#include <library/cpp/svnversion/svnversion.h>

static bool GoodForBayes(const TTextSplitter::TSplitResult& res);

#define SHINGLE_MAX 1111

typedef std::pair<ui64, TShHttpType> SHPAIR;

static constexpr size_t MaxBayesWords = 100;
static const TTrueConst<NRegexp::TPcre> RE_RNRN{"\n\r?\n", NRegexp::TSettings{}};

struct TSO_Classification {
    TStringBuf rule_name;
    TStringBuf class_name;
};

// Property descriptions
static const auto so_class = MakeTrueConstArray(
        TSO_Classification{"CL_NOTIFICATION",     "notification"},
        TSO_Classification{"CL_REGISTRATION",     "registration"},
        TSO_Classification{"CL_PEOPLE",           "people"},
        TSO_Classification{"CL_ETICKET",          "eticket"},
        TSO_Classification{"CL_ESHOP",            "eshop"},
        TSO_Classification{"CL_DISCOUNT",         "discount"},
        TSO_Classification{"CL_BOUNCE",           "bounce"},
        TSO_Classification{"CL_SCRIPT",           "script"},
        TSO_Classification{"CL_JOB",              "job"},
        TSO_Classification{"CL_NEWS",             "news"},
        TSO_Classification{"CL_SCHEMA",           "schema"},
        TSO_Classification{"CL_CANCEL",           "cancel"},
        TSO_Classification{"CL_HOTELBOOK",        "hotel"},
        TSO_Classification{"CL_DELIVERY",         "delivery"},
        TSO_Classification{"CL_INVITE",           "invite"},
        TSO_Classification{"CL_YAMONEY",          "yamoney"},
        TSO_Classification{"CL_FIRSTMAIL",        "firstmail"},

        TSO_Classification{"CL_SNDR_SOCIAL",      "s_social"},
        TSO_Classification{"CL_SNDR_AVIAETICKET", "s_aviaeticket"},
        TSO_Classification{"CL_SNDR_DATING",      "s_datingsite"},
        TSO_Classification{"CL_SNDR_COUPON",      "s_grouponsite"},
        TSO_Classification{"CL_SNDR_TRAVEL",      "s_travel"},
        TSO_Classification{"CL_SNDR_BANK",        "s_bank"},
        TSO_Classification{"CL_SNDR_ZDTICKET",    "s_zdticket"},
        TSO_Classification{"CL_SNDR_REALTY",      "s_realty"},
        TSO_Classification{"CL_SNDR_ESHOP",       "s_eshop"},
        TSO_Classification{"CL_SNDR_COMPANY",     "s_company"},
        TSO_Classification{"CL_SNDR_JOB",         "s_job"},
        TSO_Classification{"CL_SNDR_GAME",        "s_game"},
        TSO_Classification{"CL_SNDR_ADVERT",      "s_advert"},
        TSO_Classification{"CL_SNDR_PROVIDER",    "s_provider"},
        TSO_Classification{"CL_SNDR_MEDIA",       "s_media"},
        TSO_Classification{"CL_SNDR_TECH",        "s_tech"},
        TSO_Classification{"CL_SNDR_FORUM",       "s_forum"},
        TSO_Classification{"CL_SNDR_MOBILE",      "s_mobile"},
        TSO_Classification{"CL_SNDR_TRAINING",    "s_training"},
        TSO_Classification{"CL_SNDR_TRACKER",     "s_tracker"},
        TSO_Classification{"CL_SNDR_SENDER",      "s_sender"},
        TSO_Classification{"CL_SNDR_TAXI",        "s_taxi"},
        TSO_Classification{"CL_SNDR_DELIVERY",    "s_delivery"},
        TSO_Classification{"CL_SNDR_STATE",       "s_state"},
        TSO_Classification{"CL_SNDR_EVENT",       "s_event"},
        TSO_Classification{"CL_SNDR_NEWS",        "s_news"},

        TSO_Classification{"CL_EDOC",             "edoc"},

        TSO_Classification{"CL_TRUST_1",          "trust_1"},
        TSO_Classification{"CL_TRUST_2",          "trust_2"},
        TSO_Classification{"CL_TRUST_3",          "trust_3"},
        TSO_Classification{"CL_TRUST_4",          "trust_4"},
        TSO_Classification{"CL_TRUST_5",          "trust_5"},
        TSO_Classification{"CL_TRUST_6",          "trust_6"},

        TSO_Classification{"CL_SNDR_INSUR",       "s_insurance"},
        TSO_Classification{"CL_SNDR_SEO",         "s_seo"},
        TSO_Classification{"CL_TICKET_REMIND",    "remind_tic"},
        TSO_Classification{"CL_BOARDINGPASS",     "boardingpass"},

        TSO_Classification{"CL_HAMON",            "hamon"},
        TSO_Classification{"CL_TRACKERTASK",      "trackertask"},

        TSO_Classification{"YA_OUT_SPAM",         "ignorecapcha"},
        TSO_Classification{"CL_PHISHING",         "phishing"},
        TSO_Classification{"CL_FRAUD",            "fraud"},
        TSO_Classification{"CL_VIRUS",            "virus"},

        TSO_Classification{"CL_TRANSACT",         "transact"},
        TSO_Classification{"CL_PERSONAL",         "personal"},
        TSO_Classification{"CL_PROMO",            "promo"},
        TSO_Classification{"CL_TRIGGER",          "trigger"},
        TSO_Classification{"CL_REVIEW",           "review"},
        TSO_Classification{"CL_MESSAGE",          "message"},
        TSO_Classification{"CL_BILL",             "bill"},
        TSO_Classification{"CL_AVIATICKET",       "aviaticket"},
        TSO_Classification{"CL_LOST_CART",        "lostcart"},
        TSO_Classification{"CL_TAB_SOCIAL",       "t_social"},
        TSO_Classification{"CL_TAB_NEWS",         "t_news"},
        TSO_Classification{"CL_TAB_NOTIF",        "t_notification"},
        TSO_Classification{"CL_TAB_PEOPLE",       "t_people"},

        TSO_Classification{"CL_SMARTREPLY",       "smartreply"},

        TSO_Classification{"CL_GREETING",         "greeting"},

        TSO_Classification{"CL_SNDR_YANDEX_PAY",  "s_yandex_pay"},

        TSO_Classification{"CL_QUARANTINE",       "quarantine"},

        TSO_Classification{"CL_IMPORTANT",        "hamon"},
        TSO_Classification{"CL_CORRESPOND",       "correspond"},

        TSO_Classification{"CL_RES_SPF",          "spf_fail"},
        TSO_Classification{"CL_RES_DKIM",         "dkim_fail"},
        TSO_Classification{"CL_RES_DMARC",        "dmarc_fail"},
        TSO_Classification{"CL_RES_IP_RBL",       "ip_rbl"},
        TSO_Classification{"CL_RES_URL_RBL",      "url_rbl"},
        TSO_Classification{"CL_RES_DNS_FAIL",     "dns_fail"},
        TSO_Classification{"CL_RES_RFC_FAIL",     "rfc_fail"},

        //https://st.yandex-team.ru/SODEV-2347#618ba47f83b6fd43ed541a73
        TSO_Classification{"CL_RES_RFC_FAIL_OUT", "res_rfc_fail"},
        TSO_Classification{"CL_RES_URL_RBL_OUT",  "res_url_rbl"},
        TSO_Classification{"CL_RES_BAD_KARMA",    "res_bad_karma"},
        TSO_Classification{"CL_RES_LIMITS",       "res_mail_limits"},
        TSO_Classification{"CL_RES_PDD_ADMKARMA", "res_pdd_admin_karma"},
        TSO_Classification{"CL_RES_BOUNCES",      "res_bounces"},
        TSO_Classification{"CL_RES_COMPLS",       "res_spam_compl"}
);

static constexpr size_t so_classes_count = TStdArraySize<decltype(so_class)::value_type>::size;

void TRengine::SetLogger(TSpLogger **ppSpLoggerDst, TSpLogger *pSpLoggerSrc)
{
    if (pSpLoggerSrc->IsOpen())
    {
        *ppSpLoggerDst = pSpLoggerSrc;
    }
}

class TDssmDistanceProcessor : public IRengineHtmlSanAnswerProcessor {

    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto g = Guard(prof.Prof("dssm"));
        auto record = Master.m_pstat.AddStat("dssm_distance");
        if (auto maybeDistance = answer.GetDssmDistance()) {
            float distance = *maybeDistance;
            const TString& neighbourId = answer.GetDssmNeighbourId();
            const TString& neighbourLabels = answer.GetDssmNeighbourLabels();
            Master.CheckRange("visible_dssm_distance", distance);
            Master.CheckValueAllRules(FD_VISIBLE_DSSM_DISTANCE, distance);
            Master.CheckValueAllRules(FD_VISIBLE_DSSM_NEIGHBOUR_LABELS, neighbourLabels);
            Master.m_cur->m_mapValueFactors4Matrixnet.emplace("MNF_SAMPLES_VISIBLE_DSSM_DISTANCE", distance);
            record << distance << ':' << neighbourId << ':' << SubstGlobalCopy(Strip(neighbourLabels), '\n', '/');
            Master.m_cur->MlLogBuilder.Add("visible_dssm_distance", ToString(distance), distance);
            Master.m_cur->MlLogBuilder.Add("visible_dssm_neighbour_id", neighbourId);
        }
        auto record2 = Master.m_pstat.AddStat("dssm2_distance");
        if (auto maybeDistance = answer.GetDssm2Distance()) {
            float distance = *maybeDistance;
            const TString& neighbourId = answer.GetDssm2NeighbourId();
            const TString& neighbourLabels = answer.GetDssm2NeighbourLabels();
            record2 << distance << ':' << neighbourId << ':' << SubstGlobalCopy(Strip(neighbourLabels), '\n', '/');
        }
    }
};

class TDssmSubjectDistanceProcessor : public IRengineHtmlSanAnswerProcessor {

    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto g = Guard(prof.Prof("subj_dssm"));
        if (auto maybeDistance = answer.GetDssmSubjectDistance()) {
            auto record = Master.m_pstat.AddStat("dssm_subject_distance");
            float distance = *maybeDistance;
            TString neighbourId = answer.GetDssmSubjectNeighbourId();
            TString neighbourLabels = answer.GetDssmSubjectNeighbourLabels();
            Master.CheckRange("dssm_subject_distance", distance);
            Master.CheckValueAllRules(FD_DSSM_SUBJECT_DISTANCE, distance);
            Master.CheckValueAllRules(FD_DSSM_SUBJECT_NEIGHBOUR_LABELS, neighbourLabels);
            record << distance << ':' << neighbourId << ':' << SubstGlobalCopy(Strip(neighbourLabels), '\n', '/');
        }
    }
};

class TWmdDistanceProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        if(!answer.Docs())
            return;

        auto g = Guard(prof.Prof("wmd"));
        auto record = Master.m_pstat.AddStat("wmd");
        if(auto part = answer.FindVisiblePart()) {
            if(part->pureBodyWmdDistance) {
                float distance = *part->pureBodyWmdDistance;
                TString neighbourId = part->pureBodyWmdNeighbourId;
                TString neighbourLabels = part->pureBodyWmdNeighbourLabels;
                Master.CheckRange("visible_pure_body_wmd_distance", distance);
                Master.CheckValueAllRules(FD_VISIBLE_PURE_BODY_WMD_DISTANCE, distance);
                Master.CheckValueAllRules(FD_VISIBLE_PURE_BODY_WMD_NEIGHBOUR_LABELS, neighbourLabels);
                Master.m_cur->m_mapValueFactors4Matrixnet.emplace("MNF_SAMPLES_VISIBLE_BODY_DISTANCE", distance);
                record << "vb: " << distance << ':' << neighbourId << ':' << SubstGlobalCopy(Strip(neighbourLabels), '\n', '/') << ';';
                Master.m_cur->MlLogBuilder.Add("visible_pure_body_wmd_distance", ToString(distance), distance);
                Master.m_cur->MlLogBuilder.Add("visible_pure_body_wmd_neighbour_id", neighbourId, neighbourId);
            }
            if(part->subjectWmdDistance) {
                Master.CheckRange("visible_subject_wmd_distance", *part->subjectWmdDistance);
                Master.m_cur->m_mapValueFactors4Matrixnet.emplace("MNF_SAMPLES_VISIBLE_SUBJECT_DISTANCE", *part->subjectWmdDistance);
                record << "vs: " << *part->subjectWmdDistance << ';';
            }
        }

        const NHtmlSanMisc::TDoc* minPureBodyDistanceDoc = nullptr;

        for(const auto& part : answer.Docs()) {
            if(part.pureBodyWmdDistance && *part.pureBodyWmdDistance != -1) {
                const auto d = *part.pureBodyWmdDistance;
                if(!minPureBodyDistanceDoc || *minPureBodyDistanceDoc->pureBodyWmdDistance > d)
                    minPureBodyDistanceDoc = &part;
            }
        }

        if(minPureBodyDistanceDoc) {
            Master.CheckRange("min_body_wmd_distance", *minPureBodyDistanceDoc->pureBodyWmdDistance);

            Master.m_cur->m_mapValueFactors4Matrixnet.emplace("MNF_SAMPLES_MIN_BODY_DISTANCE", *minPureBodyDistanceDoc->pureBodyWmdDistance);
            record << "mb: " << *minPureBodyDistanceDoc->pureBodyWmdDistance << ':' << minPureBodyDistanceDoc->pureBodyWmdNeighbourId << ';';
        }
    }
};

class TWmdSubjectDistanceProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        if (!answer.Docs())
            return;

        auto g = Guard(prof.Prof("subj_wmd"));
        auto part = &answer.Docs().front();
        if (part->subjectWmdDistance) {
            auto record = Master.m_pstat.AddStat("wmd_subject");
            float distance = *part->subjectWmdDistance;
            TString neighbourId = part->subjectWmdNeighbourId;
            TString neighbourLabels = part->subjectWmdNeighbourLabels;
            Master.CheckRange("subject_wmd_distance", distance);
            Master.CheckValueAllRules(FD_SUBJECT_WMD_DISTANCE, distance);
            Master.CheckValueAllRules(FD_SUBJECT_WMD_NEIGHBOUR_LABELS, neighbourLabels);
            record << distance << ':' << neighbourId << ':' << SubstGlobalCopy(Strip(neighbourLabels), '\n', '/');
        }
    }
};

class TWmdAttachNameDistanceProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        if (!answer.Docs())
            return;

        auto g = Guard(prof.Prof("attach_wmd"));
        const NHtmlSanMisc::TDoc* minAttachNameDistanceDoc = nullptr;
        for (const auto& part : answer.Docs()) {
            if (part.attachNameWmdDistance) {
                const auto d = *part.attachNameWmdDistance;
                if (d != -1 && (!minAttachNameDistanceDoc || minAttachNameDistanceDoc->attachNameWmdDistance > d)) {
                    minAttachNameDistanceDoc = &part;
                }
            }
        }

        if (minAttachNameDistanceDoc) {
            auto record = Master.m_pstat.AddStat("wmd_attach_name");
            float distance = *minAttachNameDistanceDoc->attachNameWmdDistance;
            TString neighbourId = minAttachNameDistanceDoc->attachNameWmdNeighbourId;
            TString neighbourLabels = minAttachNameDistanceDoc->attachNameWmdNeighbourLabels;
            Master.CheckRange("min_attach_name_wmd_distance", distance);
            Master.CheckValueAllRules(FD_MIN_ATTACH_NAME_WMD_DISTANCE, distance);
            Master.CheckValueAllRules(FD_MIN_ATTACH_NAME_WMD_NEIGHBOUR_LABELS, neighbourLabels);
            record << distance << ':' << neighbourId << ':' << SubstGlobalCopy(Strip(neighbourLabels), '\n', '/');
        }
    }
};

class TPersonalFilterProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    static TSpClass LastTypeToSpClassSpam(const NHtmlSanMisc::TPersonalFilter& pf) {
        return AsciiEqualsIgnoreCase(pf.LastType, "ham") ? TSpClass::HAM : TSpClass::SPAM;
    }

    static bool PersonalFilterDefined(const NHtmlSanMisc::TSender* sender) {
        return sender && sender->PersonalFilter.Defined();
    }

    void Process(Y_DECLARE_UNUSED const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto g = Guard(prof.Prof("pf"));
        {
            ui32 mailsFrom = 0;
            ui32 correspondents = 0;
            TString tabPf;
            for(const NHtmlSanMisc::TSender& sender: Master.m_cur->So2Context->GetSenders()) {
                mailsFrom += sender.AbookInfo.ReceivedReadCount;
                if (sender.AbookInfo.ReceivedReadCount)
                    correspondents++;
                if (sender.TabPf.Defined()) {
                    tabPf = sender.TabPf->LastTab;
                }
            }

            if (tabPf) {
                Master.m_cur->rulesContext.SetRule("TABPF_" + to_upper(tabPf));
            }

            Master.CheckRange("mail_accepted", mailsFrom, false);
            Master.CheckRange("correspondents", correspondents, false);
        }

        Master.m_cur->NewPf = [
                rcptattrs = &Master.m_cur->m_rcptattrs,
                m_cur = Master.m_cur.Get(),
                fWebMail = Master.Config.fWebMail,
                mlLog = &Master.m_cur->MlLogBuilder](TSpClass filter_res) {
            if(AllOf(*rcptattrs, [](const auto& p){
                return p.second.sender == nullptr;
            })) {
                return false;
            }

            const auto consensus = [rcptattrs]() -> const NHtmlSanMisc::TPersonalFilter * {
                M_RCPT_ATTRS::const_iterator it = rcptattrs->cbegin();

                if(!PersonalFilterDefined(it->second.sender)) {
                    return nullptr;
                }

                const NHtmlSanMisc::TPersonalFilter * const consensusFilter = &*it->second.sender->PersonalFilter;

                if(AllOf(std::next(it), rcptattrs->cend(), [consensusFilter](const auto& p) {
                    const NHtmlSanMisc::TSender* sender = p.second.sender;
                    return PersonalFilterDefined(sender) &&
                           consensusFilter->LastType == sender->PersonalFilter->LastType;
                })) {
                    return consensusFilter;
                } else {
                    return nullptr;
                }
            }();

            if (consensus) {
                const NHtmlSanMisc::TPersonalFilter &personalFilter = *consensus;

                if (const TSpClass lastType = LastTypeToSpClassSpam(personalFilter); lastType != filter_res) {
                    double personalCorrectScore;
                    double resultHits;
                    const TSpClass resolutionBeforePf = m_cur->m_messclass;
                    if (lastType != TSpClass::HAM) {
                        m_cur->m_messclass = TSpClass::SPAM;
                        personalCorrectScore = -(m_cur->Scores.Hits - m_cur->m_required) + 8.;
                        resultHits = m_cur->m_required + 8.;
                        m_cur->fSpamPersonalCorrect = true;
                    } else {
                        m_cur->m_messclass = TSpClass::HAM;
                        personalCorrectScore = -(m_cur->Scores.Hits - m_cur->m_required) - 8.;
                        resultHits = m_cur->m_required - 18.;
                        m_cur->fHamPersonalCorrect = true;
                    }

                    if(!fWebMail || resolutionBeforePf != TSpClass::HAM) {
                        m_cur->rulesContext.SetScore("PERSONAL_CORRECT", personalCorrectScore);
                        m_cur->Scores.Hits = resultHits;
                    }
                    m_cur->rulesContext.SetRule("PERSONAL_CORRECT");

                    if(fWebMail && resolutionBeforePf == TSpClass::HAM) {
                        for(const auto& [email, attrs] : *rcptattrs) {
                            if(const TUid* uid = attrs.RobustUid()) {
                                m_cur->UidsResolutions[*uid] = lastType;
                            }
                        }
                    }

                    m_cur->RcptsWithPF = rcptattrs->size();

                    for(const auto& [email, attrs] : *rcptattrs) {
                        if(const NHtmlSanMisc::TSender* sender = attrs.sender) {
                            if (sender->PersonalFilter.Defined()) {
                                m_cur->PersonalFilterUids.emplace(sender->Uid);
                            }
                        }
                    }

                    m_cur->Logger << (TLOG_INFO) << "PR correct " << (int) m_cur->m_messclass << ' ' << personalFilter.LastType;
                    m_cur->rulesContext.SetRule("PERSONAL_CORRECT_2");
                }
            } else {
                for(const auto& [email, attrs] : *rcptattrs) {
                    if (const NHtmlSanMisc::TSender *sender = attrs.sender; sender && sender->PersonalFilter.Defined()) {
                        m_cur->PersonalFilterUids.emplace(sender->Uid);
                        if (const TSpClass lastType = LastTypeToSpClassSpam(*sender->PersonalFilter); lastType != filter_res) {
                            m_cur->rulesContext.SetRule("PERSONAL_CORRECT_2");

                            m_cur->rulesContext.SetRule("PERSONAL_CORRECT");
                            if(const TUid* uid = attrs.RobustUid()) {
                                m_cur->UidsResolutions[*uid] = lastType;
                            }
                            m_cur->RcptsWithPF++;
                        }
                    }
                }
            }

            return true;
        };

    }
};

class TAttachMlPrinter : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;
    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto g = Guard(prof.Prof("attach_ml"));
        NJson::TJsonValue attaches;

        auto& data = Master.m_cur->lsaRequestData;

        for (const auto& doc : answer.Docs()) {
            NJson::TJsonValue& docJson = attaches.AppendValue({});
            for (const auto sentence : StringSplitter(doc.BodyText.Original).SplitBySet("\n!?.;").SkipEmpty()) {
                NJson::TJsonValue& sentenceJson = docJson.AppendValue({});
                for (auto word : NText2Shingles::Text2Shingles(sentence.Token(), LANG_UNK, true, std::numeric_limits<size_t>::max())) {
                    sentenceJson.AppendValue(word);
                    data.Add(std::move(word), NLSA::TField::Attach);
                }
            }
        }

        Master.m_cur->MlLogBuilder.Add("nnattach", TStringBuilder{} << attaches);
    }
};

class TUserFeaturesProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;
public:
    void Process(Y_DECLARE_UNUSED const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto g = Guard(prof.Prof("user_feat"));
        TProfileTimer timer;
        Y_DEFER{
            const TDuration duration = timer.Get();
            Master.m_cur->Logger << (TLOG_NOTICE) << "TUserFeaturesProcessor:" << duration;
            Master.mlTimings->PushSignal(duration.MillisecondsFloat());
        };
        for(const auto& [email, attrs] : Master.m_cur->m_rcptattrs) {
            if(attrs.sender) {
                const TString fixedFrom = FixAddress(Master.m_cur->m_sMailFrom);
                NJson::TJsonValue mlLogItem;
                try {
                    NMailUserFactors::TMailUserFactorsBuilder::Get().CreateDataFromJson(attrs.sender->RawUserMlFeatures,
                                                                                  answer.GetSenderMlFeatures(),
                                                                                  attrs.sender->RawUserMlEmbeddings,
                                                                                  Master.m_cur->TextEmbed,
                                                                                  fixedFrom,
                                                                                  Master.m_cur->m_mapValueFactors4Matrixnet,
                                                                                  mlLogItem);
                } catch (...) {
                    Master.UserFeaturesErrors->PushSignal(1);
                    Master.m_cur->Logger << (TLOG_ERR) << "user features error: " << CurrentExceptionMessageWithBt();
                }
                Master.m_cur->MlLogBuilder.Add("ml_comment", std::move(mlLogItem));
                return;
            }
        }
        Master.m_cur->Logger << (TLOG_ERR) << "cannot find senderinfo:" << ' ' << MakeRangeJoiner(",", Master.m_cur->So2Context->GetSenders());
    }
};

class TBBProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto g = Guard(prof.Prof("bb"));
        for(const auto& [email, userInfo] : answer.GetUserInfos()) {
            if(userInfo.IsMailList) {
                Master.m_cur->rulesContext.SetRule("IS_MAILLIST");
                break;
            }
        }
    }
};

class TFirstShingler : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto part = answer.FindVisiblePart();
        if (!part)
            return;

        auto g = Guard(prof.Prof("first"));
        TFilteredUTF8Shingler Shingler = TFilteredUTF8Shingler(5, false);
        TVector<TUTF8Shingler::TShingle> shingles;

        {
            Shingler.ShingleText(shingles, part->htmlBody.data(), part->htmlBody.size());
        }

        if (shingles) {
            const ui64 h = std::reduce(shingles.cbegin(), shingles.cend(), TFNVHash<size_t>::Init, [](ui64 h, ui64 sh) {
                return TFNVHash<size_t>::Do(sh, h);
            });
            Master.SetShingle(spSanShingle, h);
        }
    }
};

class TSecondShingler : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;
    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto part = answer.FindVisiblePart();
        if (!part)
            return;

        auto g = Guard(prof.Prof("second"));
        TFilteredUTF8Shingler Shingler = TFilteredUTF8Shingler(5, false);
        TVector<TFilteredUTF8Shingler::TShingle> shingles;

        {
            const ui64 wordsQuantity = Shingler.ShingleText(shingles, part->PureBody.Clipped.data(), part->PureBody.Clipped.size());
            Master.CheckRange("tcwcount", wordsQuantity);
        }

        if (shingles) {
            const ui64 h = std::reduce(shingles.cbegin(), shingles.cend(), TFNVHash<size_t>::Init, [](ui64 h, ui64 sh) {
                return TFNVHash<size_t>::Do(sh, h);
            });
            Master.SetShingle(spCShingle, h);
        }
    }
};


class TKnnProcessor : public IRengineHtmlSanAnswerProcessor {
public:
    explicit TKnnProcessor(TRengine& master) noexcept
    : IRengineHtmlSanAnswerProcessor(master)
    , SEEDS(NMinHash::GenSeeds(Master.Config.MinHashSize)) {}

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto part = answer.FindVisiblePart();
        if (!part)
            return;

        auto g = Guard(prof.Prof("knn"));
        {
            Master.m_cur->MinHash = NMinHash::MinHash(part->PureBody.Clipped, SEEDS);
            NMinHash::MinHash(part->Subject.Clipped, SEEDS, Master.m_cur->MinHash);
            NMinHash::MinHash(part->normalizedFrom, SEEDS, Master.m_cur->MinHash);
        }

        if(Master.m_cur->MinHash) {
            auto hashesString = JoinSeq(",", Master.m_cur->MinHash);
            Master.GetSpStat().AddStat("coord") << hashesString;
            Master.m_cur->MlLogBuilder.Add("minhash", hashesString);
        }
    }
private:
    const TVector<NMinHash::TInt> SEEDS;
};

class TLsaDataProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    class TDiffConsumer : public NTextDeobfuscate::IDiffConsumer {
    public:
        explicit TDiffConsumer(TSpStat::TRecord record) noexcept
        : Record(std::move(record)) {}

        void Consume(const TUtf32StringBuf& word, const TUtf32StringBuf& corrected) noexcept override {
            Record << WideToUTF8(word) << " vs " << WideToUTF8(corrected) << ';';
        }
    private:
        TSpStat::TRecord Record;
    };

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto g = Guard(prof.Prof("lsa"));
        if (auto part = answer.FindVisiblePart()) {
            Master.m_cur->MlLogBuilder.Add("visible_part_hid", part->hid);

            {
                Master.m_cur->lsaRequestData.Add(part->normalizedFrom, NLSA::TField::FullFromAddr);
                Master.m_cur->lsaRequestData.Add(part->normalizedFrom.substr(0, part->normalizedFrom.find_last_of('@')),
                                                 NLSA::TField::FullFromAddrLogin);

                Master.m_cur->MlLogBuilder.Add(ToString(NLSA::TField::FullFromAddr),
                                               Master.m_cur->lsaRequestData.Get(NLSA::TField::FullFromAddr).front());
                Master.m_cur->MlLogBuilder.Add(ToString(NLSA::TField::FullFromAddrLogin),
                                               Master.m_cur->lsaRequestData.Get(
                                                       NLSA::TField::FullFromAddrLogin).front());
            }

            if (part->PureBody.Clipped) {
                Master.m_cur->lsaRequestData.Add(part->PureBody.Clipped, NLSA::TField::PureBody);
                for (auto w : NText2Shingles::Text2Shingles(part->PureBody.Clipped, LANG_UNK, true,
                                                            std::numeric_limits<size_t>::max())) {
                    Master.m_cur->lsaRequestData.Add(std::move(w), NLSA::TField::FullBody);
                }
                Master.m_cur->MlLogBuilder.Add("nnbody_full",
                                               JoinStrings(Master.m_cur->lsaRequestData.Get(NLSA::TField::FullBody),
                                                           ","));
            }

            Master.m_cur->MlLogBuilder.Add("full_subject", part->Subject.Original);

            if (Master.TextDeobfuscator) {
                if (Master.TextDeobfuscator) {
                    try {
                        TString deobfuscatedText;

                        TDiffConsumer onDiff(Master.m_pstat.AddStat("repl-body"));

                        const size_t replacements = Master.TextDeobfuscator->Replace(
                                part->PureBody.Original,
                                deobfuscatedText,
                                true,
                                true,
                                &onDiff,
                                5
                        );
                        Master.CheckValueAllRules(FD_BODY_REPLACEMETS, replacements);
                        Master.m_cur->lsaRequestData.Add(std::move(deobfuscatedText), NLSA::TField::PureDeobfuscatedBody);
                    } catch (...) {
                        Master.m_cur->Logger << (TLOG_ERR) << "TLsaDataProcessor:" << CurrentExceptionMessageWithBt();
                    }
                }

                try {
                    TString deobfuscatedText;

                    TDiffConsumer onDiff(Master.m_pstat.AddStat("repl-subj"));

                    const size_t replacements = Master.TextDeobfuscator->Replace(
                            part->Subject.Original,
                            deobfuscatedText,
                            true,
                            true,
                            &onDiff
                    );
                    Master.CheckValueAllRules(FD_SUBJECT_REPLACEMETS, replacements);

                    Master.m_cur->lsaRequestData.Add(std::move(deobfuscatedText), NLSA::TField::PureDeobfuscatedSubject);
                    Master.m_cur->MlLogBuilder.Add("nnsubj_pure_deob",
                        JoinStrings(Master.m_cur->lsaRequestData.Get(
                            NLSA::TField::PureDeobfuscatedSubject), ","));
                } catch (...) {
                    Master.m_cur->Logger << (TLOG_ERR) << "TLsaDataProcessor:" << CurrentExceptionMessageWithBt();
                }

                if (part->displayName) {
                    TString deobfuscated;
                    Master.TextDeobfuscator->Replace(part->displayName, deobfuscated, false);
                    Master.m_cur->lsaRequestData.Add(std::move(deobfuscated), NLSA::TField::PureDeobfuscatedFromname);
                }

                if (part->normalizedFrom) {
                    TString deobfuscated;
                    Master.TextDeobfuscator->Replace(part->normalizedFrom, deobfuscated, false);
                    Master.m_cur->lsaRequestData.Add(std::move(deobfuscated), NLSA::TField::PureDeobfuscatedFromaddr);
                }
            }

            if (part->Subject.Clipped) {
                Master.m_cur->lsaRequestData.Add(part->Subject.Clipped, NLSA::TField::PureSubject);
                Master.m_cur->MlLogBuilder.Add(ToString(NLSA::TField::PureSubject), part->Subject.Clipped);
                for (auto w : NText2Shingles::Text2Shingles(part->Subject.Clipped, LANG_UNK, true,
                                                            std::numeric_limits<size_t>::max())) {
                    Master.m_cur->lsaRequestData.Add(std::move(w), NLSA::TField::FullSubject);
                }
                Master.m_cur->MlLogBuilder.Add("nnsubj_full",
                                               JoinStrings(Master.m_cur->lsaRequestData.Get(NLSA::TField::FullSubject),
                                                           ","));
            }
        }

        if (Master.TextDeobfuscator) {
            for (const NHtmlSanMisc::TDoc &part : answer.Docs()) {
                if (part.attachName) {
                    TString deobfuscated;
                    Master.TextDeobfuscator->Replace(part.attachName, deobfuscated, false);
                    Master.m_cur->lsaRequestData.Add(std::move(deobfuscated),
                                                     NLSA::TField::PureDeobfuscatedAttachesNames);
                }
            }
        }
    }
};

class TMd5Shingler : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto g = Guard(prof.Prof("md5"));
        for (const auto& doc : answer.Docs()) {
            if (doc.md5) {
                auto sh = Sprintf("%015llx", static_cast<unsigned long long int>(FnvHash<ui64>(doc.md5)));
                Master.SetShingle(spAttachMd5, sh);
            }
        }
    }
};

class TUrlsProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto g = Guard(prof.Prof("urls"));
        auto hosts = [&answer, this]() {
            TVector<TStringBuf> hosts;
            THashSet<TStringBuf> urls;
            for (const auto &doc : answer.Docs()) {
                for (const auto &url : doc.urls) {
                    if (url.Size() > 8) {
                        if (AsciiHasPrefixIgnoreCase(url, "http://")) {
                            urls.emplace(TStringBuf(url).substr(7));
                        } else if (AsciiHasPrefixIgnoreCase(url, "https://")) {
                            urls.emplace(TStringBuf(url).substr(8));
                        } else {
                            urls.emplace(url);
                        }
                    }
                    const NUri::TParser parser(NUri::TFeature::FeaturesDefault | NUri::TFeature::FeatureSchemeFlexible, url);
                    const NUri::TSection& host = parser.Get(NUri::TField::FieldHost);
                    if (host.IsSet()) {
                        hosts.emplace_back(host.Get());
                    } else if (host.Get()) {
                        Master.m_cur->Logger << (TLOG_WARNING) << "Failed to parse url <" << url << ">";
                    }
                }
            }

            SortUnique(hosts, [](const TStringBuf& s1, const TStringBuf& s2){
                return AsciiCompareIgnoreCase(s1, s2) < 0;
            });
            for (const auto& url: urls) {
                Master.m_pstat.AddLimitedStat(ST_HTTP) << url;
                Master.CheckField(FD_URL_HTTP, url);
                Master.CheckField(FD_URL_HTTP_PARAM, url);
            }
            return hosts;
        }();

        if(!hosts)
            return;

        for(const auto& host: hosts) {
            if(Master.m_pRulesHolder->m_pListRuler.CheckWord(Master.m_cur->rulesContext, host, SP_LIST_SKIP_URI))
                continue;

            const auto zone = Master.m_common_zone_detector.GetCommonZoneFromHost(host);

            if(Master.m_pRulesHolder->m_pListRuler.CheckWord(Master.m_cur->rulesContext, zone, SP_LIST_SKIP_URI))
                continue;

            Master.Add2UrlReputationList(host, EUrlStaticticSource::BODY);
            Master.AddPattern(host, EN_SH_HOST);
            if(zone != host)
                Master.AddPattern(zone, EN_SH_COMMON_ZONE);
        }

        Master.m_pstat.AddStat("new_hosts") << MakeRangeJoiner(";", hosts);

        {
            auto newEnd = std::remove_if(hosts.begin(), hosts.end(), [excludes = &Master.GetSpTop().UrlsToExclude](const TStringBuf& host){
                return excludes->contains(host);
            });
            const size_t newSize = Min<size_t>(Master.Config.Hosts2MlLimitation, std::distance(hosts.begin(), newEnd));
            hosts.resize(newSize);
        }

        using TGroup = THashMap<TStringBuf, TVector<size_t>, TCIOps, TCIOps>;

        TGroup hostsByPostfixesFirstLevel, hostsByPostfixesSecondLevel;
        for(size_t i = 0; i < hosts.size(); i++) {
            const auto& host = hosts[i];
            size_t pos;

            if((pos = host.rfind('.')) == NPOS) {
                hostsByPostfixesFirstLevel[host].emplace_back(i);
                continue;
            }
            hostsByPostfixesFirstLevel[host.substr(pos)].emplace_back(i);

            if((pos = host.rfind('.', pos - 1)) == NPOS) {
                hostsByPostfixesSecondLevel[host].emplace_back(i);
                continue;
            }
            hostsByPostfixesSecondLevel[host.substr(pos)].emplace_back(i);
        }

        Master.m_cur->HostsHashes.reserve(hosts.size());

        TStringBuilder mlRecord;
        for (const auto &host: hosts) {
            Master.m_cur->HostsHashes.emplace_back(FnvCaseLess<ui64>(host));
            mlRecord << Master.m_cur->HostsHashes.back() << ',';
        }
        mlRecord << ';';

        for (const TGroup &group: {std::cref(hostsByPostfixesSecondLevel), std::cref(hostsByPostfixesFirstLevel)}) {
            for (const auto&[postfix, indexes]: group) {
                Master.m_cur->HostsHashes.emplace_back(FnvCaseLess<ui64>(postfix));
                mlRecord << Master.m_cur->HostsHashes.back() << ':' << MakeRangeJoiner(",", indexes) << ';';
            }
        }
        Master.m_cur->MlLogBuilder.Add("hosts", std::move(mlRecord));
    }
};

class TRulesProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    class TCyrReplacer {
    public:
        friend IOutputStream &operator<<(IOutputStream &stream, const TCyrReplacer &replacer) {
            for (wchar32 c : replacer.Text) {
                switch (c) {
                    case 'a':
                        stream << U'а';
                    case 'A':
                        stream << U'А';
                    case 'b':
                        stream << U'ь';
                    case 'B':
                        stream << U'В';
                    case 'c':
                        stream << U'с';
                    case 'C':
                        stream << U'С';
                    case 'e':
                        stream << U'е';
                    case 'E':
                        stream << U'Е';
                    case 'H':
                        stream << U'Н';
                    case 'k':
                        stream << U'к';
                    case 'K':
                        stream << U'К';
                    case 'm':
                        stream << U'м';
                    case 'M':
                        stream << U'М';
                    case 'n':
                        stream << U'п';
                    case 'o':
                        stream << U'о';
                    case 'O':
                        stream << U'О';
                    case 'p':
                        stream << U'р';
                    case 'P':
                        stream << U'Р';
                    case 'T':
                        stream << U'Т';
                    case 'u':
                        stream << U'и';
                    case 'x':
                        stream << U'х';
                    case 'X':
                        stream << U'Х';
                    case 'y':
                        stream << U'у';
                    case 'Y':
                        stream << U'У';
                    default:
                        stream << c;
                }
            }
            return stream;
        }

        explicit TCyrReplacer(const TUtf32StringBuf text) noexcept
                : Text(text) {}

    private:
        const TUtf32StringBuf Text;
    };

    struct TState{
        TState() = default;
        TUtf32String Word;
        bool HasLatin{};
        bool HasCyr{};
    };

    static void replace_and_write(IOutputStream& stream, TState& state) {
        if(state.HasCyr && state.HasLatin) {
            stream << TCyrReplacer(state.Word);
        } else {
            stream << state.Word;
        }
        state = {};
    }

    class TTextNormalizer{
    public:
        friend IOutputStream& operator<<(IOutputStream& stream, const TTextNormalizer& replacer) {

            auto it = reinterpret_cast<const ui8*>(replacer.Text.data());
            auto end = it + replacer.Text.size();

            TState state;

            while(it < end) {
                wchar32 c;
                if (ReadUTF8CharAndAdvance(c, it, end) != RECODE_OK) {
                    stream.Write('?');
                    continue;
                }

                const auto alpha = IsAlphabetic(c);
                const auto digit = !alpha && IsNumeric(c);

                if(alpha || digit) {
                    state.Word += c;
                    switch (ScriptByGlyph(c)) {
                        case SCRIPT_CYRILLIC:
                            state.HasCyr = true;
                            break;
                        case SCRIPT_LATIN:
                            state.HasLatin = true;
                            break;
                        default:
                            break;
                    }
                } else {
                    if (state.Word) {
                        replace_and_write(stream, state);
                    }
                    stream << c;
                }
            }

            if(state.Word) {
                replace_and_write(stream, state);
            }

            return stream;
        }

        explicit TTextNormalizer(const TStringBuf text) noexcept
        : Text(text) {}
    private:
        const TStringBuf Text;
    };

    TString Normalize(const TString& text) const {
        if(Master.Config.UseDeobfuscatorForNormalizeBeforeRe && Master.TextDeobfuscator) {
            TString result;
            Master.TextDeobfuscator->Replace(text, result, false);
            return result;
        } else {
            return TStringBuilder{} << TTextNormalizer(text);
        }
    }

    void Process(const NHtmlSanMisc::TAnswer &answer, CProf& prof) final {
        auto g = Guard(prof.Prof("rules"));
        Master.CheckFieldAllRules(FD_HBF_PROJECT_ID, answer.GetHbfProjectId());
        Master.GetSpStat().AddStat(FD_HBF_PROJECT_ID) << answer.GetHbfProjectId();
        Master.m_cur->MlLogBuilder.Add(ToString(FD_HBF_PROJECT_ID), answer.GetHbfProjectId());

        if(const NHtmlSanMisc::TDoc* doc = answer.FindVisiblePart()) {
            Master.CheckFieldAllRules(FD_FULL_PURE_BODY, Normalize(doc->PureBody.Original));
        }

        for (const auto &doc : answer.Docs()) {

            Master.CheckValueAllRules(FD_FULL_DOC, doc.Source);

            Master.CheckFieldAllRules(FD_FULL_BODY, Normalize(doc.PureBody.Original));
            Master.CheckFieldAllRules(FD_FULL_SUBJECT, Normalize(doc.Subject.Original));
            Master.CheckFieldAllRules(FD_FULL_FROM, Normalize(doc.displayName + ' ' + doc.normalizedFrom));

            Master.CheckFieldAllRules(FD_FULL_UGC_ID, doc.UgcId);
            Master.CheckFieldAllRules(FD_FULL_DEOBFUSCATED_UGC, doc.Ugc);

            if(doc.displayName)
                Master.CheckFieldAllRules(FD_FULL_FROM_NAME, doc.displayName);

            Master.CheckFieldAllRules(FD_FULL_FROM_ADDR, doc.normalizedFrom);

            if (doc.htmlBody)
                Master.CheckFieldAllRules(FD_FULL_HTML, Normalize(doc.htmlBody));
        }
    }
};

class TDomenFactorsProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer &answer, CProf& prof) final {
        if(!Y_UNLIKELY(Master.DomenFactorsBuilder))
            return;
        auto g = Guard(prof.Prof("domain"));
        if(auto part = answer.FindVisiblePart(); part && part->urls) {
            auto &targetFactors = Master.m_cur->m_mapValueFactors4Matrixnet;

            TVector<NDomenFactors::TUrl> urls(Reserve(part->urls.size()));
            for(const auto& url: part->urls) {
                const NUri::TParser parser{NUri::TFeature::FeaturesDefault | NUri::TFeature::FeatureSchemeFlexible, url};
                const NUri::TSection& scheme{parser.Get(NUri::TField::FieldScheme)};
                const NUri::TSection& host{parser.Get(NUri::TField::FieldHost)};
                if (scheme.IsSet() && host.IsSet()) {
                    urls.emplace_back(scheme.Get(), host.Get());
                } else if ((!scheme.IsSet() && scheme.Get()) || (!host.IsSet() && host.Get())) {
                    Master.m_cur->Logger << (TLOG_WARNING) << "Failed to parse url <" << url << ">";
                }
            }
            for (const auto&[factorName, value]: Master.DomenFactorsBuilder->GetFactors(part->normalizedFrom, urls)) {
                targetFactors.emplace(factorName, value);
            }
        }
    }
};

class TAbookProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(Y_DECLARE_UNUSED const NHtmlSanMisc::TAnswer &answer, CProf& prof) final {
        auto g = Guard(prof.Prof("abook"));
        auto& cur = *Master.m_cur;
        auto& spruler = *Master.m_pRulesHolder->m_spruler;

        for(auto& [email, attrs]: cur.m_rcptattrs) {
            if(!attrs.sender)
                continue;
            const auto& abookInfo = attrs.sender->AbookInfo;

            TString extractedEmail;

            spruler.CheckRange(cur, "senders_rec_count", abookInfo.ReceivedCount, false);
            Master.AddPattern(attrs.sender->Login.AsString(), EN_SH_FIRST_FROM_SNDR, abookInfo.ReceivedCount == 0);

            spruler.CheckRange(cur, "senders_read_count", abookInfo.ReceivedReadCount, false);

            spruler.CheckRange(cur, "ab_count", abookInfo.SentCount, false);

            if (!cur.m_isFreemail) {
                spruler.CheckRange(cur, "domain_sent_count", abookInfo.DomainSentCount, false);
                spruler.CheckRange(cur, "domain_rec_count", abookInfo.DomainReceivedCount, false);
                Master.AddPattern(attrs.sender->Domain, EN_SH_FIRST_FROM_DMN,
                                  abookInfo.DomainReceivedCount == 0); // do PUT when firstmail only

                spruler.CheckRange(cur, "domain_read_count", abookInfo.DomainReceivedReadCount, false);
            }

            if (abookInfo.DomainReceivedCount)
                cur.m_mapValueFactors4Matrixnet["MNF_D_READ"] = abookInfo.DomainReceivedReadCount / (1.f * abookInfo.DomainReceivedCount);
            if (abookInfo.ReceivedCount)
                cur.m_mapValueFactors4Matrixnet["MNF_S_READ"] = abookInfo.ReceivedReadCount / (1.f * abookInfo.ReceivedCount);

            Master.CheckValueAllRules(FD_ABOOK_RESPONSE, abookInfo.Original);

            if (!abookInfo.SourceType)
                attrs.SetTriade(false, attrs.sender->Uid, abookInfo.SentCount);
            else {
                attrs.SetTriade(true, attrs.sender->Uid, abookInfo.SentCount); // exists in Abook, UID, cnt
                cur.rulesContext.SetRule("SNDR_IN_ABUK");
            }
        }

        int abook_cnt = 0;
        for (const auto &attr_it : cur.m_rcptattrs)
            if (attr_it.second.aBook)
                abook_cnt++;

        const auto iSizeOr10 = Min(cur.m_rcptattrs.size(), size_t(10));
        if (iSizeOr10 && abook_cnt) {
            double abook_perc = (double) abook_cnt / (double) iSizeOr10 * 100.0;
            spruler.CheckRange(cur, "abk_perc", abook_perc, false);
        }
    }
};

class TReferencesProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(Y_DECLARE_UNUSED const NHtmlSanMisc::TAnswer &answer, CProf& prof) final {
        auto g = Guard(prof.Prof("refs"));
        auto& cur = *Master.m_cur;
        bool validInReplyTo = false;
        bool validReferences = false;

        for(auto& [email, attrs]: cur.m_rcptattrs) {
            if(!attrs.sender) {
                continue;
            }
            if (attrs.sender->ValidInReplyTo) {
                validInReplyTo = true;
            }
            if (attrs.sender->ValidReferences) {
                validReferences = true;
            }
        }

        if (validInReplyTo) {
            cur.rulesContext.SetRule("VALID_IN_REPLY_TO");
        }

        if (validReferences) {
            cur.rulesContext.SetRule("VALID_REFERENCES");
        }
    }
};

class TDkimStatsProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        for (const TString& dkimDomain: answer.GetDkimDomains()) {
            Master.m_cur->DkimDomains.emplace_back(dkimDomain);
            Master.AddPattern(dkimDomain, EN_SH_DKIM_DOMAIN);
        }

        const auto& dkimStats = answer.GetDkimStats();
        if (!dkimStats)
            return;

        auto g = Guard(prof.Prof("dkim"));
        const auto& value = *dkimStats;
        NJsonWriter::TBuf writer(NJsonWriter::HEM_UNSAFE);
        writer.WriteJsonValue(&value);
        Master.m_pstat.AddStat("dkim_stats") << writer.Str();
        Master.CheckValueAllRules(FD_DKIM_STATS, value);
    }
};

class TMatchedTemplateProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        auto g = Guard(prof.Prof("sherlock"));
        if (const TMaybe<NHtmlSanMisc::TMatchedTemplate> &matchedTemplate = answer.GetMatchedTemplate()) {
            Master.CheckValueAllRules(FD_MATCHED_TEMPLATE, matchedTemplate->Source);
        }
    }
};

class TBigBDataProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf&) final {
        const TMaybe<bool> hasCryptaUserVector{answer.GetHasCryptaUserVector()};
        if (hasCryptaUserVector.Defined()) {
            if (*hasCryptaUserVector) {
                Master.m_cur->rulesContext.SetRule("SENDER_HAS_CRYPTA_USER_VECTOR");
            } else {
                Master.m_cur->rulesContext.SetRule("NO_SENDER_CRYPTA_USER_VECTOR");
            }
        }
    }
};

class TUserinfosProcessor : public IRengineHtmlSanAnswerProcessor {
    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf& prof) final {
        const auto& mailFromUserinfo = answer.GetMailFromJsonUserinfo();
        const auto& recipientsUserinfos = answer.GetRecipientsUserinfos();
        if (!mailFromUserinfo && !recipientsUserinfos) {
            return;
        }

        if (answer.GetAllFromSameOrgId()) {
            Master.m_cur->rulesContext.SetRule("ORG_ID_SAME_ALL_SO2");
        }
        auto g = Guard(prof.Prof("userinfo"));
        if (mailFromUserinfo) {
            Master.CheckValueAllRules(FD_MAILFROM_USERINFO, *mailFromUserinfo);
        }
        if (recipientsUserinfos) {
            Master.CheckValueAllRules(FD_RECIPIENTS_USERINFOS, *recipientsUserinfos);
        }
    }
};

class TYaDiskInfoProcessor : public IRengineHtmlSanAnswerProcessor {

    using IRengineHtmlSanAnswerProcessor::IRengineHtmlSanAnswerProcessor;

    void Process(const NHtmlSanMisc::TAnswer& answer, CProf&) final {
        for (const auto& yaDiskInfo: answer.GetYaDiskInfos()) {
            Master.CheckYaDisk(yaDiskInfo);
        }
    }
};

static auto CreateStaticProcessors (TRengine& master) {
    TVector<THolder<IRengineHtmlSanAnswerProcessor>> processors;

    processors.emplace_back(MakeHolder<TBBProcessor>(master));
    processors.emplace_back(MakeHolder<TAttachMlPrinter>(master));
    processors.emplace_back(MakeHolder<TFirstShingler>(master));
    processors.emplace_back(MakeHolder<TSecondShingler>(master));
    processors.emplace_back(MakeHolder<TMd5Shingler>(master));
    processors.emplace_back(MakeHolder<TUrlsProcessor>(master));
    processors.emplace_back(MakeHolder<TLsaDataProcessor>(master));
    processors.emplace_back(MakeHolder<TRulesProcessor>(master));
    processors.emplace_back(MakeHolder<TDomenFactorsProcessor>(master));
    processors.emplace_back(MakeHolder<TPersonalFilterProcessor>(master));
    processors.emplace_back(MakeHolder<TAbookProcessor>(master));
    processors.emplace_back(MakeHolder<TWmdDistanceProcessor>(master));
    processors.emplace_back(MakeHolder<TWmdSubjectDistanceProcessor>(master));
    processors.emplace_back(MakeHolder<TWmdAttachNameDistanceProcessor>(master));
    processors.emplace_back(MakeHolder<TDssmDistanceProcessor>(master));
    processors.emplace_back(MakeHolder<TDssmSubjectDistanceProcessor>(master));
    processors.emplace_back(MakeHolder<TReferencesProcessor>(master));
    processors.emplace_back(MakeHolder<TDkimStatsProcessor>(master));
    processors.emplace_back(MakeHolder<TMatchedTemplateProcessor>(master));
    processors.emplace_back(MakeHolder<TBigBDataProcessor>(master));
    processors.emplace_back(MakeHolder<TUserinfosProcessor>(master));
    processors.emplace_back(MakeHolder<TYaDiskInfoProcessor>(master));

    return processors;
}

NHyperscan::TScratch MakeScratch(const THashMap<TSpFields, NHyperscan::TDatabase>& dbs) {
    if(!dbs)
        return {};

    auto it = dbs.cbegin();

    auto scratch = NHyperscan::MakeScratch(it->second);

    for(++it; it != dbs.cend(); ++it) {
        NHyperscan::GrowScratch(scratch, it->second);
    }

    return scratch;
}

TRengine::TRengine(const T_SpParams &p_params,
                   TTrueAtomicSharedPtr<TRulesHolder> rulesHolder,
                   IThreadPool & ThreadPool,
                   TTrueAtomicSharedPtr<TRengineCurlPools> Pools,
                   kipv6::TStringsLists* rulePrintList,
                   ip_match local_matcher,
                   TTrueAtomicSharedPtr<TAppliersMap> models,
                   TTrueAtomicSharedPtr<NNeuralNetApplier::TModel> text2VecModel,
                   TTrueAtomicSharedPtr<NTextDeobfuscate::TTextDeobfuscator> textDeobfuscator,
                   TTrueAtomicSharedPtr<NDomenFactors::TDomenFactorsBuilder> domenFactorsBuilder,
                   TUserWeightsPairPtr usersWeights,
                   TTrueAtomicSharedPtr<TUidsStats> uidsStats)
        : Config(p_params.pSpTop)
        , ThreadPool(ThreadPool)
        , Pools(std::move(Pools))
        , m_common_zone_detector(TZoneDetector::TE_KOI8R)
        , Text2VecModel(std::move(text2VecModel))
        , TextDeobfuscator(std::move(textDeobfuscator))
        , DomenFactorsBuilder(std::move(domenFactorsBuilder))
        , UidsStats(std::move(uidsStats))
        , Processors(CreateStaticProcessors(*this))
        , UsersWeights(std::move(usersWeights))
        , LocalMatcher(std::move(local_matcher)) {
    ReloadRules(std::move(rulesHolder));
    ActualizeRules();
    ReloadModels(std::move(models));
    ActualizeModels();

    is_spk = m_pRulesHolder->IsSPK();

    RulePrintList = rulePrintList;

    m_p_ml_logger       = &m_null_logger;

    ShortLog = p_params.p_spLoggers->ShortLog;
    SetLogger(&m_p_ml_logger,       p_params.p_spLoggers->GetSpLoggerPtr(SO_ML_LOGGER));

    if (!m_pRulesHolder->IsOK())
    {
        SysLogger() << TLOG_ERR << "Rules reading error!";
        return;
    }

    SetTimeForStatLog();
}

void TRengine::ReloadRules(TTrueAtomicSharedPtr<TRulesHolder> rulesHolder) {
    auto newLuaRulesRunner = MakeTrueAtomicShared<NLua::TRunner>(rulesHolder->LuaRulesRunner.Clone());
    auto newLuaRulesContext = MakeTrueAtomicShared<NLua::TContext>();
    TRulesMap newRules;
    TLuaRulesMap newLuaRules;
    for(const THolder<TRuleDef>& ruleDef: rulesHolder->m_ppRules) {
        switch(ruleDef->rt) {
            case RT_UNKNOWN:break;
            case RT_RGEX:
                Y_VERIFY(newRules.emplace(std::addressof(*ruleDef), std::get<TReDef>(ruleDef->rules).MakeRule().Release()).first);
                break;
            case RT_RGEXGROUP:break;
            case RT_BF:break;
            case RT_ARITHMETIC:break;
            case RT_PRESENT:
                Y_VERIFY(newRules.emplace(std::addressof(*ruleDef), TPresentDef::MakeRule().Release()).first);
                break;
            case RT_ABSENT:break;
            case RT_ALG:break;
            case RT_RECEIVED_NUM:break;
            case RT_ANTI:break;
            case RT_LV:break;
            case RT_RANGE:break;
            case RT_RANGE_DATE:break;
            case RT_LEVENSTEIN:break;
            case RT_HS:break;
            case RT_LUA: {
                newLuaRules.emplace(std::addressof(*ruleDef), std::get<TLuaRuleDef>(ruleDef->rules).MakeRule(*newLuaRulesRunner, *newLuaRulesContext));
                break;
            }
        }
    }
    auto newHsScratch = MakeTrueAtomicShared<NHyperscan::TScratch>(MakeScratch(rulesHolder->HsDbsByField));

    const TWriteGuard g(RulesLock);

    RulesHolderIdeal = std::move(rulesHolder);
    rulesIdeal = MakeTrueAtomicShared<TRulesMap>(std::move(newRules));
    LuaRulesIdeal = MakeTrueAtomicShared<TLuaRulesMap>(std::move(newLuaRules));
    LuaRulesRunnerIdeal = std::move(newLuaRulesRunner);
    LuaRulesContextIdeal = std::move(newLuaRulesContext);
    HsScratchIdeal = std::move(newHsScratch);
}

void TRengine::ReloadModels(TTrueAtomicSharedPtr<TAppliersMap> models) {
    const TWriteGuard g(ModelsLock);
    ModelsIdeal = std::move(models);
}

void TRengine::ActualizeRules() {
    const TReadGuard g(RulesLock);

    m_pRulesHolder = RulesHolderIdeal;
    rules = rulesIdeal;
    LuaRules = LuaRulesIdeal;
    LuaRulesRunner = LuaRulesRunnerIdeal;
    LuaRulesContext = LuaRulesContextIdeal;
    HsScratch = HsScratchIdeal;
}

void TRengine::ActualizeModels() {
    const TReadGuard g(ModelsLock);
    if(ModelsIdeal) {
        Models = ModelsIdeal;
    } else {
        Models = MakeTrueAtomicShared<TAppliersMap>();
    }
}

TString TRengine::MNF2String() const
{
    TStringStream strResult;

    for (const auto &it : m_cur->m_mapValueFactors4Matrixnet)
        strResult << it.first << ":" << it.second << ";";

    return std::move(strResult.Str());
}

NJson::TJsonValue TRengine::MNF2Json() const {
    NJson::TJsonValue res;
    for (const auto &it : m_cur->m_mapValueFactors4Matrixnet) {
        if(IsNan(it.second)) {
            m_cur->Logger << (TLOG_ERR) << it.first << " is nan";
        } else {
            res[it.first] = it.second;
        }
    }
    return res;
}

bool TRengine::InitMessage(
    const TStringBuf& pMessageId,
    TSimpleSharedPtr<const NHtmlSanMisc::TAnswer> so2Context,
    TSimpleSharedPtr<const TActivityShingleRequestVector> activityInfo,
    const TString& ocrText,
    TSpClass prevSpClass,
    TLog logger)
{
    ActualizeRules();
    ActualizeModels();

    m_cur = MakeHolder<TCurMessageEngine>(
            *m_pRulesHolder,
            std::move(logger),
            Config.tskvFormat,
            pMessageId,
            Config.score,
            Config.delivery_score
    );

    m_cur->So2Context = std::move(so2Context);
    m_cur->ActivityInfo = std::move(activityInfo);
    m_cur->OcrText = ocrText;
    m_cur->PrevSpClass = prevSpClass;
    m_pstat.Reset();
    m_bodypart = MakeHolder<TSpBodyPart>();
    m_shingler = MakeHolder<TShingleEngine>(*m_pRulesHolder);
    m_alg = MakeHolder<TSpAlg>(*m_pRulesHolder, Config, m_pstat, LocalMatcher, m_cur.Get());
    m_html = MakeHolder<TSpHtml>(&m_cur->rulesContext, *m_pRulesHolder, m_pstat, m_cur->Logger);

    LuaRulesContext->SetSo2Answer(m_cur->So2Context.Get());
    LuaRulesContext->SetRuleProvider(&m_cur->rulesContext);
    LuaRulesContext->SetMlLogProvider(&m_cur->MlLogBuilder);
    LuaRulesContext->SetDlvLogProvider(&m_pstat);
    LuaRulesContext->SetShinglerProvider(this);

    const char* pmes = pMessageId ? pMessageId.data() : "";
    const char* pPureMessID = strchr(pmes, ':');
    if (pPureMessID)
        pPureMessID++;
    {
//        Dec  2 07:29:22.105 - 1575260962:
        const auto now = Now();

        const auto date = now.FormatLocalTime("%b %d %H:%M:%S");

        const TString messStr = TStringBuilder{} << date << '.' << now.MilliSecondsOfSecond() << " - " << now.TimeT() << ":  " << GetMessageId();;

        m_cur->DlvLogRequest.MessTime = now;

        m_pstat.AddStat(ST_MESSAGE) << messStr;
        if (HostName()) {
            m_pstat.AddStat(ST_HOST_LOCAL) << HostName();
            m_cur->DlvLogRequest.LocL = HostName();
        }

        m_cur->MlLogBuilder.Add(ToString(ST_MESSAGE), ToString(now.TimeT()));
        m_cur->MlLogBuilder.Add("unixtime", ToString(now.Seconds()));

        m_pstat.AddStat(ST_SP_VERSION) << spVersion << ' ' << m_pRulesHolder->GetCS() << " mnSum:" << GetMNModelsIdSum();
    }

    m_cur->messageId = pPureMessID;

    m_cur->c_rcpt = 1;

    CheckLuaInitMessage();

    return true;
}

void TRengine::EndMessage(const TLog& deliveryLog, const TLog& mlLog) noexcept try {
    reTimings->PushSignal(m_cur->reProf.GetDuration().MilliSeconds());
    PushRulesSignals();
    m_pstat.PrintLimitedRecordsCount();

    if(m_cur->fieldsById[TSpFields::FD_FROM_ADDR]) {
        const auto& fromAddr = *m_cur->fieldsById[TSpFields::FD_FROM_ADDR];
        const size_t rcptsWithoutPf = m_cur->m_rcptattrs.size() - m_cur->RcptsWithPF;
        if(fromAddr && Pools->DkimRequester && rcptsWithoutPf && IsHam(m_cur->m_messclass) && m_cur->fieldsById[TSpFields::FD_FROM_ADDR].Defined()) {
            if(!ThreadPool.AddFunc([
                    requester = Pools->DkimRequester,
                    logger = m_cur->Logger,
                    DkimDomains = m_cur->DkimDomains,
                    fromAddr,
                    rcptsWithoutPf](){
                requester->Update(logger, fromAddr, DkimDomains, Now(), rcptsWithoutPf);
            })) {
                m_pstat.AddStat("lost_dkim_domains") << fromAddr << ' ' << MakeRangeJoiner(",", m_cur->DkimDomains);
            }
        }
    }

    {
        const auto processingDuration = Now() - m_cur->Creation;
        m_pstat.AddStat(ST_TIME) << processingDuration.MilliSeconds();
    }
    {
        auto statRec = m_pstat.AddStat("rule_prof ");
        auto sysRec = std::move(m_cur->Logger << (TLOG_NOTICE) << "rule_prof ");
        TTeeOutput tee(&statRec, &sysRec);

        const TMaybe<TRuleCurrentRef> mostSlowlyRule = m_cur->rulesContext.GetMostSlowlyRule();

        if(mostSlowlyRule) {
            const TRuleCurrent& worstRule = *mostSlowlyRule;
            tee << worstRule.GetDef().pRuleName << ':' << worstRule.GetProfiler().GetDuration().MilliSeconds() << ';';
        }
    }

    {
        NJson::TJsonValue report = TClassifiersContext::DiffReport(m_cur->ClassifiersContext,
            MakeClassifiersContext());

        m_cur->MlLogBuilder.Add("diff_report", TStringBuilder{} << std::move(report));
    }

    TString pstat = Config.NewDlvFormat ? (TStringBuilder{} << TSpStat::TJsonDlvLogPrinter(m_pstat)) : (
        TStringBuilder{} << TSpStat::TDlvLogPrinter(m_pstat));

    if (!CancelLog() && pstat) {
        NThreading::Async([
            requester = Pools->StatLogRequester,
            requester2 = Pools->SpLogggerRequester,
            pstat = std::move(pstat),
            request = std::move(m_cur->DlvLogRequest),
            logRegim = Config.DeliveryLogRegim,
            DeliveryLog = deliveryLog,
            logger = m_cur->Logger]() mutable {
          bool logSent = false;
          if (requester) {
              logSent = requester->SendDlvLog(request, pstat, logger);
          }
          if (requester2) {
              logSent = requester2->SendDlvLog(std::move(request), pstat, logger);
          }

          if (logRegim == 1 || (logRegim == 2 && !logSent)) {
              DeliveryLog << pstat;
          }
        }, ThreadPool);
    }

    const auto mlRow = TStringBuilder{} << m_cur->MlLogBuilder << '\n';
    m_p_ml_logger->spprintlen(mlRow.c_str(), mlRow.size());
    mlLog << m_cur->MlLogBuilder.JsonLog << '\n';
} catch(...) {
    m_cur->Logger << CurrentExceptionMessageWithBt();
}

void TRengine::InitHeader() {
    // is not processed enveloped messages
    //    m_alg->InitMessage();
    ++m_cur->m_cHeaders;
    if (m_cur->m_cHeaders > 1) {
        char str[str_short_size];
        snprintf(str, sizeof(str), "new header m_cHeaders = %d - temporary ", m_cur->m_cHeaders);
        SET_STR_NULL(str);
        m_pstat.AddStat(ST_LOG) << str;
    }
}

void TRengine::CheckField(TStringBuf fieldname, TStringBuf field, bool fInternalField) {
    char str[str_large_size];

    snprintf(str, sizeof(str), "fieldlen_%.*s", int(fieldname.size()), fieldname.data());
    SET_STR_NULL(str);
    CheckRange(str, (ui64)field.size(), false);
    CheckRange("fieldlen_all", (ui64)field.size(), false);

    if (field.size() >= str_large_size - 24) {
        snprintf(str, sizeof(str), "long field: %zu %.*s", field.size(), int(fieldname.size()), fieldname.data());
        SET_STR_NULL(str);
        m_pstat.AddStat(ST_LOG) << str;
        field = field.substr(0, str_large_size - 25); // !!! protection from long fields
    }

    const TString sFldname = to_lower(TString{fieldname});
    TSpFields fid = FD_UNKNOWN;
    const bool fFidExist = TryFromString<TSpFields>(sFldname, fid);

    // generate field MESSAGEID
    // Field Message-Id:
    if (TRulesHolder::m_pcre->Check("is_messageid", fieldname))
        CheckField(FD_MESSAGEID, field);

    if (TRulesHolder::m_pcre->Check("is_list", fieldname))
        CheckField(FD_LIST_, field, fid != FD_LIST_ && fid != FD_LIST_POST && fid != FD_LIST_OWNER && fid != FD_LIST_SUBSCRIBE);

    if (fFidExist)
        CheckField(fid, field);

    fInternalField = fInternalField ||
                     sFldname.StartsWith("x-yandex-") ||
                     sFldname.StartsWith("iy-") ||
                     sFldname.StartsWith("x-skipped-") ||
                     sFldname.StartsWith("x-original-size");

    if (!m_cur->m_mapCurrentFields.contains(sFldname))
    {
        if (!fInternalField &&
            !sFldname.Contains(':') &&
            (fFidExist || sFldname.length() < 256) &&
            fid != FD_RAWSUBJECT &&
            m_cur->m_mapCurrentFields.size() < SP_CURRENT_FIELDS &&
            m_cur->m_sCurrentFields.Size() < 512)
            m_cur->m_sCurrentFields << TTextAdder{sFldname, ' '};
        m_cur->m_mapCurrentFields.emplace(sFldname, fid);
    }
}

void TRengine::CheckDomain(TStringBuf sdomain, bool bExtractDomain) {
    if (!sdomain)
        return;

    if (bExtractDomain) {
        sdomain = GetDomain(sdomain);
        sdomain.ChopSuffix(">");

        // if bExtractDomain set, then append domain to url list for reputation check; called for messageID fld
        if(sdomain.size() > 1)
            Add2UrlReputationList(sdomain, EUrlStaticticSource::RCVDS);
    }

    m_pRulesHolder->m_pListRuler.CheckDomainName(m_cur->rulesContext, sdomain, SP_LIST_DOMAIN, SP_LIST_DOMAIN_PART);

}

void TRengine::CheckLevensteinRules(TSpFields /*fid*/, const char* field, int fieldlen) {
    if (!fieldlen)
        return;

    TString sDomain, sMasterDomain;
    TString sStr;
    std::set<TString> keysDone;

    TStringBuf sExtractedDomain = GetDomain({field, size_t(fieldlen)});
    if (!sExtractedDomain.empty() && sExtractedDomain.EndsWith('>')) // if message-id, then remove closing bracket
        sExtractedDomain.Chop(1);

    sDomain.assign(sExtractedDomain);
    ToLower(sDomain.begin(), sDomain.size());
    sMasterDomain = m_common_zone_detector.GetCommonZone(sDomain.c_str());
    auto lastDot = sMasterDomain.find_last_of(".");
    if (lastDot != TString::npos) {
        sMasterDomain.erase(lastDot); // truncate full domain to sensitive form: yandex.com to yandex
    }

    auto rec = m_pstat.AddStat(ST_LEVENSTEIN);
    for (const THolder<TRuleDef>& ruleHolder : m_pRulesHolder->GetRules()) {
        if (ruleHolder->rt == RT_LEVENSTEIN) // found levenstein-type rule, need check
        {
            const TRangeDef& rule = std::get<TRangeDef>(ruleHolder->rules);
            if (keysDone.find(rule.key) == keysDone.end()) // do check once per key
            {
                rec << m_pRulesHolder->m_spruler->CheckLevenstein(*m_cur, rule.key.c_str(), sMasterDomain.c_str());
                keysDone.insert(rule.key);
            }
        }
    }
}

static constexpr bool SubjAllCaps(const TStringBuf& subj) {
    //don't match short subjects
    if (subj.size() < 10)
        return false;

    int cupper = 0, cspace = 0;
    for (char c : subj) {
        switch (GetCaseSymbols(static_cast<ui8>(c))) {
            case SP_UPPER:
                cupper++;
                break;
            case SP_SPACE:
                cspace++;
                break;
            case SP_LOWER:
                return false;
            case SP_DIGIT:
            default:
                break;
        }
    }

    // don't match one word subjects
    if (!cspace)
        return false;

    return cupper >= 3;
}

void TRengine::CheckField(TSpFields fid, TStringBuf field, bool fCancelSaveListField) {
    if (!field)
        return;

    switch (fid) {
        case FD_X_FORM_ID:
            AddPattern(field, EN_SH_X_FORM_ID);
            break;
        case FD_X_SENDER_ACCOUNT:
            AddPattern(field, EN_SH_X_SENDER_ACCOUNT);
            break;
        case FD_DKIM_SIGNATURE: {
            const auto parsedDkim = TDkim::Parse(field);
            AddPattern(TStringBuilder{} << parsedDkim.GetB() << ' ' << parsedDkim.GetDomain() << ' ' << parsedDkim.GetS(), EN_SH_DKIM);
            break;
        }
        case FD_SUBJECT:
            m_html->CheckLingv(field);
            if (SubjAllCaps(field))
                m_cur->rulesContext.SetRule("SUBJ_ALL_CAPS");
            break;
        case FD_RECEIVED:
            if (m_cur->m_cHeaders <= 1)
                m_alg->SetField(fid, field, m_cur->m_cFields, fCancelSaveListField, *this);
            // check re in spalg after spaces deleting
            return;
        case FD_RECEIVED_SPF:
            if (m_cur->m_cHeaders > 1)
                return;
            break;
        case FD_MESSAGE_ID:
            CheckDomain(field, true); // do extract domain from field, add domain to url_rep check
            break;
        case FD_FROM_ADDR:
            CheckLevensteinRules(fid, field.data(), field.size()); // extract From::domain and checkup
            break;
        case FD_X_YANDEX_TIMEMARK:
            if(!m_cur->FieldAlreadyFilled[fid]) {
                CheckFieldAllRules(fid, TInstant::MilliSeconds(FromString<double>(field) * 1000).ToRfc822StringLocal());
            }
            break;
        case FD_X_YANDEX_CF_RECEIPT:
            try {
                auto [receipt, json] = TReceipt::Unbase64AndDecompress(field);
                m_cur->Logger << (TLOG_NOTICE) << "receipt:" << receipt;
                CheckValueAllRules(FD_RECEIPT_VALUE, json);
            } catch (...) {
                m_cur->rulesContext.SetRule("RECEIPT_FAILED");
                m_cur->Logger << (TLOG_ERR) << "failed receipt: " << CurrentExceptionMessageWithBt();
            }
            break;
        default:
            break;
    }

    if (m_cur->m_cHeaders <= 1)
        m_alg->SetField(fid, field, m_cur->m_cFields, fCancelSaveListField, *this);

    CheckFieldAllRules(fid, field);

    m_cur->FieldAlreadyFilled[fid] = true;
}

void TRengine::CheckValueAllRules(TSpFields fid, NJson::TJsonValue value) {
    m_cur->m_rgCurFieldsExists[fid] = true;

    m_cur->FieldsValues[fid].emplace_back(std::move(value));
}

void TRengine::CheckFieldAllRules(TSpFields fid, const TStringBuf& field) {
    m_cur->m_rgCurFieldsExists[fid] = true;
    m_cur->FieldsValues[fid].emplace_back(NJson::TJsonValue(SafeRecode(field)));
    const TTrueAtomicSharedPtr<TRulesMap> rules = this->rules;
    for (const TRuleDef &ruleDef : m_pRulesHolder->RulesByField(fid)) {
        try {
            TRuleCurrent& rule = m_cur->rulesContext.GetRuleCurrentMutable(ruleDef.id);
            if (rule.IsActive())
                continue;

            if(const TTrueAtomicSharedPtr<IRule>* runnableRulePtr = MapFindPtr(*rules, std::addressof(ruleDef))) {
                IRule& runnableRule = **runnableRulePtr;
                auto g = Guard(m_cur->reProf);
                auto g1 = Guard(rule.GetProfiler());
                if (runnableRule.Run(field))
                    m_cur->rulesContext.WorkedRule(rule);
            }
        } catch (...) {
            m_cur->Logger << (TLOG_ERR) << "Matching error " << CurrentExceptionMessageWithBt();
        }
    }

    auto callback = [context = &m_cur->rulesContext](unsigned int id, Y_DECLARE_UNUSED unsigned long long from, Y_DECLARE_UNUSED unsigned long long to){
        context->WorkedRule(id);
    };
    {
        auto g = Guard(m_cur->reProf);
        if (auto db = MapFindPtr(m_pRulesHolder->HsDbsByField, fid)) {
            NHyperscan::Scan(*db, *HsScratch, field, callback);
        }
    }
}

void TRengine::CheckLuaInitMessage() {
    for(const TRuleDef& ruleDef : m_pRulesHolder->RulesByField(FD_INIT_MESSAGE)) {
        if (ruleDef.rt != TRulesType::RT_LUA) {
            continue;
        }
        TLuaRule *rule = MapFindPtr(*LuaRules, std::addressof(ruleDef));
        if (!rule) {
            continue;
        }
        if(auto error = rule->RunMultiArgs({})) {
            m_pstat.AddStat(FD_INIT_MESSAGE) << rule->GetFuncName() << ':' << error->Message;
        }
    }
}

void TRengine::CheckLuaRules() {
    auto g = Guard(m_cur->reProf);
    for(const TRuleCurrent& ruleCurrent : m_cur->rulesContext.GetRules()) {
        if (ruleCurrent.IsActive() || ruleCurrent.GetDef().rt != TRulesType::RT_LUA)
            continue;
        TLuaRule *rule = MapFindPtr(*LuaRules, std::addressof(ruleCurrent.GetDef()));
        if (!rule) {
            continue;
        }

        const TVector<TSpFields> &fields = ruleCurrent.GetDef().pfields;

        TVector<size_t> limits(Reserve(fields.size()));
        {
            bool allFieldsEmpty = true;
            for (const TSpFields field : fields) {
                limits.emplace_back(m_cur->FieldsValues[field].size());
                if(limits.back()) {
                    allFieldsEmpty = false;
                }
            }
            if(allFieldsEmpty) {
                continue;
            }
        }

        TMultiCounter multiCounter(limits);
        do {
            TVector<NJson::TJsonValue> values(Reserve(fields.size()));
            const TVector<size_t> &valuesIndexes = multiCounter.GetCounters();
            for (size_t fieldIndex = 0; fieldIndex < fields.size(); fieldIndex++) {
                const TVector<NJson::TJsonValue> &fieldsValues = m_cur->FieldsValues[fields[fieldIndex]];
                if (fieldsValues) {
                    values.emplace_back(fieldsValues[valuesIndexes[fieldIndex]]);
                } else {
                    values.emplace_back(NJson::TJsonValue(NJson::JSON_NULL));
                }
            }
            if(auto error = rule->RunMultiArgs(values)) {
                m_pstat.AddStat("lerr") << rule->GetFuncName() << ':' << error->Message;
            }
        } while (multiCounter.Next());
    }
}

void TRengine::CheckRulesByfieldsExists() {
    for (auto [fid, rid] : m_pRulesHolder->GetFieldExistsVector()) {
        if (m_cur->m_rgCurFieldsExists[fid]) {
            if (m_pRulesHolder->RuleTypeById(rid) == RT_PRESENT)
                m_cur->rulesContext.WorkedRule(rid);
        } else if (m_pRulesHolder->RuleTypeById(rid) == RT_ABSENT)
            m_cur->rulesContext.WorkedRule(rid);
    }
}

void TRengine::ProcessingAlias(const THashMap<TString, ui8>& aliases, const TString& name, const char* prefix, const char* rule, IOutputStream& stream) {
    auto it = aliases.find(name);
    if (it != aliases.end()) {
        if (rule != nullptr)
            m_cur->rulesContext.SetRule(rule);
        stream << ' ' << prefix << static_cast<ui32>(it->second);
    }
}

bool TRengine::CheckUrlReputation(bool up) try{
    if (m_cur->m_url_reputation_vec.empty())
        return false;

    const bool isSpam = IsSpam(TSpClass::SPAM);
    for (auto& item: m_cur->m_url_reputation_vec) {
        item.isSpam = isSpam;
    }

    auto result = Pools->UrlRepRequester->Get(m_cur->m_url_reputation_vec, m_cur->Logger);
    if(up) {

        const auto urlInfos = NFuncClient::GetUrlInfo(const_cast<const TUrlStatisticVector&>(m_cur->m_url_reputation_vec), false).first;
        if (urlInfos) {
            auto urlPutRequest = NFuncClient::TUrlPutRequest(urlInfos);
            NThreading::Async([urlPutRequest,
                               logger = m_cur->Logger,
                               requester = Pools->UrlRepRequester.Get()](){
                requester->Put(urlPutRequest, logger);
            }, ThreadPool);
        }
    }
    if (!result) {
        m_cur->Logger << TLOG_WARNING << "url_reputation http error: " << result;
        m_cur->rulesContext.SetRule("URLREP_FAIL");
        return false;
    }

    bool fRulereset = false;
    for (const auto& item : m_cur->m_url_reputation_vec) {
        const auto& info = item.isLink ? item.longUrl : item.checkUrl;

        {
            auto record = m_pstat.AddStat(ST_URL_REP);
            record << info.url << ' ';
            if (item.isLink)
                record << item.checkUrl.url;
            else
                record << Hex(info.shingle_url, 0);

            record << ' ' << info.FirstTimeElapsed() << ' ' << info.LastTimeElapsed() << "  "
                   << info.today.ham << ' ' << info.today.spam << ' ' << "  "
                   << info.today.complaint_ham << ' ' << info.today.complaint_spam << " H  "
                   << info.history.ham << ' ' << info.history.spam << ' ' << "  "
                   << info.history.complaint_ham << ' ' << info.history.complaint_spam << ' '
                   << info.Phishing() << ' ' << info.Virus() << " C:";

            if (item.flags & EUrlStaticticSource::SENDER)
                record << 's';
            if (item.flags & EUrlStaticticSource::RCVDS)
                record << 'r';
            if (item.flags & EUrlStaticticSource::BODY)
                record << 'b';

            ProcessingAlias(info.aliases, "surbl", "S:", nullptr, record);
            ProcessingAlias(info.aliases, "dbl", "D:", "RPUR_DBL", record);
            ProcessingAlias(info.aliases, "malware", "M:", "RPUR_MD", record);
            ProcessingAlias(info.aliases, "malwareaggress", "MA:", "RPUR_MDA", record);
        }


        if (!m_pRulesHolder->m_pListRuler.CheckWord(m_cur->rulesContext, info.host.c_str(), SP_LIST_SKIP_SURBL))
            SetSURBLRules(info.server_surbl);

        CheckDomain(info.host.c_str());

        if (!(item.flags & EUrlStaticticSource::BODY)) // rise URLREP rules for body URLs only
            continue;

        if (info.IsPhishingCBB())
            m_cur->rulesContext.SetRule("RPUR_CBB77");
        if (info.IsPhishingVDirect())
            m_cur->rulesContext.SetRule("RPUR_PH_VD");
        if (info.virus)
            m_cur->rulesContext.SetRule("RPUR_VIR_VD");

        int total = 0;
        fRulereset = true;
        int total_hist = (int)(info.history.ham + info.history.spam);
        if (item.isLink) {
            m_html->CheckUrl(this, info.url, TSpHtml::URI_HTML, true, '1');

            total = (int)(info.today.ham + info.today.spam);

            if (info.today.ham > 50)
                CheckRange("repshorturl_compl_spam_perc", (int)((100 * info.today.complaint_spam) / info.today.ham), false);

            if (info.history.ham > 100)
                CheckRange("repshorturl_hist_compl_spam_perc", (int)((100 * info.history.complaint_spam) / info.history.ham), false);

            if (total_hist >= 1000) {
                m_cur->m_mapValueFactors4Matrixnet["MNF_UR_SP"] = (info.history.spam * 1000.) / total_hist;
                int perc = (info.history.spam * 1000) / total_hist;

                if (total_hist < 100000)
                    CheckRange("repurl_hist_1000", perc, false);
                else
                    CheckRange("repurl_hist_100000", perc, false);
            }
        } else // long url check
        {
            m_cur->m_mapValueFactors4Matrixnet["MNF_UR_AGE"] = info.FirstTimeElapsed();
            CheckRange("repurl_age", (int)(info.FirstTimeElapsed()), false);

            int spam = 0;

            decltype(m_cur->m_mapurl_rep_shingle)::const_iterator it = m_cur->m_mapurl_rep_shingle.find(ShingleToStroka(info.shingle_url));

            if (m_cur->m_mapurl_rep_shingle.cend() != it) {
                const auto & url_count = it->second;
                total = (int)(url_count.first + url_count.second);
                spam = (int)(url_count.second);
            } else {
                total = (int)(info.today.ham + info.today.spam);
                spam = (int)(info.today.spam);
            }

            if (total > 100) {
                int perc = (spam * 1000) / total;

                if (info.FirstTimeElapsed() <= 3)
                    CheckRange("repurl_0_3_perc", perc, false);
                else if (info.FirstTimeElapsed() <= 30)
                    CheckRange("repurl_3_30_perc", perc, false);
                else
                    CheckRange("repturl_30_perc", perc, false);
            }

            if (total_hist >= 1000) {
                m_cur->m_mapValueFactors4Matrixnet["MNF_UR_SP"] = (info.history.spam * 1000.) / total_hist;

                int perc = (info.history.spam * 1000) / total_hist;
                if (total_hist < 100000)
                    CheckRange("repurl_hist_1000", perc, false);
                else
                    CheckRange("repurl_hist_100000", perc, false);
            }
        }

        if (info.today.spam > 10) {
            m_cur->m_mapValueFactors4Matrixnet["MNF_UR_CSP"] = 100. * info.today.complaint_ham / info.today.spam;
            CheckRange("repurl_compl_ham_perc", (int)((100 * info.today.complaint_ham) / info.today.spam), false);
        }

        if (info.today.ham > 50) {
            m_cur->m_mapValueFactors4Matrixnet["MNF_UR_CHP"] = (100. * info.today.complaint_spam) / info.today.ham;
            CheckRange("repurl_compl_spam_perc", (int)((100 * info.today.complaint_spam) / info.today.ham), false);
        }

        if (info.history.spam > 100) {
            m_cur->m_mapValueFactors4Matrixnet["MNF_UR_CHP_H"] = (100. * info.history.complaint_ham) / info.history.spam;
            CheckRange("repurl_hist_compl_ham_perc", (int)((100 * info.history.complaint_ham) / info.history.spam), false);
        }

        if (info.history.ham > 100) {
            m_cur->m_mapValueFactors4Matrixnet["MNF_UR_SHP_H"] = (100. * info.history.complaint_spam) / info.history.ham;
            CheckRange("repurl_hist_compl_spam_perc", (int)((100 * info.history.complaint_spam) / info.history.ham), false);
        }

        if (info.FirstTimeElapsed() <= 3) {
            m_cur->m_mapValueFactors4Matrixnet["MNF_UR_C3"] = total;
            CheckRange("repurl_0_3", total, false);
        } else if (info.FirstTimeElapsed() <= 30) {
            m_cur->m_mapValueFactors4Matrixnet["MNF_UR_C30"] = total;
            CheckRange("repurl_3_30", total, false);
        } else {
            m_cur->m_mapValueFactors4Matrixnet["MNF_UR_CM"] = total;
            CheckRange("repurl_30", total, false);
        }
    }

    return fRulereset;
} catch(...) {
    m_cur->Logger << (TLOG_ERR) << __FUNCTION__ << ' ' << CurrentExceptionMessageWithBt();
    return {};
}

TClassifiersContext TRengine::MakeClassifiersContext() const {
    TVector<TString> rulesNames(Reserve(m_cur->rulesContext.GetOccuredRules().size() + m_cur->rulesContext.GetCanceledRules().size()));
    for(const TRulesContext::TRulesRefs& rules : {std::cref(m_cur->rulesContext.GetOccuredRules()), std::cref(m_cur->rulesContext.GetCanceledRules())}) {
        for (const TRuleCurrent &rule : rules) {
            rulesNames.emplace_back(rule.GetDef().pRuleName);
        }
    }

    return TClassifiersContext{std::move(rulesNames), m_cur->m_mapValueFactors4Matrixnet, m_cur->m_factorsAddons};
}

void TRengine::CheckTextClassifier(const TClassifiersContext& context, TString& sMNLabels) {
    TStringStream logStream;
    NJson::TJsonValue jsonLog;

    IModelApplier::TFeaturesMap featuresMap = context.Features;

    for(const auto& ruleName : context.RulesNames)
        featuresMap.emplace(ruleName, 1.f);

    THashMap<ui64, TVector<double>> modelsResultsCache;

    for (const auto &[modelInfo, model] : *Models) {
        try {
            auto addonMapPtr = MapFindPtr(context.Addons, modelInfo.slot);

            const auto features = addonMapPtr ?
                                  model->dict.MakeFeatures({std::cref(*addonMapPtr), std::cref(featuresMap)}) :
                                  model->dict.MakeFeatures({std::cref(featuresMap)});

            const ui64* formulaId = modelInfo.GetFormulaId().Get();

            const TVector<double> *cached = formulaId ? MapFindPtr(modelsResultsCache, *formulaId) : nullptr;
            const TVector<double> &modelResult = cached ? *cached : model->Apply(features);

            if (formulaId && !cached) {
                modelsResultsCache.emplace(*formulaId, modelResult);
            }

            if (!modelResult) {
                m_cur->Logger << (TLOG_ERR) << "zero answer for model " << modelInfo;
                continue;
            }

            if (modelResult.size() == 1) {
                const double actualThreshold = [rulesContext = &m_cur->rulesContext,
                        slot = modelInfo.slot,
                        threshold = modelInfo.threshold]() {
                    if (const TRuleCurrent *rule = rulesContext->FindRule(to_upper(slot) + "_THRESHOLD");
                            rule && rule->IsActive()) {
                        return rule->GetScore();
                    }
                    return threshold;
                }();

                const bool mnResolution = (modelResult[0] >= actualThreshold);

                logStream << modelInfo.briefName << ": "
                          << modelInfo.formulaId << ' '
                          << modelInfo.resourceId << ' '
                          << (mnResolution ? 1 : 0) << ' '
                          << modelResult[0] << ' '
                          << modelInfo.threshold << ' '
                          << actualThreshold << ';';
                {
                    NJson::TJsonValue &jsonLogItem = jsonLog.InsertValue(modelInfo.briefName, {});
                    jsonLogItem.InsertValue("fid", modelInfo.formulaId.GetOrElse(0));
                    jsonLogItem.InsertValue("rid", modelInfo.resourceId);
                    jsonLogItem.InsertValue("res", modelResult[0]);
                    jsonLogItem.InsertValue("thr", modelInfo.threshold);
                    jsonLogItem.InsertValue("act_thr", actualThreshold);
                }

                m_cur->rulesContext.SetRule(
                        to_upper(modelInfo.slot) + (mnResolution ? "_SAYS_SPAM" : "_SAYS_HAM"));

                switch (m_cur->m_messclass) {
                    case TSpClass::HAM:
                    case TSpClass::DLVR:
                        if (mnResolution) // MN detected spam
                        {
                            m_cur->rulesContext.SetRule(to_upper(modelInfo.slot) + "_SPAM");
                            sMNLabels.append("_M" + to_upper(modelInfo.briefName));
                        }
                        break;
                    case TSpClass::SPAM:
                    case TSpClass::MALIC:
                        if (!mnResolution) // MN detected ham
                        {
                            m_cur->rulesContext.SetRule(to_upper(modelInfo.slot) + "_HAM");
                            sMNLabels.append("_M" + to_upper(modelInfo.briefName));
                        }
                        break;
                    default:
                        break;
                }
            } else if (const auto &mapper = modelInfo.OneVsAllMapper) {
                if (auto features = mapper->Map(m_cur->Logger, modelResult)) {
                    auto it = MaxElementBy(features, [](const auto &p) {
                        return p.second;
                    });

                    m_cur->rulesContext.SetRule(it->first);
                    featuresMap[it->first] = 1.0;
                }
            } else {
                m_cur->Logger << (TLOG_WARNING) << modelInfo << ": answer has more then 1 features: "
                                     << MakeRangeJoiner(",", modelResult) << ", but there is no mappers";
            }
        } catch (...) {
            m_cur->Logger << (TLOG_ERR) << "fail to cacl model " << modelInfo << ':' << CurrentExceptionMessageWithBt();
        }
    }

    if (logStream.Size() > 3) {
        m_pstat.AddStat(ST_RECOGNIZER) << logStream.Str();
        m_cur->MlLogBuilder.Add(ToString(ST_RECOGNIZER), std::move(logStream.Str()));
    }

    m_cur->rulesContext.Freeze();
}


void TRengine::ProceedActivity(const TActivityShingleRequestVector& shActList, TStringBuf sFromNormal, TStringBuf sFromDomain) {

    char str[DATE_BUF_LEN];
    constexpr int inactivePeriod = 90;
    int inactiveRcptos = 0;
    TString sLogBuf;

    const time_t today = time(0);
    bool empty = true;
    for (const auto &act : shActList) {
        rcptattrs* attr;
        {
            auto it = FindIf(m_cur->m_rcptattrs.begin(), m_cur->m_rcptattrs.end(), [uid = ToString(act.Uid())](const auto & p){
                return p.second.sUid == uid;
            });
            if(it == m_cur->m_rcptattrs.end()) {
                continue;
            }
            attr = &it->second;
        }

        empty = false;

        const auto & p_uid = attr->sUid;
        ui32 activityDT = 1440622800; // activity DB creation date, `date -d2015-08-27 +%s`
        if (act.Empty()) {
            m_cur->rulesContext.SetRule("ACTIVITY_EMPTY");
        } else {

            activityDT = Max(
                    activityDT,
                    act.GetMobile(),
                    act.GetSpamReport(),
                    act.GetWmi(),
                    act.GetMailboxOper(),
                    act.GetDaria(),
                    act.GetLiza(),
                    act.GetTouch(),
                    act.GetLite(),
                    act.GetMobileIOs(),
                    act.GetMobileAndroid(),
                    act.GetTabletIOs(),
                    act.GetTabletAndroid(),
                    act.GetYServerImap(),
                    act.GetYServerPop(),
                    act.GetHound()
            );

            if (act.GetComplaintsCount()) // days since last complaint
            {
                time_t lastComplDT = time_t(act.GetLasttimeComplaints());
                CheckRange("days_since_compl", (int)(today - lastComplDT) / 3600 / 24);
                CheckRange("days_with_compl", (int) act.GetComplaintDaysCount());
            }
        }

        int daysSinceActive = (today - activityDT) / 3600 / 24;

        attr->SetActivity(daysSinceActive);

        CheckRange("rcpt_active", daysSinceActive);
        DateToString(str, activityDT);
        if (sLogBuf.length())
            sLogBuf.append("; ");
        else
            sLogBuf.append("active at ");
        sLogBuf.append(str);

        if (daysSinceActive >= inactivePeriod)
            inactiveRcptos++;

        m_cur->UidsActivities.emplace(p_uid, daysSinceActive);
    }

    if (empty) {
        m_cur->rulesContext.SetRule("ACTIVITY_EMPTY");
    }

    // do put for inactive only
    AddPattern(sFromNormal, EN_SH_SNDR_TO_INACTIVE, inactiveRcptos > 0, false, inactiveRcptos);
    if (!m_cur->m_isFreemail)
        AddPattern(sFromDomain, EN_SH_DOMN_TO_INACTIVE, inactiveRcptos > 0, false, inactiveRcptos);

    if (sLogBuf.length())
        m_pstat.AddStat(ST_LOG) << sLogBuf;
}

bool TRengine::CheckYaDisk(const TString& answer) try{
    char attrs[256];
    struct tm uiTime;

    NFuncClient::TDiskUserInfo userInfo(answer);
    if (!userInfo.name.empty()) {
        if (userInfo.type == "dir") // ignore folders
            return false;

        if (m_cur->m_sMailFromUID == userInfo.uid)
            m_cur->rulesContext.SetRule("YD_OWNER");
        else {
            if(std::any_of(m_cur->m_rcptattrs.cbegin(), m_cur->m_rcptattrs.cend(), [&userInfo](const auto & p){
                return AsciiCompareIgnoreCase(p.first, userInfo.login) == 0;
            }))
                m_cur->rulesContext.SetRule("YD_OWNER");
        }

        m_cur->rulesContext.SetRule("YDINFO");

        localtime_r(&userInfo.ctime, &uiTime);
        snprintf(attrs, 255, " %02d-%02d-%d %02d:%02d:%02d %d T:%s L:%s %s %s",
                 uiTime.tm_mday, uiTime.tm_mon, uiTime.tm_year + 1900, uiTime.tm_hour, uiTime.tm_min, uiTime.tm_sec,
                 userInfo.size, userInfo.type.c_str(), userInfo.locale.c_str(), userInfo.hid.c_str(), userInfo.login.c_str());

        // decode UTF filename for log  !!! outdated! do not need wide decoding for new disk API
        //                    TUtf16String wtr;
        //                    wtr.AssignAscii (userInfo.name);
        //                    TString str = WideToChar (UnescapeC (wtr), CODES_KOI8);

        TString str = Recode(CODES_UTF8, CODES_KOI8, userInfo.name);

        // handle shingles now
        AddPattern(str.c_str(), EN_SH_ATTACH_NAME);
        if (userInfo.hid.length()) {
            AddPattern(userInfo.hid, EN_SH_FCRC);
            AddPattern(userInfo.hid, EN_SH_ATTACH_MD5);
        }
        CheckField(FD_YADISK_FILE, str); // yadisk filename

        str.append(attrs);
        m_pstat.AddStat(ST_YDISK) << str;

    }
    return true;
} catch(...) {
    m_cur->Logger << (TLOG_ERR) << __FUNCTION__  << ' ' << __LINE__ << ' ' << CurrentExceptionMessageWithBt();
    return {};
}

void TRengine::CheckComplList(TComplaintRequestList* compllist, TComplaintRequestList* correct_compllist, TComplaintRequestList* reset_compllist) {
    TString s_id; // suid or uid
    bool fFirst = true;
    bool fMessageSpam = IsSpam(m_cur->m_messclass);
    TReputationComplaintSet complaintset;
    TReputationComplaint* pComplaint;
    std::vector<TComplaintRequestListIt> v_it_compllist, v_it_resetlist;
    std::vector<TComplaintRequestListIt>::iterator v_it_it_compllist, v_it_it_resetlist;

    memset(&complaintset, 0, sizeof(TReputationComplaintSet));
    for (auto cit = compllist->begin(); cit != compllist->end(); ++cit) {
        if ((*cit).ResEmpty())
            continue;

        if (fFirst) {
            fFirst = false;
            s_id.assign((*cit).Userid());
        } else if (strcmp(s_id.c_str(), (*cit).Userid())) {
            if (CheckUserId(complaintset, s_id.c_str(), fMessageSpam))
                for (v_it_it_compllist = v_it_compllist.begin(); v_it_it_compllist != v_it_compllist.end(); v_it_it_compllist++)
                    correct_compllist->push_back(**v_it_it_compllist);
            else if (!m_cur->rulesContext.IsRuleWorked("UNDO_RESET_COMPLAINTS") && !v_it_resetlist.empty()) {
                for (v_it_it_resetlist = v_it_resetlist.begin(); v_it_it_resetlist != v_it_resetlist.end(); v_it_it_resetlist++)
                    reset_compllist->push_back(**v_it_it_resetlist);
                m_pstat.AddStat(ST_COMPLAINT) << "will be removed pf";
            }
            s_id.assign(cit->Userid());
            v_it_compllist.clear();
            v_it_resetlist.clear();
            memset(&complaintset, 0, sizeof(TReputationComplaintSet));
        }

        v_it_compllist.push_back(cit);

        switch ((*cit).Type()) {
            case KCFROM:
                pComplaint = &(complaintset.From);
                break;
            case KCFROMDOMEN:
                pComplaint = &(complaintset.FromDomain);
                break;
            case KCMESS:
                pComplaint = &(complaintset.Messid);
                break;
            case KCSUBJECT:
                pComplaint = &(complaintset.Subj);
                break;
            case KCBEENSENDER:
                pComplaint = &(complaintset.BeenSender);
                break;
            case KCLIST:
                pComplaint = &(complaintset.List);
                break;
            case KCPSNDR:
                pComplaint = &(complaintset.Paysender);
                break;
            case KCPSNDR_NOFROM:
                pComplaint = &(complaintset.Paysender_Nofrom);
                break;
            default:
                //                sperror ("type reputation error: %d", res);
                continue;
        }

        if (SetComplaint((*cit).Type(), pComplaint, cit, fMessageSpam, cit->Userid()))
            v_it_resetlist.push_back(cit);
    }

    if (CheckUserId(complaintset, s_id.c_str(), fMessageSpam)) // last iteration
        for (v_it_it_compllist = v_it_compllist.begin(); v_it_it_compllist != v_it_compllist.end(); v_it_it_compllist++)
            correct_compllist->push_back(**v_it_it_compllist);
    else if (!m_cur->rulesContext.IsRuleWorked("UNDO_RESET_COMPLAINTS") && !v_it_resetlist.empty()) {
        for (v_it_it_resetlist = v_it_resetlist.begin(); v_it_it_resetlist != v_it_resetlist.end(); v_it_it_resetlist++)
            reset_compllist->push_back(**v_it_it_resetlist);
        m_pstat.AddStat(ST_COMPLAINT) << "will be removed pf";
    }
}

// if true rules must be updated
bool TRengine::CheckReputation() {
    bool fres = false;
    TSpClass filter_res = TSpClass::UNKNOWN;
    TSpClass filter_res_fromto = TSpClass::UNKNOWN;
    //    TSoPersonal *pres_personal;
    //    const char *psuid_ft;
    int c_fromto = 0;
    int c_mails_from = 0;
    //    TLSInfoList lrep_info;
    TKMailType mail_type = KUNKNOWN, mail_type_to = KINMAIL;
    //    TCurShingle *pcur_shingle;
    const TString& pFrom = m_alg->GetFromAddr();
    TEMailFunction email_func;
    TStringBuf sFromNormal = email_func.GetFirstEMailAddress(pFrom);
    const char* pReturnPath = m_alg->GetReturnPath();
    TStringBuf sReturnPath = email_func.GetFirstEMailAddress(pReturnPath);
    const char* pSender = m_cur->rulesContext.IsRuleWorked("__IS_FORWARD") ? "" : m_alg->GetFieldSender();
    const TStringBuf pTo = m_alg->GetToAddr();
    const char* pHost = 0;
    const char* pIp = 0;
    const char* const TrustedHost = "trustedzone.trustedzone";
    const char* pSubject = m_alg->GetSubject();
    const TString& pRealMessageId = m_alg->GetRealMessageId();
    std::list<THostRequest> hostlist;
    std::list<TFromToRequest> fromtolist;
    TString s_shingles, s_complaint;
    TSuid spoppersuid;
    TString sXBeenThereFields;
    TString sMailListField;

    const TFieldListStrings mail_list_fields(m_alg->GetFieldList_(),
                                             m_alg->GetFieldListPost(),
                                             m_alg->GetFieldListOwner(),
                                             m_alg->GetFieldListSubscribe(),
                                             m_alg->GetFieldListAll());

    if ((sFromNormal.length() == 0) && pFrom)
        sFromNormal = pFrom;

    if ((sReturnPath.length() == 0) && pReturnPath)
        sReturnPath = pReturnPath;

    if (m_cur->rulesContext.IsRuleWorked("__POP3_AUTH")) {
        pHost = "pop.pop";
        pIp = "";
    } else if (*(m_alg->GetForwardHost())) {
        pHost = m_alg->GetForwardHost();
        pIp = m_alg->GetForwardIp();
    } else if (Config.fWebMail) {
        pHost = "yandex.ru";
        pIp = "";
    } else {
        pHost = m_alg->GetHostSender();
        pIp = m_alg->GetIpSender();
    }

    if (m_cur->rulesContext.IsRuleWorked("DOMAINRPTN"))
        m_cur->fFromDomainReputation = true;
    if (m_cur->rulesContext.IsRuleWorked("SIMPLIFIEDRPTN"))
        m_cur->fSimplifiedReputation = true;
    if (m_cur->rulesContext.IsRuleWorked("SIMPLIFIEDRPTNMID"))
        m_cur->fSimplifiedReputationMid = true;

    if (*pHost == 0 && (!strcmp(pIp, "0.0.0.0") || *pIp == 0 || *pIp == '-'))
        pHost = TrustedHost;


    if(!m_cur->m_rcptattrs.empty())
    {
        auto it = m_cur->m_rcptattrs.cbegin(); // take the very first adressee and ignore BeenThere, MailList for same addr
        const auto& prcpt_login = it->first;
        sXBeenThereFields = ParseSenderXBeenThereFields(sFromNormal, pSender, m_alg->GetFieldBeenThere(), nullptr);
        if(AsciiCompareIgnoreCase(sXBeenThereFields, pTo) == 0 ||
           AsciiCompareIgnoreCase(sXBeenThereFields, prcpt_login) == 0)
            sXBeenThereFields.clear();

        sMailListField = ParseListFields(sFromNormal, sXBeenThereFields, nullptr, mail_list_fields);
        if ((AsciiCompareIgnoreCase(sMailListField, pTo) == 0) ||
            (AsciiCompareIgnoreCase(sMailListField, prcpt_login) == 0))
            sMailListField.clear();
    }


    switch (m_cur->m_messclass) {
        case TSpClass::HAM:
        case TSpClass::DLVR:
            if (Config.fWebMail) {
                mail_type = KOUTMAILHAM; // for complaints just kIN-OUTmail
                mail_type_to = KINMAILHAM;
            } else
                mail_type = KINMAILHAM;
            filter_res = TSpClass::HAM;
            break;
        case TSpClass::SPAM:
        case TSpClass::MALIC:
            if (Config.fWebMail) {
                mail_type = KOUTMAILSPAM;
                mail_type_to = KINMAILSPAM;
            } else
                mail_type = KINMAILSPAM;
            filter_res = TSpClass::SPAM;
            break;
        default:
            filter_res = TSpClass::UNKNOWN;
            mail_type = (Config.fWebMail) ? KOUTMAIL : KINMAIL;
    }

    TFromStatRequestList statreclist;

    TString sFromBeenList;
    if (sFromNormal.length()) {
        statreclist.emplace_back(sFromNormal, KSFROM, mail_type);
        sFromBeenList.assign(" from=");
        sFromBeenList.append(sFromNormal);
    }
    if (sXBeenThereFields.length()) {
        statreclist.emplace_back(sXBeenThereFields, KSBEENSENDER, mail_type);
        sFromBeenList.append(" been=");
        sFromBeenList.append(sXBeenThereFields);
    }
    if (sMailListField.length()) {
        statreclist.emplace_back(sMailListField, KSLIST, mail_type);
        sFromBeenList.append(" list=");
        sFromBeenList.append(sMailListField);
    }
    if (sFromBeenList.length() == 0)
        sFromBeenList.assign("from-");

    //m_pstat.AddStat(ST_REPUTATION) << "reqst - " << (Config.fWebMail ? 0 : 1) << " " << (int)mail_type << " " << sFromBeenList << " to=" << pTo << " host=" << pHost << " ip=" << pIp << " subj=" << SafeRecode(pSubject);

    if (Config.fWebMail) {
        if (m_cur->m_sMailFromSuid.length() > 0) {
            s_shingles.append("ft=");

            for (auto &it : m_cur->m_rcptattrs) {
                if (it.first.length() == 0)
                    continue;

                // direct shingle, rcpt-mailfromSuid
                fromtolist.emplace_back(TFromToShingle(it.first, m_cur->m_sMailFromSuid, m_cur->m_sMailFrom), mail_type);
                s_shingles.append(" ");
                s_shingles.append(fromtolist.back().sShingle());

                // reverse shingle, mailfrom-rcptSuid
                if (it.second.sSuid.length() && it.second.sSuid[0] != '0')
                    fromtolist.emplace_back(TFromToShingle(m_cur->m_sMailFrom, it.second.sSuid, it.first), mail_type_to);
            }
        }
    }

    const bool fundo_check_from_to = m_cur->rulesContext.IsRuleWorked("UNDO_MCOUNT");
    //    m_cur->m_mapRcpt.SetFirst();
    //    while (m_cur->m_mapRcpt.GetNext(&prcpt_login, &p_suid))
    for (auto &it : m_cur->m_rcptattrs) {
        if (!it.second.sSuid || !it.second.sUid) {
            m_cur->Logger << TLOG_WARNING << "<%s> NO SUID for '%s'" << it.first;
            continue;
        }
        const auto & prcpt_login = it.first;
        auto p_suid = it.second.sSuid;

        if (p_suid.length() == 1 && p_suid[0] == '0') {
            spoppersuid.assign(m_alg->GetPopperSuid());
            if (spoppersuid.length()) // popper
                p_suid = spoppersuid;
            else
                continue;
        }

        s_complaint.append(" suid=");
        s_complaint.append(p_suid);
        s_complaint.append(" login=");
        s_complaint.append(prcpt_login);

        if (m_cur->rulesContext.IsRuleWorked("PSNDR") && m_cur->m_sPaysenderRule.length()) {
            TString tmpPsndr(m_cur->m_sPaysenderRule);
            tmpPsndr.append("pay.sndr");
            m_cur->m_compllist.emplace_back(m_common_zone_detector, tmpPsndr, "", "", p_suid, mail_type, KCPSNDR_NOFROM, "");
            s_complaint.append(" P ");
            s_complaint.append(m_cur->m_compllist.back().sShingle());
        }

        if (sFromNormal.length()) {
            if (m_cur->rulesContext.IsRuleWorked("PSNDR")) {
                m_cur->m_compllist.emplace_back(m_common_zone_detector, "pay.sndr", "", sFromNormal, p_suid, mail_type, KCPSNDR, "");
                s_complaint.append(" Q ");
                s_complaint.append(m_cur->m_compllist.back().sShingle());
            }

            //complaint
            {
                m_cur->m_compllist.emplace_back(m_common_zone_detector, pHost, pIp, sFromNormal, p_suid, mail_type, KCFROM, "");
                s_complaint.append(" F ");
                s_complaint.append(m_cur->m_compllist.back().sShingle());
                m_cur->Logger << (TLOG_DEBUG) << "<" << m_alg->GetQueueID() << "> KCFROM " << pRealMessageId << ' ' << pHost << ' ' << p_suid << ' ' << sFromNormal;
            }

            if (!Config.fWebMail) {
                if (it.second.sUid && it.second.sUid != m_cur->m_sMailFromUID && !fundo_check_from_to) // not check if from == to
                {
                    fromtolist.emplace_back(TFromToShingle(pFrom, p_suid, prcpt_login), mail_type);
                    s_shingles.append(" ft=");
                    s_shingles.append(fromtolist.back().sShingle());
                }
            }
        }

        if (sXBeenThereFields.length() > 0) {
            m_cur->m_compllist.emplace_back(m_common_zone_detector, pHost, pIp, sXBeenThereFields, p_suid, mail_type, KCBEENSENDER, "");
            s_complaint.append(" B ");
            s_complaint.append(m_cur->m_compllist.back().sShingle());
            m_cur->Logger << (TLOG_DEBUG) << "<" << m_alg->GetQueueID() << "> KCBEENSENDER " << pRealMessageId << ' ' << pHost << ' ' << p_suid << ' ' << sXBeenThereFields;
        }

        if (sMailListField.length() > 0) {
            m_cur->m_compllist.emplace_back(m_common_zone_detector, pHost, pIp, sMailListField, p_suid, mail_type, KCLIST, "");
            s_complaint.append(" L ");
            s_complaint.append(m_cur->m_compllist.back().sShingle());
            m_cur->Logger << (TLOG_DEBUG) << "<" << m_alg->GetQueueID() << "> KCLIST " << pRealMessageId << ' ' << pHost << ' ' << p_suid << ' ' << sMailListField;
        }

        if (*pSubject != 0) {
            m_cur->m_compllist.emplace_back(m_common_zone_detector, pHost, pIp, pSubject, p_suid, mail_type, KCSUBJECT, "");
            s_complaint.append(" S ");
            s_complaint.append(m_cur->m_compllist.back().sShingle());
            m_cur->Logger << (TLOG_DEBUG) << "<" << m_alg->GetQueueID() << "> KCSUBJECT " << pRealMessageId << ' ' << pHost << ' ' << p_suid << ' ' << pSubject;
        }

        if (m_cur->fFromDomainReputation) {
            m_cur->m_compllist.emplace_back(m_common_zone_detector, pHost, pIp, GetFromDomen(sFromNormal, m_common_zone_detector), p_suid, mail_type, KCFROMDOMEN, "");
            s_complaint.append(" D ");
            s_complaint.append(m_cur->m_compllist.back().sShingle());
            m_cur->Logger << (TLOG_DEBUG) << "<" << m_alg->GetQueueID() << "> KCFROMDOMEN " << pRealMessageId << ' ' << pHost << ' ' << p_suid << ' ' << GetFromDomen(sFromNormal, m_common_zone_detector);
        }

        if (sReturnPath.length()) {
            m_cur->m_compllist.emplace_back(m_common_zone_detector, pHost, pIp, sReturnPath, p_suid, mail_type, KCRETURNPATH, "");
            s_complaint.append(" R ");
            s_complaint.append(m_cur->m_compllist.back().sShingle());
            m_cur->Logger << (TLOG_DEBUG) << "<" << m_alg->GetQueueID() << "> KCRETURNPATH " << pRealMessageId << ' ' << pHost << ' ' << p_suid << ' ' << sReturnPath;
        }

        const size_t hostMessageIdInd = pRealMessageId.rfind('@');

        if(hostMessageIdInd != TString::npos) {
            TStringBuf hostMessageId{pRealMessageId.c_str() + hostMessageIdInd + 1, pRealMessageId.cend()};
            if (hostMessageId && hostMessageId.Contains('.') && !hostMessageId.Contains("localhost")) {

                hostMessageId.ChopSuffix(">");

                m_cur->m_compllist.emplace_back(m_common_zone_detector, pHost, pIp, hostMessageId, p_suid, mail_type, KCMESS, "");
                s_complaint.append(" M ");
                s_complaint.append(m_cur->m_compllist.back().sShingle());
            }
        }
    }

    //if (s_complaint.length() > 0)
    //    m_pstat.AddStat(ST_COMPLAINT) << s_complaint;

    //if (Config.fWebMail || s_shingles.length() > 4)
    //    m_pstat.AddStat(ST_REPUTATION) << s_shingles;

    bool newPf = false;
    {
        const bool undo = m_cur->rulesContext.IsRuleWorked("UNDO_PERSONAL_FILTER");
        const bool undoBlack = m_cur->rulesContext.IsRuleWorked("UNDO_BLACK_PERSONAL_FILTER");
        const bool undoWhite = m_cur->rulesContext.IsRuleWorked("UNDO_WHITE_PERSONAL_FILTER");
        if (m_cur->NewPf && !undo && (IsSpam(filter_res) ? !undoWhite : !undoBlack)) {
            newPf = m_cur->NewPf(filter_res);
        }
    }

    for (auto& ifs : statreclist)
        if (!ifs.ResEmpty())
            PrintFromStat(ifs, true);

    if ((filter_res == TSpClass::SPAM) && c_fromto) {
        if (Config.fWebMail) {
            if (m_cur->Scores.Hits < m_cur->m_required + 1. + (c_fromto - 1) * 2 + c_mails_from * 2) {
                if (m_cur->rulesContext.IsRuleWorked("__UNDO_MCOUNT"))
                    m_pstat.AddStat(ST_COMPLAINT) << "undo MCOUNT by rule __UNDO_MCOUNT";
                else {
                    m_cur->m_messclass = TSpClass::HAM;
                    m_pstat.AddStat(ST_FROMTO) << "mcount set";
                }
            }
        } else {
            if (m_cur->Scores.Hits < m_cur->m_required + c_fromto) {
                filter_res_fromto = TSpClass::HAM;
                if (m_cur->m_rcptattrs.size() > 1)
                    m_pstat.AddStat(ST_FROMTO) << "mcount set";
                for (auto &it : m_cur->m_rcptattrs)
                    it.second.pres_personal.fromto = TSpClass::HAM;
            } else {
                const bool fAllHam = std::all_of(m_cur->m_rcptattrs.cbegin(), m_cur->m_rcptattrs.cend(), [](const auto& p){
                    return p.second.pres_personal.fromto == TSpClass::HAM;
                });

                if (fAllHam)
                    filter_res_fromto = TSpClass::HAM;
            }
        }
    }

    TString s_suid;
    TComplaintRequestList correct_compllist, reset_compllist, correct_compllist2, reset_compllist2;

    CheckComplList(&m_cur->m_compllist, &correct_compllist, &reset_compllist);   // old ID

    if (m_cur->fUndoMcountByCompl)
        filter_res_fromto = TSpClass::UNKNOWN;

    if (!newPf && SetPersonalresult(filter_res, filter_res_fromto))
        fres = false; // can not undo the result of checking

    return fres;
}

void TRengine::CheckOutmailReputation() {
    if (!Config.fWebMail || !m_cur->rulesContext.IsRuleWorked("YA_PERS_SPAM"))
        return;

    for (const auto&[login, attr] : m_cur->m_rcptattrs) {
        if (attr.sUid) {
            m_cur->UidsResolutions[attr.sUid] = TSpClass::SPAM;
        }
    }
}

bool TRengine::PrepareScore() {
    bool fDelivery = false;

    m_cur->m_sSO_Class.clear();
    m_cur->m_sSO_Class_Short.clear();

    CheckExpressions(0);

    if (m_cur->rulesContext.GetAntiRules()) {
        m_cur->rulesContext.UnsetCanceledRules();

        CheckExpressions(1);
    };

    m_cur->rulesContext.Recalc();

    m_cur->Scores = m_cur->rulesContext.CalcScores();

    CheckRange("score_range", m_cur->Scores.Hits);
    CheckExpressions(1);

    m_cur->rulesContext.Recalc();

    if (m_cur->Scores.Hits >= m_cur->m_required)
        m_cur->m_messclass = TSpClass::SPAM;
    else {
        //        snprintf(str, 255, "CheckExpressions hits = %.1f score = %.1f", m_cur->Scores.Hits, m_cur->m_required);
        //        m_pstat.AddStat(ST_LOG, str);
        if (m_cur->Scores.HitsDlvr < m_cur->m_delivery_required)
            m_cur->m_messclass = TSpClass::HAM;
        else {
            m_cur->m_messclass = TSpClass::DLVR;
        }
    }
    return fDelivery;
}

bool TRengine::IsRedirect(TString sSrcUrl, TString* sInnerUrl) {
    TMaybe<NRegexp::TResult> res;
    if (
            (res = TRulesHolder::m_redir_pcre->Check("liru1", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("liru2", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("liru3", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("mail1", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("mail2", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("vk1", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("subs1", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("63ru", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("yndx1", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("gvom1", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("fedg1", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("wtfm1", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("googl", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("rambl", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("jobs1", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("jstcl", sSrcUrl)) ||
            (res = TRulesHolder::m_redir_pcre->Check("slow1", sSrcUrl))) {

        if (TStringBuf pattern = res->GetPattern(1)) {
            if ((strncasecmp(sSrcUrl.c_str(), "r.mail.ru", 9) == 0) && strcasestr(pattern.data(), ".mail.ru/")) // mailru internal redirect, ignore
                return false;

            m_pstat.AddLimitedStat(ST_HTTP) << "redirect " << sSrcUrl;
            sInnerUrl->assign(pattern);
            return true;
        }
    }

    return false;
}

bool compareWordsAndLangs(TWordAndLang arg1, TWordAndLang arg2) {
    return arg1.wWord > arg2.wWord;
}

bool equalWordsAndLangs(TWordAndLang arg1, TWordAndLang arg2) {
    return arg1.wWord == arg2.wWord;
}

TString TRengine::ActivityRow() const {
    TString result;

    for (auto &attrIt : m_cur->m_rcptattrs) {
        if (!result.empty())
            result.append(";");
        result.append(IntToStroka(attrIt.second.iDaysSinceActive));
    }

    return result;
}

NJson::TJsonValue TRengine::ActivityJson() const {
    NJson::TJsonValue res;
    for (const auto& m_rcptattr : m_cur->m_rcptattrs) {
        res.InsertValue(m_rcptattr.second.sUid, m_rcptattr.second.iDaysSinceActive);
    }
    return res;
}

TString TRengine::GeozoneRow() const {
    TString result;

    for (const auto& m_rcptattr : m_cur->m_rcptattrs) {
        if (!result.empty())
            result.append(";");
        result.append(m_rcptattr.second.sCountry);
    }

    return result;
}

NJson::TJsonValue TRengine::GeozoneJson() const {
    NJson::TJsonValue res;
    for (const auto& m_rcptattr : m_cur->m_rcptattrs) {
        res.InsertValue(m_rcptattr.second.sUid, m_rcptattr.second.sCountry);
    }
    return res;
}

TString TRengine::ExperimentsRow() const {
    if(m_cur->m_rcptattrs.empty())
        return {};

    return JoinSeq(" ", NFuncTools::Map(
            [](const auto& p) -> TString {
                return TStringBuilder{} << p.second.sUid << ':' << p.second.ExpBoxes->RawExpBoxes;
            },
            NFuncTools::Filter([](const auto& p){
                return p.second.sUid && p.second.ExpBoxes;
            }, m_cur->m_rcptattrs)
    ));
}

void TRengine::CheckForSameDomains(const TStringBuf& from, const TStringBuf& mfrm) {
    if (!from || !mfrm)
        return;

    const size_t p_at = from.find('@');
    if (p_at == TStringBuf::npos)
        return;

    const size_t p_mfrm_at = mfrm.find('@');
    if (p_mfrm_at == TStringBuf::npos)
        return;

    if (AsciiEqualsIgnoreCase(from.SubStr(p_at), mfrm.SubStr(p_mfrm_at))) // domains of envelope From and From:addr are same
        m_cur->rulesContext.SetRule("FROM_MFRM_DMN");

    if (AsciiEqualsIgnoreCase(
            m_common_zone_detector.GetCommonZone(from.SubStr(p_at + 1)),
            m_common_zone_detector.GetCommonZone(mfrm.SubStr(p_mfrm_at + 1))))
        m_cur->rulesContext.SetRule("FROM_MFRM_DMN_PART"); // 1st-level domains same
}


static const auto soClassesStatters = MakeTrueConst([] {
    std::array<NUnistat::IHolePtr, so_classes_count> soClassesStatters;

    for (size_t i = 0; i < so_classes_count; i++) {
        soClassesStatters[i] = TUnistat::Instance().DrillFloatHole(
                TStringBuilder{} << so_class[i].class_name, "summ", NUnistat::TPriority{0});
    }
    return soClassesStatters;
}());

static bool KnnDiffs(double so, bool mlSpam) {
    return (so > 9 && so <= 24 && mlSpam) or (so > 9 && so <= 44 && !mlSpam);
}

void TRengine::CheckMessage(NProf::Profiler& prof, const TStringBuf mes, TCheckedMessage& checkedMessage, const TShinglesGroupClass& fmData) {
    auto totalProf = Guard(prof.Prof("all"));

    bool fEmptyBody = false;
    bool fMalicious = false;
    bool fDelivery = false;
    std::vector<TString> v_uids; // uids for Bayesian and personal dicts

    {
        auto g = Guard(prof.Prof("p1"));
        if (m_cur->YandexTeam && m_cur->rulesContext.IsRuleWorked("ALLTRUSTEDIP")) {
            m_cur->rulesContext.SetRule("UNDO_PERSONAL_FILTER");
            m_cur->fPersonalFilterUndo = true;
        }

        if (Config.fCorpMode)
            m_cur->rulesContext.SetRule("CORP");

        if (Config.dumbMode) {
            m_cur->rulesContext.SetRule(TStringBuilder{} << "DUMB_" << Config.dumbMode);
        }

        if (/*Config.fWebMail && */ (m_cur->m_rcptattrs.size() == 1) && !m_cur->m_sMailFrom.empty() && !m_cur->m_sMailFromSuid.empty()) {
            auto it = m_cur->m_rcptattrs.cbegin();
            if (m_cur->m_sMailFromSuid == it->second.sSuid)
                m_cur->rulesContext.SetRule("RCPT_MAILFROM_SAME");
        }
    }
    // check for blind rcptos and collect recipients' domains for FML
    bool fHasBlindRcpto{};
    {
        auto g = Guard(prof.Prof("recipients"));
        TString rcptDomains;
        NJson::TJsonValue rcptDomainsJson;

        for(const auto & p : m_cur->m_rcptattrs) {
            const auto & prcpt_addr = p.first;
            size_t rcpt_at_pos = prcpt_addr.find("@");
            const auto & rcptLogin = rcpt_at_pos > 0 ? prcpt_addr.substr(0, rcpt_at_pos) : "";
            const auto & rcptDomain = rcpt_at_pos != TString::npos ? prcpt_addr.substr(rcpt_at_pos + 1) : "";

            if (!m_cur->m_sMailFrom.empty()) {
                if (m_pRulesHolder->m_pListRuler.CheckListNoRuleSet(rcptLogin, SP_LIST_RCPT_LOGIN)) {
                    AddPattern(m_cur->m_sMailFrom.c_str(), EN_SH_BLIND_RCPTO, false); // do not PUT by default, will enable put later
                    fHasBlindRcpto = true;
                }
            }

            if (!rcptDomains.empty())
                rcptDomains += ";";

            if (rcptDomain) {
                rcptDomainsJson.AppendValue(rcptDomain);
                rcptDomains.append(rcptDomain);
            }

        }

        if (!fHasBlindRcpto)
            AddPattern(m_cur->m_sMailFrom, EN_SH_BLIND_RCPTO, false); // always get 39th

        m_cur->MlLogBuilder.Add(ToString(ST_RCP_TO), std::move(rcptDomains));

        if(!m_cur->m_rcptattrs.empty()) {
            m_cur->MlLogBuilder.Add("rcpt_uid",
                                    JoinSeq(";", NFuncTools::Map([](const auto& p){
                                        return p.second.sUid;
                                    }, m_cur->m_rcptattrs)));
        }
    }

    {
        auto g = Guard(prof.Prof("p3"));
        //if (m_pRulesHolder->m_cCorruptRules)
        //    m_pstat.AddStat(ST_CORRUPT) << m_pRulesHolder->m_cCorruptRules;
        m_pstat.AddStat(ST_FIELDS) << m_cur->m_sCurrentFields.Str();

        m_pRulesHolder->m_spruler->CheckRange(*m_cur, "field_to_names", m_cur->cFiledToNames, false);
    }

    {
        auto g = Guard(prof.Prof("blen"));

        const auto body = RE_RNRN->MatchAndGetResult(mes, 1).GetRestPattern();

        if (body.size() < 50) {
            fEmptyBody = AllOf(body, [](char c){
                return c <= 32;
            });
        }

        if (!body)
            m_cur->rulesContext.SetRule("__MESS_NO_BODY");
        else if (!body || fEmptyBody)
            m_cur->rulesContext.SetRule("__MESS_LEN_0");
        else if (body.size() < 10)
            m_cur->rulesContext.SetRule("__MESS_LEN_0_10");
        else if (body.size() < 100)
            m_cur->rulesContext.SetRule("__MESS_LEN_10_100");
        else if (body.size() < 500)
            m_cur->rulesContext.SetRule("__MESS_LEN_100_500");
        else if (body.size() < 1000)
            m_cur->rulesContext.SetRule("__MESS_LEN_500_1000");
        else if (body.size() < 10000)
            m_cur->rulesContext.SetRule("__MESS_LEN_1000_10000");
        else if (body.size() < 20000)
            m_cur->rulesContext.SetRule("__MESS_LEN_10000_20000");
        else if (body.size() < 50000)
            m_cur->rulesContext.SetRule("__MESS_LEN_20000_50000");
        else if (body.size() < 100000)
            m_cur->rulesContext.SetRule("__MESS_LEN_50000_100000");
        else if (body.size() < 1000000)
            m_cur->rulesContext.SetRule("__MESS_LEN_100000_1000000");
        else if (body.size() >= 1000000)
            m_cur->rulesContext.SetRule("__MESS_LEN_MAX");
    }

    {
        auto g = Guard(prof.Prof("p5"));
        // процент "безымянных" адресов типа info@
        m_pRulesHolder->m_spruler->CheckRange(*m_cur, "blind_to", std::max(m_cur->c_blindRcptos, m_cur->c_blindTos), false);

        // количество адресов, не совпавших в rcpt и to
        if (m_cur->m_rcptattrs.size() < m_RcptMax)
            m_pRulesHolder->m_spruler->CheckRange(*m_cur, "rcpto_missmatch", m_cur->c_rcpt_missmatch, false);

        if (m_cur->m_cHeaders > 1) {
            if (m_cur->m_cHeaders == 2)
                m_cur->rulesContext.SetRule("ENVELOPED_2");
            else if (m_cur->m_cHeaders == 3)
                m_cur->rulesContext.SetRule("ENVELOPED_3");
            else
                m_cur->rulesContext.SetRule("ENVELOPED_4");
        }
    }
    {
        auto g = Guard(prof.Prof("p"
                                 "6"));
#ifndef SP_STAT_IP
        m_html->CheckMessage(this);
        m_alg->CheckMessage(*this);
        m_bodypart->CheckMessage(this);
    }

    if (m_cur->m_rgCurFieldsExists[FD_FROM_ADDR] && m_cur->m_rgCurFieldsExists[FD_MAIL_FROM]) {
        auto g = Guard(prof.Prof("p" "7"));
        TString hostZone;

        if (m_cur->m_rgCurFieldsExists[FD_REC_RDNS]) {
            hostZone = m_common_zone_detector.GetCommonZone(m_alg->GetFirstRdns());

            if (hostZone.length() &&
                (hostZone == m_common_zone_detector.GetCommonZoneFromAddr(m_alg->GetFromAddr()) &&
                 (hostZone == m_common_zone_detector.GetCommonZoneFromAddr(m_cur->m_sMailFrom))))
                m_cur->rulesContext.SetRule("HOST_FROM");
        }
        if (m_alg->GetHostFirst()) {
            hostZone = m_common_zone_detector.GetCommonZone(m_alg->GetHostFirst());
            if (hostZone.length() &&
                (hostZone == m_common_zone_detector.GetCommonZoneFromAddr(m_alg->GetFromAddr()) &&
                 (hostZone == m_common_zone_detector.GetCommonZoneFromAddr(m_cur->m_sMailFrom))))
                m_cur->rulesContext.SetRule("HOST_FROM_LAST");
        }

        // now check for same domains
        CheckForSameDomains(m_alg->GetFromAddr(), m_cur->m_sMailFrom);
    }

    if (Config.fWebMail) {
        auto g = Guard(prof.Prof("p" "9"));
        if (m_cur->c_rcpt - m_cur->c_InternalPdd > 1) {
            if (m_cur->m_sMailFromUID.length())
                AddPattern(m_cur->m_sMailFromUID, EN_SH_MAILFROM_UID, true, false, m_cur->c_rcpt - m_cur->c_InternalPdd);
            else
                AddPattern(m_cur->m_sMailFrom, EN_SH_MAILFROM_UID, true, false, m_cur->c_rcpt - m_cur->c_InternalPdd);
        }

        if (m_cur->m_sMailFromUID.length()) {
            // uniq geozones for sender suid
            TString sKey = m_alg->GetGeoZone();
            if (sKey.length()) {
                sKey.insert(0, "_");
                sKey.insert(0, m_cur->m_sMailFromUID);
                AddPattern(sKey, EN_SH_GEO_UID_UNIQ);
                AddPattern(m_cur->m_sMailFromUID, EN_SH_GEO_UID);
            }

            for(const auto & [login, _] : m_cur->m_rcptattrs) {
                sKey.assign(m_cur->m_sMailFromUID);
                sKey.append("_");
                sKey.append(login);
                AddPattern(sKey, EN_SH_MAILFROM_UID_UNIQ);
                AddPattern(m_cur->m_sMailFromUID, EN_SH_UID_RCPT);
            }

            if (m_alg->GetMessageId() && strlen(m_alg->GetMessageId())) // shingle 36 put for outgoing uid+msgid
            {
                sKey.assign(m_cur->m_sMailFromUID);
                sKey.append("_");
                sKey.append(m_alg->GetMessageId());
                AddPattern(sKey, EN_SH_UID_MSGID);
            }
        }
    } else {
        auto g = Guard(prof.Prof("p" "10"));
        if (m_cur->m_shingle36source.length()) // shingler 36 get for incoming uid+msgid from attached msg
            AddPattern(m_cur->m_shingle36source, EN_SH_UID_MSGID, false);

        // 47 shingle for imap spam/ham folders count per sender
        AddPattern(m_alg->GetFromAddr(), EN_SH_RPOP_MAIL_GMAIL, false);
    }


    bool fFixedSource = m_alg->IsFixedSource();
    bool fFixedSourceLevel1 = fFixedSource && m_cur->rulesContext.IsRuleWorked("__FIXED_SOURCE_LEVEL_1");
    {
        auto g = Guard(prof.Prof("p" "checcs"));
        m_html->CheckCS(this);
    }

    if(m_cur->m_sMailFromUID && m_cur->fieldsById[TSpFields::FD_FROM_NAME].Defined()) {
        const TString& fromName = *m_cur->fieldsById[TSpFields::FD_FROM_NAME];

        AddPattern(m_cur->m_sMailFromUID + "_" + fromName, EN_SH_FROMNAME_UID_UNIQ);
        AddPattern(m_cur->m_sMailFromUID, EN_SH_FROMNAME_UID);
    }

    ;

    const TStringBuf sFromNormal = TEMailFunction{}.GetFirstEMailAddressSoft(m_alg->GetFromAddr());
    const size_t atPos = sFromNormal.find_last_of("@");
    const TStringBuf sDomainNormal = atPos != TString::npos ? sFromNormal.substr(atPos + 1) : sFromNormal;


    if(sDomainNormal)
        AddPattern(sDomainNormal, EN_SH_LOGINS_AT_DOMAIN, false);
    if(sFromNormal)
        AddPattern(sFromNormal, EN_SH_SNDR_TO_INACTIVE, false);
    if (!m_cur->m_isFreemail && sDomainNormal)
        AddPattern(sDomainNormal, EN_SH_DOMN_TO_INACTIVE, false);

    AddPattern(sDomainNormal, EN_SH_FIRST_FROM_DMN, false);

    NThreading::TFuture<TMaybe<NSenderReputation::TGetData>> senderFuture;
    NThreading::TFuture<std::pair<TMaybe<NFreeMail::TInfo>, TMaybe<NFreeMail::TBounceInfo>>> freemailFuture;

    prof.Prof("clients").Start();
    {
        auto g = Guard(prof.Prof("p"
                                 "11"));
#ifndef __SO_CLIENTS__

        if (Pools->FreeMailRequester && Config.fWebMail) {
            freemailFuture = NThreading::Async([
                   FreeMailRequester = Pools->FreeMailRequester,
                   logger = m_cur->Logger,
                   uid = m_cur->m_sMailFromUID]() mutable {
                return GetFreeMailInfoGeneral(
                        NFreeMail::TEmailInfo::UUID(uid),
                        std::move(FreeMailRequester),
                        std::move(logger)
                );
            }, ThreadPool);
        }

        if (Pools->SenderRepRequester && !Config.fWebMail) {
            senderFuture = GetSenderReputationInfo(m_alg->GetFromAddr());
        }
        {
            if(m_cur->So2Context) {
                const auto &matchedTemplate = m_cur->So2Context->GetMatchedTemplate();
                if (!matchedTemplate.Empty()) {
                    SherlockSignDefined->PushSignal(1);
                    const auto &hashesList = matchedTemplate->CalculateHashes();
                    for(const auto& hashes: hashesList) for(const auto& hash: hashes) {
                         m_cur->lsaRequestData.Add(hash, NLSA::TField::Sherlock);
                    }
                    m_cur->MlLogBuilder.Add("sher", NHtmlSanMisc::TMatchedTemplate::HashesAsJson(hashesList));
                    m_cur->MlLogBuilder.Add("sher_id", TStringBuilder{} << matchedTemplate->StableSign);
                    m_cur->StableSign = matchedTemplate->StableSign;
                    m_pstat.AddStat("sher_id") << matchedTemplate->StableSign;
                }
            }
        }

        if (Pools->LSARequester) {
            GetLSA();
        }

        if(m_cur->So2Context) {
            TUserFeaturesProcessor processor(*this);
            processor.Process(*m_cur->So2Context, prof);
        }
    }
#endif

    if(freemailFuture.Initialized()) {
        try {
            if (auto [freemailInfo, bounceInfo] = freemailFuture.ExtractValueSync(); freemailInfo || bounceInfo)
                ProceedFreeMailInfo(NFreeMail::TEmailInfo::UUID(m_cur->m_sMailFromUID), freemailInfo, bounceInfo);
            else
                m_cur->rulesContext.SetRule("FREEMAIL_FAIL");
        } catch (...) {
            m_cur->rulesContext.SetRule("FREEMAIL_FAIL");
            m_cur->Logger << TLOG_ERR << CurrentExceptionMessageWithBt();
        }
    }
    m_cur->m_isFreemail = m_free_mail_hosts.IsFreemailByEmail(m_alg->GetFromAddr());

    if (m_cur->ActivityInfo) {
        const TActivityShingleRequestVector& activityInfo = *m_cur->ActivityInfo;
        if (activityInfo.empty()) {
            m_cur->rulesContext.SetRule("ACTIVITY_EMPTY");
        } else {
            ProceedActivity(activityInfo, sFromNormal, sDomainNormal);
        }
    } else {
        m_cur->rulesContext.SetRule("ACTIVITY_FAIL");
    }

    TCntShinglerResolution cntShinglerResolution{};

    if (Pools->CountRequester && !m_cur->m_newsh.empty()) {
        auto [shList, shAbuseList] = PrepareForHttp2(fmData);

        try {
            if (!Pools->CountRequester->GetNs(shList, m_cur->Logger)) {
                shList = {};
            }
        } catch (...) {
            m_cur->Logger << (TLOG_ERR) << "CountRequester->GetNs:" << CurrentExceptionMessageWithBt();
            shList = {};
        }

        if (Pools->ComplRequester && !shAbuseList.empty()) {
            try {
                if (!Pools->ComplRequester->Get(shAbuseList, m_cur->Logger)) {
                    shAbuseList = {};
                }
                m_cur->ShComplaintRequest.ConstructInPlace(NFuncClient::TCompl::MakePutRequest(shAbuseList, ""));
            } catch (...) {
                m_cur->Logger << (TLOG_ERR) << "ComplRequester->Get:" << CurrentExceptionMessageWithBt();
                shAbuseList = {};
            }
        }
        if (!shList && Pools->CountRequester && !m_cur->m_newsh.empty()) {
            m_cur->MassShinglerFailed = true;
            m_cur->rulesContext.SetRule("SHIN2_FAIL");
        }
        if (!shAbuseList && Pools->ComplRequester)
            m_cur->rulesContext.SetRule("ABUSE_FAIL");
        shList.sort();
        cntShinglerResolution = ProcessCntShinglersResponses(shList, shAbuseList, fHasBlindRcpto);
    }

    if(senderFuture.Initialized()) {
        try {
            if (auto senderRepInfo = senderFuture.ExtractValueSync()) {
                ProceedSenderReputationInfo(*senderRepInfo, m_alg->GetFromAddr(), cntShinglerResolution.sndr_today, cntShinglerResolution.sh14_ham);
                m_cur->fNextPrepareScore = true;
            } else {
                m_cur->rulesContext.SetRule("SENDER_FAIL");
            }
        } catch (...) {
            m_cur->rulesContext.SetRule("SENDER_FAIL");
            m_cur->Logger << TLOG_ERR << CurrentExceptionMessageWithBt();
        }
    }

    prof.Prof("clients").Stop();

    if (Config.fWebMail)
        m_cur->MlLogBuilder.Add("x-yandex-queueid", m_alg->GetQueueID());
    else
        m_cur->MlLogBuilder.Add("rcpt_active", ActivityRow());

    m_cur->MlLogBuilder.Add("test-buckets", ExperimentsRow());
    m_cur->MlLogBuilder.Add("rcpt_zones", GeozoneRow());

    if ((m_cur->c_ShSpam[EN_SH_FIRST_FROM_SNDR] + m_cur->c_ShHam[EN_SH_FIRST_FROM_SNDR]) > 10)
    {
        CheckRange("kuba_sndr_perc", (1000. * (double)(m_cur->c_ShSpam[EN_SH_FIRST_FROM_SNDR] + m_cur->c_ShHam[EN_SH_FIRST_FROM_SNDR]) / (m_cur->c_ShSpam[EN_SH_FROM_ADDR] + m_cur->c_ShHam[EN_SH_FROM_ADDR] + 1)));
        m_cur->m_mapValueFactors4Matrixnet["MNF_S_KUBA"] = 1000. * (double)(m_cur->c_ShSpam[EN_SH_FIRST_FROM_SNDR] + m_cur->c_ShHam[EN_SH_FIRST_FROM_SNDR]) / (m_cur->c_ShSpam[EN_SH_FROM_ADDR] + m_cur->c_ShHam[EN_SH_FROM_ADDR] + 1);
    }

    if (!m_cur->m_isFreemail && ((m_cur->c_ShSpam[EN_SH_FIRST_FROM_DMN] + m_cur->c_ShHam[EN_SH_FIRST_FROM_DMN]) > 10))
    {
        CheckRange("kuba_domn_perc", (1000. * (double)(m_cur->c_ShSpam[EN_SH_FIRST_FROM_DMN] + m_cur->c_ShHam[EN_SH_FIRST_FROM_DMN]) / (m_cur->c_ShSpam[EN_SH_FROM_DOMAIN] + m_cur->c_ShHam[EN_SH_FROM_DOMAIN] + 1)));
        m_cur->m_mapValueFactors4Matrixnet["MNF_D_KUBA"] = 1000. * (double)(m_cur->c_ShSpam[EN_SH_FIRST_FROM_DMN] + m_cur->c_ShHam[EN_SH_FIRST_FROM_DMN]) / (m_cur->c_ShSpam[EN_SH_FROM_DOMAIN] + m_cur->c_ShHam[EN_SH_FROM_DOMAIN] + 1);
    }

    if ((m_cur->c_ShSpam[EN_SH_SNDR_TO_INACTIVE] + m_cur->c_ShHam[EN_SH_SNDR_TO_INACTIVE]) > 10)
    {
        double dValue = 1000. * (double)(m_cur->c_ShSpam[EN_SH_SNDR_TO_INACTIVE] + m_cur->c_ShHam[EN_SH_SNDR_TO_INACTIVE]) / (m_cur->c_ShSpam[EN_SH_FROM_ADDR] + m_cur->c_ShHam[EN_SH_FROM_ADDR] + 1);
        CheckRange("inactive_sndr_perc", dValue);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S_INA"] = dValue;
    }

    if (!m_cur->m_isFreemail && ((m_cur->c_ShSpam[EN_SH_DOMN_TO_INACTIVE] + m_cur->c_ShHam[EN_SH_DOMN_TO_INACTIVE]) > 10))
    {
        double dValue = 1000. * (double)(m_cur->c_ShSpam[EN_SH_DOMN_TO_INACTIVE] + m_cur->c_ShHam[EN_SH_DOMN_TO_INACTIVE]) / (m_cur->c_ShSpam[EN_SH_FROM_DOMAIN] + m_cur->c_ShHam[EN_SH_FROM_DOMAIN] + 1);
        CheckRange("inactive_domn_perc", dValue);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D_INA"] = dValue;
    }

    {
        auto g = Guard(prof.Prof("p" "prepare_sc"));
        PrepareScore(); // will need rules prepared for Sender Reputations
    }

    if (m_cur->rulesContext.IsRuleWorked("__RDSL_OR_NORESOLVED_FIRST") && cntShinglerResolution.set_dsl_first)
        m_cur->rulesContext.SetRule("DSLFIRST");

    // copy body extracted URLs from internal map to urls vector for url-rep query
    {
        auto g = Guard(prof.Prof("p" "isredir"));
        for (auto& [purl, purl_ind] : m_cur->m_mapurl_reputation) {

            if (purl_ind == 0) {
                purl_ind = 1;
                m_cur->AddUrl(purl, {"surbl"});
                TString sRedir;
                if (IsRedirect(purl, &sRedir)) {
                    m_cur->AddUrl(sRedir, {"surbl"});
                    m_cur->rulesContext.SetRule("URL_REDIR");
                }
            }
        }
    }

    if (m_cur->mails > 1)
        CheckRange("boxes_mails", (m_cur->boxes * 100) / m_cur->mails);
#endif

    {
        auto g = Guard(prof.Prof("p" "14"));
        CheckRulesByfieldsExists();
    }

    if (Pools->UrlRepRequester) {
        auto g = Guard(prof.Prof("p" "16"));
        if (CheckUrlReputation(!m_cur->rulesContext.IsRuleWorked("TEST_SO_VERSION") &&
                               !m_cur->rulesContext.IsRuleWorked("SKIP_PUT") &&
                               !m_cur->rulesContext.IsRuleWorked("SKIP_URL_PUT")))
            m_cur->fNextPrepareScore = true;
    }

    CheckLuaRules();

    {
        auto g = Guard(prof.Prof("p" "17"));
        fDelivery = PrepareScore();
    }

    if (m_cur->KnnFuture.Initialized()) try {
            if(auto response = m_cur->KnnFuture.ExtractValueSync(); response && response->Neighbors)
                ProceedKNNSpamness(*response);
        } catch (...) {
            m_cur->Logger << (TLOG_ERR) << "knn: " << CurrentExceptionMessageWithBt();
        }

    if(UidsStats) {
        if(m_cur->m_rcptattrs) {
            const auto & [login, attrs] = *m_cur->m_rcptattrs.cbegin();
            if(attrs.sUid) {
                if(ui64 uid; TryIntFromString<10>(attrs.sUid, uid)) {
                    const TUidStats *uidStats = UidsStats->GetStatsOrNull(uid);
                    if (uidStats) {
                        uidStats->ToFeatures(m_cur->m_mapValueFactors4Matrixnet);
                    }
                } else {
                    m_cur->Logger << (TLOG_ERR) << "cannot parse uid from " << m_cur->messageId << ' ' << attrs.sUid << ' ' << login;
                }
            } else {
                m_cur->Logger << (TLOG_ERR) << "empty uid for " << m_cur->messageId << ' ' << login;
            }
        } else {
            m_cur->Logger << (TLOG_ERR) << "empty rcpts for " << m_cur->messageId;
        }
    }

    TString sMNLabels;
    m_cur->ClassifiersContext = MakeClassifiersContext();
    {
        auto g = Guard(prof.Prof("p" "18"));
        CheckTextClassifier(m_cur->ClassifiersContext, sMNLabels);
        {
            auto g = Guard(prof.Prof("p" "19"));
            fDelivery = PrepareScore();
        }
    }

    if (m_shingler) {
        auto g = Guard(prof.Prof("p" "20"));
        m_shingler->SetReputation(Config.fWebMail);
    }

    if (!(m_cur->rulesContext.IsRuleWorked("USE_GREYLISTING") && m_cur->m_messclass == TSpClass::SPAM)) {
        auto g = Guard(prof.Prof("p" "21"));
        if (CheckReputation() || Config.fWebMail)
            m_cur->fNextPrepareScore = true;
    }

    if (m_cur->fNextPrepareScore) {
        auto g = Guard(prof.Prof("p" "22"));
        fDelivery = PrepareScore();
    }

    if(Config.fWebMail) {
        CheckOutmailReputation();
    }

    if (m_cur->rulesContext.IsRuleWorked("CANCEL_GREYLISTING") || m_cur->rulesContext.IsRuleWorked("PERSONAL_CORRECT"))
        m_cur->fCancelGreyListing = true;

    m_cur->m_sSO_Class.clear();
    m_cur->m_sSO_Class_Short.clear();

    for(const TString& type : LuaRulesContext->GetTypes()) {
        m_cur->m_sSO_Class.emplace(type);
        m_cur->m_sSO_Class_Short << TTextAdder{type.substr(0, 8), ';'};
    }

    for(const auto& [soType, statter]: NFuncTools::Zip(*so_class, *soClassesStatters)) {
        if (m_cur->rulesContext.IsRuleWorked(soType.rule_name)) {
            m_cur->m_sSO_Class.emplace(soType.class_name);
            m_cur->m_sSO_Class_Short << TTextAdder{soType.class_name.substr(0, 8), ';'};

            statter->PushSignal(1);
        }
    }

    if (!Config.fWebMail) {
        if (*(m_alg->GetDomainLabel()) != '\0' && !m_cur->rulesContext.IsRuleWorked("UNDO_LABEL")) {
            const TStringBuf domain = m_alg->GetDomainLabel();
            m_cur->m_sSO_Class.emplace(domain);
            m_cur->m_sSO_Class_Short << TTextAdder{domain.substr(0, 8), ';'};
        }
    }

    if (!m_cur->m_sSO_Class.empty()) {
        const TString strClasses = TStringBuilder{} << MakeRangeJoiner(",", m_cur->m_sSO_Class);
        m_pstat.AddStat(ST_LOG) << "cl " << strClasses;
        m_cur->MlLogBuilder.Add("logcl", strClasses);
    }

    if (const auto& antiRules = m_cur->rulesContext.GetAntiRules(); antiRules) {
        m_pstat.AddStat(ST_LOG) << "Anti:" << MakeRangeJoiner(" ", antiRules);
    }

    if (const auto& canceledRules = m_cur->rulesContext.GetCanceledRules(); canceledRules) {
        m_pstat.AddStat(ST_LOG) << "Cancel:" << MakeRangeJoiner(" ", canceledRules);
        m_cur->MlLogBuilder.Add("r_cancel",
                                JoinSeq(";", canceledRules));
    }

    TShingleCount check_res = HAM;
    switch (m_cur->m_messclass) {
        case TSpClass::HAM:
            break;
        case TSpClass::SPAM:
        case TSpClass::MALIC:
            check_res = SPAM;
            break;
        case TSpClass::DLVR:
            check_res = HAM;
            break;
        default:
            m_cur->Logger << TLOG_WARNING << "Unknown message type: " << int(m_cur->m_messclass);
            break;
    }

    if (m_cur->OcrText) {
        m_cur->rulesContext.SetRule("OCR");
        if (IsSpam(m_cur->PrevSpClass)) {
            if (IsSpam(m_cur->m_messclass)) {
                m_cur->rulesContext.SetRule("OCR_SPAM_SPAM");
            } else {
                m_cur->rulesContext.SetRule("OCR_SPAM_HAM");
            }
        } else {
            if (IsSpam(m_cur->m_messclass)) {
                m_cur->rulesContext.SetRule("OCR_HAM_SPAM");
            } else {
                m_cur->rulesContext.SetRule("OCR_HAM_HAM");
            }
        }
    }

    // добавляем шинглы только на запись
    if (!m_cur->rulesContext.IsRuleWorked("USE_GREYLISTING")) {

        auto g = Guard(prof.Prof("p" "23"));
        AddPattern(m_alg->GetIpSender(), EN_SH_IP_BOX);

        if ((m_cur->c_ShSpam[EN_SH_FROM_ADDR] + m_cur->c_ShHam[EN_SH_FROM_ADDR]) == 0) {
            if (m_alg->GetFromAddr()) {
                const TStringBuf& pfrom_addr = m_alg->GetFromAddr();
                const size_t p_at = pfrom_addr.find('@');

                if (p_at != TStringBuf::npos)
                    AddPattern(pfrom_addr.SubStr(p_at + 1), EN_SH_LOGINS_AT_DOMAIN);
            }
        }

        if (m_cur->fPutForward)
            AddPattern(m_alg->GetIpSender(), EN_SH_IP_FWD);
    }
    {
        auto g = Guard(prof.Prof("p" "24"));
        m_pRulesHolder->m_pListRuler.CheckMXZone(m_cur->rulesContext, m_alg->GetIpSender(), m_alg->GetHostSender());
    }

    {
        TStringStream m_sYaml;
        m_sYaml << TTextAdder{ToString(m_cur->c_rcpt), '\t'} << TTextAdder{ToString<int>(m_cur->rulesContext.IsRuleWorked("IMPORTANT_ABUSE")), '\t'};

        m_pstat.AddStat(ST_YAML) << m_sYaml.Str();
        m_cur->MlLogBuilder.Add(ToString(ST_YAML), m_sYaml.Str());
    }

    {
        auto g = Guard(prof.Prof("p" "26"));
        checkedMessage.Report = Report();
    }

    const char* pmesclass;
    switch (m_cur->m_messclass) {
        case TSpClass::SPAM:
            pmesclass = "yes";
            break;
        case TSpClass::HAM:
            pmesclass = "no";
            break;
        case TSpClass::DLVR:
            pmesclass = "dlv";
            break;
        default:
            pmesclass = "";
    }
    {
        if (m_cur->rulesContext.IsRuleWorked("KNN_NSPAM_HOT_SPAM_02") &&
            ::KnnDiffs(m_cur->Scores.Hits, m_cur->rulesContext.IsRuleWorked("MN_PROD_SAYS_SPAM"))) {
            KnnDiffs->PushSignal(1);
            m_pstat.AddStat("knn_diff") << 1;
            m_cur->MlLogBuilder.Add("knn_diff", "1", 1);
        }
    }
    {
        m_pstat.AddStat(ST_WEIGHT) << Prec(m_cur->Scores.Hits, PREC_POINT_DIGITS_STRIP_ZEROES, 2) << ' ' << Prec(m_cur->Scores.HitsDlvr, PREC_POINT_DIGITS_STRIP_ZEROES, 2);
        m_pstat.AddStat(ST_SPAM) << pmesclass;
        m_cur->MlLogBuilder.Add(ToString(ST_WEIGHT), ToString(Prec(m_cur->Scores.Hits, PREC_POINT_DIGITS_STRIP_ZEROES, 2)));

        auto stSpamReportRec = m_pstat.AddStat(ST_REPORT_SPAM);
        auto stReportDlvRec = m_pstat.AddStat(ST_REPORT_DLV);
        auto stReportNullRec = m_pstat.AddStat(ST_REPORT_NULL);
        {
            auto g = Guard(prof.Prof("p" "29"));
            TStringStream reportSpam;

            StableSortBy(m_cur->vr_null.begin(), m_cur->vr_null.end(), [](const TRuleCurrent& r) { return r.GetDef().pRuleName; });
            StableSortBy(m_cur->vr_spam.begin(), m_cur->vr_spam.end(), [](const TRuleCurrent& r) { return -Abs(r.GetScore()); });
            StableSortBy(m_cur->vr_dlv.begin(), m_cur->vr_dlv.end(), [](const TRuleCurrent& r) { return -Abs(r.GetScore()); });

            for(auto& [vr, record] : {
                    std::make_tuple(std::cref(m_cur->vr_spam), std::ref(stSpamReportRec)),
                    std::make_tuple(std::cref(m_cur->vr_dlv), std::ref(stReportDlvRec)),
                    std::make_tuple(std::cref(m_cur->vr_null), std::ref(stReportNullRec)),
            }) {
                for (const TRuleCurrent &rule: vr) {
                    record << rule << ',';
                    reportSpam << rule << ';';
                }
            }

            m_cur->MlLogBuilder.Add(ToString(ST_SPAM), pmesclass);
            m_cur->MlLogBuilder.Add(ToString(ST_REPORT_SPAM), std::move(reportSpam.Str()));
            m_cur->MlLogBuilder.Add("mnf", MNF2String());

#ifndef SP_STAT_IP
            TString s_level;
            if (m_cur->m_messclass == TSpClass::SPAM)
                s_level.assign("SPAM LEVEL ");
            else if (m_cur->m_messclass == TSpClass::HAM)
                s_level.assign("HAM LEVEL ");
            else if (m_cur->m_messclass == TSpClass::DLVR)
                s_level.assign("DELIVERY LEVEL ");
            else
                s_level.assign("UNKNOWN LEVEL ");

            if (fFixedSourceLevel1)
                s_level.append("1");
            else if (fFixedSource)
                s_level.append("2");
            else
                s_level.append("3");

            m_pstat.AddStat(ST_LEVEL) << s_level;
#endif
        }

        if (m_cur->rulesContext.IsRuleWorked("PERSONAL_CORRECT") && m_cur->m_messclass == TSpClass::DLVR || m_cur->fHamPersonalCorrect)
            m_cur->m_messclass = TSpClass::HAM;

        // only for outgoing mails
        if (m_cur->m_messclass == TSpClass::SPAM) {
            if (m_cur->rulesContext.IsRuleWorked("MALICIOUS_SPAMER"))
                m_cur->m_messclass = TSpClass::MALIC;
            //        if (IsRuleWorked("PHISHING_SPAM"))
            //            m_cur->m_messclass |= (int)spPhishing_Spamer; // Sic! make bit mask
        } else if (m_cur->m_messclass == TSpClass::UNKNOWN)
            m_cur->Logger << TLOG_ERR << "mess_res = " << int(m_cur->m_messclass);

        if (fMalicious && m_cur->Scores.Hits > m_cur->m_required && !m_cur->fHamPersonalCorrect)
            m_cur->m_messclass = TSpClass::MALIC;

        stReportNullRec << "R" << (int)m_cur->m_messclass << ", ";
        m_cur->DlvLogRequest.Code = (int)m_cur->m_messclass;
        if(m_cur->rulesContext.IsRuleWorked("PERSONAL_CORRECT")) {
            switch (m_cur->DlvLogRequest.Code) {
                case 1:
                    m_cur->DlvLogRequest.Code = 8;
                    break;
                case 4:
                    m_cur->DlvLogRequest.Code = 127;
                    break;
            }
        }
    }
    TShingleCount m_MailGmailResolution = UNKNOWN;
    if (m_alg->GetMailGmailIMAP()) {
        if (m_cur->rulesContext.IsRuleWorked("SPAM_FOLDER_S"))
            m_MailGmailResolution = SPAM;
        else if (!m_cur->rulesContext.IsRuleWorked("HAM_FOLDER_S"))
            m_MailGmailResolution = HAM;
    }

    // TEMP: remove this later; Delivery type will be removed
    TShingleCount tmp_checkres = check_res;
    if (m_cur->m_messclass == TSpClass::MALIC)
        tmp_checkres = MALIC;
    else if (m_cur->m_messclass == TSpClass::DLVR)
        tmp_checkres = HAM;

    const bool bPutAllowed = !m_cur->rulesContext.IsRuleWorked("__POP3_AUTH") &&
                       !m_cur->rulesContext.IsRuleWorked("SKIP_SH_PUT") &&
                       !m_cur->rulesContext.IsRuleWorked("SKIP_PUT") &&
                       !m_cur->rulesContext.IsRuleWorked("USE_GREYLISTING") &&
                       !m_cur->rulesContext.IsRuleWorked("TEST_SO_VERSION");

    if (Pools->CountRequester && m_cur->m_newsh.size()) {
        auto g = Guard(prof.Prof("p" "31"));

        TCntComplShingleRequestList shAbuseList;

        if (bPutAllowed) {
            TMaybe<TShingleStat> sh19, sh31, sh33, sh35, sh56, sh57, sh58, sh59;

            ui64 sh24value = 0;
            std::vector<TShingleStat> v17th, v30th, v32th, v34th;

            for (const auto& [shingle, type] : m_cur->m_newsh) {
                if (type.bPut || (shingle.second == EN_SH_BLIND_RCPTO && cntShinglerResolution.fPutBlindRcpto)) // force put for 39th when 17th < 2
                {
                    switch (shingle.second) {
                        case EN_SH_IP_RCP_TO:
                            v17th.emplace_back(shingle.first, EN_SH_IP_RCP_TO, type.Count());
                            break;
                        case EN_SH_IP_BOX:
                            sh19.ConstructInPlace(shingle.first, EN_SH_IP_BOX, type.Count());
                            break;
                        case EN_SH_MAILFROM_UID_UNIQ:
                            v30th.emplace_back(shingle.first, EN_SH_MAILFROM_UID_UNIQ, type.Count());
                            break;
                        case EN_SH_UID_RCPT:
                            if (Config.fWebMail)
                                sh31.ConstructInPlace(shingle.first, EN_SH_UID_RCPT, type.Count());
                            break;
                        case EN_SH_PDD_DOMN_UNIQ:
                            v32th.emplace_back(shingle.first, EN_SH_PDD_DOMN_UNIQ, type.Count());
                            break;
                        case EN_SH_UID_PDD_DOMN:
                            if (Config.fWebMail)
                                sh33.ConstructInPlace(shingle.first, EN_SH_UID_PDD_DOMN, type.Count());
                            break;
                        case EN_SH_GEO_UID_UNIQ:
                            v34th.emplace_back(shingle.first, EN_SH_GEO_UID_UNIQ, type.Count());
                            break;
                        case EN_SH_GEO_UID:
                            if (Config.fWebMail)
                                sh35.ConstructInPlace(shingle.first, EN_SH_GEO_UID, type.Count());
                            break;
                        case EN_SH_RPOP_MAIL_GMAIL:
                            m_cur->ShList.emplace_back(shingle.first, shingle.second, 1);
                            break;
                        case EN_SH_FROM_PDD_ADM_UNIQ:
                            if (Config.fWebMail)
                                sh57.ConstructInPlace(shingle.first, EN_SH_FROM_PDD_ADM_UNIQ, type.Count());
                            break;
                        case EN_SH_FROM_PDD_ADMIN:
                            sh56.ConstructInPlace(shingle.first, EN_SH_FROM_PDD_ADMIN, type.Count());
                            break;
                        case EN_SH_FROMNAME_UID:
                            sh58.ConstructInPlace(shingle.first, EN_SH_FROMNAME_UID, type.Count());
                            break;
                        case EN_SH_FROMNAME_UID_UNIQ:
                            sh59.ConstructInPlace(shingle.first, EN_SH_FROMNAME_UID_UNIQ, type.Count());
                            break;
                        case EN_SH_MAILFROM_UID:
                            sh24value = shingle.first;
                        default: // all other types add to common put list
                            m_cur->ShList.emplace_back(shingle.first, shingle.second, type.Count());
                    }
                }
            }

            if(!m_cur->MassShinglerFailed) {
                if (sh19) {
                    for (auto & v17 : v17th)
                        m_cur->ShList.emplace_back(v17.Shingle(), EN_SH_IP_RCP_TO, // shingle1, type
                                                   sh19->Shingle(), EN_SH_IP_BOX,   // shingle2, type2
                                                   v17.GetCount());               // shingle1 count
                }

                if (sh31) // counters for outgoing rcptos from suid
                {
                    for (auto & v30 : v30th)
                        m_cur->ShList.emplace_back(v30.Shingle(), EN_SH_MAILFROM_UID_UNIQ,
                                                   sh31->Shingle(), EN_SH_UID_RCPT,
                                                   v30.GetCount());
                }

                if (sh33) // counters for outgoung rcptos from pdd domain
                {
                    for (auto & v32 : v32th)
                        m_cur->ShList.emplace_back(v32.Shingle(), EN_SH_PDD_DOMN_UNIQ,
                                                   sh33->Shingle(), EN_SH_UID_PDD_DOMN,
                                                   v32.GetCount());
                }

                if (sh35) // counters for outgoing rcptos from geozone
                {
                    for (auto & v34 : v34th)
                        m_cur->ShList.emplace_back(v34.Shingle(), EN_SH_GEO_UID_UNIQ,
                                                   sh35->Shingle(), EN_SH_GEO_UID,
                                                   v34.GetCount());
                }

                if (sh57 && sh56)
                    m_cur->ShList.emplace_back(sh57->Shingle(), EN_SH_FROM_PDD_ADM_UNIQ,
                                               sh56->Shingle(), EN_SH_FROM_PDD_ADMIN,
                                               sh57->GetCount());

                if (sh58.Defined() && sh59.Defined()) {
                    m_cur->ShList.emplace_back(sh59->Shingle(), EN_SH_FROMNAME_UID_UNIQ,
                                               sh58->Shingle(), EN_SH_FROMNAME_UID,
                                               sh59->GetCount());

                }
            }

            if (Config.fWebMail && m_cur->rulesContext.IsRuleWorked("MALIC_ML") && sh24value)                // user limit reached. Will check this for fast reject by 52th
                m_cur->ShList.emplace_back(sh24value, EN_SH_MFRM_UID_LIMIT, 50); // 50 for immediate db update

            try {
                Pools->CountRequester->PutNs(m_cur->ShList, tmp_checkres, m_cur->fHamPersonalCorrect ? PERSHAM : m_cur->fSpamPersonalCorrect ? PERSSPAM : PERSUNDEF, m_cur->Logger);
            } catch(...) {
                m_cur->Logger << (TLOG_ERR) << __FUNCTION__ << ' ' << CurrentExceptionMessageWithBt();
            }
        }

        // separate put for imap folder name 47th
        if (bPutAllowed && (m_MailGmailResolution != UNKNOWN))
        {
            auto m_it = m_cur->m_newsh.begin();
            for (; m_it != m_cur->m_newsh.end(); m_it++) // locate 47 shingle text
                if (m_it->first.second == EN_SH_RPOP_MAIL_GMAIL)
                    break;

            if (m_it != m_cur->m_newsh.end()) try{
                    m_cur->ShList.emplace_back(m_it->first.first, EN_SH_RPOP_MAIL_GMAIL, 1);
                    Pools->CountRequester->PutNs(m_cur->ShList, m_MailGmailResolution, PERSUNDEF, m_cur->Logger);
                } catch(...) {
                    m_cur->Logger << (TLOG_ERR) << __FUNCTION__ << ' ' << CurrentExceptionMessageWithBt();
                }
        }
    }

#ifndef __SO_CLIENTS__

    if (Pools->FreeMailRequester &&
        !m_cur->rulesContext.IsRuleWorked("TEST_SO_VERSION") &&
        !m_cur->rulesContext.IsRuleWorked("SKIP_PUT")) {
        auto g = Guard(prof.Prof("p" "33"));
        PutFreeMailInfoGeneral();
    }

    if (Config.fWebMail) {
        auto g = Guard(prof.Prof("p" "34"));
        if (m_cur->rulesContext.IsRuleWorked("SET_KARMA_2000"))
            CleanUserKarma(m_cur->m_sMailFromUID, "2");
        if (m_cur->rulesContext.IsRuleWorked("UNSET_KARMA_6000"))
            CleanUserKarma(m_cur->m_sMailFromUID, "0");
    }

    if (Pools->SenderRepRequester &&
        !m_cur->rulesContext.IsRuleWorked("SKIP_SENDER_PUT") &&
        !m_cur->rulesContext.IsRuleWorked("TEST_SO_VERSION") &&
        !m_cur->rulesContext.IsRuleWorked("SKIP_PUT")) {
        auto g = Guard(prof.Prof("p" "35"));
        PutSenderReputationInfo(IsSpam(m_cur->m_messclass));
    }

#endif

    auto g = Guard(prof.Prof("p42"));
    // ShortLog
    Print2Shortlog(std::move(sMNLabels));

    checkedMessage.spClass = m_cur->m_messclass;
}

void TRengine::ProceedKNNSpamness(const NSoKNN::TGetNeighborsResponse& response) {
    TStringStream distancesStr;

    for(const auto& [ns, neighbours]: response.Neighbors) {
        double simSum = {};
        double weightedSum = {};
        TMaybe<double> minDistanceToSpam, minDistanceToHam;
        for (const auto& n : neighbours) {
            const auto& isSpam = n.Value["spam"].GetBooleanSafe();

            const double sim = 1 - n.Distance;

            if (isSpam) {
                weightedSum += sim;

                if (!minDistanceToSpam)
                    minDistanceToSpam = n.Distance;
            } else {
                if (!minDistanceToHam)
                    minDistanceToHam = n.Distance;
            }
            simSum += sim;

            distancesStr << ' ' << n.Distance << ':' << int(isSpam);
        }


        const double spamness = weightedSum / simSum;

        m_cur->Logger << (TLOG_INFO) << ns << " size:" << neighbours.size()
                          << " knn_spam_perc:" << spamness
                          << " distances:" << distancesStr.Str();

        m_cur->MlLogBuilder.Add("knn_spam_perc_" + ns, TStringBuilder{} << spamness);
        m_pstat.AddStat("knn_" + ns) << "perc:" << spamness << ' ' << neighbours.size() << ";ch:" << minDistanceToHam << ";cs:" << minDistanceToSpam;


        CheckRange("knn_spam_" + ToLowerUTF8(ns), spamness);
        m_cur->m_mapValueFactors4Matrixnet["MNF_KNN_SPAMNESS_" + ToUpperUTF8(ns)] = spamness;

        if(minDistanceToHam) {
            CheckRange("knn_nham_" + ToLowerUTF8(ns), *minDistanceToHam);
            m_cur->m_mapValueFactors4Matrixnet["MNF_KNN_NEAREST_HAM_" + ToUpperUTF8(ns)] = *minDistanceToHam;
        }

        if(minDistanceToSpam) {
            CheckRange("knn_nspam_" + ToLowerUTF8(ns), *minDistanceToSpam);
            m_cur->m_mapValueFactors4Matrixnet["MNF_KNN_NEAREST_SPAM_" + ToUpperUTF8(ns)] = *minDistanceToSpam;
        }
    }
}

void TRengine::PrepareBodyDataForVw(const TString& body, ELanguage langCode) {
    for (auto w : NText2Shingles::Text2Shingles(body, langCode, true))
        m_cur->lsaRequestData.Add(std::move(w), NLSA::TField::Body);
}

void TRengine::PrepareSubjDataForVw(const TString& koi8subject) {
    const TString utf8subject = Recode(CODES_KOI8, CODES_UTF8, koi8subject);

    for(auto w : NText2Shingles::Text2Shingles(utf8subject, LANG_UNK, true))
        m_cur->lsaRequestData.Add(std::move(w), NLSA::TField::Subject);
}

void TRengine::PrepareFromNameDataForVw(const TString& koi8FromName) {
    const TString utf8FromName = Recode(CODES_KOI8, CODES_UTF8, koi8FromName);

    for(auto w : NText2Shingles::Text2Shingles(utf8FromName, LANG_UNK, false))
        m_cur->lsaRequestData.Add(std::move(w), NLSA::TField::Fromname);
}

void TRengine::PrepareFromAddrDataForVw(const TString& asciiFromAddr) {
    const TString utf8FromAddr = Recode(CODES_ASCII, CODES_UTF8, asciiFromAddr);

    for(auto w : NText2Shingles::Text2Shingles(utf8FromAddr, LANG_UNK, false))
        m_cur->lsaRequestData.Add(std::move(w), NLSA::TField::Fromaddr);
}

TCheckMessageReport TRengine::Report() {
    TCheckMessageReport report;
    report.Hits = m_cur->Scores.Hits;
    report.DlvHits = m_cur->Scores.HitsDlvr;

    for(const TRuleCurrent& current: m_cur->rulesContext.GetOccuredRules()) {
        if (current.IsCancelled())
            continue;

        if (current.IsSignificant()) {
            if (current.GetDef().fDelivery)
                m_cur->vr_dlv.emplace_back(current);
            else
                m_cur->vr_spam.emplace_back(current);
        } else {
            m_cur->vr_null.emplace_back(current);
        }
    }

    return report;
}

void TRengine::CheckExpressions(Y_DECLARE_UNUSED int loop) {
//    int cnt = 0;
//    bool fWorked;
//    bool fWorkedAll = false;
//    bool fRuleWorked = false;

    m_cur->indCheckExpr++;

    m_cur->rulesContext.CheckExpressions();

//    do {
//        fWorkedAll = false;
//
//        for (int rid : m_pRulesHolder->GetExprRulesVector()) {
//            fWorked = false;
//            auto& ruleCurrent = m_cur->rulesContext.GetRuleCurrent(rid);
//
//            fRuleWorked = ruleCurrent.IsActive();
//            if (ruleCurrent.GetDef().rt == RT_BF) {
//                if (ruleCurrent.GetBfResult())
//                    fWorked = true;
//            } else //  == RT_ARITHMETIC
//            {
//                if (ruleCurrent.GetArResult())
//                    fWorked = true;
//            }
//
//            fWorkedAll |= (!fRuleWorked && fWorked) || (fRuleWorked && !fWorked);
//
//            if (loop) {
//                if (fWorked) {
//                    if (!ruleCurrent.IsCancelled() && !ruleCurrent.IsActive()) {
//                        m_cur->rulesContext.WorkedRule(rid);
//                    }
//                } else {
//                    if (m_cur->rulesContext.IsRuleWorked(rid))
//                        m_cur->rulesContext.UnSetWorkedRule(rid);
//                }
//            } else if (fWorked)
//                m_cur->rulesContext.WorkedRule(rid); // fist loop
//        }
//
//        if (!fWorkedAll || (++cnt > 10))
//            break;
//
//    } while (true);
}

void TRengine::BayesFillLetterDict(const char* clear_utf8_text, TString /*langName*/, ELanguage langCode) {
    const size_t len = strlen(clear_utf8_text);

    if (!len) {
        m_cur->Logger << (TLOG_WARNING) << "Empty clean UTF8 text";
        return;
    }

    NUnicode::TNormalizer<NUnicode::NFC> yandexNormalizer;
    for (TTextSplitter it(clear_utf8_text, len, MaxBayesWords); it.FindNextWord();) {
        const TTextSplitter::TSplitResult& res = it.Get();

        TUtf16String loweredWord = res.word;
        loweredWord.to_lower();

        TUtf16String normalizedWord = yandexNormalizer.Normalize(loweredWord);
        if (normalizedWord != loweredWord)
        {
            m_cur->Logger << (TLOG_NOTICE) << "ICU !NORM '" << WideToUTF8(loweredWord) << "' vs '" << WideToUTF8(normalizedWord) << '\'';
            loweredWord.assign(normalizedWord);
            m_cur->rulesContext.SetRule("ICU_NORM");
        }

        const bool goodForBayes = GoodForBayes(res);
        if (!goodForBayes)
            continue;

        const TString& loweredUTF8 = WideToUTF8(loweredWord);

        bool isStop = NText2Shingles::IsStopWord(loweredUTF8);

        if (isStop) {
            m_cur->Logger << (TLOG_DEBUG) << "STOP WORD '" << loweredUTF8 << '\'';
            continue;
        }

        m_cur->letter_dict.emplace_back(res.word, langCode);
    }

    m_cur->Logger << (TLOG_DEBUG) << "DICT LENGTH: " << m_cur->letter_dict.size();
}

bool greater_pair(std::pair<double, ui32> arg1, std::pair<double, ui32> arg2) {
    return arg1.first > arg2.first;
}

void TRengine::RaiseQuantileRules(std::vector<std::pair<double, ui32>>& vQuantiles, const char* key_label) {
    double xValue = 0;
    bool b80 = false,
            b70 = false;
    TString sLabel;

    for (std::vector<std::pair<double, ui32>>::iterator it = vQuantiles.begin(); it != vQuantiles.end(); it++) {
        xValue += it->first;
        if ((xValue > 70) && !b70) {
            sLabel.assign(key_label);
            sLabel += "70";
            b70 = true;
            CheckRange(sLabel.c_str(), (int)it->second, false);
        }
        if ((xValue > 80) && !b80) {
            sLabel.assign(key_label);
            sLabel += "80";
            b80 = true;
            CheckRange(sLabel.c_str(), (int)it->second, false);
        }
        if (xValue > 90) {
            sLabel.assign(key_label);
            sLabel += "90";
            CheckRange(sLabel.c_str(), (int)it->second, false);
            break;
        }
    }
}

std::pair<TShingleStatList, TCntComplShingleRequestList> TRengine::PrepareForHttp2(const TShinglesGroupClass& fmData) const {

    TShingleStatList shList;
    TCntComplShingleRequestList shAbuseList;

    ui64 sh64;

    for (const auto& [shingleWithType, shAttrs]: m_cur->m_newsh) {
        shList.emplace_back(shingleWithType.first, shingleWithType.second, 1); // it_new->second.Count())); // shingle, type, count

        if (Pools->ComplRequester) {
            // ignore abuses to 45, 46 if not actually first message from sender, domain
            if ((shingleWithType.second == EN_SH_FIRST_FROM_SNDR) && !IS_CMPL_FIRSTSNDR(fmData.ui_flags))
                continue;
            if ((shingleWithType.second == EN_SH_FIRST_FROM_DMN) && !IS_CMPL_FIRSTDMN(fmData.ui_flags))
                continue;

            shAbuseList.emplace_back(shingleWithType.first, shingleWithType.second);
        }
    }

    for (const auto& site : m_cur->m_vsite_netc_cs)
        if (sscanf(site.c_str(), "%" PRIx64, &sh64) == 1)
            shList.emplace_back(sh64);

    return {std::move(shList), std::move(shAbuseList)};
}

TCntShinglerResolution TRengine::ProcessCntShinglersResponses(
        const TShingleStatList& shList,
        const TCntComplShingleRequestList& shAbuseList,
        bool fHasBlindRcpto
) {

    TCntShinglerResolution resolution{};

    TString sLocalStr = HostName();
    THashMap<std::pair<ui64, TShHttpType>, TShAttrs>::iterator it_m_newsh;

    TShHttpType oldType = EN_SH_MAX;
    int cnt = 0;

    bool fShprint = false;
    bool fSkipShingle = false;
    float resultWeight = 0.0;

    TStringStream shinglesInfoForMl;
    std::array<ui32, EN_SH_MAX> c_Sh{};

    const auto ProceedUserWeight = [this](ui64 shingle, TShHttpType type) {
        if(!UsersWeights)
            return 0.;
        const auto *userWeightsPair = MapFindPtr(*UsersWeights, std::make_pair(shingle, int(type)));

        const auto &userWeight = !userWeightsPair ? TUserWeights{} : Config.fWebMail
                                                                     ? userWeightsPair->Out
                                                                     : userWeightsPair->In;


        if (userWeight.GetEffectiveShouldBeMalic()) {
            m_cur->rulesContext.SetRule("RPTN_MAN_MALIC");
        }

        double tmpWeight = userWeight.GetEffectiveWeight();

        if (fabs(tmpWeight) > 0.0099) {

            char str_rule[str_short_size]{};
            if (tmpWeight > 0)
                sprintf(str_rule, "RPTN_MAN_%d", type);
            else
                sprintf(str_rule, "RPTN_MAN_COMP%d", type);

            m_cur->rulesContext.SetScore(str_rule, tmpWeight);
            m_cur->rulesContext.SetRule(str_rule);
            if(const TString& user = userWeight.GetUser()) {
                CheckValueAllRules(FD_WEIGHTS_USER, user);
            } else {
                m_cur->rulesContext.SetRule("WEIGHTS_USER_EMPTY");
            }
            return tmpWeight;
        }

        return 0.;
    };

    auto banRecord = m_pstat.AddStat("ban_reports");

    NJson::TJsonValue shinglesLuaContext;

    for (auto& shingleStat : shList) {
        if (shingleStat.ShingleForm() == TShingleStat::TSHSHINGLE) {
            const auto sh_type = (TShHttpType)shingleStat.Type();

            NJson::TJsonValue& shingleJsonStat = shinglesLuaContext[IntToString<10>(int(sh_type))]
                    .InsertValue(IntToString<10>(shingleStat.Shingle()), shingleStat.toJson());


            if (sh_type != oldType) {
                oldType = sh_type;
                cnt = 0; // will print first 5 same-type shingles
            }

            fShprint = cnt++ < 5;
            fSkipShingle = m_shingler->SetCountToday(this, sh_type, shingleStat.ShingleStr().c_str(),
                                                     shingleStat.Ham(TODAY), shingleStat.Malic(TODAY) + shingleStat.Spam(TODAY));

            it_m_newsh = m_cur->m_newsh.find(std::make_pair(shingleStat.Shingle(), sh_type)); // Locate corresp item in shingles map for ShAttrs

            auto shAbuseIt = FindIf(shAbuseList.begin(), shAbuseList.end(), [&shingleStat](const auto& abuse){
                return (abuse.Shingle() == shingleStat.Shingle()) && (abuse.Type() == shingleStat.Type());
            });

            ui32 all = shingleStat.TotalSpam() + shingleStat.TotalMalic() + shingleStat.TotalHam();

            if (sh_type < EN_SH_MAX) {
                m_cur->c_ShSpam[sh_type] = (ui32)shingleStat.TotalMalic() + (ui32)shingleStat.TotalSpam();
                m_cur->c_ShHam[sh_type] = (ui32)shingleStat.TotalHam();
                //                        m_cur->c_ShMalic [sh_type] = (ui32) it_new->TotalMalic();
            }

            c_Sh[sh_type]++;

            switch (sh_type) {
                case EN_SH_HOST: {
//                            TCurShingle* pcursh = m_shingler->FindCurShingle(sh_type, it_new->ShingleStr().c_str());
                    int ind_url = 1; // url already added
                    const auto lowered = to_lower(it_m_newsh->second.GetLabel());
                    auto it = m_cur->m_mapurl_reputation.find(lowered);
                    if(it != end(m_cur->m_mapurl_reputation)) {
                        ind_url = it->second;
                    } else {
                        ind_url = -1; // url not found
                    }


                    if (ind_url < 1 &&
                        m_pRulesHolder->m_pListRuler.CheckList(it_m_newsh->second.GetLabel(), SP_LIST_URL_HOST, "LIST_SHORT_URL") == false) {
                        Add2UrlReputationList(it_m_newsh->second.GetLabel().c_str(), EUrlStaticticSource::BODY);
                        TString sRedirected;
                        if (IsRedirect(it_m_newsh->second.GetLabel(), &sRedirected)) {
                            Add2UrlReputationList(sRedirected.c_str(), EUrlStaticticSource::BODY);
                            m_cur->rulesContext.SetRule("URL_REDIR");
                        }

                        if (ind_url == 0)
                            it->second = 1;
                        else
                            m_cur->m_mapurl_reputation.emplace(lowered, 1); // ind_url == -1
                        m_cur->m_mapurl_rep_shingle.emplace(shingleStat.ShingleStr(), std::make_pair(m_cur->c_ShHam[sh_type], m_cur->c_ShSpam[sh_type]));
                    }
                } break;
                case EN_SH_IP_FWD:
                    if (all > 0)
                        m_cur->rulesContext.SetRule("__FWD");
                    break;
                case EN_SH_IP_RCP_TO:
                    if ((all < 2) && fHasBlindRcpto)
                        resolution.fPutBlindRcpto = true;
                    else if (all == 0) {
                        m_cur->boxes++;
                    } else if (all == 9)
                        m_cur->fPutForward = true;
                    break;
                case EN_SH_IP_BOX:
                    m_cur->boxes += all;
                    break;
                case EN_SH_LAST_SENDER_IP:
                    m_cur->mails = all;
                    if ((shingleStat.Spam(TODAY) + shingleStat.Malic(TODAY) == 1) && (shingleStat.Ham(TODAY) == 0))
                        resolution.set_dsl_first = true;
                    break;
                case EN_SH_URL_FILE_NAME:
                case EN_SH_URL_PATH:
                    if (c_Sh[sh_type] > 2)
                        fShprint = false;
                    break;
                case EN_SH_MAILFROM_UID:
                    if (Config.fWebMail && all >= 5) {
                        m_cur->fFullFL = true;
                        if (all > 10)
                            CheckRange("bounce_percent", (int)(m_cur->c_bounce_perc / all), false);
                    }

                    resolution.rcpts_today_spam = shingleStat.Spam(TODAY) + shingleStat.Malic(TODAY);
                    resolution.rcpts_today_ham = shingleStat.Ham(TODAY);
                    resolution.rcpts_today_malic = shingleStat.Malic(TODAY);
                    if (c_Sh[sh_type] > 1)
                        fShprint = false;
                    break;
                case EN_SH_FROM_ADDR:
                    resolution.sndr_today = shingleStat.Ham(TODAY) + shingleStat.Spam(TODAY) + shingleStat.Malic(TODAY);
                    resolution.sndr_today_spam = shingleStat.Spam(TODAY) + shingleStat.Malic(TODAY);
                    resolution.sndr_today_ham = shingleStat.Ham(TODAY);
                    resolution.sndr_today_malic = shingleStat.Malic(TODAY);
                    resolution.sndr_yesterday = shingleStat.Ham(YESTERDAY) + shingleStat.Spam(YESTERDAY) + shingleStat.Malic(YESTERDAY);
                    resolution.sh14_ham = shingleStat.TotalHam();
                    break;
                case EN_SH_FROM_DOMAIN:
                    resolution.sndr_domain_today = shingleStat.Ham(TODAY) + shingleStat.Spam(TODAY) + shingleStat.Malic(TODAY);
                    break;
                default:
                    break;
            }

            if (fShprint) {
                shinglesInfoForMl << static_cast<int>(sh_type) << ':' << shingleStat.ShingleStr() << ':'

                                  << shingleStat.TotalSpam() << ':'
                                  << shingleStat.TotalMalic() << ':'
                                  << shingleStat.TotalHam() << ':'

                                  << shingleStat.Spam(TODAY) << ':'
                                  << shingleStat.Malic(TODAY) << ':'
                                  << shingleStat.Ham(TODAY) << ':'

                                  << shingleStat.PersSpam(TODAY) << ':'
                                  << shingleStat.PersHam(TODAY) << ';';

                m_pstat.AddStat(ST_LOG) << "t = " << int(sh_type)
                                        << ", s = " << (shingleStat.TotalSpam() + shingleStat.TotalMalic())
                                        << ", h = " << shingleStat.TotalHam()
                                        << ", u = " << all
                                        << ", w = 0.0000 (0) 1 S= " << shingleStat.Spam(TODAY)
                                        << " H= " << shingleStat.Ham(TODAY)
                                        << " m= " << shingleStat.Malic(TODAY)
                                        << " ph= " << shingleStat.PersHam(TODAY)
                                        << " ps= " << shingleStat.PersSpam(TODAY) << " " << shingleStat.ShingleStr() << " " << SafeRecode(it_m_newsh->second.BracketedLabel());
                shingleJsonStat.InsertValue("text", it_m_newsh->second.GetLabel());
            }

            if(auto w = ProceedUserWeight(shingleStat.Shingle(), sh_type)) {
                resultWeight += w;
                BanByWeights->PushSignal(1.);
            }

            banRecord << "sh" << int(sh_type) << ':';
            for (const auto hostHash : m_cur->HostsHashes) {
                const auto fullHash = shingleStat.Shingle() ^ hostHash;
                if (auto w = ProceedUserWeight(fullHash, sh_type)) {
                    resultWeight += w;
                    BanByReports->PushSignal(1.);
                    m_cur->rulesContext.SetRule("BANED_BY_REPORT");
                    banRecord << Hex(fullHash) << ':' << w << ',';
                }
            }
            banRecord << ';';

            if (shAbuseIt != shAbuseList.end()) {
                if (fShprint && (!shAbuseIt->ResTodayEmpty() || !shAbuseIt->ResHistoryEmpty())) {
                    m_pstat.AddStat(ST_COMPLLOG) << "t = " << int(sh_type) << ", "
                                                 << shAbuseIt->GetTodayComplaintHam() << " "
                                                 << shAbuseIt->GetTodayComplaintSpam() << " "
                                                 << shAbuseIt->GetTodayUniqSDHam() << " "
                                                 << shAbuseIt->GetTodayUniqSDSpam() << "  "
                                                 << shAbuseIt->GetHistoryHam() << " "
                                                 << shAbuseIt->GetHistorySpam() << " "
                                                 << shAbuseIt->GetHistoryComplaintHam() << " "
                                                 << shAbuseIt->GetHistoryComplaintSpam() << " "
                                                 << shAbuseIt->GetHistoryDayCountWithComplaint() << "  "
                                                 << shAbuseIt->GetHistoryVirusCount() << " "
                                                 << shAbuseIt->GetHistoryDayCountWithVirus() << "  "
                                                 << shAbuseIt->sShingle();
                }

                char str_short[str_short_size];
                if (shAbuseIt->GetTodayVirusCount()) {
                    sprintf(str_short, "virus_td_key_%d", sh_type);
                    CheckRange(str_short, (int)shAbuseIt->GetTodayVirusCount());
                }
                if (shAbuseIt->GetHistoryVirusCount()) {
                    sprintf(str_short, "virus_hist_key_%d", sh_type);
                    CheckRange(str_short, (int)shAbuseIt->GetHistoryVirusCount());
                }
                if (shAbuseIt->GetHistoryDayCountWithVirus()) {
                    sprintf(str_short, "virus_days_key_%d", sh_type);
                    CheckRange(str_short, (int)shAbuseIt->GetHistoryDayCountWithVirus());
                }
            }

            if (!fSkipShingle) {
                SetRulePattern(sh_type, shingleStat.TotalHam(), 0, (ui32)shingleStat.TotalMalic() + (ui32)shingleStat.TotalSpam(), all);
                SetRulePatternRange(sh_type, shingleStat.TotalHam(), 0, (ui32)shingleStat.TotalMalic() + (ui32)shingleStat.TotalSpam(),
                                    shAbuseIt != shAbuseList.end() ? shAbuseIt->GetTodayComplaintHam() : -1,
                                    shAbuseIt != shAbuseList.end() ? shAbuseIt->GetTodayComplaintSpam() : -1);
            }
        }

        if (shingleStat.ShingleForm() == TShingleStat::TSHSTORAGE) // site netC
        {
            std::vector<TString>::iterator it_value = m_cur->m_vsite_netcinfo.begin();
            auto it_key = m_cur->m_vsite_netc_cs.begin();
            auto it_domen = m_cur->m_vsite_netc.begin();
            for (it_key = m_cur->m_vsite_netc_cs.begin(); it_key != m_cur->m_vsite_netc_cs.end(); it_key++, it_value++, it_domen++)
                if (*it_key == shingleStat.ShingleStr())
                    break;

            if (it_key != m_cur->m_vsite_netc_cs.end()) {
                m_alg->AddSiteNetcInfo(*this, (*it_domen).c_str(), (*it_key).c_str(), shingleStat.GetSTData().c_str());
            }

            m_alg->WriteDeliveryStat(true);
            m_alg->CompareFixMethods();
        }

    } // loop for shList

    CheckValueAllRules(FD_SHINGLES_MAP, shinglesLuaContext);

    m_cur->MlLogBuilder.Add("sh", std::move(shinglesInfoForMl.Str()));

    if (resultWeight > 0) {
        m_cur->rulesContext.SetScore("RPTN_MAN", resultWeight);
        m_cur->rulesContext.SetRule("RPTN_MAN");
    }
    if (resultWeight < 0) {
        m_cur->rulesContext.SetScore("RPTN_MAN_COMP", resultWeight);
        m_cur->rulesContext.SetRule("RPTN_MAN_COMP");
    }

    return resolution;
}

void TRengine::GetLSA() try{
    if(Text2VecModel) {
        const TVector<TString> annotations{
            TString("body"),
            TString("subject"),
            TString("fromname"),
            TString("fromaddr")};

        const TVector<TString> dssmInputs{
            JoinStrings(m_cur->lsaRequestData.Get(NLSA::TField::Body), " "),
            JoinStrings(m_cur->lsaRequestData.Get(NLSA::TField::Subject), " "),
            JoinStrings(m_cur->lsaRequestData.Get(NLSA::TField::Fromname), " "),
            JoinStrings(m_cur->lsaRequestData.Get(NLSA::TField::Fromaddr), " "),
        };

        Text2VecModel->Apply(
                MakeAtomicShared<NNeuralNetApplier::TSample>(
                        annotations,
                        dssmInputs),
                {"query_embedding"},
                m_cur->TextEmbed
        );
    }
    {
        if (m_cur->TextEmbed) {
            auto embedsString = JoinSeq(",", m_cur->TextEmbed);
            //GetSpStat().AddStat("body_embed") << embedsString;
            m_cur->MlLogBuilder.Add("body_embed", std::move(embedsString));
        }

        if(Pools->KnnRequester) {
            m_cur->KnnFuture = NThreading::Async(
                    [req = NSoKNN::TGetNeighborsRequest(NSoKNN::TCoordinate(m_cur->MinHash),
                                                        m_cur->TextEmbed),
                    Getter = &Pools->KnnRequester->Getter,
                    logger = m_cur->Logger]() {
                        return Getter->Perform(std::move(req), logger);
                    }, ThreadPool);
        }
    }

    m_cur->MlLogBuilder.Add("nnfromname",
                            JoinStrings(m_cur->lsaRequestData.Get(NLSA::TField::Fromname), ","));
    m_cur->MlLogBuilder.Add("nnfromaddr",
                            JoinStrings(m_cur->lsaRequestData.Get(NLSA::TField::Fromaddr), ","));
    m_cur->MlLogBuilder.Add("nnsubj",
                            JoinStrings(m_cur->lsaRequestData.Get(NLSA::TField::Subject), ","));
    m_cur->MlLogBuilder.Add("nnbody",
                            JoinStrings(m_cur->lsaRequestData.Get(NLSA::TField::Body), ","));

    for(const auto & [modelInfo, _] : *Models) {
        for(const auto id : modelInfo.slaveModelsIds)
            m_cur->lsaRequestData.AddModelId(ToString(id));
    }

    NLSA::TResponseData responseData;
    try {
        responseData = Pools->LSARequester->Perform(m_cur->lsaRequestData, m_cur->rulesContext.IsRuleWorked("TEST_SO_VERSION"));
    } catch (...) {
        m_cur->rulesContext.SetRule("LSA_FAIL");
        m_cur->Logger << TLOG_ERR << CurrentExceptionMessageWithBt();
        return;
    }

    {
        CheckValueAllRules(FD_LSA_RESPONSE, responseData.GetOriginalJson());
    }

    {
        {
            TStringStream msg;

            msg << "found: " << responseData.GetMatchsPercent();

            for(const NLSA::TResponseData::TTheme & theme : responseData.GetThemes())
                msg << "; " << theme.GetDescription() << ':' << theme.GetDistance();
            //m_pstat.AddStat(ST_LSA) << msg.Str();
            m_cur->MlLogBuilder.Add("lsa", msg.Str());
        }

        /*const auto & testing = responseData.GetTesting();
        if(!testing.empty())
        {
            auto record = m_pstat.AddStat(ST_LSA_TESTING);
            for(const auto & p : testing)
            {
                const auto & name = p.first;

                record << name << ": ";
                for(const NLSA::TResponseData::TTheme & theme : p.second)
                    record << "; " << theme.GetDescription() << ':' << theme.GetDistance();
                record << ';';
            }
        }*/
    }

    {
        for(const auto & theme : responseData.GetThemes())
        {
            TString rule = "LSA_" + theme.GetDescription();
            ToUpper(rule.begin(), rule.size());

            m_cur->rulesContext.SetRule(rule);
        }
    }

    {
        for (const auto& [name, themesChain] : responseData.GetTesting()) {
            auto maxIt = MaxElementBy(themesChain, [](const auto& theme) {
                return theme.Distance;
            });

            const ssize_t index = std::distance(themesChain.cbegin(), maxIt);

            constexpr TStringBuf TAB_RULES_TEMPLATES[] = {"LSA_TAB_NEWS_", "LSA_TAB_NOTIF_", "LSA_TAB_PEOPLE_", "LSA_TAB_SOCIAL_"};

            if (0 <= index && index < (ssize_t)Y_ARRAY_SIZE(TAB_RULES_TEMPLATES)) {
                m_cur->rulesContext.SetRule(TAB_RULES_TEMPLATES[index] + to_upper(name));
            } else {
                ythrow TWithBackTrace<yexception>() << "invalid theme: " << name << ", index: " << index;
            }
        }
    }

    {
        struct TBestTab{
            TBestTab(TStringBuf name, double score) noexcept : Name(name), Score(score) {}
            TStringBuf Name;
            double Score{};
        };

        THashMap<TString, TBestTab> bestTabsByPrefix;
        {
            const auto& predictions = responseData.GetPredictionsByModel();

            THashSet<ui64> missedModelsIds;
            for (const auto& [modelInfo, _] : *Models) {
                const auto briefName = to_upper(modelInfo.briefName);
                auto& factors = modelInfo.slot == "mn_prod" ? m_cur->m_mapValueFactors4Matrixnet : m_cur->m_factorsAddons[modelInfo.slot];
                for (const auto& slaveModelId : modelInfo.slaveModelsIds) {
                    auto featuresMap = MapFindPtr(predictions, ToString(slaveModelId));
                    if (!featuresMap || !*featuresMap) {
                        missedModelsIds.emplace(slaveModelId);
                        continue;
                    }

                    factors.insert(featuresMap->cbegin(), featuresMap->cend());

                    //todo make it better
                    {
                        if (!Config.TabPrefixesToCalcTheBest)
                            continue;
                        for (const auto& p : *featuresMap) {
                            const auto& feature = p.first;
                            const auto& value = p.second;

                            const auto prefix = FindIfPtr(Config.TabPrefixesToCalcTheBest, [&feature](const auto& prefix) {
                                return feature.StartsWith(prefix);
                            });

                            if (!prefix)
                                continue;

                            if (auto bestTab = MapFindPtr(bestTabsByPrefix, *prefix)) {
                                if (bestTab->Score < value) {
                                    *bestTab = TBestTab(feature, value);
                                }
                            } else {
                                bestTabsByPrefix.emplace(*prefix, TBestTab(feature, value));
                            }
                        }
                    }
                }
            }

            MissedModels->PushSignal(missedModelsIds.size());
            if(missedModelsIds) {
                NJson::TJsonValue missedModelsAsJson;
                for(const ui64 id : missedModelsIds) {
                    missedModelsAsJson.AppendValue(id);
                }
                m_cur->MlLogBuilder.Add("missed_models", std::move(missedModelsAsJson));
            }
        }

        for(const auto& [_, bestTab]: bestTabsByPrefix) {
            m_cur->rulesContext.SetRule(bestTab.Name);
        }
    }
    {
        const auto& rulesByModel = responseData.GetRulesByModel();
        for (const auto& [modelInfo, _] : *Models) {
            const auto briefName = to_upper(modelInfo.briefName);
            for (const auto& slaveModelId : modelInfo.slaveModelsIds) {
                if(auto rules = MapFindPtr(rulesByModel, ToString(slaveModelId))) {
                    for(const auto& rule : *rules) {
                        m_cur->rulesContext.SetRule(TStringBuilder{} << rule << '_' << briefName);
                    }
                }
            }
        }
    }
    {
        for(const auto& rule : responseData.GetIndepRules()) {
            m_cur->rulesContext.SetRule(rule);
        }
    }

    {
        const auto& distances = responseData.GetDistancesToCompls();

        if(distances)
            m_cur->MlLogBuilder.Add("compl_distances",
                                    TStringBuilder{} << MakeRangeJoiner(",", distances));
    }
} catch(...) {
    m_cur->Logger << (TLOG_ERR) << "error in GetLSA: " << CurrentExceptionMessageWithBt();
}

static const auto languageMapping = MakeTrueConst(
    THashMap<TString, TString>{
        {"POLISH", "EUROPE"},
        {"CZECH", "EUROPE"},
        {"SLOVAK", "EUROPE"},
        {"LITHUANIAN", "EUROPE"},
        {"HUNGARIAN", "EUROPE"},
        {"AZERBAIJANI", "EUROPE"},
        {"ROMANIAN", "EUROPE"},
        {"GEORGIAN", "EUROPE"},
        {"GREEK", "EUROPE"},
        {"LATVIAN", "EUROPE"},
        {"SLOVENIAN", "EUROPE"},
        {"DUTCH", "EUROPE"},
        {"CROATIAN", "EUROPE"},
        {"ALBANIAN", "EUROPE"},
        {"FINNISH", "EUROPE"},
        {"SWEDISH", "EUROPE"},
        {"DANISH", "EUROPE"},
        {"ESTONIAN", "EUROPE"},
        {"NORWEGIAN", "EUROPE"},
        {"EUROPE_N", "EUROPE"},

        {"SERBIAN", "RUSSIAN"},
        {"BULGARIAN", "RUSSIAN"},
        {"BELARUSIAN", "RUSSIAN"},
        {"RUSSIAN", "RUSSIAN"},

        {"INDONESIAN", "ASIA"},
        {"VIETNAMESE", "ASIA"},
        {"BURMESE", "ASIA"},
        {"THAI", "ASIA"},
        {"TAGALOG", "ASIA"},
        {"MALAY", "ASIA"},

        {"BENGALIAN", "HINDI"},
        {"BENGALI", "HINDI"},
        {"URDU", "HINDI"},
        {"SINHALESE", "HINDI"},
        {"SANSKRIT", "HINDI"},
        {"PUNJABI", "HINDI"},
        {"HINDI", "HINDI"},

        {"PORTUGUESE", "SPANISH"},
        {"CATALAN", "SPANISH"},
        {"SPANISH", "SPANISH"},

        {"CHINESET", "CHINESE"},
        {"JAPANESE", "CHINESE"},
        {"KOREAN", "CHINESE"},
        {"CHINESE", "CHINESE"},

        {"PERSIAN", "ARABIC"},
        {"HEBREW", "ARABIC"},
        {"ARABIC", "ARABIC"},

        {"UNKNOWN", "UNKNOWN"},
        {"ENGLISH", "ENGLISH"},
        {"TURKISH", "TURKISH"},
        {"FRENCH", "FRENCH"},
        {"GERMAN", "GERMAN"},
        {"UKRAINIAN", "UKRAINIAN"},
        {"ITALIAN", "ITALIAN"},
    }
);

TString TRengine::GetGroupedLanguage(TString sLanguage) {
    ToUpper(sLanguage.begin(), sLanguage.size());

    auto it = languageMapping->find(sLanguage);
    if(it != languageMapping->cend())
        return it->second;

    return "OTHER";
}

bool compareUnicodeCodesAndCnts(std::pair<int, int> arg1, std::pair<int, int> arg2) {
    return arg1.second > arg2.second;
}

TString TRengine::UnicodeBlocks(const char* pPureText) {
    size_t bodyLen = 0;
    if (pPureText)
        bodyLen = strlen(pPureText);

    std::map<int, int> mUnicodeBlockCodesCnt;

    TUtf16String unicodeBody;
    //    rprof utftowide ("UTF8ToWide");
    try {
        unicodeBody = UTF8ToWide<true>(pPureText, bodyLen);
    } catch (yexception& ye) {
        m_cur->Logger <<TLOG_WARNING << CurrentExceptionMessageWithBt();
    }
    //    utftowide.stop ();

    //    rprof script_lang ("scripts-langs");
    for (TUtf16String::const_iterator wChar = unicodeBody.begin(); wChar != unicodeBody.end(); wChar++) {
        mUnicodeBlockCodesCnt[NUnicodeRange::UnicodeBlockCodeBySymbol(*wChar)]++;
    }

    std::vector<std::pair<int, int>> vScripts;
    for (std::map<int, int>::iterator it = mUnicodeBlockCodesCnt.begin(); it != mUnicodeBlockCodesCnt.end(); it++) {
        if (it->second)
            vScripts.push_back(std::make_pair(it->first, it->second));
    }
    sort(vScripts.begin(), vScripts.end(), compareUnicodeCodesAndCnts);

    TString sScripts;
    int nScripts = 0;
    int nRareScripts = 0;
    for (std::vector<std::pair<int, int>>::iterator vIt = vScripts.begin(); vIt != vScripts.end(); vIt++) {
        if (vIt->second) {
            if (!sScripts.empty())
                sScripts.append("-");
            sScripts.append(NUnicodeRange::UnicodeBlockNameByCode(vIt->first));
            sScripts.append(":");
            sScripts.append(IntToStroka(vIt->second));
            nScripts++;
            if (NUnicodeRange::IsRare(vIt->first))
                nRareScripts++;
        } else
            break;
    }
    //    script_lang.stop ();

    CheckRange("unicode_scripts", nScripts);
    CheckRange("rare_unicode_scripts", nRareScripts);

    return sScripts;
}

void TRengine::CheckBody(NProf::Profiler& prof, const char *pbufbody, size_t bodylen, TSpMesType mestype,
                         TString& pPureText, const char **pPureUtf8Text,
                         TBodyPartProperty *property) {

    auto totalProf = Guard(prof.Prof("all"));

    pPureText.clear();
    *pPureUtf8Text = nullptr;

    if (bodylen <= 0)
        return;

    char lngg[str_short_size];

    snprintf(lngg, sizeof(lngg), "%s", NameByLanguage(property->lang));
    m_pstat.AddStat(ST_LANGUAGE) << NameByLanguage(property->lang);

    // now try google lang detector
    CLD2::Language googleLang;
    {
        bool is_reliable;
        int valid_prefix_length;
        auto g = Guard(prof.Prof("p"
                                 "1"));
        googleLang = CLD2::DetectLanguageCheckUTF8(pbufbody, bodylen, false, &is_reliable, &valid_prefix_length);
    };
    //    cld2.stop ();

    //    rprof stat2 ("cld2 stat");
    m_pstat.AddStat(ST_LANGUAGE) << property->pcharset << ", google says:" << LanguageName(googleLang) << ' ' << LanguageCode(googleLang);

    // do ignore ya-recognizer and trust goo recognizer for now
    ELanguage yandexLang = LANG_UNK;
    if (googleLang == CLD2::HEBREW)
        yandexLang = LANG_HEB;
    else
        yandexLang = LanguageByName(LanguageCode(googleLang));

    snprintf(lngg, sizeof(lngg), "%s", LanguageCode(googleLang));
    property->lang = yandexLang;


    m_cur->MlLogBuilder.Add(ToString(ST_LANGUAGE), lngg);

    CheckField(FD_IY_LANGUAGE, lngg);

    TString geoLang;

    {
        auto g = Guard(prof.Prof("p" "2"));
        for (const auto& [login, attr] : m_cur->m_rcptattrs) {
            geoLang.assign(attr.sCountry, 0, 2);

            ToUpper(geoLang.begin(), geoLang.size());
            geoLang += "-" + GetGroupedLanguage(LanguageName(googleLang));
            CheckField(FD_GEO_LANG, geoLang);
        }
    }

    if (property->pcharset && TRulesHolder::m_pcre->Check("cyrillic", TStringBuf(property->pcharset))) {
        m_cur->rulesContext.SetRule("__CYRILLIC_BODY");
    }

    TString body;
    {
        auto g = Guard(prof.Prof("p" "3"));
        TStringOutput stringOutput(body);
        TestUuencode({pbufbody, bodylen}, stringOutput);
    }

    const ECharset m_singleByteCP = GetDestinationEncoding(property->lang);
    m_html->SetSingleByteCP(m_singleByteCP);
    const ELanguage knownLang = (
                                        property->lang == LANG_RUS ||
                                        property->lang == LANG_ENG ||
                                        property->lang == LANG_UKR ||
                                        property->lang == LANG_BEL ||
                                        property->lang == LANG_TUR) ? property->lang : LANG_UNK;

    m_html->CheckBody(this, prof.Sub("html"), body, mestype, pPureText, pPureUtf8Text);

    if (*pPureUtf8Text)
    {
        auto g = Guard(prof.Prof("p"
                                 "5"));
        {
            auto g = Guard(prof.Prof("p"
                                     "6"));
            BayesFillLetterDict(*pPureUtf8Text, lngg, knownLang);
        }
        m_cur->langCode = property->lang;

        m_pstat.AddStat(ST_LANGUAGE) << UnicodeBlocks(*pPureUtf8Text);
        {
            auto g = Guard(prof.Prof("p"
                                     "7"));
            if (m_cur->OcrText) {
                PrepareBodyDataForVw(m_cur->OcrText, knownLang);
            } else {
                PrepareBodyDataForVw(*pPureUtf8Text, knownLang);
            }
        }
    } else
        m_cur->Logger <<TLOG_WARNING << "Empty UTF8 body";
}

void TRengine::Check8Bit(const TStringBuf& fieldname, const TStringBuf& field) {
    int c = 0;

    TString sFldname = TString{fieldname};
    sFldname.to_lower();

    TSpFields fid;
    if (!TryFromString<TSpFields>(sFldname, fid))
        return;
    if (fid != FD_TO && fid != FD_FROM && fid != FD_SUBJECT)
        return;

    for (size_t i = 0; i < field.size(); i++) {
        if (field[i] < 0) {
            if (++c > 2) {
                m_cur->rulesContext.SetRule("HEADER_8BITS");
                return;
            }
        } else
            c = 0;
    }
}

void TRengine::CheckCharSet(const TStringBuf& pcharset) {
    CheckField(FD_FIELD_CODING, pcharset);

#ifdef SO_OLD_BUILD
    const CodePage* cp = CodePageByName(pcharset);
    if (!cp)
#else
    ECharset code = CharsetByName(pcharset);
    if (!ValidCodepage(code))
#endif
        m_cur->rulesContext.SetRule("CHARSET_FARAWAY_HEADERS");
}

void TRengine::CheckField(TStringBuf pFieldName, TStringBuf pField, TStringBuf pcharset, bool fValid, bool fMime) {
    ++m_cur->m_cFields;

    if (!fValid)
        m_cur->rulesContext.SetRule("MAIL_INVALID_FIELD");

    if (pcharset && pcharset != "US-ASCII")
        CheckCharSet(pcharset);

    if (pFieldName && pField && !fMime)
        Check8Bit(pFieldName, pField);

    CheckField(pFieldName, pField);
}
void TRengine::TestUuencode(TStringBuf src, IOutputStream& target) {
    size_t j = 20;

    TStringBuf uuencode;
    while (src && --j > 0 && GetUuencode(src, uuencode) && uuencode) {
        const size_t offset = uuencode.data() - src.data();
        target << src.substr(0, offset);
        src.Skip(offset + uuencode.size());
    }

    target << src;
}

bool TRengine::GetUuencode(const TStringBuf& buf, TStringBuf & uuencode) {
    if (TRulesHolder::m_pcre->Check("uuencode_check", buf)) {
        if (uuencode = TRulesHolder::m_pcre->GetPattern("uuencode_end", buf, 0))
            return true;
        else if (uuencode = TRulesHolder::m_pcre->GetPattern("uuencode_bad", buf, 0))
            return true;
    }
    return false;
}

void TRengine::CheckBodyPart(const char* pBody, int Len, TBodyPartProperty* property) {
    m_bodypart->CheckBodyPart(this, pBody, Len, property);
}

void TRengine::CheckQuoted(const TStringBuf& text) {
    m_html->CheckQuoted(text);
}

const char* TRengine::GetMessageId() {
    return m_cur->m_sMessageId.c_str();
}

void TRengine::SetShingle(TSpShingleType shtype, ui64 shingle) {
    char buff[64]{};
    sprintf(buff, "%015" PRIx64, shingle);
    SetShingle(shtype, buff);
}

void TRengine::SetShingle(TSpShingleType shtype, const TStringBuf& szshingle) {
    if(!szshingle)
        return;
    switch (shtype) {
        case spShingle:
            AddShingle(szshingle, EN_SH_SHINGLE);
            m_shingler->Add(nullptr, szshingle, EN_SH_SHINGLE, false);
            break;
        case spCShingle:
            AddShingle(szshingle, EN_SH_CLEAR);
            m_shingler->Add(nullptr, szshingle, EN_SH_CLEAR, false);
            break;
        case spSanShingle:
            AddShingle(szshingle, EN_SH_SAN_TEXT);
            m_shingler->Add(nullptr, szshingle, EN_SH_SAN_TEXT, false);
            break;
        case spAttachMd5:
            AddShingle(szshingle, EN_SH_ATTACH_MD5);
            m_shingler->Add(nullptr, szshingle, EN_SH_ATTACH_MD5, false);
            break;
        case spFcrc:
            AddShingle(szshingle, EN_SH_FCRC);
            m_shingler->Add(nullptr, szshingle, EN_SH_FCRC, false);
            break;
    }
}

const TRuleDef* TRengine::GetSuperNetRule(ui32 net) const {
    return m_pRulesHolder->FindDeliverySuperNetRule(net);
}

const TRuleDef* TRengine::GetNetRule(ui32 net) const {
    return m_pRulesHolder->FindDeliveryNetRule(net);
}

const TRuleDef* TRengine::GetIpRule(ui32 ip) const {
    return m_pRulesHolder->FindDeliveryIpRule(ip);
}

void TRengine::SetTimeForStatLog() {
    const auto& pTimeX = Config.sStatTimeX;

    int hour, min;
    time_t ltime;

    time(&ltime);

    if (auto res = TRulesHolder::m_pcre->Check("time_x", pTimeX)) {
        if (!res->GetPattern(1, hour) ||
            !res->GetPattern(2, min) ||
            hour > 23 || min > 59)
        {
            m_cur->Logger << (TLOG_ERR) << "Time format error for StatTimeX: " << pTimeX;
            return;
        }
    } else
        return;

    struct tm* curtm = localtime(&ltime);
    time_t ltimex, next_day = 0;
    if (curtm->tm_hour > hour || curtm->tm_hour == hour && curtm->tm_min > min)
        next_day = 24 * 3600;
    curtm->tm_hour = hour;
    curtm->tm_min = min;
    curtm->tm_sec = 0;
    ltimex = mktime(curtm);
}

//fPutAdd == true �� ������ ������ � put � get, ����� ������ get
void TRengine::AddShingle(const TStringBuf& pPattern, TShHttpType shtype, const TStringBuf& pText, bool fPutAdd, int cnt) {
    if(!pPattern)
        return;
    char str[str_short_size];
    const auto len = pPattern.size();

    if (m_cur->fInternalPddMail && shtype != EN_SH_PDD)
        return;

    if (len < 5)
        m_cur->Logger << TLOG_WARNING << "Strange shingle size: " << len;

    if (len > SP_HTTP_MAX_SHINGLE_SIZE) {
        m_cur->Logger << TLOG_ERR << "Too high shingle size: " << len;
        return;
    }

    if (m_cur->m_newsh.size() > SP_HTTP_MAX_SHINGLE_COUNT) {
        m_cur->Logger << TLOG_ERR << "Too high new shingles count: " << m_cur->m_newsh.size();
        return;
    }

    if (shtype == EN_SH_MAILFROM_UID) {
        if (m_cur->c_ShAdd[shtype] > 127) {
            m_pstat.AddStat(ST_LOG) << "sh limit " << (int)shtype << ' ' << m_cur->c_ShAdd[shtype];
            return;
        }
    } else if (m_cur->c_ShAdd[shtype] > SP_HTTP_MAX_ALONE_SHINGLE_COUNT) {
        m_pstat.AddStat(ST_LOG) << "sh limit " << (int)shtype << ' ' << m_cur->c_ShAdd[shtype];
        return;
    }

    sprintf(str, "shvalue_%d", shtype);
    CheckField(str, pPattern, true);

    ++m_cur->c_ShAdd[shtype];

    ui64 sh64;
    if (sscanf(pPattern.data(), "%" PRIx64, &sh64) == 1) {
        m_cur->m_newsh[std::make_pair(sh64, shtype)].Increment(pText, pPattern, fPutAdd, cnt);
    }
}

TString TRengine::AddPattern(const TStringBuf& str, TShHttpType shtype, bool fPutAdd, bool fBanPattern, int cnt) {
    char crs_str[32];

    if (m_cur->fInternalPddMail && shtype != EN_SH_PDD)
        return {};

    if (str.empty()) //  || (len < 2 && !fPutAllLen)) do not check string length anymore
        return {};

    if (!calc_strcrc64(str, crs_str))
        return {};

    TStringBuf pDupText;

    switch(shtype) {
        // ��� ���� ����� �������� ����� ������ � �������
        case EN_SH_HOST: {
            auto lowered = to_lower(TString{str});
            if (!m_cur->m_mapurl_reputation.contains(lowered) &&
                !m_pRulesHolder->m_pListRuler.CheckList(str, SP_LIST_URL_HOST, "LIST_SHORT_URL"))
                m_cur->m_mapurl_reputation.emplace(std::move(lowered), 0);
        }
        case EN_SH_MAIL:
        case EN_SH_PHONE:
        case EN_SH_NUMBER:
        case EN_SH_ICQ:
        case EN_SH_FROM_NAME:
        case EN_SH_FROM_ADDR:
        case EN_SH_FIRST_LAST_SENDER_IP:
        case EN_SH_LAST_SENDER_IP:
        case EN_SH_IP_RCP_TO:
        case EN_SH_MAILFROM_UID:
        case EN_SH_MAILFROM_UID_UNIQ:
        case EN_SH_PDD_DOMN_UNIQ:
        case EN_SH_GEO_UID_UNIQ:
        case EN_SH_UID_MSGID:
        case EN_SH_FROM_DOMAIN:
        case EN_SH_BLIND_RCPTO:
        case EN_SH_ATTACH_EXEC:
        case EN_SH_ATTACH_NAME:
        case EN_SH_FIRST_FROM_SNDR:
        case EN_SH_FIRST_FROM_DMN:
        case EN_SH_SOURCE_IP_MASKED:
        case EN_SH_FROM_PDD_ADM_UNIQ:
        case EN_SH_FROM_PDD_ADMIN:
        case EN_SH_FROMNAME_UID:
        case EN_SH_FROMNAME_UID_UNIQ:
        case EN_SH_COMMON_ZONE:
        case EN_SH_DKIM:
        case EN_SH_DKIM_DOMAIN:
        case EN_SH_X_SENDER_ACCOUNT:
        case EN_SH_ATTACH_MD5:
        case EN_SH_LUA_GENERATED: {
            pDupText = str;
        }
        default:
            break;
    };

    AddShingle(crs_str, shtype, pDupText, fPutAdd, cnt);
    if (!pDupText && str.size() < 128)
        pDupText = str;
    m_shingler->Add(pDupText, crs_str, shtype, fBanPattern);

    return crs_str;
}

void TRengine::SetRulePatternRange(TShHttpType sh_type, ui32 ham_inp, ui32 dlv_inp, ui32 spam_inp, int abuseComplHam, int abuseComplSpam) {
    const ui32 white = (ui32)ham_inp + (ui32)dlv_inp;
    const ui32 spam = spam_inp;
    const ui32 all = white + spam;
    char str[str_short_size];

    sprintf(str, "sh_key_%d", sh_type);
    CheckRange(str, (int)all, false); // shingle count
    sprintf(str, "MNF_SH_%d", sh_type);
    m_cur->m_mapValueFactors4Matrixnet[str] = all;

    if (abuseComplHam >= 0) {
        sprintf(str, "shch_key_%d", sh_type);
        CheckRange(str, abuseComplHam, false);
    }
    if (abuseComplSpam >= 0) {
        sprintf(str, "shcs_key_%d", sh_type);
        CheckRange(str, abuseComplSpam, false);
    }

    if (white == spam || (white < 20 && spam < 20))
        return;

    const int value_ratio = (spam * 1000) / all;

    if (all < 100) {
        sprintf(str, "sh_ratio_key_100_%d", sh_type);
        CheckRange(str, value_ratio, false); // shingle count
    } else {
        if (all < 1000) {
            if (abuseComplHam >= 0) {
                const double dValue = 1000. * abuseComplHam / (spam + 1);
                sprintf(str, "shch_ratio_key_1000_%d", sh_type);
                CheckRange(str, (int)dValue);
                sprintf(str, "MNF_SHCH_%d", sh_type);
                m_cur->m_mapValueFactors4Matrixnet[str] = dValue;
            }
            if (abuseComplSpam >= 0) {
                const double dValue = 1000.0 * abuseComplSpam / (white + 1);
                sprintf(str, "shcs_ratio_key_1000_%d", sh_type);
                CheckRange(str, (int)dValue);
                sprintf(str, "MNF_SHCS_%d", sh_type);
                m_cur->m_mapValueFactors4Matrixnet[str] = dValue;
            }

            sprintf(str, "sh_ratio_key_1000_%d", sh_type);
            CheckRange(str, value_ratio, false); // shingle count range rule
        } else {
            if (abuseComplHam >= 0) {
                const double dValue = 1000.0 * abuseComplHam / (spam + 1);
                sprintf(str, "shch_ratio_key_1111_%d", sh_type);
                CheckRange(str, (int)dValue);
                sprintf(str, "MNF_SHCH_%d", sh_type);
                m_cur->m_mapValueFactors4Matrixnet[str] = dValue;
            }
            if (abuseComplSpam >= 0) {
                const double dValue = 1000.0 * abuseComplSpam / (white + 1);
                sprintf(str, "shcs_ratio_key_1111_%d", sh_type);
                CheckRange(str, (int)dValue);
                sprintf(str, "MNF_SHCS_%d", sh_type);
                m_cur->m_mapValueFactors4Matrixnet[str] = dValue;
            }

            sprintf(str, "sh_ratio_key_1111_%d", sh_type);
            CheckRange(str, value_ratio, false); // shingle count
        }
    }
}

void TRengine::SetRulePattern(TShHttpType sh_type, ui32 ham_inp, ui32 dlv_inp, ui32 spam_inp, ui32 unknown_inp) {
    ui32 ham = ham_inp;
    ui32 dlv = dlv_inp;
    ui32 spam = spam_inp;
    ui32 unknown = unknown_inp;

    // range[i] < val < range[i+1]
    //    ui32 range[] =    {1,2,5,10,100,200,500,1000, SHINGLE_MAX};
    ui32 range_wb[] = {9, 99, 499, 999, SHINGLE_MAX};
    ui32 range_ratio[] = {3, 8, 32, 128, 512, SHINGLE_MAX};
    char prefix_let[] = {'0', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J'};
    ui32 white = ham + dlv;
    const char* pwb;
    ui32 value;
    ui32 value_ratio;
    int rv_wb, rv_ratio;

    if (unknown != 65000 && unknown < ham + dlv + spam)
        m_cur->Logger << TLOG_ERR << "shinhle info:  unknown = " << unknown << " ham = " << ham << " spam = " << spam << " dlv = " << dlv;

    if (white == spam || (white < 10 && spam < 10))
        return;

    if (white > spam) {
        pwb = "WB";
        value = white;
        value_ratio = spam ? white / spam : 0;
    } else {
        pwb = "BW";
        value = spam;
        value_ratio = white ? spam / white : 0;
    }

    rv_wb = GetRangeValue(range_wb, sizeof(range_wb) / sizeof(range_wb[0]), value);

    if (rv_wb < 2 || value_ratio == 1)
        return;

    if (value_ratio > 1) {
        rv_ratio = GetRangeValue(range_ratio, sizeof(range_ratio) / sizeof(range_ratio[0]), value_ratio);
        if (!rv_ratio)
            return;
    } else
        rv_ratio = 5;

    SetShingleRule(pwb, sh_type, prefix_let[rv_wb], prefix_let[rv_ratio]);
}

int TRengine::GetRangeValue(ui32 range[], int count, ui32 value) {
    int ind;
    for (ind = 0; ind < count - 1; ind++)
        if (value <= range[ind])
            return ind;

    return ind;
}

void TRengine::SetShingleRule(const char* prefix, ui8 sh_type, char value1, char value2) {
    char str[str_short_size];

    sprintf(str, "%s_%d_%c_%c", prefix, sh_type, value1, value2);
    m_cur->rulesContext.SetRule(str);
}

void TRengine::AddSiteNetc(const TString& pDomen) {
    char crs_str[32];
    if (pDomen.size() < 3)
        return;

    if (calc_strcrc64(pDomen, crs_str)) {
        m_cur->m_vsite_netc_cs.push_back(crs_str);
        m_cur->m_vsite_netc.push_back(pDomen);
    }
}
// 2 sc
// 4 ws
// 8 ph
// 16 ob
// 32 ab
// 64 jp

bool TRengine::SetSURBLRules(int _mask) // return true for all zones except yandex
{
    size_t position = 0;
    std::bitset<8> mask(_mask);
    for (const char *rule : { "PATTERN_YA", "PATTERN_SC", "PATTERN_WS", "PATTERN_PH", "PATTERN_OB", "PATTERN_AB", "PATTERN_JB", "PATTERN_CR" }) {
        if (mask.test(position++))
            m_cur->rulesContext.SetRule(rule);
    }

    const bool result = (mask >>= 1).any(); //skip yandex
    if (result)
        m_cur->rulesContext.SetRule("PATTERN_BL");

    return result;
}

void TRengine::CheckRange(const TString& key, int value, bool fPrintWarning) {
    m_pRulesHolder->m_spruler->CheckRange(*m_cur, key, value, fPrintWarning);
}

void TRengine::CheckRange(const TString& key, double value, bool fPrintWarning) {
    m_pRulesHolder->m_spruler->CheckRange(*m_cur, key, value, fPrintWarning);
}

void TRengine::CheckRange(const TString& key, ui32 value, bool fPrintWarning) {
    return CheckRange(key, int(value), fPrintWarning);
}

void TRengine::CheckRange(const TString& key, ui64 value, bool fPrintWarning) {
    m_pRulesHolder->m_spruler->CheckRange(*m_cur, key, value, fPrintWarning);
}

std::pair<TStringBuf, rcptattrs> ParseAttrsFromEnvelope(const TStringBuf& rcpt) {
    const TStringBuf rcptString = StripString(rcpt);
    TStringBuf email, tail;

    if(const size_t space = rcptString.find(' ')) {
        email = rcptString.substr(0, space);
        tail = rcptString.substr(space);
    } else {
        email = rcptString;
    }

    rcptattrs attrs;
    if (auto bornDate = Parseout(tail, "borndate=", " ")) {
        attrs.SetBornDate(TString(bornDate));
    }

    if (auto suid = Parseout(tail, " id=", " ")) {
        attrs.SetSUID(TSuid{suid});
    }

    // fill rcpt-UID dictionary
    if (auto uid = Parseout(tail, " uid=", " ")) {
        attrs.SetUID(TUid{uid});
    }

    // fill country codes
    if (auto country = Parseout(tail, "country=", " ")) {
        attrs.SetCountry(TString{country});
    }

    return {email, std::move(attrs)};
}

// rcpt field format
// old - login;login;...login
// new - login suid;login suid;...login suid
void TRengine::CheckRcpt(TStringBuf rcpt, TStringBuf originalRcpt) {
    using namespace NHtmlSanMisc;
    originalRcpt.SkipPrefix("<");
    originalRcpt.ChopSuffix(">");

    m_pstat.AddStat("orcp") << originalRcpt;
    m_pstat.AddStat("old_rcpt") << rcpt;

    bool allYa = true;
    TString sYandTeamSuid;
    TString sYandTeamUid;
    rcpt.SkipPrefix("<");
    rcpt.ChopSuffix(">");

    auto senderRecord = m_pstat.AddStat("senders");

    for (auto it : StringSplitter(rcpt).Split(';').SkipEmpty().Take(m_RcptMax)) {
        auto [email, attrs] = ParseAttrsFromEnvelope(it.Token());

        CheckFieldAllRules(FD_RCP_TO, it.Token());

        if(m_cur->So2Context) {
            attrs.sender = m_cur->So2Context->FindSenderByUid(attrs.sUid);
            if(attrs.sender)
                senderRecord << *attrs.sender << ',';
        }

        if (attrs.BornDate) {
            CheckField(FD_RCPT_BORN_DATE, attrs.BornDate);
        }

        if (attrs.sSuid) {
            if (m_pRulesHolder->m_pListRuler.CheckWord(m_cur->rulesContext, attrs.sSuid, SP_LIST_YTEAM_SUID)) {
                if (sYandTeamSuid.length() > 0)
                    sYandTeamSuid.append(", ");
                sYandTeamSuid.append(attrs.sSuid);
            }

            m_pRulesHolder->m_pListRuler.CheckWord(m_cur->rulesContext, attrs.sSuid, SP_LIST_SUIDS_RCPT);
            m_cur->m_mapSuidReputation.emplace(attrs.sSuid, TSoPersonal{});
        }

        // fill rcpt-UID dictionary
        if (attrs.sUid) {
            m_cur->DlvLogRequest.RcptsUids.emplace_back(attrs.sUid);
            AddPattern(attrs.sUid, EN_SH_UID);

            if (m_pRulesHolder->m_pListRuler.CheckWord(m_cur->rulesContext, attrs.sUid, SP_LIST_YTEAM_UID)) {
                if (sYandTeamUid.length() > 0)
                    sYandTeamUid.append(", ");
                sYandTeamUid.append(attrs.sUid);
            }
            m_pRulesHolder->m_pListRuler.CheckWord(m_cur->rulesContext, attrs.sUid, SP_LIST_UIDS_RCPT);

            if (Pools->UaasRequester) {
                try {
                    std::optional<TExpBoxes> expBoxes;
                    NJson::TJsonValue::TArray expFlags;
                    bool res = Pools->UaasRequester->CheckUaas(attrs.sUid, expBoxes, expFlags, m_cur->Logger);
                    if (res) {
                        if (expBoxes) {
                            m_pstat.AddStat(ST_UAAS) << expBoxes->RawExpBoxes;
                            CheckField(FD_UAAS_IDS, expBoxes->RawExpBoxes);
                            attrs.ExpBoxes = std::move(expBoxes);
                        }

                        for (const auto &flag : expFlags)
                            CheckValueAllRules(FD_X_YANDEX_EXPFLAGS, flag);
                    } else {
                        m_cur->rulesContext.SetRule("UAAS_FAIL");
                    }
                } catch (...) {
                    m_cur->rulesContext.SetRule("UAAS_FAIL");
                    m_cur->Logger << (TLOG_ERR) << __FUNCTION__ << ' ' << CurrentExceptionMessageWithBt();
                }
            }
        } else {
            allYa = false;
        }

        m_cur->m_rcptattrs.emplace(email, std::move(attrs));

        m_pRulesHolder->m_pListRuler.CheckWord(m_cur->rulesContext, email, SP_LIST_RCPT);
    }

    if(allYa) {
        m_cur->rulesContext.SetRule("RCP_ALL_YA");
    }

    if (sYandTeamSuid.length())
        m_pstat.AddStat(ST_LOG) << "ytsuid " << sYandTeamSuid;
    if (sYandTeamUid.length())
        m_pstat.AddStat(ST_LOG) << "ytUID " << sYandTeamUid;


    char first_letter[256];
    memset(first_letter, 0, 256);
    int c_letters = 0, c_rcpt = 0, c_unsorted = 0;
    TStringBuf lastLogin;
    for(auto it : StringSplitter(originalRcpt).Split(';').SkipEmpty().Take(m_RcptMax) ) {
        c_rcpt++;
        auto [email, attrs] = ParseAttrsFromEnvelope(it.Token());

        if(m_cur->So2Context) {
            attrs.sender = m_cur->So2Context->FindSenderByUid(attrs.sUid);
        }

        m_cur->m_original_rcptattrs.emplace(email, std::move(attrs));

        const auto fl = (ui8)email.front();
        if (first_letter[fl] == 0) {
            first_letter[fl] = 1;
            ++c_letters;
        }

        if(lastLogin) {
            if (lastLogin >= email)
                ++c_unsorted;
        } else {
            lastLogin = email;
        }

    }

    m_cur->c_rcpt = c_rcpt;

    m_pRulesHolder->m_spruler->CheckRange(*m_cur, "rcp_to_count", c_rcpt, false);
    m_cur->MlLogBuilder.Add("rcp_to_count", ToString(c_rcpt));
    if (c_rcpt > 1) {
        m_pRulesHolder->m_spruler->CheckRange(*m_cur, "rcpto_letters", c_letters, false);
        m_pRulesHolder->m_spruler->CheckRange(*m_cur, "rcpto_unsorted", c_unsorted, false);
    }

    FilterModelsAccordingToUaas();
}

void TRengine::FilterModelsAccordingToUaas() {
    TAppliersMap& models = *Models;

    const M_RCPT_ATTRS& rcptAttrs = m_cur->m_rcptattrs;

    models.erase(std::remove_if(models.begin(), models.end(), [&rcptAttrs](const auto & p) {
        const TModelConfig& modelConfig = p.first;
        const bool accepted = rcptAttrs.empty() || AnyOf(rcptAttrs, [&modelConfig](const auto& p){
            const rcptattrs& attrs = p.second;
            return !attrs.ExpBoxes || modelConfig.AcceptExperimentId(*attrs.ExpBoxes);
        });

        return !accepted;
    }), models.cend());
}

void TRengine::SetPopperSuid(TStringBuf field) {
    if(m_cur->m_rcptattrs.empty())
        return;

    const size_t fieldlen = field.size();
    while(!field.empty() && isdigit(field.front()))
        field.Skip(1);

    if (field.empty())
        return;

    const auto it = m_cur->m_rcptattrs.cbegin();
    auto prcpt_login = it->first;
    auto p_suid = it->second.sSuid;

    if (prcpt_login.size() == 0 || prcpt_login.size() > 1024)
        return;

    if (!p_suid || p_suid.length() == 0 || p_suid.length() == 1 && p_suid[0] == '0') {

        if (prcpt_login.size() != fieldlen - field.size() - 1)
            m_pstat.AddStat(ST_LOG) << "SetPoperSuid fault";

        m_cur->m_mapSuidReputation.clear();
        m_cur->m_rcptattrs.clear();

        m_cur->m_mapSuidReputation.emplace(field, TSoPersonal{});
        m_cur->m_rcptattrs[prcpt_login].SetSUID(TSuid{field});
    }
}

void TRengine::CheckMailFrom(const TString& mf) {

    if (!mf)
        return;

    auto inp_envfrom = mf.c_str();

    char *pDupText = 0, *envfrom = 0;
    int len = strlen(inp_envfrom);
    // do not calc 32-33 shingles for same domains
    const bool fPddLocal = m_cur->rulesContext.IsRuleWorked("ORG_ID_SAME_ALL_SO2");

    STRDUPLWR(&pDupText, inp_envfrom, len);

    if (*pDupText == '<')
        envfrom = pDupText + 1;
    else
        envfrom = pDupText;

    if (pDupText[len - 1] == '>')
        pDupText[len - 1] = 0;

    CheckFieldAllRules(FD_MAIL_FROM, envfrom);
    if (!is_spk)
        m_pstat.AddStat(ST_MAIL_FROM) << envfrom;

    if (Config.fWebMail) {
        const char* psuid;
        int suidlen;
        int login_len;
        const char* plogin_end;

        if (strstr(inp_envfrom, "tel=1"))
            m_cur->rulesContext.SetRule("USER_HAS_PHONE");

        if (m_cur->m_sMailFromUID.length()) // m_cur->m_sMailFromUID already set outside, by SpSetSenderUID call
            m_pRulesHolder->m_pListRuler.CheckWord(m_cur->rulesContext, m_cur->m_sMailFromUID, SP_LIST_UIDS_MAILFROM);

        if ((psuid = strstr(pDupText, " id="))) {
            login_len = psuid - pDupText;
            psuid += 4;
            if (!(suidlen = GetDigitTextLen((const unsigned char*)psuid)))
                m_cur->Logger << TLOG_ERR << "mailfrom format: pDupText (psuid = " << psuid << ")";
            else {
                m_cur->m_sMailFromSuid.assign(psuid, suidlen);
                m_cur->DlvLogRequest.Suid = m_cur->m_sMailFromSuid;
                m_pRulesHolder->m_pListRuler.CheckWord(m_cur->rulesContext, {psuid, size_t(suidlen)}, SP_LIST_SUIDS_MAILFROM);
            }

            // extract karma, karma_status and borndate from envelopefrom for so-out
            const char* start = nullptr;
            size_t lenkarma = 0;

            if ((start = strstr(pDupText, "karma="))) {
                start += 6;
                lenkarma = GetDigitTextLen((const unsigned char*)start);
                if (lenkarma)
                    CheckField(FD_X_YANDEX_KARMA, {start, lenkarma});
            }

            if ((start = strstr(pDupText, "karma_status="))) {
                start += 13;
                lenkarma = GetDigitTextLen((const unsigned char*)start);
                if (lenkarma)
                    CheckField(FD_X_YANDEX_KARMA_STATUS, {start, lenkarma});
            }

            if ((start = strstr(pDupText, "borndate="))) {
                start += 9;
                lenkarma = GetDigitTextLen((const unsigned char*)start);
                if (lenkarma)
                    CheckField(FD_X_BORNDATE, {start, lenkarma});
            }

        } else if ((plogin_end = strchr(pDupText, ' ')))
            login_len = plogin_end - pDupText;
        else
            login_len = strlen(pDupText);

        TStringBuf domain;
        if (m_cur->rulesContext.IsRuleWorked("PDD") && !(domain = GetDomain({pDupText, size_t(login_len)})).empty() && !IsYandexDomain(domain)) {
            TString s_res;
            TString sPddDomain_Rcpto;
            ;

            const auto aliasesDomains = [this] {
                auto range = NFuncTools::Map(GetDomain, m_cur->MailFromAliases);
                return THashSet<TStringBuf>(range.begin(), range.end());
            }();

            for(const auto & [prcpt_login, _] : m_cur->m_rcptattrs) {
                const TStringBuf rcpt_domain = GetDomain(prcpt_login);
                if (rcpt_domain != domain && !aliasesDomains.contains(rcpt_domain)) // rcpt domain differs login domain
                {
                    sPddDomain_Rcpto.assign(domain);
                    sPddDomain_Rcpto.append("_");
                    sPddDomain_Rcpto.append(prcpt_login);
                    AddPattern(sPddDomain_Rcpto, EN_SH_PDD_DOMN_UNIQ);
                    AddPattern(domain, EN_SH_UID_PDD_DOMN);
                }
                if (fPddLocal) {
                    m_cur->c_InternalPdd++;
                }
            }

            if (fPddLocal && m_cur->m_rcptattrs && !m_cur->rulesContext.IsRuleWorked("PDD_OPENREG")) {
                m_pstat.AddStat(ST_LOG) << "pdd local";
                m_cur->fInternalPddMail = true;
                AddPattern(domain, EN_SH_PDD);
            }
        }
    }

    char* pend = 0;
    if ((pend = strchr(pDupText, ' ')))
        *pend = 0; // login = id

    m_pRulesHolder->m_pListRuler.CheckWord(m_cur->rulesContext, envfrom, SP_LIST_MAILFROM);

    char* pdomain_envfrom = 0;
    if ((pdomain_envfrom = strchr(envfrom, '@'))) {
        ++pdomain_envfrom;
        m_pRulesHolder->m_pListRuler.CheckDomainName(m_cur->rulesContext, pdomain_envfrom, SP_LIST_DOMAINMAILFROM, SP_LIST_DOMAINMAILFROM_PART);
        m_pRulesHolder->m_pListRuler.CheckDomainName(m_cur->rulesContext, pdomain_envfrom, SP_LIST_DOMAIN, SP_LIST_DOMAIN_PART);

        Add2UrlReputationList(pdomain_envfrom, EUrlStaticticSource::SENDER);
    }

    {
        m_cur->MlLogBuilder.Add(ToString(ST_MAIL_FROM), Config.fWebMail ? inp_envfrom : envfrom);
    }

    // for big PDDs perform all shingles; for small - 29th only
    // https://st.yandex-team.ru/SODEV-971
    if (Config.fWebMail && fPddLocal && m_cur->rulesContext.IsRuleWorked("PDD_OPENREG"))
        m_cur->fInternalPddMail = false;

    m_cur->m_sMailFrom.assign(envfrom);

    if(m_cur->m_sMailFrom) {
        AddPattern(m_cur->m_sMailFrom, EN_SH_MAIL_FROM);
    }

    if (!m_cur->fInternalPddMail) {
        if (m_cur->m_sMailFromUID.length())
            AddPattern(m_cur->m_sMailFromUID, EN_SH_MAILFROM_UID);
        else
            AddPattern(envfrom, EN_SH_MAILFROM_UID);

        if (m_cur->m_sMailFromUID)
            AddPattern(m_cur->m_sMailFromUID, EN_SH_MFRM_UID);
        else
            AddPattern(envfrom, EN_SH_MFRM_UID);
    }

    DELETE_ARR(pDupText);
}

void TRengine::CheckRcptMail(const TString& to) {
    if (!m_cur->m_rcptattrs.contains(TLogin(to_lower(to))))
        ++m_cur->c_rcpt_missmatch;
}

void TRengine::AddPatternIpTo(const TStringBuf& pIpLast) {
    //    const char * prcpt_login;
    //    TString *p_suid;
    TString s_res;
    int clogins = 0;

    for (M_RCPT_ATTRS::iterator mIt = m_cur->m_rcptattrs.begin(); (mIt != m_cur->m_rcptattrs.end()) && (++clogins < 4); mIt++) {
        s_res.assign(pIpLast);
        s_res.append("_");
        if (mIt->second.sUid.length() == 0 || (mIt->second.sUid.length() == 1 && mIt->second.sUid[0] == '0')) {
            if (mIt->first.length())
                s_res.append(mIt->first);
            else
                continue;
        } else
            s_res.append(mIt->second.sUid);

        AddPattern(s_res, EN_SH_IP_RCP_TO);
    }

    // 17th shingle changed to UID

    /*
    m_cur->m_mapRcpt.SetFirst();
    for(int clogins = 0; m_cur->m_mapRcpt.GetNext(&prcpt_login, &p_suid) && clogins < 4; clogins++)
    {
        s_res.assign(pIpLast);
        s_res.append("_");
        if (p_suid->length() == 0 || p_suid->length() == 1 && p_suid->c_str()[0] == '0')
        {
            if (!prcpt_login  || *prcpt_login == 0)
                continue;
            s_res.append(prcpt_login);
        }
        else
            s_res.append(p_suid->c_str());
        clogins++;
        AddPattern(s_res.c_str(),  s_res.length(), EN_SH_IP_RCP_TO);
    }
    */
}

// true if change message result for suid
bool TRengine::CheckUserId(TReputationComplaintSet& cs, const char* userId, bool fMessageSpam) {
    //    bool fYandexTeamUser = false;
    int criterion;
    if (!cs.From.fset && !cs.BeenSender.fset && !cs.List.fset && !cs.Subj.fset &&
        !cs.Messid.fset && !cs.FromDomain.fset && !cs.Paysender.fset && !cs.Paysender_Nofrom.fset)
        return false;

    //  SP_LIST_YTEAM_SUID already checked at CheckRcpt
    //    if (GetListRuler()->CheckList(psuid, SP_LIST_YTEAM_SUID))
    //        fYandexTeamUser = true;

    TSpClass complaint_res = CheckComplaints(cs, &criterion);

    //    TSoPersonal *pres_personal = 0;
    if (complaint_res == TSpClass::HAM || complaint_res == TSpClass::SPAM) {
        m_pstat.AddStat(ST_COMPLAINT) << "crit2" << ' ' << userId << ' ' << (int)complaint_res << ' ' << criterion;
        m_cur->fNextPrepareScore = false; // must not consider ABOOK if personal filter
    }

    M_RCPT_ATTRS::iterator it;
    for (it = m_cur->m_rcptattrs.begin(); it != m_cur->m_rcptattrs.end(); it++) {
        if (it->second.sSuid.length() && strcmp(userId, it->second.sSuid.c_str()) == 0)
            break;
    }

    if (it != m_cur->m_rcptattrs.end()) // userId found
    {
        it->second.pres_personal.compl_spam = cs.From.complaintspam;
        it->second.pres_personal.compl_ham = cs.From.complaintham;

        if (it->second.pres_personal.fromto == TSpClass::HAM && cs.From.complaintspam > 0) {
            m_pstat.AddStat(ST_COMPLAINT) << "undo MCOUNT by complaintspam " << userId;
            m_cur->fUndoMcountByCompl = true;
            it->second.pres_personal.fromto = TSpClass::UNKNOWN;
        }

        if (m_cur->fPersonalFilterUndo || m_cur->rulesContext.IsRuleWorked("UNDO_PERSONAL_FILTER") ||
            m_cur->rulesContext.IsRuleWorked("UNDO_WHITE_PERSONAL_FILTER") && fMessageSpam ||
            m_cur->rulesContext.IsRuleWorked("UNDO_BLACK_PERSONAL_FILTER") && !fMessageSpam) {
            if (!fMessageSpam && complaint_res == TSpClass::SPAM || fMessageSpam && complaint_res == TSpClass::HAM)
                m_pstat.AddStat(ST_COMPLAINT) << "undo " << userId;
            it->second.pres_personal.complaint = TSpClass::UNKNOWN;
            return false;
        } else
            it->second.pres_personal.complaint = complaint_res;
    }
    return true;
}

//
TSpClass TRengine::CheckComplaints(const TReputationComplaintSet& cs, int* criterion) {
    bool fAllowSubjMid = !(m_cur->rulesContext.IsRuleWorked("UNDO_SUBJ_MESS_PERSONAL_FILTER"));

    ui32 _FS = cs.From.complaintspam;
    ui32 FH = cs.From.complaintham;
    ui32 _DS = cs.FromDomain.complaintspam;
    ui32 DH = cs.FromDomain.complaintham;
    ui32 BS = cs.BeenSender.complaintspam;
    ui32 BH = cs.BeenSender.complaintham;
    ui32 LS = cs.List.complaintspam;
    ui32 LH = cs.List.complaintham;
    ui32 _SS = cs.Subj.complaintspam;
    ui32 SH = cs.Subj.complaintham;
    ui32 MS = cs.Messid.complaintspam;
    ui32 MH = cs.Messid.complaintham;
    ui32 PS = cs.Paysender.complaintspam;
    ui32 PH = cs.Paysender.complaintham;
    ui32 PFS = cs.Paysender_Nofrom.complaintspam;
    ui32 PFH = cs.Paysender_Nofrom.complaintham;

    m_pstat.AddStat(ST_LOG) << "SH "
                            << _FS << ' ' << FH << "  "
                            << _DS << ' ' << DH << "  "
                            << BS  << ' ' << BH << "  "
                            << LS  << ' ' << LH << "  "
                            << _SS << ' ' << SH << "  "
                            << MS  << ' ' << MH << "  "
                            << PS  << ' ' << PH << ' ' << PFS << ' ' << PFH;

    if (cs.From.fUnsubscribe || cs.BeenSender.fUnsubscribe || cs.List.fUnsubscribe) {
        *criterion = 23;
        m_cur->rulesContext.SetRule("U_PF");
        return TSpClass::SPAM;
    }

    *criterion = 1;
    // ham
    if (!_FS) {
        if (FH || !BS && BH || !LS && LH || fAllowSubjMid && !_SS && !MS && SH && MH)
            return TSpClass::HAM;
    }

    *criterion = 17;
    if (!_DS && DH)
        return TSpClass::HAM;

    // spam
    bool FSign = cs.From.fSignificantSpamComplaint;
    bool DSign = cs.FromDomain.fSignificantSpamComplaint;
    bool BSign = cs.BeenSender.fSignificantSpamComplaint;
    bool LSign = cs.List.fSignificantSpamComplaint;
    bool SSign = cs.Subj.fSignificantSpamComplaint;
    bool MSign = cs.Messid.fSignificantSpamComplaint;
    bool PNFSign = cs.Paysender_Nofrom.fSignificantSpamComplaint;

    m_pstat.AddStat(ST_LOG) << "Sign " << FSign << " " << DSign << "  " << BSign << " " << LSign << "  " << SSign << " " << MSign;

    *criterion = 2;
    if (!FH) {
        if (_FS > 1 && FSign || BSign && !BH && BS > 1 || LSign && !LH && LS > 1 ||
            fAllowSubjMid && SSign && MSign && !SH && !MH && _SS > 2 && MS > 2)
            return TSpClass::SPAM;
    }

    *criterion = 3;
    if (!DH) {
        if (_DS > 2 && DSign)
            return TSpClass::SPAM;
    }

    int LastFH = 0, LastFS = 0, LastDH = 0, LastDS = 0, LastBH = 0, LastBS = 0, LastLH = 0, LastLS = 0, LastSH = 0, LastSS = 0, LastMH = 0, LastMS = 0,
            LastPH = 0, LastPS = 0, LastPFH = 0, LastPFS = 0;

    GetLastComplaints(cs.From, LastFH, LastFS);
    GetLastComplaints(cs.FromDomain, LastDH, LastDS);
    GetLastComplaints(cs.BeenSender, LastBH, LastBS);
    GetLastComplaints(cs.List, LastLH, LastLS);
    GetLastComplaints(cs.Subj, LastSH, LastSS);
    GetLastComplaints(cs.Messid, LastMH, LastMS);
    GetLastComplaints(cs.Paysender, LastPH, LastPS);
    GetLastComplaints(cs.Paysender_Nofrom, LastPFH, LastPFS);

    m_pstat.AddStat(ST_LOG) << "L "
                            << LastFH << ' ' << LastFS << "  "
                            << LastDH << ' ' << LastDS << "  "
                            << LastBH << ' ' << LastBS << "  "
                            << LastLH << ' ' << LastLS << "  "
                            << LastSH << ' ' << LastSS << "  "
                            << LastMH << ' ' << LastMS << "  "
                            << LastPH << ' ' << LastPS << ' ' << LastPFH << ' ' << LastPFS;

    if (LastFH > 2 || LastBH > 2 || LastLH > 2) {
        *criterion = 21;
        return TSpClass::HAM;
    }

    if (LastFH > 0 && FH > _FS - 2 || LastBH > 0 && BH > BS - 2 || LastLH > 0 && LH > LS - 2) {
        *criterion = 22;
        return TSpClass::HAM;
    }

    bool fSimplifiedHam = LastFH || LastDH || LastBH || LastLH || fAllowSubjMid && LastSH && LastMH;
    bool fSimplifiedSpam = LastFS || LastDS || LastBS || LastLS || fAllowSubjMid && LastSS && LastMS;

    if (m_cur->fSimplifiedReputation) {
        *criterion = 4;
        if (fSimplifiedHam && !fSimplifiedSpam)
            return TSpClass::HAM;
        *criterion = 5;
        if (fSimplifiedSpam && !fSimplifiedHam)
            return TSpClass::SPAM;
    }

    if (m_cur->fSimplifiedReputationMid) {
        *criterion = 20;
        if (LastFH)
            return TSpClass::HAM;
        if (LastFS)
            return TSpClass::SPAM;
        if (LastDH || LastBH || LastLH)
            return TSpClass::HAM;
        if (LastDS || LastBS || LastLS)
            return TSpClass::SPAM;
        if (LastMH)
            return TSpClass::HAM;
        if (LastMS)
            return TSpClass::SPAM;
    }

    *criterion = 6;
    // ham
    if (_FS <= 1 && (LastFH > 2 || BS <= 1 && LastBH > 2 || LS <= 1 && LastLH > 2 ||
                     fAllowSubjMid && MS <= 1 && _SS <= 1 && LastSH > 2 && LastMH > 2))
        return TSpClass::HAM;

    *criterion = 7;
    if (_FS <= 2 && (LastFH > 4 || BS <= 2 && LastBH > 4 || LS <= 2 && LastLH > 4 ||
                     fAllowSubjMid && MS <= 2 && _SS <= 2 && LastSH > 4 && LastMH > 4))
        return TSpClass::HAM;

    *criterion = 8;
    if ((LastFH > 10 || LastBH > 10 || LastLH > 10 || (fAllowSubjMid && LastSH > 10 && LastMH > 10)) && !LastFS && !LastSS && !LastMS)
        return TSpClass::HAM;

    *criterion = 9;
    if (_DS <= 1 && LastDH > 2)
        return TSpClass::HAM;

    *criterion = 10;
    if (_DS <= 2 && LastDH > 4)
        return TSpClass::HAM;

    *criterion = 11;

    if (LastDH > 10)
        return TSpClass::HAM;

    if (!FSign)
        LastFS = 0;

    if (!DSign)
        LastDS = 0;

    if (!BSign)
        LastBS = 0;

    if (!LSign)
        LastLS = 0;

    if (!MSign)
        LastMS = 0;

    //spam
    *criterion = 12;
    if (FH <= 1 && (LastFS > 6 || fAllowSubjMid && LastSS > 6 && LastMS > 6))
        return TSpClass::SPAM;

    *criterion = 13;
    if (FH <= 2 && (LastFS > 19 || fAllowSubjMid && MH <= 2 && SH <= 2 && LastSS > 19 && LastMS > 19))
        return TSpClass::SPAM;

    *criterion = 14;
    if (DH <= 1 && LastDS > 6)
        return TSpClass::SPAM;

    *criterion = 15;
    if (DH <= 2 && LastDS > 19)
        return TSpClass::SPAM;

    *criterion = 24; // paysenders
    if (LastPH)
        return TSpClass::HAM;

    *criterion = 25; // paysenders spam
    if (LastPS)
        return TSpClass::SPAM;

    *criterion = 26; // paysender no from
    if (LastPFH > 2)
        return TSpClass::HAM;

    *criterion = 27; // paysender no from spam
    if (LastPFS > 2 && PNFSign)
        return TSpClass::SPAM;

    *criterion = 16;

    //spam
    //    if (!fSimplifiedHam && (LastFS > 30 || LastDS > 30 || LastBS > 30 || LastLS > 30 || fAllowSubjMid && LastSS > 30 && LastMS  > 30)
    //        return spSpam;
    //    *criterion = 17;

    return TSpClass::UNKNOWN;
}

void TRengine::PrintFromStat(TFromStatRequest& fromstatreq, bool bNewShingler) {
    m_pstat.AddStat(ST_COMPLAINT_FROM)  << (bNewShingler ? "frmstat2 " : "")
                                        << fromstatreq.From() << ' '
                                        << fromstatreq.ResFirsttimeElapsed() << ' '
                                        << fromstatreq.ResLasttimeElapsed() << "  "
                                        << fromstatreq.ResComplaintHam() << ' '
                                        << fromstatreq.ResComplaintSpam() << "  "
                                        << fromstatreq.ResHam() << ' '
                                        << fromstatreq.ResSpam() << "  "
                                        << fromstatreq.ResUniquserHam() << ' '
                                        << fromstatreq.ResUniquserSpam() << ' '
                                        << fromstatreq.ResUniquser();
}

// if true reset complaints
bool TRengine::SetComplaint(TComplaintType type, TReputationComplaint* pComplaint, TComplaintRequestListIt cit, bool fMessageSpam, const char* puserId) {
    bool reset_complatins = false;
    pComplaint->fset = true;
    pComplaint->firsttime = (*cit).ResFirsttimeElapsed();
    pComplaint->lasttime = (*cit).ResLasttimeElapsed();
    pComplaint->complaintham = (*cit).ResComplaintHam();
    pComplaint->complaintspam = (*cit).ResComplaintSpam();
    pComplaint->resham = (*cit).ResHam();
    pComplaint->resspam = (*cit).ResSpam();
    pComplaint->history = (*cit).ResHistory();
    pComplaint->replace_to_ham = (*cit).ResReplaceToHam();
    pComplaint->replace_to_spam = (*cit).ResReplaceToSpam();
    pComplaint->fUnsubscribe = (*cit).ResUnsubscribe() && !m_cur->rulesContext.IsRuleWorked("NO_UPF_LIST"); // if rule then always false

    pComplaint->fSignificantSpamComplaint = (pComplaint->resham > 0) && (2 * pComplaint->complaintspam >= pComplaint->resham);

    //    if (fMessageSpam && (*cit).ResComplaintHam() || !fMessageSpam && (*cit).ResComplaintSpam())
    {
        ui32 history = pComplaint->history;
        bool last_complaint = false;
        ui32 h_spam = 0, h_ham = 0;
        unsigned int last_spam = 0;

        auto record = m_pstat.AddStat(ST_COMPLAINT);

        record  << "cmpl" << " - " << puserId << (int)type << ' '
                << pComplaint->firsttime << " "
                << pComplaint->lasttime << "  "
                << pComplaint->complaintham << " "
                << pComplaint->complaintspam << "  "
                << pComplaint->resham << " "
                << pComplaint->resspam << "  "
                << pComplaint->replace_to_ham << " "
                << pComplaint->replace_to_spam << " "
                << pComplaint->fUnsubscribe << " ";

        int cSpamHam = 0;
        for (ui32 i = 0; i < 32 && i < (pComplaint->complaintham + pComplaint->complaintspam); i++) {
            if (pComplaint->complaintham == 0)
                history = 1;
            else if (pComplaint->complaintspam == 0)
                history = 0;
            else if (i > 0)
                history >>= 1;

            if (history & 1) // oldest compl SPAM
            {
                record << '1';
                last_complaint = true;
                ++h_spam;
                if (!h_ham)
                    ++last_spam;

                if (pComplaint->complaintspam && cSpamHam < 3)
                    cSpamHam++;
            } else // oldest compl HAM
            {
                record << '0';
                last_complaint = false;
                ++h_ham;

                if (pComplaint->complaintham && cSpamHam > -3)
                    cSpamHam--;
            }
        }

        if ((cSpamHam >= 2) && !fMessageSpam)
            m_cur->rulesContext.SetRule("NEWPF_SPAM");
        if ((cSpamHam < 0) && fMessageSpam)
            m_cur->rulesContext.SetRule("NEWPF_HAM");

        if (!last_spam && pComplaint->fUnsubscribe) {
            m_pstat.AddStat(ST_COMPLAINT) << "pf is corrupted - fUnsubscribe && last_spam == 0";
            pComplaint->fUnsubscribe = false;
        }

        if ((pComplaint->complaintham + pComplaint->complaintspam < 32) && h_spam != pComplaint->complaintspam && h_ham != pComplaint->complaintham)
            m_pstat.AddStat(ST_COMPLAINT) << "pf is corrupted";

        if (!pComplaint->fSignificantSpamComplaint && last_complaint)
            record << " cancel";

        if (m_cur->fSimplifiedReputation)
            record << " s";
        if (m_cur->fSimplifiedReputationMid)
            record << " m";

        if (!m_cur->fSimplifiedReputation && !pComplaint->fUnsubscribe &&
            last_spam > 1 && (type == KCFROM || type == KCLIST || type == KCBEENSENDER) && !fMessageSpam &&
            ((pComplaint->complaintham > 0 && last_spam > 3) || (2 * pComplaint->complaintspam + 4) < pComplaint->resham))
            reset_complatins = true;
    }

    return reset_complatins;
}

void TRengine::GetLastComplaints(const TReputationComplaint& Complaint, int& cLastHam, int& cLastSpam) const {
    ui32 history = Complaint.history;

    for (ui32 i = 0; (i < 32) && i < (Complaint.complaintham + Complaint.complaintspam); i++) {
        if (i > 0)
            history >>= 1;
        if (history & 1) {
            if (i && !cLastSpam)
                break;
            ++cLastSpam;
        } else {
            if (i && !cLastHam)
                break;
            ++cLastHam;
        }
    }
}

// true - if set personal result
bool TRengine::SetPersonalresult(TSpClass filter_res, TSpClass filter_res_fromto) {
    bool fres = false;
    bool fAll = true;
    TSpClass res;

    bool fSetfilter_res_fromto = false;
    bool fallnotcompl = true;


    {
        auto record = m_pstat.AddStat(ST_LOG);
        record << "persdeb";
        for(const auto & p : m_cur->m_mapSuidReputation) {
            const auto & pres_personal = p.second;

            record  << ' '
                    << (int)pres_personal.complaint << ' '
                    << (int)pres_personal.fromto << ' '
                    << pres_personal.compl_spam << ' '
                    << pres_personal.compl_ham;
        }
    }
    auto record = m_pstat.AddStat(ST_LOG);
    record << "pers";

    if (m_cur->m_rcptattrs.empty()) {
        m_cur->Logger << TLOG_ERR << "Set personal result error!";
        return fres;
    }

    auto it = m_cur->m_rcptattrs.begin();
    const TSpClass res_first = (it->second.pres_personal.complaint != TSpClass::UNKNOWN) ? it->second.pres_personal.complaint : it->second.pres_personal.fromto;
    record << ' ' << (int)it->second.pres_personal.complaint << ' ' << (int)it->second.pres_personal.fromto;
    if (it->second.pres_personal.compl_spam > 0)
        fallnotcompl = false;

    for (; it != m_cur->m_rcptattrs.end(); it++) {
        if (it->second.pres_personal.complaint == TSpClass::SPAM)
            filter_res_fromto = TSpClass::UNKNOWN;
        if (res_first != ((it->second.pres_personal.complaint != TSpClass::UNKNOWN) ? it->second.pres_personal.complaint : it->second.pres_personal.fromto))
            fAll = false;
        else {
            record << ' ' << (int)it->second.pres_personal.complaint << ' ' << (int)it->second.pres_personal.fromto;
        }
    }

    if (!(fAll && (res_first == TSpClass::UNKNOWN)) || (filter_res_fromto == TSpClass::HAM && filter_res != TSpClass::HAM))
        if (m_alg->CheckAddrInReceived())
            m_pstat.AddStat(ST_LOG) << "pers - forward";

    if (fAll) {
        if (res_first == TSpClass::UNKNOWN) {
            if (filter_res_fromto == TSpClass::HAM && filter_res != TSpClass::HAM && !fallnotcompl) {
                m_pstat.AddStat(ST_LOG) << "filter_res_fromto == spHam 1";
                m_cur->Logger << TLOG_INFO << "PR filter_res_fromto == spHam 1";
                m_cur->m_messclass = TSpClass::HAM;
                fres = true;
                if (m_cur->Scores.Hits >= m_cur->m_required) {
                    m_cur->rulesContext.SetScore("PERSONAL_CORRECT", -(m_cur->Scores.Hits - m_cur->m_required) - 8.);
                    m_cur->Scores.Hits = m_cur->m_required - 18.;
                    m_cur->rulesContext.SetRule("PERSONAL_CORRECT");
                    m_cur->fHamPersonalCorrect = true;
                }
            }
        } else if (res_first != filter_res) {
            if (res_first == TSpClass::HAM) {
                m_cur->rulesContext.SetScore("PERSONAL_CORRECT", -(m_cur->Scores.Hits - m_cur->m_required) - 8.);
                m_cur->Scores.Hits = m_cur->m_required - 18.;
                m_cur->fHamPersonalCorrect = true;
            } else if (res_first == TSpClass::SPAM) {
                m_cur->rulesContext.SetScore("PERSONAL_CORRECT", -(m_cur->Scores.Hits - m_cur->m_required) + 8.);
                m_cur->Scores.Hits = m_cur->m_required + 8.;
                m_cur->fSpamPersonalCorrect = true;
            }
            m_cur->rulesContext.SetRule("PERSONAL_CORRECT");
            m_cur->Logger << TLOG_INFO << "PR correct " << m_cur->m_messclass << ' ' << res_first;
            m_cur->m_messclass = res_first;

            fres = true;

            for (it = m_cur->m_rcptattrs.begin(); it != m_cur->m_rcptattrs.end(); it++) {
                if (it->second.pres_personal.complaint != TSpClass::UNKNOWN) {
                    m_cur->PersonalFilterUids.emplace(it->second.sUid);
                }
            }
        }
    } else {
        int fpersandfromto = 0;
        if (filter_res_fromto == TSpClass::HAM && filter_res != TSpClass::HAM) {
            m_pstat.AddStat(ST_LOG) << "filter_res_fromto == spHam 2";
            m_cur->Logger << TLOG_INFO << "PR filter_res_fromto == spHam 2";
            filter_res = TSpClass::HAM;
            m_cur->m_messclass = TSpClass::HAM;
            fpersandfromto = 1;
            fSetfilter_res_fromto = true;
            if (m_cur->Scores.Hits >= m_cur->m_required) {
                m_cur->rulesContext.SetScore("PERSONAL_CORRECT", -(m_cur->Scores.Hits - m_cur->m_required) - 8.);
                m_cur->Scores.Hits = m_cur->m_required - 18.;
                m_cur->rulesContext.SetRule("PERSONAL_CORRECT");
                m_cur->fHamPersonalCorrect = true;
            }
        }

        for (it = m_cur->m_rcptattrs.begin(); it != m_cur->m_rcptattrs.end(); it++) {
            if (fSetfilter_res_fromto)
                it->second.pres_personal.fromto = TSpClass::HAM;
            res = (it->second.pres_personal.complaint != TSpClass::UNKNOWN) ? it->second.pres_personal.complaint : it->second.pres_personal.fromto;
            if (res != TSpClass::UNKNOWN && res != filter_res || fSetfilter_res_fromto && it->second.pres_personal.compl_spam > 0) {
                if (it->second.pres_personal.complaint != TSpClass::UNKNOWN) {
                    m_cur->PersonalFilterUids.emplace(it->second.sUid);
                }
                if (fpersandfromto)
                    fpersandfromto = 2;
                fres = true;
            }
        }

        if (fpersandfromto == 2)
            m_pstat.AddStat(ST_LOG) << "pers and ft";
    }

    return fres;
}

void TRengine::PrintNewSenderRepRow(const TString& pfrom, const NSenderReputation::TGetData* getData) {
    TString record = TStringBuilder{} << *getData;

    m_pstat.AddStat(ST_SENDER) << pfrom << ':' << record;
    m_cur->MlLogBuilder.Add(ToString(ST_SENDER), std::move(record));
}

void TRengine::PrintNewDomainRepRow(const NSenderReputation::TGetData* getData) {
    char str[512];
    TString sStr;
    TString sTmp;

    sTmp = " ";
    if (getData->ResIsDomenDL_FBR())
        sTmp.append('D');
    if (getData->ResIsDomenFIXED_SOURCE())
        sTmp.append('S');
    if (getData->ResIsDomenSOURCE_LEVEL1())
        sTmp.append('L');
    if (getData->ResIsDomenWEBMASTER())
        sTmp.append('W');
    if (getData->ResIsDomenMALIC())
        sTmp.append('M');

    snprintf(str, sizeof(str) - 1, "%d %" PRId64 "_ %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 "_ %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 "_ %" PRId64 " %" PRId64 "_ %" PRId64 " %" PRId64 " %" PRId64 "_%s",
             getData->ResDomenFirsttimeElapsed(), getData->ResDomenActiveDays(),
             getData->ResDomen2WeeksHamCount(), getData->ResDomen2WeeksSpamCount(),
             getData->ResDomen2WeeksComplHamCount(), getData->ResDomen2WeeksComplSpamCount(),
             getData->ResDomen2WeeksPersHamCount(), getData->ResDomen2WeeksPersSpamCount(), getData->ResDomen2WeeksKoobaCount(),
             getData->ResDomenHam(), getData->ResDomenSpam(),
             getData->ResDomenComplaintHam(), getData->ResDomenComplaintSpam(),
             getData->ResDomenPersonalHam(), getData->ResDomenPersonalSpam(),
             getData->ResDomenAbookCount(), getData->ResDomenRcvdFromYandexCount(),
             getData->DomenDeletedWithoutRead2Weeks(), getData->DomenDeletedWithoutRead(), getData->DomenDkimDeletedWithoutRead(),
             sTmp.c_str());

    sStr.assign(str);
    sTmp.clear();

    m_pstat.AddStat(ST_DOMAINSENDER) << sStr;
    m_cur->MlLogBuilder.Add(ToString(ST_DOMAINSENDER), sStr);
}

void TRengine::PrintNewPaysenderRepRow(const NSenderReputation::TGetData* getData) {
    TStringStream str;

    str
            << getData->ResPSndrFirsttimeElapsed() << ' '
            << getData->ResPSndrActiveDays() << "_ "

            << getData->ResPSndr2WeeksHamCount() << ' '
            << getData->ResPSndr2WeeksSpamCount() << ' '
            << getData->ResPSndr2WeeksComplHamCount() << ' '
            << getData->ResPSndr2WeeksComplSpamCount() << ' '
            << getData->ResPSndr2WeeksPersHamCount() << ' '
            << getData->ResPSndr2WeeksPersSpamCount() << ' '
            << getData->ResPSndr2WeeksKoobaCount() << "_ "

            << getData->ResPSndrHam() << ' '
            << getData->ResPSndrSpam() << ' '
            << getData->ResPSndrComplaintHam() << ' '
            << getData->ResPSndrComplaintSpam() << ' '
            << getData->ResPSndrPersonalHam() << ' '
            << getData->ResPSndrPersonalSpam() << "_ "

            << getData->ResPSndrAbookCount();

    m_pstat.AddStat(ST_PAYSENDER) << str.Str();
    m_cur->MlLogBuilder.Add(ToString(ST_PAYSENDER), str.Str());
}

static NJson::TJsonValue SendersToJson(const NSenderReputation::TSender& sender) {
    NJson::TJsonValue jsonContext;
    {
        const NSenderReputation::TDBSender &totals = sender.totals;

        jsonContext["active_days"] = totals.activedays;
        jsonContext["firsttime"] = totals.firsttime;
        jsonContext["lasttime"] = totals.lasttime;
        jsonContext["ham"] = totals.ham;
        jsonContext["spam"] = totals.spam;
        jsonContext["compl_ham"] = totals.complaint_ham;
        jsonContext["compl_spam"] = totals.complaint_spam;
        jsonContext["pers_ham"] = totals.personalham;
        jsonContext["pers_spam"] = totals.personalspam;
        jsonContext["pop3_ham"] = totals.pop3_ham;
        jsonContext["pop3_spam"] = totals.pop3_spam;
        jsonContext["rcvd_fromyandex"] = totals.rcvd_fromyandex_count;
        jsonContext["abook_count"] = totals.abook_count;
        jsonContext["kooba_count"] = totals.kooba_count;
        jsonContext["read"] = totals.read;
        jsonContext["del_wo_read"] = totals.del_wo_read;
        jsonContext["dkim_del_wo_read"] = totals.dkim_del_wo_read;
        jsonContext["dkim_ham"] = totals.dkim_ham;
        jsonContext["dkim_spam"] = totals.dkim_spam;
        jsonContext["compl_dkim_spam"] = totals.compl_dkim_spam;
        jsonContext["ps_ham"] = totals.del_wo_read;
        jsonContext["ps_spam"] = totals.del_wo_read;
    }
    {
        const NSenderReputation::TDBPeriodicSender &periodic = sender.periodic;

        jsonContext["ham_count_2w"] = periodic.ham_count;
        jsonContext["spam_count_2w"] = periodic.spam_count;
        jsonContext["compl_ham_count_2w"] = periodic.compl_ham_count;
        jsonContext["compl_spam_count_2w"] = periodic.compl_spam_count;
        jsonContext["pers_ham_count_2w"] = periodic.pers_ham_count;
        jsonContext["pers_spam_count_2w"] = periodic.pers_spam_count;
        jsonContext["kooba_count_2w"] = periodic.kooba_count;
        jsonContext["read_2w"] = periodic.read;
        jsonContext["del_wo_read_2w"] = periodic.del_wo_read;
        jsonContext["ps_ham_2w"] = periodic.ps_ham;
        jsonContext["ps_spam_2w"] = periodic.ps_spam;
        jsonContext["dkim_ham_count_2w"] = periodic.dkim_ham_count;
        jsonContext["dkim_spam_count_2w"] = periodic.dkim_spam_count;
        jsonContext["dkim_del_wo_read_2w"] = periodic.dkim_del_wo_read;
    }

    return jsonContext;
}

void TRengine::ProceedSenderReputationInfo(const NSenderReputation::TGetData& senderRepInfo, const TString& pfrom, size_t sndr_today, ui64 sh14_ham) {

    NJson::TJsonValue luaSendersContext;

    if(m_cur->iSenderReputationFlags.Test(NSenderReputation::TSenderType::Sender)) {
        PrintNewSenderRepRow(pfrom, &senderRepInfo);

        luaSendersContext["sender"] = SendersToJson(senderRepInfo.sender);
        CheckRange("sndr_age", (int)(senderRepInfo.ResSenderFirsttimeElapsed()), false);
        CheckRange("sndr_active_days", senderRepInfo.ResSenderActiveDays(), false);
        CheckRange("sndr_active_days_perc", 100 * senderRepInfo.ResSenderActiveDays() / (1 + senderRepInfo.ResSenderFirsttimeElapsed()), false);
        CheckRange("sndr_ham", senderRepInfo.ResSenderHam(), false);
        CheckRange("sndr_spam", senderRepInfo.ResSenderSpam(), false);
        CheckRange("sndr_complaint_hamcount", senderRepInfo.ResSenderComplaintHam(), false);
        CheckRange("sndr_complaint_spamcount", senderRepInfo.ResSenderComplaintSpam(), false);
        CheckRange("sndr_personal_hamcount", senderRepInfo.ResSenderPersonalHam(), false);
        CheckRange("sndr_personal_spamcount", senderRepInfo.ResSenderPersonalSpam(), false);

        CheckRange("sndr_pop3_ham_perc",
                   (int)(1000 * senderRepInfo.SenderPop3Ham() / (senderRepInfo.SenderPop3Ham() + senderRepInfo.SenderPop3Spam() + 1)));
        CheckRange("sndr_pop3_spam_perc",
                   (int)(1000 * senderRepInfo.SenderPop3Spam() / (senderRepInfo.SenderPop3Ham() + senderRepInfo.SenderPop3Spam() + 1)));

        if (senderRepInfo.ResSenderActiveDays())
            CheckRange("sndr_mean", (senderRepInfo.ResSenderHam() + senderRepInfo.ResSenderSpam()) / senderRepInfo.ResSenderActiveDays());

        CheckRange("sndr_complaint_hamcount_perc",
                   (int)(1000 * senderRepInfo.ResSenderComplaintHam() / (1 + senderRepInfo.ResSenderSpam())), false);
        CheckRange("sndr_complaint_spamcount_perc",
                   (int)(1000 * senderRepInfo.ResSenderComplaintSpam() / (1 + senderRepInfo.ResSenderHam())), false);
        if (senderRepInfo.ResSenderPersonalHam() > 100)
            CheckRange("sndr_personal_hamcount_perc",
                       (int)(100 * senderRepInfo.ResSenderPersonalHam() / (1 + senderRepInfo.ResSenderSpam() + senderRepInfo.ResSenderPersonalHam())), false);
        if (senderRepInfo.ResSenderPersonalSpam() > 100)
            CheckRange("sndr_personal_spamcount_perc",
                       (int)(100 * senderRepInfo.ResSenderPersonalSpam() / (1 + senderRepInfo.ResSenderHam() + senderRepInfo.ResSenderPersonalSpam())), false);
        if (senderRepInfo.ResSenderHam() + senderRepInfo.ResSenderHam() > 20)
            CheckRange("sndr_ham_perc",
                       (int)(100 * senderRepInfo.ResSenderHam() / (1 + senderRepInfo.ResSenderHam() + senderRepInfo.ResSenderSpam())), false);

        CheckRange("abk_total", senderRepInfo.ResSenderAbookCount(), false);
        CheckRange("abk_total_perc",
                   (int)(100 * senderRepInfo.ResSenderAbookCount() / (1 + senderRepInfo.ResSenderSpam() + senderRepInfo.ResSenderHam())), false);

        CheckRange("sndr_rcvd_from_yandex", senderRepInfo.ResSenderRcvdFromYandexCount(), false);

        CheckRange("sndr_ham_2w", senderRepInfo.ResSender2WeeksHamCount(), false);
        CheckRange("sndr_spam_2w", senderRepInfo.ResSender2WeeksSpamCount(), false);
        CheckRange("sndr_complaint_hamcount_2w", senderRepInfo.ResSender2WeeksComplHamCount(), false);
        CheckRange("sndr_complaint_spamcount_2w", senderRepInfo.ResSender2WeeksComplSpamCount(), false);
        CheckRange("sndr_complaint_hamcount_perc_2w",
                   (int)(1000 * senderRepInfo.ResSender2WeeksComplHamCount() / (1 + senderRepInfo.ResSender2WeeksSpamCount())), false);
        CheckRange("sndr_complaint_spamcount_perc_2w",
                   (int)(1000 * senderRepInfo.ResSender2WeeksComplSpamCount() / (1 + senderRepInfo.ResSender2WeeksHamCount())), false);
        CheckRange("sndr_personal_hamcount_perc_2w",
                   (int)(100 * senderRepInfo.ResSender2WeeksPersHamCount() / (1 + senderRepInfo.ResSender2WeeksSpamCount() + senderRepInfo.ResSender2WeeksPersHamCount())), false);
        CheckRange("sndr_personal_spamcount_perc_2w",
                   (int)(100 * senderRepInfo.ResSender2WeeksPersSpamCount() / (1 + senderRepInfo.ResSender2WeeksHamCount() + senderRepInfo.ResSender2WeeksPersSpamCount())), false);
        if (senderRepInfo.ResSender2WeeksHamCount() + senderRepInfo.ResSender2WeeksSpamCount() > 20)
            CheckRange("sndr_ham_perc2w",
                       (int)(100 * senderRepInfo.ResSender2WeeksHamCount() / (1 + senderRepInfo.ResSender2WeeksHamCount() + senderRepInfo.ResSender2WeeksSpamCount())), false);

        CheckRange("sndr_kuba_perc_2w",
                   (int)(100 * senderRepInfo.ResSender2WeeksKoobaCount() / (1 + senderRepInfo.ResSender2WeeksHamCount() + senderRepInfo.ResSender2WeeksSpamCount())), false);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S2W_KUBA"] = 100 * senderRepInfo.ResSender2WeeksKoobaCount() / (1 + senderRepInfo.ResSender2WeeksHamCount() + senderRepInfo.ResSender2WeeksSpamCount());

        CheckRange("sndr_del_total", senderRepInfo.SenderDeletedWithoutRead(), false);
        CheckRange("sndr_del_2w", senderRepInfo.SenderDeletedWithoutRead2Weeks(), false);
        CheckRange("sndr_del_perc",
                   (int)(1000 * senderRepInfo.SenderDeletedWithoutRead() / (1 + senderRepInfo.ResSenderHam())), false);
        CheckRange("sndr_del_perc_2w",
                   (int)(1000 * senderRepInfo.SenderDeletedWithoutRead2Weeks() / (1 + senderRepInfo.ResSender2WeeksHamCount())), false);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S2W_DH"] = senderRepInfo.SenderDeletedWithoutRead2Weeks() / (senderRepInfo.ResSender2WeeksHamCount() + 10.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S2W_DA"] = senderRepInfo.SenderDeletedWithoutRead2Weeks() / (senderRepInfo.ResSender2WeeksHamCount() + senderRepInfo.ResSender2WeeksSpamCount() + 10.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S_DH"] = senderRepInfo.SenderDeletedWithoutRead() / (senderRepInfo.ResSenderHam() + 10.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S_DA"] = senderRepInfo.SenderDeletedWithoutRead() / (senderRepInfo.ResSenderHam() + senderRepInfo.ResSenderSpam() + 10.);

        ui64 sndr_total = senderRepInfo.ResSenderSpam() + senderRepInfo.ResSenderHam();
        if (sndr_total != sndr_today) // SODEV-101
        {
            if ((senderRepInfo.ResSenderActiveDays() > 2) &&
                (sndr_total > 5000) &&
                (sndr_today / (sndr_total - sndr_today) > 2))
                m_cur->rulesContext.SetRule("SNDR_BURST");
            if (senderRepInfo.ResSenderHam() != sh14_ham)
                if ((sh14_ham / (senderRepInfo.ResSenderHam() - sh14_ham) > 2) &&
                    (senderRepInfo.ResSenderHam() >= 100) &&
                    (senderRepInfo.ResSenderActiveDays() > 2))
                    m_cur->rulesContext.SetRule("SNDR_BURST_H");
        }

        m_cur->m_mapValueFactors4Matrixnet["MNF_S2W_CS"] = senderRepInfo.ResSender2WeeksComplSpamCount() / (senderRepInfo.ResSender2WeeksHamCount() + 10.0);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S2W_CH"] = senderRepInfo.ResSender2WeeksComplHamCount() / (senderRepInfo.ResSender2WeeksSpamCount() + 10.0);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S2W_CSA"] = senderRepInfo.ResSender2WeeksComplSpamCount() / (senderRepInfo.ResSender2WeeksHamCount() + senderRepInfo.ResSender2WeeksSpamCount() + 10.0);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S2W_CHA"] = senderRepInfo.ResSender2WeeksComplHamCount() / (senderRepInfo.ResSender2WeeksHamCount() + senderRepInfo.ResSender2WeeksSpamCount() + 10.0);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S2W_CA"] = (senderRepInfo.ResSender2WeeksComplSpamCount() + senderRepInfo.ResSender2WeeksComplHamCount()) /
                                                           (senderRepInfo.ResSender2WeeksHamCount() + senderRepInfo.ResSender2WeeksSpamCount() + 10.0);

        m_cur->m_mapValueFactors4Matrixnet["MNF_S_CS"] = senderRepInfo.ResSenderComplaintSpam() / (senderRepInfo.ResSenderHam() + 50.0);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S_CH"] = senderRepInfo.ResSenderComplaintHam() / (senderRepInfo.ResSenderSpam() + 50.0);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S_CSA"] = senderRepInfo.ResSenderComplaintSpam() / (senderRepInfo.ResSenderHam() + senderRepInfo.ResSenderSpam() + 50.0);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S_CHA"] = senderRepInfo.ResSenderComplaintHam() / (senderRepInfo.ResSenderHam() + senderRepInfo.ResSenderSpam() + 50.0);
        m_cur->m_mapValueFactors4Matrixnet["MNF_S_CA"] = (senderRepInfo.ResSenderComplaintSpam() + senderRepInfo.ResSenderComplaintHam()) /
                                                         (senderRepInfo.ResSenderHam() + senderRepInfo.ResSenderSpam() + 50.0);

        const NSenderReputation::TDBSender& totals = senderRepInfo.sender.totals;
        const NSenderReputation::TDBPeriodicSender& periodic = senderRepInfo.sender.periodic;
        CheckRange("sndr_dkim_ham_perc", (int)(1000 * totals.dkim_ham / (totals.dkim_ham + totals.dkim_spam + 1)));
        CheckRange("sndr_dkim_spam_perc", (int)(1000 * totals.dkim_spam / (totals.dkim_ham + totals.dkim_spam + 1)));
        CheckRange("sndr_dkim_perc", (int)(1000 * (totals.dkim_ham + totals.dkim_spam) / (senderRepInfo.ResSenderHam() + senderRepInfo.ResSenderSpam() + 1)));
        CheckRange("sndr_dkim_ham_perc_2w", (int)(1000 * periodic.dkim_ham_count / (periodic.dkim_ham_count + periodic.dkim_spam_count + 1)));
        CheckRange("sndr_dkim_spam_perc_2w", (int)(1000 * periodic.dkim_spam_count / (periodic.dkim_ham_count + periodic.dkim_spam_count + 1)));
        CheckRange("sndr_dkim_perc_2w", (int)(1000 * (periodic.dkim_ham_count + periodic.dkim_spam_count) / (senderRepInfo.ResSender2WeeksHamCount() + senderRepInfo.ResSender2WeeksSpamCount() + 1)));

    }

    if(m_cur->iSenderReputationFlags.Test(NSenderReputation::TSenderType::Domain)) {
        PrintNewDomainRepRow(&senderRepInfo);

        luaSendersContext["domain"] = SendersToJson(senderRepInfo.domain);

        CheckRange("domn_age", (int)(senderRepInfo.ResDomenFirsttimeElapsed()), false);
        CheckRange("domn_active_days", senderRepInfo.ResDomenActiveDays(), false);
        CheckRange("domn_active_days_perc", 100 * senderRepInfo.ResDomenActiveDays() / (1 + senderRepInfo.ResDomenFirsttimeElapsed()), false);
        CheckRange("domn_ham", senderRepInfo.ResDomenHam(), false);
        CheckRange("domn_spam", senderRepInfo.ResDomenSpam(), false);
        CheckRange("domn_complaint_hamcount", senderRepInfo.ResDomenComplaintHam(), false);
        CheckRange("domn_complaint_spamcount", senderRepInfo.ResDomenComplaintSpam(), false);
        CheckRange("domn_personal_hamcount", senderRepInfo.ResDomenPersonalHam(), false);
        CheckRange("domn_personal_spamcount", senderRepInfo.ResDomenPersonalSpam(), false);

        const NSenderReputation::TDBSender& totals = senderRepInfo.domain.totals;
        const NSenderReputation::TDBPeriodicSender& periodic = senderRepInfo.domain.periodic;
        CheckRange("domn_dkim_ham_perc", (int)(1000 * totals.dkim_ham / (totals.dkim_ham + totals.dkim_spam + 1)));
        CheckRange("domn_dkim_spam_perc", (int)(1000 * totals.dkim_spam / (totals.dkim_ham + totals.dkim_spam + 1)));
        CheckRange("domn_dkim_perc", (int)(1000 * (totals.dkim_ham + totals.dkim_spam) / (senderRepInfo.ResSenderHam() + senderRepInfo.ResSenderSpam() + 1)));
        CheckRange("domn_dkim_ham_perc_2w", (int)(1000 * periodic.dkim_ham_count / (periodic.dkim_ham_count + periodic.dkim_spam_count + 1)));
        CheckRange("domn_dkim_spam_perc_2w", (int)(1000 * periodic.dkim_spam_count / (periodic.dkim_ham_count + periodic.dkim_spam_count + 1)));
        CheckRange("domn_dkim_perc_2w", (int)(1000 * (periodic.dkim_ham_count + periodic.dkim_spam_count) / (senderRepInfo.ResSender2WeeksHamCount() + senderRepInfo.ResSender2WeeksSpamCount() + 1)));

        CheckRange("domn_ps_ham_perc",
                   (int)(1000 * senderRepInfo.DomenPsHam() / (senderRepInfo.DomenPsHam() + senderRepInfo.DomenPsSpam() + 1)));
        CheckRange("domn_ps_spam_perc",
                   (int)(1000 * senderRepInfo.DomenPsSpam() / (senderRepInfo.DomenPsHam() + senderRepInfo.DomenPsSpam() + 1)));
        CheckRange("domn_ps_perc",
                   (int)(1000 * (senderRepInfo.DomenPsHam() + senderRepInfo.DomenPsSpam()) / (senderRepInfo.ResDomenHam() + senderRepInfo.ResDomenSpam() + 1)));
        CheckRange("domn_ps_ham_perc_2w",
                   (int)(1000 * senderRepInfo.DomenPsHam2Weeks() / (senderRepInfo.DomenPsHam2Weeks() + senderRepInfo.DomenPsSpam2Weeks() + 1)));
        CheckRange("domn_ps_spam_perc_2w",
                   (int)(1000 * senderRepInfo.DomenPsSpam2Weeks() / (senderRepInfo.DomenPsHam2Weeks() + senderRepInfo.DomenPsSpam2Weeks() + 1)));
        CheckRange("domn_ps_perc_2w",
                   (int)(1000 * (senderRepInfo.DomenPsHam2Weeks() + senderRepInfo.DomenPsSpam2Weeks()) / (senderRepInfo.ResDomen2WeeksHamCount() + senderRepInfo.ResDomen2WeeksSpamCount() + 1)));

        if (senderRepInfo.ResDomenActiveDays())
            CheckRange("domn_mean", (senderRepInfo.ResDomenHam() + senderRepInfo.ResDomenSpam()) / senderRepInfo.ResDomenActiveDays());

        CheckRange("domn_complaint_hamcount_perc",
                   (int)(1000 * senderRepInfo.ResDomenComplaintHam() / (1 + senderRepInfo.ResDomenSpam())), false);
        CheckRange("domn_complaint_spamcount_perc",
                   (int)(1000 * senderRepInfo.ResDomenComplaintSpam() / (1 + senderRepInfo.ResDomenHam())), false);
        CheckRange("domn_personal_hamcount_perc",
                   (int)(100 * senderRepInfo.ResDomenPersonalHam() / (1 + senderRepInfo.ResDomenPersonalHam() + senderRepInfo.ResDomenSpam())), false);
        CheckRange("domn_personal_spamcount_perc",
                   (int)(100 * senderRepInfo.ResDomenPersonalSpam() / (1 + senderRepInfo.ResDomenPersonalSpam() + senderRepInfo.ResDomenHam())), false);

        CheckRange("domn_rcvd_from_yandex", senderRepInfo.ResDomenRcvdFromYandexCount(), false);

        CheckRange("domn_ham_2w", senderRepInfo.ResDomen2WeeksHamCount(), false);
        CheckRange("domn_spam_2w", senderRepInfo.ResDomen2WeeksSpamCount(), false);
        CheckRange("domn_complaint_hamcount_2w", senderRepInfo.ResDomen2WeeksComplHamCount(), false);
        CheckRange("domn_complaint_spamcount_2w", senderRepInfo.ResDomen2WeeksComplSpamCount(), false);
        CheckRange("domn_complaint_hamcount_perc_2w",
                   (int)(1000 * senderRepInfo.ResDomen2WeeksComplHamCount() / (1 + senderRepInfo.ResDomen2WeeksSpamCount())), false);
        CheckRange("domn_complaint_spamcount_perc_2w",
                   (int)(1000 * senderRepInfo.ResDomen2WeeksComplSpamCount() / (1 + senderRepInfo.ResDomen2WeeksHamCount())), false);
        CheckRange("domn_personal_hamcount_perc_2w",
                   (int)(100 * senderRepInfo.ResDomen2WeeksPersHamCount() / (1 + senderRepInfo.ResDomen2WeeksSpamCount() + senderRepInfo.ResDomen2WeeksPersHamCount())), false);
        CheckRange("domn_personal_spamcount_perc_2w",
                   (int)(100 * senderRepInfo.ResDomen2WeeksPersSpamCount() / (1 + senderRepInfo.ResDomen2WeeksHamCount() + senderRepInfo.ResDomen2WeeksPersSpamCount())), false);

        CheckRange("domn_kuba_perc_2w",
                   (int)(100 * senderRepInfo.ResDomen2WeeksKoobaCount() / (1 + senderRepInfo.ResDomen2WeeksHamCount() + senderRepInfo.ResDomen2WeeksSpamCount())), false);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D2W_KUBA"] = 100 * senderRepInfo.ResDomen2WeeksKoobaCount() / (1 + senderRepInfo.ResDomen2WeeksHamCount() + senderRepInfo.ResDomen2WeeksSpamCount());

        CheckRange("domn_del_total", senderRepInfo.DomenDeletedWithoutRead(), false);
        CheckRange("domn_del_2w", senderRepInfo.DomenDeletedWithoutRead2Weeks(), false);
        CheckRange("domn_dkim_del", senderRepInfo.DomenDkimDeletedWithoutRead(), false);
        CheckRange("domn_del_perc",
                   (int)(1000 * (senderRepInfo.DomenDeletedWithoutRead() / (1 + senderRepInfo.ResDomenHam()))), false);
        CheckRange("domn_del_perc_2w",
                   (int)(1000 * (senderRepInfo.DomenDeletedWithoutRead2Weeks() / (1 + senderRepInfo.ResDomen2WeeksHamCount()))), false);
        CheckRange("domn_dkim_del_perc",
                   (int)(1000 * (senderRepInfo.DomenDkimDeletedWithoutRead() / (1 + senderRepInfo.DomenDKimHam()))), false);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D2W_DH"] = senderRepInfo.DomenDeletedWithoutRead2Weeks() / (senderRepInfo.ResDomen2WeeksHamCount() + 10.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D2W_DA"] = senderRepInfo.DomenDeletedWithoutRead2Weeks() / (senderRepInfo.ResDomen2WeeksHamCount() + senderRepInfo.ResDomen2WeeksSpamCount() + 10.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D_DH"] = senderRepInfo.DomenDeletedWithoutRead() / (senderRepInfo.ResDomenHam() + 10.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D_DA"] = senderRepInfo.DomenDeletedWithoutRead() / (senderRepInfo.ResDomenHam() + senderRepInfo.ResDomenSpam() + 10.);

        m_cur->m_mapValueFactors4Matrixnet["MNF_D2W_CS"] = senderRepInfo.ResDomen2WeeksComplSpamCount() / (senderRepInfo.ResDomen2WeeksHamCount() + 10.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D2W_CH"] = senderRepInfo.ResDomen2WeeksComplHamCount() / (senderRepInfo.ResDomen2WeeksSpamCount() + 10.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D2W_CSA"] = senderRepInfo.ResDomen2WeeksComplSpamCount() / (senderRepInfo.ResDomen2WeeksHamCount() + senderRepInfo.ResDomen2WeeksSpamCount() + 10.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D2W_CHA"] = senderRepInfo.ResDomen2WeeksComplHamCount() / (senderRepInfo.ResDomen2WeeksHamCount() + senderRepInfo.ResDomen2WeeksSpamCount() + 10.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D2W_CA"] = (senderRepInfo.ResDomen2WeeksComplSpamCount() + senderRepInfo.ResDomen2WeeksComplHamCount()) /
                                                           (senderRepInfo.ResDomen2WeeksHamCount() + senderRepInfo.ResDomen2WeeksSpamCount() + 10.);

        m_cur->m_mapValueFactors4Matrixnet["MNF_D_CH"] = senderRepInfo.ResDomenComplaintHam() / (senderRepInfo.ResDomenSpam() + 50.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D_CSA"] = senderRepInfo.ResDomenComplaintSpam() / (senderRepInfo.ResDomenSpam() + senderRepInfo.ResDomenHam() + 50.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D_CHA"] = senderRepInfo.ResDomenComplaintHam() / (senderRepInfo.ResDomenSpam() + senderRepInfo.ResDomenHam() + 50.);
        m_cur->m_mapValueFactors4Matrixnet["MNF_D_CA"] = (senderRepInfo.ResDomenComplaintHam() + senderRepInfo.ResDomenComplaintSpam()) /
                                                         (senderRepInfo.ResDomenSpam() + senderRepInfo.ResDomenHam() + 50.);
    }

    if(m_cur->iSenderReputationFlags.Test(NSenderReputation::TSenderType::Paysender)) {
        PrintNewPaysenderRepRow(&senderRepInfo);

        luaSendersContext["pay_sender"] = SendersToJson(senderRepInfo.psndr);

        CheckRange("psnd_ham", senderRepInfo.ResPSndrHam(), false);
        CheckRange("psnd_spam", senderRepInfo.ResPSndrSpam(), false);
        CheckRange("psnd_complaint_hamcount", senderRepInfo.ResPSndrComplaintHam(), false);
        CheckRange("psnd_complaint_spamcount", senderRepInfo.ResPSndrComplaintSpam(), false);

        if (senderRepInfo.ResPSndrHam() + senderRepInfo.ResPSndrSpam() > 20)
            CheckRange("psnd_ham_perc",
                       (int)(100 * senderRepInfo.ResPSndrHam() / (1 + senderRepInfo.ResPSndrHam() + senderRepInfo.ResPSndrSpam())), false);
        CheckRange("psnd_complaint_hamcount_perc",
                   (int)(10000 * senderRepInfo.ResPSndrComplaintHam() / (1 + senderRepInfo.ResPSndrSpam())), false);
        CheckRange("psnd_complaint_spamcount_perc",
                   (int)(10000 * senderRepInfo.ResPSndrComplaintSpam() / (1 + senderRepInfo.ResPSndrHam())), false);
        CheckRange("psnd_personal_hamcount_perc",
                   (int)(100 * senderRepInfo.ResPSndrPersonalHam() / (1 + senderRepInfo.ResPSndrPersonalHam() + senderRepInfo.ResPSndrSpam())), false);
        CheckRange("psnd_personal_spamcount_perc",
                   (int)(100 * senderRepInfo.ResPSndrPersonalSpam() / (1 + senderRepInfo.ResPSndrPersonalSpam() + senderRepInfo.ResPSndrHam())), false);

        CheckRange("psnd_ham_2w", senderRepInfo.ResPSndr2WeeksHamCount(), false);
        CheckRange("psnd_spam_2w", senderRepInfo.ResPSndr2WeeksSpamCount(), false);
        CheckRange("psnd_complaint_hamcount_2w", senderRepInfo.ResPSndr2WeeksComplHamCount(), false);
        CheckRange("psnd_complaint_spamcount_2w", senderRepInfo.ResPSndr2WeeksComplSpamCount(), false);

        if (senderRepInfo.ResPSndr2WeeksHamCount() + senderRepInfo.ResPSndr2WeeksSpamCount() > 20)
            CheckRange("psnd_ham_perc_2w",
                       (int)(100 * senderRepInfo.ResPSndr2WeeksHamCount() / (1 + senderRepInfo.ResPSndr2WeeksHamCount() + senderRepInfo.ResPSndr2WeeksSpamCount())), false);
        CheckRange("psnd_complaint_hamcount_perc_2w",
                   (int)(10000 * senderRepInfo.ResPSndr2WeeksComplHamCount() / (1 + senderRepInfo.ResPSndr2WeeksSpamCount())), false);
        CheckRange("psnd_complaint_spamcount_perc_2w",
                   (int)(10000 * senderRepInfo.ResPSndr2WeeksComplSpamCount() / (1 + senderRepInfo.ResPSndr2WeeksHamCount())), false);
        CheckRange("psnd_personal_hamcount_perc_2w",
                   (int)(100 * senderRepInfo.ResPSndr2WeeksPersHamCount() / (1 + senderRepInfo.ResPSndr2WeeksPersHamCount() + senderRepInfo.ResPSndr2WeeksSpamCount())), false);
        CheckRange("psnd_personal_spamcount_perc_2w",
                   (int)(100 * senderRepInfo.ResPSndr2WeeksPersSpamCount() / (1 + senderRepInfo.ResPSndr2WeeksPersSpamCount() + senderRepInfo.ResPSndr2WeeksHamCount())), false);

        CheckRange("psndr_kuba_perc_2w",
                   (int)(100 * senderRepInfo.ResPSndr2WeeksKoobaCount() / (1 + senderRepInfo.ResPSndr2WeeksHamCount() + senderRepInfo.ResPSndr2WeeksSpamCount())), false);
    }

    CheckValueAllRules(FD_SENDERS_VALUE, luaSendersContext);
}

NThreading::TFuture<TMaybe<NSenderReputation::TGetData>> TRengine::GetSenderReputationInfo(const TString& pfrom) {
    if (pfrom.size() < 6 || !pfrom.Contains('@')) // x@x.x or x.xx  not valid
        return NThreading::MakeFuture<TMaybe<NSenderReputation::TGetData>>(Nothing());

    m_cur->iSenderReputationFlags.Reset();
    if (m_cur->rulesContext.IsRuleWorked("PSNDR")) {
        for(const TRuleCurrent& ruleCurrent : m_cur->rulesContext.GetOccuredRules()) {
            if (strncasecmp("PSNDR_", ruleCurrent.GetDef().pRuleName.c_str(), 6) == 0) {
                // check for rulename like PSNDR_digits
                const char* strP = ruleCurrent.GetDef().pRuleName.c_str() + 6;
                while (*strP && isdigit(*strP))
                    strP++;
                if (*strP) // rulename has not-digits
                    continue;

                m_cur->m_sPaysenderRule = ruleCurrent.GetDef().pRuleName;
                break;
            }
        }

        ToLower(m_cur->m_sPaysenderRule.begin(), m_cur->m_sPaysenderRule.size());
        m_pstat.AddStat(ST_LOG) << pfrom << ' ' << m_cur->m_sPaysenderRule;
        m_cur->iSenderReputationFlags |= NSenderReputation::TSenderBitset(NSenderReputation::TSenderType::Paysender);
    }

    if (!m_free_mail_hosts.IsFreemailByEmail(pfrom))
        m_cur->iSenderReputationFlags |= NSenderReputation::TSenderBitset(NSenderReputation::TSenderType::Domain);
    //   will use all senders w/o sndr_counters check
    m_cur->iSenderReputationFlags |= NSenderReputation::TSenderBitset(NSenderReputation::TSenderType::Sender);

    // for new sender rep
    const NSenderReputation::TShingle senderShingle(pfrom, m_cur->m_sPaysenderRule, m_cur->iSenderReputationFlags);
    //m_pstat.AddStat("sender-shingle") << senderShingle;


    return NThreading::Async([
            senderGetRequest = NSenderReputation::TGeneralGetRequest(senderShingle),
            requester = Pools->SenderRepRequester,
            logger = m_cur->Logger]() mutable -> TMaybe<NSenderReputation::TGetData> {
        try {
            bool cltRes = requester->GetSenderReputation(senderGetRequest, logger);
            if (!cltRes) {
                return Nothing();
            }
            return MakeMaybe(std::move(senderGetRequest.getDataMutable()));
        } catch(...) {
            logger << (TLOG_ERR) << __FUNCTION__ << ' ' << CurrentExceptionMessageWithBt();
            return Nothing();
        }
    }, ThreadPool);
}

void TRengine::PutSenderReputationInfo(bool f_spam) {
    ui32 mask = 0;
    NSenderReputation::TPersonalAction pers_action = NSenderReputation::PA_NONE;
    const TString& pfrom = m_alg->GetFromAddr();

    if (Config.fWebMail) {
        NSenderReputation::TPutDataOutMail outmailData;

        TString s_debug;

        for(const auto & p : m_cur->m_rcptattrs) {
            const auto & prcpt_login = p.first;


            if (!m_free_mail_hosts.IsFreemailByEmail(prcpt_login)) {
                outmailData.emplace_back(prcpt_login, NSenderReputation::TSenderBitset(NSenderReputation::TSenderType::Sender));
                s_debug.append(prcpt_login);
                s_debug.append(" ");
            }
        }

        if (outmailData.empty())
            return;

        auto res = Pools->SenderRepRequester->PutOutMail(NSenderReputation::TGeneralPutOutMail(outmailData), m_cur->Logger);
        if (!res) {
            m_cur->Logger << TLOG_ERR << "sender_reputation error: " << res;
            m_cur->rulesContext.SetRule("SENDER_FAIL");
            return;
        }

        return;
    }

    NSenderReputation::TPutDataInMail putData;
    if (pfrom.size() < 6 || !pfrom.Contains('@')) // x@x.x or x.xx  not valid
        return;

    if (m_cur->rulesContext.IsRuleWorked("DL_FBR"))
    SET_SNDR_DL_FBR(mask);
    if (m_cur->rulesContext.IsRuleWorked("__FIXED_SOURCE"))
    SET_SNDR_FIXED_SOURCE(mask);
    if (m_cur->rulesContext.IsRuleWorked("SRC_LEVEL_1"))
    SET_SNDR_SOURCE_LEVEL1(mask);
    if (m_cur->rulesContext.IsRuleWorked("SNDR_WEBMASTER"))
    SET_SNDR_WEBMASTER(mask);
    putData.SetMask(mask);

    if (m_cur->rulesContext.IsRuleWorked("PERSONAL_CORRECT")) {
        if (m_cur->Scores.Hits < m_cur->m_required)
            pers_action = NSenderReputation::PA_PERSHAM;
        else
            pers_action = NSenderReputation::PA_PERSSPAM;
    }
    putData.SetPers(pers_action);

    if (m_cur->iSenderReputationFlags) // do not PUT if no PSNDR && SENDER && DOMAIN
    {
        putData.SetAbook(m_cur->rulesContext.IsRuleWorked("ABUK_PERC_50_MAX"));
        putData.SetKooba(m_cur->rulesContext.IsRuleWorked("KUBA_REC_0"));
        putData.SetSpam(f_spam);

        if (m_cur->iSenderReputationFlags.Test(NSenderReputation::TSenderType::Domain)) // for domains
        {
            if (m_cur->rulesContext.IsRuleWorked("TRUST_FROM"))
                putData.SetDkimSpam(f_spam);

            // mail-gmail resolution for domains
            if (m_cur->rulesContext.IsRuleWorked("SPAM_FOLDER_S"))
                putData.SetPop3Spam(true);
            if (m_cur->rulesContext.IsRuleWorked("HAM_FOLDER_S"))
                putData.SetPop3Spam(false);

            if (m_cur->iSenderReputationFlags.Test(NSenderReputation::TSenderType::Paysender)) // domain and paysender
                putData.SetPsSpam(f_spam);
        }

        NSenderReputation::TShingle shingle(pfrom, m_cur->m_sPaysenderRule, m_cur->iSenderReputationFlags);

        m_cur->SendersPutRequest = NSenderReputation::TGeneralPutInMail(shingle, putData);
        NThreading::Async([request = *m_cur->SendersPutRequest,
                                  requester = Pools->SenderRepRequester,
                                  putData = std::move(putData),
                                  shingle = std::move(shingle),
                                  logger = m_cur->Logger]() {
            try {
                if (!requester->PutInMail(std::move(request), logger)) {
                    logger << (TLOG_ERR) << __FUNCTION__ << "sender_reputation error " << CurrentExceptionMessageWithBt();
                }
            } catch(...) {
                logger << (TLOG_ERR) << __FUNCTION__ << "sender_reputation error " << CurrentExceptionMessageWithBt();
            }
        }, ThreadPool);
    }
}

TMaybe<NBlackbox2::TKarmaInfo> TRengine::DiscoverPddAdminKarma(const TString& admin_uid) {
    if (Pools->BBRequester && Config.fWebMail)
    {
        try {
            using namespace NFuncClient;
            auto response = Pools->BBRequester->GetUserInfo(CBB::MakeRequest(TUid{admin_uid}), m_cur->Logger);

            return MakeMaybe<NBlackbox2::TKarmaInfo>(response.Get());
        } catch (...) {
            m_cur->Logger << (TLOG_ERR) << "GetUserKarma error: " << CurrentExceptionMessageWithBt();
        }
    }

    return Nothing();
}

void TRengine::ProceedFreeMailInfo(const NFreeMail::TEmailInfo& key, const TMaybe<NFreeMail::TInfo>& info, const TMaybe<NFreeMail::TBounceInfo>& bounce) {

    if (!info.Defined()) {
        CheckRange("fm_abuse_spam", 0, false);
        CheckRange("fm_abuse_ham", 0, false);
        CheckRange("fm_count", 0, false);
        CheckRange("fm_age", 0, false);
        CheckRange("fm_uniqgeo", 0, false);
        CheckRange("fm_activedays", 0, false);
        m_pstat.AddStat(ST_FREE_MAIL) << "empty";
    }
    else {
        m_pstat.AddStat(ST_FREE_MAIL) << "ResActivDay " << info->active_days;

        TStringStream stream;
        stream << Hex(key.GetHash(), HF_FULL) << ' ' << info->create_time << ' ' << info->update_time << ' ' << info->GetCreateElapsedTime() << ' ' << info->GetUpdateElapsedTime() << "   "
               << info->complaint << "  " << info->send << "  " << info->receive << "  U  " << info->unique_complaint_user << "  "
               << info->unique_geo_zone << ' ' << info->active_days << ' ' << info->max_recipients_count << Flush;

        m_cur->MlLogBuilder.Add(ToString(ST_FREE_MAIL), stream.Str());

        if (info->send.ham > 0 || info->send.spam > 1) {
            CheckRange("fm_abuse_spam", static_cast<int>(info->complaint.spam), false);
            CheckRange("fm_abuse_ham", static_cast<int>(info->complaint.ham), false);
            CheckRange("fm_count", static_cast<int>(info->send.ham + info->send.spam), false);
            CheckRange("fm_age", static_cast<int>(info->GetCreateElapsedTime()), false);
            if (info->send.ham > 30 && info->complaint.spam < info->send.ham)
                CheckRange("fm_abuse_spam_rat", static_cast<int>(100 * info->complaint.spam / info->send.ham), false);
            CheckRange("fm_uniqcompluser", static_cast<int>(info->unique_complaint_user.spam), false);
            CheckRange("fm_uniqgeo", static_cast<int>(info->unique_geo_zone), false);
            CheckRange("fm_activedays", static_cast<int>(info->active_days), false);
        }

        m_pstat.AddStat(ST_FREE_MAIL) << stream.Str();
    }

    if (bounce.Defined()) {
        TStringStream stream;
        stream << bounce->unknown.today << ' ' << bounce->spam.today << ' '
               << bounce->unknown.total << ' ' << bounce->spam.total << ' '
               << bounce->last_type << ' ' << bounce->GetElapsedTime() << Flush;

        m_pstat.AddStat(ST_USER_BOUNCE) << stream.Str();

        CheckRange("bounce_t_unknown", static_cast<int>(bounce->unknown.today), false);
        CheckRange("bounce_t_spam", static_cast<int>(bounce->spam.today), false);
        CheckRange("bounce_all_unknown", static_cast<int>(bounce->unknown.total), false);
        CheckRange("bounce_all_spam", static_cast<int>(bounce->spam.total), false);
        CheckRange("bounce_t", static_cast<int>(bounce->unknown.today + bounce->spam.today), false);
        CheckRange("bounce_days", static_cast<int>(bounce->GetElapsedTime()), false);

        if (bounce->unknown.today + bounce->spam.today > 3)
            m_cur->c_bounce_perc = (bounce->unknown.today + bounce->spam.today) * 100;
    }

    TStringBuf domain;
    TMaybe<NFreeMail::TPDDInfo> pdd;
    if (!m_cur->rulesContext.IsRuleWorked("PDD") ||
        (domain = GetDomain(m_cur->m_sMailFrom)).empty() ||
        !Pools->FreeMailRequester->Get(TString{domain}, pdd, m_cur->Logger) ||
        !pdd.Defined())
        return;

    m_pstat.AddStat(ST_PDD_INFO) << pdd->GetElapsedTime() << ' ' <<  pdd->mailbox_count << ' ' << "  " << pdd->admin_uid << "  "
                                 << pdd->quota << ' ' << pdd->karma << ' ' << pdd->ip.toStroka2() << ' '
                                 << ' ' << pdd->bounce << ' ' << pdd->org_id << ' ' << MakeRangeJoiner(",", pdd->domains);

    if (!pdd->admin_uid.empty()) {
        CheckFieldAllRules(FD_IY_PDDADMIN, pdd->admin_uid);

        const TString domain_login = m_alg->GetFromDomain() + TString("_") + pdd->admin_uid;
        AddPattern(domain_login, EN_SH_FROM_PDD_ADM_UNIQ);
        AddPattern(pdd->admin_uid, EN_SH_FROM_PDD_ADMIN);

        const auto pddAdminKarmaInfo = DiscoverPddAdminKarma(pdd->admin_uid);
        m_pstat.AddStat("pddk") << "k:" << pddAdminKarmaInfo;

        if (pddAdminKarmaInfo) {
            CheckFieldAllRules(FD_IY_PDDADMKARMA, pddAdminKarmaInfo->Karma());

            if(i32 karma; TryFromString(pddAdminKarmaInfo->Karma(), karma))
                CheckRange("pdd_bb_admin_karma", karma, false);
            else
                m_cur->Logger << (TLOG_ERR) << "cannot read karma from " << pddAdminKarmaInfo;

            if(i32 status; TryFromString(pddAdminKarmaInfo->KarmaStatus(), status))
                CheckRange("pdd_bb_admin_karma_status", status, false);
            else
                m_cur->Logger << (TLOG_ERR) << "cannot read status from " << pddAdminKarmaInfo;
        }
    }

    const TString ipstring = pdd->ip.toStroka2();
    if (!ipstring.empty())
        CheckFieldAllRules(FD_IY_PDDIP, ipstring);

    CheckRange("pdd_age", static_cast<int>(pdd->GetElapsedTime()), false);
    CheckRange("pdd_mailbox", static_cast<int>(pdd->mailbox_count), false);
    CheckRange("pdd_quota", static_cast<int>(pdd->quota), false);
    CheckRange("pdd_karma", static_cast<int>(pdd->karma), false);
    CheckRange("pdd_bounce_today", static_cast<int>(pdd->bounce.today), false);
    CheckRange("pdd_bounce_all", static_cast<int>(pdd->bounce.total), false);
}

std::pair<TMaybe<NFreeMail::TInfo>, TMaybe<NFreeMail::TBounceInfo>> TRengine::GetFreeMailInfoGeneral(
        const NFreeMail::TEmailInfo& key,
        TTrueAtomicSharedPtr<NFuncClient::TFreeMail> requester,
        const TLog logger) try{
    TMaybe<NFreeMail::TInfo> info;
    TMaybe<NFreeMail::TBounceInfo> bounce;
    auto result = requester->Get(key, info, bounce, logger);
    if (!result) {
        logger << (TLOG_ERR) << "free_mail_reputation error: " << int(result);
        return {Nothing(), Nothing()};
    }

    return {std::move(info), std::move(bounce)};

} catch(...) {
    logger << (TLOG_ERR) << __FUNCTION__ << ' ' << CurrentExceptionMessageWithBt();
    return {Nothing(), Nothing()};
}

void TRengine::LogoutUser(TString uid, TString qid) {
    if (Pools->PassInternalRequester && uid) {
        NThreading::Async([uid = std::move(uid),
                           qid = std::move(qid),
                           logger = m_cur->Logger,
                           requester = Pools->PassInternalRequester]() mutable {
            requester->Logout.Perform(logger, uid, qid);
        },
                          ThreadPool);
    }
}

void TRengine::CleanUserKarma(TString uid, TString prefix) {
    if (Pools->PassInternalRequester && uid) {
        NThreading::Async([uid = std::move(uid),
                           prefix = std::move(prefix),
                           logger = m_cur->Logger,
                           requester = Pools->PassInternalRequester]() mutable {
            requester->SetKarma.Perform(logger, uid, prefix);
        }, ThreadPool);
    }
}

static NFreeMail::ESpamType Convert(TSpClass type) {
    switch (type) {
        case TSpClass::HAM:
        case TSpClass::DLVR:
            return NFreeMail::HAM;
        case TSpClass::SPAM:
            return NFreeMail::SPAM;
        case TSpClass::MALIC:
            return NFreeMail::MALIC;
        default:
            return NFreeMail::UNKNOWN;
    }
}

static TString GetShortGeo(const TString& geo) {
    return geo.substr(0, geo.find(' '));
}

void TRengine::PutFreeMailInfoGeneral() {
    if (Config.fWebMail && m_cur->rulesContext.IsRuleWorked("TO_FMS_HACK")) {
        LogoutUser(m_cur->m_sMailFromUID, queueID);
    }

    if (m_cur->m_isFreemail || Config.fWebMail) try{
            TVector<NFreeMail::TEmailInfo> recepients(Reserve(m_cur->m_rcptattrs.size()));
            for (const auto& [login, attr] : m_cur->m_rcptattrs) {
                recepients.emplace_back(NFreeMail::TEmailInfo::Create(attr.sUid, login, !attr.sUid.empty()));
            }

            NThreading::Async([requester = Pools->FreeMailRequester,
                                      mailFromUid = m_cur->m_sMailFromUID,
                                      mailFrom = m_cur->m_sMailFrom,
                                      isOut = Config.fWebMail,
                                      geozone = GetShortGeo(m_alg->GetGeoZone()),
                                      messClass = Convert(m_cur->m_messclass),
                                      recepients = std::move(recepients),
                                      logger = m_cur->Logger](){
                requester->Put(NFreeMail::TEmailInfo::Create(mailFromUid, mailFrom, isOut), geozone, messClass, recepients, logger);
            }, ThreadPool);

        } catch(...) {
            m_cur->Logger << (TLOG_ERR) << __FUNCTION__ << ' ' << CurrentExceptionMessageWithBt();
        }
}

void TRengine::AddShortUrl(const TStringBuf & url) {
    m_cur->AddShortUrl(url);
}

void TRengine::CheckFieldWord(TSpHeaderFields headerfield, const char* pword, int wordlen) {
    if (!(headerfield == spHFieldFrom || headerfield == spHFieldReplyTo || headerfield == spHFieldTo || headerfield == spHFieldCc))
        return;

    if (m_pRulesHolder->m_pListRuler.CheckListNoRuleSet(pword, wordlen, SP_LIST_USERNAMES)) {
        if (headerfield == spHFieldFrom || headerfield == spHFieldReplyTo)
            m_cur->rulesContext.SetRule("FLD_FROM_NAME");
        else if (headerfield == spHFieldTo || headerfield == spHFieldCc)
            m_cur->cFiledToNames++;
    }
}

bool TRengine::CancelLog() {
    return (m_cur->rulesContext.IsRuleWorked("CANCEL_DELIVERY_LOG") ||
            (m_cur->YandexTeam && m_cur->rulesContext.IsRuleWorked("ALLTRUSTEDIP") && m_cur->Scores.Hits < m_cur->m_required));
    return true;

    return false;
}

void TRengine::SendFreemailComplaint(const TShinglesGroupClass& fmData) try{
    if(!Pools->FreeMailRequester)
        return;
    if (Config.fWebMail) {
        const NFreeMail::TEmailInfo::UUID sender(fmData.sender_id.GetNewUid());
        const NFreeMail::TEmailInfo::External complainant(fmData.id.GetNewUid());
        Pools->FreeMailRequester->Complaint(sender, complainant, (fmData.spam == CS_HAM)? NFreeMail::HAM: NFreeMail::SPAM, m_cur->Logger);
    }
} catch(...) {
    m_cur->Logger << (TLOG_ERR) << __FUNCTION__ << ' ' << CurrentExceptionMessageWithBt();
}

NSenderReputation::TGeneralPutComplaint TRengine::MakeSenderComplaintRequest(const TShinglesGroupClass& fmData, const TRulesContext& rulesContext) {
    NSenderReputation::TShingle shingle(fmData.from, fmData.psnd, NSenderReputation::TSenderBitset(NSenderReputation::TSenderType::Sender, NSenderReputation::TSenderType::Domain));

    NSenderReputation::TComplaintData complaintData(fmData.spam != CS_HAM, rulesContext.IsRuleWorked("TRUST_FROM"));
    return NSenderReputation::TGeneralPutComplaint(shingle, complaintData, TInstant::Seconds(fmData.dlv_time));
}

const M_RCPT_ATTRS & TRengine::GetEnvRcptos() const {
    return m_cur->m_rcptattrs;
}

TString TRengine::GetMesageTypesCodes(const TShinglesGroupClass& fmData) {
    TString sResult;
    TString sRulename, sLabel;

    // complaint types as defined in http://wiki.yandex-team.ru/AntiSpam/Complaints
    if (m_cur->rulesContext.IsRuleWorked("HIDDEN_DLVR"))
        sResult.append("_HD");
    if (m_cur->rulesContext.IsRuleWorked("DL_FBR"))
        sResult.append("_DL");
    if (m_cur->rulesContext.IsRuleWorked("FREE_MAIL_COND"))
        sResult.append("_FM");
    if (m_cur->rulesContext.IsRuleWorked("DSN_NO_SENT_BY_YAMAIL") || m_cur->rulesContext.IsRuleWorked("CL_BOUNCE"))
        sResult.append("_BN");
    if (m_cur->m_sSoFront == "yaback")
        sResult.append("_YB");
    else if (m_cur->m_sSoFront == "outback")
        sResult.append("_OB");
    else if (m_cur->rulesContext.IsRuleWorked("BY_YANDEX_HTTP"))
        sResult.append("_YW");
    else if (m_cur->rulesContext.IsRuleWorked("BY_YANDEX_SMTP"))
        sResult.append("_YS");
    else if (m_cur->rulesContext.IsRuleWorked("ALLTRUSTEDIP") && fmData.compl_src != TComplSource::SRC_FBL)
        sResult.append("_ZY");
    if (m_cur->rulesContext.IsRuleWorked("TR_GEO_RCP"))
        sResult.append("_ZT");
    if (m_cur->rulesContext.IsRuleWorked("USR_UA"))
        sResult.append("_ZU");
    if (m_cur->rulesContext.IsRuleWorked("USR_BY"))
        sResult.append("_ZB");
    if (m_cur->rulesContext.IsRuleWorked("__CSO_FROM_FORWARD"))
        sResult.append("_DF");
    if (m_cur->rulesContext.IsRuleWorked("YANDEX_MAILER") || m_cur->rulesContext.IsRuleWorked("YANDEX_SMTP"))
        sResult.append("_DC");
    if (m_cur->rulesContext.IsRuleWorked("__POP3_AUTH"))
        sResult.append("_P3");
    if (m_cur->rulesContext.IsRuleWorked("SH_27_1111"))
        sResult.append("_1K");
    if (m_cur->rulesContext.IsRuleWorked("__IS_FORWARD"))
        sResult.append("_FW");
    if (m_cur->rulesContext.IsRuleWorked("PAY_SENDER"))
        sResult.append("_PS");
    if (m_cur->rulesContext.IsRuleWorked("YA_DEVNULL_RP"))
        sResult.append("_DN");
    if (m_cur->rulesContext.IsRuleWorked("SHIN_FAIL"))
        sResult.append("_SF");
    if (m_cur->rulesContext.IsRuleWorked("SHIN2_FAIL"))
        sResult.append("_S2");
    if (m_cur->rulesContext.IsRuleWorked("PDD") || IS_CMPL_PDD(fmData.ui_flags)) {
        sResult.append("_PD");
        if (m_cur->rulesContext.IsRuleWorked("PDD_MAILBOX_5K"))
            sResult.append("_PO");
    }
    if (m_cur->rulesContext.IsRuleWorked("BY_YANDEX_HTTP_SPAM"))
        sResult.append("_YC");
    if (m_cur->rulesContext.IsRuleWorked("TR_GEO_USER"))
        sResult.append("_OT");
    if (m_cur->rulesContext.IsRuleWorked("REC_IPV6"))
        sResult.append("_V6");
    if (m_cur->rulesContext.IsRuleWorked("__IMAP_AUTH"))
        sResult.append("_IM");
    if (m_cur->rulesContext.IsRuleWorked("__BORN_DATE_0_30"))
        sResult.append("_FR");
    if (m_cur->rulesContext.IsRuleWorked("NEWS_NO_UNSIB_LIST"))
        sResult.append("_NU");
    if (m_cur->rulesContext.IsRuleWorked("__SPF_PASS"))
        sResult.append("_SP");
    if (m_cur->rulesContext.IsRuleWorked("SPF_FAIL"))
        sResult.append("_SN");
    if (m_cur->rulesContext.IsRuleWorked("__YA_DKIM_PASS"))
        sResult.append("_DP");
    if (m_cur->rulesContext.IsRuleWorked("DOMN_ROLL"))
        sResult.append("_DR");
    if (m_cur->rulesContext.IsRuleWorked("YA_POP3"))
        sResult.append("_YP");
    if (m_cur->rulesContext.IsRuleWorked("YA_IMAP"))
        sResult.append("_YM");
    if (m_cur->rulesContext.IsRuleWorked("RDSL_FR"))
        sResult.append("_DS");
    //    if (m_cur->rulesContext.IsRuleWorked ("Z_MARKET"))
    //        sResult.append ("_MZ");
    //    if (m_cur->rulesContext.IsRuleWorked ("CL_SNDR_ESHOP"))
    //        sResult.append ("_ES");
    if (m_cur->rulesContext.IsRuleWorked("U_PF"))
        sResult.append("_UF");
    if (m_cur->rulesContext.IsRuleWorked("PERSONAL_CORRECT"))
        sResult.append("_PF");
    if (m_cur->rulesContext.IsRuleWorked("SP_LIST_YTEAM"))
        sResult.append("_LY");
    if (m_cur->rulesContext.IsRuleWorked("ABUK_PERC_1_50") || m_cur->rulesContext.IsRuleWorked("ABUK_PERC_50_90") || m_cur->rulesContext.IsRuleWorked("ABUK_PERC_90_MAX"))
        sResult.append("_AB");
    if (m_cur->rulesContext.IsRuleWorked("FAKE_RESOLV") || m_cur->rulesContext.IsRuleWorked("FAKE_RESOLV_V6") || m_cur->rulesContext.IsRuleWorked("FRNR"))
        sResult.append("_FA");
    if (m_cur->rulesContext.IsRuleWorked("ACT_US"))
        sResult.append("_AU");
    if (m_cur->rulesContext.IsRuleWorked("CLNDR_CMPL"))
        sResult.append("_CA");
    if (m_cur->rulesContext.IsRuleWorked("MAILISH"))
        sResult.append("_MA");
    if (m_cur->rulesContext.IsRuleWorked("RPTN_MAN"))
        sResult.append("_RP");

    if (Config.fWebMail) // so_out
    {
        if (m_cur->rulesContext.IsRuleWorked("MALIC_ML"))
            sResult.append("_ML");
        if (m_cur->rulesContext.IsRuleWorked("PDD_LOCAL"))
            sResult.append("_PL");
        if (m_cur->rulesContext.IsRuleWorked("YA_CAPTCHA"))
            sResult.append("_CY");
        if (m_cur->rulesContext.IsRuleWorked("YA_CAPTCHA_BAD"))
            sResult.append("_CN");
    }

    if (m_cur->rulesContext.IsRuleWorked("SEO_ABUSE") || IS_CMPL_SEO_ABUSE(fmData.ui_flags))
        sResult.append("_SA");

    if (IS_CMPL_MOVED(fmData.ui_flags))
        sResult.append("_MV");
    if (IS_CMPL_SEEN(fmData.ui_flags))
        sResult.append("_SN");
    if (IS_CMPL_PERSONAL(fmData.ui_flags))
        sResult.append("_PF");
    if (IS_CMPL_SKIPPED(fmData.ui_flags))
        sResult.append("_SK");
    if (IS_CMPL_YTEAM(fmData.ui_flags))
        sResult.append("_YT");
    if (IS_CMPL_WRONGSPAM(fmData.ui_flags))
        sResult.append("_WS");
    if (IS_CMPL_WRONGHAM(fmData.ui_flags))
        sResult.append("_WH");
    if (IS_CMPL_YAFWD(fmData.ui_flags))
        sResult.append("_YF");
    if (IS_CMPL_WAS(fmData.ui_flags))
        sResult.append("_XX");

    if (fmData.spam == CS_UNSUBSCRIBE)
        sResult.append("_UN");
    if (fmData.compl_src == SRC_IMAP)
        sResult.append("_IC");
    if (fmData.compl_src == SRC_MOBILE)
        sResult.append("_MC");
    if (fmData.compl_src == SRC_FURITA)
        sResult.append("_FC");
    if (fmData.compl_src == SRC_FBL)
        sResult.append("_FB");
    if (fmData.compl_src == SRC_FASTSRVLOG)
        sResult.append("_FS");

    for (int j = 0; j < 10; j++) {
        sprintf(sRulename, "LOGTYPE_A%d", j);
        if (m_cur->rulesContext.IsRuleWorked(sRulename.c_str())) {
            sprintf(sLabel, "_A%d", j);
            sResult.append(sLabel);
        }
        sprintf(sRulename, "FLAG_%d", j);
        if (m_cur->rulesContext.IsRuleWorked(sRulename.c_str())) {
            sprintf(sLabel, "_F%d", j);
            sResult.append(sLabel);
        }
    }

    if (sResult.empty())
        sResult.assign("-");
    sResult.append("\t");

    return sResult;
}

struct TCharsAsSpace{
    explicit TCharsAsSpace(const TStringBuf src, const TStringBuf chars) noexcept : Src(src), Chars(chars.cbegin(), chars.cend()) {}
    friend IOutputStream& operator<<(IOutputStream& stream, const TCharsAsSpace& as) {
        for(const char c : as.Src) {
            stream << (as.Chars.contains(c) ? ' ' : c);
        }
        return stream;
    }

    const TStringBuf Src;
    const THashSet<char> Chars;
};

void TRengine::Print2Shortlog(const TString& sMNLabel) {
    TString sStr;
    time_t now, creation;
    struct tm* dateTime;

    auto rec = ShortLog << "";

    if (Config.fWebMail) // so_out
    {
        rec << (m_cur->m_sMailFromSuid.empty() ? "-" : m_cur->m_sMailFromSuid) << '\t';
    }

    if(m_cur->m_sMailFrom)
        rec << TCharsAsSpace(TStringBuf(m_cur->m_sMailFrom).Head(MAX_LEN), "\t\r\n") << '\t';
    else
        rec << "-\t";


    if(m_alg->GetFromAddr())
        rec << TCharsAsSpace(TStringBuf(m_alg->GetFromAddr()).Head(MAX_LEN), "\t\r\n") << '\t';
    else
        rec << "-\t";

    switch (m_cur->m_messclass) {
        case TSpClass::HAM:
            rec << "H\t";
            break;
        case TSpClass::DLVR:
            rec << "D\t";
            break;
        case TSpClass::SPAM:
            rec << "S\t";
            break;
        case TSpClass::MALIC:
            rec << "M\t";
            break;
        default:
            rec << "U\t";
    }

    if (Config.fWebMail && m_alg) // so_out
    {
        // carma && creation date
        rec << m_alg->GetKarmaStatus() << '\t';

        creation = m_alg->GetBornDate();
        dateTime = localtime(&creation);
        sprintf(sStr, "%02d.%02d.%d", dateTime->tm_mday, ++dateTime->tm_mon, dateTime->tm_year + 1900);
        rec << sStr << '\t';
    }

    // check date
    now = time(nullptr);
    struct tm dt;
    dateTime = localtime_r(&now, &dt);
    sprintf(sStr, "%02d.%02d.%d %02d:%02d:%02d", dateTime->tm_mday, ++dateTime->tm_mon, dateTime->tm_year + 1900, dateTime->tm_hour, dateTime->tm_min, dateTime->tm_sec);
    rec << sStr << '\t';

    // mID


    if(m_alg->GetRealMessageId())
        rec << TCharsAsSpace(TStringBuf(m_alg->GetRealMessageId()).Head(MAX_LEN * 2), "\t\r\n") << '\t';
    else
        rec << "-\t";

    // types
    rec << sMNLabel << GetMesageTypesCodes({});

    // spammer IP
    sStr = m_alg->GetIpSender();
    rec << (sStr ? TStringBuf(sStr).Head(MAX_LEN) : "-") << '\t';

    // spammer host
    sStr = m_alg->GetHostSender();
    rec << (sStr ? TStringBuf(sStr).Head(MAX_LEN) : "-") << '\t';

    // geozone
    sStr = m_alg->GetGeoZone();
    rec << (sStr ? TStringBuf(sStr).Head(MAX_LEN) : "-") << '\t';

    // first IP
    sStr = m_alg->GetIpFirst();
    rec << (sStr ? TStringBuf(sStr).Head(MAX_LEN) : "-") << '\t';

    // first host - TODO
    rec << "-\t";

    // first geozone
    sStr = m_alg->GetFirstGeoZone();
    rec << (sStr.length() ? TStringBuf(sStr).Head(2) : "-") << "\t";

    sStr.erase();
    if (Config.fWebMail) // for so-out print sender' suid
    {
        if (m_cur->m_sMailFromUID.length())
            sStr.append(m_cur->m_sMailFromUID);
        else
            sStr.append("-");
    } else // for so-in print suids for all rcptos
    {
        for(const auto & p : m_cur->m_rcptattrs) {
            const auto & prcpt = p.first;
            const auto & sSuid = p.second.sSuid;
            if (!sStr.empty())
                sStr.append(",");

            if (!prcpt || prcpt.front() == 0)
                sStr.append("-");
            else
                sStr.append(prcpt);

            if (!Config.fWebMail) // so_in
            {
                sStr.append("_");

                if (sSuid && sSuid.length() > 0)
                    sStr.append(sSuid);
                else
                    sStr.append("0");
            }
        }
    }
    rec << sStr << '\t';
    rec << HostName() << '\t';

    sStr.erase();
    if (Config.fWebMail) // for so-out print login(s) list
    {
        for(const auto & p : m_cur->m_rcptattrs) {
            const auto & prcpt = p.first;
            if (!sStr.empty())
                sStr.append(",");
            sStr.append(prcpt);
        }
    } else // for so-in print uids
    {
        for(const auto & p : m_cur->m_rcptattrs) {
            if (!sStr.empty())
                sStr.append(",");
            sStr.append(p.second.sUid);
        }
    }
    rec << (sStr.empty() ? "-" : sStr) << '\t';

    // Queue-ID
    sStr = m_alg->GetQueueID();
    rec << (sStr ? sStr : "-") << '\t';

    // recipients countries list, OR sender country code for so-out
    sStr.clear();
    if (Config.fWebMail)
        sStr.assign(m_cur->m_sMailFromGeo);
    else
        for (M_RCPT_ATTRS::iterator mIt = m_cur->m_rcptattrs.begin(); mIt != m_cur->m_rcptattrs.end(); mIt++) {
            if (!sStr.empty())
                sStr.append(",");
            sStr.append(mIt->second.sCountry.length() ? mIt->second.sCountry : "-");
        }
    rec << (sStr ? sStr : "-") << '\t';

    // spammer nik-name
    if(m_alg->GetFromName())
        rec << TCharsAsSpace(TStringBuf(m_alg->GetFromName()).Head(MAX_LEN), "'\t\n\r") << '\t';
    else
        rec << "-\t";

    // spam subject
    if(m_alg->GetSubject())
        rec << TCharsAsSpace(TStringBuf(m_alg->GetSubject()).Head(MAX_LEN), "'\t\n\r") << '\t';
    else
        rec << "-\t";

    if (Config.fWebMail) // so_out
    {
        rec << '\t' << m_alg->GetXMailer() << '\t' << m_alg->GetUserAgent();
    }

    rec << '\t' << GetSOClassShort() << '\n';
}

void TRengine::SetSenderUID(TUid sUID) {
    m_cur->m_sMailFromUID = std::move(sUID);
    m_cur->DlvLogRequest.Uid = m_cur->m_sMailFromUID;
}

void TRengine::SetSenderGeo(TStringBuf sGeo) {
    m_cur->m_sMailFromGeo.assign(sGeo);
}

void TRengine::SetShingle36Src(const char* sSrc) {
    m_cur->m_shingle36source.assign(sSrc);
}

void TRengine::SetFrmFlag(bool bFlag) {
    m_cur->bFrmFlag = bFlag;
}

void TRengine::Add2UrlReputationList(TStringBuf domainOrHost, EUrlStaticticSource code) {
    if (domainOrHost.empty() ||
        !domainOrHost.Contains('.') ||
        domainOrHost[0] == '[' ||
        domainOrHost == "undef")
        return;

    if (domainOrHost.back() == '>')
        domainOrHost.Chop(1);

    const TStringBuf sCommonZone = m_common_zone_detector.GetCommonZone(domainOrHost);

    const bool bDoCount = code == EUrlStaticticSource::BODY; // increment counters for body urls only

    TVector<TString> aliases = { "dbl", "malware", "malwareaggress" };
    if (code == EUrlStaticticSource::BODY) {
        aliases.emplace_back("surbl");
    }

    if (m_pRulesHolder->m_pListRuler.CheckWord(m_cur->rulesContext, sCommonZone, SP_LIST_SKIP_URI))
        return;

    if (auto [added, stat] = m_cur->AddUrl(sCommonZone, aliases, bDoCount, code); !added){
        m_cur->Logger << (TLOG_DEBUG) << "URL code added " << static_cast<ui32>(code)
                           << " for " << sCommonZone
                           << ", flag " << stat->flags
                           << "->" << static_cast<ui32>(stat->flags | code);

        stat->flags |= code;
        stat->noAddStorage &= bDoCount;
    }
}

ui32 TRengine::LangCode2LangBit(ELanguage langCode) {
    ui32 langBit = 0;

    if (langCode == LANG_RUS)
        langBit = RUL_RUS;
        //    else if (langCode == LANG_ENG)        // treat English as "use always"
        //        langBit = RUL_ENG;
    else if (langCode == LANG_UKR)
        langBit = RUL_UKR;
    else if (langCode == LANG_BEL)
        langBit = RUL_BLR;
    else if (langCode == LANG_TUR)
        langBit = RUL_TUR;

    return langBit;
}

//*******************************

TShingleEngine::TShingleEngine(const TRulesHolder &pRulesHolder)
: m_pRulesHolder(pRulesHolder){
}

void TShingleEngine::Add(const TStringBuf & pattern, const TStringBuf& shingle, TShHttpType type, bool fBanPattern) {
    if (!shingle)
        return;

    char str_pattern[str_short_size], str_shingle[str_short_size];

    sh_reputation_list.emplace_back(shingle, type);

    if (!pattern)
        return;

    snprintf(str_pattern, sizeof(str_pattern), "%.*s__%d", static_cast<int>(pattern.size()), pattern.data(), static_cast<int>(type));
    SET_STR_NULL(str_pattern);
    snprintf(str_shingle, sizeof(str_shingle), "%.*s__%d", static_cast<int>(shingle.size()), shingle.data(), type);
    SET_STR_NULL(str_shingle);

    if (count >= m_Max_Engine_Shingles)
        return;

    auto it = m_mappatsh.find(str_pattern);
    if(it != m_mappatsh.end()) {
        auto & pref_pattern = it->second;
        pref_pattern.shingle.assign(str_shingle);
        pref_pattern.fSpBan = fBanPattern;
    } else {
        m_mappatsh.emplace(str_pattern, TCurPattern{type, TString(str_shingle), fBanPattern});
    }

    if (m_mapsh.contains(str_shingle))
        return;

    ++count;

    TCurShingle cursh;
    cursh.type = type;

    cursh.value.assign(str_pattern);
    // ���������� �������� � ���������
    if (type == EN_SH_HOST && m_pRulesHolder.m_pListRuler.CheckShingleGeo(pattern.data(), pattern.size()))
        cursh.fGeo = true;

    // ���������� ������, �� ������� �� ������ ��������
    if (m_pRulesHolder.m_pListRuler.CheckShingleWl(pattern.data(), pattern.size()))
        cursh.fWl = true;
    m_mapsh.emplace(str_shingle, std::move(cursh));
}

// ������� �� ������� ������
void TShingleEngine::AddBanPattern(const char* pattern, TShHttpType type) {
    if (!pattern)
        return;

    char str_pattern[str_short_size];
    snprintf(str_pattern, sizeof(str_pattern), "%s__%d", pattern, type);
    SET_STR_NULL(str_pattern);

    if (count >= m_Max_Engine_Shingles)
        return;

    auto patShIt = m_mappatsh.find(str_pattern);
    if(m_mappatsh.end() != patShIt) {
        TCurPattern & pref_pattern = patShIt->second;
        pref_pattern.fSpBan = true;

        auto shIt = m_mapsh.find(pref_pattern.shingle);
        if (shIt != m_mapsh.end())
            shIt->second.fSpBan = true;
        return;
    }
    m_mappatsh.emplace(str_pattern, TCurPattern{type, nullptr, true});
}

TCurShingle* TShingleEngine::FindCurShingle(TShHttpType type, const char* shingle) {
    if (!shingle)
        return nullptr;

    char str_shingle[str_short_size];;

    snprintf(str_shingle, sizeof(str_shingle), "%s__%d", shingle, type);
    SET_STR_NULL(str_shingle);


    auto it = m_mapsh.find(str_shingle);

    return (it != m_mapsh.end()) ? &it->second : nullptr;
}

// return true if need skip shingle
bool TShingleEngine::SetCountToday(TRengine* m_prengine, TShHttpType type, const char* shingle, ui16 ham, ui16 spam) {
    if (TCurShingle* pref_shingle = FindCurShingle(type, shingle)) {
        pref_shingle->ftoday_set = true;
        pref_shingle->cspam = spam;
        pref_shingle->cham = ham;

        if (pref_shingle->fSpBan)
            m_prengine->CheckRange("spban_count", ham + spam, false);

        if (pref_shingle->fWl || pref_shingle->fGeo)
            return true; // skip
    }

    return false;
}

void TShingleEngine::SetSurbl(TShHttpType type, const char* shingle) {
    TCurShingle* pref_shingle;

    if (type == EN_SH_HOST && (pref_shingle = FindCurShingle(type, shingle)))
        pref_shingle->fSurbl = true;
}

void TShingleEngine::SetReputation(bool fWebMail) {
    char str[str_short_size];
    TCurShingle* pref_shingle;

    /*if (fWebMail)
        m_pstat.AddStat(ST_COMPL_SHINGL) << "debug webmail";
    else
        m_pstat.AddStat(ST_COMPL_SHINGL) << "debug";*/

    for (auto& sit : sh_reputation_list) {
        if (sit.ResEmpty())
            continue;

        str[0] = 0;

        if (sit.ResHam() > 2 || sit.ResSpam() > 2 || Abs(sit.ResWeight(fWebMail)) > 0.0099) {
            snprintf(str, sizeof(str), "%d %s %d %d (%d %d %d %d %d %.1f)", sit.Type(), sit.sShingle(),
                     sit.ResSpam(), sit.ResHam(),
                     sit.ResTodaySpam(), sit.ResTodayHam(),
                     sit.ResYesterdaySpam(), sit.ResYesterdayHam(),
                     sit.ResInWhiteList() ? 1 : 0, sit.ResWeight(fWebMail));
            SET_STR_NULL(str);
        }

        if ((pref_shingle = FindCurShingle((TShHttpType)(sit.Type()), sit.sShingle()))) {
            pref_shingle->compl_empty = false;
            pref_shingle->compl_inwhitelist = sit.ResInWhiteList();
            pref_shingle->compl_ham = sit.ResHam();
            pref_shingle->compl_spam = sit.ResSpam();
            pref_shingle->compl_today_ham = sit.ResTodayHam();
            pref_shingle->compl_today_spam = sit.ResTodaySpam();
            pref_shingle->compl_yesterday_ham = sit.ResYesterdayHam();
            pref_shingle->compl_yesterday_spam = sit.ResYesterdaySpam();
            pref_shingle->compl_weight = sit.ResWeight(fWebMail);
            if (sit.ResHam() > 2 || sit.ResSpam() > 2) {
                int len_tmp = strlen(str);
                snprintf(str + len_tmp, sizeof(str) - len_tmp, "%d %d %d %d", pref_shingle->cspam, pref_shingle->cham,
                         sit.ResSpam() * 100 / (pref_shingle->cham + 1), sit.ResHam() * 100 / (pref_shingle->cspam + 1));
                SET_STR_NULL(str);
            }
        }

        //if (strlen(str) > 0)
        //    m_pstat.AddStat(ST_COMPL_SHINGL) << str;
    }
}

TAppliersMap LoadModels(const TSoConfig& config) {
    if (config.mnPluginLibDir && config.mnConfigPath && config.dumbMode < 2) {
        const TFsPath workingDir(config.mnPluginLibDir);
        TMappedFileInput fileInput(config.mnConfigPath);
        return LoadAppliers(workingDir, NConfig::TConfig::FromJson(fileInput));
    }
    return {};
}

bool GoodForBayes(const TTextSplitter::TSplitResult& res) {
    return !(res.has_cyrillic && res.has_latin) && !res.has_digit && (res.word.length() > 3) && (res.word.length() <= 16);
}

void TRengine::CheckIP(const TStringBuf & word)
{
    m_pRulesHolder->m_pListRuler.CheckIp(m_cur->rulesContext, word);
}

void TRengine::CheckRdns(const TStringBuf & word)
{
    m_pRulesHolder->m_pListRuler.CheckRdns(m_cur->rulesContext, word);
}

void TRengine::CheckGEO(const TStringBuf & word)
{
    m_pRulesHolder->m_pListRuler.CheckGEO(m_cur->rulesContext, word);
}

bool TRengine::GetPattern(const char *PatternName, const TString& Field, TString& Match) {

    if (TStringBuf pattern = TRulesHolder::m_pcre->GetPattern(PatternName, Field, 1)) {
        Match.assign(pattern.data(), 0, pattern.size());
        return true;
    } else
        return false;
}

struct THostInfoK {
    TString host;
    const char *Pattern{};
    size_t PatternLen{};
    bool ok{};
};

void TRengine::AddStat(TStringBuf fdname, TStringBuf pfield) {
    m_pstat.AddStat(fdname) << pfield;
}

bool TRengine::IsHTMLText(TString& text) {
    return m_html->IsHTMLText(text);
}

void TRengine::GetWorkedCriterionBanRule(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv4BanRuleWeight, "_CR", "_H");
}

void TRengine::GetWorkedCriterionRPTRule(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv4RPTRuleWeight, nullptr, "_R");
}

void TRengine::GetWorkedCriterionFRWDRule(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv4FRWDRuleWeight, nullptr, "_T");
}

void TRengine::GetWorkedCriterionPROBBANRule(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv4PROBBANRuleWeight, nullptr, "_CR");
}

void TRengine::GetWorkedCriterionBAN2Rule(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv4BAN2RuleWeight, nullptr, nullptr);
}

void TRengine::GetWorkedCriterionBAN7Rule(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv4BAN7RuleWeight, nullptr, nullptr);
}

void TRengine::GetWorkedCriterionBAN8Rule(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv4BAN8RuleWeight, nullptr, nullptr);
}

void TRengine::GetWorkedCriterionBanRuleIPv6(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv6BanRuleWeight, "_CR", "_H");
}

void TRengine::GetWorkedCriterionRPTRuleIPv6(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv6RPTRuleWeight, nullptr, "_R");
}

void TRengine::GetWorkedCriterionFRWDRuleIPv6(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv6FRWDRuleWeight, nullptr, "_T");
}

void TRengine::GetWorkedCriterionPROBBANRuleIPv6(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv6PROBBANRuleWeight, nullptr, "_CR");
}

void TRengine::GetWorkedCriterionBAN2RuleIPv6(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv6BAN2RuleWeight, nullptr, nullptr);
}

void TRengine::GetWorkedCriterionBAN7RuleIPv6(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv6BAN7RuleWeight, nullptr, nullptr);
}

void TRengine::GetWorkedCriterionBAN8RuleIPv6(TCritRuleList& critlist) const {
    critlist = GetWorkedCriterionRule(IPv6BAN8RuleWeight, nullptr, nullptr);
}

static ui16 ParseIfTagFound(const TStringBuf& str, const TStringBuf& tag) {
    if (tag.IsInited()) {
        const size_t p = str.find(tag);
        if (p != TStringBuf::npos && str.size() - p >= 3 + tag.size()) {
            return FromString<ui16>(str.substr(p + tag.size(), 3));
        }
    }
    return 0;
}

TCritRuleList TRengine::GetWorkedCriterionRule(double weight, const TStringBuf& criterionTag, const TStringBuf& hourTag) const {
    TCritRuleList critlist;
    for(const TRuleCurrent& ruleCurrent : m_cur->rulesContext.GetOccuredRules()) {
        const TRuleDef& prule = ruleCurrent.GetDef();

        if (ruleCurrent.GetScore() == weight) {
            const TStringBuf rule_name = prule.pRuleName;
            critlist.emplace_back(TString{rule_name},
                                  ParseIfTagFound(rule_name, criterionTag),
                                  ParseIfTagFound(rule_name, hourTag));
        }
    }
    if (hourTag.IsInited())
        critlist.sort([](const TCritRule& cr1, const TCritRule& cr2) { return cr1.Hour < cr2.Hour; });
    return critlist;
}

bool TRengine::ExistsbanInternal(const TStringBuf& rulename) const {
    if (!rulename.IsInited())
        return false;

    for(const TRuleCurrent& ruleCurrent : m_cur->rulesContext.GetOccuredRules()) {
        if (ruleCurrent.GetDef().pRuleName == rulename) {
            return true;
        }
    }

    return false;
}

bool TRengine::ExistsRule(char *rulename) const {
    return ExistsbanInternal(rulename);
}

void TRengine::CheckSuidRoll(const TStringBuf& word) const {
    m_pRulesHolder->m_pListRuler.CheckSuidRoll(m_cur->rulesContext, word);

}

void TRengine::CheckLongLines(const TStringBuf body) {
    size_t start = 0;

    while(start < body.size()){
        const auto eol = body.find('\n', start + 1);

        const size_t length = (eol == NPOS ? body.size() : eol) - start;

        if(length > 997) {
            m_cur->rulesContext.SetRule("CAN_BE_TRUNCATED");
            break;
        }
        start = eol;
    };
}

void TRengine::ProcessClearText(CProf& prof, const TString& text) {
    auto fullProf = Guard(prof.Prof("all"));

    {
        auto fullProf = Guard(prof.Prof("ck_997"));
        const TStringBuf body = RE_RNRN->MatchAndGetResult(text, 1).GetRestPattern();
        CheckLongLines(body);
    }
    if (!m_cur->So2Context) {
        m_cur->rulesContext.SetRule("ABOOK_FAIL");
        m_cur->rulesContext.SetRule("HTML_SAN_FAIL");
        return;
    }

    if (!m_cur->So2Context->GetSenders()) {
        m_cur->rulesContext.SetRule("SO2_SENDERS_FAIL");
    }

    if (!m_cur->So2Context->Docs())
        return;

    TVector<THolder<IRengineHtmlSanAnswerProcessor>> localProcessors;

    if (Pools->KnnRequester) {
        localProcessors.emplace_back(MakeHolder<TKnnProcessor>(*this));
    }

    CProf& procProf = prof.Sub("proc");
    for(const auto& processors: {std::cref(Processors), std::cref(localProcessors)}) {
        for (const auto& processorHolder : processors.get()) {
            try {
                processorHolder->Process(*m_cur->So2Context, procProf);
            } catch (...) {
                m_cur->Logger << (TLOG_ERR) << "error in process htmlsan answer: " << CurrentExceptionMessageWithBt();
            }
        }
    }
}

void TRengine::PushRuleSignal(const TRuleDef &ruleDef,
                              TSpClass spClass,
                              int value) {

    TRuleUnistat &rulesUnistat = rulesUnistats.try_emplace(std::addressof(ruleDef),
                                                           TStringBuf(ruleDef.pRuleName)).first->second;

    {
        const TKWYasmTagsBitset &yasmTags = ruleDef.YasmTags;
        if (yasmTags.Test(TKWYasmTags::HAM) && IsHam(spClass) ||
            yasmTags.Test(TKWYasmTags::SPAM) && spClass == TSpClass::SPAM ||
            yasmTags.Test(TKWYasmTags::MALIC) && spClass == TSpClass::MALIC) {
            rulesUnistat.Hole(spClass)->PushSignal(value);
        }
    }

    {
        const TKWYasmTagsBitset &solomonTags = ruleDef.SolomonTags;
        if (solomonTags.Test(TKWYasmTags::HAM) && IsHam(spClass) ||
            solomonTags.Test(TKWYasmTags::SPAM) && spClass == TSpClass::SPAM ||
            solomonTags.Test(TKWYasmTags::MALIC) && spClass == TSpClass::MALIC) {
            rulesUnistat.Metric(spClass)->Add(value);
        }
    }
}

TString TRengine::CheckShingle(ui32 shingleType, const TStringBuf& text) {
    return AddPattern(text, TShHttpType(shingleType));
}

void TRengine::PushRulesSignals() {
    const TSpClass spClass = m_cur->m_messclass;
    if(spClass == TSpClass::UNKNOWN) {
        m_cur->Logger << (TLOG_WARNING) << "unknown resolution";
        return;
    }

    const IModelApplier::TFeaturesMap& features = m_cur->m_mapValueFactors4Matrixnet;

    for(const TRuleDef& rule: m_pRulesHolder->GetMnfRules()) {
        const TString& name = rule.pRuleName;

        PushRuleSignal(rule, spClass, features.Value(name, 0));
    }

    for(const TRuleCurrent& rule : m_cur->rulesContext.GetOccuredRules()) {
        if(rule.IsCancelled())
            continue;
        PushRuleSignal(rule.GetDef(), spClass, 1);
    }
}
