#pragma once

#include <mail/so/spamstop/tools/so-common/regexp.h>
#include <mail/so/spamstop/tools/so-common/so_log.h>

#include <mail/so/libs/lua_rules/runner.h>
#include <mail/so/libs/syslog/so_log.h>

#include <library/cpp/lua/eval.h>
#include <library/cpp/lua/json.h>
#include <library/cpp/threading/atomic_shared_ptr/atomic_shared_ptr.h>

#include <util/string/strip.h>

struct IRule{
    [[nodiscard]] virtual bool Run(const TStringBuf& value) = 0;
    [[nodiscard]] virtual bool Run(const NJson::TJsonValue& value) = 0;
    virtual ~IRule() = default;
};

struct TReRule : public IRule{
    [[nodiscard]] bool Run(const TStringBuf& value) override;
    [[nodiscard]] bool Run(const NJson::TJsonValue& value) override {
        Y_UNUSED(value);
        ythrow TWithBackTrace<yexception>() << "not available";
    }

    TReRule(const NRegexp::IExpression& expression, bool fMatch) noexcept
        : expression(expression), fMatch(fMatch) {}

private:
    const NRegexp::IExpression& expression;
    bool fMatch;
};

struct TReDef {
    [[nodiscard]] THolder<TReRule> MakeRule() const {
        return MakeHolder<TReRule>(*re_comp, fMatch);
    }

    bool CheckRegExp(const TStringBuf& text, NRegexp::TMatches* matches = {}, size_t coffs = {}) const {
        const auto matchResult = (matches == nullptr || coffs == 0) ? re_comp->Match(text) : re_comp->Match(text, coffs, *matches);

        return (matchResult == NRegexp::EMatchResult::NotMatch && !fMatch) ||
               (matchResult == NRegexp::EMatchResult::Match && fMatch);
    }

    explicit TReDef(const TString& text);
    explicit TReDef(TString text, bool fMatch) noexcept : text(std::move(text)), fMatch(fMatch) {}

    TReDef(TReDef &&) noexcept = default;

    TString text;        // regular expression
    bool fMatch;       // =~ || ! (true if =~)
    THolder<NRegexp::IExpression> re_comp;     // pcre compilation data
    TString test;        // test string for checking reg. expr.
};

struct THsDef {
    explicit THsDef(TStringBuf text) noexcept;
    TString Text;
};

struct TPresentRule : public IRule{
    [[nodiscard]] bool Run(const TStringBuf& value) override {
        Y_UNUSED(value);
        return true;
    }
    [[nodiscard]] bool Run(const NJson::TJsonValue& value) override {
        Y_UNUSED(value);
        return true;
    }
};

struct TPresentDef {
    static THolder<TPresentRule> MakeRule() {
        return MakeHolder<TPresentRule>();
    }
};

struct TLuaRule {
    struct TError {
        TString Message;
    };

    [[nodiscard]] TMaybe<TError> RunMultiArgs(const TVector<NJson::TJsonValue>& values) try{
        LuaRunner.Run(LuaContext.AsPtr(), func, values);
        return Nothing();
    } catch(...) {
        return TError{CurrentExceptionMessageWithBt()};
    }

    [[nodiscard]] const TString& GetFuncName() const {
        return func;
    }

    TLuaRule(const TString& func, NLua::TRunner& luaRunner, NLua::TContext& luaContext) noexcept
    : func(func)
    , LuaRunner(luaRunner)
    , LuaContext(luaContext) {}

private:
    const TLuaEval::TExpression expression;
    const TString& func;
    NLua::TRunner& LuaRunner;
    NLua::TContext& LuaContext;
};

struct TLuaRuleDef {
    TLuaRule MakeRule(NLua::TRunner& luaRunner, NLua::TContext& luaContext) const {
        return TLuaRule(func, luaRunner, luaContext);
    }

    explicit TLuaRuleDef(TString func) : func(std::move(func)) {}
    TString func;
};

struct TLvDef: public TReDef {
    using TReDef::TReDef;
    int kid{};
    int after{};
    int to{};
    int option{};
};

struct TRuleDef;
using TRulesMap = THashMap<const TRuleDef*, TTrueAtomicSharedPtr<IRule>>;
using TLuaRulesMap = THashMap<const TRuleDef*, TLuaRule>;

