#include "rulesholder.h"
#include "spruler.h"

#include <mail/so/spamstop/tools/so-common/so_log.h>

#include <util/string/split.h>

#include <utility>

#define NOT_ANTI_RULE -1

static const auto EngineRe = MakeTrueConstArray(
    TPcreUnit{ "login", "/^\\s*([^%@]+)[%@]/", true },
    TPcreUnit{ "uuencode_check", "/\nbegin\\s+[0-7]{3}\\s/", true },
    TPcreUnit{ "uuencode_end", "/\nbegin\\s+[0-7]{3}\\s.{0,72}\\n[\\x20-\\x61\\r\\n]+\\nend\\r?\\n/s", true },
    TPcreUnit{ "uuencode_bad", "/\nbegin\\s+[0-7]{3}\\s.{0,72}\\n[\\x20-\\x61\\r\\n]+/s", true },
    TPcreUnit{ "bin_attach", "/attachment.+\\.(?:jpe?g|png|gif|tiff|bmp|zip|arc|rar|pdf|doc|xls)/is", true },
    TPcreUnit{ "time_x", "/^(\\d\\d?):(\\d\\d?)$/", true },
    TPcreUnit{ "is_messageid", "/^message-id$/i", true },
    TPcreUnit{ "is_list", "/^list-|^mailing-list/i", true },
    TPcreUnit{ "emptybody", "/\\S/s", true },
    TPcreUnit{ "cyrillic", "/windows-1251|koi8-r/i", true },
    TPcreUnit{ "statipresult", "/^begin ban (\\d+) save \\d+ end/", true },
    TPcreUnit{ "url",            "/(https?:\\/\\/[^\\s]+)/i",    true },
    TPcreUnit{ "host",           "/https?:\\/\\/([^\\/<>\\s\"']+)|(([\\w\\d-]+\\.){2,}[a-z]{2,4})\\b/i",  true },
    TPcreUnit{ "ya_team_message", "/yandex-team\\.(?:ru|com(\\.ua|\\.tr)?|tr|ua)(?:[^-.a-z0-9A-Z]|$)/i", true },
    TPcreUnit{"by_smtp_host", "/by [0-9a-z-]+\\.mail\\.yandex\\.net (smtp[-a-z]*\\/Yandex)/i", true },
    TPcreUnit{"mail_backend",
     "/by [a-z\\-]+[0-9][0-9a-z-]*?\\.(?:c?mail|qloud-c)\\.yandex\\.net\\s+\\((?:([a-z\\-]+)\\/)?Yandex\\)\\s+with"
     "\\s+E?SMTPS?A?\\s+id\\s+(\\w+\\-\\w+)/i", true },
    TPcreUnit{"by_http_host", "/by [a-z0-9.-]+\\.qloud-c\\.yandex\\.net with HTTP/i", true },
    TPcreUnit{"name", "/[a-z\\-]+/", true }
);

static const auto Redirects = MakeTrueConstArray(
    TPcreUnit{ "liru1", "/^li\\.ru\\/go\\?(\\S*)/", true },
    TPcreUnit{ "liru2", "/^liveinternet\\.ru\\/click\\?(\\S*)/", true },
    TPcreUnit{ "liru3", "/^liveinternet\\.ru\\/go;li_face\\?(htt[pT]s?:\\/\\/\\S*)/", true },
    TPcreUnit{ "mail1", "/^r.mail.ru\\/cln[0-9]+\\/(\\S*)/", true },
    TPcreUnit{ "mail2", "/^kp\\.ru\\/go\\/(\\S*)/", true },
    TPcreUnit{ "vk1", "/^vk\\.com\\/away\\.php\\?to=(\\S*)/", true },
    TPcreUnit{ "subs1", "/^redirect\\.subscribe\\.ru\\/.*-\\/([^\\/ ]+)/", true },
    TPcreUnit{ "63ru", "/^63\\.ru\\/go\\/\\?url=(\\S*)/", true },
    TPcreUnit{ "yndx1", "/^r\\.mail\\.yandex\\.net\\/url\\/[a-zA-Z0-9]+,\\d+%2f(.+)/", true },
    TPcreUnit{ "gvom1", "/^gvomail\\.com\\/redir\\.php\\?.*url=http(?:%3A|:)(?:%2f|\\/){2}(.+)/", true },
    TPcreUnit{ "fedg1", "/^feedgee\\.com\\/gobylink\\.aspx\\?.*trgt=http(?:%3A|:)(?:%2f|\\/){2}(.+)/", true },
    TPcreUnit{ "wtfm1", "/^wtfmedia\\.ru\\/go\\/url=http:\\/\\/(.+)/", true },
    TPcreUnit{ "googl", "/^(?:www\\.)?google\\.[a-z]{2,3}\\/url\\?.*?=http(?:%3A|:)(?:%2f|\\/){2}(.+)/", true },
    TPcreUnit{ "rambl", "/^mail\\.rambler\\.ru\\/m\\/redirect\\?url=http(?:%3A|:)(?:%2f|\\/){2}(.+)/", true },
    TPcreUnit{ "jobs1", "/^jobs\\.ua\\/go_url\\/\\?link=(.+)/", true },
    TPcreUnit{ "jstcl", "/^syub\\.justclick\\.ru\\/.*?=http(?:%3A|:)(?:%2f|\\/){2}(.+)/", true },
    TPcreUnit{ "slow1", "/\\/(?:go|redirect|r|goto)\\.php(?:\\?|=).*(http\\S+)/", true }
);

const TTrueConst<PcreTool> TRulesHolder::m_pcre{NRegexp::TSettings{}, MakeArrayRef(*EngineRe)};
const TTrueConst<PcreTool> TRulesHolder::m_redir_pcre{NRegexp::TSettings{}, MakeArrayRef(*Redirects)};

void TRulesHolder::RulesHolderBase(bool is_spkA){
    is_spk = is_spkA;
    rulesLoadedOK = ecOK;
}

TRulesHolder::TRulesHolder(bool isSpk,
                           const TVector<TFsPath>& rulesFolders,
                           TSpLoggers& pSpLoggers,
                           const NRegexp::TSettings& pcreSettings,
                           const TMaybe<TFsPath>& hsRulesCache,
                           const TMaybe<TFsPath>& rulesDict) :
    m_p_SpLoggers(pSpLoggers)
{
    m_p_daemon_logger = m_p_SpLoggers.GetSpLoggerPtr(SO_DAEMON_LOGGER);
    m_p_rules_logger  = m_p_SpLoggers.GetSpLoggerPtr(SO_RULES_LOGGER);

    RulesHolderBase(isSpk);

    int i;
    int cDlvrNet = 1;
    int cDlvrIp = 1;
    int cDlvrSuperNet = 1;
    int cDlvrDomens = 1;

    m_rgfFieldsSave.fill(false);

    if(rulesDict) {
        TFileInput f(*rulesDict);
        RulesDict = ReadRulesDict(f);
    }

    {
        TSetRules setRules(rulesFolders, m_p_rules_logger, pcreSettings);

        if (setRules.IsOk() != ecOK) {
            rulesLoadedOK = setRules.IsOk();
            return;
        }

        m_mapBanListSender = std::move(setRules.m_pmapBanListSender);
        m_mapBanListHost = std::move(setRules.m_pmapBanListHost);
        m_vLvKeys = std::move(setRules.m_pvLvKeys);
        m_rgfFieldsSave = std::move(setRules.m_pfFieldsSave);

        rulesCS = setRules.GetCs();

        m_cRules = setRules.GetRules(m_ppRules, hsRulesCache);
        for(const THolder<TRuleDef>& rule: m_ppRules) {
            MnfRules.emplace_back(*rule);
        }

        LuaRulesRunner = std::move(setRules.LuaRulesRunner);

        m_pListRuler = std::move(setRules.m_listRuler);

        HsDbsByField = std::move(setRules.HsDbsByField);
    }

    int crul = m_cRules + 1;
    m_mapAllRules.reserve(crul);

    m_pRuleToAntiRule.resize(m_cRules);

    for (i = 0; i < m_cRules; i++) {
        m_mapAllRules.emplace(m_ppRules[i]->pRuleName, i);
        m_pRuleToAntiRule[i] = NOT_ANTI_RULE;
    }

    m_mapDomens.reserve(5 * cDlvrDomens + 1);
    m_hmDlvrIp.reserve(cDlvrIp);
    m_hmDlvrNet.reserve(cDlvrNet);
    m_hmDlvrSuperNet.reserve(cDlvrSuperNet * 256);

    for (i = 0; i < m_cRules; i++) {
        switch (m_ppRules[i]->rt) {
        case RT_BF:
        case RT_ARITHMETIC:
            //              if (m_ppRules[i]->fGroupSource)  ??????????? ????????, ?? ?? ? ????? ????
            //                  continue;
            m_vexprrl.emplace_back(*m_ppRules[i]);
            break;

        case RT_RECEIVED_NUM:
            m_vReceivedNum.emplace_back(std::get<TReDef>(m_ppRules[i]->rules).test.c_str(), i, *m_ppRules[i]);
            break;

        case RT_ANTI:
            SetAntiRule(i);
            break;

        case RT_LV:
            m_vLvRules.push_back(i);
            break;

        default:
            break;
        }

    }

    for (i = 0; i < m_cRules; i++) {
        for (const auto fid : m_ppRules[i]->pfields) {
            rulesByField[fid].emplace_back(*m_ppRules[i]);
        }
    }

    for (i = 0; i < m_cRules; i++) {
        if (m_ppRules[i]->rt != RT_PRESENT && m_ppRules[i]->rt != RT_ABSENT)
            continue;
        for (const auto fid : m_ppRules[i]->pfields) {
            vFieldsExists.emplace_back(fid, i);
        }
    }

//    SetFields();

    m_spruler = MakeHolder<TSpRuler>(m_p_rules_logger, m_cRules, m_ppRules);
    m_spruler->SetRules();

#ifdef SP_STAT_IP
    if (!m_cRules)
        return ecOK;
#else
    if (!m_cRules) {
        m_p_daemon_logger->splog(TLOG_ERR, "%s", "Couldn't define filter rules.");
        rulesLoadedOK = ecReading;
        return;
    }
#endif

    m_rgfFieldsSave[FD_FROM] = true;
    m_rgfFieldsSave[FD_FROM_NAME] = true;
    m_rgfFieldsSave[FD_FROM_ADDR] = true;
    m_rgfFieldsSave[FD_FROM_DOMAIN] = true;
    m_rgfFieldsSave[FD_MAIL_FROM] = true;
    m_rgfFieldsSave[FD_TO] = true;
    m_rgfFieldsSave[FD_TO_ADDR] = true;
    m_rgfFieldsSave[FD_SUBJECT] = true;
    m_rgfFieldsSave[FD_MESSAGEID] = true;
    m_rgfFieldsSave[FD_MESSAGE_ID] = true; //real message-id
    m_rgfFieldsSave[FD_DATE] = true;
    m_rgfFieldsSave[FD_RESENT_DATE] = true;
    m_rgfFieldsSave[FD_X_MAILER] = true;
    m_rgfFieldsSave[FD_SENDER] = true;
    m_rgfFieldsSave[FD_X_SENDER] = true;
    m_rgfFieldsSave[FD_REPLY_TO_ADDR] = true;
    m_rgfFieldsSave[FD_RAWSUBJECT] = true;
    m_rgfFieldsSave[FD_X_YAMAIL_AUTO_REPLY] = true;
    m_rgfFieldsSave[FD_X_BEENTHERE] = true;
    m_rgfFieldsSave[FD_LIST_] = true;
    m_rgfFieldsSave[FD_LIST_POST] = true;
    m_rgfFieldsSave[FD_LIST_OWNER] = true;
    m_rgfFieldsSave[FD_LIST_SUBSCRIBE] = true;
    m_rgfFieldsSave[FD_DELIVERED_TO] = true;
    m_rgfFieldsSave[FD_IY_GEOZONE] = true;
    m_rgfFieldsSave[FD_IY_FIRSTGEOZONE] = true;
    m_rgfFieldsSave[FD_RETURN_PATH] = true;
    m_rgfFieldsSave[FD_USER_AGENT] = true;
    m_rgfFieldsSave[FD_X_YANDEX_QUEUEID] = true;
    m_rgfFieldsSave[FD_X_YANDEX_RPOP_FOLDER] = true;

    m_rgfFieldsSave[FD_AUTHENTICATION_RESULTS] = true;
    m_rgfFieldsSave[FD_X_YANDEX_AUTHENTICATION_RESULTS] = true;

    rulesLoadedOK = ecOK;

}

//TRulesHolder::TRulesHolder(const char *rulesFolder, const char* pPrepare, TSpLogger *pSpDaemonLogger, TSpLogger *pSpFilterLogger, int* pcErrors) : TRulesHolder(false, rulesFolder, pPrepare, pSpLogger, pcErrors) {}

void TRulesHolder::SetAntiRule(int rid) {
    TAntiDef* anti = &std::get<TAntiDef>(m_ppRules[rid]->rules);

    const TString& antiruleName = m_ppRules[rid]->pRuleName;
    const TString& masterRuleName = anti->pAntiRuleName;

    const i32* masterRuleIndex = MapFindPtr(m_mapAllRules, masterRuleName);
    if (!masterRuleIndex) {
        ythrow TWithBackTrace<yexception>() << "Could not find rule (" << masterRuleName << ") for antirule: " << antiruleName;
    }

    anti->pCancelRules.reserve(anti->vCancelRulesNames.size());

    for (const TString& ruleToCancelName : anti->vCancelRulesNames) {
        if(const i32* ruleToCancelIndex = MapFindPtr(m_mapAllRules, ruleToCancelName)) {
            anti->pCancelRules.emplace_back(*m_ppRules[*ruleToCancelIndex]);
        } else {
            ythrow TWithBackTrace<yexception>() << "Could not find rule (" << ruleToCancelName << ") for antirule: " << antiruleName;
        }
    }

    m_pRuleToAntiRule[*masterRuleIndex] = rid;
}

const TRuleDef* TRulesHolder::RuleById(int rid) const
{
    if (rid < m_cRules)
        return m_ppRules[rid].Get();

    return nullptr;
}

TRuleDef *TRulesHolder::RuleById(int rid)
{
    if (rid < m_cRules)
        return m_ppRules[rid].Get();

    return nullptr;
}

const TVector<std::reference_wrapper<const TRuleDef>>& TRulesHolder::RulesByField(TSpFields fid) const {
    return rulesByField[fid];
}

TRulesType TRulesHolder::RuleTypeById(int rid) const
{
    return m_ppRules[rid]->rt;
}

TString TRulesHolder::RuleNameById(int rid) const
{
    if (rid >= 0 && rid < (int)m_ppRules.size())
        return m_ppRules[rid]->pRuleName;

    return nullptr;
}

bool TRulesHolder::AllRulesFindRid(const TStringBuf& rule, int &rid) const {
    auto it = m_mapAllRules.find(rule);
    if(cend(m_mapAllRules) == it)
        return false;

    rid = it->second;
    return true;
}

int TRulesHolder::GetRuleToAntirule(int rid) const {
    return m_pRuleToAntiRule[rid];
}

const TRuleDef *TRulesHolder::FindDomainRule(const TStringBuf& domain) const {
    auto it = m_mapDomens.find(domain);

    if (it != m_mapDomens.cend())
        return it->second;
    return nullptr;
}

const TRuleDef* TRulesHolder::FindDeliverySuperNetRule(ui32 net) const {
    auto it = m_hmDlvrSuperNet.find(net);

    if (it != m_hmDlvrSuperNet.cend())
        return (*it).second;

    return nullptr;
}

const TRuleDef* TRulesHolder::FindDeliveryNetRule(ui32 net) const {
    auto it = m_hmDlvrNet.find(net);

    if (it != m_hmDlvrNet.cend())
        return (*it).second;

    return nullptr;
}

const TRuleDef* TRulesHolder::FindDeliveryIpRule(ui32 ip) const {
    auto it = m_hmDlvrIp.find(ip);

    if (it != m_hmDlvrIp.cend())
        return (*it).second;

    return nullptr;
}

THashMap<TString, int> TRulesHolder::ReadRulesDict(IInputStream &s) {
    THashMap<TString, int> dict;
    TString line;
    int index;
    TStringBuf name, tmp;
    while (s.ReadLine(line)) {
        if (line) {
            Split(line, '\t', index, name, tmp, tmp);
            dict.emplace(name, index);
        }
    }
    return dict;
}
