#include "sptop.h"
#include <string.h>
#include <algorithm>
#include "sptypes.h"
#include "mail/so/spamstop/tools/so-common/sputil.h"
#include "setlistrule.h"
#include "rengine.h"
#include "spstat.h"
//#include "sperror.h"
#include "setrules.h"
#include <time.h>
#include <fcntl.h>
#ifdef WIN32
#include <stdio.h>
#else
#include <unistd.h> // unlink
#include <util/generic/string.h>
#include "processing_context.h"

#endif

TSpListRuler::TSpListRuler(TSpLogger *pSpLogger)
{
    m_p_sp_logger = pSpLogger;
//    is_spk = is_spkA;
    m_crules = 0;
    m_cwordsall = 0;

    m_fTextWithSpaces = false;

    memset(m_cKeyWords, 0, sizeof(m_cKeyWords));
}

void TSpListRuler::InitFile() {
    m_idList = 0;
    m_ckeywords = 0;
    m_cfilewords = 0;

    m_list_info.ind = -1;
    m_list_info.indvrules = -1;
    m_list_info.sKeyword.assign("");

    m_list_info.fSubject = false;
    m_list_info.fMessageId = false;
    m_list_info.fIgnoreDsl = false;
    m_list_info.sValue.assign("");
    m_list_info.fdlv = false;
    m_list_info.fipv46 = false;
    m_list_info.fipv46first = false;
    m_list_info.fMX = false;
    m_list_info.sRulename.erase();
}

bool TSpListRuler::Add(const char* pstr, const char* fn) {
    int len;

    if (*pstr == '#') // commentary
    {
        if (!STRNCMP(pstr, "#@skip_package"))
            ++m_ckeywords;
        else if (!STRNCMP(pstr, "#@list_type")) {
            ++m_ckeywords;
            len = TSetRules::GetNextWord(&pstr, 0);
            const auto lowered = to_lower(TString(pstr, len));

            TSpListType type{};
            if (!len || !TryFromString<TSpListType>(lowered, type)) {
                Syslog(TLOG_ERR) << "Undefined list type: " << pstr << ", file: " << fn << ", available: " << GetEnumAllNames<TSpListType>();
                return false;
            }

            m_idList = type;
            m_list_info.ind = m_idList;
            if (m_idList == SP_LIST_DELIVERY)
                m_list_info.fdlv = true;
            else if (m_idList == SP_LIST_IPV46)
                m_list_info.fipv46 = true;
            else if (m_idList == SP_LIST_IPV46_FIRST)
                m_list_info.fipv46first = true;
            else if (m_idList == SP_LIST_MX)
                m_list_info.fMX = true;
        } else if (!STRNCMP(pstr, "#@rule")) {
            if (m_ckeywords != 2) {
                m_p_sp_logger->splog(TLOG_ERR, "Fault roll format: %s", fn);
                return false;
            }
            len = TSetRules::GetNextWord(&pstr, 0);

            TString sRule;
            sRule.assign(pstr, len);
            m_list_info.indvrules = m_crules++;
            m_list_info.sRulename.assign(sRule);
            m_vrules.push_back(sRule);
            if (m_list_info.fipv46) {
                mZoneIPV46 = {};
            }
            if (m_list_info.fipv46first) {
                mZoneIPV46First = {};
            }
        } else if (!STRNCMP(pstr, "#@textwithspaces"))
            m_fTextWithSpaces = true;
        else if (!STRNCMP(pstr, "#@dlv_subject"))
            m_list_info.fSubject = true;
        else if (!STRNCMP(pstr, "#@dlv_messageid"))
            m_list_info.fMessageId = true;
        return true;
    } else if (TSetRules::End_Of_Str(*pstr)) // empty line
        return true;

    if (m_ckeywords != 2 || m_list_info.ind == -1 || m_list_info.indvrules == -1) {
        m_p_sp_logger->splog(TLOG_ERR, "Fault roll format: %s", fn);
        return false;
    }

    auto ptext = pstr;
    int textlen = 0;
    SpTrim(pstr, (const char**)(&ptext), &textlen);

    if (m_fTextWithSpaces && !m_list_info.fdlv)
        len = strlen(ptext);
    else
        len = TSetRules::GetFirstWord(&ptext, 0);

    if (!len)
        return true;

    m_list_info.sKeyword.assign(ptext, len);

    if (m_list_info.fdlv) {
        if (textlen > len + 1)
            m_list_info.sValue.assign(ptext + (len + 1), textlen - len - 1);
        else
            m_list_info.sValue.assign("");

        m_vlist_info.push_back(m_list_info);
    } else if (m_list_info.fipv46) {
        mZoneIPV46[m_list_info.sRulename].Add(TIpAddressRange::FromString(TString{StripString(TStringBuf(ptext))}));
    } else if (m_list_info.fipv46first) {
        mZoneIPV46First[m_list_info.sRulename].Add(TIpAddressRange::FromString(TString{StripString(TStringBuf(ptext))}));
    } else if (m_list_info.fMX) {
        char *tok, *brk, *row = strdup(ptext + len + 1);
        if (row) {
            for (tok = strtok_r(row, " \n\r", &brk); tok; tok = strtok_r(nullptr, " \n\r", &brk)) {
                if (strlen(tok) == 0)
                    continue;

                ToLower(m_list_info.sKeyword.begin(), m_list_info.sKeyword.size());
                bool bNeedInit = m_v_mxs.find(m_list_info.sRulename) == m_v_mxs.end();
                if (!bNeedInit) {
                    auto zone_it = m_v_mxs[m_list_info.sRulename].find(m_list_info.sKeyword);
                    bNeedInit = zone_it == m_v_mxs[m_list_info.sRulename].end();
                }
                if (bNeedInit) {
                    m_v_mxs[m_list_info.sRulename][m_list_info.sKeyword] = {};
                }

                m_v_mxs[m_list_info.sRulename][m_list_info.sKeyword].Add(TIpAddressRange::FromString(TString{StripString(TStringBuf(tok))}));
            }
            free(row);
        }
    } else
        m_vlist_info.push_back(m_list_info);

    ++m_cKeyWords[m_idList];
    ++m_cfilewords;
    ++m_cwordsall;

    return true;
}

void TSpListRuler::Prepare() {
    int count_pair = 0;
    for (int j = 0; j < SP_LIST_MAX; j++) {
        if (j == SP_LIST_DELIVERY)
            m_map_dlv.reserve(m_cKeyWords[j]);

//        m_spmap[j].AssignName(sp_list_names[j].name);
        m_spmap[j].reserve(m_cKeyWords[j]);
        count_pair += m_cKeyWords[j];
    }

    THashSet<std::tuple<TString, TString>> mAllStrings;
    mAllStrings.reserve(count_pair);

    for (const auto &iv : m_vlist_info) {
        auto lowered = to_lower(iv.sKeyword);
        if (!mAllStrings.emplace_noresize(std::tie(lowered, m_vrules[iv.indvrules])).second) {
            m_p_sp_logger->splog(TLOG_ERR, "Duplicated pair: word = %s, rule = %s", iv.sKeyword.c_str(), m_vrules[iv.indvrules].c_str());
            continue;
        }

        if (iv.fdlv) {
            decltype(m_map_dlv)::const_iterator it = m_map_dlv.find(lowered);

            if (m_map_dlv.cend() != it) {
                m_p_sp_logger->splog(TLOG_ERR, "Duplicated delivery list key:  %s, rule = %s", iv.sKeyword.c_str(), m_vrules[iv.indvrules].c_str());
                continue;
            } else {

                TSpListValueDlv dlv_value;
                dlv_value.indvrules = iv.indvrules;
                dlv_value.fSubject = iv.fSubject;
                dlv_value.fMessageId = iv.fMessageId;
                if (GetDeliveryValue(iv.sKeyword.c_str(), iv.sValue.c_str(), &dlv_value)) {
                    m_map_dlv.emplace(std::move(lowered), std::move(dlv_value));
                } else
                    m_p_sp_logger->splog(TLOG_ERR, "Bad delivery list value for key:  %s, rule = %s", iv.sValue.c_str(), iv.sKeyword.c_str());
            }
        } else {
            auto it = m_spmap[iv.ind].find(lowered);

            if(it == m_spmap[iv.ind].end()) {
                m_spmap[iv.ind].emplace(std::move(lowered), TVector<int>{iv.indvrules});
            } else {
                it->second.emplace_back(iv.indvrules);
            }
        }
    }

    m_vlist_info.clear();
}

bool TSpListRuler::GetDeliveryValue(const char* pkey, const char* pvalue, TSpListValueDlv* dlv_value) {
    const char* pnextword;
    int len = SpGetFirstWord(&pvalue, &pnextword);
    ui32 ip, netc;
    const int cfrom_prefix = strlen("from:");
    TString sZoneRow;
    TIpAddr tipIP;

    while (len > 0) {
        if (!STRNCMP(pvalue, "domain_")) {
            dlv_value->sDomainLabel.assign(pvalue, len);
        } else if (!STRNCMP(pvalue, "from:domain_")) {
            dlv_value->fFromFieldDomainLabel = true;
            dlv_value->sDomainLabel.assign(pvalue + cfrom_prefix, len - cfrom_prefix);
        } else
            switch (SetValue(pvalue, len, &ip, &netc, &sZoneRow)) {
                case SP_LIST_DLV_IP:
                    if (++dlv_value->cIp == 1)
                        dlv_value->indIp = m_vip_dlv.size();
                    m_vip_dlv.push_back(ip);
                    break;

                case SP_LIST_DLV_NET:
                    if (++dlv_value->cNet == 1)
                        dlv_value->indNet = m_vnet_dlv.size();
                    m_vnet_dlv.push_back(netc);
                    break;

                case SP_LIST_DLV_MASK:
                    if (!sZoneRow.empty()) {
                        if (++dlv_value->cMask == 1)
                            dlv_value->indMask = (int)m_vmask_dlv.size();
                        ip_match matcher;
                        m_vmask_dlv.push_back(matcher);

                        size_t netBegin = 0, netEnd = 0;
                        TString sSubnet;
                        while (true) {
                            netEnd = sZoneRow.find_first_of(" \t", netBegin);
                            if (netEnd != TString::npos)
                                sSubnet = sZoneRow.substr(netBegin, netEnd - netBegin);
                            else
                                sSubnet = sZoneRow.substr(netBegin);

                            m_vmask_dlv.back().add_mask(sSubnet);

                            if (netEnd == TString::npos)
                                break;

                            netBegin = (int)sZoneRow.find_first_not_of(" \t", netEnd);
                            if (netBegin == TString::npos || netBegin > sZoneRow.length())
                                break;
                        }
                    }
                    break;

                case SP_LIST_DLV_RELAY:
                    if (++dlv_value->cRelay == 1)
                        dlv_value->indRelay = m_vrelay_dlv.size();
                    m_vrelay_dlv.emplace_back(pvalue, len);
                    break;
                case SP_LIST_DLV_TRUSTED:
                    dlv_value->CheckTrusted = true;
                    break;
                case SP_LIST_DLV_FAULT:
                default:
                    m_p_sp_logger->splog(TLOG_ERR, "Bad delivery list value for key:  %s, rule = %s", pvalue, pkey);
            }

        pvalue = pnextword;
        len = SpGetNextWord(&pvalue, &pnextword);
    }

    return true;
}

const TSpListValueDlv * TSpListRuler::SetDeliveryDomen(TRulesContext& rulesContext, const TString& pdomen, const bool* fields, bool* fCancelDomainLabel, bool* fBanOrFreem) const {
    *fBanOrFreem = false;

    const auto & lowered = to_lower(pdomen);

    decltype(m_map_dlv)::const_iterator it = m_map_dlv.find(lowered);

    if(it == m_map_dlv.cend())
        return {};

    const TSpListValueDlv & dlvValue = it->second;

    *fCancelDomainLabel = !fields[en_SR_FROM] && dlvValue.fFromFieldDomainLabel;

    if(!(fields[en_SR_FROM] ||
         fields[en_SR_REPLYTO] ||
         fields[en_SR_LIST] ||
         fields[en_SR_MESSID] && dlvValue.fMessageId ||
         fields[en_SR_SUBJ] && dlvValue.fSubject))
        return {};

    if (STRNCMP(m_vrules[dlvValue.indvrules].c_str(), "DLV_BAN") &&
        STRNCMP(m_vrules[dlvValue.indvrules].c_str(), "DLV_FREEM"))
        rulesContext.SetRule("DOMN_ROLL");

    if ((STRNCMP(m_vrules[dlvValue.indvrules].c_str(), "DLV_BAN") == 0) ||
        (STRNCMP(m_vrules[dlvValue.indvrules].c_str(), "DLV_FREEM") == 0)) {
        *fBanOrFreem = true;
    }

    return &dlvValue;
}

bool TSpListRuler::CheckDeliveryIp(TRulesContext& rulesContext, const TSpListValueDlv& dlvValue, ui32 ip, TStringBuf& pRuleName, TStringBuf& pDomainLabel) const {
    for (int i = 0; i < dlvValue.cIp; i++)
        if (m_vip_dlv[dlvValue.indIp + i] == ip) {
            pRuleName = m_vrules[dlvValue.indvrules];
            rulesContext.SetRule(pRuleName);
            if (dlvValue.sDomainLabel)
                pDomainLabel = dlvValue.sDomainLabel;
            return true;
        }

    return false;
}

bool TSpListRuler::CheckDeliveryNet(TRulesContext& rulesContext, const TSpListValueDlv& dlvValue, ui32 net, TStringBuf& pRuleName, TStringBuf& pDomainLabel) const {
    for (int i = 0; i < dlvValue.cNet; i++)
        if (m_vnet_dlv[dlvValue.indNet + i] == net) {
            pRuleName = m_vrules[dlvValue.indvrules];
            rulesContext.SetRule(pRuleName);
            if (dlvValue.sDomainLabel)
                pDomainLabel = dlvValue.sDomainLabel;
            return true;
        }

    return false;
}

bool TSpListRuler::CheckDeliveryMask(TRulesContext& rulesContext, const TSpListValueDlv& dlvValue, TIpAddr IP, TStringBuf& pRuleName, TStringBuf& pDomainLabel) const {
    for (int i = 0; i < dlvValue.cMask; i++)
        if (m_vmask_dlv[dlvValue.indMask + i].match(IP)) {
            pRuleName = m_vrules[dlvValue.indvrules];
            rulesContext.SetRule(pRuleName);
            if (dlvValue.sDomainLabel)
                pDomainLabel = dlvValue.sDomainLabel;
            return true;
        }

    return false;
}

bool TSpListRuler::CheckMXZone(TRulesContext& rulesContext, const char* ip_word, const char* sender_host) const {
    if (!ip_word || !sender_host)
        return false;

    TKIPv6 ip(ip_word);
    if (strlen(sender_host) && !ip.Undefined()) {
        for (const auto& m_v_mx : m_v_mxs) {
            for (auto zone_it = m_v_mx.second.cbegin(); zone_it != m_v_mx.second.cend(); zone_it++) {
                bool ok;
                if (cmpdomensrevi(zone_it->first.c_str(), sender_host) && zone_it->second.Contains(TIpv6Address::FromString(ip_word, ok))) {
                    rulesContext.SetRule(m_v_mx.first);
                    return true;
                }
            }
        }
    }

    return false;
}

bool TSpListRuler::CheckDeliveryRelay(TRulesContext& rulesContext, const TSpListValueDlv& dlvValue, TString sRelay, TStringBuf& pRuleName, TStringBuf& pDomainLabel) const {
    for (int i = 0; i < dlvValue.cRelay; i++)
        if (cmpdomensrevi(m_vrelay_dlv[dlvValue.indRelay + i], sRelay)) {
            pRuleName = m_vrules[dlvValue.indvrules];
            rulesContext.SetRule(pRuleName);
            if (dlvValue.sDomainLabel)
                pDomainLabel = dlvValue.sDomainLabel;
            return true;
        }

    return false;
}

bool TSpListRuler::CheckDeliveryTrusted(TRulesContext& rulesContext, const TSpListValueDlv& dlvValue, TStringBuf& pRuleName, TStringBuf& pDomainLabel) const {
    if(dlvValue.CheckTrusted && rulesContext.IsRuleWorked("ALLTRUSTEDIP")) {
        pRuleName = m_vrules[dlvValue.indvrules];
        rulesContext.SetRule(pRuleName);
        if (dlvValue.sDomainLabel)
            pDomainLabel = dlvValue.sDomainLabel;
        return true;
    }

    return false;
}

bool TSpListRuler::CheckDeliverySpfDkim(TRulesContext& rulesContext, const TSpListValueDlv& dlvValue, TStringBuf& pRuleName, TStringBuf& pDomainLabel, bool dkim, bool spf) const {
    //  need not check IPs etc for spf-dkim
    //    if (m_dlv_value.cIp || m_dlv_value.cNet || m_dlv_value.cMask || m_dlv_value.cRelay)
    pRuleName = m_vrules[dlvValue.indvrules];

    if (pRuleName) {
        rulesContext.SetRule(pRuleName);
        if(spf)
            rulesContext.SetRule("DL_SPF");
        if(dkim)
            rulesContext.SetRule("DL_DKIM");

        if (dlvValue.sDomainLabel)
            pDomainLabel = dlvValue.sDomainLabel;

        return true;
    }

    return false;
}

TSpListDlvValueType TSpListRuler::SetValue(const char* pvalue, int len, ui32* ip, ui32* netc, TString* sZoneRow) {
    if(TStringBuf(pvalue, len) == "trustedzone") {
        return SP_LIST_DLV_TRUSTED;
    }

    int i = 0, cdot = 0, c_colon = 0;
    bool fip = true;
    bool fipv6 = true;
    const char* plast_dot = 0;
    bool fCidr = false;

    for (i = 0; i < len; i++)
        if (pvalue[i] == '.') {
            ++cdot;
            plast_dot = pvalue + i;
        } else if (pvalue[i] == '/') {
            if (fCidr)
                return SP_LIST_DLV_FAULT;
            fCidr = true;
        } else if (pvalue[i] < '0' || pvalue[i] > '9') {
            fip = false;
            if (pvalue[i] == ':')
                ++c_colon;
            else if (!((pvalue[i] >= 'a' && pvalue[i] <= 'f') || (pvalue[i] >= 'A' && pvalue[i] <= 'F')))
                fipv6 = false;
        }

    if (sZoneRow && (fCidr && fip && cdot == 3 || fipv6 && c_colon > 1)) {
        sZoneRow->assign(pvalue);
        return SP_LIST_DLV_MASK;
    }

    if (fip) {
        if (cdot == 3) {
            if (IpToInt(pvalue, len, netc, ip))
                return SP_LIST_DLV_IP;
            else
                return SP_LIST_DLV_FAULT;
        } else if (cdot == 2) {
            if (IpToInt(pvalue, len, netc, nullptr))
                return SP_LIST_DLV_NET;
            else
                return SP_LIST_DLV_FAULT;
        } else
            return SP_LIST_DLV_FAULT;
    }

    if (!plast_dot)
        return SP_LIST_DLV_FAULT;

    while (++plast_dot < (pvalue + len))
        if (!isletter((unsigned char)(*plast_dot)))
            return SP_LIST_DLV_FAULT;

    return SP_LIST_DLV_RELAY;
}

bool TSpListRuler::CheckWord(TRulesContext& rulesContext, const TStringBuf& word, TSpListType type) const {
    if (word.empty())
        return false;

    auto it = m_spmap[type].find(to_lower(TString{word.substr(0, Min(word.size(), size_t(1024)))}));

    if (it == m_spmap[type].end())
        return false;

    for(int i : it->second)
        rulesContext.SetRule(m_vrules[i]);

    return true;
}

bool TSpListRuler::FindWord(const char* word, int len, TSpListType type, const TStringBuf & rule) const {
    if (len <= 0)
        return false;

    auto it = m_spmap[type].find(to_lower(TString{word, Min(size_t(len), size_t(1024))}));

    if (it == m_spmap[type].end())
        return false;

    const auto & pv_rules = it->second;
    for (const auto & vr : pv_rules)
        if (m_vrules[vr] == rule)
            return true;
    return false;
}

void TSpListRuler::CheckDomainName(TRulesContext& rulesContext, const TStringBuf & word, TSpListType type, TSpListType type_part) const {
    if (word.size() < 2)
        return;

    CheckWord(rulesContext, word, type);
    CheckWord(rulesContext, word, type_part);

    for (size_t i = 0; i < word.size() - 2; i++)
        if (word[i] == '.')
            CheckWord(rulesContext, word.substr(i + 1), type_part);
}

void TSpListRuler::CheckDomainNamePart(TRulesContext& rulesContext, const TStringBuf & word, TSpListType type_part) const {
    if (word.size() < 2)
        return;

    CheckWord(rulesContext, word, type_part);

    for (size_t i = 0; i < word.size() - 2; i++)
        if (word[i] == '.')
            CheckWord(rulesContext, word.substr(i + 1), type_part);
}

void TSpListRuler::CheckIpNet(TRulesContext& rulesContext, const TStringBuf & word, TSpListType type, TSpListType type_part) const {
    if (word.empty())
        return;

    CheckWord(rulesContext, word, type);
    CheckWord(rulesContext, word, type_part);

    for (size_t i = word.size() - 1; i > 0; i--)
        if (word[i] == '.')
            CheckWord(rulesContext, word.substr(0, i), type_part);
}

// shingle without counter
bool TSpListRuler::CheckShingleWl(const char* word, int len) const {
    return FindWord(word, len, SP_LIST_WL, "SP_LIST_WL");
}

// geo domens and hostings
bool TSpListRuler::CheckShingleGeo(const char* word, int len) const {
    return FindWord(word, len, SP_LIST_GEO, "SP_LIST_GEO");
}

bool TSpListRuler::CheckList(const TStringBuf& word, TSpListType type, const TStringBuf& rule_name) const {
    if (word.empty())
        return false;


    auto it = m_spmap[type].find(to_lower(TString{word.substr(0, Min(word.size(), size_t(1024)))}));

    if(it != m_spmap[type].end()) {
        for (const auto & vr : it->second)
            if (rule_name == m_vrules[vr])
                return true;
    }

    return false;
}

bool TSpListRuler::CheckListNoRuleSet(const char* word, int len, TSpListType type) const {
    if (!word || len <= 0)
        return false;

    TString tmpStr;
    tmpStr.assign(word, Min((size_t)len, (size_t)1024));

    return m_spmap[type].contains(to_lower(tmpStr));
}

bool TSpListRuler::CheckListNoRuleSet(const TStringBuf& word, TSpListType type) const {
    return CheckListNoRuleSet(word.data(), word.size(), type);
}

bool TSpListRuler::CheckPhrase(const char* word, int len, std::vector<const char*>& vphraserules) const {
    if (len <= 0)
        return false;

    auto it = m_spmap[SP_LIST_PHRASE].find(to_lower(TString{word, Min(size_t(len), size_t(1024))}));

    if(m_spmap[SP_LIST_PHRASE].cend() != it) {
        for (const auto & vr : it->second)
            vphraserules.push_back(m_vrules[vr].c_str());
        return true;
    }

    return false;
}

bool TSpListRuler::CheckIpv46(TRulesContext& rulesContext, TIpAddr ipa) const {
    TKIPv6 ip = ipa.ToTKIPv6();
    return CheckIpv46(rulesContext, &ip);
}

bool TSpListRuler::CheckIpv46(TRulesContext& rulesContext, TKIPv6* ip) const {
    bool bResult = false;

    if (ip->Undefined())
        return bResult;

    bool ok;
    for (auto it = mZoneIPV46.begin(); it != mZoneIPV46.end(); ++it)
        if (it->second.Contains(TIpv6Address::FromString(ip->toStroka(), ok))) {
            rulesContext.SetRule(it->first);
            bResult = true;
        }

    return bResult;
}

bool TSpListRuler::CheckIpv46First(TRulesContext& rulesContext, TIpAddr ipa) const {
    TKIPv6 ip = ipa.ToTKIPv6();
    return CheckIpv46First(rulesContext, &ip);
}

bool TSpListRuler::CheckIpv46First(TRulesContext& rulesContext, TKIPv6* ip) const {
    bool bResult = false;

    if (ip->Undefined())
        return bResult;

    bool ok;
    for (auto it = mZoneIPV46First.begin(); it != mZoneIPV46First.end(); ++it)
        if (it->second.Contains(TIpv6Address::FromString(ip->toStroka(), ok))) {
            rulesContext.SetRule(it->first);
            bResult = true;
        }

    return bResult;
}

void TSpListRuler::CheckIp(TRulesContext& rulesContext, const TStringBuf& word) const {
    CheckWord(rulesContext, word, SP_LIST_IP);
    CheckWord(rulesContext, word, SP_LIST_IP_PART);

    for (size_t i = word.size() - 1; i > 0; i--)
        if (word[i] == '.')
            CheckWord(rulesContext, word.substr(0, i), SP_LIST_IP_PART);
}

void TSpListRuler::CheckRdns(TRulesContext& rulesContext, const TStringBuf& word) const {

    CheckWord(rulesContext, word, SP_LIST_RDNS);
    CheckWord(rulesContext, word, SP_LIST_RDNS_PART);

    for (size_t i = 0; i < word.size() - 2; i++)
        if (word[i] == '.')
            CheckWord(rulesContext, word.substr(i + 1), SP_LIST_RDNS_PART);
}

void TSpListRuler::CheckGEO(TRulesContext& rulesContext, const TStringBuf& word) const {
    //int i;
    //const char *pword = word;
    TString geocountry = "";
    TString geocity = "";
    bool endcountry = false;
    char symb = 0x00;

    if(word.empty())
        return;

    for (size_t i = 0; i < word.size(); i++) {
        if (word[i] == ' ') {
            endcountry = true;
            symb = '_';
        } else
            symb = word[i];

        if (!endcountry)
            geocountry = geocountry + TString(symb);
        geocity = geocity + TString(symb);
    }

    CheckWord(rulesContext, geocountry, SP_LIST_GEO_COUNTRY);
    CheckWord(rulesContext, geocity, SP_LIST_GEO_CITY);
}

void TSpListRuler::CheckSuidRoll(TRulesContext& rulesContext, const TStringBuf& word) const {
    CheckWord(rulesContext, word, SP_LIST_SUIDS_MSEARCH);
}
