#include <algorithm>

#include <mail/so/spamstop/tools/so-common/safe_recode.h>

#include <mail/so/libs/protect/protect.h>

#include "sptypes.h"
#include "spalg.h"
#include "sptri.h"
#include "sptop.h"
#include "spstat.h"
#include "setrules.h"
#include "sphtml.h"
#include "setlistrule.h"
#include "spruler.h"
#include "rengine.h"

#include <util/string/builder.h>
#include <util/string/ascii.h>
#include <util/string/strip.h>
#include <util/string/type.h>

static const auto AlgRe = MakeTrueConstArray(
    // Subject spam Id
    TPcreUnit{"id_1", "/[-_\\.\\s]{7,}([-a-z0-9]{4,})$/", true},
    TPcreUnit{"id_2", "/\\s{10,}(?:\\S\\s)?(\\S+)$/", true},
    TPcreUnit{"id_3", "/\\s{3,}[-:\\#\\(\\[]+([-a-z0-9]{4,})[\\]\\)]+$/", true},
    TPcreUnit{"id_4", "/\\s{3,}[:\\#\\(\\[]*([a-f0-9]{4,})[\\]\\)]*$/", true},
    TPcreUnit{"id_5", "/\\s{3,}[-:\\#]([a-z0-9]{5,})$/", true},
    TPcreUnit{"id_6", "/[\\s._]{3,}([^0\\s._]\\d{3,})$/", true},
    TPcreUnit{"id_7", "/[\\s._]{3,}\\[(\\S+)\\]$/", true},
    //    (7217vPhZ0-478TLdy5829qicU9-0@26) and similar
    TPcreUnit{"id_8", "/\\(([-a-zA-Z]{7,}\\@\\d+)\\)$/", true},
    //    Seven or more digits at the end of a subject is almost certainly a id
    TPcreUnit{"id_9", "/\\b(\\d{7,})\\s*$/", true},
    //    stuff at end of line after "!" or "?" is usually an id
    TPcreUnit{"id_10", "/[!\\?]\\s*(\\d{4,}|[a-zA-Z]+(-[a-zA-Z]+)+)\\s*$/", true},
    //    9095IPZK7-095wsvp8715rJgY8-286-28 and similar
    TPcreUnit{"id_11", "/\\b([a-zA-Z]{7,}-[a-zA-Z]{7,}(-[a-zA-Z]+)*)\\s*$/", true},
    //    #30D7 and similar
    TPcreUnit{"id_12", "/\\s#\\s*([a-f0-9]{4,})\\s*$/", true},
    TPcreUnit{"id_test0", "/\\s{8,}$/", true},
    TPcreUnit{"id_test1", "/\\d{5,}/", true},
    TPcreUnit{"id_test2", "/(?:item|invoice|order|number|confirmation).{1,6}\\s*$/", true},
    TPcreUnit{"id_test3", "/[^a-z\\']/", true},
    TPcreUnit{"id_test4", "/(?:linux|nix|bsd)/", true},
    TPcreUnit{"id_test5", "/(?:whew|phew|attn|tha?nx)/", true},
    TPcreUnit{"id_test7", "/�����|��������|���|����|�����|��[ţ]�|(?:\\s�|N\\s*$)/i", true},
    TPcreUnit{"id_test8", "/\\s�|N\\s*$/i", true},
    // MessageId
    TPcreUnit{"mesid_outlooktime", "/^<[0-9a-f]{4}([0-9a-f]{8})\\$[0-9a-f]{8}\\$[0-9a-f]{8}\\@/", true},
    // Date
    TPcreUnit{"dd_mon_yyyy", "/ (\\d+) (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) (\\d{4}) /i", true},
    TPcreUnit{"mon_dd_yyyy", "/ (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) +(\\d+) \\d+:\\d+:\\d+ (\\d{4}) /i", true},
    TPcreUnit{"dd_mon_yy", "/ (\\d+) (Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec) (\\d{2,3}) /i", true},
    TPcreUnit{"hh_mm_ss", "/ (\\d?\\d):(\\d\\d)(:(\\d\\d))? /", true},
    TPcreUnit{"numeric_timezone", "/ ([-+]\\d{4}) /", true},
    TPcreUnit{"text_timezone", "/\\b([A-Z]{2,4})\\b/", true},
    TPcreUnit{"tzoff", "/([-+])(\\d\\d)(\\d\\d)$/", true},
    // GatedThroughReceivedHdrRemover
    TPcreUnit{"run_by_ezmlm", "/^contact \\S+\\@\\S+\\; run by ezmlm$/im", true},
    TPcreUnit{"delivered_to", "/^mailing list \\S+\\@\\S+/im", true},
    TPcreUnit{"qmail_invoked_by", "/qmail \\d+ invoked by .{3,20}\\); \\d+ ... \\d+/im", true},
    TPcreUnit{"empty_field", "/\\S/mi", false},
    TPcreUnit{"from_groups_msn", "/from groups\\.msn\\.com \\(\\S+\\.msn\\.com /mi", true},
    // MessageIdNotUsable
    TPcreUnit{"list_unsubscribe", "/<mailto:(?:leave-\\S+|\\S+-unsubscribe)\\@\\S+>$/mi", true},
    TPcreUnit{"cwt_dce", "/\\/CWT\\/DCE\\)/mi", true},
    // GetReceivedHeaderTimes
    TPcreUnit{"localhost", "/\\bfrom (?:localhost\\s|(?:\\S+ ){1,2}\\S*\\b127\\.0\\.0\\.1\\b)/m", true},
    TPcreUnit{"invoked_by_uid", "/qmail \\d+ invoked by uid \\d+/m", true},
    TPcreUnit{"localhost_with", "/\\bby localhost with \\w+ \\(fetchmail-[\\d.]+/m", true},
    TPcreUnit{"received_date", "/(\\s.?\\d+ \\S\\S\\S \\d+ \\d+:\\d+:\\d+ \\S+)/m", true},
    //    TPcreUnit{ "received_date",      "/(\\s.?(?:\\d+ \\S\\S\\S|\\S\\S\\S \\d+) \\d+ \\d+:\\d+:\\d+(?: \\S+)?)/m", true },
    //CheckForFromToSame
    TPcreUnit{"from_to_same", "/^\\s*\\S+\\s*$/", true},
    TPcreUnit{"not_empty", "/\\S/", true},
    //CheckRecipients
    TPcreUnit{"recipient", "/([\\w.=-]+\\@\\w+(?:[\\w.-]+\\.)+\\w+)/m", true},
    //CheckForForgedReceivedTrail
    TPcreUnit{"by_received", "/\\bby[\\t ]+(\\w+(?:[\\w.-]+\\.)+\\w+)/si", true},
    TPcreUnit{"from_received", "/\\bfrom[\\t ]+(\\w+(?:[\\w.-]+\\.)+\\w+)/si", true},
    TPcreUnit{"ehlo_received", "/\\\"EHLO[\\t\\n ]+(\\w+(?:[\\w.-]+\\.)+\\w+)/si", true},
    TPcreUnit{"prepare_received", "/.*\\.(\\S+\\.\\S+)$/", true},
#ifdef SO_WITH_IPV6
    TPcreUnit{"ip_address", "/$IPV4_ADDRESS|$IPV6_ADDRESS/i", true},
#else
    TPcreUnit{"ip_address", "/$IP_ADDRESS/", true},
#endif
    TPcreUnit{"from_localhost", "/^localhost(?:\\.localdomain|)$/", true},
    //CheckForForgedHotmailReceivedHeaders
    TPcreUnit{"hotmail_smtpsvc", "/from mail pickup service by hotmail\\.com with Microsoft SMTPSVC;/s", true},
    TPcreUnit{"hotmail_1", "/from \\S*hotmail.com \\(\\S+\\.hotmail(?:\\.msn|)\\.com[ \\)]/s", true},
    TPcreUnit{"hotmail_2", "/from \\S+ by \\S+\\.hotmail(?:\\.msn|)\\.com with HTTP\\;/s", true},
    TPcreUnit{"hotmail_3", "/from \\[66\\.218.\\S+\\] by \\S+\\.hotmail\\.com/s", true},
    TPcreUnit{"hotmail_4", "/hotmail\\.com$/", true},
    //    TPcreUnit{ "hotmail_helo",       "/(?:^from |HELO[ =]*)\\s*hotmail\\.com\\b/i",                    true },
    TPcreUnit{"hotmail_from", "/hotmail.com/", true},
    //CheckForMsnGroupsHeaders
    //      TPcreUnit{ "x_loop",             "/^notifications\\@groups\\.msn\\.com/",    true },
    //    TPcreUnit{ "to_msn",             "/<(\\S+)\\@groups\\.msn\\.com>/i",         true },
    //    TPcreUnit{ "message_id_msn",     "/\\S+\\@groups\\.msn\\.com>/",             true },
    //    TPcreUnit{ "return_path_msn",    "/^bounce\\@groups\\.msn\\.com>/",          true },
    //CheckForForgedHotmailReceivedHeaders
    TPcreUnit{"rly", "/\\brly-[a-z][a-z]\\d\\d\\./i", true},
    TPcreUnit{"rly_aol", "/\\/AOL-\\d+\\.\\d+\\.\\d+\\)/", true},
    TPcreUnit{"rly_relay", "/ESMTP id (?:RELAY|MAILRELAY|MAILIN)/", true},
    //CheckForForgedEudoraMailReceivedHeaders
    TPcreUnit{"eudoramail", "/eudoramail.com/", true},
    TPcreUnit{"whowhere", "/by \\S*whowhere.com\\;/", true},
    //CheckForForgedYahooReceivedHeaders
    //    TPcreUnit{ "yahoo_com",           "/yahoo\\.com(\\.\\w\\w)?$/",      true }, // FromAddr
    TPcreUnit{"yahoo_com", "/yahoo\\.com$/", true}, // FromAddr
    TPcreUnit{"yahoo_web", "/by web\\S+\\.mail(?:\\.scd)?\\.yahoo\\.com via HTTP/", true},
    TPcreUnit{"yahoo_smtp", "/by smtp\\.\\S+\\.yahoo\\.com with SMTP/", true},
    TPcreUnit{"yahoo_ip", "/from \\[$REC_IP\\] by \\S+\\.(?:groups|grp\\.scd)\\.yahoo\\.com with NNFMP/i", true},
    TPcreUnit{"yahoo_mailer", "/\\bmailer\\d+\\.bulk\\.scd\\.yahoo\\.com\\b/", true},
    TPcreUnit{"yahoo_reply", "/\\@reply\\.yahoo\\.com$/", true}, // m_pFromAddr
    TPcreUnit{"yahoo_friend", "/by (?:\\w+\\.){2,3}yahoo\\.com \\(\\d+\\.\\d+\\.\\d+\\/\\d+\\.\\d+\\.\\d+\\) id \\w+/", true},
    //    TPcreUnit{ "yahoo_helo",             "/(?:^from |HELO[ =]*)\\s*yahoo\\.com\\b/i",                    true },
    TPcreUnit{"yahoo_4", "/yahoo\\.com/", true},
    //CheckForForgedJunoReceivedHeaders
    TPcreUnit{"juno_com", "/@juno.com/i", true},
    TPcreUnit{"juno_from", "/from.*\\bjuno\\.com.*[\\[\\(]$REC_IP[\\]\\)].* by /i", true},
    TPcreUnit{"juno_cookie", "/ cookie\\.juno\\.com /", true},
    TPcreUnit{"juno_mailer", "/Juno /", true},
    TPcreUnit{"juno_from_2", "/from.*\\bmail\\.com.*\\[$REC_IP\\].* by /i", true},
    TPcreUnit{"juno_mailer_2", "/\\bmail\\.com/", true},
    //CheckForForgedGw05ReceivedHeaders
    TPcreUnit{"gw05", "/\bfrom (\\S+) by (\\S+) with ESMTP\\; \\S{3}, \\d+ \\S{3} \\d{4} \\d\\d:\\d\\d:\\d\\d [-+]*\\d{4}\\b/s", true},
    TPcreUnit{"gw05_dot", "/\\./", true},
    //CheckMtaMessageId
    TPcreUnit{"mta_yahoo", "/\\@[a-z0-9.-]+\\.(?:yahoo|wanadoo)(?:\\.[a-z]{2,3}){1,2}>/", true},
    TPcreUnit{"mta_hostfrom", "/.*\\@(.+)$/", true},
    TPcreUnit{"mta_hostid", "/.*\\@(.+)[>\\s]+$/", true},
    TPcreUnit{"prepare_from", "/\\.?(\\S+)\\.\\S+$/", true},
    TPcreUnit{"prepare_messageid", "/.*\\.(\\S+)\\.\\S+$/", true},
    //CheckForRoundTheWorldReceived
    //    TPcreUnit{ "world_relays",     "/from\\b.{0,20}\\s(\\S+\\.${CCTLDS_WITH_LOTS_OF_OPEN_RELAYS})\\s\\(.{0,200}\\sfrom\\b.{0,20}\\s([-_A-Za-z0-9.]+)\\s.{0,30}\\[($REC_IP)\\]/i", true },
    //CheckReceivedHelos
    TPcreUnit{"helos_ip", "/\\[([\\d.]+)\\]\\)/", true},
    TPcreUnit{"helos_ip_reserved", "/$IP_IN_RESERVED_RANGE/", true},
    TPcreUnit{"helos_1", "/from ([-\\w.]+\\.[-\\w.]+) \\(\\S+ helo=([-\\w.]+)\\) by ([-\\w.]+)/", true},
    TPcreUnit{"helos_2", "/from ([-\\w.]+\\.[-\\w.]+) \\(HELO ([-\\w.]+)\\) \\(\\S+\\) by ([-\\w.]+)/", true},
    TPcreUnit{"helos_3", "/from ([-\\w.]+\\.[-\\w.]+) \\(\\[\\S+\\]\\).* by ([-\\w.]+)/", true},
    TPcreUnit{"helos_4", "/from ([-\\w.]+) \\(([-\\w.]+\\.[-\\w.]+).* by ([-\\w.]+)/", true},
    TPcreUnit{"helos_5", "/(?:\\.|^)(lycos\\.com|lycos\\.co\\.uk|hotmail\\.com|localhost\\.com|excite\\.com|caramail\\.com|cs\\.com|aol\\.com|msn\\.com|yahoo\\.com|drizzle\\.com)$/i", true},
    //GetClassificStat
    TPcreUnit{"with_pop3", "/ by .* with POP3/i", true},
    TPcreUnit{"rec_ip", "/[\\(\\[]($REC_IP)[\\)\\]] by /i", true},
    TPcreUnit{"rec_ip_2", "/($REC_IP).* by /i", true},
    //    TPcreUnit{ "rec_ip_imap",        "/by ($REC_IP) with IMAP/i",   true },
    TPcreUnit{"ip_unknown", "/[\\(\\[]$REC_IP[\\)\\]].*[\\(\\[]$REC_IP[\\)\\]].* by /i", true},
    TPcreUnit{"rec_ip_pure", "/($REC_IP)/i", true},
    TPcreUnit{"rec_rdns_clear", "/($REC_MAILADDR)/", true},
    TPcreUnit{"rec_rdns", "/(?:^|\\s|\\()($REC_MAILADDR).*\\[.+\\].* by /", true},
    TPcreUnit{"rec_rdns_2", "/($REC_RDNS).* by /", true},
    TPcreUnit{"rec_rdns_3", "/($REC_RDNS).*($REC_RDNS).* by /", true},
    TPcreUnit{"not_resolved", "/$REC_RDNS$/", true},
    TPcreUnit{"rec_helo", "/(?:HELO|[EL]HLO)[- =\":<\\(\\[]*([^ \\)\\[\"\\]]*)(?:[ \\)\\]\"]|$)/i", true},
    // zmailer exim
    TPcreUnit{"rec_zmailer1", "/^(\\S*)\\s*\\(\\s*\\[($REC_IP)\\](?:\\:\\d+)?[\" ]*(?:HELO|EHLO)[- =\":<\\(\\[]*([^ \">\\)\\]]*).*\\)\\s* by /i", true},
    TPcreUnit{"rec_zmailer", "/^(\\S*)\\s*\\(\\s*\\[($REC_IP)\\](?:\\:\\d+)?(.*)\\)\\s* by /i", true},
    TPcreUnit{"zmailer_ip", "/\\(\\s*\\[($REC_IP)\\](?:\\:\\d+)?.*\\)\\s* by /i", true},
    TPcreUnit{"zmailer_dns", "/^([^\\s\\(]+)\\s*\\)/", true},
    TPcreUnit{"zmailer_helo", "/(?:HELO|EHLO)[=\\s]+([-.\\w)]+)/", true},
    // sendmil Postfix
    TPcreUnit{"rec_sendmil", "/^(\\S*)\\s*\\(\\s*(\\S*)\\s*\\[($REC_IP)\\](?:\\:\\d+)?\\s*\\)\\s* by/i", true},
    TPcreUnit{"rec_exim", "/Exim/i", true},
    TPcreUnit{"rec_exim1", "/^\\[($REC_IP)\\](?:\\:\\d+)? \\(([^)]*?)\\) by/i", true},
    TPcreUnit{"rec_exim2", "/^(\\S+) \\(\\[($REC_IP)\\](?:\\:\\d+)? helo=(\\S+)(?: .*\\)(?: ident=[^ )])?) by/i", true},
    TPcreUnit{"rec_exim3", "/^(\\S+) \\(?\\[($REC_IP)\\](?:\\:\\d+)?\\)? by/i", true},
    TPcreUnit{"rec_exim4", "/^(\\S+) \\(\\[($REC_IP)\\](?:\\:\\d+)? ident=(\\S+) .*\\) by/i", true},
    TPcreUnit{"rec_exim5", "/^[^ .] + by /", true},
    TPcreUnit{"rec_meims", "/^(\\S+) \\((\\S+) \\[($REC_IP)\\](?:\\:\\d+)?\\) by \\S+ with \\S+ \\(/i", true},
    TPcreUnit{"rec_11", "/^(\\S+) \\(\\[($REC_IP)\\](?:\\:\\d+)? ((?:HE|EH)LO.*)\\) by \\S+ with/i", true},
    TPcreUnit{"rec_12", "/^(\\S+) \\(<unknown\\S*>\\[($REC_IP)\\](?:\\:\\d+)?\\) by/i", true},
    TPcreUnit{"rec_13", "/^(\\S+) \\((\\S+\\.\\S+)\\[($REC_IP)\\](?:\\:\\d+)?\\) by/i", true},

    TPcreUnit{"rec_sendmail", "/^(\\S+) \\((\\S+) \\[($REC_IP)\\](?:\\:\\d+)?.*\\) by \\S+ \\(/i", true},
    TPcreUnit{"rec_sendmail-8-14", "/^(\\S+) \\((?:(\\S+) )?\\[($REC_IP)\\](?:\\:\\d+)?.*\\) by \\S+ \\(/i", true},

    TPcreUnit{"rec_postfix", "/ \\(Postfix\\) with/i", true},
    TPcreUnit{"rec_postfix1", "/^(\\S+) \\((\\S+) ?\\[($REC_IP)\\](?:\\:\\d+)?\\) by/i", true},
    TPcreUnit{"rec_msn", "/^($REC_IP) .*by (\\S+\\.hotmail\\.msn\\.com) /i", true},
    TPcreUnit{"rec_via", "/^(\\S+) \\(\\[($REC_IP)\\](?:\\:\\d+)?\\) by \\S+ via smtpd \\(for \\S+\\) with SMTP\\(/i", true},
    TPcreUnit{"rec_14", "/^(\\S+) \\( \\[($REC_IP)\\](?:\\:\\d+)?\\).*? by/i", true},
    TPcreUnit{"rec_15", "/^(\\S+) \\( \\[($REC_IP)\\](?:\\:\\d+)?\\).*? by/i", true},
    TPcreUnit{"rec_16", "/^\\[($REC_IP)\\](?:\\:\\d+)? by/i", true},
    TPcreUnit{"rec_17", "/^(\\S+) \\[($REC_IP)\\](?:\\:\\d+)? by (\\S+) \\[(\\S+)\\] with /i", true},
    TPcreUnit{"rec_18", "/^(\\S+) \\(\\[($REC_IP)\\](?:\\:\\d+)?\\) by \\S+ \\(/i", true},
    TPcreUnit{"rec_19", "/^(\\S+) \\((\\S+) \\[($REC_IP)\\](?:\\:\\d+)?\\).*? by/i", true},
    TPcreUnit{"rec_20", "/^(\\S+) \\(\\[($REC_IP)\\](?:\\:\\d+)?\\).*? by/i", true},
    TPcreUnit{"rec_21", "/^(\\S+) \\((\\S+) \\[($REC_IP)\\](?:\\:\\d+)?\\)(?: \\(authenticated bits=\\d+\\)|) by \\S+ \\(/i", true},
    TPcreUnit{"rec_22", "/^(\\S+) \\[($REC_IP)\\](?:\\:\\d+)? by \\S+ with IMAP \\(fetchmail/i", true},
    TPcreUnit{"rec_24", "/^(\\S+) \\[($REC_IP)\\](?:\\:\\d+)? by \\S+ with POP3 /i", true},
    TPcreUnit{"rec_25", "/^(\\S+)\\(($REC_IP)\\) by \\S+ via smap /i", true},
    TPcreUnit{"rec_27", "/^(\\S+) \\(\\[($REC_IP)\\](?:\\:\\d+)?\\) by \\S+ with /i", true},
    TPcreUnit{"rec_28", "/^($REC_IP) by/i", true},
    TPcreUnit{"rec_29", "/^(\\S+) \\((\\S+) \\[($REC_IP)\\](?:\\:\\d+)?\\) by \\S+ with/i", true},
    TPcreUnit{"rec_30", "/^(\\S+) \\(HELO (\\S+)\\) \\(($REC_IP).*?\\) by \\S+ with /i", true},
    TPcreUnit{"rec_31", "/^(\\S+) \\((\\S+)\\[($REC_IP)\\](?:\\:\\d+)?\\) by/i", true},
    TPcreUnit{"rec_32", "/^(\\S+) \\((?:\\S+\\@)?($REC_IP)(?:\\:\\d+)?\\) by/i", true},

    TPcreUnit{"rec_qmail", "/^\\S+( \\(HELO \\S+\\))? \\((\\S+\\@)?\\[?$REC_IP\\]?\\)( \\(envelope-sender <\\S+>\\))? by \\S+( \\(.+\\))* with (.* )?(SMTP|QMQP)/i", true},
    TPcreUnit{"rec_qmail1", "/^(\\S+) \\(HELO (\\S+)\\) \\((\\S+)\\@\\[?($REC_IP)\\]?\\)( \\(envelope-sender <\\S+>\\))? by/i", true},
    TPcreUnit{"rec_qmail2", "/^(\\S+) \\(HELO (\\S+)\\) \\(\\[?($REC_IP)\\]?\\)( \\(envelope-sender <\\S+>\\))? by/i", true},
    TPcreUnit{"rec_qmail3", "/^(\\S+) \\((\\S+)\\@\\[?($REC_IP)\\]?\\)( \\(envelope-sender <\\S+>\\))? by/i", true},
    TPcreUnit{"rec_qmail4", "/^(\\S+) \\(\\[?($REC_IP)\\]?\\)( \\(envelope-sender <\\S+>\\))? by/i", true},

    //    TPcreUnit{ "rec_ecelerity",      "/^(\\S+) \\(\\[($REC_IP):\\d+\\]\\) by .+\\(ecelerity/i", true },
    TPcreUnit{"rec_ecelerity", "/^(\\S+) \\(\\[($REC_IP):\\d+\\](?: helo=(\\S+))?\\) by .+\\(ecelerity/i", true},

    TPcreUnit{"get_by_host", "/ by ([^ ;]+)/i", true},
    TPcreUnit{"spf_domain", "/domain of \\s*([a-zA-Z0-9._-]+)/", true},

    //Delivery
    TPcreUnit{"get_sender", R"(/\@([-\w.]*[-\w]+\.[-\w]+)(?:[^-\w.][^@]*$|$)/i)", true},
    TPcreUnit{"get_domen2", "/(?:^|(\\.))([-\\w]+\\.[-\\w]+)$/i", true},
    TPcreUnit{"get_domen3", "/(?:^|(\\.))([-\\w]+\\.[-\\w]+\\.[-\\w]+)$/i", true},
    TPcreUnit{"get_domen4", "/(?:^|(\\.))([-\\w]+\\.[-\\w]+\\.[-\\w]+\\.[-\\w]+)$/i", true},
    TPcreUnit{"geo_domain", "/^(arpa|asia|coop|name|biz|cat|co|com|net|info|gov|edu|org)\\.[a-z]+$/i", true},
    TPcreUnit{"get_sender_rec", "/^[-\\w]+\\.(.+)$/i", true},
    TPcreUnit{"get_subj_sender_mail", "/[-\\w]*\\@\\S+\\.[a-z][a-z]+/i", true},
    TPcreUnit{"get_subj_sender_host", "/(?:[-\\w]+\\.)+[a-z][a-z]+/i", true},
    TPcreUnit{"get_spf_domain", "/spf=pass \\(\\S*:\\s*domain\\s*of\\s*(\\S*)\\s*designates/i", true},
    TPcreUnit{"get_dkim_domain", "/dkim=pass\\s*header.i=\\S*@(\\S*)/i", true},
    TPcreUnit{"ip_local", "/^(?:10|127|169\\.254|172\\.(?:1[6-9]|2[0-9]|30|31)|192\\.168|22[4-9]|23[0-9])\\./", true}
);

static const auto rgMacroSubstRe = MakeTrueConstArray(
    TPcreUnit{"$IPV4_ADDRESS", R"((?:^|[^\d])(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})(?:$|[^\d]))", true},
    TPcreUnit{"$IPV6_ADDRESS", "(?:[0-9a-f.ipv]*:){2}[0-9a-f:.]*(?:%[0-9a-z]+)?", true},
    TPcreUnit{"$REC_IP", R"(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}|(?:[0-9a-f.ipv]*:){2}[0-9a-f:.]*(?:%[0-9a-z]+)?)", true},
    TPcreUnit{"$IP_IN_RESERVED_RANGE", R"(^(?:192\.168|10|172\.(?:1[6-9]|2[0-9]|3[01])|169\.254|127|00[01257]|02[37]|03[179]|04[12]|05[89]|060|07[0-9]|08[2-9]|09[0-9]|1[01][0-9]|12[0-6]|197|22[23]|24[0-9]|25[0-5])\.)", true},
    TPcreUnit{"$REC_RDNS", "[-.\\w]+\\.[a-zA-Z]+", true},
    TPcreUnit{"$REC_MAILADDR", R"([-.\w\@]+\.[a-zA-Z]+)", true}
);

static const TTrueConst<PcreTool> m_pcre{NRegexp::TSettings{}, MakeArrayRef(*AlgRe), MakeArrayRef(*rgMacroSubstRe)};

TSpAlg::TSpAlg(const TRulesHolder& rulesHolder, const TSoConfig& spTop, TSpStat& pstat, const ip_match& local_matcher, TCurMessageEngine* curMessageContext)
        : m_pstat(pstat)
        , m_rulesHolder(rulesHolder)
        , m_curMessageContext(curMessageContext)
        , local_matcher(local_matcher) {

    is_spk = m_rulesHolder.IsSPK();

    m_fWebMail = spTop.fWebMail;

    m_mapSender.reserve(MAX_SENDERS + 1);
}

void TSpAlg::SetField(TSpFields fid, const TStringBuf& field, int FieldNumber, bool fCancelSaveListField, TRengine& rengine) {
    if (m_rulesHolder.GetFieldsSave()[fid]) {
        switch (fid) {
            case FD_LIST_:
            case FD_LIST_POST:
            case FD_LIST_OWNER:
            case FD_LIST_SUBSCRIBE:
                if (!fCancelSaveListField && m_rgpFields[fid].empty())
                    m_rgpFields[fid] = field;
                break;
            case FD_DELIVERED_TO:
                if (m_rgpFields[fid].empty())
                    m_rgpFields[fid] = field;
                break;
            case FD_FROM_ADDR:
                m_rgpFields[fid] = ToLowerUTF8(field);
                break;
            default:
                m_rgpFields[fid] = field;
                break;
        }
    }

    m_faddprint = true;

    m_curMessageContext->fieldsById[fid].ConstructInPlace(field);

    switch (fid) {
        case FD_FROM:
            if (m_cur.cFrom++ > 3)
                return;
            m_pstat.AddStat(FD_FROM) << SafeRecode(field);
            break;
        case FD_FROM_ADDR: {
            m_curMessageContext->DlvLogRequest.FromAddr = SafeRecode(field);
            AddStat(fid, m_curMessageContext->DlvLogRequest.FromAddr);
            AddSender(fid, field, 3000);
            rengine.PrepareFromAddrDataForVw(TString(field));
            CheckFromSender(rengine);

            auto domain = GetDomain(field);

            if(domain.empty()) {
                domain = field;
            } else {
                rengine.Add2UrlReputationList(domain, EUrlStaticticSource::SENDER);
            }
            rengine.m_cur->MlLogBuilder.Add(ToString(FD_FROM), SafeRecode(field));
            break;
        }
        case FD_FROM_NAME:
            AddStat(fid, SafeRecode(field));
            rengine.PrepareFromNameDataForVw(TString(field));
            break;
        case FD_TO:
            if (m_cur.cTo++ > 10)
                return;
            AddAddrToCc(rengine, field);
            AddStat(fid, SafeRecode(field));
            break;
        case FD_SUBJECT:
            if (m_cur.cSubject++ > 3) {
                SetRule("STRANGE_FIELD_REPEAT");
                return;
            }

            if (CheckSubjectId(field))
                SetRule("SUBJ_HAS_UNIQ_ID");

            if (field) // = 4)    do not check subj len anymore
            {
                rengine.AddPattern(field, EN_SH_SUBJ);
                if (field.size() > 8)
                    rengine.AddPattern({field.data(), 8}, EN_SH_SUBJ8);
                if (field.size() > 16)
                    rengine.AddPattern({field.data(), 16}, EN_SH_SUBJ16);
            }

            {
                auto utf8Subj = SafeRecode(m_rgpFields[FD_SUBJECT]);
                m_pstat.AddStat(ST_SUBJ) << utf8Subj;
                rengine.m_cur->MlLogBuilder.Add(ToString(ST_SUBJ), std::move(utf8Subj));
            }
            rengine.PrepareSubjDataForVw(m_rgpFields[FD_SUBJECT]);
            CheckSubject();
            break;
        case FD_RAWSUBJECT:
            break;
        case FD_MESSAGEID:
            m_cur.fMessageId = true;
            if (FieldNumber > 1 && m_cur.iReceived == FieldNumber - 1)
                m_cur.mta_check = true;
            //            m_pstat.AddStat(ST_MESSID, m_pMessageid);
            //            m_pstat.AddStat(ST_MESSID, m_rgpFields[FD_MESSAGEID]);
            CheckMessageId(field);
            AddSender(fid, field, 50);
            m_curMessageContext->DlvLogRequest.MessageId = field;
            AddStat(fid, field);
            if (field.size()) {
                rengine.m_cur->MlLogBuilder.Add(ToString(ST_MESSID), TString{field});
            }
            break;
        case FD_RECEIVED: {
            m_cur.iReceived = FieldNumber;
            if (m_cur.fMessageId)
                m_cur.fLaterMta = true;
            TString pReceived = TString{field};
            DelRepeatedSpaces(pReceived);
            if (pReceived) {
                rengine.CheckFieldAllRules(fid, pReceived);
                m_vReceived.push_back(pReceived);
                m_sReceived.append("  ");
                m_sReceived.append(pReceived);
            }
        } break;
        case FD_MAILING_LIST:
            m_cur.fMailingList = !!m_pcre->Check("run_by_ezmlm", field);
            break;
        case FD_LIST_UNSUBSCRIBE:
            // Lyris eats message-ids.
            if (!m_cur.fMessageIdNotUsable)
                m_cur.fMessageIdNotUsable = !!m_pcre->Check("list_unsubscribe", field);
            break;
        case FD_LIST_ID:
            break;
        case FD_REFERENCES:
            break;
        case FD_CC:
            AddAddrToCc(rengine, field);
            break;
        case FD_X_ORIGINATING_IP:
            m_cur.fXOriginatingIpExists = true;
            m_cur.fXOriginatingIp = !!m_pcre->Check("ip_address", field);
            AddStat(fid, field);
            break;
        case FD_X_SENDER_IP:
            m_cur.fXSenderIp = !!m_pcre->Check("ip_address", field);
            break;
        case FD_X_MAILER:
        case FD_USER_AGENT: {
            TString recoded = SafeRecode(field);
            m_pstat.AddStat(ST_XMAILER) << recoded;
            rengine.m_cur->MlLogBuilder.Add(ToString(ST_XMAILER), std::move(recoded));
            if (field.size())
                rengine.AddPattern(field, EN_SH_XMLR);
            break;
        }
        case FD_CONTENT_TYPE: {
            if (field.StartsWith("text")) {
                m_cur.fTextType = true;
                if (field.substr(4).StartsWith("/html"))
                    SetRule("__MIME_HTML_ONLY");
            }
        } break;
        case FD_CONTENT_TRANSFER_ENCODING:
            m_cur.fBase64 = field.Contains("base64");
            break;
        case FD_PRECEDENCE:
            m_cur.fDelivery = true;
            break;
        case FD_RETURN_PATH:
        case FD_RESENT_FROM:
        case FD_RESENT_SENDER:
            AddStat(fid, field);
            break;
        case FD_DELIVERED_TO:
            //            AddSender(fid, field, fieldlen, 1000);
            m_cur.fDeliveredTo = !!m_pcre->Check("delivered_to", field);
            m_cur.fDelivery = true;
            break;
        case FD_X_ORIGINAL_TO:
            AddSender(fid, field, 900);
            break;
        case FD_SENDER: {
            AddSender(fid, field, 800);

            const auto& domain = GetDomain(field);

            if (!domain.empty())
                rengine.Add2UrlReputationList(domain, EUrlStaticticSource::SENDER);

            break;
        }
        case FD_X_SENDER:
            AddSender(fid, field, 700);
            break;
        case FD_LIST_:
            if (fCancelSaveListField) {
                m_sListAll.append(",");
                m_sListAll.append(field);
            }
            m_cur.fDelivery = true;
            AddSender(fid, field, 600);
            break;
        case FD_REPLY_TO_ADDR: {
            if (field) {
                rengine.AddPattern(field, EN_SH_REPLYTO_ADDR);
                rengine.m_cur->MlLogBuilder.Add(ToString(FD_REPLY_TO), TString{field});
            }

            const auto& domain = GetDomain(field);
            if (domain)
                rengine.Add2UrlReputationList(domain, EUrlStaticticSource::SENDER);

            break;
        }
        case FD_RECEIVED_SPF:
            if (m_cur.cReseivedSpf) {
                SetRule("RESEIVED_SPF_MORE_1");
            } else {
                m_cur.cReseivedSpf = true;
            }
            AddStat(fid, field);
            break;
        case FD_X_BORNDATE:
            if (m_fWebMail)
                CheckBornDate(field);
            break;
        case FD_RCPT_BORN_DATE: {
            if (time_t born_t{}; m_fWebMail && TryIntFromString<10>(field, born_t)) {
                const auto instant = TInstant::Seconds(born_t);
                const auto days = (Now() - instant).Days();
                m_rulesHolder.m_spruler->CheckDateRange(*m_curMessageContext, "rcpt_bb_age", instant.TimeT(), false);
                m_pstat.AddStat(FD_RCPT_BORN_DATE) << "field:" << field << " " << instant << ", days:" << days;
            }
            break;
        }
        case FD_X_MSGDAYCOUNT:
            m_cur.cMsgDayCount = CheckDigitField("MSG_COUNT", field);
            if (m_cur.cMsgDayCount)
                AddStat(fid, field);
            break;
        case FD_X_YANDEX_KARMA:
            m_cur.karma = CheckDigitField("YANDEX_KARMA", field);
            AddStat(fid, field);
            if (m_fWebMail)
                rengine.m_cur->MlLogBuilder.Add("karma", TString{field});
            break;
        case FD_X_YANDEX_KARMA_STATUS:
            m_cur.karma_status = CheckDigitField("YANDEX_KARMA_STATUS", field);
            AddStat(fid, field);
            break;
        case FD_X_ORIGINAL_SIZE:
            CheckDigitField("original_message_size", field);
            AddStat(fid, field);
            rengine.m_cur->MlLogBuilder.Add(ToString(FD_X_ORIGINAL_SIZE), TString{field});
            break;
        case FD_DATE: {
            AddStat(fid, field);
            mimepp::DateTime mesDate({field.data(), field.size()});
            mesDate.parse();
            rengine.m_cur->MlLogBuilder.Add("date", ToString(mesDate.asUnixTime()));
            break;
        }
        case FD_X_YANDEX_RPOP_FOLDER:
            AddStat(fid, SafeRecode(field));
            rengine.m_cur->MlLogBuilder.Add(ToString(FD_X_YANDEX_RPOP_FOLDER), SafeRecode(TString{field}));
            break;
        case FD_X_YANDEX_SO_FRONT:
            m_curMessageContext->m_sSoFront = TString{field};
            AddStat(fid, field);
            m_curMessageContext->DlvLogRequest.Mx = field;
            break;
        case FD_X_YANDEX_FRONT:
        case FD_X_YANDEX_SPAM:
        case FD_X_YAMAIL_AUTO_REPLY:
        case FD_X_SKIPPED_RECEIVED:
        case FD_X_SKIPPED_SPF:
        case FD_X_DMARC:
        case FD_X_YANDEX_FOLDERNAME:
        case FD_AUTHENTICATION_RESULTS:
        case FD_IY_SPLIT_ID:
        case FD_X_YANDEX_SOURCE:
            AddStat(fid, field);
            break;
        case FD_IY_DAYS_SINCE_RCVD:
            CheckDigitField("email_age", field);
            AddStat(fid, field);
            break;
        case FD_X_YANDEX_CAPTCHA:
            if (m_fWebMail && AsciiCompareIgnoreCase(StripString(field), "bad")) {
                m_curMessageContext->rulesContext.SetRule("SKIP_PUT");
                m_curMessageContext->Logger << TLOG_WARNING << "CAPTCHA disabled PUTs";
            }
            AddStat(fid, field);
            break;
        case FD_X_YANDEX_QUEUEID:
            m_curMessageContext->DlvLogRequest.QueueId = field;
            AddStat(fid, field);
            if (!m_fWebMail)
                rengine.m_cur->MlLogBuilder.Add(ToString(FD_X_YANDEX_QUEUEID), TString{field});
            break;
        case FD_X_YANDEX_POP_SERVER:
            AddStat(fid, field);
            rengine.m_cur->MlLogBuilder.Add(ToString(FD_X_YANDEX_POP_SERVER), TString{field});
            break;
        case FD_X_YANDEX_RPOP_INFO:
            m_cur.fMailGmailIMAP = (AsciiCompareIgnoreCase(field, "@imap.mail.ru") == 0) || (AsciiCompareIgnoreCase(field, "@imap.gmail.com") == 0);
            break;
        case FD_IY_GEOZONE:
            if (field.size() >= 2)
                m_rulesHolder.m_pListRuler.CheckWord(m_curMessageContext->rulesContext, field.Before(' '), SP_LIST_GEO_ZONE);
            AddStat(fid, field);
            rengine.m_cur->MlLogBuilder.Add(ToString(fid), TString{field});
            break;
        case FD_IY_FIRSTGEOZONE:
            if (field.size() >= 2)
                m_rulesHolder.m_pListRuler.CheckWord(m_curMessageContext->rulesContext, field.Before(' '), SP_LIST_FIRST_GEO);
            AddStat(fid, field);
            break;
        case FD_IY_AS:
            AddStat(fid, field);
            if (!field.empty())
                rengine.AddPattern(field, EN_SH_AS);
            break;
        case FD_IY_TOR:
            AddStat(fid, field);
            SetRule("TOR");
            break;
        case FD_X_SKIPPED_SO_FORWARD:
            AddStat(fid, field);
            SetForwardHost(field, en_MTA_SENDMAIL);
            break;
        case FD_X_YANDEX_SO_GREYLISTING:
            SetRule("Y_GREYLISTING_NEW");
        case FD_X_YANDEX_GREYLISTING:
            SetRule("Y_GREYLISTING");
            AddStat(fid, field);
            break;
        case FD_X_YANDEX_SUID:
            if (!field.empty())
                rengine.SetPopperSuid(field);
            AddStat(fid, field);
            break;
        case FD_IY_LANGUAGE:
            m_pstat.AddStat(ST_LANGUAGE) << field;
            break;
        case FD_IY_LOGTEMP:
            m_pstat.AddStat(ST_SOLOGTEMP) << field;
            break;
        //case FD_X_MAILRU_TYPE:
        //    AddStat(fid, field);
        //    break;
        case FD_X_YANDEX_AUTHENTICATION_RESULTS:
            AddStat(fid, field);
            if (!m_fWebMail)
                rengine.m_cur->MlLogBuilder.Add(ToString(FD_X_YANDEX_AUTHENTICATION_RESULTS), TString{field});
            break;
        case FD_IN_REPLY_TO:
            if (m_fWebMail)
                rengine.m_cur->MlLogBuilder.Add(ToString(FD_IN_REPLY_TO), TString{field});
            break;
        case FD_x_yandex_mailish:
            AddStat(fid, field);
            if(IsNumber(StripString(field)))
                m_curMessageContext->IsMailish = true;
            break;
        default:
            break;
    }

    if (m_faddprint && (fid < FD_X_IP) &&
        fid != FD_RECEIVED &&
        fid != FD_FROM && fid != FD_SUBJECT &&
        fid != FD_TO && fid != FD_TO_ADDR &&
        fid != FD_CC && fid != FD_CC_ADDR &&
        fid != FD_REPLY_TO && fid != FD_REPLY_TO_ADDR &&
        fid != FD_MESSAGE_ID &&
        m_pcre->Check("get_sender", field))
        AddStat(fid, field);
}

static const auto m_maptriplets = MakeTrueConst([]{
    constexpr size_t count = Y_ARRAY_SIZE(SpEngTrigr);

    THashMap<TString, i32> mapTriplets(count);

    for(size_t i = 0; i < count; i++)
        mapTriplets.emplace_noresize(SpEngTrigr[i], i);

    return mapTriplets;
}());

static const auto m_tabl = MakeTrueConst([]() noexcept {
    std::array<TSpSymbols, 256> tabl{};
    tabl[0x00] = SP_NULL;
    tabl[0x09] = SP_SPACE;
    tabl[0x0A] = SP_SPACE;
    tabl[0x0B] = SP_SPACE;
    tabl[0x0D] = SP_SPACE;
    tabl[0x20] = SP_SPACE;
    return tabl;
}());

static const auto m_SpSenderTypeStr = MakeTrueConstArray(
        "en_SR_UNDEF",
        "en_SR_HOST",
        "en_SR_MESSID",
        "en_SR_FROM",
        "en_SR_REPLYTO",
        "en_SR_SENDER",
        "en_SR_LIST",
        "en_SR_SUBJ",
        "en_SR_RBL",
        "en_SR_DSL",
        "en_SR_NRV",
        "en_SR_BL_NOT",
        "en_SR_L2ASL1",
        "en_SR_MAX"
);

void TSpAlg::CheckMessage(TRengine& rengine) {
    if (m_cur.fBase64 && m_cur.fTextType)
        SetRule("BASE64_ENC_TEXT");

    GetHeaderTime();

    if (CheckOutlookTimeStamp())
        SetRule("MSGID_OUTLOOK_TIME");

    m_cur.fGatedThrough = GatedThroughReceivedHdrRemover();
    //    #ifndef SP_CHECK_RULES // for first message only
    if (rengine.GetCountHeaders() < 2)
        CheckForShiftedDate();
    //    #endif
    if (CheckForFromToSame())
        SetRule("FROM_AND_TO_SAME");

    if (CheckMessageIdNotUsable())
        SetRule("__UNUSABLE_MSGID");

    int mismatch_addr;
    if (CheckForForgedReceivedTrail(&mismatch_addr))
        SetRule("FORGED_RCVD_TRAIL");
    else if (mismatch_addr > 1)
        SetRule("FORGED_RCVD_TRAIL_2");

    CheckForForgedHotmailReceivedHeaders();
    //    if (m_cur.fHotmailAddrWithForgedHotmailReceived)
    //        SetRule("FORGED_HOTMAIL_RCVD");
    if (m_cur.fHotmailAddrButNoHotmailReceived)
        SetRule("SEMIFORGED_HOTMAIL_RCVD");

    if (CheckForFakeAolRelayInRcvd())
        SetRule("FORGED_AOL_RCVD");
    if (CheckForForgedEudoraMailReceivedHeaders())
        SetRule("FORGED_EUDORAMAIL_RCVD");
    if (CheckForForgedYahooReceivedHeaders())
        SetRule("FORGED_YAHOO_RCVD");
    if (CheckForForgedJunoReceivedHeaders())
        SetRule("FORGED_JUNO_RCVD");
    if (CheckForForgedGw05ReceivedHeaders())
        SetRule("FORGED_GW05_RCVD");

    if (CheckReceivedHelos())
        SetRule("RCVD_FAKE_HELO_DOTCOM_2");

    CheckForToInSubject();
    if (m_cur.mta_check || m_cur.fLaterMta)
        CheckMtaMessageId();

    if (!m_cur.fCheckDelivery)
        DomenPrepare(rengine);

    CheckSender();
    CheckReceivedNum();
    WriteDeliveryStat(false);

    if (m_rgpFields[FD_FROM_NAME].Size() > 0) // 2)    // do not check field length anymore
        rengine.AddPattern(m_rgpFields[FD_FROM_NAME], EN_SH_FROM_NAME);
    if (m_rgpFields[FD_FROM_ADDR].Size() > 0) // 2)
        rengine.AddPattern(m_rgpFields[FD_FROM_ADDR], EN_SH_FROM_ADDR, true, false, rengine.GetRcptCount());

    if (    !stricmp(m_rgpFields[FD_TO_ADDR].c_str(), m_rgpFields[FD_FROM_ADDR].c_str()) &&
            m_rgpFields[FD_FROM_ADDR].Size() > 5 && strchr(m_rgpFields[FD_FROM_ADDR].c_str(), '@'))
        SetRule("FROM_TO_ADDR_SAME");

    const char* patfrom = strchr(m_rgpFields[FD_FROM_ADDR].c_str(), '@');
    const char* patto = strchr(m_rgpFields[FD_TO_ADDR].c_str(), '@');

    if (patfrom && patto && !strcmp(patfrom, patto))
        SetRule("FROM_TO_DOMAIN_SAME");

    if (CheckAddrInReceived())
        m_pstat.AddStat(ST_LOG) << "may be forward";

    if (m_cur.cTo > 10)
        m_pstat.AddStat(ST_LOG) << " too many fields To: " << m_cur.cTo;

}

void TSpAlg::CheckBornDate(const TStringBuf& field) {
    if (field.empty())
        return;

    time_t born_t = {};
    if (TryIntFromString<10>(field, born_t) && born_t) {
        m_cur.tBornDate = born_t;
        m_rulesHolder.m_spruler->CheckDateRange(*m_curMessageContext, "LOGIN_BORN_DATE", born_t, false);
        m_pstat.AddStat(ST_LOG) << "CheckBornDate:  " << field << ", " << ctime(&born_t);
    }
}

int TSpAlg::CheckDigitField(const char* fieldkey, const TStringBuf& field) {
    if (field.empty())
        return 0;

    int count = 0;
    TryIntFromString<10>(field, count);

    m_rulesHolder.m_spruler->CheckRange(*m_curMessageContext, fieldkey, count);
    return count;
}

void TSpAlg::CheckReceivedNum() {
    const std::vector<TReceivedNum>* pvReceivedNum = m_rulesHolder.GetReceivedNum();

    for (const auto& it : *pvReceivedNum) {
        auto& ruleDef = it.ruleDef;
        int ind = 1;

        if (it.Direction == '+') {
            for (auto itRec = m_vReceived.begin(); itRec != m_vReceived.end(); itRec++, ind++) {
                if (ind < it.iFirst)
                    continue;
                if (ind > it.iLast)
                    break;
                if(std::get<TReDef>(ruleDef.rules).CheckRegExp(*itRec))
                    m_curMessageContext->rulesContext.WorkedRule(it.rid);
            }
        } else {
            for (auto revitRec = m_vReceived.rbegin(); revitRec != m_vReceived.rend(); revitRec++, ind++) {
                if (ind < it.iFirst)
                    continue;
                if (ind > it.iLast)
                    break;
                if(std::get<TReDef>(ruleDef.rules).CheckRegExp(*revitRec))
                    m_curMessageContext->rulesContext.WorkedRule(it.rid);
            }
        }
    }
}

void TSpAlg::CheckFromSender(TRengine& rengine) {
    if (m_rgpFields[FD_FROM_ADDR].Empty())
        return;

    const TStringBuf pfrom_addr = m_rgpFields[FD_FROM_ADDR];

    m_rulesHolder.m_pListRuler.CheckWord(m_curMessageContext->rulesContext, pfrom_addr, SP_LIST_FROM);

    const size_t at = pfrom_addr.find('@');
    if (at != TString::npos) {
        {
            const TStringBuf& domain = pfrom_addr.substr(at + 1);
            m_rulesHolder.m_pListRuler.CheckDomainName(m_curMessageContext->rulesContext, domain, SP_LIST_FROM_DOMAIN,
                                                       SP_LIST_FROM_DOMAIN_PART);
            m_rulesHolder.m_pListRuler.CheckDomainName(m_curMessageContext->rulesContext, domain, SP_LIST_DOMAIN, SP_LIST_DOMAIN_PART);
            if (!m_fWebMail) // 38th shingle from domains for incoming
            {
                //            Syslog (TLOG_WARNING, "38th for %s", p_at + 1);
                rengine.AddPattern(domain, EN_SH_FROM_DOMAIN, true, false, rengine.GetRcptCount());
            }
            m_rgpFields[FD_FROM_DOMAIN] = TString{domain};
        }
        rengine.CheckRange("from_addr_len", (int)at);
    }
}

bool TSpAlg::IsSpace(ui8 c) const {
    return (m_tabl[c] == SP_SPACE);
}

bool TSpAlg::IsSpace(char p) const {
    return (m_tabl[(ui8)p] == SP_SPACE);
}

bool TSpAlg::CheckSubjectId(const TStringBuf& subject) {
    const auto psubj = to_lower(TString{subject});
    {
        TStringBuf pUrl;
        if(auto pUrl = m_pcre->GetPattern("get_subj_sender_mail", psubj, 0)) {
            AddSender(FD_SUBJECT, pUrl, 100);
        } else if (auto pUrl = m_pcre->GetPattern("get_subj_sender_host", psubj, 0)) {
            TString sSubject;
            sSubject.assign("0@");
            sSubject.append(pUrl);
            AddSender(FD_SUBJECT, sSubject, 100);
        }

        if (pUrl) {
            m_pstat.AddStat(ST_LOG) << "subject sender: " << pUrl;
        }
    }

    return CheckSubjectIdLow(psubj);
}

bool TSpAlg::CheckSubjectIdLow(const TStringBuf& subject) {
    TMaybe<NRegexp::TResult> res;
    for(TStringBuf pat : {
            "id_1", "id_2", "id_3", "id_4", "id_5", "id_6", "id_7", "id_8", "id_9", "id_10", "id_11", "id_12"
    })
        if(res = m_pcre->Check(pat, subject)) {
            break;
        }

    if (!res)
        return false;

    const TStringBuf pattern = res->GetPattern(1);

    if (pattern.empty())
        return false;

    if (m_pcre->Check("id_test0", {subject, size_t(pattern.data() - subject.data())}))
        return true;

    if (m_pcre->Check("id_test7", {subject, size_t(pattern.data() - subject.data())}))
        return false;

    if (auto res = m_pcre->Check("id_test1", {subject, pattern.size()})) {
        const auto withoutPattern = res->GetTextWithoutPattern(1);
        const bool ftest2 = !!m_pcre->Check("id_test2", TStringBuf{subject.data(), withoutPattern.first.size() + withoutPattern.second.size()});

        if (ftest2)
            return false;
    }
    char* pId = new char[pattern.size() + 1];
    size_t IdLen = 0;
    for (size_t i = 0; i < pattern.size(); i++)
        if (pattern[i] != '-')
            pId[IdLen++] = pattern[i];

    pId[IdLen] = 0;

    bool fres = CheckIdByDictionary(pId, IdLen);

    DELETE_ARR(pId);

    return fres;
}

bool TSpAlg::CheckIdByDictionary(char* pId, int IdLen) {
    int i, csp = 0;
    while (IsSpace(*pId))
        ++csp;

    if (csp) {
        IdLen -= csp;
        memmove(pId, pId + csp, IdLen + 1);
    }

    csp = 0;
    for (i = IdLen - 1; i > 0 && IsSpace(pId[i]); i--)
        ++csp;

    if (csp) {
        IdLen -= csp;
        pId[IdLen] = 0;
    }

    if (IdLen <= 0)
        return false;

    // Unique IDs probably aren't going to be only one or two letters long
    if (IdLen < 3 || m_maptriplets->empty())
        return false;

    if (m_pcre->Check("id_test3", TStringBuf{pId, size_t(IdLen)}))
        return true;

    if (!strcmp(pId, "ot")) // off-topic
        return false;

    if (m_pcre->Check("id_test4", TStringBuf{pId, size_t(IdLen)})) // not in most dicts
        return false;
    if (m_pcre->Check("id_test5", TStringBuf{pId, size_t(IdLen)})) // not in most dicts
        return false;

    for (i = 0; i < IdLen - 2; i++)
        if (!m_maptriplets->contains(TStringBuf{pId + i, 3}))
            return true;

    return false;
}

void TSpAlg::CheckSubject() {
    int c;
    const char* pSubject = m_rgpFields[FD_SUBJECT].c_str();
    if (!(*pSubject))
        return;

    if (!strncmp(pSubject, "Re: ", 4) || !strncmp(pSubject, "RE: ", 4)) {
        c = FoundReInSubj(pSubject + 4, 0);
        if (c == 2)
            SetRule("SUBJ_RE_2");
        else if (c == 1)
            SetRule("SUBJ_RE_1");
        else
            SetRule("SUBJ_RE");
        SetRule("__SUBJ_RE");

    } else if (!strncmp(pSubject, "Re[", 3) &&
               isdigit(((ui8*)pSubject)[3]) &&
               !strncmp(pSubject + 4, "]: ", 3)) {
        c = FoundReInSubj(pSubject + 7, 0);
        if (c == 2)
            SetRule("SUBJ_REN_2");
        else if (c == 1)
            SetRule("SUBJ_REN_1");
        else
            SetRule("SUBJ_REN");
        SetRule("__SUBJ_RE");

    } else if (!strncmp(pSubject, "FW: ", 4)) {
        c = FoundReInSubj(pSubject + 4, 0);
        if (c == 2)
            SetRule("SUBJ_FW_2");
        else if (c == 1)
            SetRule("SUBJ_FW_1");
        else
            SetRule("SUBJ_FW");
        SetRule("SUBJ_FW_01");
    } else if (!strncmp(pSubject, "Fwd: ", 5)) {
        c = FoundReInSubj(pSubject + 5, 0);
        if (c == 2)
            SetRule("SUBJ_FWD_2");
        else if (c == 1)
            SetRule("SUBJ_FWD_1");
        else
            SetRule("SUBJ_FWD");
        SetRule("SUBJ_FW_01");
    }
}

int TSpAlg::FoundReInSubj(const char* szSubject, int ind) {
    const char* p;
    if ((p = strstr(szSubject, "Re: ")) ||
        (p = strstr(szSubject, "RE: ")) ||
        (p = strstr(szSubject, "Re[")) ||
        (p = strstr(szSubject, "FW: ")) ||
        (p = strstr(szSubject, "Fwd: "))) {
        if (ind)
            return ind + 1;
        return FoundReInSubj(p + 3, 1);
    }

    return ind;
}

void TSpAlg::CheckMessageId(const TStringBuf& field) {
    if (auto res = m_pcre->Check("mesid_outlooktime", field)) {
        const auto Pattern = res->GetPattern(1);

        if (Pattern.empty())
            return;
        if (!StrToHex((const ui8*)Pattern.data(), Pattern.size(), &m_cur.outlooktime))
            m_cur.outlooktime = 0;
    }
}

int TSpAlg::DelSpaces(char* szField) {
    int i, c;
    for (i = 0, c = 0; szField[i] != 0; i++)
        if (!IsSpace(szField[i]))
            szField[c++] = szField[i];
    szField[c] = 0;
    return c;
}

void TSpAlg::DelRepeatedSpaces(TString& szField) {
    bool fSpace = false;

    size_t c = 0;
    for (size_t i = 0; i < szField.size(); i++)
        if (IsSpace(szField[i])) {
            if (!fSpace) {
                fSpace = true;
                szField[c++] = ' ';
            }
        } else {
            fSpace = false;
            szField[c++] = szField[i];
        }

    szField.erase(c);
}

bool TSpAlg::GetDate(const char* field, int fieldlen, time_t* ptime) const {
    // make it a bit easier to match
    TArrayHolder<char> pDate{new char[fieldlen + 10]};
    pDate[0] = ' ';
    pDate[fieldlen + 1] = ' ';
    memcpy(pDate.Get() + 1, field, fieldlen);
    fieldlen += 2;
    pDate[fieldlen] = 0;

    int i, c;
    bool fSpace = true;
    for (i = 1, c = 1; i < fieldlen; i++)
        if (IsSpace(pDate[i]) || pDate[i] == ',') {
            if (!fSpace) {
                fSpace = true;
                pDate[c++] = ' ';
            }
        } else {
            fSpace = false;
            pDate[c++] = pDate[i];
        }

    pDate[c] = 0;
    fieldlen = c;

    if(TInstant instant; TInstant::TryParseRfc822(TStringBuf(pDate.Get(), fieldlen), instant)) {
        *ptime = instant.TimeT();
        return true;
    }

    return false;
}

void TSpAlg::AssignField(const char* field, int fieldlen, char** pdupfield) {
    DELETE_ARR(*pdupfield);
    STRDUP(pdupfield, field, fieldlen);
}

void TSpAlg::InitField(char** pfield) {
    DELETE_ARR(*pfield);
    STRDUP(pfield, "", 1);
}

bool TSpAlg::CheckOutlookTimeStamp() {
    if (!m_cur.outlooktime)
        return false;

    double x = 0.0023283064365387;
    double y = 27111902.8329849;

    // quite generous, but we just want to be in the right ballpark, so we
    // can handle mostly-correct values OK, but catch random strings.

    time_t t_date = 0;
    if (m_cur.fheader_date)
        t_date = m_cur.header_time;
    else {
        const char* pDate = m_rgpFields[FD_DATE].c_str();
        if (*pDate && !GetDate(pDate, strlen(pDate), &t_date))
            return false;
    }

    if (!t_date)
        return false;

    int expected = int((t_date * x) + y);

    if (abs((int)m_cur.outlooktime - expected) < 250)
        return false;

    //also try last date in Received header, Date could have been rewritten
    if (!m_vReceived.empty()) {
        if (!GetTimeReceived(m_vReceived.back(), &t_date))
            return false;
        expected = int((t_date * x) + y);
        if (abs((int)m_cur.outlooktime - expected) < 250)
            return false;
    }

    return true;
}

// ezmlm has a very bad habit of removing Received: headers! bad ezmlm.
//  gated_through_received_hdr_remover
bool TSpAlg::GatedThroughReceivedHdrRemover() {
    if (m_cur.fMailingList && m_cur.fDeliveredTo &&
        m_pcre->Check("qmail_invoked_by", m_sReceived))
        return true;

    // Empty received headers!  These tests cannot run in that case
    if (m_pcre->Check("empty_field", m_sReceived))
        return true;

    //  MSN groups removes Received lines.
    if (m_pcre->Check("from_groups_msn", m_sReceived))
        return true;

    return false;
}

// check_messageid_not_usable
bool TSpAlg::CheckMessageIdNotUsable() const {
    // Lyris eats message-ids.  also some ezmlm, I think :(
    //  $_ = $self->get ("List-Unsubscribe");
    //  return 1 if (/<mailto:(?:leave-\S+|\S+-unsubscribe)\@\S+>$/);
    // Lyris eats message-ids,  also some ezmlm
    if (m_cur.fMessageIdNotUsable)
        return true;

    // ezmlm
    if (m_cur.fGatedThrough)
        return true;

    // this as 'Wacky sendmail version?'
    if (m_pcre->Check("cwt_dce", m_sReceived))
        return true;

// too old; older versions of clients used different formats
#ifndef SP_CHECK_RULES
    if (ReceivedWithinMonths(9, 0))
        return true;
#endif
    //    return 1 if ($self->received_within_months('6','undef'));

    return false;
}

bool TSpAlg::GetTimeReceived(TString& received, time_t* pt_date) {
    TStringBuf pRecDate;

    *pt_date = 0;
    TMaybe<NRegexp::TResult> res;
    if (!(res = m_pcre->Check("received_date", received)) ||
        !res->GetPattern(1, pRecDate) ||
        !GetDate(pRecDate.data(), pRecDate.size(), pt_date) ||
        !(*pt_date)) {
        const char* strHot = "by HotBOX.Ru WebMail";
        if (strncmp(received.c_str(), strHot, strlen(strHot)))
            SetRule("RCVD_DATE_NOT_STANDART");
        return false;
    }
    return true;
}

void TSpAlg::CheckForShiftedDate() {
    GetReceivedHeaderTimes();
    GetDateDiff();
    CheckDateReceived();

    // 0 - value not limited
    if (CheckForShiftedDate(-6, -3))
        SetRule("DATE_IN_PAST_03_06");
    else if (CheckForShiftedDate(-12, -6))
        SetRule("DATE_IN_PAST_06_12");
    else if (CheckForShiftedDate(-24, -12))
        SetRule("DATE_IN_PAST_12_24");
    else if (CheckForShiftedDate(-48, -24))
        SetRule("DATE_IN_PAST_24_48");
    else if (CheckForShiftedDate(-96, -48))
        SetRule("DATE_IN_PAST_48_96");
    else if (CheckForShiftedDate(0, -96))
        SetRule("DATE_IN_PAST_96_XX");
    else if (CheckForShiftedDate(3, 6))
        SetRule("DATE_IN_FUTURE_03_06");
    else if (CheckForShiftedDate(6, 12))
        SetRule("DATE_IN_FUTURE_06_12");
    else if (CheckForShiftedDate(12, 24))
        SetRule("DATE_IN_FUTURE_12_24");
    else if (CheckForShiftedDate(24, 48))
        SetRule("DATE_IN_FUTURE_24_48");
    else if (CheckForShiftedDate(48, 96))
        SetRule("DATE_IN_FUTURE_48_96");
    else if (CheckForShiftedDate(96, 0))
        SetRule("DATE_IN_FUTURE_96_XX");
}

// DateTime
//###########################################################################

bool TSpAlg::CheckForShiftedDate(int hh_min, int hh_max) {
    return ((hh_min == 0 || m_cur.date_diff >= (3600 * hh_min)) &&
            (hh_max == 0 || m_cur.date_diff < (3600 * hh_max)));
}
//received_within_months
bool TSpAlg::ReceivedWithinMonths(int mon_min, int mon_max) const {
    // filters out some false positives in old corpus mail

    int diff = (int)(m_cur.curTime - m_cur.date_received);

    // 365.2425 * 24 * 60 * 60 = 31556952 = seconds in year (including leap)

    if ((mon_min == 0 || diff >= ((31556952 * mon_min) / 12)) && (mon_max == 0 || diff < ((31556952 * mon_max) / 12)))
        return true;
    return false;
}

void TSpAlg::GetHeaderTime() {
    const char* pResentDate = m_rgpFields[FD_RESENT_DATE].c_str();
    if (*pResentDate)
        GetDate(pResentDate, strlen(pResentDate), &m_cur.header_time);
    else {
        const char* pDate = m_rgpFields[FD_DATE].c_str();
        if (*pDate) {
            m_cur.fheader_date = true;
            GetDate(pDate, strlen(pDate), &m_cur.header_time);
        }
    }
}

void TSpAlg::GetReceivedHeaderTimes() {
    //sub _get_received_header_times
    //  my $self = $_[0];

    //  $self->{received_header_times} = [ () ];
    //  $self->{received_fetchmail_time} = undef;

    //  my(@received);
    //  my $received = $self->get('Received');
    //  if (defined($received) && length($received)) {
    //    @received = grep {$_ =~ m/\S/} (split(/\n/,$received));
    //  }
    //  if we have no Received: headers, chances are we're archived mail
    //  with a limited set of headers
    //  if (!scalar(@received)) {
    //    return;
    //  }

    if (m_vReceived.empty())
        return;
    bool fLocal = false;
    int cReceived = m_vReceived.size();
    if (cReceived > 1 && (m_pcre->Check("localhost", m_vReceived[cReceived - 1]) || m_pcre->Check("invoked_by_uid", m_vReceived[cReceived - 1])) && m_pcre->Check("localhost_with", m_vReceived[cReceived - 2]))
        fLocal = true;
    // handle fetchmail headers
    //my(@local);
    //if (($received[0] =~
    //    m/\bfrom (?:localhost\s|(?:\S+ ){1,2}\S*\b127\.0\.0\.1\b)/) ||
    //    ($received[0] =~ m/qmail \d+ invoked by uid \d+/))
    //  push @local, (shift @received);
    // }
    //  if (scalar(@received) &&
    //      ($received[0] =~ m/\bby localhost with \w+ \(fetchmail-[\d.]+/)) {
    //    push @local, (shift @received);
    //  }
    //  elsif (scalar(@local)) {
    //    unshift @received, (shift @local);
    //  }

    //  my $rcvd;

    //  if (scalar(@local)) {
    //    my(@fetchmail_times);
    //    foreach $rcvd (@local) {
    //     if ($rcvd =~ m/(\s.?\d+ \S\S\S \d+ \d+:\d+:\d+ \S+)/) {
    //    my $date = $1;
    //    dbg ("trying Received fetchmail header date for real time: $date",
    //         "datediff", -2);
    //    my $time = Mail::SpamAssassin::Util::parse_rfc822_date($date);
    //    if (defined($time) && (time() >= $time)) {
    //      dbg ("time_t from date=$time, rcvd=$date", "datediff", -2);
    //      push @fetchmail_times, $time;
    //    }
    //    }
    //    }
    if (fLocal) {
        time_t t_date;
        if (GetTimeReceived(m_vReceived[cReceived - 1], &t_date) && m_cur.curTime > t_date) {
            m_cur.received_fetchmail_time = t_date;
            if (GetTimeReceived(m_vReceived[cReceived - 2], &t_date) && m_cur.curTime > t_date && t_date > m_cur.received_fetchmail_time) {
                m_cur.received_fetchmail_time = t_date;
            }
        } else if (GetTimeReceived(m_vReceived[cReceived - 2], &t_date) && m_cur.curTime > t_date) {
            m_cur.received_fetchmail_time = t_date;
        }
        cReceived -= 2;
    }
    //    if (scalar(@fetchmail_times) > 1) {
    //      $self->{received_fetchmail_time} =
    //       (sort {$b <=> $a} (@fetchmail_times))[0];
    //    } elsif (scalar(@fetchmail_times)) {
    //      $self->{received_fetchmail_time} = $fetchmail_times[0];
    //    }
    //  }
    time_t t_date;
    for (int i = 0; i < cReceived; i++) {
        if (GetTimeReceived(m_vReceived[i], &t_date))
            m_vReceivedHeaderTimes.push_back(t_date);
    }
    //    my(@header_times);
    //  foreach $rcvd (@received) {
    //    if ($rcvd =~ m/(\s.?\d+ \S\S\S \d+ \d+:\d+:\d+ \S+)/) {
    //      my $date = $1;
    //      dbg ("trying Received header date for real time: $date", "datediff", -2);
    //    my $time = Mail::SpamAssassin::Util::parse_rfc822_date($date);
    //  if (defined($time)) {
    //    dbg ("time_t from date=$time, rcvd=$date", "datediff", -2);
    //    push @header_times, $time;
    //    }
    // }
    //}

    //  if (scalar(@header_times)) {
    //  $self->{received_header_times} = [ @header_times ];
    //} else {
    // dbg ("no dates found in Received headers", "datediff", -1);
    //}
}

//_check_date_received
void TSpAlg::CheckDateReceived() {
    //  my $self = $_[0];

    //  my(@dates_poss);

    //  $self->{date_received} = 0;

    //  if (!exists($self->{date_header_time})) {
    //    $self->_get_date_header_time();
    //  }

    std::vector<time_t> v_poss;
    v_poss.reserve(5);
    time_t t;

    if (m_cur.header_time)
        v_poss.push_back(m_cur.header_time);

    if (!m_vReceivedHeaderTimes.empty()) {
        t = m_vReceivedHeaderTimes.back();
        v_poss.push_back(t);
    }

    if (m_cur.received_fetchmail_time)
        v_poss.push_back(m_cur.received_fetchmail_time);

    if (m_cur.header_time && m_cur.date_diff) {
        t = m_cur.header_time - m_cur.date_diff;
        v_poss.push_back(t);
    }

    if (!v_poss.empty()) {
        //  descending
        std::sort(v_poss.begin(), v_poss.end(), std::greater<time_t>());
        m_cur.date_received = v_poss[v_poss.size() / 2];
    }

    //if (scalar(@dates_poss))     use median
    //    $self->{date_received} = (sort {$b <=> $a}
    //                  (@dates_poss))[int($#dates_poss/2)];
}

bool compareabstime(time_t x, time_t y) {
    bool f = (std::abs(x) < std::abs(y));
    return f;
}

//sub _check_date_diff()
void TSpAlg::GetDateDiff() {
    //  my $self = $_[0];

    //  $self->{date_diff} = 0;

    //  if (!exists($self->{date_header_time})) {
    //    $self->_get_date_header_time();
    //  }

    //  if (!defined($self->{date_header_time})) {
    //    return;            already have tests for this
    //  }

    if (!m_cur.header_time)
        return;

    if (m_vReceivedHeaderTimes.empty())
        return;
    //    if (!exists($self->{received_header_times})) {
    //        $self->_get_received_header_times();
    //    }
    //  my(@header_times) = @{ $self->{received_header_times} };

    //  if (!scalar(@header_times)) {
    //    return;            archived mail?
    //  }
    //
    //  my(@diffs) = map {$self->{date_header_time} - $_} (@header_times);

    std::vector<time_t> v_diffs(m_vReceivedHeaderTimes);
    std::vector<time_t>::iterator iv;
    for (iv = v_diffs.begin(); iv != v_diffs.end(); iv++)
        (*iv) -= m_cur.header_time;

    // if the last Received: header has no difference, then we choose to
    // exclude it
    //  if ($#diffs > 0 && $diffs[$#diffs] == 0) {
    //    pop(@diffs);
    //  }
    if (v_diffs.back() == 0)
        v_diffs.pop_back();

    if (v_diffs.empty())
        return;
    // use the date with the smallest absolute difference
    // (experimentally, this results in the fewest false positives)
    std::sort(v_diffs.begin(), v_diffs.end(), compareabstime);
    //  @diffs = sort { abs($a) <=> abs($b) } @diffs;
    //  $self->{date_diff} = $diffs[0];
    m_cur.date_diff = v_diffs[0];
}

void TSpAlg::CheckDelivery(TRengine& rengine) {
    std::vector<TString>::iterator vi;
    const char* prcvd;
    TString pIp;
    TString pIpFirst = "-";
    TString pIpLast = "-";
    TString pRdns;
    TString pHelo;
    const TRuleDef* pDlvrRule = 0;
    char* pHost = 0;
    char* pIpRem = 0;
    ui32 rgnet[REC_MAX_LEVEL];
    ui32 rgip[REC_MAX_LEVEL];
    ui8 level = 0;
    char str[512];

    m_cur.fCheckDelivery = true;

    memset(&rgnet, 0, sizeof(rgnet));
    memset(&rgip, 0, sizeof(rgip));

    for (vi = m_vReceived.begin(); vi != m_vReceived.end() && level < REC_MAX_LEVEL; vi++) {
        prcvd = (*vi).c_str();
        snprintf(str, 511, "Received: %s", prcvd);
        m_pstat.AddStat(ST_RECEIVED) << str;

        pIp = "-";
        pRdns = "-";
        pHelo = "-";

        while (prcvd < vi->cend() && IsSpace(*prcvd))
            ++prcvd;
        if (!strncmp(prcvd, "from", 4)) {
            prcvd += 4;
            while (prcvd < vi->cend() && IsSpace(*prcvd))
                ++prcvd;
        }

        size_t rcvdlen = strlen(prcvd);

        TMaybe<NRegexp::TResult> res;
        if (vi == m_vReceived.begin()) {
            GetReceivedFirst({prcvd, rcvdlen}, en_MTA_SENDMAIL, pIp, pRdns, pHelo);
            if (pIp.StartsWith('-'))
                GetReceivedNext({prcvd, rcvdlen}, pIp, pRdns, pHelo);
            if (pIp.StartsWith('-')) {
                m_curMessageContext->Logger << TLOG_ERR << "Could not define relay ip : " << prcvd;
#ifdef SO_WITH_IPV6
            } else {
                m_ip_addr_last.FromString(pIp);

                m_cur.fIpv6 = m_ip_addr_last.IsIpv6();
                if (m_ip_addr_last.IsValid()) {
                    pIp = m_ip_addr_last.ToString();
                    if (m_ip_addr_last.IsIpv6()) {
                        TIpAddr mask;
                        mask.MakeMask(m_fWebMail ? 32 : 64, true);
                        TIpAddr maskedIP = m_ip_addr_last & mask;
                        rengine.AddPattern(maskedIP.ToString().c_str(), EN_SH_SOURCE_IP_MASKED);
                    }
                } else
                    pIp = "-";
            }
#else
            }
#endif
            snprintf(str, 511, "source ip = %s  rdns = %s  helo = %s", pIp.c_str(), pRdns.c_str(), pHelo.c_str());
            m_curMessageContext->DlvLogRequest.SourceIp = pIp;
            m_pstat.AddStat(ST_RECEIVED) << str;
            rengine.m_cur->MlLogBuilder.Add(ToString(ST_RECEIVED), str);

            if (!pHelo.StartsWith('-')) {
                rengine.CheckFieldAllRules(FD_REC_HELO, pHelo);
                rengine.AddPattern(pHelo, EN_SH_HELO);
                rengine.CheckDomain(pHelo);
                rengine.Add2UrlReputationList(pHelo, EUrlStaticticSource::RCVDS);
            }
            if (!pRdns.StartsWith('-')) {
                TStringBuf pHelo_level2;
                TStringBuf pRdns_level2;
                rengine.CheckFieldAllRules(FD_REC_RDNS, pRdns);
                m_rulesHolder.m_pListRuler.CheckDomainName(m_curMessageContext->rulesContext, pRdns, SP_LIST_RDNS, SP_LIST_RDNS_PART);
                if (!pHelo.StartsWith('-') &&
                    (pHelo_level2 = m_pcre->GetPattern("get_domen2", pHelo, 2)) &&
                    (pRdns_level2 = m_pcre->GetPattern("get_domen2", pRdns, 2)) &&
                    pHelo_level2 && pRdns_level2 && pHelo_level2 != pRdns_level2)
                    SetRule("HELODOMAIN_NOIN_RESOLVED_2");
                if (pHelo != pRdns)
                    rengine.Add2UrlReputationList(pRdns, EUrlStaticticSource::RCVDS);
            }
            if (!pIp.StartsWith('-')) {
                rengine.CheckFieldAllRules(FD_REC_IP, pIp);
                if (!m_cur.fIpv6)
                    m_rulesHolder.m_pListRuler.CheckIpNet(m_curMessageContext->rulesContext, pIp, SP_LIST_IP, SP_LIST_IP_PART);
#ifdef SO_WITH_IPV6
                m_rulesHolder.m_pListRuler.CheckIpv46(m_curMessageContext->rulesContext, m_ip_addr_last);
#endif
                pIpLast = pIp;
                m_cur.fValidIp = true;
            }
        } else {
            if (level > REC_MAX_LEVEL + 5)
                break;
            GetReceivedNext({prcvd, rcvdlen}, pIp, pRdns, pHelo);
#ifdef SO_WITH_IPV6
            m_ip_addr_first.FromString(pIp);
            if (m_ip_addr_first.IsValid()) {
                pIp = m_ip_addr_first.ToString();
            } else
                pIp = "-";
#endif
            snprintf(str, 511, "next ip = %s  rdns = %s  helo = %s", pIp.c_str(), pRdns.c_str(), pHelo.c_str());
            m_pstat.AddStat(ST_RECEIVED) << str;
            {
                if (pIp.StartsWith('-') || !(pIp = m_pcre->GetPattern("rec_ip_pure", pIp, 1)))
                    continue;
            }
            bool fipv6_next = m_ip_addr_first.IsIpv6();

            if (local_matcher.match(m_ip_addr_first) || !fipv6_next && (res = m_pcre->Check("ip_local", pIp)))
            {
                m_pstat.AddStat(ST_LOG) << "local ip";
                continue;
            }
            m_sIpFirst.assign(pIp);
            m_sHostFirst.assign(pRdns);

            if (pIp.size() > 6 || fipv6_next && pIp.size() > 2)
                pIpFirst = pIp;
            if (!pHelo.StartsWith('-')) {
                rengine.CheckFieldAllRules(FD_REC_HELO_NEXT, pHelo);
                rengine.CheckDomain(pHelo);
                rengine.Add2UrlReputationList(pHelo, EUrlStaticticSource::RCVDS);
            }
            if (pRdns.StartsWith('-')) {
                if (m_fWebMail)
                    SetRule("RELAY_NORESOLVED_IN_RCVD");
            } else {
                rengine.CheckFieldAllRules(FD_REC_RDNS_NEXT, pRdns);
                rengine.Add2UrlReputationList(pRdns, EUrlStaticticSource::RCVDS);
            }
            if (!pIp.StartsWith('-'))
                rengine.CheckFieldAllRules(FD_REC_IP_NEXT, pIp);
            //     The Internet Assigned Numbers Authority (IANA) has reserved the following three blocks of the IP address space for private internets:
            //     10/8              Private Use
            //     127/8             Private Use (localhost)
            //     169.254/16        Private Use (APIPA)
            //     172.16.0.0      -   172.31.255.255  (172.16/12 ) Private Use
            //     192.168/16        Private Use
            //     224-239/8:        IANA Rsvd, Mcast
            //     230-239/8:        IANA Multicast // sources:
            //     rfc1918
            //   IANA  = <http://www.iana.org/assignments/ipv4-address-space>,
            //           <http://duxcw.com/faq/network/privip.htm>,
            //   APIPA = <http://duxcw.com/faq/network/autoip.htm>,
            //   3330  = <ftp://ftp.rfc-editor.org/in-notes/rfc3330.txt>
            //   CYMRU = <http://www.cymru.com/Documents/bogon-list.html>
            //
        }

        if (!pRdns.StartsWith('-') && !(pRdns = m_pcre->GetPattern("rec_rdns_clear", pRdns, 1)))
            pRdns = "-";

        if (m_pcre->Check("with_pop3", TStringBuf{prcvd, size_t(rcvdlen)})) {
            if (vi == m_vReceived.begin())
                m_cur.fForward = true;
            continue;
        }

        if (m_cur.fForward)
            continue;

        ui32 ip = 0;
        ui32 net = 0;

        if (!m_cur.fIpv6) {
            IpToInt(pIp.c_str(), pIp.size(), &net, &ip);

            if (level && rgnet[level - 1] == net)
                continue;

            if (!ip) {
                if (level)
                    continue;
                else
                    break; // unknown sender
            }
        }

        if (pRdns.StartsWith('-'))
            m_rgsReceivedHost[level].assign("");
        else
            m_rgsReceivedHost[level].assign(pRdns);

        if (!level) {
            if (!pRdns.StartsWith('-')) {
                AddSender(FD_RECEIVED, pRdns, 0);
                STRDUP(&pHost, pRdns.c_str(), pRdns.size());
                m_sRelay.assign(pHost);
            }
            if (!m_cur.fIpv6) {
                pDlvrRule = rengine.GetIpRule(ip);
                if (!pDlvrRule) {
                    pDlvrRule = rengine.GetNetRule(net);
                    if (!pDlvrRule)
                        pDlvrRule = rengine.GetSuperNetRule(net);
                }
                if (pDlvrRule)
                    AssignField(pIp.c_str(), pIp.size(), &pIpRem);
            }
        }

        if (level < REC_MAX_LEVEL && !m_cur.fIpv6) {
            rgnet[level] = net;
            rgip[level] = ip;
            //#ifdef SO_WITH_IPV6
            //            m_ip_addr_last = ip_addr;
            //#endif
        }

        ++level;
    }

#ifdef SO_WITH_IPV6
    if ((pIpLast.size() > 2) && !local_matcher.match(m_ip_addr_last))
#else
        if (strlen(pIpLast) > 6 && !(m_pcre->Check("ip_local", pIpLast)))
#endif
    {
        rengine.AddPattern(pIpLast, EN_SH_LAST_SENDER_IP);
        if (pIpFirst.StartsWith('-'))
            rengine.AddPattern(pIpLast, EN_SH_FIRST_LAST_SENDER_IP); // message source
        rengine.AddPattern(pIpLast, EN_SH_IP_FWD, false);            // ������ ������ get
        rengine.AddPattern(pIpLast, EN_SH_IP_BOX, false);            // ������ ������ get
        rengine.AddPattern(pIpLast, EN_SH_IP_BOX_NO_UNIQ, true, false, rengine.GetRcptCount());
        rengine.AddPatternIpTo(pIpLast);
        m_sIpSender.assign(pIpLast);
    }

    if (!pIpFirst.StartsWith('-')) {
        rengine.AddPattern(pIpFirst, EN_SH_FIRST_LAST_SENDER_IP); // message source
        if (pIpFirst != pIpLast) {
            m_rulesHolder.m_pListRuler.CheckIpNet(m_curMessageContext->rulesContext, pIpFirst, SP_LIST_IP_FIRST, SP_LIST_IP_FIRST_PART);
#ifdef SO_WITH_IPV6
            if (m_ip_addr_first.IsValid())
                m_rulesHolder.m_pListRuler.CheckIpv46First(m_curMessageContext->rulesContext, m_ip_addr_first);
#endif
        }
    }

    if (!pDlvrRule || (pDlvrRule && pDlvrRule->pRuleName != "DLV_FREEM")) {
        m_cur.level = level;
        if (level) {
            if (level > REC_MAX_LEVEL) {
                level = REC_MAX_LEVEL;
            }
            memcpy(m_cur.rgnet, rgnet, level * sizeof(m_cur.rgnet[0]));
            memcpy(m_cur.rgip, rgip, level * sizeof(m_cur.rgip[0]));

            if (level == 1)
                SetRule("__REC_LEVEL_1");
            else
                SetRule("__REC_LEVEL_2");
        } else
            SetRule("__REC_LEVEL_0");
    }

    TStringBuf pattern;
    if (m_vReceived.rbegin() != m_vReceived.rend() &&
        (pattern = m_pcre->GetPattern("get_by_host", m_vReceived.back(), 1)))
        m_rulesHolder.m_pListRuler.CheckDomainName(m_curMessageContext->rulesContext, pattern, SP_LIST_SENDER, SP_LIST_SENDER_PART);

    DELETE_ARR(pHost);
    DELETE_ARR(pIpRem);
}

void TSpAlg::SetForwardHost(TStringBuf prcvd, TSpMTAType mtaType) {
    TString pIp;
    TString pRdns;
    TString pHelo;

    for(size_t i = 0; i < prcvd.size(); i++) {
        if(!IsSpace(prcvd[i]) && i) {
            prcvd.Skip(i);
            break;
        }
    }

    while(!prcvd.empty() && IsSpace(prcvd.front()))
        prcvd.Skip(1);


    if (prcvd.StartsWith("from")) {
        prcvd.Skip(4);
        while(!prcvd.empty() && IsSpace(prcvd.front()))
            prcvd.Skip(1);
    }

    GetReceivedFirst(prcvd, mtaType, pIp, pRdns, pHelo);

    m_sForwardHost.assign(pRdns);
    m_sForwardIp.assign(pIp);
}

void TSpAlg::GetReceivedFirst(TStringBuf prcvd, TSpMTAType mtaType, TString& ppIp, TString& ppRdns, TString& ppHelo) {
    {
        const auto n = prcvd.find(" by ");
        if(n != TStringBuf::npos)
            prcvd.Chop(prcvd.size() - n - 4);
    }

    switch (mtaType) {
        case en_MTA_ZMAILER:

            //Received: from energo.tmb.ru ([213.135.141.5]:3998 "EHLO
            //        proxy.tamben.elektra.ru" smtp-auth: <none> TLS-CIPHER: <none>
            //        TLS-PEER-CN1: <none>) by mail.yandex.ru with ESMTP id S966712AbVATSF4

            if (auto res = m_pcre->Check("rec_zmailer", prcvd)) {
                ppRdns = res->GetPattern(1);
                ppIp = res->GetPattern(2);
                const auto pattern = res->GetPattern(3);
                if (!pattern.empty())
                    ppHelo = m_pcre->GetPattern("zmailer_helo", pattern, 1);
            }

            break;

        case en_MTA_SENDMAIL:
        case en_MTA_QMAIL:
        case en_MTA_CGP:
        case en_MTA_POSTFIX:
        case en_MTA_EXIM:

            // snprintf(rh,(1024*5)-1,"Received: from %s (%s [%s]) \r\n\tby "
            //           "%s; %3s, %d %3s %d %02d:%02d:%02d %+.2ld%02ld (%s)\r\n",
            // helohost,ci->hostname,ci->peer,hst, wday[tm.tm_wday],tm.tm_mday, mon[tm.tm_mon], tm.tm_year+1900,
            // tm.tm_hour, tm.tm_min, tm.tm_sec, tm.tm_gmtoff/3600,(tm.tm_gmtoff/60)%60,tm.tm_zone

            if (auto res = m_pcre->Check("rec_sendmil", prcvd)) {
                ppHelo = res->GetPattern(1);
                ppRdns = res->GetPattern(2);
                ppIp = res->GetPattern(3);
            }
            break;
        default:
            GetReceivedNext(prcvd, ppIp, ppRdns, ppHelo);
    }
    if (ppRdns == "unknown")
        ppRdns = "-";
}

bool TSpAlg::GetReceivedNext_2(TStringBuf prcvd, TString& ppIp, TString& ppRdns, TString& ppHelo) {
    bool fip2 = false;

    bool res = false;

    ppHelo = "-";
    ppRdns = "-";
    ppIp = "-";

    if (auto res = m_pcre->Check("rec_helo", prcvd)) {
        ppHelo = res->GetPattern(1);
        const auto& beforeAndAfter = res->GetTextWithoutPattern(0);

        const auto s = TString{beforeAndAfter.first} + beforeAndAfter.second;

        if (!(ppIp = m_pcre->GetPattern("rec_ip", s, 1)))
            ppIp = m_pcre->GetPattern("rec_ip_2", s, 1);
        ppRdns = m_pcre->GetPattern("rec_rdns_2", s, 1);
        return true;
    }

    if (!(ppIp = m_pcre->GetPattern("rec_ip", prcvd, 1))) {
        if (ppIp = m_pcre->GetPattern("rec_ip_2", prcvd, 1))
            fip2 = true;
        else {
            {
                ppIp = "-";
                m_pstat.AddStat(ST_LOG) << "undefined ip (0)";
            }
        }
        //Received: from ppp85-140-31-127.pppoe.mtu-net.ru (HELO stasis) (Links@samokopirka.ru@85.140.31.127)
        //  by smtp2.masterhost.ru with SMTP; 25 Jan 2005 17:50:10 -0000
        //Received: from 165.146.22.2 by by23fd.bay23.hotmail.msn.com with HTTP;
    }

    if (m_pcre->Check("ip_unknown", prcvd)) {
        m_cur.fUndefinedIp = true;
        m_pstat.AddStat(ST_LOG) << "undefined ip (2)";
    }

    return res;
}

//TSpMTAType TSpAlg::GetMTAType(const char * prcvd, int len)
TSpMTAType TSpAlg::GetMTAType(const TStringBuf& /*prcvd*/) {
    return EN_MTA_UNKNOWN;
}

void TSpAlg::AddSender(TSpFields fid, const TStringBuf& pfield, int Weight) {

    TMaybe<NRegexp::TResult> res;
    m_faddprint = false;
    if (Weight) {
        if (!(res = m_pcre->Check("get_sender", pfield)))
            return;
    } else if (!(res = m_pcre->Check("get_sender_rec", pfield)))
        return;

    if (fid != FD_MESSAGEID && fid != FD_RECEIVED && fid != FD_FROM_ADDR && fid != FD_SUBJECT)
        m_pstat.AddStat(fid) << SafeRecode(pfield);

    if (TStringBuf pattern = res->GetPattern(1)) {
        if (!(res = m_pcre->Check("get_domen2", pattern)) || PutSender(*res, fid, Weight, true))
            return;
        if (!(res = m_pcre->Check("get_domen3", pattern)) || PutSender(*res, fid, Weight, false))
            return;
        if (!(res = m_pcre->Check("get_domen4", pattern)) || PutSender(*res, fid, Weight, false))
            return;
    }
}

bool TSpAlg::PutSender(const NRegexp::TResult& res, TSpFields fid, int Weight, bool CheckGeoDomain) {
    int ind;
    bool fCoincide = true;

    TStringBuf Domen;
    if (res.GetPattern(1, Domen))
        fCoincide = false;

    if (!res.GetPattern(2, Domen))
        return fCoincide;

    // Check geo-domain
    if (CheckGeoDomain) {
        if (m_pcre->Check("geo_domain", Domen))
            return false;
    }
    {
        auto loweredDomen = to_lower(TString{Domen});
        auto it = m_mapSender.find(loweredDomen);
        if (it == end(m_mapSender)) {
            if (m_cur.cSenders >= MAX_SENDERS)
                return fCoincide;
            m_mapSender.emplace(std::move(loweredDomen), m_cur.cSenders);
            ind = m_cur.cSenders++;
        } else {
            ind = it->second;
        }
    }

    if (Weight > 0)
        m_cur.senders[ind].fSender = true;
    else if (Weight == 0)
        m_cur.senders[ind].fRelay = true;

    switch (fid) {
        case FD_MESSAGEID:
            m_cur.senders[ind].fields[en_SR_MESSID] = true;
            break;
        case FD_FROM_ADDR:
            m_cur.senders[ind].fields[en_SR_FROM] = true;
            m_cur.senders[ind].fDeliveryField = true;
            break;
        case FD_REPLY_TO_ADDR:
            m_cur.senders[ind].fields[en_SR_REPLYTO] = true;
            m_cur.senders[ind].fDeliveryField = true;
            break;
        case FD_SENDER:
            m_cur.senders[ind].fields[en_SR_SENDER] = true;
            m_cur.senders[ind].fDeliveryField = true;
            break;
        case FD_LIST_:
            m_cur.senders[ind].fields[en_SR_LIST] = true;
            m_cur.senders[ind].fDeliveryField = true;
            break;
        case FD_SUBJECT:
            m_cur.senders[ind].fields[en_SR_SUBJ] = true;
            break;
        default:
            break;
    }
    m_cur.senders[ind].count++;
    if (Weight > m_cur.senders[ind].weight_sort)
        m_cur.senders[ind].weight_sort = Weight;
    if (fCoincide) {
        m_cur.senders[ind].weight_sort++;
        if (Weight > 0)
            m_cur.senders[ind].fCoincide = true;
    }

    return fCoincide;
}

void TSpAlg::DomenPrepare(TRengine& rengine) {
    char strip[32];
    bool fCancelDomainLabel;
    bool fBanOrFreem = false;

    if (m_cur.fDomenPrepare)
        return;

    m_cur.fDomenPrepare = true;

    CheckDelivery(rengine);

//    if (!m_cur.level || !m_cur.rgip[0] && !m_cur.fIpv6 || !m_cur.fValidIp)
//        return; // finish

    bool fdomains = false;
    TString s_domains = IntToIp(m_cur.rgip[0], strip);

    TVector<std::pair<TStringBuf, std::reference_wrapper<const TSpSender>>> vSenders(Reserve(m_mapSender.size()));
    for(const auto & [domain, ind] : m_mapSender) {
        const TSpSender& sender = m_cur.senders[ind];

        vSenders.emplace_back(domain, sender);
        if (sender.fSender && sender.fDeliveryField) {
            s_domains.append(" ");
            s_domains.append(domain);
            fdomains = true;
        }
    }

    if (fdomains && !m_fWebMail)
        m_pstat.AddStat(ST_DOMAINS) << s_domains;

    if (vSenders.empty())
        return; // finish

    sort(vSenders.begin(), vSenders.end(), [](const auto& p1, const auto& p2){
        return p1.second.get().weight_sort > p2.second.get().weight_sort;
    });

    const TDomains domains = CheckAuthDomains();

    {
        if (domains.Dkim)
            m_rulesHolder.m_pListRuler.CheckDomainNamePart(m_curMessageContext->rulesContext, domains.Dkim, SP_LIST_SPF);
        if (domains.Spf)
            m_rulesHolder.m_pListRuler.CheckDomainNamePart(m_curMessageContext->rulesContext, domains.Spf, SP_LIST_SPF);
    }

    for (const auto& [domain, senderPtr] : vSenders) {
        const auto& sender = senderPtr.get();
        if (sender.fSender) {
            rengine.AddSiteNetc(TString{domain});

            const bool bDkimOK = domains.Dkim && cmpdomensrevi(domain, domains.Dkim);// sender domain corresponds with verified dkim domain
            const bool bSpfOK = domains.Spf && cmpdomensrevi(domain, domains.Spf);// sender domain corresponds with verified spf domain
            if (sender.fields[en_SR_FROM] && (bDkimOK || bSpfOK))
                SetRule("TRUST_FROM");

            fBanOrFreem = false;
            if (const TSpListValueDlv * spListValueDlv = m_rulesHolder.m_pListRuler.SetDeliveryDomen(m_curMessageContext->rulesContext, TString{domain}, sender.fields, &fCancelDomainLabel, &fBanOrFreem); spListValueDlv) {

                const TSpListValueDlv& dlvValue = *spListValueDlv;
                TStringBuf pRuleName, pDomainLabel;

                TStringBuilder slabel;

                if (m_rulesHolder.m_pListRuler.CheckDeliveryRelay(m_curMessageContext->rulesContext, dlvValue, m_sRelay, pRuleName, pDomainLabel))
                    slabel << "list " << pRuleName << " domen " << domain << " relay " << m_sRelay;
                else if (m_rulesHolder.m_pListRuler.CheckDeliveryTrusted(m_curMessageContext->rulesContext, dlvValue, pRuleName, pDomainLabel))
                    slabel << "list " << pRuleName << " domen " << domain << " trusted ";
                else if (m_cur.fIpv6) {
#ifdef SO_WITH_IPV6
                    if (m_ip_addr_last.IsValid() && m_rulesHolder.m_pListRuler.CheckDeliveryMask(m_curMessageContext->rulesContext, dlvValue, m_ip_addr_last, pRuleName, pDomainLabel))
                        slabel << "list " << pRuleName << " domen " << domain << " mask " << m_ip_addr_last.ToString();
#endif
                } else {
                    if (m_rulesHolder.m_pListRuler.CheckDeliveryIp(m_curMessageContext->rulesContext, dlvValue, m_cur.rgip[0], pRuleName, pDomainLabel))
                        slabel << "list " << pRuleName << " domen " << domain << " ip " << IntToIp(m_cur.rgip[0], strip);
                    else if (m_rulesHolder.m_pListRuler.CheckDeliveryNet(m_curMessageContext->rulesContext, dlvValue, m_cur.rgnet[0], pRuleName, pDomainLabel))
                        slabel << "list " << pRuleName << " domen " << domain << " net " << IntToIp(m_cur.rgnet[0], strip);
                }

                if (fBanOrFreem) {
                    continue;
                }

                if ((bDkimOK || bSpfOK) && !pRuleName && m_rulesHolder.m_pListRuler.CheckDeliverySpfDkim(m_curMessageContext->rulesContext, dlvValue, pRuleName, pDomainLabel, bDkimOK, bSpfOK))
                    slabel << "list " << pRuleName << " domen " << domain << " spf-dkim auth";

                const bool fakeSender = !bDkimOK && !bSpfOK &&
                                        (m_curMessageContext->rulesContext.IsRuleWorked("FRNR") ||
                                         m_curMessageContext->rulesContext.IsRuleWorked("FAKE_RESOLV") ||
                                         m_curMessageContext->rulesContext.IsRuleWorked("FAKE_RESOLV_V6"));

                if (!fakeSender && pRuleName && pRuleName != "DLV_FREEM") // do not set for fake senders and DLV_FREEM
                {
                    m_cur.fFixedSource = true;
                    if (pDomainLabel) {
                        slabel << " label " << pDomainLabel;
                        if (fCancelDomainLabel)
                            slabel << " cancel";
                        else
                            m_sDomainLabel.assign(pDomainLabel);
                    }
                    m_pstat.AddStat(ST_FIXED_BY_RULE) << slabel;

                    SetRule("DL_FBR");
                }
            }
        }
    }
}

float TSpAlg::UnpackYandWeight(ui32 yand_weight) {
    ui32 tmpw = yand_weight;
    float weightf = 0.;
    if (tmpw) {
        char* pc = (char*)(&tmpw);
        char arr[4];
        if (tmpw != 0xFFFFFFFF)
            tmpw ^= 0xFFFFFFFF;
        arr[0] = pc[3];
        arr[1] = pc[1];
        arr[2] = pc[0];
        arr[3] = pc[2];

        weightf = *((float*)arr);
        memcpy(&tmpw, arr, 4);
    }
    return weightf;
}

bool TSpAlg::CheckDomenNameNew(TRengine& rengine, TSiteNetcInfo* sni, const char* pDomen, const char* pDomen_cs, int level, bool fTestTop) {
    int i, ind = 0;
    bool f_correspond = false;
    ui32 netc = m_cur.rgnet[level];

    if (!netc)
        return false;

    // DEBUG !!!
    // Check for block here

    if (!fTestTop) {
        const auto lowered = to_lower(TString(pDomen));
        if(m_rulesHolder.GetMapBanListSender().contains(lowered)) {
            //m_pstat.AddStat(ST_FIXED_BY_SHINGLER) << "ban list sender";
            return false;
        }
    }

    if (!fTestTop && sni->bl_arr[en_SR_HOST]) {
        //m_pstat.AddStat(ST_FIXED_BY_SHINGLER) << "ban new list sender: " << pDomen;
        return false;
    }

    {
        const auto lowered = to_lower(TString(pDomen));
        auto it = m_mapSender.find(lowered);
        if (it == end(m_mapSender))
            m_curMessageContext->Logger << TLOG_ERR << "Internal sender error for domain: " << pDomen;
        else
            ind = it->second;
    }

    for (i = 0; i < sni->c_netc; i++) {
        if (netc == sni->netc[i]) {
            f_correspond = true;
            //m_pstat.AddStat(ST_FIXED_BY_SHINGLER) << "fixed by ip";
            //            m_cur.fShFixedSource = true;
            break;
        } else if (abs((int)(sni->netc[i] - netc)) <= 1024) {
            f_correspond = true;
            //m_pstat.AddStat(ST_FIXED_BY_SHINGLER) << "fixed by ip (near)";
            //            m_cur.fShFixedSource = true;
        }
    }

    if (!f_correspond && sni->c_host_rcvd && m_rgsReceivedHost[level].length() > 3) {
        if (CheckHostReceivedNew(sni, level, "get_domen2") ||
            CheckHostReceivedNew(sni, level, "get_domen3") ||
            CheckHostReceivedNew(sni, level, "get_domen4")) {
            //m_pstat.AddStat(ST_FIXED_BY_SHINGLER) << "fixed by host";
            f_correspond = true;
        }
    }

    //if (!fTestTop && !f_correspond && m_cur.senders[ind].fRelay && m_cur.senders[ind].fSender) {
        //m_pstat.AddStat(ST_FIXED_BY_SHINGLER) << "fixed domen unknown net";
    //}

    if (!f_correspond)
        return false;

    if (fTestTop)
        return true;
    else if (CheckNewSender(en_SR_FROM, ind, sni) ||
             CheckNewSender(en_SR_REPLYTO, ind, sni) ||
             CheckNewSender(en_SR_SENDER, ind, sni) ||
             CheckNewSender(en_SR_LIST, ind, sni) ||
             CheckNewSender(en_SR_SUBJ, ind, sni) ||
             CheckNewSender(en_SR_MESSID, ind, sni)) {
        // set level
        m_cur.sh_level = m_cur.level;
        if (m_cur.sh_level > 1) {
            for (i = 1; i < m_cur.sh_level; i++)
                if (!CheckDomenNameNew(rengine, sni, pDomen, pDomen_cs, i, true))
                    break;
            if (i == m_cur.sh_level)
                m_cur.sh_level = 1; // internal relays
            else if (sni->bl_arr[en_SR_L2ASL1]) {
                //m_pstat.AddStat(ST_FIXED_BY_SHINGLER) << "level2 as level1";
                m_cur.fShFixedSource = true;
                m_cur.sh_level = 1; // internal relays
            }
        }
    } else
        return false;

    float fweight = UnpackYandWeight(sni->w_yand);
    if (fweight >= 0.5) {
        if (fweight >= 30.)
            SetRule("__PGR_30");
        else if (fweight >= 10.)
            SetRule("__PGR_10");
        else if (fweight >= 1.)
            SetRule("__PGR_1");
        else
            SetRule("__PGR_05");
    }

    if (sni->w_rul) {
        m_curMessageContext->rulesContext.SetScore("NEWSH_RULW", sni->w_rul);
        SetRule("NEWSH_RULW");
    }
    // delivery detection is no longer important
    //    if (sni->w_dlv)
    //    {
    //        rengine.SetScore("NEWSH_DLVW", sni->w_dlv);
    //        SetRule ("NEWSH_DLVW");
    //    }

    SetRule("NEWSH_FS");
    if (m_cur.sh_level == 1 && !m_cur.fUndefinedIp &&
        !m_curMessageContext->rulesContext.IsRuleWorked("__PRESENT_X_ORIGINATING_IP"))
        SetRule("NEWSH_FSL1");

    if (sni->bl_arr[en_SR_RBL])
        SetRule("NEWSH_RBL_PERMIT");
    if (sni->bl_arr[en_SR_DSL])
        SetRule("NEWSH_DSL_PERMIT");
    if (!sni->bl_arr[en_SR_NRV])
        SetRule("NEWSH_NRV_NOT_PERMIT");
    if (sni->bl_arr[en_SR_BL_NOT])
        SetRule("NEWSH_ALL_PERMIT");

    m_cur.fShFixedSource = true;
    return true;
}

bool TSpAlg::CheckHostReceivedNew(TSiteNetcInfo* sni, int level, const char* domen_level) {
    TStringBuf pRdnsDomen;
    char crs_str[32];
    int i;

    if (*(m_rgsReceivedHost[level].c_str()) == 0)
        return false;

    if ((pRdnsDomen = m_pcre->GetPattern(domen_level, m_rgsReceivedHost[level], 2)) &&
        calc_strcrc64(pRdnsDomen, crs_str)) {
        for (i = 0; i < sni->c_host_rcvd && i < 15; i++)
            if (!memcmp(sni->host_rcvd[i], crs_str, 16))
                return true;
    }

    return false;
}

bool TSpAlg::CheckNewSender(TSpSenderType sendertype, int ind, TSiteNetcInfo* sni) {
    if (m_cur.senders[ind].fields[sendertype]) {
        if (sni->bl_arr[sendertype]) {
            //m_pstat.AddStat(ST_FIXED) << "ban new sender (" << pDomen << ") field: " << m_SpSenderTypeStr[sendertype];
            return false;
        }
    } else
        return false;

    return true;
}

void TSpAlg::AddSiteNetcInfo(TRengine& rengine, const char* pDomen, const char* pDomen_cs, const char* sitenecinfo) {
    TSiteNetcInfo sni;
    const int StrSize = 350;
    char str[StrSize + 2];

    if (m_cur.fIpv6) {
        m_pstat.AddStat(ST_LOG) << "ipv6 skip netcInfo";
        return;
    }

    //    m_mapSiteNetcInfo.Add(pDomen, sitenecinfo);
    memset(&sni, 0, sizeof(TSiteNetcInfo));
    sni.bl_arr[en_SR_MESSID] = true;
    sni.bl_arr[en_SR_SENDER] = false;
    sni.bl_arr[en_SR_SUBJ] = false;
    sni.bl_arr[en_SR_RBL] = false;
    sni.bl_arr[en_SR_DSL] = false;
    sni.bl_arr[en_SR_NRV] = false;
    sni.phost = pDomen;
    sni.phost_cs = pDomen_cs;
    snprintf(str, StrSize, "Check new %s %s %s", pDomen_cs, pDomen, sitenecinfo);
    //m_pstat.AddStat(ST_LOG) << str;
    if (ParseSiteNetcInfo(&sni, sitenecinfo)) {
        CheckDomenNameNew(rengine, &sni, pDomen, pDomen_cs, 0, false);
    }
    /*
  if (m_cur.fShFixedSource)
  {
      SetRule("__FIXED_SOURCE_SHINGLER");
      if (m_cur.level == 1 && !m_cur.fUndefinedIp &&
          !rengine.PerformedRule("__PRESENT_X_ORIGINATING_IP"))
          SetRule("__FIXED_SOURCE_LEVEL_SHINGLER_1");
  }
*/
}

bool TSpAlg::ParseSiteNetcInfo(TSiteNetcInfo* psni, const char* sitenecinfo) {
    int i;
    const char* pstr = sitenecinfo;
    int len = TSetRules::GetFirstWord(&pstr, 0);
    char pnetchex[9];
    const int StrSize = 350;
    char str[StrSize + 2];
    bool res = true;

    pnetchex[6] = '0';
    pnetchex[7] = '0';
    pnetchex[8] = 0;

    if (!len || !GetInt(pstr, len, &(psni->version)) || psni->version > 2)
        return false;

    len = TSetRules::GetNextWord(&pstr, 0);
    if (!len)
        return false;
    psni->w_yand = strtoul(pstr, nullptr, 10);

    if (psni->w_yand) {
    }

    len = TSetRules::GetNextWord(&pstr, 0);
    if (!len)
        return false;

    snprintf(str, StrSize, "%s", "Check ip new");
    for (i = 0; i < len / 6 && i < 43; i++) {
        memcpy(pnetchex, pstr + (i * 6), 6);
        char tmp = pnetchex[1];
        pnetchex[1] = pnetchex[5];
        pnetchex[5] = tmp;
        psni->netc[i] = strtoul(pnetchex, nullptr, 16);
        if (strlen(str) < StrSize - 50)
            snprintf(str + strlen(str), StrSize, " %ul", psni->netc[i]);
    }
    //if (strlen(str) < StrSize - 50)
    //    m_pstat.AddStat(ST_LOG) << str;
    psni->c_netc = i;

    if (psni->version == 0)
        return res;

    len = TSetRules::GetNextInt(&pstr, 0, &(psni->w_rul));
    if (!len)
        return res;
    len = TSetRules::GetNextInt(&pstr, 0, &(psni->w_dlv));
    if (!len)
        return res;
    len = TSetRules::GetNextInt(&pstr, 0, &(psni->max_shin));
    if (!len)
        return res;
    len = TSetRules::GetNextInt(&pstr, 0, &(psni->max_all));
    if (!len)
        return res;

    len = TSetRules::GetNextWord(&pstr, 0);
    if (!len)
        return res;

    strcpy(str, "bl=");
    for (i = 0; i < len && i < 15; i++)
        snprintf(str + strlen(str), StrSize, "%d", pstr[i]);
    m_pstat.AddStat(ST_LOG) << str;

    i = 0;
    if (pstr[i])
        psni->bl_arr[en_SR_HOST] = true;
    if (++i < len && pstr[i] == '0')
        psni->bl_arr[en_SR_MESSID] = false;
    if (++i < len && pstr[i] == '1')
        psni->bl_arr[en_SR_FROM] = true;
    if (++i < len && pstr[i] == '1')
        psni->bl_arr[en_SR_REPLYTO] = true;
    if (++i < len && pstr[i] == '0')
        psni->bl_arr[en_SR_SENDER] = false;
    if (++i < len && pstr[i] == '1')
        psni->bl_arr[en_SR_LIST] = true;
    if (++i < len && pstr[i] == '0')
        psni->bl_arr[en_SR_SUBJ] = false;
    if (++i < len && pstr[i] == '0')
        psni->bl_arr[en_SR_RBL] = false;
    if (++i < len && pstr[i] == '0')
        psni->bl_arr[en_SR_DSL] = false;
    if (++i < len && pstr[i] == '0')
        psni->bl_arr[en_SR_NRV] = false;
    if (++i < len && pstr[i] == '1')
        psni->bl_arr[en_SR_BL_NOT] = true;
    if (++i < len && pstr[i] == '1')
        psni->bl_arr[en_SR_L2ASL1] = false;

    len = TSetRules::GetNextWord(&pstr, 0);
    if (!len)
        return res;

    psni->c_host_rcvd = 0;
    for (i = 0; i < len / 16 && i < 15; i++) {
        memcpy(&(psni->host_rcvd[psni->c_host_rcvd++]), pstr + (i * 16), 16);
    }

    return res;
}

bool TSpAlg::IsFixedSource() const {
    return m_cur.fFixedSource;
}

void TSpAlg::AddStat(TSpFields fid, const TStringBuf& pfield) {
    m_faddprint = false;
    m_pstat.AddStat(fid) << pfield;
}

//###########################################################################

// From and To have same address, but are not exactly the same and
// neither contains intermediate spaces.
//
bool TSpAlg::CheckForFromToSame() {
    const auto& pFrom = m_rgpFields[FD_FROM];
    const auto& pTo = m_rgpFields[FD_TO];
    const auto& pFromAddr = m_rgpFields[FD_FROM_ADDR];
    const auto& pToAddr = m_rgpFields[FD_TO_ADDR];
    if (!pFromAddr || !pToAddr || pFrom == pTo)
        return false;

    char *pfrom_addr = 0, *pto_addr = 0;
    AssignField(pFromAddr.c_str(), pFromAddr.size(), &pfrom_addr);
    AssignField(pToAddr.c_str(), pToAddr.size(), &pto_addr);
    DelSpaces(pfrom_addr);
    DelSpaces(pto_addr);

    bool res = false;
    if (!strcmp(pfrom_addr, pto_addr) && m_pcre->Check("not_empty", m_rgpFields[FD_FROM_NAME]) &&
        m_pcre->Check("from_to_same", pFrom) && m_pcre->Check("from_to_same", pTo))
        res = true;

    DELETE_ARR(pfrom_addr);
    DELETE_ARR(pto_addr);

    return res;
}

void TSpAlg::AddAddrToCc(TRengine& rengine, TStringBuf field) {
    TStringBuf Pattern;

    TMaybe<NRegexp::TResult> res;
    m_rec.Init();
    for (int i = 0; i < RecMax && m_rec.c < RecMax; i++) {
        if (!(res = m_pcre->Check("recipient", field)) ||
            !res->GetPattern(1, Pattern))
            break;

        rengine.CheckRcptMail(TString{Pattern});

        if (m_rec.fSorted) {
            if (m_rec.c > 0 && Pattern < m_rec.LastAddr)
                m_rec.fSorted = false;
            else
                m_rec.LastAddr.assign(Pattern);
        }
        m_rec.rghost_2[m_rec.c][0] = 0;
        if (Pattern.size() > TOCC_SIMILAR_LENGTH - 1) {
            memcpy(m_rec.rguser[m_rec.c], Pattern.data(), TOCC_SIMILAR_LENGTH);

            const size_t p = Pattern.find('@');
            if (p != TStringBuf::npos)
            {
                const auto & domain = Pattern.substr(p + 1);
                if (domain.size() >= TOCC_SIMILAR_LENGTH)
                    memcpy(m_rec.rghost[m_rec.c], domain.data(), TOCC_SIMILAR_LENGTH);
                STRNCPY(m_rec.rghost_2[m_rec.c], domain.data(), std::min(size_t(TOCC_SIMILAR_LENGTH_2), domain.size()));
                m_rec.rghost_2[m_rec.c][TOCC_SIMILAR_LENGTH_2 - 1] = 0;
            }
        }
        m_rec.c++;

        if (!(field = res->GetRestPattern()) || field.size() < 3)
            break;
    }
    CheckRecipients();
}

// best experimentally derived values
//use constant TOCC_SORTED_COUNT => 7;
//use constant TOCC_SIMILAR_COUNT => 5;
//use constant TOCC_SIMILAR_LENGTH => 2;

// sub _check_recipients
void TSpAlg::CheckRecipients() {
    const int TOCC_SORTED_COUNT = 7;
    const int TOCC_SIMILAR_COUNT = 5;

    if (m_rec.c > TOCC_SORTED_COUNT && m_rec.fSorted)
        SetRule("SORTED_RECIPS");

    if (m_rec.c < TOCC_SIMILAR_COUNT)
        return;

    int hits = 0;
    int combinations = 0;

    for (int i = 0; i < m_rec.c; i++) {
        for (int j = i + 1; j < m_rec.c; j++) {
            if (!memcmp(m_rec.rguser[i], m_rec.rguser[j], TOCC_SIMILAR_LENGTH))
                hits++;
            if (!memcmp(m_rec.rghost[i], m_rec.rghost[j], TOCC_SIMILAR_LENGTH) &&
                strcmp(m_rec.rghost_2[i], m_rec.rghost_2[j]))
                hits++;
            combinations++;
        }
    }

    double cToCcSimilar = ((double)hits) / combinations;

    if (cToCcSimilar >= 0.6 && cToCcSimilar < 1.2)
        SetRule("SUSPICIOUS_RECIPS");
    else if (cToCcSimilar >= 1.2)
        SetRule("VERY_SUSP_RECIPS");

    //    my $to = $self->get('ToCc');    get all recipients
    //  $to =~ s/\(.*?\)//g;        strip out the (comments)
    //  my @address = ($to =~ m/([\w.=-]+\@\w+(?:[\w.-]+\.)+\w+)/g);

    //   ideas that had both poor S/O ratios and poor hit rates:
    //   - testing for reverse sorted recipient lists
    //   - testing To: and Cc: headers separately
    //  $self->{tocc_sorted} = (scalar(@address) >= TOCC_SORTED_COUNT &&
    //              join(',', @address) eq (join(',', sort @address)));

    //   a good S/O ratio and hit rate is achieved by comparing 2-byte
    //   substrings and requiring 5 or more addresses
    //  $self->{tocc_similar} = 0;
    //  if (scalar (@address) >= TOCC_SIMILAR_COUNT) {
    //    my @user = map { substr($_,0,TOCC_SIMILAR_LENGTH) } @address;
    //    my @fqhn = map { m/\@(.*)/ } @address;
    //    my @host = map { substr($_,0,TOCC_SIMILAR_LENGTH) } @fqhn;
    //    my $hits = 0;
    //    my $combinations = 0;
    //    for (my $i = 0; $i <= $#address; $i++) {
    //      for (my $j = $i+1; $j <= $#address; $j++) {
    //    $hits++ if $user[$i] eq $user[$j];
    //    $hits++ if $host[$i] eq $host[$j] && $fqhn[$i] ne $fqhn[$j];
    //    $combinations++;
    //      }
    //    }
    //    $self->{tocc_similar} = $hits / $combinations;
    //  }
}

//###########################################################################

// The MTA probably added the Message-ID if either of the following is true:
//
// (1) The Message-ID: comes before a Received: header.
//
// (2) The Message-ID is the first header after all Received headers and
//     the From address domain is not the same as the Message-ID domain and
//     the Message-ID domain matches the last Received "by" domain.
//
// These two tests could be combined into a single rule, but they are
// separated because the first test is more accurate than the second test.
// However, we only run the primary function once for better performance.

bool TSpAlg::GetHost(const TStringBuf& pfiled, const TStringBuf& re, const TStringBuf& re_prepare, TString& phost) {
    TStringBuf pattern;
    if ((pattern = m_pcre->GetPattern(re, pfiled, 1)) &&
        (pattern = m_pcre->GetPattern(re_prepare, pattern, 1))) {
        phost = to_lower(TString{pattern});
        return true;
    }
    return false;
}

void TSpAlg::CheckMtaMessageId() {
    // Yahoo! and Wanadoo.fr do add their Message-Id on transport time:
    // Yahoo! MIDs can depend on the country: yahoo.com, yahoo.fr, yahoo.co.uk, etc.
    // Wanadoo MIDs end always in wanadoo.fr


    // no further checks in simple case
    if (m_cur.fMessageId && m_vReceived && m_cur.fLaterMta && !m_pcre->Check("mta_yahoo", m_rgpFields[FD_MESSAGEID])) {
        SetRule("MSG_ID_ADDED_BY_MTA_2");
    }
}

void TSpAlg::CheckSender() {
    const char* pFromAddr = m_rgpFields[FD_FROM_ADDR].c_str();
    const char* pSender = m_rgpFields[FD_SENDER].c_str();
    const char* pReplyTo = m_rgpFields[FD_REPLY_TO_ADDR].c_str();

    TString pHostSender;
    TString pHostFrom;
    TString pHostReplyTo;

    if (!(*pFromAddr) ||
        !GetHost(pFromAddr, "mta_hostfrom", "prepare_from", pHostFrom) ||
        !pHostFrom)
        return;

    if (*pSender && GetHost(pSender, "mta_hostfrom", "prepare_from", pHostSender) &&
        pHostSender && pHostSender != pHostFrom)
        SetRule("MSG_SENDER_CHECK");

    if (*pReplyTo && GetHost(pReplyTo, "mta_hostfrom", "prepare_from", pHostReplyTo) &&
        pHostReplyTo && pHostReplyTo != pHostFrom)
        SetRule("MSG_REPLYTO_CHECK");
}

//###########################################################################

// yet another test for faked Received: headers (FORGED_RCVD_TRAIL).

bool TSpAlg::CheckForForgedReceivedTrail(int* mismatch_addr) {
    std::vector<TString>::iterator vi;
    int mismatch = 0;
    bool fFirst = true;
    TStringBuf pattern;
    char* pRecFrom = 0;
    char* pRecBy = 0;
    char* pRecFrom_1 = 0;
    char* pRecBy_1 = 0;
    InitField(&pRecFrom);
    InitField(&pRecBy);
    InitField(&pRecFrom_1);
    InitField(&pRecBy_1);

    *mismatch_addr = 0;
    for (vi = m_vReceived.begin(); vi != m_vReceived.end() && mismatch < 2; vi++) {
        if (!fFirst && *pRecFrom) {
            if (pattern = m_pcre->GetPattern("by_received", *vi, 1)) {
                AssignField(pattern.data(), pattern.size(), &pRecBy);
                if (stricmp(pRecBy, pRecFrom)) {
                    (*mismatch_addr)++;
                    if (pattern = m_pcre->GetPattern("prepare_received", TStringBuf(pRecBy), 1))
                        AssignField(pattern.data(), pattern.size(), &pRecBy_1);
                    {
                        if (pattern = m_pcre->GetPattern("prepare_received", TStringBuf(pRecFrom), 1)) {
                            AssignField(pattern.data(), pattern.size(), &pRecFrom_1);
                            if (*pRecBy_1 && *pRecFrom_1 && stricmp(pRecBy_1, pRecFrom_1))
                                mismatch++;
                        }
                    }
                }
            }
        }
        fFirst = false;
        InitField(&pRecFrom);
        if (vi != m_vReceived.end() - 1) {
            if (pattern = m_pcre->GetPattern("ehlo_received", *vi, 1))
                AssignField(pattern.data(), pattern.size(), &pRecFrom);
            else if (pattern = m_pcre->GetPattern("from_received", *vi, 1))
                AssignField(pattern.data(), pattern.size(), &pRecFrom);

            // valid: bouncing around inside 1 machine, via the localhost interface.
            // freshmeat newsletter does this.
            if (*pRecFrom && m_pcre->Check("from_localhost", TStringBuf(pRecFrom)) &&
                (pattern = m_pcre->GetPattern("ip_address", *vi, 1)) &&
                !strncmp(pattern.data(), "127.0.0.1", pattern.size()))
                InitField(&pRecFrom);
        }
    }

    DELETE_ARR(pRecFrom);
    DELETE_ARR(pRecBy);
    DELETE_ARR(pRecFrom_1);
    DELETE_ARR(pRecBy_1);
    return (mismatch > 1);
}

// FORGED_HOTMAIL_RCVD
void TSpAlg::CheckForForgedHotmailReceivedHeaders() {
    //        /from mail pickup service by hotmail\.com with Microsoft SMTPSVC;/);

    const char* pFromAddr = m_rgpFields[FD_FROM_ADDR].c_str();
    if (m_pcre->Check("hotmail_smtpsvc", m_sReceived))
        return;

    //Hotmail formats its received headers like this:
    // Received: from hotmail.com (f135.law8.hotmail.com [216.33.241.135])
    // spammers do not ;)

    if (m_cur.fGatedThrough)
        return;
    if (m_cur.fXOriginatingIp) {
        if (m_pcre->Check("hotmail_4", m_sRelay) ||
            m_pcre->Check("hotmail_1", m_sReceived) ||
            m_pcre->Check("hotmail_2", m_sReceived) ||
            m_pcre->Check("hotmail_3", m_sReceived))
            return;
    }

        //  if (m_pcre->Check("hotmail_helo", m_sReceived))
        //      m_cur.fHotmailAddrWithForgedHotmailReceived = true;
    else if (m_pcre->Check("hotmail_from", TStringBuf(pFromAddr)))
        m_cur.fHotmailAddrButNoHotmailReceived = true;
}

//###########################################################################

bool TSpAlg::CheckForFakeAolRelayInRcvd() {
    // this is the hostname format used by AOL for their relays. Spammers love
    // forging it.  Don't make it more specific to match aol.com only, though --
    // there's another set of spammers who generate fake hostnames to go with it

    if (m_pcre->Check("rly", m_sReceived)) {
        if (m_pcre->Check("rly_aol", m_sReceived))
            return false;
        if (m_pcre->Check("rly_relay", m_sReceived))
            return false;
        return true;
    }
    return false;

    // spam: Received: from unknown (HELO mta05bw.bigpond.com) (80.71.176.130) by
    //    rly-xw01.mx.aol.com with QMQP; Sat, 15 Jun 2002 23:37:16 -0000

    // non: Received: from  rly-xj02.mx.aol.com (rly-xj02.mail.aol.com [172.20.116.39]) by
    //    omr-r05.mx.aol.com (v83.35) with ESMTP id RELAYIN7-0501132011; Wed, 01
    //    May 2002 13:20:11 -0400

    // non: Received: from logs-tr.proxy.aol.com (logs-tr.proxy.aol.com [152.163.201.132])
    //    by rly-ip01.mx.aol.com (8.8.8/8.8.8/AOL-5.0.0)
    //    with ESMTP id NAA08955 for <sapient-alumni@yahoogroups.com>;
    //    Thu, 4 Apr 2002 13:11:20 -0500 (EST)
}

bool TSpAlg::CheckForForgedEudoraMailReceivedHeaders() {
    if (m_cur.fGatedThrough)
        return false;
    ;
    if (!m_pcre->Check("eudoramail", m_rgpFields[FD_FROM_ADDR]))
        return false;
    if (m_cur.fXSenderIp && m_pcre->Check("whowhere", m_sReceived))
        return false;

    return true;

    //Eudoramail formats its received headers like this:
    // Received: from Unknown/Local ([?.?.?.?]) by shared1-mail.whowhere.com;
    //      Thu Nov 29 13:44:25 2001
    // Message-Id: <JGDHDEHPPJECDAAA@shared1-mail.whowhere.com>
    // Organization: QUALCOMM Eudora Web-Mail  (http://www.eudoramail.com:80)
    // X-Sender-Ip: 192.175.21.146
    // X-Mailer: MailCity Service
}

//###########################################################################

bool TSpAlg::CheckForForgedYahooReceivedHeaders() {
    const auto& pFromAddr = m_rgpFields[FD_FROM_ADDR];

    if (!m_pcre->Check("yahoo_4", m_sRelay))
        return false;

    if (m_cur.fGatedThrough)
        return false;
    if (!m_pcre->Check("yahoo_com", pFromAddr))
        return false;
    if (m_pcre->Check("yahoo_web", m_sReceived) ||
        m_pcre->Check("yahoo_smtp", m_sReceived) ||
        m_pcre->Check("yahoo_ip", m_sReceived))
        return false;

    //   search for msgid <20020929140301.451A92940A9@xent.com>, subject "Yahoo!
    //
    if (m_pcre->Check("yahoo_mailer", m_sReceived) && m_pcre->Check("yahoo_reply", pFromAddr))
        return false;
    if (m_pcre->Check("yahoo_friend", m_sReceived))
        return false;
    return true;
}

bool TSpAlg::CheckForForgedJunoReceivedHeaders() {
    const auto& pFromAddr = m_rgpFields[FD_FROM_ADDR];
    const auto& pXMailer = m_rgpFields[FD_X_MAILER];

    if (m_cur.fGatedThrough)
        return false;
    ;
    if (!m_pcre->Check("juno_com", pFromAddr))
        return false;
    if (!m_cur.fXOriginatingIpExists) {
        if (!m_pcre->Check("juno_from", m_sReceived) &&
            !m_pcre->Check("juno_cookie", m_sReceived))
            return true;
        if (!m_pcre->Check("juno_mailer", pXMailer))
            return true;
    } else {
        if (!m_pcre->Check("juno_from_2", m_sReceived))
            return true;
        if (!m_cur.fXOriginatingIp)
            return true;
        if (!m_pcre->Check("juno_mailer_2", pXMailer))
            return true;
    }

    return false;
}

//###########################################################################

bool TSpAlg::CheckForForgedGw05ReceivedHeaders() {
    // e.g.
    // Received: from mail3.icytundra.com by gw05 with ESMTP; Thu, 21 Jun 2001 02:28:32 -0400
    TStringBuf pattern;
    auto res = m_pcre->Check("gw05", m_sReceived);
    return  res &&
            res->GetPattern(1, pattern) &&
            res->GetPattern(2, pattern) &&
            !m_pcre->Check("gw05_dot", pattern);

}

//###########################################################################
// Some spammers will, through HELO, tell the server that their machine
// name *is* the relay; don't know why. An example:

// from mail1.mailwizards.com (m448-mp1.cvx1-b.col.dial.ntli.net
//        [213.107.233.192])
//        by mail1.mailwizards.com

// When this occurs for real, the from name and HELO name will be the
// same, unless the "helo" name is localhost, or the from and by hostsnames
// themselves are localhost
bool TSpAlg::CheckReceivedHelos() {
    TStringBuf ip, helo_host, from_host, by_host;
    int from_ind{}, helo_ind{}, by_ind{};

    for (const auto& prec : m_vReceived) {
        // Ignore where HELO is in reserved IP space; regexp matches
        // "[W.X.Y.Z]" immediatly followed by a ")", which should only
        // sho up at the end of the HELO part of a Received header

        TMaybe<NRegexp::TResult> res;
        if ((res = m_pcre->Check("helos_ip", prec)) &&
            (ip = res->GetPattern(1)) &&
            (res = m_pcre->Check("helos_ip_reserved", ip)))
            continue;

        // $helo_host regexp is "([\w.-]+\.[\w.-]+)" so that at least
        // one "." must be present, thus avoiding domainless hostnames
        // and "(HELO hostname)" situations.
        //
        // $from_host and $by_host regexps are "([\w.-]+)" to exclude
        // things like "[1.2.3.4]"; we don't deal with numeric-only
        // addresses
        // Exim: from ns.egenix.com ([217.115.138.139] helo=www.egenix.com) by
        // mail.python.org with esmtp (Exim 4.05) id 1829w0-0007uf-00; Thu, 17 Oct
        if (res = m_pcre->Check("helos_1", prec)) {
            from_ind = 1;
            helo_ind = 2;
            by_ind = 3;
        } else if (res = m_pcre->Check("helos_2", prec)) {
            // qmail: from 64-251-145-11-cablemodem-roll.fidnet.com (HELO gabriels)
            // (64.251.145.11) by three.fidnet.com with SMTP; 4 Dec 2002 16:01:35 -0000
            from_ind = 1;
            helo_ind = 2;
            by_ind = 3;
        } else if (res = m_pcre->Check("helos_3", prec)) {
            // Received: from ralph.jamiemccarthy.com ([65.88.171.80]) by red.harvee.home
            // (8.11.6/8.11.6) with ESMTP id gB4KuQ130187 for <zzzzzzzz@tb.tf>;
            // Wed, 4 Dec 2002 15:56:27 -0500   [helo = from == good]
            from_ind = 1;
            helo_ind = 1;
            by_ind = 2;
        } else if (res = m_pcre->Check("helos_4", prec)) {
            // I'm pretty sure from and HELO were the wrong way around here.  e.g.  in
            // "from lycos.co.uk (newwww-37.st1.spray.net [212.78.202.47]) by
            // outmail-3.st1.spray.net", the HELO is 'lycos.co.uk', NOT
            // 'newwww-37.st1.spray.net' -- the latter is from reverse DNS, and is
            // therefore trustworthy, whereas HELO is not.
            from_ind = 2;
            helo_ind = 1;
            by_ind = 3;
        } else
            continue;
        if (!(helo_host = res->GetPattern(helo_ind)))
            continue;
        if (!(from_host = res->GetPattern(from_ind)))
            continue;
        if (!(by_host = res->GetPattern(by_ind)))
            continue;

        // Check for a faked dotcom HELO, e.g.
        // Received: from mx02.hotmail.com (www.sucasita.com.mx [148.223.251.99])...
        // this can be a stronger spamsign than the normal case, since the
        // big dotcoms don't screw up their rDNS normally ;), so less FPs.
        // Since spammers like sending out their mails from the dotcoms (esp.
        // hotmail and AOL) this will catch those forgeries.
        //
        // allow stuff before the dot-com for both from-name and HELO-name,
        // so HELO="outgoing.aol.com" and from="mx34853495.mx.aol.com" works OK.
        //
        TStringBuf dom;
        if ((dom = m_pcre->GetPattern("helos_5", helo_host, 1)) &&
            !m_pcre->Check("ip_address", from_host)) {
            if (dom && (dom.size() > from_host.size() || !from_host.EndsWith(dom)))
                return true;
        }
    }
    return false;
}

//###########################################################################

void TSpAlg::CheckForToInSubject() {
    const char* pSubject = m_rgpFields[FD_SUBJECT].c_str();
    const char* pToAddr = m_rgpFields[FD_TO_ADDR].c_str();

    if (!(*pToAddr) || !(*pSubject))
        return;

    const char* p_at = strchr(pToAddr, '@');
    if (!p_at)
        return;

    int at_len = p_at - pToAddr;
    char* pAddr = 0;
    AssignField(pToAddr, strlen(pToAddr), &pAddr);
    strlwr(pAddr);
    pAddr[at_len] = 0;

    int i = 0;
    while (IsSpace(pSubject[i]))
        i++;
    char* pSubj = 0;
    AssignField(pSubject + i, strlen(pSubject + i), &pSubj);
    strlwr(pSubj);

    int len = strlen(pAddr);
    if (!strncmp(pSubj, pAddr, len) && pSubj[len] == ',') // alex, ...
        SetRule("USERNAME_IN_SUBJECT");
    else if (strstr(pSubj, pAddr))
        SetRule("USERNAME_IN_SUBJECT_2");         // ... alex...
    else if (strstr(pSubj, pToAddr + at_len + 1)) // .. yandex-team.ru ..
        SetRule("ADDRESS_TO_IN_SUBJECT");

    DELETE_ARR(pAddr);
    DELETE_ARR(pSubj);

    return;
}

void TSpAlg::SetRule(const TStringBuf& szname) {
    m_curMessageContext->rulesContext.SetRule(szname);
}

//---------------------------------------------------------------------------

void TSpAlg::GetReceivedNext(const TStringBuf& prcvd, TString& ppIp, TString& ppRdns, TString& ppHelo, bool fPrintStat) {
    TStringBuf psubj;
    int mta_looked_up_dns = 0;
    TString rec_type;

    ppHelo = "-";
    ppRdns = "-";
    ppIp = "-";

    // Received: (qmail 27981 invoked by uid 225); 14 Mar 2003 07:24:34 -0000
    // Received: (qmail 84907 invoked from network); 13 Feb 2003 20:59:28 -0000
    // Received: (ofmipd 208.31.42.38); 17 Mar 2003 04:09:01 -0000
    // we don't care about this kind of gateway noise

    if (prcvd.StartsWith(' '))
        return;

    //   OK -- given knowledge of most Received header formats,
    //   break them down.  We have to do something like this, because
    //   some MTAs will swap position of rdns and helo -- so we can't
    //   simply use simplistic regexps.
    TMaybe<NRegexp::TResult> res;
    if (m_pcre->Check("rec_postfix", prcvd)) {
        if (res = m_pcre->Check("rec_postfix1", prcvd)) {
            mta_looked_up_dns = 1;
            ppHelo = res->GetPattern(1);
            ppRdns = res->GetPattern(2);
            ppIp = res->GetPattern(3);
            rec_type.assign("rec_postfix1");
            goto enough;
        }

        //      Received: from 207.8.214.3 (unknown[211.94.164.65])
        //      by puzzle.pobox.com (Postfix) with SMTP id 9029AFB732;
        //      Sat,  8 Nov 2003 17:57:46 -0500 (EST)
        //      (Pobox.com version: reported in bug 2745)
    }

    if (m_pcre->Check("rec_exim", prcvd)) {
        //       one of the HUGE number of Exim formats :(
        //       This must be scriptable.

        //      Received: from [61.174.163.26] (helo=host) by sc8-sf-list1.sourceforge.net with smtp (Exim 3.31-VA-mm2 #1 (Debian)) id 18t2z0-0001NX-00 for <razor-users@lists.sourceforge.net>; Wed, 12 Mar 2003 01:57:10 -0800
        //       Received: from [218.19.142.229] (helo=hotmail.com ident=yiuhyotp) by yzordderrex with smtp (Exim 3.35 #1 (Debian)) id 194BE5-0005Zh-00; Sat, 12 Apr 2003 03:58:53 +0100
        if (res = m_pcre->Check("rec_exim1", prcvd)) {
            ppIp = res->GetPattern(1);
            if (psubj = res->GetPattern(2))
                ppHelo = m_pcre->GetPattern("rec_helo", psubj, 1);
            rec_type.assign("rec_exim1");
            goto enough;
        }

        //      Received: from sc8-sf-list1-b.sourceforge.net ([10.3.1.13] helo=sc8-sf-list1.sourceforge.net) by sc8-sf-list2.sourceforge.net with esmtp (Exim 3.31-VA-mm2 #1 (Debian)) id 18t301-0007Bh-00; Wed, 12 Mar 2003 01:58:13 -0800
        //      Received: from dsl092-072-213.bos1.dsl.speakeasy.net ([66.92.72.213] helo=blazing.arsecandle.org) by sc8-sf-list1.sourceforge.net with esmtp (Cipher TLSv1:DES-CBC3-SHA:168) (Exim 3.31-VA-mm2 #1 (Debian)) id 18lyuU-0007TI-00 for <SpamAssassin-talk@lists.sourceforge.net>; Thu, 20 Feb 2003 14:11:18 -0800
        //      Received: from eclectic.kluge.net ([66.92.69.221] ident=[W9VcNxE2vKxgWHD05PJbLzIHSxcmZQ/O]) by sc8-sf-list1.sourceforge.net with esmtp (Cipher TLSv1:DES-CBC3-SHA:168) (Exim 3.31-VA-mm2 #1 (Debian)) id 18m0hT-00031I-00 for <spamassassin-talk@lists.sourceforge.net>; Thu, 20 Feb 2003 16:06:00 -0800
        if (res = m_pcre->Check("rec_exim2", prcvd)) {
            ppRdns = res->GetPattern(1);
            ppIp = res->GetPattern(2);
            ppHelo = res->GetPattern(3);
            rec_type.assign("rec_exim2");
            goto enough;
        }

        //      Received: from mail.ssccbelen.edu.pe ([216.244.149.154]) by yzordderrex
        //      with esmtp (Exim 3.35 #1 (Debian)) id 18tqiz-000702-00 for
        //      <jm@example.com>; Fri, 14 Mar 2003 15:03:57 +0000
        if (res = m_pcre->Check("rec_exim3", prcvd)) {
            ppRdns = res->GetPattern(1);
            ppIp = res->GetPattern(2);
            ppHelo = res->GetPattern(1);
            rec_type.assign("rec_exim3");
            goto enough;
        }
        if (res = m_pcre->Check("rec_exim4", prcvd)) {
            ppRdns = res->GetPattern(1);
            ppIp = res->GetPattern(2);
            ppHelo = res->GetPattern(1);
            rec_type.assign("rec_exim4");
            goto enough;
        }
        if (m_pcre->Check("rec_exim5", prcvd)) {
            rec_type.assign("rec_exim5");
            goto enough;
        }

        //     Received: from boggle.ihug.co.nz [203.109.252.209] by grunt6.ihug.co.nz
        //     with esmtp (Exim 3.35 #1 (Debian)) id 18SWRe-0006X6-00; Sun, 29 Dec
        //     2002 18:57:06 +1300

        if (fPrintStat)
            m_pstat.AddStat(ST_LOG) << "unknown exim type";
    }

    //    Received: from ns.elcanto.co.kr (66.161.246.58 [66.161.246.58]) by
    //    mail.ssccbelen.edu.pe with SMTP (Microsoft Exchange Internet Mail Service
    //    Version 5.5.1960.3) id G69TW478; Thu, 13 Mar 2003 14:01:10 -0500

    if (res = m_pcre->Check("rec_meims", prcvd)) {
        ppHelo = res->GetPattern(1);
        ppRdns = res->GetPattern(2);
        ppIp = res->GetPattern(3);
        rec_type.assign("rec_meims");
        goto enough;
    }

    if (res = m_pcre->Check("rec_zmailer1", prcvd)) {
        ppHelo = res->GetPattern(3);
        ppRdns = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_zmailer1");
        goto enough;
    }

    //    from mail2.detr.gsi.gov.uk ([51.64.35.18] helo=ahvfw.dtlr.gsi.gov.uk)
    //    by mail4.gsi.gov.uk with smtp id 190K1R-0000me-00

    if (res = m_pcre->Check("rec_11", prcvd)) {
        ppRdns = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        psubj = res->GetPattern(3);
        ppHelo = m_pcre->GetPattern("rec_helo", psubj, 1);
        rec_type.assign("rec_11");
        goto enough;
    }

    //   from 12-211-5-69.client.attbi.com (<unknown.domain>[12.211.5.69]) by rwcrmhc53.attbi.com (rwcrmhc53) with SMTP id <2002112823351305300akl1ue>; Thu, 28 Nov 2002 23:35:13 +0000
    if (res = m_pcre->Check("rec_12", prcvd)) {
        ppHelo = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_12");
        goto enough;
    }

    //    from attbi.com (h000502e08144.ne.client2.attbi.com[24.128.27.103]) by rwcrmhc53.attbi.com (rwcrmhc53) with SMTP id <20030222193438053008f7tee>; Sat, 22 Feb 2003 19:34:39 +0000
    if (res = m_pcre->Check("rec_13", prcvd)) {
        mta_looked_up_dns = 1;
        ppHelo = res->GetPattern(1);
        ppRdns = res->GetPattern(2);
        ppIp = res->GetPattern(3);
        rec_type.assign("rec_13");
        goto enough;
    }

    //    sendmail:
    //    Received: from mail1.insuranceiq.com (host66.insuranceiq.com [65.217.159.66] (may be forged)) by dogma.slashnull.org (8.11.6/8.11.6) with ESMTP id h2F0c2x31856 for <jm@jmason.org>; Sat, 15 Mar 2003 00:38:03 GMT
    //    Received: from BAY0-HMR08.adinternal.hotmail.com (bay0-hmr08.bay0.hotmail.com [65.54.241.207]) by dogma.slashnull.org (8.11.6/8.11.6) with ESMTP id h2DBpvs24047 for <webmaster@efi.ie>; Thu, 13 Mar 2003 11:51:57 GMT
    //    Received: from ran-out.mx.develooper.com (IDENT:qmailr@one.develooper.com [64.81.84.115]) by dogma.slashnull.org (8.11.6/8.11.6) with SMTP id h381Vvf19860 for <jm-cpan@jmason.org>; Tue, 8 Apr 2003 02:31:57 +0100
    //    Received: from rev.net (natpool62.rev.net [63.148.93.62] (may be forged)) (authenticated) by mail.rev.net (8.11.4/8.11.4) with ESMTP id h0KKa7d32306 for <spamassassin-talk@lists.sourceforge.net>
    if (res = m_pcre->Check("rec_sendmail", prcvd)) {
        mta_looked_up_dns = 1;
        ppHelo = res->GetPattern(1);
        ppRdns = res->GetPattern(2);
        ppIp = res->GetPattern(3);
        rec_type.assign("rec_sendmail");
        goto enough;
    }

    //    Received: from 87.98.253.105 ([87.98.253.105]) (authenticated bits=0) by apteka-ifk.ru (8.14.7/8.14.5) with ESMTP id tBABDKCT099911 for <elaizina@yandex.ru>; Thu, 10 Dec 2015 15:13:21 +0400 (MSK) (envelope-from OVDabaza@mail.ru)
    if (res = m_pcre->Check("rec_sendmail-8-14", prcvd)) {
        mta_looked_up_dns = 1;
        ppHelo = res->GetPattern(1);
        ppRdns = res->GetPattern(2);
        ppIp = res->GetPattern(3);
        rec_type.assign("rec_sendmail-8-14");
        goto enough;
    }

    //      Received: from localhost (unknown [127.0.0.1])
    //      by cabbage.jmason.org (Postfix) with ESMTP id A96E18BD97
    //      for <jm@localhost>; Thu, 13 Mar 2003 15:23:15 -0500 (EST)
    //    Received: from 213.123.174.21 by lw11fd.law11.hotmail.msn.com with HTTP;
    //    Wed, 24 Jul 2002 16:36:44 GMT
    if (res = m_pcre->Check("rec_msn", prcvd)) {
        ppIp = res->GetPattern(1);
        rec_type.assign("rec_msn");
        goto enough;
    }

    //    Moved some tests up because they might match on qmail tests, where this
    //    is not qmail
    //    #
    //    Received: from imo-m01.mx.aol.com ([64.12.136.4]) by eagle.glenraven.com
    //    via smtpd (for [198.85.87.98]) with SMTP; Wed, 08 Oct 2003 16:25:37 -0400
    if (res = m_pcre->Check("rec_via", prcvd)) {
        ppHelo = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_via");
        goto enough;
    }

    //  ecelerity MTA, aka MetaMail
    //   from [10.156.251.33] ([10.156.251.33:48169]) by smout018.07.ash2.facebook.com (envelope-from <update+zrdol6ldeppf@facebookmail.com>)
    //        (ecelerity 3.6.0.39694 r(Platform:3.6.0.0))
    //        with ECSTREAM id 5F/17-61555-FEFF0E25; Thu, 23 Jan 2014 03:41:35 -0800
    if (res = m_pcre->Check("rec_ecelerity", prcvd)) {
        ppHelo = res->GetPattern(3);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_ecelerity");
        goto enough;
    }

    //    Try to match most of various qmail possibilities
    //
    //    General format:
    //    Received: from postfix3-2.free.fr (HELO machine.domain.com) (foobar@213.228.0.169) by totor.bouissou.net with SMTP; 14 Nov 2003 08:05:50 -0000
    //    "from (remote.rDNS|unknown)" is always there
    //    "(HELO machine.domain.com)" is there only if HELO differs from remote rDNS
    //    "foobar@" is remote IDENT info, specified only if ident given by remote
    //    Remote IP always appears between (parentheses), with or without IDENT@
    //    "by local.system.domain.com" always appears
    //
    //    Protocol can be different from "SMTP", i.e. "RC4-SHA encrypted SMTP" or "QMQP"
    //    qmail's reported protocol shouldn't be "ESMTP", so by allowing only "with (.* )(SMTP|QMQP)"
    //    we should avoid matching on some sendmailish Received: lines that reports remote IP
    //    between ([218.0.185.24]) like qmail-ldap does, but use "with ESMTP".
    //
    //    Normally, qmail-smtpd remote IP isn't between square brackets [], but some versions of
    //    qmail-ldap seem to add square brackets around remote IP. These versions of qmail-ldap
    //    use a longer format that also states the (envelope-sender <sender@domain>) and the
    //    qmail-ldap version. Example:
    //    Received: from unknown (HELO terpsichore.farfalle.com) (jdavid@[216.254.40.70]) (envelope-sender <jdavid@farfalle.com>) by mail13.speakeasy.net (qmail-ldap-1.03) with SMTP for <jm@jmason.org>; 12 Feb 2003 18:23:19 -0000
    //    Some others of the numerous qmail patches out there can also add variants of their own
    //
    if (res = m_pcre->Check("rec_qmail", prcvd)) {
        if (res = m_pcre->Check("rec_qmail1", prcvd)) {
            ppRdns = res->GetPattern(1);
            ppHelo = res->GetPattern(2);
            ppIp = res->GetPattern(4);
            rec_type.assign("rec_qmail1");
        } else if (res = m_pcre->Check("rec_qmail2", prcvd)) {
            ppRdns = res->GetPattern(1);
            ppHelo = res->GetPattern(2);
            ppIp = res->GetPattern(3);
            rec_type.assign("rec_qmail2");
        } else if (res = m_pcre->Check("rec_qmail3", prcvd)) {
            ppRdns = res->GetPattern(1);
            ppIp = res->GetPattern(3);
            rec_type.assign("rec_qmail3");
        } else if (res = m_pcre->Check("rec_qmail4", prcvd)) {
            ppRdns = res->GetPattern(1);
            ppIp = res->GetPattern(2);
            rec_type.assign("rec_qmail4");
        }
        //      qmail doesn't perform rDNS requests by itself, but is usually called
        //      by tcpserver or a similar daemon that passes rDNS information to qmail-smtpd.
        //      If qmail puts something else than "unknown" in the rDNS field, it means that
        //      it received this information from the daemon that called it. If qmail-smtpd
        //      writes "Received: from unknown", it means that either the remote has no
        //      rDNS, or qmail was called by a daemon that didn't gave the rDNS information.
        goto enough;
    }

    //     Received: from [193.220.176.134] by web40310.mail.yahoo.com via HTTP;
    //     Wed, 12 Feb 2003 14:22:21 PST
    //    if (/^from \[(${IP_ADDRESS})\] by (\S+) via HTTP\;/) {
    //      $ip = $1; $by = $2; goto enough;
    //    }

    //    Received: from 192.168.5.158 ( [192.168.5.158]) as user jason@localhost by mail.reusch.net with HTTP; Mon, 8 Jul 2002 23:24:56 -0400

    if (res = m_pcre->Check("rec_14", prcvd)) {
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_14");
        goto enough;
    }

    //    Received: from (64.52.135.194 [64.52.135.194]) by mail.unearthed.com with ESMTP id BQB0hUH2 Thu, 20 Feb 2003 16:13:20 -0700 (PST)

    if (res = m_pcre->Check("rec_15", prcvd)) {
        ppHelo = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_15");
        goto enough;
    }

    //    Received: from [65.167.180.251] by relent.cedata.com (MessageWall 1.1.0) with SMTP; 20 Feb 2003 23:57:15 -0000

    if (res = m_pcre->Check("rec_16", prcvd)) {
        ppIp = res->GetPattern(1);
        rec_type.assign("rec_16");
        goto enough;
    }

    //    Received: from acecomms [202.83.84.95] by mailscan.acenet.net.au [202.83.84.27] with SMTP (MDaemon.PRO.v5.0.6.R) for <spamassassin-talk@lists.sourceforge.net>; Fri, 21 Feb 2003 09:32:27 +1000

    if (res = m_pcre->Check("rec_17", prcvd)) {
        mta_looked_up_dns = 1;
        ppHelo = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_17");
        goto enough;
    }

    //    Received: from mail.sxptt.zj.cn ([218.0.185.24]) by dogma.slashnull.org
    //    (8.11.6/8.11.6) with ESMTP id h2FH0Zx11330 for <webmaster@efi.ie>;
    //    Sat, 15 Mar 2003 17:00:41 GMT

    if (res = m_pcre->Check("rec_18", prcvd)) {
        mta_looked_up_dns = 1;
        ppHelo = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_18");
        goto enough;
    }

    //    Received: from umr-mail7.umr.edu (umr-mail7.umr.edu [131.151.1.64]) via ESMTP by mrelay1.cc.umr.edu (8.12.1/) id h06GHYLZ022481; Mon, 6 Jan 2003 10:17:34 -0600
    //    Received: from Agni (localhost [::ffff:127.0.0.1]) (TLS: TLSv1/SSLv3, 168bits,DES-CBC3-SHA) by agni.forevermore.net with esmtp; Mon, 28 Oct 2002 14:48:52 -0800
    //    Received: from gandalf ([4.37.75.131]) (authenticated bits=0) by herald.cc.purdue.edu (8.12.5/8.12.5/herald) with ESMTP id g9JLefrm028228 for <spamassassin-talk@lists.sourceforge.net>; Sat, 19 Oct 2002 16:40:41 -0500 (EST)
    if (res = m_pcre->Check("rec_19", prcvd)) {
        mta_looked_up_dns = 1;
        ppHelo = res->GetPattern(1);
        ppRdns = res->GetPattern(2);
        ppIp = res->GetPattern(3);
        rec_type.assign("rec_19");
        goto enough;
    }

    if (res = m_pcre->Check("rec_20", prcvd)) {
        mta_looked_up_dns = 1;
        ppHelo = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_20");
        goto enough;
    }

    //    Received: from roissy (p573.as1.exs.dublin.eircom.net [159.134.226.61])
    //    (authenticated bits=0) by slate.dublin.wbtsystems.com (8.12.6/8.12.6)
    //    with ESMTP id g9MFWcvb068860 for <jm@jmason.org>;
    //    Tue, 22 Oct 2002 16:32:39 +0100 (IST)
    if (res = m_pcre->Check("rec_21", prcvd)) {
        mta_looked_up_dns = 1;
        ppHelo = res->GetPattern(1);
        ppRdns = res->GetPattern(2);
        ppIp = res->GetPattern(3);
        rec_type.assign("rec_21");
        goto enough;
    }

    //    Received: from cabbage.jmason.org [127.0.0.1] by localhost with IMAP (fetchmail-5.9.0)
    //    for jm@localhost (single-drop); Thu, 13 Mar 2003 20:39:56 -0800 (PST)
    if (res = m_pcre->Check("rec_22", prcvd)) {
        ppRdns = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_22");
        goto enough;
    }

    //    Received: from [129.24.215.125] by ws1-7.us4.outblaze.com with http for
    //    _bushisevil_@mail.com; Thu, 13 Feb 2003 15:59:28 -0500

    //    Received: from po11.mit.edu [18.7.21.73]
    //    by stark.dyndns.tv with POP3 (fetchmail-5.9.7)
    //    for stark@localhost (single-drop); Tue, 18 Feb 2003 10:43:09 -0500 (EST)
    //    by po11.mit.edu (Cyrus v2.1.5) with LMTP; Tue, 18 Feb 2003 09:49:46 -0500

    if (res = m_pcre->Check("rec_24", prcvd)) {
        ppRdns = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_24");
        goto enough;
    }

    //    Received: from snake.corp.yahoo.com(216.145.52.229) by x.x.org via smap (V1.3)
    //    id xma093673; Wed, 26 Mar 03 20:43:24 -0600

    if (res = m_pcre->Check("rec_25", prcvd)) {
        ppRdns = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_25");
        goto enough;
    }

    //    Received: from [192.168.0.71] by web01-nyc.clicvu.com (Post.Office MTA
    //    v3.5.3 release 223 ID0-64039U1000L100S0V35) with SMTP id com for
    //    <x@x.org>; Tue, 25 Mar 2003 11:42:04 -0500

    //    Received: from [127.0.0.1] by euphoria (ArGoSoft Mail Server
    //    Freeware, Version 1.8 (1.8.2.5)); Sat, 8 Feb 2003 09:45:32 +0200

    //    Received: from inet-vrs-05.redmond.corp.microsoft.com ([157.54.6.157]) by
    //    INET-IMC-05.redmond.corp.microsoft.com with Microsoft SMTPSVC(5.0.2195.6624);
    //    Thu, 6 Mar 2003 12:02:35 -0800

    if (res = m_pcre->Check("rec_27", prcvd)) {
        ppHelo = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_27");
        goto enough;
    }

    //    Received: from tthompson ([217.35.105.172] unverified) by
    //    mail.neosinteractive.com with Microsoft SMTPSVC(5.0.2195.5329);
    //    Tue, 11 Mar 2003 13:23:01 +0000

    //    Received: from 157.54.8.23 by inet-vrs-05.redmond.corp.microsoft.com
    //    (InterScan E-Mail VirusWall NT); Thu, 06 Mar 2003 12:02:35 -0800

    if (res = m_pcre->Check("rec_28", prcvd)) {
        ppIp = res->GetPattern(1);
        rec_type.assign("rec_28");
        goto enough;
    }

    //    Received: from faerber.muc.de by slarti.muc.de with BSMTP (rsmtp-qm-ot 0.4)
    //    for asrg@ietf.org; 7 Mar 2003 21:10:38 -0000

    //    Received: from spike (spike.ig.co.uk [193.32.60.32]) by mail.ig.co.uk with
    //    SMTP id h27CrCD03362 for <asrg@ietf.org>; Fri, 7 Mar 2003 12:53:12 GMT

    if (res = m_pcre->Check("rec_29", prcvd)) {
        mta_looked_up_dns = 1;
        ppHelo = res->GetPattern(1);
        ppRdns = res->GetPattern(2);
        ppIp = res->GetPattern(3);
        rec_type.assign("rec_29");
        goto enough;
    }

    //    Received: from customer254-217.iplannetworks.net (HELO AGAMENON)
    //    (baldusi@200.69.254.217 with plain) by smtp.mail.vip.sc5.yahoo.com with
    //    SMTP; 11 Mar 2003 21:03:28 -0000

    if (res = m_pcre->Check("rec_30", prcvd)) {
        mta_looked_up_dns = 1;
        ppRdns = res->GetPattern(1);
        ppHelo = res->GetPattern(2);
        ppIp = res->GetPattern(3);
        rec_type.assign("rec_30");
        goto enough;
    }

    //    Received: from raptor.research.att.com (bala@localhost) by
    //    raptor.research.att.com (SGI-8.9.3/8.8.7) with ESMTP id KAA14788
    //    for <asrg@example.com>; Fri, 7 Mar 2003 10:37:56 -0500 (EST)

    //    Received: from mmail by argon.connect.org.uk with local (connectmail/so/exim) id 18tOsg-0008FX-00; Thu, 13 Mar 2003 09:20:06 +0000

    //    Received: from [192.168.1.104] (account nazgul HELO [192.168.1.104])
    //    by somewhere.com (CommuniGate Pro SMTP 3.5.7) with ESMTP-TLS id 2088434;
    //    Fri, 07 Mar 2003 13:05:06 -0500

    //    Received: from ([10.0.0.6]) by mail0.ciphertrust.com with ESMTP ; Thu,
    //    13 Mar 2003 06:26:21 -0500 (EST)

    //    Received: from ironport.com (10.1.1.5) by a50.ironport.com with ESMTP; 01 Apr 2003 12:00:51 -0800
    //    Received: from dyn-81-166-39-132.ppp.tiscali.fr (81.166.39.132) by cpmail.dk.tiscali.com (6.7.018)
    //    note: must be before 'Content Technologies SMTPRS' rule, cf. bug 2787

    //    Received: from scv3.apple.com (scv3.apple.com) by mailgate2.apple.com (Content Technologies SMTPRS 4.2.1) with ESMTP id <T61095998e1118164e13f8@mailgate2.apple.com>; Mon, 17 Mar 2003 17:04:54 -0800

    //    Received: from 01al10015010057.ad.bls.com ([90.152.5.141] [90.152.5.141])
    //    by aismtp3g.bls.com with ESMTP; Mon, 10 Mar 2003 11:10:41 -0500

    //    Received: from 206.47.0.153 by dm3cn8.bell.ca with ESMTP (Tumbleweed MMS
    //    SMTP Relay (MMS v5.0)); Mon, 24 Mar 2003 19:49:48 -0500

    //    Received: from pobox.com (h005018086b3b.ne.client2.attbi.com[66.31.45.164])
    //    by rwcrmhc53.attbi.com (rwcrmhc53) with SMTP id <2003031302165605300suph7e>;
    //    Thu, 13 Mar 2003 02:16:56 +0000

    if (res = m_pcre->Check("rec_31", prcvd)) {
        mta_looked_up_dns = 1;
        ppHelo = res->GetPattern(1);
        ppRdns = res->GetPattern(2);
        ppIp = res->GetPattern(3);
        rec_type.assign("rec_31");
        goto enough;
    }

    //    Received: from [10.128.128.81]:50999 (HELO dfintra.f-secure.com) by fsav4im2 ([10.128.128.74]:25) (F-Secure Anti-Virus for Internet Mail 6.0.34 Release) with SMTP; Tue, 5 Mar 2002 14:11:53 -0000

    //    Received: from 62.180.7.250 (HELO daisy) by smtp.altavista.de (209.228.22.152) with SMTP; 19 Sep 2002 17:03:17 +0000

    //    Received: from oemcomputer [63.232.189.195] by highstream.net (SMTPD32-7.07) id A4CE7F2A0028; Sat, 01 Feb 2003 21:39:10 -0500
    //
    //    Received: from [192.168.0.13] by <server> (MailGate 3.5.172) with SMTP;
    //    Tue, 1 Apr 2003 15:04:55 +0100

    //    Received: from jmason.org (unverified [195.218.107.131]) by ni-mail1.dna.utvinternet.net <B0014212518@ni-mail1.dna.utvinternet.net>; Tue, 11 Feb 2003 12:18:12 +0000

    //    from 165.228.131.11 (proxying for 139.130.20.189) (SquirrelMail authenticated user jmmail) by jmason.org with HTTP

    //    Received: from [212.87.144.30] (account seiz [212.87.144.30] verified) by x.imd.net (CommuniGate Pro SMTP 4.0.3) with ESMTP-TLS id 5026665 for spamassassin-talk@lists.sourceforge.net; Wed, 15 Jan 2003 16:27:05 +0100

    //    Received: from mtsbp606.email-info.net (?dXqpg3b0hiH9faI2OxLT94P/YKDD3rQ1?@64.253.199.166) by kde.informatik.uni-kl.de with SMTP; 30 Apr 2003 15:06:29
    if (res = m_pcre->Check("rec_32", prcvd)) {
        ppRdns = res->GetPattern(1);
        ppIp = res->GetPattern(2);
        rec_type.assign("rec_32");
        goto enough;
    }

    if (fPrintStat)
        m_pstat.AddStat(ST_LOG) << "GetReceivedNext_2";
    GetReceivedNext_2(prcvd, ppIp, ppRdns, ppHelo);

    enough:

    if (fPrintStat && !rec_type.empty())
        m_pstat.AddStat(ST_LOG) << rec_type;

    if (ppRdns == "unknown" || ppRdns == "not_resolved") {
        if (ppIp != "-" && fPrintStat)
            m_pstat.AddStat(ST_LOG) << "host not resolved";
        ppRdns = "-";
    }
}

//
// Compare fix methods (old and new - by shingler)
//
void TSpAlg::CompareFixMethods() {
    char str[256];
    if (m_cur.fFixedSource != m_cur.fShFixedSource) {
        snprintf(str, 256, "old: %u, new (by shingler): %u", m_cur.fFixedSource, m_cur.fShFixedSource);
        //m_pstat.AddStat(ST_FIXED_DIFF) << str;
    }
}

//
// Write statistics on delivery fixes
//
void TSpAlg::WriteDeliveryStat(bool UseShingler) {
    if (UseShingler)
        if (m_cur.fShFixedSource) {
            m_cur.fFixedSource = true;
            m_cur.level = m_cur.sh_level;
        }

    if (m_cur.fFixedSource) {
        SetRule("__FIXED_SOURCE");
        if (m_cur.level == 1 && !m_cur.fUndefinedIp &&
            !m_curMessageContext->rulesContext.IsRuleWorked("__PRESENT_X_ORIGINATING_IP"))
            SetRule("__FIXED_SOURCE_LEVEL_1");
    }

    //  if (m_cur.level && m_cur.rgip[0])
    /*if (m_cur.level && (m_cur.rgip[0] || m_cur.fIpv6)) {
        if (m_cur.fDelivery)
        {
            if (m_cur.fFixedSource)
                m_pstat.AddStat(ST_FIXED) << "fixed delivery";
            else
                m_pstat.AddStat(ST_FIXED) << "miss delivery";
        }
        else
        {
            if (m_cur.fFixedSource)
                m_pstat.AddStat(ST_FIXED) << "fixed domen";
            else
                m_pstat.AddStat(ST_FIXED) << "miss domen";
        }
    } else if (!m_cur.fForward && !m_curMessageContext->rulesContext.IsRuleWorked("ALLTRUSTEDIP"))
        m_pstat.AddStat(ST_FIXED) << "unknown sender ip";
    */
}

bool TSpAlg::CheckAddrInReceived() {
    std::vector<TString>::iterator vi;

    if (m_rgpFields[FD_FROM_ADDR].Size() < 6 || !strchr(m_rgpFields[FD_FROM_ADDR].c_str(), '@'))
        return false;

    for (vi = m_vReceived.begin(); vi != m_vReceived.end(); vi++) {
        for (const char* prcvd = (*vi).c_str(); prcvd = strstr(prcvd, "for ");) {
            prcvd += 4;
            while (*prcvd == ' ' || *prcvd == '<' || *prcvd == '(')
                ++prcvd;
            if (!STRNICMP(prcvd, m_rgpFields[FD_FROM_ADDR].c_str()))
                return true;
        }
    }

    return false;
}

TSpAlg::TDomains TSpAlg::CheckAuthDomains() const{

    const TString& authRes = m_rgpFields[m_curMessageContext->IsMailish ? FD_AUTHENTICATION_RESULTS : FD_X_YANDEX_AUTHENTICATION_RESULTS];
    return TDomains {
            TDomains::TSpf{m_pcre->GetPattern("get_spf_domain", authRes, 1)},
            TDomains::TDKim{m_pcre->GetPattern("get_dkim_domain", authRes, 1)},
    };
}
