#include <util/generic/array_ref.h>
#include <mail/so/spamstop/tools/pcre_wrapper/pcre_wrapper.h>
#include <util/string/subst.h>
#include "sptop.h"
#include "spamrule.h"

//**********
// TRuleDef
//**********

bool TReRule::Run(const TStringBuf& value) {
    const auto matchResult = expression.Match(value);

    return (matchResult == NRegexp::EMatchResult::NotMatch && !fMatch) ||
           (matchResult == NRegexp::EMatchResult::Match && fMatch);
}

//**********
// TRulePcre
//**********

TString TRulePcre::SetRe(const TStringBuf& szre, TArrayRef<const TPcreUnit> rgmacro) {
    TString res(szre);

    for (const auto& unit : rgmacro) {
        SubstGlobal(res, unit.name, unit.re);
    }

    return res;
}

void SetPcre(TReDef& re, const NRegexp::TSettings& pcreSettings) {
    re.re_comp = MakeHolder<NRegexp::TPcre>(NormalizeRe(re.text), pcreSettings);
}

TString NormalizeRe(TString re) {
    Y_VERIFY(re);

    size_t p = 0;

    //  Get the delimiter and seek the end of the pattern; if is isn't
    //  complete, read more.

    const char delimiter = re[p++];

    if (isalnum(delimiter) || delimiter == '\\') {
        ythrow TWithBackTrace<yexception>() << "Delimiter can not be alphanumeric or \\  Re: " << re;
    }

    while (p < re.size()) {
        if (re[p] == '\\' && p + 1 < re.size())
            p++;
        else if (re[p] == delimiter)
            break;
        p++;
    }

    if (p >= re.size() || re[p] != delimiter) {
        Syslog(TLOG_ERR) << "Delimiter error (" << delimiter << ")! Re: " << re;
    }

    // If the first character after the delimiter is backslash, make
    // the pattern end with backslash. This is purely to provide a way
    // of testing for the error message when a pattern ends with backslash.

    if (re[p + 1] == '\\')
        re[p++] = '\\';

    // Terminate the pattern at the delimiter

    const auto preend = p++;

    TStringStream options;
    // Look for options after final delimiter
    options << '(' << '?';
    bool multiLine = false;
    bool singleLine = false;

    while (p < re.size()) {
        char c = re[p++];
        switch (c) {
            case 's':
            case 'x':
            case 'A':
            case 'E':
            case 'N':
            case 'U':
            case 'X':
                singleLine = true;
                options << c;
                break;
            case 'm':
                multiLine = true;
            case '8':
            case 'i':
                options << c;
                break;
            case '\n':
            case ' ':
                break;
            default:
                ythrow TWithBackTrace<yexception>() << "Rule has unknown option! '" << re.at(p - 1) << "\' in regexp: " << re;
        }
    }

    if (multiLine || !singleLine)
        options << 'm';

    options << ')';

    TStringStream resultRe;
    if (!options.empty())
        resultRe << options.Str();

    resultRe << re.substr(1, preend - 1);

    return std::move(resultRe.Str());
}

PcreTool::PcreTool(const NRegexp::TSettings& settings, TArrayRef<const TPcreUnit> units, TArrayRef<const TPcreUnit> rgmacro) {
    for (const auto& unit : units) {
        if (unit.re) {
            auto newPcreIt = PcreMap.emplace(
                                        std::piecewise_construct,
                                        std::forward_as_tuple(unit.name),
                                        std::forward_as_tuple(&unit, rgmacro))
                                 .first;
            SetPcre(std::get<TReDef>(newPcreIt->second.rules), settings);
        }
    }
}

TMaybe<NRegexp::TResult> PcreTool::Check(const TStringBuf& name, const TStringBuf& text) const {
    if (auto re = MapFindPtr(PcreMap, name)) {
        NRegexp::TMatches matches(Reserve(MAX_PCRE_PATTERNS * 2));
        if (std::get<TReDef>(re->rules).CheckRegExp(text, &matches, MAX_PCRE_PATTERNS * 2)) {
            return MakeMaybe<NRegexp::TResult>(text, std::move(matches));
        } else {
            return Nothing();
        }
    }
    ythrow TWithBackTrace<yexception>() << "PcreTool: not found re: " << name;
}

// numeration pattern from 1 ...
// 0 - pattern for  regexp in all

TStringBuf PcreTool::GetPattern(const TStringBuf& szname, const TStringBuf& text, size_t ind) const {
    if (auto res = Check(szname, text)) {
        return res->GetPattern(ind);
    }
    return {};
}
