#pragma once

#include "fields.h"
#include "rules.h"
#include "rules_key_words.h"

#include <mail/so/spamstop/tools/pcre_wrapper/pcre_wrapper.h>

#include <library/cpp/enumbitset/enumbitset.h>

#include <util/generic/variant.h>
#include <util/generic/array_ref.h>

#include <utility>

#define RUL_UNK 0x00000000
#define RUL_RUS 0x00000001
#define RUL_ENG 0x00000002
#define RUL_UKR 0x00000004
#define RUL_TUR 0x00000008
#define RUL_BLR 0x00000016

#define FD_MAX_LEN 260

// delivery fields and properties
enum TSpSenderType { en_SR_UNDEF,
                     en_SR_HOST,
                     en_SR_MESSID,
                     en_SR_FROM,
                     en_SR_REPLYTO,
                     en_SR_SENDER,
                     en_SR_LIST,
                     en_SR_SUBJ,
                     en_SR_RBL,
                     en_SR_DSL,
                     en_SR_NRV,
                     en_SR_BL_NOT,
                     en_SR_L2ASL1,
                     en_SR_MAX
};

using TKWYasmTagsBitset = TEnumBitSet<TKWYasmTags, static_cast<int>(TKWYasmTags::SPAM), static_cast<int>(TKWYasmTags::MALIC) + 1>;

#define SP_MAX_RULE_NAME 128
#define SP_MAX_RULE_LENGTH 1024
#define SP_MAX_SCORE_LENGTH 1024

// Spam types of rules
enum TRulesType{
    RT_UNKNOWN,      // not defined
    RT_RGEX,         // regular expression
    RT_RGEXGROUP,    // group regular expression
    RT_BF,           // rules boolean expression -  rule1 | rule2 & rule3
    RT_ARITHMETIC,   // rules arithmetic expression -  rule1 + rule2 - rule3 > 3.0
    RT_PRESENT,      // field exists and not empty
    RT_ABSENT,       // field is not exists or emty
    RT_ALG,          // algorithmic rule
    RT_RECEIVED_NUM, // rule for field 'received' with defined number
    RT_ANTI,         // antirule cancel defined rule
    RT_LV,           // linguistic rule
    RT_RANGE,        // range rule
    RT_RANGE_DATE,   // date range rule
    RT_LEVENSTEIN,   // levenstein distance rule,
    RT_LUA,          // lua script
    RT_HS,
};

struct TRuleDef;

struct TBfDef {
    TString text; // boolean expression
    TVector<ui8> pResult;
};

// + or -  is allowed only
// < or >  is allowed only
// parenthesis is not allowed
// all arguments has value 0 or 1 if correspondent rule was worked
// syntax: rule1  + or - rule2 ... < or > Number

struct TArDef {
    TString text;               // arithmetic expression - a+b-c > 5
    TVector<char> pSignes;      // array of signes(+) or (-)
    char comp = '>';            // '>' or '<' - expression operation
    int value{};                //  right part of expression
};

struct TAntiDef {
    TString pAntiRuleName;                    // antirule name
    std::vector<TString> vCancelRulesNames{}; // names of the rules to cancel
    std::vector<std::reference_wrapper<const TRuleDef>> pCancelRules{};       // indexes of the rules to cancel
};

struct TRangeDef /*: public TReDef*/ {
    bool fMin{};
    bool fMax{};
    bool fToday{};
    double d_min{};
    double d_max{};
    time_t time_point{};
    TString key;

    bool CheckRange(double value) const {
        if (!fMin && value < d_min)
            return false;
        if (!fMax && value >= d_max)
            return false;
        return true;
    }
    bool CheckDateRange(time_t value) {
        if (fToday)
            time_point = time(nullptr);
        if (!fMin && (time_point - value) < d_min)
            return false;
        if (!fMax && (time_point - value) >= d_max)
            return false;
        return true;
    }
};

#define REC_MAX_LEVEL 15

struct TRuleDef {
    TString pRuleName{};        // rule name
    TRulesType rt = RT_UNKNOWN; // rule type

    double score{}; // score

    TString comment;            // rule commentary
    TVector<TSpFields> pfields; // fields to use

    using TRules = std::variant<std::tuple<>, TReDef, TBfDef, TArDef, TAntiDef, TLvDef, TRangeDef, TLuaRuleDef, THsDef>;
    TRules rules;
    bool fDelivery{};    // delivery weight flag

    int id{};

    TKWYasmTagsBitset YasmTags;
    TKWYasmTagsBitset SolomonTags;

    std::vector<std::reference_wrapper<const TRuleDef>> Masters, Dependencies;

    friend IOutputStream& operator<<(IOutputStream& stream, const TRuleDef& ruleDef) {
        return stream << "rule:" << ruleDef.pRuleName << " type:" << ruleDef.rt;
    }

    virtual ~TRuleDef() = default;
};

struct TPcreUnit {
    TPcreUnit(TStringBuf name, TStringBuf re, bool fMatch = false) noexcept
        : name(name)
        , re(re)
        , fMatch(fMatch) {
    }
    TStringBuf name;
    TStringBuf re;
    bool fMatch{};
};

struct TRulePcre : public TRuleDef {
    TRulePcre(const TPcreUnit* pcreunit, TArrayRef<const TPcreUnit> rgmacro) {
        pRuleName = pcreunit->name;
        rt = RT_RGEX;

        TString ptext = nullptr;
        if (rgmacro) {
            ptext = SetRe(pcreunit->re, rgmacro);
        }

        if (!ptext)
            ptext = pcreunit->re;

        rules.emplace<TReDef>(std::move(ptext), pcreunit->fMatch);
    }
    static TString SetRe(const TStringBuf& szre, TArrayRef<const TPcreUnit> rgmacro);
};

#define MAX_PCRE_PATTERNS 10

class PcreTool {
private:
    THashMap<TString, TRulePcre> PcreMap;

public:

public:
    explicit PcreTool(const NRegexp::TSettings& settings,  TArrayRef<const TPcreUnit> units, TArrayRef<const TPcreUnit> rgmacro = {});
    [[nodiscard]] TMaybe<NRegexp::TResult> Check(const TStringBuf& szname, const TStringBuf& text) const;
    [[nodiscard]] TStringBuf GetPattern(const TStringBuf& szname, const TStringBuf& text, size_t ind) const;
    TStringBuf GetPattern(const TStringBuf& szname, TString&& text, size_t ind) const = delete;
};

void SetPcre(TReDef& re, const NRegexp::TSettings& pcreSettings);

TString NormalizeRe(TString re);
