#ifndef WIN32
#include <unistd.h> // unlink
#include <sys/file.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <vector>
#include <algorithm>
#include <utility>
#include <locale.h>
#include <time.h>

#include "util/generic/string.h"

#include "sptypes.h"
#include "mail/so/spamstop/tools/so-common/sputil.h"
#include "spruler.h"
#include "spstat.h"
#include "rengine.h"
#include "rulesholder.h"

TSpRuler::TSpRuler(TSpLogger* pSpLogger, int m_cRulesA, TVector<THolder<TRuleDef>>& m_ppRulesA) : m_ppRules(m_ppRulesA) {
    m_p_sp_logger = pSpLogger;
    m_cRules = m_cRulesA;

    memset(&m_cur, 0, sizeof(TCurRuler));

//    m_mapKeyRange.AssignName("TRengine::KeyRange");
//    m_mapKeyLevenstein.AssignName("TRengine::KeyLevenstein");
}

ecRet TSpRuler::SetRules() {
    int i;
    int cDlvrRanges = 1;
    int cLevenstein = 0;

    for (i = 0; i < m_cRules; i++) {
        switch (m_ppRules[i]->rt) {
            case RT_RANGE:
            case RT_RANGE_DATE:
                ++cDlvrRanges;
                break;
            case RT_LEVENSTEIN:
                ++cLevenstein;
                break;
            default:
                break;
        }
    }

    m_mapKeyRange.reserve(cDlvrRanges + 1);
    if (cLevenstein)
        m_mapKeyLevenstein.reserve(cLevenstein);

    for (i = 0; i < m_cRules; i++) {
        switch (m_ppRules[i]->rt) {
            case RT_RANGE:
            case RT_RANGE_DATE: {
                auto it = m_mapKeyRange.find(std::get<TRangeDef>(m_ppRules[i]->rules).key);
                if (end(m_mapKeyRange) != it)
                    it->second.push_back(i);
                else {
                    m_mapKeyRange.emplace(std::get<TRangeDef>(m_ppRules[i]->rules).key, std::vector<int>{i});
                }
                break;
            }
            case RT_LEVENSTEIN: {
                auto it = m_mapKeyLevenstein.find(std::get<TRangeDef>(m_ppRules[i]->rules).key);
                if (end(m_mapKeyLevenstein) != it)
                    it->second.push_back(i);
                else {
                    m_mapKeyLevenstein.emplace(std::get<TRangeDef>(m_ppRules[i]->rules).key, std::vector<int>{i});
                }
                break;
            }
            default:
                break;
        }
    }

    return ecOK;
}

const std::vector<int> *TSpRuler::FindRangeRules(const TString& key, bool fPrintWarning) const {
    auto it = m_mapKeyRange.find(to_lower(key));
    if (Y_LIKELY(end(m_mapKeyRange) != it)) {
        return &it->second;
    }
#ifndef __SO_CLIENTS__
    else if (fPrintWarning && (m_p_sp_logger != nullptr))
            m_p_sp_logger->splog(TLOG_ERR, "unknown key range checking: %s", key.c_str());
#endif
    return nullptr;
}

const std::vector<int>* TSpRuler::FindLevensteinRules(const TString& key, bool /*fPrintWarning*/) const {
    auto it = m_mapKeyLevenstein.find(to_lower(key));
    if (Y_LIKELY(cend(m_mapKeyRange) != it)) {
        return &it->second;
    }
    return nullptr;
}

void TSpRuler::CheckDateRange(TCurMessageEngine& curMessageEngine, const TString& key, time_t value, bool fPrintWarning) const {

    auto *pvrange = FindRangeRules(key, fPrintWarning);
    if (!pvrange)
        return;

    for (auto n : *pvrange) {
        auto& rule = m_ppRules[n];
        if (rule->rt == RT_RANGE_DATE) {
            if (std::get<TRangeDef>(rule->rules).CheckDateRange(value))
                curMessageEngine.rulesContext.WorkedRule(n);
        } else {
            if (m_p_sp_logger != nullptr)
                m_p_sp_logger->splog(TLOG_ERR, "rule type error in range checking: %s", rule->pRuleName.c_str());
        }
    }
}

TString TSpRuler::CheckLevenstein(TCurMessageEngine& curMessageEngine, const char* key, const char* word) const {
    const std::vector<int>* pvrange = FindLevensteinRules(key, true);

    if (!pvrange)
        return nullptr;

    size_t levValue = GetLevensteinDistance(key, word);

    if ((levValue > 3) || (strcasecmp(key, word) == 0)) // ignore equal and all far distances
        return nullptr;

    for (auto n : *pvrange) {
        const auto & rule = m_ppRules[n];
        if (rule->rt == RT_LEVENSTEIN) {
            if (std::get<TRangeDef>(rule->rules).CheckRange((int)levValue))
                curMessageEngine.rulesContext.WorkedRule(n);
        } else {
            if (m_p_sp_logger != nullptr)
                m_p_sp_logger->splog(TLOG_ERR, "rule type error in range checking: %s", rule->pRuleName.c_str());
        }
    }

    char str[256] = {};
    snprintf(str, 255, "%s=%d ", word, (int)levValue);
    return str;
}

size_t TSpRuler::GetLevensteinDistance(const TString& src, const TString& dst) const {
    size_t m = src.size();
    size_t n = dst.size();
    if (m == 0)
        return n;

    if (n == 0)
        return m;

    std::vector<std::vector<size_t>> matrix(m + 1);

    for (size_t i = 0; i <= m; ++i) {
        matrix[i].resize(n + 1);
        matrix[i][0] = i;
    }
    for (size_t i = 0; i <= n; ++i)
        matrix[0][i] = i;

    size_t above_cell, left_cell, diagonal_cell, cost;

    for (size_t i = 1; i <= m; ++i) {
        for (size_t j = 1; j <= n; ++j) {
            cost = src[i - 1] == dst[j - 1] ? 0 : 1;
            above_cell = matrix[i - 1][j];
            left_cell = matrix[i][j - 1];
            diagonal_cell = matrix[i - 1][j - 1];
            matrix[i][j] = std::min(std::min(above_cell + 1, left_cell + 1), diagonal_cell + cost);
        }
    }

    return matrix[m][n];
}
