#pragma once

#include <util/stream/str.h>
#include <util/stream/null.h>
#include <util/generic/hash.h>
#include <util/generic/deque.h>
#include <util/generic/map.h>
#include <util/string/builder.h>
#include <util/system/rwlock.h>
#include <mail/so/libs/lua_rules/kv_logger.h>

#include <memory>
#include "fields.h"

enum TClassificStat{
    ST_MESSAGE /* "mess" */,
    ST_SPAM /* "spam" */,
    ST_FROM /* "from" */,
    ST_SUBJ /* "subj" */,
    ST_MESSID /* "msid" */,
    ST_XMAILER /* "xmlr" */,
    ST_RAW_BODYSIZE /* "size" */,
    ST_WEIGHT /* "wght" */,
    ST_TIME /* "time" */,
    ST_IP /* "ip  " */,
    ST_NEAR_C /* "near" */,
    ST_HOSTNAME /* "host" */,
    ST_SOURCE /* "srce" */,
    ST_FIXED /* "fixd" */,
    ST_FIXED_BY_SHINGLER /* "fixs" */,
    ST_FIXED_DIFF /* "fixf" */,
    ST_FIXED_BY_RULE /* "fxrl" */,
    ST_REPORT_SPAM /* "r_sp" */,
    ST_REPORT_DLV /* "r_dl" */,
    ST_REPORT_NULL /* "r_nl" */,
    ST_LEVEL /* "levl" */,
    ST_RECEIVED /* "rcvd" */,
    ST_HTTP /* "http" */,
    ST_MAILTO /* "mlto" */,
    ST_PHONE /* "phon" */,
    ST_LOG /* "log " */,
    ST_FL /* "flin" */,
    ST_CS /* "spcs" */,
    ST_HOST_LOCAL /* "locl" */,
    ST_SPAM_PAT /* "spat" */,
    ST_RCP_TO /* "rcpt" */,
    ST_MAIL_FROM /* "mfrm" */,
    ST_COMPLAINT /* "cmpl" */,
    ST_COMPLAINT_FROM /* "cmpf" */,
    ST_FROMTO /* "frto" */,
    ST_REPUTATION /* "rptn" */,
    ST_PERSONAL /* "pers" */,
    ST_DOMAINS /* "domn" */,
    ST_RULE_STAT /* "r_st" */,
    ST_FULL_FL /* "ffln" */,
    ST_COMPL_SHINGL /* "cosh" */,
    ST_HISTORY /* "hist" */,
    ST_FREE_MAIL /* "frml" */,
    ST_FREE_MAIL2 /* "frm2" */,
    ST_TEXT_PROC /* "txpr" */,
    ST_URL_REP /* "rpur" */,
    ST_CLIENT_ERR /* "clnt" */,
    ST_SP_VERSION /* "vers" */,
    ST_PHISH_Y /* "phiy" */,
    ST_FIELDS /* "flds" */,
    ST_PDD_INFO /* "pdd " */,
    ST_PDD_INFO2 /* "pdd2" */,
    ST_USER_BOUNCE /* "ubns" */,
    ST_USER_BOUNCE2 /* "ubn2" */,
    ST_RECOGNIZER /* "rcgn" */,
    ST_LANGUAGE /* "lngg" */,
    ST_SOLOGTEMP /* "iyso" */,
    ST_SENDER /* "sndr" */,
    ST_DOMAINSENDER /* "sdmn" */,
    ST_FROMADDR /* "fadr" */,
    ST_YAML /* "yaml" */,
    ST_RBL /* "rbl " */,
    ST_SHIN /* "shin" */,
    ST_PAYSENDER /* "psnd" */,
    ST_ARCHIVE /* "arch" */,
    ST_ATTACH /* "atch" */,
    ST_YDISK /* "ydsk" */,
    ST_WHOIS /* "whis" */,
    ST_COMPLLOG /* "clog" */,
    ST_DICT /* "dict" */,
    ST_UAAS /* "uaas" */,
    ST_LSA /* "lsa " */,
    ST_LSA_TESTING /* "lsa2" */,
    ST_LEVENSTEIN /* "lvst" */,
    ST_CORRUPT /* "crpt" */,
    ST_TMP1 /* "tmp1" */,
    ST_TMP2 /* "tmp2" */,
    ST_CACHE_HIT /* "chit" */
};

namespace NJsonWriter {
    class TBuf;
}

class TSpStat : public NLua::IKVLogProvider{
private:
    TDeque<std::pair<TString, TStringStream>> Records;
    TMap<TString, long> LimitedRecords;
    mutable TRWMutex Lock;
    static const long DEFAULT_RECORDS_LIMIT = 5;

public:
    struct TRecord : IOutputStream{
        TRecord(TRecord&&) noexcept = default;

        explicit TRecord(IOutputStream& dst) noexcept : Row(dst) {}

        template <class T>
        TRecord& operator<<(const T& t) {
            static_cast<IOutputStream&>(*this) << t;
            return *this;
        }

        void DoWrite(const void* buf, size_t len) override;

        IOutputStream& Row;
    };

    struct TPrinter {
        explicit TPrinter(const TSpStat& master) noexcept : Master(master) {}
    protected:
        const TSpStat& Master;
    };

    struct TDlvLogPrinter : public TPrinter{
    public:
        friend IOutputStream& operator<<(IOutputStream& stream, const TDlvLogPrinter& p);
        using TPrinter::TPrinter;
    };

    struct TJsonDlvLogPrinter : public TPrinter{
    public:
        friend IOutputStream& operator<<(IOutputStream& stream, const TJsonDlvLogPrinter& p);
        using TPrinter::TPrinter;
    };

    struct TJsonPrinter : public TPrinter{
    public:
        friend NJsonWriter::TBuf& PrintJson(const TJsonPrinter& p, NJsonWriter::TBuf& stream);
        friend IOutputStream& operator<<(IOutputStream& stream, const TJsonPrinter& p);
        using TPrinter::TPrinter;
    };

    void Log(const TStringBuf &key, const TStringBuf &value) override {
        AddStat(key) << value;
    }

public:

    template<class T> TRecord AddStat(const T &prefix) {
        TString key = TStringBuilder{} << prefix;

        auto guard = TWriteGuard(Lock);
        return TRecord(Records.emplace_back(std::move(key), TStringStream{}).second);
    }

    template<class T> TRecord AddLimitedStat(const T &prefix) {
        TString key = TStringBuilder{} << prefix;
        auto guard = TWriteGuard(Lock);
        if (LimitedRecords[key]++ < DEFAULT_RECORDS_LIMIT) {
            return TRecord(Records.emplace_back(std::move(key), TStringStream{}).second);
        } else {
            return TRecord(Cnull);
        }
    }

    void PrintLimitedRecordsCount() {
        auto guard = TWriteGuard(Lock);
        for (const auto &record : LimitedRecords) {
            if (record.second > DEFAULT_RECORDS_LIMIT) {
                Records.emplace_back(record.first + "_count", TStringBuilder{} << record.second);
            }
        }
    }

    template<class T> auto VisitRecords(T && recordsVisitor) const {
        auto guard = TReadGuard (Lock);
        return recordsVisitor(Records);
    }

    bool Empty() const {
        auto guard = TReadGuard (Lock);
        return Records.empty();
    }

    void Reset() {
        auto guard = TWriteGuard (Lock);
        Records.clear();
    }
};
