#include <sys/types.h>

#include "sptop.h"
#include <util/system/maxlen.h>
#include "util/generic/string.h"
#include "mail/so/spamstop/tools/so-common/sputil.h"
#include "sptypes.h"
#include <util/string/split.h>
#include <util/string/strip.h>
#include <util/system/fs.h>
#include <mail/so/spamstop/tools/so-common/so_log.h>
#include <mail/so/libs/talkative_config/config.h>
#include <util/stream/file.h>
#include <mail/so/libs/syslog/so_log.h>


TClientsConfigs TClientsConfigs::FromConfig(const NConfig::TDict &map) {
    TClientsConfigs configs;
    if (auto it = MapFindPtr(map, "RblProducer")) {
        configs.RblProducerConfig = TClientConfig::ReadFromConfig(*it);
        if (auto it1 = MapFindPtr(it->Get<NConfig::TDict>(), "TTL"))
            configs.RblProducerTTL = TDuration::Parse(it1->Get<TString>());
        if (auto it1 = MapFindPtr(it->Get<NConfig::TDict>(), "Period"))
            configs.RblProducerPeriod = TDuration::Parse(it1->Get<TString>());
    }

    if (auto it = MapFindPtr(map, "Cache")) {
        configs.CacheConfig = TClientConfig::ReadFromConfig(*it);
    }

    if (auto it = MapFindPtr(map, "Http2"))
        configs.Http2Config = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "Statlog")) {
        configs.StatlogConfig = TClientConfig::ReadFromConfig(*it);
    }

    if (auto it = MapFindPtr(map, "SpLogger")) {
        configs.SpLoggerConfig = TClientConfig::ReadFromConfig(*it);
    }

    if (auto it = MapFindPtr(map, "FreeMailReputation"))
        configs.FreeMailReputationConfig = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "SenderReputation"))
        configs.SenderReputationConfig = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "UrlReputation"))
        configs.UrlReputationConfig = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "PassInternal"))
        configs.PassInternalConfig = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "Abuse"))
        configs.AbuseConfig = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "UserWeights"))
        configs.UserWeightsConfig = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "UserWeightsNg"))
        configs.UserWeightsNgConfig = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "Uaas"))
        configs.UaasConfig = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "Knn"))
        configs.Knn = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "LSA")) {
        configs.LSAConfig = TClientConfig::ReadFromConfig(*it);
    }

    if (auto it = MapFindPtr(map, "UserReputation"))
        configs.UserReputationConfig = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "Blackbox"))
        configs.BBConfig = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "Rbl"))
        configs.RblConfig = TClientConfig::ReadFromConfig(*it);

    if (!configs.UserWeightsConfig)
        configs.UserWeightsConfig = configs.AbuseConfig;

    if (auto it = MapFindPtr(map, "DkimUpdater"))
        configs.DkimUpdaterConfig = TClientConfig::ReadFromConfig(*it);

    if (auto it = MapFindPtr(map, "Ocr"))
        configs.OcrConfig = TClientConfig::ReadFromConfig(*it);

    return configs;
}

TTvmConfig TTvmConfig::FromConfig(const NConfig::TConfig &config) {
    const size_t id = NTalkativeConfig::As<ui64>(config, "id");
    TString secret = NTalkativeConfig::Get<TString>(config, "secret");

    THashMap<TString, size_t> services;

    for (const auto&[key, value]: NTalkativeConfig::Get<NConfig::TDict>(config)) {
        if (key == "id" || key == "secret")
            continue;

        services.emplace(key, NTalkativeConfig::As<ui64>(value));
    }

    return {id, std::move(secret), std::move(services)};
}

TDeobfuscatorConfig TDeobfuscatorConfig::FromConfig(const NConfig::TConfig &config) {
    return {NTalkativeConfig::As<TFsPath>(config, "remap"), NTalkativeConfig::As<TFsPath>(config, "trie")};
}

TDeobfuscatorConfigs TDeobfuscatorConfigs::FromConfig(const NConfig::TConfig &config) {
    TDeobfuscatorConfigs res;
    for (const auto&[key, c] : NTalkativeConfig::Get<NConfig::TDict>(config)) {
        res.Configs.emplace(key, TDeobfuscatorConfig::FromConfig(c));
    }
    return res;
}

TSoConfig::TSoConfig(bool spk)
        : is_spk(spk) {
}

TSoConfig::TSoConfig(const NConfig::TConfig &config)
        : TSoConfig(false) {
    const auto &map = config.Get<NConfig::TDict>();

    if (auto it = MapFindPtr(map, "security")) {
        if (auto it1 = MapFindPtr(it->Get<NConfig::TDict>(), "default_domain")) {
            defDomain = it1->Get<TString>();
        }
    }

    ClientsConfigs = TClientsConfigs::FromConfig(map);

    if (auto it = MapFindPtr(map, "ComplLog")) {
        ComplLog = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "slowlog")) {
        if (auto it1 = MapFindPtr(it->Get<NConfig::TDict>(), "enabled")) {
            bSlowlog = it1->As<bool>();
        }

        if (auto it1 = MapFindPtr(it->Get<NConfig::TDict>(), "threshold")) {
            iSlowlogThreshold = it1->As<int>();
        }

        if (auto it1 = MapFindPtr(it->Get<NConfig::TDict>(), "logfile")) {
            sSlowlog = it1->Get<TString>();
        }
    }

    if (auto it = MapFindPtr(map, "fastspam")) {
        if (auto it1 = MapFindPtr(it->Get<NConfig::TDict>(), "sh14sLim")) {
            fsSh14sLim = it1->As<int>();
        }

        if (auto it1 = MapFindPtr(it->Get<NConfig::TDict>(), "sh17sLim")) {
            fsSh17sLim = it1->As<int>();
        }

        if (auto it1 = MapFindPtr(it->Get<NConfig::TDict>(), "sh17sLim2")) {
            fsSh17sLim2 = it1->As<int>();
        }

        if (auto it1 = MapFindPtr(it->Get<NConfig::TDict>(), "sh17mLim2")) {
            fsSh17mLim2 = it1->As<int>();
        }
    }

    if (auto it = MapFindPtr(map, "SolverThreads")) {
        solverThreads = it->As<size_t>();
    }

    if (auto it = MapFindPtr(map, "log_level")) {
        logLevel = FromString(it->Get<TString>());
    }

    if (auto it = MapFindPtr(map, "dns_timeout")) {
        dns_timeout = it->As<int>();
        if (dns_timeout < 10) {
            Syslog(TLOG_ERR) << "dns_timeout turned to milliseconds, set to 100";
            dns_timeout = 100;
        }
    }

    if (auto it = MapFindPtr(map, "SmartReject")) {
        smart_reject = it->As<int>();
    }

    if (auto it = MapFindPtr(map, "fake_syslog")) {
        fakeSyslog = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "tskv_format")) {
        tskvFormat = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "non-check-wl")) {
        TString strNet, strMask;
        Split(it->Get<TString>(), ' ', strNet, strMask);

        uNet.emplace_back(strNet);
        uMask.emplace_back(strMask);
    }

    if (auto it = MapFindPtr(map, "dnsbl_server")) {
        dnsbl_server = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "stat_folder")) {
        statFolder = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "add_rcvdheader")) {
        add_rcvd = it->As<int>();
    }

    if (auto it = MapFindPtr(map, "discard_trusted")) {
        discard_trusted = it->As<int>();
    }

    if (auto it = MapFindPtr(map, "DetectForegnMx")) {
        bDetectForeignMx = it->As<int>();
    }

    if (auto it = MapFindPtr(map, "DomesticZones")) {
        domesticZones = StringSplitter(it->Get<TString>()).SplitBySet(" \t").SkipEmpty();
    }

    if (auto it = MapFindPtr(map, "fnFilterLog")) {
        fnFilterLog = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "HttpLog")) {
        fnHttpLog = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "RulesLog")) {
        fnRulesLog = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "MailTruncateSize")) {
        MailTruncateSize = it->As<size_t>();
    }

    if (auto it = MapFindPtr(map, "MailProceedSize")) {
        MailProceedSize = it->As<size_t>();
    }

    if (auto it = MapFindPtr(map, "HeaderMaxSize")) {
        HeaderMaxSize = it->As<size_t>();
    }

    if (auto it = MapFindPtr(map, "dnRules")) {
        for (const auto tok : StringSplitter(it->Get<TString>()).SplitBySet(" ,;\t").SkipEmpty()) {
            dnRules.emplace_back(TFsPath(tok.Token()).RealPath());
            NFs::EnsureExists(dnRules.back());
        }
    }

    if (auto it = MapFindPtr(map, "TotalRcptsToProceed")) {
        TotalRcptsToProceed = it->As<size_t>();
    }

    if (auto it = MapFindPtr(map, "CorpMail")) {
        corp_mail = it->As<int>();
    }

    if (auto it = MapFindPtr(map, "RBLHost")) {
        rblHosts = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "fnMLLog")) {
        MLLog = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "fnUserReputationLog")) {
        UserRepLog = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "CorpMode")) {
        fCorpMode = it->As<i32>();
    }

#if !defined(SP_FILTER_CLIENTS) || defined(SO_CMAIL)

    if (auto it = MapFindPtr(map, "score")) {
        score = it->As<double>();
    }

    if (auto it = MapFindPtr(map, "delivery_score")) {
        delivery_score = it->As<double>();
    }
#endif

    if (auto it = MapFindPtr(map, "LvThreshold")) {
        LvThreshold = it->As<double>();
    }

    if (auto it = MapFindPtr(map, "fnDeliveryLog")) {
        fnClassification = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "DeliveryLogRegim")) {
        DeliveryLogRegim = it->As<i32>();
    }

    if (auto it = MapFindPtr(map, "StatTimeX")) {
        sStatTimeX = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "fnRulesList")) {
        fnRulesList = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "PersonalFilterDebug")) {
        fPersonalFilterDebug = 0 < it->As<i32>();
    }

    if (auto it = MapFindPtr(map, "ShortLog")) {
        ShortLog = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "MatrixnetLibDir")) {
        mnPluginLibDir = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "MatrixnetConfigPath")) {
        mnConfigPath = it->Get<TString>();
    }

    if (auto it = MapFindPtr(map, "UnistatPrefix")) {
        UnistatPrefix = it->Get<TString>();
    }


    if (auto it = MapFindPtr(map, "WebMail")) {
        fWebMail = 0 < it->As<int>();
    }

    if (auto it = MapFindPtr(map, "DumbMode")) {
        dumbMode = it->As<int>();
    }

    if (auto it = MapFindPtr(map, "TabPrefixesToCalcTheBest")) {
        for (const auto tok : StringSplitter(it->Get<TString>()).SplitBySet(" ,;\t.").SkipEmpty())
            TabPrefixesToCalcTheBest.emplace_back(tok.Token());
    }

    if (auto it = MapFindPtr(map, "server")) {
        ServerOptions = TServerOptions::ReadOptionsFromConfig(NTalkativeConfig::Get<NConfig::TDict>(*it));
    }

    if (auto it = MapFindPtr(map, "NewDlvFormat")) {
        NewDlvFormat = NTalkativeConfig::As<bool>(*it);
    }

    if (auto it = MapFindPtr(map, "tvm")) {
        TvmConfig = TTvmConfig::FromConfig(*it);
    }

    if (auto it = MapFindPtr(map, "MinHashSize")) {
        MinHashSize = NTalkativeConfig::As<size_t>(*it);
    }

    if (auto it = MapFindPtr(map, "Hosts2MlLimitation")) {
        Hosts2MlLimitation = NTalkativeConfig::As<size_t>(*it);
    }

    if (auto it = MapFindPtr(map, "UrlsToExclude")) {
        UrlsToExclude = StringSplitter(NTalkativeConfig::Get<TString>(*it)).SplitBySet(" ,;\t").SkipEmpty();
    }

    if (auto it = MapFindPtr(map, "Text2VecDssm")) {
        Text2VecDssm = NTalkativeConfig::As<TFsPath>(*it);
    }

    if (auto it = MapFindPtr(map, "PersFilterGetOnly")) {
        PersFilterGetOnly = NTalkativeConfig::As<bool>(*it);
    }


    if (auto it = MapFindPtr(map, "Deobfuscator")) {
        DeobfuscatorConfigs = TDeobfuscatorConfigs::FromConfig(*it);
    }

    if (auto it = MapFindPtr(map, "UseDeobfuscatorForNormalizeBeforeRe")) {
        UseDeobfuscatorForNormalizeBeforeRe = NTalkativeConfig::As<bool>(*it);
    }

    if (auto it = MapFindPtr(map, "HsRulesCache")) {
        HsRulesCache = NTalkativeConfig::Get<TString>(*it);
    }

    if (auto it = MapFindPtr(map, "DomenFactorsTrie")) {
        DomenFactorsTrie = NTalkativeConfig::As<TFsPath>(*it);
    }

    if (auto it = MapFindPtr(map, "UserWeightsFetchingPeriod")) {
        UserWeightsFetchingPeriod = NTalkativeConfig::As<TDuration>(*it);
    }

    if (auto it = MapFindPtr(map, "RulesDict")) {
        RulesDictPath = NTalkativeConfig::As<TFsPath>(*it);
    }

    if (auto it = MapFindPtr(map, "UidsFeatures")) {
        UidsFeaturesPath = NTalkativeConfig::As<TFsPath>(*it);
    }

    if (auto it = MapFindPtr(map, "KasperskyLogger")) {
        KasperskyLogger = NTalkativeConfig::Get<TString>(*it);
    }

    if (auto it = MapFindPtr(map, "JsonMlLog")) {
        JsonMlLog = NTalkativeConfig::Get<TString>(*it);
    }

    if (auto it = MapFindPtr(map, "PcreSettings")) {
        PcreSettings = NRegexp::TSettings::FromConfig(NTalkativeConfig::Get<NConfig::TDict>(*it));
    }


    if (auto it = MapFindPtr(map, "PrintPersonalResolutions")) {
        PrintPersonalResolutions = NTalkativeConfig::As<bool>(*it);
    }


    if (auto it = MapFindPtr(map, "ResolutionsMapping")) {
        for(auto tok : StringSplitter(NTalkativeConfig::Get<TString>(*it)).Split(',').SkipEmpty()) {
            TStringBuf uid, spClass;
            tok.Token().Split(":", uid, spClass);

            ResolutionsMapping.emplace(TUid{StripString(uid)}, FromString(StripString((spClass))));
        }
    }

    if (auto it = MapFindPtr(map, "ThreadStackSize")) {
        ThreadStackSize = NTalkativeConfig::As<size_t>(*it);
    }

    if (auto it = MapFindPtr(map, "RecorderDictPath")) {
        RecorderDictPath = NTalkativeConfig::As<TFsPath>(*it);
    }

    if (auto it = MapFindPtr(map, "TrustedZonesPath")) {
        TrustedZonesPath = NTalkativeConfig::As<TFsPath>(*it);
    }

    if (auto it = MapFindPtr(map, "IntranetZonesPath")) {
        IntranetZonesPath = NTalkativeConfig::As<TFsPath>(*it);
    }

    if (auto it = MapFindPtr(map, "LocalZonesPath")) {
        LocalZonesPath = NTalkativeConfig::As<TFsPath>(*it);
    }

    ServerOptions.SetThreads(solverThreads);
}

double TSoConfig::GetLvThreshold() const {
    return (LvThreshold < 0.001) ? (3. * score) / 4. : LvThreshold;
}

//#endif
