#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/charset/codepage.h>
#include "context_symbols.h"

Y_UNIT_TEST_SUITE(ContextTableTest) {
    Y_UNIT_TEST(HtmlTable) {

        const auto codePage = CodePageByCharset(CODES_KOI8);
        for(size_t i = 0; i < 256; i++) {
            const auto c = static_cast<char>(i);

            if(codePage->IsAlpha(c)) {
                UNIT_ASSERT_C(TestHtmlSymbol(c, SP_LETTER), c << " SP_LETTER");
                UNIT_ASSERT_C(!TestHtmlSymbol(c, SP_DIGIT), c << " SP_DIGIT");
                UNIT_ASSERT_EQUAL(GetHTMLSymbols(c), SP_LETTER);
                UNIT_ASSERT_UNEQUAL(GetHTMLSymbols(c), SP_DIGIT);
            } else if(codePage->IsDigit(c)) {
                UNIT_ASSERT_C(TestHtmlSymbol(c, SP_DIGIT), c << " SP_DIGIT");
                UNIT_ASSERT_C(!TestHtmlSymbol(c, SP_LETTER), c << " SP_LETTER");
                UNIT_ASSERT_EQUAL(GetHTMLSymbols(c), SP_DIGIT);
                UNIT_ASSERT_UNEQUAL(GetHTMLSymbols(c), SP_LETTER);
            }

            switch(c) {
                case 0x00:
                    UNIT_ASSERT(TestHtmlSymbol(c, SP_NULL));
                    UNIT_ASSERT_EQUAL(GetHTMLSymbols(c), SP_NULL);
                    break;
                case 0x09:
                case 0x0A:
                case 0x0B:
                case 0x0D:
                case 0x20:
                case char(160):
                    UNIT_ASSERT_C(TestHtmlSymbol(char(c), SP_SPACE), ui32(c) << ' ' << int(GetHTMLSymbols(c)));
                    UNIT_ASSERT_EQUAL(GetHTMLSymbols(c), SP_SPACE);
                    break;
                case (ui8)('&'):
                    UNIT_ASSERT(TestHtmlSymbol(c, SP_AMPERSAND));
                    UNIT_ASSERT_EQUAL(GetHTMLSymbols(c), SP_AMPERSAND);
                    break;
                case (ui8)('<'):
                    UNIT_ASSERT(TestHtmlSymbol(c, SP_LESS));
                    UNIT_ASSERT_EQUAL(GetHTMLSymbols(c), SP_LESS);
                    break;
                case (ui8)('!'):
                    UNIT_ASSERT(TestHtmlSymbol(c, SP_EXCLAM));
                    UNIT_ASSERT_EQUAL(GetHTMLSymbols(c), SP_EXCLAM);
                    break;
                case (ui8)('.'):
                    UNIT_ASSERT(TestHtmlSymbol(c, SP_DOT));
                    UNIT_ASSERT_EQUAL(GetHTMLSymbols(c), SP_DOT);
                    break;
                case (ui8)('@'):
                    UNIT_ASSERT(TestHtmlSymbol(c, SP_AT));
                    UNIT_ASSERT_EQUAL(GetHTMLSymbols(c), SP_AT);
                    break;
                case (ui8)('$'):
                    UNIT_ASSERT(TestHtmlSymbol(c, SP_DOLLAR));
                    UNIT_ASSERT_EQUAL(GetHTMLSymbols(c), SP_DOLLAR);
                    break;
            }
        }
    }
    Y_UNIT_TEST(CaseTable) {

        const auto codePage = CodePageByCharset(CODES_KOI8);
        for(size_t i = 0; i < 256; i++) {
            const auto c = static_cast<char>(i);

            if(codePage->IsUpper(c)) {
                UNIT_ASSERT(TestCaseSymbol(c, SP_UPPER));
                UNIT_ASSERT_EQUAL(GetCaseSymbols(c), SP_UPPER);
            } else if(codePage->IsLower(c)) {
                UNIT_ASSERT(TestCaseSymbol(c, SP_LOWER));
                UNIT_ASSERT_EQUAL(GetCaseSymbols(c), SP_LOWER);
            } else if (i <= 32 && TestHtmlSymbol(c, SP_SPACE)) {
                UNIT_ASSERT(TestCaseSymbol(c, SP_SPACE));
                UNIT_ASSERT_EQUAL(GetCaseSymbols(c), SP_SPACE);
            }
        }
    }

    Y_UNIT_TEST(TelTable) {
        const auto codePage = CodePageByCharset(CODES_KOI8);

        UNIT_ASSERT(TestTelSymbol('O', SP_DIGIT_FORGED));
        UNIT_ASSERT_EQUAL(GetTelSymbols('O'), SP_DIGIT_FORGED);
        UNIT_ASSERT(TestTelSymbol('I', SP_DIGIT_FORGED));
        UNIT_ASSERT_EQUAL(GetTelSymbols('I'), SP_DIGIT_FORGED);
        UNIT_ASSERT(TestTelSymbol('l', SP_DIGIT_FORGED));
        UNIT_ASSERT_EQUAL(GetTelSymbols('l'), SP_DIGIT_FORGED);
        UNIT_ASSERT(TestTelSymbol('Z', SP_DIGIT_FORGED));
        UNIT_ASSERT_EQUAL(GetTelSymbols('Z'), SP_DIGIT_FORGED);
        UNIT_ASSERT(TestTelSymbol('S', SP_DIGIT_FORGED));
        UNIT_ASSERT_EQUAL(GetTelSymbols('S'), SP_DIGIT_FORGED);
        UNIT_ASSERT(TestTelSymbol('b', SP_DIGIT_FORGED));
        UNIT_ASSERT_EQUAL(GetTelSymbols('b'), SP_DIGIT_FORGED);

        for (size_t i = 0; i < 256; i++)
            if (codePage->IsDigit(i)) {
                UNIT_ASSERT(TestTelSymbol(char(i), SP_DIGIT));
                UNIT_ASSERT_EQUAL(GetTelSymbols(char(i)), SP_DIGIT);
            }
    }

    Y_UNIT_TEST(UrlTable) {
        const auto codePage = CodePageByCharset(CODES_KOI8);
        for(size_t i = 0; i < 256; i++) {
            const auto c = static_cast<char>(i);

            if (codePage->IsAlpha(c)) {
                UNIT_ASSERT_C(TestUrlSymbol(c, SP_LAT_LETTER), c << " SP_LAT_LETTER");
                UNIT_ASSERT_EQUAL(GetUrlSymbols(c), SP_LAT_LETTER);
            } else if (codePage->IsDigit(c)) {
                UNIT_ASSERT_C(TestUrlSymbol(c, SP_DIGIT), c << " SP_DIGIT");
                UNIT_ASSERT_EQUAL(GetUrlSymbols(c), SP_DIGIT);
            }
        }
    }
}

