
#include <library/cpp/testing/unittest/registar.h>
#include <mail/so/spamstop/tools/batch_hnsw_index/index.h>
#include <util/generic/buffer.h>

struct TPoint{
    TPoint() = default;
    explicit TPoint(float v) noexcept : Value(v) {};
    float Value{};
    Y_SAVELOAD_DEFINE(Value)
};

Y_DECLARE_OUT_SPEC(inline, TPoint, s, p) {
    s << p.Value;
}

struct TDistance{
    float operator()(const TPoint& p1, const TPoint& p2) const noexcept {
        return Abs(p1.Value - p2.Value);
    }
};

using NS = NBatchHnsw::TypedNs<TPoint, TDistance>;
using TIndexManager = NS::TIndexManager;
using TIndexItem = NS::TUpdatablePoint;
using TIndexItemPtr = NS::TUpdatablePointPtr;

Y_UNIT_TEST_SUITE(BatchHnsw) {
    Y_UNIT_TEST(Ttl) {
        NBatchHnsw::TConfig config;
        config.BuffersMaxSize = 5;
        config.MaxIndexes = 1;
        config.Ttl = TDuration::MilliSeconds(200);

        TIndexManager manager(config);

        auto point = MakeAtomicShared<TIndexItem>(Now(), 5.f);
        manager.AddItem(point);

        UNIT_ASSERT_EQUAL(manager.Size(), 1);

        auto neighbors = manager.GetNearestNeighbors(point);
        UNIT_ASSERT_EQUAL(neighbors.size(), 1);
        UNIT_ASSERT_EQUAL(neighbors.front().Distance, 0);
        UNIT_ASSERT_EQUAL(neighbors.front().Item, point);

        Sleep(config.Ttl);

        neighbors = manager.GetNearestNeighbors(point);
        UNIT_ASSERT_EQUAL(neighbors.size(), 0);
    }
    Y_UNIT_TEST(Deleting) {
        NBatchHnsw::TConfig config;
        config.BuffersMaxSize = 5;
        config.MaxIndexes = 1;

        TIndexManager manager(config);
        {
            auto point = MakeAtomicShared<TIndexItem>(Now(), 5.f);
            manager.AddItem(point);
        }
        {
            auto point = MakeAtomicShared<TIndexItem>(Now(), 7.f);
            manager.AddItem(point);
        }
        auto point = MakeAtomicShared<TIndexItem>(Now(), 5.5f);

        UNIT_ASSERT_EQUAL(manager.Size(), 2);

        auto neighbors = manager.GetNearestNeighbors(point);
        UNIT_ASSERT_EQUAL(neighbors.size(), 2);
        UNIT_ASSERT_EQUAL(neighbors.front().Distance, 0.5);

        neighbors.front().Item->MarkDeleted();

        neighbors = manager.GetNearestNeighbors(point);
        UNIT_ASSERT_EQUAL(neighbors.size(), 1);
        UNIT_ASSERT_EQUAL(neighbors.front().Distance, 1.5);
    }
    Y_UNIT_TEST(SaveLoad) {
        NBatchHnsw::TConfig config;
        config.BuffersMaxSize = 500;
        config.MaxIndexes = 1;

        TBuffer buffer;
        size_t N = 1000;
        TFakeThreadPool threadPool;
        {
            TIndexManager manager(config, TDistance{}, &threadPool, &threadPool);

            const auto now = Now();
            for (size_t i = 0; i < N; i++) {
                manager.AddItem(MakeAtomicShared<TIndexItem>(now, i));
            }

            for (size_t i = 0; i < N; i++) {
                auto point = MakeAtomicShared<TIndexItem>(now, float(i));
                auto neighbors = manager.GetNearestNeighbors(point);

                UNIT_ASSERT_EQUAL_C(neighbors.front().Distance, 0, i << ' ' << neighbors.front().Distance);
                UNIT_ASSERT_EQUAL(neighbors.front().Item->GetPoint().Value, point->GetPoint().Value);
            }
            UNIT_ASSERT_EQUAL_C(manager.Size(), N, N << ' ' << manager.Size());
            TBufferOutput s(buffer);
            ::Save(&s, manager);
        }
        {
            TIndexManager manager(config, TDistance{}, &threadPool, &threadPool);
            {
                TBufferInput s(buffer);
                ::Load(&s, manager);
                UNIT_ASSERT_EQUAL_C(manager.Size(), N, N << ' ' << manager.Size());
            }

            const auto now = Now();
            for (size_t i = 0; i < N; i++) {
                auto point = MakeAtomicShared<TIndexItem>(now, float(i));
                auto neighbors = manager.GetNearestNeighbors(point);

                UNIT_ASSERT_EQUAL_C(neighbors.front().Distance, 0, neighbors.front().Distance);
                UNIT_ASSERT_EQUAL(neighbors.front().Item->GetPoint().Value, point->GetPoint().Value);
            }
        }
    }
}
