#include <library/cpp/threading/future/async.h>
#include "sofilter_cf.h"
#include <stdlib.h>
#include <library/cpp/iterator/enumerate.h>
#include <util/generic/hash.h>
#include <util/string/reverse.h>
#include "tstoragenosql.h"


CSoFilterCF::CSoFilterCF()
    : poolTraits(new NCurl::TPoolTraits{NCurl::TOptions{}})
    , curlPool(TPoolParams(128, TDuration::MilliSeconds(50)))
    , bbCache(TDuration::Minutes(10)) {
    m_configobj = NULL;
    LogsGroup = NULL;
    GroupFilter = NULL;
    LoginCheck = NULL;
    FieldCheck = NULL;
    AntiDDOSList = NULL;
    m_storagetype = STORAGE_INTERNAL;
    m_signhash = NULL;
    BlackList = NULL;
    WhiteList = NULL;

    RBLHostObj = NULL;
    AsyncResolvCache = NULL;
    m_async_resolv = false;
}

CSoFilterCF::~CSoFilterCF() {
    if (AsyncResolvCache != NULL) {
        delete AsyncResolvCache;
        AsyncResolvCache = NULL;
    }

    if (RBLHostObj != NULL) {
        delete RBLHostObj;
        RBLHostObj = NULL;
    }

    if (BlackList != NULL) {
        delete BlackList;
        BlackList = NULL;
    }
    if (WhiteList != NULL) {
        delete WhiteList;
        WhiteList = NULL;
    }
}

//
// Init
//
bool CSoFilterCF::InitBeforeFork(TKConfig* configobjA, TLogsGroupCF* LogsGroupA, TSignatureClass* signhashA, STORAGETYPE storagetypeA, TGroupFilter* GroupFilterA, TKIniFile* FieldCheckA, TString& ipcachedump, TKIniFile* LoginCheckA, TKIniFile* AntiDDOSListA) {
    TString blacklistpath = "";
    TString whitelistpath = "";

    m_configobj = configobjA;
    LogsGroup = LogsGroupA;

    m_storagetype = storagetypeA;
    GroupFilter = GroupFilterA;
    LoginCheck = LoginCheckA;
    FieldCheck = FieldCheckA;
    AntiDDOSList = AntiDDOSListA;
    m_signhash = signhashA;

    ipcache.Init(LogsGroup, ipcachedump);

    //heavystatistik
    if (m_configobj != NULL) {
        TLogClass* srvlog = NULL;
        if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
            srvlog = LogsGroup->GetServerLog();

        int addtreshold_normtext = m_configobj->ReadInteger("heavystat", "addtreshold_normtext", -1);
        int addtreshold_suid = m_configobj->ReadInteger("heavystat", "addtreshold_suid", -1);
        int addtreshold_ip = m_configobj->ReadInteger("heavystat", "addtreshold_ip", -1);

        blacklistpath = m_configobj->ReadStroka("lists", "BlackListNet", "");
        whitelistpath = m_configobj->ReadStroka("lists", "WhiteListNet", "");

        normtext_heavystat.Init("HS_NORMTEXT", srvlog, addtreshold_normtext);
        suid_heavystat.Init("HS_SUID", srvlog, addtreshold_suid);
        ip_heavystat.Init("HS_IP", srvlog, addtreshold_ip);

        if(m_configobj->ReadBool("geoserver", "enable", false)) {
            TClientConfig config;
            config.Host = m_configobj->ReadStroka("geoserver", "host", "");
            config.Port = m_configobj->ReadInteger("geoserver", "port", 80);
            config.RequestTimeout = TDuration::MilliSeconds(m_configobj->ReadInteger("geoserver", "timeout", 150));

            rblClient = MakeHolder<NFuncClient::TRbl>(TRequestClient::TArgs(std::move(config)));
        }

        if (RBLHostObj == NULL)
            RBLHostObj = new TRBLHostClass();
        if (RBLHostObj != NULL) {
            TMultiStringList rblconf_status_list;
            TMultiStringListIt rcslit;

            RBLHostObj->InitByConfig(m_configobj, LogsGroup->GetLogTrace2());

            RBLHostObj->GetInitStatus(rblconf_status_list);
            rcslit = rblconf_status_list.begin();
            while (rcslit != rblconf_status_list.end()) {
                if (!(*rcslit).empty()) {
                    if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                        LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "%s", (*rcslit).c_str());
                }

                ++rcslit;
            }
        }

        //blacklist
        if (!blacklistpath.empty()) {
            BlackList = new TNetIPv6("       BLACKLIST");
            if (BlackList != NULL)
                BlackList->Init(blacklistpath, srvlog);
            ReloadBlackList();
        }

        //whitelist
        if (!whitelistpath.empty()) {
            WhiteList = new TNetIPv6("       WHITELIST");
            if (WhiteList != NULL)
                WhiteList->Init(whitelistpath, srvlog);
            ReloadWhiteList();
        }

        const auto bbHost = m_configobj->ReadStroka("filter", "bb_host", "");
        if (bbHost) {

            const auto tvmSelfId = static_cast<const NTvmAuth::TTvmId>(m_configobj->ReadInteger("filter", "self_tvm_id", 0));
            const auto tvmBBId = static_cast<const NTvmAuth::TTvmId>(m_configobj->ReadInteger("filter", "bb_tvm_id", 0));
            const TString tvmSelfSecret = m_configobj->ReadStroka("filter", "self_tvm_secret", nullptr);

            THolder<NTvmAuth::TTvmClient> tvmClient;
            if (tvmSelfId && tvmBBId && tvmSelfSecret) {
                NTvmAuth::NTvmApi::TClientSettings tvmBBClientSettings;
                tvmBBClientSettings.SetSelfTvmId(tvmSelfId); // tvm resource ID for sp-daemon, refer to ABC

                tvmBBClientSettings.EnableServiceTicketsFetchOptions(tvmSelfSecret,
                                                                     NTvmAuth::NTvmApi::TClientSettings::TDstMap{
                                                                         {"blackbox", tvmBBId}});

                tvmClient = MakeHolder<NTvmAuth::TTvmClient>(tvmBBClientSettings, NTvmAuth::TDevNullLogger::IAmBrave());
            }
            {
                TClientConfig config;
                config.Host = bbHost;
                config.TvmServiceName = "blackbox";
                bbClient = MakeHolder<NFuncClient::CBB>(TRequestClient::TArgs(config).SetTvm(std::move(tvmClient)));
            }
        }

        if(const auto host = m_configobj->ReadStroka("filter", "logout_host", "")) {
            TClientConfig config;
            config.Host = host;

            config.ConnectTimeout = TDuration::MilliSeconds(100);
            config.RequestTimeout = TDuration::Parse(m_configobj->ReadStroka("filter", "logout_timeout", "1s"));
            logoutClient = MakeHolder<NFuncClient::TLogoutUser>(TRequestClient::TArgs(std::move(config)));
        }

        cleanWebHost = m_configobj->ReadStroka("filter", "clean_web_host", "");
        cleanWebCACert = m_configobj->ReadStroka("filter", "clean_web_ca_cert", "");
        if (cleanWebCACert) {
            NCurl::TSSL ssl;
            ssl.SetCACertFile(cleanWebCACert);
            poolTraits.Reset(new NCurl::TPoolTraits{ssl, Nothing()});
        }

        const auto urlRepHost = m_configobj->ReadStroka("filter", "url_rep_host", "");
        if(urlRepHost) {
            TClientConfig config;
            config.Host = urlRepHost;
            urlReputationClient = MakeHolder<NFuncClient::TUrlReputation>(TRequestClient::TArgs(std::move(config)));
        }

        const auto resolution_cache_ttl_s = m_configobj->ReadInteger("filter", "resolution_cache_ttl_s", 3600);
        resCache.Init(TDuration::Seconds(resolution_cache_ttl_s));
    }

    //list for exclude field to check by host, email and phone
    exclude_field_list.push_back("so_name");
    exclude_field_list.push_back("so_login");

    return true;
}

bool CSoFilterCF::InitAfterFork(void*) {
    threadPool.Start(0, 0);

    bool res = true;

    if (m_configobj != NULL) {
        bool resolv_enable = false;

        m_async_resolv = m_configobj->ReadBool("resolv", "async_mode", false);
        resolv_enable = m_configobj->ReadBool("resolv", "enable", false);

        if (m_async_resolv && resolv_enable) {
            AsyncResolvCache = new TResolvCache();
            if (AsyncResolvCache != NULL) {
                AsyncResolvCache->Init(RBLHostObj, m_configobj, LogsGroup->GetLogTrace2());
            }
        }
    }

    return res;
}

TReqParams* CSoFilterCF::DecodeParams(const TString& Numbrequest, TReqParams* pReqParams) {
    TReqParams* res = NULL;
    ui32 size1 = 0, size2 = 0, size3 = 0;
    TReqParams::iterator par_it;
    TString doccodes = "";
    char m_RequestPr[1096];
    char *ptarray = NULL, *p = NULL;
    bool extbuffer = false;
    //TString                st1 = "", st2 = "";
    ui8 codes_type = 0;
    TString sfirst = "", ssecond = "";
    TString sfirstnew = "", ssecondnew = "";
    TString stt1 = "", stt2 = "";

    if (pReqParams != NULL) {
        par_it = pReqParams->find("so_codes");
        if (par_it != pReqParams->end())
            doccodes = par_it->second[0];

        TReqParams* pReqParamsNew = new TReqParams();
        par_it = pReqParams->begin();
        while (par_it != pReqParams->end()) {
            if (par_it->second.empty()) {
                ++par_it;
                continue;
            }

            stt1 = par_it->first;
            stt2 = par_it->second[0];
            size1 = par_it->first.size();
            size2 = par_it->second[0].size();
            size3 = size1 > size2 ? size1 : size2;
            TString ts = par_it->second[0];

            if (size3 > (sizeof(m_RequestPr) - 1)) {
                ptarray = new char[size3 + 1];
                p = ptarray;
                extbuffer = true;
            } else {
                memset(m_RequestPr, 0, sizeof(m_RequestPr));
                p = m_RequestPr;
                extbuffer = false;
            }

            //text decoding
            if ((doccodes == "UTF8") || (doccodes == "utf8")) //utf8
                codes_type = 1;
            else if ((doccodes == "WIN1251") || (doccodes == "win1251")) //win
                codes_type = 2;
            else
                codes_type = 1;

            switch (codes_type) {
                case 1: //utf8
                    if (extbuffer)
                        memset(ptarray, 0, size3 + 1);
                    else
                        memset(m_RequestPr, 0, sizeof(m_RequestPr));
                    //CGIUnescape(p, par_it->first.c_str(), CODES_UTF8);
                    //sfirst = TString(p);
                    sfirst = Trim(par_it->first);

                    if (extbuffer)
                        memset(ptarray, 0, size3 + 1);
                    else
                        memset(m_RequestPr, 0, sizeof(m_RequestPr));
                    //CGIUnescape(p, par_it->second[0].c_str(), CODES_UTF8);
                    //ssecond = TString(p);
                    ssecond = par_it->second[0];

                    try {
                        sfirstnew = Recode(CODES_UTF8, CODES_WIN, sfirst);
                    } catch (...) {
                        if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                            LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "%s BROKEN_RECODE sfirst='%s'", Numbrequest.c_str(), sfirst.c_str());
                    }

                    try {
                        ssecondnew = Recode(CODES_UTF8, CODES_WIN, ssecond);
                    } catch (...) {
                        if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                            LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "%s BROKEN_RECODE sfirst='%s', *ssecond='%s'", Numbrequest.c_str(), sfirst.c_str(), ssecond.c_str());
                    }

                    break;
                case 2: //win1251
                    if (extbuffer)
                        memset(ptarray, 0, size3 + 1);
                    else
                        memset(m_RequestPr, 0, sizeof(m_RequestPr));
                    //CGIUnescape(p, par_it->first.c_str(), CODES_WIN);
                    //sfirst = TString(p);
                    sfirst = Trim(par_it->first);

                    if (extbuffer)
                        memset(ptarray, 0, size3 + 1);
                    else
                        memset(m_RequestPr, 0, sizeof(m_RequestPr));
                    //CGIUnescape(p, par_it->second[0].c_str(), CODES_WIN);
                    //ssecond = TString(p);
                    ssecond = par_it->second[0];

                    sfirstnew = sfirst;
                    ssecondnew = ssecond;

                    break;
            };
            //add to hash
            (*pReqParamsNew)[sfirstnew] = TReqValues();
            (*pReqParamsNew)[sfirstnew].push_back(RemoveCR(DecodeFromHTML(ssecondnew)));

            if (ptarray != NULL) {
                delete[] ptarray;
                ptarray = NULL;
            }

            ++par_it;
        }
        res = pReqParamsNew;
    }

    return res;
}

void CSoFilterCF::DeleteParams(TReqParams* pReqParamsNew) {
    if (pReqParamsNew != NULL) {
        pReqParamsNew->clear();
        delete pReqParamsNew;
        pReqParamsNew = NULL;
    }
}

TString CSoFilterCF::GetKey(TReqParams::iterator& it) {
    return it->first;
}

TString CSoFilterCF::GetValue(TReqParams::iterator& it) {
    TString res = "";

    if (it->second[0].length() > MAX_FIELD_SIZE)
        res = it->second[0].substr(0, MAX_FIELD_SIZE);
    else
        res = it->second[0];

    return res;
}

time_t ConvertToTimestamp(const TString& value) {
    time_t res = 0;
    TString shablon = "YYYY-MM-DD hh:mm:ss";

    if (value.length() >= shablon.length()) {
        ui32 a1 = 0, a2 = 0, a3 = 0, a4 = 0, a5 = 0, a6 = 0;
        tm tt;

        sscanf(value.c_str(), "%u-%u-%u %u:%u:%u", &a1, &a2, &a3, &a4, &a5, &a6);
        if ((a2 > 0) && (a1 > 1900)) {
            tt.tm_sec = a6;
            tt.tm_min = a5;
            tt.tm_hour = a4;
            tt.tm_mday = a3;
            tt.tm_mon = a2 - 1;
            tt.tm_year = a1 - 1900;
            res = mktime(&tt);
        }
    }

    return res;
}

ui64 CSoFilterCF::CalcGeoShingle(TString& geos) {
    ui64 res = 0;

    if (!geos.empty()) {
        ui64 shingle = 0;
        char sshingle[32];
        TString geos_t = "geodata_" + geos;

        calc_strcrc64(geos_t.c_str(), geos_t.length(), sshingle);
        sscanf(sshingle, "%lx", &shingle);

        res = shingle;
    }

    return res;
}

ui64 CSoFilterCF::CalcGeoCountryShingle(TString& geos) {
    ui64 res = 0;

    if (!geos.empty()) {
        ui64 shingle = 0;
        char sshingle[32];
        TString geos_t = "";
        TString geos_t2 = "";
        const char* p = NULL;
        int countsymb = 0;

        geos_t2 = TString{Trim(geos)};
        geos_t = "geocountry_" + geos_t2;
        p = strchr(geos_t2.c_str(), ' ');
        if (p != NULL) {
            countsymb = p - geos_t2.c_str();
            if (countsymb > 0)
                geos_t = "geocountry_" + TString(geos_t2.c_str(), countsymb);
        }

        calc_strcrc64(geos_t.c_str(), geos_t.length(), sshingle);
        sscanf(sshingle, "%lx", &shingle);

        res = shingle;
    }

    return res;
}

bool CSoFilterCF::ExcludeKey(const TString& so_service, const TString& skey_field) {
    bool res = false;

    if (("so_ip" == skey_field) ||
        ("so_form_name" == skey_field) ||
        ("so_codes" == skey_field) ||
        ("so_urlenc" == skey_field) ||
        ("so_name" == skey_field) ||
        ("so_login" == skey_field) ||
        ("so_serv_reg" == skey_field) ||
        ("sp_serv_reg" == skey_field) ||
        ("so_logn_age" == skey_field) ||
        ("so_service" == skey_field) ||
        ("so_rcpt" == skey_field)) {
        res = true;
    }

    if ((so_service == "POSTCARD") && ("so_q2" == skey_field))
        res = true;

    return res;
}

ui32 CSoFilterCF::ParseHostsMailsPhones(TReqParams* pReqParams, const TString& so_service,
                                        TZoneDetector::TKUrlHash& urls_hash, bool& urlmail_parse_max,
                                        TZoneDetector::TStrokaHash& host_list, ui32& max_host_count_in_field,
                                        TZoneDetector::TStrokaHash& mail_list, ui32& max_email_count_in_field,
                                        common::TPhoneParser::TPhoneList& phone_list, ui32& max_phone_count_in_field,
                                        stordata::TStorDataByHMP& stordata_hmp) {
    ui32 res = CShingleTime::GetMs();
    TString skey_field = "";
    TString svalue_field = "";
    TZoneDetector HostParser;
    TZoneDetector::TStrokaHash hlist_src;
    TZoneDetector::TStrokaHash mlist_src;

    urls_hash.clear();
    urlmail_parse_max = false;
    host_list.clear();
    max_host_count_in_field = 0;
    mail_list.clear();
    max_email_count_in_field = 0;
    phone_list.clear();
    max_phone_count_in_field = 0;

    for (TReqParams::iterator i = pReqParams->begin(); i != pReqParams->end(); i++) {
        skey_field = i->first;
        svalue_field = GetValue(i);

        if (ExcludeKey(so_service, skey_field))
            continue;

        // Check host stat
        bool disablecheck = false;
        for (size_t j = 0; j < exclude_field_list.size(); j++) {
            if ((skey_field.length() >= exclude_field_list[j].length()) && (!memcmp(skey_field.c_str(), exclude_field_list[j].c_str(), exclude_field_list[j].length()))) {
                disablecheck = true;
                break;
            }
        }

        bool parse_host = false;
        bool parse_email = false;
        bool need_parse_host = (!disablecheck) && (FieldCheck != NULL) && (FieldCheck->KeyExists("host", skey_field));
        bool need_parse_email = (!disablecheck) && (FieldCheck != NULL) && (FieldCheck->KeyExists("email", skey_field));
        if (need_parse_host || need_parse_email) {
            if (HostParser.GetEmailsUrlsUniqLimited(svalue_field.c_str(), svalue_field.length(), false, hlist_src, mlist_src, urls_hash, MAX_PARSE_URLMAIL_COUNT + 1) > 0) {
                parse_host = true;
                parse_email = true;

                if ((hlist_src.size() + mlist_src.size()) > MAX_PARSE_URLMAIL_COUNT)
                    urlmail_parse_max = true;
            }
        }

        //hosts
        if (parse_host) {
            TZoneDetector::TStrokaHashIt it;
            TZoneDetector::TStrokaHashIt it2;
            ui32 hostcount = 0;
            int thishostcount = 0;
            TZoneDetector::TStrokaHash hlist_dob;
            TString host = "";
            TString host2 = "";
            TString host3 = "";
            ui8 thishostcount_byte = 0;
            TString urlbody = "";

            //�������� ������ ������� � �������� �������
            hlist_dob.clear();
            it = hlist_src.begin();
            while (it != hlist_src.end()) {
                host = (*it).first;
                host2 = GetDomen2level(host);
                host3 = GetDomen3level(host);
                thishostcount = (*it).second;
                if (thishostcount > 0xFF)
                    thishostcount_byte = 0xFF;
                else
                    thishostcount_byte = thishostcount;

                if (!host2.empty()) {
                    if (!HostParser.CheckCommonZone(host2.c_str())) {
                        it2 = hlist_dob.find(host2);
                        if (it2 != hlist_dob.end())
                            (*it2).second = IncMax8((*it2).second, thishostcount_byte);
                        else
                            hlist_dob[host2] = thishostcount_byte;
                    }
                }

                if (!host3.empty()) {
                    if (!HostParser.CheckCommonZone(host3.c_str())) {
                        it2 = hlist_dob.find(host3);
                        if (it2 != hlist_dob.end())
                            (*it2).second = IncMax8((*it2).second, thishostcount_byte);
                        else
                            hlist_dob[host3] = thishostcount_byte;
                    }
                }

                ++it;
            }
            //��������� ������ ������� � �������� �������
            it = hlist_dob.begin();
            while (it != hlist_dob.end()) {
                host = (*it).first;
                thishostcount = (*it).second;
                if (thishostcount > 0xFF)
                    thishostcount_byte = 0xFF;
                else
                    thishostcount_byte = thishostcount;

                if (!host.empty()) {
                    it2 = hlist_src.find(host);
                    if (it2 != hlist_src.end())
                        (*it2).second = IncMax8((*it2).second, thishostcount_byte);
                    else
                        hlist_src[host] = thishostcount_byte;
                }

                ++it;
            }

            //������������ ����� � ������, �������� ��������� � ������ ����������
            hostcount = 0;
            it = hlist_src.begin();
            while (it != hlist_src.end()) {
                host = (*it).first;
                thishostcount = (*it).second;
                //������� ������, ��������� � excludehosts (fieldcheck.txt)
                if (!FieldCheck->KeyExistsCompare("excludehosts", host)) {
                    host_list[(*it).first] = (*it).second;
                    hostcount = IncMax32(hostcount, 1);
                }

                ++it;
            }
            if (hostcount > max_host_count_in_field)
                max_host_count_in_field = hostcount;

            //fill vector to receive data from storage
            stordata_hmp.host_list_size = 0;
            TZoneDetector::TStrokaHash::iterator host_it = host_list.begin();
            while (host_it != host_list.end()) {
                if (stordata_hmp.host_list_size >= MAX_HOST_COUNT)
                    break;

                if (!host_it->first.empty()) {
                    stordata_hmp.host_list[stordata_hmp.host_list_size] = stordata::TTxtData(host_it->first);
                    stordata_hmp.host_list_size = IncMax32(stordata_hmp.host_list_size, 1);
                }

                ++host_it;
            }
        }

        //email
        if (parse_email) {
            TZoneDetector::TStrokaHashIt it;
            ui32 emailcount = 0;
            int thisemailcount = 0;
            TString email = "";

            emailcount = 0;
            it = mlist_src.begin();
            while (it != mlist_src.end()) {
                email = (*it).first;
                thisemailcount = (*it).second;
                //������������ ������ ������, �� ��������� � excludeemail (fieldcheck.txt)
                if (!FieldCheck->KeyExistsCompare("excludeemail", email)) {
                    mail_list[(*it).first] = (*it).second;
                    emailcount = IncMax32(emailcount, 1);
                }

                ++it;
            }
            if (emailcount > max_email_count_in_field)
                max_email_count_in_field = emailcount;

            //fill vector to receive data from storage
            stordata_hmp.mail_list_size = 0;
            TZoneDetector::TStrokaHash::iterator mail_it = mail_list.begin();
            while (mail_it != mail_list.end()) {
                if (stordata_hmp.mail_list_size >= MAX_HOST_COUNT)
                    break;

                if (!mail_it->first.empty()) {
                    stordata_hmp.mail_list[stordata_hmp.mail_list_size] = stordata::TTxtData(mail_it->first);
                    stordata_hmp.mail_list_size = IncMax32(stordata_hmp.mail_list_size, 1);
                }

                ++mail_it;
            }
        }

        //phone
        if ((!disablecheck) && (FieldCheck != NULL) && (FieldCheck->KeyExists("phone", skey_field))) {
            TIStrokaList slist;
            ui32 phonecount = 0;
            common::TPhoneParser PhoneParser;
            common::TPhoneParser::TPhoneList plist_src;
            common::TPhoneParser::TPhoneListIt pit;
            TString phone = "";

            if (PhoneParser.GetAllPhones(svalue_field.c_str(), svalue_field.length(), plist_src) > 0) {
                phonecount = 0;
                pit = plist_src.begin();
                while (pit != plist_src.end()) {
                    phone = (*pit);
                    //������������ ������ ��������, �� ��������� � excludephone (fieldcheck.txt)
                    if (!FieldCheck->KeyExistsCompare("excludephone", phone)) {
                        phone_list.push_back(phone);
                        phonecount = IncMax32(phonecount, 1);
                    }

                    ++pit;
                }
                if (phonecount > max_phone_count_in_field)
                    max_phone_count_in_field = phonecount;
            }

            //fill vector to receive data from storage
            stordata_hmp.phone_list_size = 0;
            common::TPhoneParser::TPhoneList::iterator phone_it = phone_list.begin();
            while (phone_it != phone_list.end()) {
                if (stordata_hmp.phone_list_size >= MAX_HOST_COUNT)
                    break;

                if (!(*phone_it).empty()) {
                    stordata_hmp.phone_list[stordata_hmp.phone_list_size] = stordata::TTxtData(*phone_it);
                    stordata_hmp.phone_list_size = IncMax32(stordata_hmp.phone_list_size, 1);
                }

                ++phone_it;
            }
        }
    }

    res = CShingleTime::GetMs() - res;

    return res;
}

ui64 CSoFilterCF::CreateShingleWithService(TKIPv6 ip, const TString& so_service) {
    ui64 res = 0;

    if (!ip.Undefined() && !so_service.empty())
        res = ShingleFromStroka(so_service + "_" + ip.toStroka2());

    return res;
}

ui64 CSoFilterCF::CreateShingleWithService(const TString& text, const TString& so_service) {
    ui64 res = 0;

    if (!text.empty() && !so_service.empty())
        res = ShingleFromStroka(so_service + "_" + text);

    return res;
}

ui64 CSoFilterCF::CreateShingleWithService(ui64 shingle, const TString& so_service) {
    ui64 res = 0;

    if ((shingle != 0) && !so_service.empty())
        res = ShingleFromStroka(so_service + "_" + ShingleToStroka(shingle));

    return res;
}

void CSoFilterCF::CreateServiceShingleList(stordata::TShingleDataList& srvc_shingles_list, TObrabParams& odata) {
    TString so_service = odata.so_service;

    if (!odata.ip.Undefined())
        srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.ip, so_service), so_service, stordata::IPSTAT, &odata.srvc_res.ipstat));

    if ((!odata.login.empty()) && (odata.login != "-"))
        srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.login, so_service), so_service, stordata::LOGINSTAT, &odata.srvc_res.loginstat));

    if (odata.so_service == "XMPP") //� ������� XMPP � ��������� ���� ����� �������� ���������� �� ������� xmpp
    {
        if (odata.xmpp_server_name_shingle != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.xmpp_server_name_shingle, so_service), so_service, stordata::NICKSTAT, &odata.srvc_res.xmpp_server_stat));
    } else {
        if (odata.so_nick_shingle != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.so_nick_shingle, so_service), so_service, stordata::NICKSTAT, &odata.srvc_res.nickstat));
    }

    if (odata.so_comment_shingle != 0)
        srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.so_comment_shingle, so_service), so_service, stordata::COMMENTSTAT, &odata.srvc_res.commentstat));

    if (odata.so_subject_shingle != 0)
        srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.so_subject_shingle, so_service), so_service, stordata::SUBJECTSTAT, &odata.srvc_res.subjectstat));

    if (odata.geo_shingle != 0)
        srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.geo_shingle, so_service), so_service, stordata::GEO, &odata.srvc_res.geostat));

    if (odata.geocountry_shingle != 0)
        srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.geocountry_shingle, so_service), so_service, stordata::GEOCOUNTRY, &odata.srvc_res.geocountrystat));

    //������ ��� ������� POSTCARD
    if (odata.so_idpostcard_shingle != 0)
        srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.so_idpostcard_shingle, so_service), so_service, stordata::IDPOSTCARD, &odata.srvc_res.idpostcardstat));

    //������ ��� ������� MSEARCH-PROXY
    if (odata.so_service == "MSEARCH-PROXY") {
        if (odata.ip_suid_total_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.ip_suid_total_hash, so_service), so_service, stordata::SUIDTOTAL, &odata.srvc_res.ip_suid_total_stat));

        if (odata.ip_text_total_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.ip_text_total_hash, so_service), so_service, stordata::TEXTTOTAL, &odata.srvc_res.ip_text_total_stat));

        if (odata.ip_suid_text_total_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.ip_suid_text_total_hash, so_service), so_service, stordata::IPSUIDTEXTTOTAL, &odata.srvc_res.ip_suid_text_total_stat));

        if (odata.ip_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.ip_found_hash, so_service), so_service, stordata::IPFOUND, &odata.srvc_res.ip_found_stat));

        if (odata.suid_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.suid_found_hash, so_service), so_service, stordata::LOGINFOUND, &odata.srvc_res.suid_found_stat));

        if (odata.text_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.text_found_hash, so_service), so_service, stordata::TEXTFOUND, &odata.srvc_res.text_found_stat));

        if (odata.geos_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.geos_found_hash, so_service), so_service, stordata::GEOSFOUND, &odata.srvc_res.geos_found_stat));

        if (odata.ip_suid_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.ip_suid_found_hash, so_service), so_service, stordata::IPSUIDFOUND, &odata.srvc_res.ip_suid_found_stat));

        if (odata.ip_text_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.ip_text_found_hash, so_service), so_service, stordata::IPTEXTFOUND, &odata.srvc_res.ip_text_found_stat));
    }

    if ((odata.stordata_hmp.host_list_size > 0) && (odata.stordata_hmp.host_list_size < MAX_HOST_COUNT)) {
        for (size_t k = 0; k < odata.stordata_hmp.host_list_size; k++) {
            if (odata.stordata_hmp.host_list[k].m_shingle != 0)
                srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.stordata_hmp.host_list[k].m_shingle, so_service), so_service, stordata::HOSTSTAT, &odata.stordata_hmp.host_list[k].m_service_data));
        }
    }

    if ((odata.stordata_hmp.mail_list_size > 0) && (odata.stordata_hmp.mail_list_size < MAX_HOST_COUNT)) {
        for (size_t k = 0; k < odata.stordata_hmp.mail_list_size; k++) {
            if (odata.stordata_hmp.mail_list[k].m_shingle != 0)
                srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.stordata_hmp.mail_list[k].m_shingle, so_service), so_service, stordata::EMAILSTAT, &odata.stordata_hmp.mail_list[k].m_service_data));
        }
    }

    if ((odata.stordata_hmp.phone_list_size > 0) && (odata.stordata_hmp.phone_list_size < MAX_HOST_COUNT)) {
        for (size_t k = 0; k < odata.stordata_hmp.phone_list_size; k++) {
            if (odata.stordata_hmp.phone_list[k].m_shingle != 0)
                srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.stordata_hmp.phone_list[k].m_shingle, so_service), so_service, stordata::PHONESTAT, &odata.stordata_hmp.phone_list[k].m_service_data));
        }
    }
}

void CSoFilterCF::CreateShingleListMSearchFound(stordata::TShingleDataList& srvc_shingles_list, TObrabParams& odata) {
    TString so_service = odata.so_service;

    //������ ��� ������� MSEARCH-PROXY
    if (odata.so_service == "MSEARCH-PROXY") {
        if (odata.ip_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.ip_found_hash, so_service), so_service, stordata::IPFOUND, &odata.srvc_res.ip_found_stat));

        if (odata.suid_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.suid_found_hash, so_service), so_service, stordata::LOGINFOUND, &odata.srvc_res.suid_found_stat));

        if (odata.text_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.text_found_hash, so_service), so_service, stordata::TEXTFOUND, &odata.srvc_res.text_found_stat));

        if (odata.geos_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.geos_found_hash, so_service), so_service, stordata::GEOSFOUND, &odata.srvc_res.geos_found_stat));

        if (odata.ip_suid_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.ip_suid_found_hash, so_service), so_service, stordata::IPSUIDFOUND, &odata.srvc_res.ip_suid_found_stat));

        if (odata.ip_text_found_hash != 0)
            srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.ip_text_found_hash, so_service), so_service, stordata::IPTEXTFOUND, &odata.srvc_res.ip_text_found_stat));
    }
}

void CSoFilterCF::CreateStatShingleList(stordata::TShingleDataList& srvc_shingles_list, TObrabParams& odata) {
    //TString so_service = odata.so_service;
    TString so_service = odata.so_service_from_fitem;

    srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(BASE_STAT_SHINGLE_RQST, so_service), TString(BASE_STAT_SHINGLE_DOPSTR) + so_service, stordata::SERVICE_STAT_RQST, &odata.srvc_data.m_request, true));
    srvc_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(BASE_STAT_SHINGLE_RCPT, so_service), TString(BASE_STAT_SHINGLE_DOPSTR) + so_service, stordata::SERVICE_STAT_RCPT, &odata.srvc_data.m_rcpt, true));
}

ui32 CSoFilterCF::GetServiceStorage(const TString& NumbRequest, TStoreBase* store, TObrabParams& odata, ui32& service_stat_collision_count, ui32& shingles_collision_count) {
    ui32 res = CShingleTime::GetMs();

    service_stat_collision_count = 0;
    shingles_collision_count = 0;

    if (store != NULL) {
        stordata::TShingleDataList srvc_shingles_list;

        CreateServiceShingleList(srvc_shingles_list, odata);
        CreateCommonShingleList(srvc_shingles_list, odata, false);
        CreateStatShingleList(srvc_shingles_list, odata);

        //!!!NEED REQUEST BY SERVICE STATISTIK (odata.srvc_data)!!!

        //need read service function
        store->GetDataFromStorage(NumbRequest, srvc_shingles_list, service_stat_collision_count, shingles_collision_count);

        //anti ddos
        if (!odata.ip.Undefined())
            AddAntiDDOSActionIP(store, odata.so_service, odata.ip, odata.srvc_res.ipstat);

        if ((!odata.login.empty()) && (odata.login != "-"))
            AddAntiDDOSActionLogin(store, odata.so_service, odata.login, odata.srvc_res.loginstat);

        if (odata.so_service == "XMPP") //� ������� XMPP � ��������� ���� ����� �������� ���������� �� ������� xmpp
        {
            if (odata.xmpp_server_name_shingle != 0)
                AddAntiDDOSActionXMPPServerName(store, odata.so_service, odata.xmpp_server_name_shingle, odata.srvc_res.xmpp_server_stat);

        } else {
            if (odata.so_nick_shingle != 0)
                AddAntiDDOSActionNick(store, odata.so_service, odata.so_nick_shingle, odata.srvc_res.nickstat);
        }

        if (odata.so_comment_shingle != 0)
            AdddAntiDDOSActionComment(store, odata.so_service, odata.so_comment_shingle, odata.srvc_res.commentstat);

        if (odata.so_subject_shingle != 0)
            AdddAntiDDOSActionSubject(store, odata.so_service, odata.so_subject_shingle, odata.srvc_res.subjectstat);
    }

    res = CShingleTime::GetMs() - res;

    return res;
}

struct ToCGIParam {
    TString operator()(const TLogin& login) const {
        return TStringBuilder{} << "login=" << (const TString&)login;
    }
    TString operator()(const TUid& uid) {
        return TStringBuilder{} << "uid=" << (const TString&)uid;
    }
};

struct ToHash {
    size_t operator()(const TLogin& login) const {
        return ComputeHash(login);
    }
    size_t operator()(const TUid& uid) {
        return ComputeHash(uid);
    }
};

TAtomicSharedPtr<NBlackbox2::TResponse> CSoFilterCF::GetBB(const NFuncClient::CBB::TBbKey& key) {
    const auto loginHash = std::visit(ToHash{}, key);
    {
        TBBInfo bbInfo;
        if (bbCache.Get(loginHash, bbInfo))
            return bbInfo;
    }

    NBlackbox2::TOptions options;
    {
        NBlackbox2::TDBFields dbFields;
        dbFields << "userinfo.reg_date.uid";
        options << dbFields;
    }

    if (TBBInfo bbInfo = bbClient->GetUserInfo(NFuncClient::CBB::MakeRequest(key, options), SysLogger())) {
        bbCache.Add(loginHash, bbInfo);
        return bbInfo;
    } else {
        ythrow yexception() << "for login " << std::visit(ToCGIParam{}, key) << " empty response";
    }
}

TString CSoFilterCF::CleanWebRequest(const TString& comment) {
    const auto commentHash = ComputeHash(comment);
    {
        TString value;
        if (cleanWebCache.Get(commentHash, value))
            return value;
    }

    const auto utf8Comment = Recode(ECharset::CODES_WIN, ECharset::CODES_UTF8, comment);

    auto curlHolder = curlPool.get(*poolTraits);

    NCurl::TRequestContext requestContext;

    {
        TStringStream postData;
        {
            NJson::TJsonWriter writer(&postData, false);
            writer.OpenMap();
            writer.Write("jsonrpc", "2.0");
            writer.Write("method", "process");
            writer.Write("id", commentHash);
            writer.OpenMap("params");
            writer.Write("service", "so");
            writer.Write("type", "text");
            writer.Write("key", "doc3");
            writer.OpenMap("body");
            writer.Write("text", utf8Comment);
            writer.CloseMap();
            writer.CloseMap();
            writer.CloseMap();
            writer.Flush();
        }
        requestContext
            .SetContentType("application/json")
            .SetHost(cleanWebHost)
            .SetPostData(std::move(postData.Str()));
    }

    NCurl::TSimpleArtifacts artifacts;


    if(const auto error = curlHolder.Get().Setup(std::move(requestContext)).Perform(artifacts)) {
        ythrow yexception() << *error;
    }

    return artifacts.body.ReadAll();
}

ui32 CSoFilterCF::UpdateServiceStorage(const TString& NumbRequest, TStoreBase* store, TObrabParams& odata, const TString& normtext, bool Spam, int rcpt_count, chkfrm::TDelayClass& /*DelayClass*/) {
    ui32 res = CShingleTime::GetMs();

    if ((!odata.no_write_storage) && (store != NULL)) {
        stordata::TShingleDataList srvc_shingles_list;

        CreateServiceShingleList(srvc_shingles_list, odata);
        CreateCommonShingleList(srvc_shingles_list, odata, true);
        CreateStatShingleList(srvc_shingles_list, odata);

        //need update service function
        stordata::RESTYPE res_d = Spam ? stordata::RT_SPAM : stordata::RT_HAM;
        store->UpdateDataToStorage(NumbRequest, srvc_shingles_list, res_d, rcpt_count);

        if (odata.so_service == "MSEARCH-PROXY") {
            normtext_heavystat.AddData(odata.srvc_res.commentstat, normtext, Spam, odata.ip, odata.so_nick, "");
            suid_heavystat.AddData(odata.srvc_res.nickstat, odata.so_nick, Spam, odata.ip, odata.so_nick, "");
            ip_heavystat.AddData(odata.srvc_res.ipstat, odata.ip.toStroka2(), Spam, odata.ip, odata.so_nick, odata.geos);
        }
    }

    res = CShingleTime::GetMs() - res;

    return res;
}

void CSoFilterCF::AddStatError(const TString& NumbRequest, TStoreBase* store, TObrabParams& odata, ui16 rcpt_count) {
    if ((!odata.no_write_storage) && (store != NULL)) {
        stordata::TShingleDataList srvc_shingles_list;

        CreateStatShingleList(srvc_shingles_list, odata);

        //need update service function
        store->UpdateDataToStorage(NumbRequest, srvc_shingles_list, stordata::RT_ERROR, rcpt_count);
    }
}

ui32 CSoFilterCF::UpdateMSearchFound(const TString& NumbRequest, TStoreBase* store, TObrabParams& odata, bool Spam, int rcpt_count, chkfrm::TDelayClass& /*DelayClass*/) {
    ui32 res = CShingleTime::GetMs();

    if ((!odata.no_write_storage) && (store != NULL)) {
        stordata::TShingleDataList srvc_shingles_list;

        CreateShingleListMSearchFound(srvc_shingles_list, odata);

        //need update service function
        stordata::RESTYPE res_d = Spam ? stordata::RT_SPAM : stordata::RT_HAM;
        store->UpdateDataToStorage(NumbRequest, srvc_shingles_list, res_d, rcpt_count);
    }

    res = CShingleTime::GetMs() - res;

    return res;
}

ui32 CSoFilterCF::GetLongIPStat(const TString& NumbRequest, TStoreBase* store, const TString& so_service, const TString& ip, TLongIPData& longipdata, ui32& ipstat_collision_count) {
    ui32 res = CShingleTime::GetMs();

    ipstat_collision_count = 0;
    if (store != NULL && ip) {
        ui64 shingle = 0;
        TString tstr = "";
        bool collision_error = false;

        tstr = so_service + "_" + ip;
        shingle = ShingleFromStroka(tstr);
        stordata::TShingleData ipstat = stordata::TShingleData(shingle, so_service);

        longipdata = store->GetLongIPStat(NumbRequest, ipstat, collision_error);
        if (collision_error)
            ipstat_collision_count = IncMax32(ipstat_collision_count, 1);
    }

    res = CShingleTime::GetMs() - res;

    return res;
}

ui32 CSoFilterCF::UpdateLongIPStat(const TString& NumbRequest, TStoreBase* store, const TString& so_service, const TString& ip, stordata::TLongIPCnt cnt_type) {
    ui32 res = CShingleTime::GetMs();

    if (store != NULL && ip) {
        ui64 shingle = 0;
        TString tstr = "";

        tstr = so_service + "_" + ip;
        shingle = ShingleFromStroka(tstr);
        stordata::TShingleData ipstat = stordata::TShingleData(shingle, so_service);

        store->UpdateLongIPStat(NumbRequest, ipstat, cnt_type);
    }

    res = CShingleTime::GetMs() - res;

    return res;
}

void CSoFilterCF::CreateCommonShingleList(stordata::TShingleDataList& common_shingles_list, TObrabParams& odata, bool up) {
    TString so_service = COMMON_SERVICE;

    if (!odata.ip.Undefined())
        common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.ip, so_service), so_service, stordata::IPSTAT, &odata.common_res.ipstat));

    if ((!odata.login.empty()) && (odata.login != "-"))
        common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.login, so_service), so_service, stordata::LOGINSTAT, &odata.common_res.loginstat));

    if (odata.so_service == "XMPP") //� ������� XMPP � ���������� ��� ����� ������ �� �����������, ����� �� ������ ����������
    {
    } else {
        if (odata.so_nick_shingle != 0)
            common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.so_nick_shingle, so_service), so_service, stordata::NICKSTAT, &odata.common_res.nickstat));
    }

    if (odata.so_comment_shingle != 0)
        common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.so_comment_shingle, so_service), so_service, stordata::COMMENTSTAT, &odata.common_res.commentstat));

    if (odata.so_login_chatid_shingle != 0) {
        common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.so_login_chatid_shingle, so_service), so_service, stordata::LOGIN_CHATID_STAT, &odata.common_res.loginchatidstat));
    }

    if (odata.so_uniq_chats_by_login_shingle &&
        (!up || (odata.so_login_chatid_shingle && odata.common_res.loginchatidstat.All() == 0)))
        common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.so_uniq_chats_by_login_shingle, so_service), so_service, stordata::UNIQ_CHATIDS_BY_LOGIN_STAT, &odata.common_res.uniquechatsbyloginstat));

    if (odata.so_subject_shingle != 0)
        common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.so_subject_shingle, so_service), so_service, stordata::SUBJECTSTAT, &odata.common_res.subjectstat));

    if (odata.geo_shingle != 0)
        common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.geo_shingle, so_service), so_service, stordata::GEO, &odata.common_res.geostat));

    if (odata.geocountry_shingle != 0)
        common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.geocountry_shingle, so_service), so_service, stordata::GEOCOUNTRY, &odata.common_res.geocountrystat));

    if ((odata.stordata_hmp.host_list_size > 0) && (odata.stordata_hmp.host_list_size < MAX_HOST_COUNT)) {
        for (size_t k = 0; k < odata.stordata_hmp.host_list_size; k++) {
            if (odata.stordata_hmp.host_list[k].m_shingle != 0)
                common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.stordata_hmp.host_list[k].m_shingle, so_service), so_service, stordata::HOSTSTAT, &odata.stordata_hmp.host_list[k].m_common_data));
        }
    }

    if ((odata.stordata_hmp.mail_list_size > 0) && (odata.stordata_hmp.mail_list_size < MAX_HOST_COUNT)) {
        for (size_t k = 0; k < odata.stordata_hmp.mail_list_size; k++) {
            if (odata.stordata_hmp.mail_list[k].m_shingle != 0)
                common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.stordata_hmp.mail_list[k].m_shingle, so_service), so_service, stordata::EMAILSTAT, &odata.stordata_hmp.mail_list[k].m_common_data));
        }
    }

    if ((odata.stordata_hmp.phone_list_size > 0) && (odata.stordata_hmp.phone_list_size < MAX_HOST_COUNT)) {
        for (size_t k = 0; k < odata.stordata_hmp.phone_list_size; k++) {
            if (odata.stordata_hmp.phone_list[k].m_shingle != 0)
                common_shingles_list.push_back(stordata::TShingleData(CreateShingleWithService(odata.stordata_hmp.phone_list[k].m_shingle, so_service), so_service, stordata::PHONESTAT, &odata.stordata_hmp.phone_list[k].m_common_data));
        }
    }
}

TFilterResult CSoFilterCF::AppleFilter(
        const TString& sid,
        const NJson::TJsonValue& js,
        const TString& filter_index,
        TReqParams* pReqParams,
        TRengine* m_hSp,
        TObrabParams& odata,
        chkfrm::TDelayClass& DelayClass,
        ui32& filter_work_tick,
        TMaybe<TReceipt>& receipt,
        TLogsGroupCF* logsGroup,
        const TString& service) {
    TFilterResult res = TFilterResult();
    ui32 ttt = 0;

    filter_work_tick = CShingleTime::GetMs();

    if (!m_hSp) {
        return res;
    }
    TString so_comment_log = "";
    TString so_subject_log = "";
    TString skey_field = "";
    TString svalue_field = "";
    bool nowritetodlvr = false;
    TString allfields = " ";
    TString formname_fieldname = "";
    TString url = "";
    TBodyPartProperty property;
    const char* pPureUtf8Text = NULL;

    // Invoke filter
    ttt = CShingleTime::GetMs();
    m_hSp->InitMessage(sid, nullptr, nullptr, "", TSpClass::UNKNOWN, SysLogger());
    Y_DEFER{
        m_hSp->EndMessage(logsGroup->GetDeliveryLog()->GetLogger(service), {});
    };

    SpSetSenderUID(m_hSp, odata.login);

    if(js.IsMap())
        m_hSp->CheckValueAllRules(FD_request_json, js);

    m_hSp->AddStat("*srvc", odata.so_service);
    m_hSp->AddStat("fidx", filter_index);
    m_hSp->AddStat("*  ip", ("(l:" + UI32ToStroka(odata.srvc_res.ipstat.Spam()) + "/" + UI32ToStroka(odata.srvc_res.ipstat.All()) + ", c:" + UI32ToStroka(odata.common_res.ipstat.Spam()) + "/" + UI32ToStroka(odata.common_res.ipstat.All()) + ") " + odata.ip.toStroka()));

    if (odata.is_black_ip)
        SpSetRule(m_hSp, "IS_BLACK_LIST");
    if (odata.is_white_ip)
        SpSetRule(m_hSp, "IS_WHITE_LIST");

    m_hSp->AddStat("*form", odata.form_name);
    if ((!odata.login.empty()) && (odata.login != "-"))
        m_hSp->AddStat("*logn", ("(l:" + UI32ToStroka(odata.srvc_res.loginstat.Spam()) + "/" + UI32ToStroka(odata.srvc_res.loginstat.All()) + ", c:" + UI32ToStroka(odata.common_res.loginstat.Spam()) + "/" + UI32ToStroka(odata.common_res.loginstat.All()) + ") " + odata.login));
    m_hSp->AddStat("*nick", ("(s:" + IntToStroka(odata.so_nick_length) + ", w:" + IntToStroka(odata.so_nick_words) + ", l:" + UI32ToStroka(odata.srvc_res.nickstat.Spam()) + "/" + UI32ToStroka(odata.srvc_res.nickstat.All()) + ", c:" + UI32ToStroka(odata.common_res.nickstat.Spam()) + "/" + UI32ToStroka(odata.common_res.nickstat.All()) + ") " + odata.so_nick));
    if (odata.so_service == "XMPP")
        m_hSp->AddStat("*xmppserv", ("(l:" + UI32ToStroka(odata.srvc_res.xmpp_server_stat.Spam()) + "/" + UI32ToStroka(odata.srvc_res.xmpp_server_stat.All()) + ") " + odata.xmpp_server_name));

    if (odata.so_comment.length() > MAX_COMMENT_LOG)
        so_comment_log = odata.so_comment.substr(0, MAX_COMMENT_LOG);
    else
        so_comment_log = odata.so_comment;
    m_hSp->AddStat("*cmmt", ("(s:" + IntToStroka(odata.so_comment_length) + ", w:" + IntToStroka(odata.so_comment_words) + ", l:" + UI32ToStroka(odata.srvc_res.commentstat.Spam()) + "/" + UI32ToStroka(odata.srvc_res.commentstat.All()) + ", c:" + UI32ToStroka(odata.common_res.commentstat.Spam()) + "/" + UI32ToStroka(odata.common_res.commentstat.All()) + ") " + so_comment_log));

    if (odata.so_subject.length() > MAX_COMMENT_LOG)
        so_subject_log = odata.so_subject.substr(0, MAX_COMMENT_LOG);
    else
        so_subject_log = odata.so_subject;
    m_hSp->AddStat("*subj", ("(s:" + IntToStroka(odata.so_subject_length) + ", w:" + IntToStroka(odata.so_subject_words) + ", l:" + UI32ToStroka(odata.srvc_res.subjectstat.Spam()) + "/" + UI32ToStroka(odata.srvc_res.subjectstat.All()) + ", c:" + UI32ToStroka(odata.common_res.subjectstat.Spam()) + "/" + UI32ToStroka(odata.common_res.subjectstat.All()) + ") " + so_subject_log));
    m_hSp->AddStat("*pcid", ("(l:" + UI32ToStroka(odata.srvc_res.idpostcardstat.Spam()) + "/" + UI32ToStroka(odata.srvc_res.idpostcardstat.All()) + ") " + odata.id_postcard));

    if (odata.so_service == "MSEARCH-PROXY") {
        m_hSp->AddStat("*ipts", ("ipsuid(a/s/h)=" + IntToStroka(odata.srvc_res.ip_suid_total_stat.All()) + "/" + IntToStroka(odata.srvc_res.ip_suid_total_stat.Spam()) + "/" + IntToStroka(odata.srvc_res.ip_suid_total_stat.Ham()) + ", iptext(a/s/h)=" + IntToStroka(odata.srvc_res.ip_text_total_stat.All()) + "/" + IntToStroka(odata.srvc_res.ip_text_total_stat.Spam()) + "/" + IntToStroka(odata.srvc_res.ip_text_total_stat.Ham()) + ", ipsuidtext(a/s/h)=" + IntToStroka(odata.srvc_res.ip_suid_text_total_stat.All()) + +"/" + IntToStroka(odata.srvc_res.ip_suid_text_total_stat.Spam()) + "/" + IntToStroka(odata.srvc_res.ip_suid_text_total_stat.Ham())));
        m_hSp->AddStat("*fond", ("ip(0/1)=" + IntToStroka(odata.srvc_res.ip_found_stat.Spam()) + "/" + IntToStroka(odata.srvc_res.ip_found_stat.Ham()) + ", suid(0/1)=" + IntToStroka(odata.srvc_res.suid_found_stat.Spam()) + "/" + IntToStroka(odata.srvc_res.suid_found_stat.Ham()) + ", text(0/1)=" + IntToStroka(odata.srvc_res.text_found_stat.Spam()) + "/" + IntToStroka(odata.srvc_res.text_found_stat.Ham()) + ", geos(0/1)=" + IntToStroka(odata.srvc_res.geos_found_stat.Spam()) + "/" + IntToStroka(odata.srvc_res.geos_found_stat.Ham()) + ", ipsuid(0/1)=" + IntToStroka(odata.srvc_res.ip_suid_found_stat.Spam()) + "/" + IntToStroka(odata.srvc_res.ip_suid_found_stat.Ham()) + ", iptext(0/1)=" + IntToStroka(odata.srvc_res.ip_text_found_stat.Spam()) + "/" + IntToStroka(odata.srvc_res.ip_text_found_stat.Ham())));
        //if (odata.msearch_offset >= 0)
        //   m_hSp->AddStat("*ofst", IntToStrokaS(odata.msearch_offset));
        m_hSp->CheckRange("mp_ip_suid_total_stat_all", odata.srvc_res.ip_suid_total_stat.All());
        m_hSp->CheckRange("mp_ip_suid_total_stat_spam", odata.srvc_res.ip_suid_total_stat.Spam());
        m_hSp->CheckRange("mp_ip_suid_total_stat_ham", odata.srvc_res.ip_suid_total_stat.Ham());
        m_hSp->CheckRange("mp_ip_text_total_stat_all", odata.srvc_res.ip_text_total_stat.All());
        m_hSp->CheckRange("mp_ip_text_total_stat_spam", odata.srvc_res.ip_text_total_stat.Spam());
        m_hSp->CheckRange("mp_ip_text_total_stat_ham", odata.srvc_res.ip_text_total_stat.Ham());
        m_hSp->CheckRange("mp_ip_suid_text_total_stat_all", odata.srvc_res.ip_suid_text_total_stat.All());
        m_hSp->CheckRange("mp_ip_suid_text_total_stat_spam", odata.srvc_res.ip_suid_text_total_stat.Spam());
        m_hSp->CheckRange("mp_ip_suid_text_total_stat_ham", odata.srvc_res.ip_suid_text_total_stat.Ham());
        m_hSp->CheckRange("mp_ip_found_stat0", odata.srvc_res.ip_found_stat.Spam());
        m_hSp->CheckRange("mp_login_found_stat0", odata.srvc_res.suid_found_stat.Spam());
        m_hSp->CheckRange("mp_text_found_stat0", odata.srvc_res.text_found_stat.Spam());
        m_hSp->CheckRange("mp_geos_found_stat0", odata.srvc_res.geos_found_stat.Spam());
        m_hSp->CheckRange("mp_ip_suid_found_stat0", odata.srvc_res.ip_suid_found_stat.Spam());
        m_hSp->CheckRange("mp_ip_text_found_stat0", odata.srvc_res.ip_text_found_stat.Spam());
        m_hSp->CheckRange("mp_ip_found_stat1", odata.srvc_res.ip_found_stat.Ham());
        m_hSp->CheckRange("mp_login_found_stat1", odata.srvc_res.suid_found_stat.Ham());
        m_hSp->CheckRange("mp_text_found_stat1", odata.srvc_res.text_found_stat.Ham());
        m_hSp->CheckRange("mp_geos_found_stat1", odata.srvc_res.geos_found_stat.Ham());
        m_hSp->CheckRange("mp_ip_suid_found_stat1", odata.srvc_res.ip_suid_found_stat.Ham());
        m_hSp->CheckRange("mp_ip_text_found_stat1", odata.srvc_res.ip_text_found_stat.Ham());
        m_hSp->CheckRange("mp_offset", odata.msearch_offset);
    }

    m_hSp->AddStat("*lage", odata.login_age_str);
    m_hSp->AddStat("*sage", odata.so_reg_age_str);
    m_hSp->AddStat("spst", odata.spst_answer.ToText());

    //rules by spamstat
    if (odata.spst_answer.m_exists_data) {
        if (odata.spst_answer.m_today_autoban)
            SpSetRule(m_hSp, "SPST_TODAY_AUTOBAN");
        if (odata.spst_answer.m_today_manualban)
            SpSetRule(m_hSp, "SPST_TODAY_MANUALBAN");
        if (odata.spst_answer.m_today_white)
            SpSetRule(m_hSp, "SPST_TODAY_WHITE");
        if (odata.spst_answer.m_today_rbl)
            SpSetRule(m_hSp, "SPST_TODAY_RBL");
        m_hSp->CheckRange("spst_today_ham", odata.spst_answer.m_today_ham);
        m_hSp->CheckRange("spst_today_spam", odata.spst_answer.m_today_spam);
        m_hSp->CheckRange("spst_today_malic", odata.spst_answer.m_today_malic);
        m_hSp->CheckRange("spst_today_percentspam", odata.spst_answer.m_today_spam_percent);

        if (odata.spst_answer.m_yesterday_autoban)
            SpSetRule(m_hSp, "SPST_YESTERDAY_AUTOBAN");
        if (odata.spst_answer.m_yesterday_manualban)
            SpSetRule(m_hSp, "SPST_YESTERDAY_MANUALBAN");
        if (odata.spst_answer.m_yesterday_white)
            SpSetRule(m_hSp, "SPST_YESTERDAY_WHITE");
        if (odata.spst_answer.m_yesterday_rbl)
            SpSetRule(m_hSp, "SPST_YESTERDAY_RBL");
        m_hSp->CheckRange("spst_yesterday_ham", odata.spst_answer.m_yesterday_ham);
        m_hSp->CheckRange("spst_yesterday_spam", odata.spst_answer.m_yesterday_spam);
        m_hSp->CheckRange("spst_yesterday_malic", odata.spst_answer.m_yesterday_malic);
        m_hSp->CheckRange("spst_yesterday_percentspam", odata.spst_answer.m_yesterday_spam_percent);

        m_hSp->CheckRange("spst_history_ham", odata.spst_answer.m_history_ham);
        m_hSp->CheckRange("spst_history_spam", odata.spst_answer.m_history_spam);
        m_hSp->CheckRange("spst_history_rejectdays", odata.spst_answer.m_history_reject);
        m_hSp->CheckRange("spst_history_norejectdays", odata.spst_answer.m_history_noreject);
        m_hSp->CheckRange("spst_history_percentspam", odata.spst_answer.m_history_spam_percent);

        //if (!odata.spst_answer.m_geodata.empty())
        //   SpCheckField(m_hSp, "spst_geo", odata.spst_answer.m_geodata.c_str(), (int)odata.spst_answer.m_geodata.length(), "cp1251", true, false);

        //if (!odata.spst_answer.m_host.empty())
        //   SpCheckField(m_hSp, "spst_host", odata.spst_answer.m_host.c_str(), (int)odata.spst_answer.m_host.length(), "cp1251", true, false);

    } else {
        SpSetRule(m_hSp, "SPST_NODATA");
    }

    if (!odata.geos.empty()) {
        TString geotxt = odata.geos + " (SRVC[" + IntToStroka(odata.srvc_res.geostat.Spam()) + "/" + IntToStroka(odata.srvc_res.geostat.All()) + ", country: " + IntToStroka(odata.srvc_res.geocountrystat.Spam()) + "/" + IntToStroka(odata.srvc_res.geocountrystat.All()) + "] COMMON[" + IntToStroka(odata.common_res.geostat.Spam()) + "/" + IntToStroka(odata.common_res.geostat.All()) + ", country: " + IntToStroka(odata.common_res.geocountrystat.Spam()) + "/" + IntToStroka(odata.common_res.geocountrystat.All()) + "])";
        m_hSp->AddStat("geos", geotxt);
        SpCheckFieldTick(m_hSp, "geos", odata.geos.c_str(), (int)odata.geos.length(), "cp1251", true, false, DelayClass.GetRuleHash());
        m_hSp->AddStat("iptm", odata.geotime);
        SpCheckFieldTick(m_hSp, "iptm", odata.geotime.c_str(), (int)odata.geotime.length(), "cp1251", true, false, DelayClass.GetRuleHash());
        //service
        m_hSp->CheckRange("srvc_geo_ham", odata.srvc_res.geostat.Ham());
        m_hSp->CheckRange("srvc_geo_spam", odata.srvc_res.geostat.Spam());
        m_hSp->CheckRange("srvc_geo_all", odata.srvc_res.geostat.All());
        m_hSp->CheckRange("srvc_geocountry_ham", odata.srvc_res.geocountrystat.Ham());
        m_hSp->CheckRange("srvc_geocountry_spam", odata.srvc_res.geocountrystat.Spam());
        m_hSp->CheckRange("srvc_geocountry_all", odata.srvc_res.geocountrystat.All());
        //common
        m_hSp->CheckRange("cmmn_geo_ham", odata.common_res.geostat.Ham());
        m_hSp->CheckRange("cmmn_geo_spam", odata.common_res.geostat.Spam());
        m_hSp->CheckRange("cmmn_geo_all", odata.common_res.geostat.All());
        m_hSp->CheckRange("cmmn_geocountry_ham", odata.common_res.geocountrystat.Ham());
        m_hSp->CheckRange("cmmn_geocountry_spam", odata.common_res.geocountrystat.Spam());
        m_hSp->CheckRange("cmmn_geocountry_all", odata.common_res.geocountrystat.All());
    }

    if (!odata.hosts.empty()) {
        m_hSp->AddStat("hsts", odata.hosts);
        SpCheckFieldTick(m_hSp, "hosts", odata.hosts.c_str(), (int)odata.hosts.length(), "cp1251", true, false, DelayClass.GetRuleHash());
    }

    m_hSp->CheckRange("total_from_ip", odata.srvc_res.ipstat.All());
    m_hSp->CheckRange("spam_from_ip", odata.srvc_res.ipstat.Spam());
    if ((!odata.login.empty()) && (odata.login != "-")) {
        m_hSp->CheckRange("total_from_login", odata.srvc_res.loginstat.All());
        m_hSp->CheckRange("spam_from_login", odata.srvc_res.loginstat.Spam());
    }
    m_hSp->CheckRange("total_from_nick", odata.srvc_res.nickstat.All());
    m_hSp->CheckRange("spam_from_nick", odata.srvc_res.nickstat.Spam());
    if (odata.so_service == "XMPP") {
        m_hSp->CheckRange("total_from_xmppserv", odata.srvc_res.xmpp_server_stat.All());
        m_hSp->CheckRange("spam_from_xmppserv", odata.srvc_res.xmpp_server_stat.Spam());
    }
    m_hSp->CheckRange("total_from_comment", odata.srvc_res.commentstat.All());
    m_hSp->CheckRange("spam_from_comment", odata.srvc_res.commentstat.Spam());
    m_hSp->CheckRange("total_from_subject", odata.srvc_res.subjectstat.All());
    m_hSp->CheckRange("spam_from_subject", odata.srvc_res.subjectstat.Spam());
    m_hSp->CheckRange("total_from_psid", odata.srvc_res.idpostcardstat.All());
    m_hSp->CheckRange("spam_from_psid", odata.srvc_res.idpostcardstat.Spam());

    if (odata.login && odata.formId) {
        m_hSp->CheckRange("user_chat_total", odata.common_res.loginchatidstat.All());
        m_hSp->CheckRange("user_chat_spam_p", (100 * odata.common_res.loginchatidstat.Spam()) / (odata.common_res.loginchatidstat.All() + 1));
    }

    if (odata.login)
        m_hSp->CheckRange("user_uniq_chats", odata.common_res.uniquechatsbyloginstat.All());

    m_hSp->CheckRange("common_total_from_ip", odata.common_res.ipstat.All());
    m_hSp->CheckRange("common_spam_from_ip", odata.common_res.ipstat.Spam());
    if ((!odata.login.empty()) && (odata.login != "-")) {
        m_hSp->CheckRange("common_total_from_login", odata.common_res.loginstat.All());
        m_hSp->CheckRange("common_spam_from_login", odata.common_res.loginstat.Spam());
    }
    m_hSp->CheckRange("common_total_from_nick", odata.common_res.nickstat.All());
    m_hSp->CheckRange("common_spam_from_nick", odata.common_res.nickstat.Spam());
    m_hSp->CheckRange("common_total_from_comment", odata.common_res.commentstat.All());
    m_hSp->CheckRange("common_spam_from_comment", odata.common_res.commentstat.Spam());
    m_hSp->CheckRange("common_total_from_subject", odata.common_res.subjectstat.All());
    m_hSp->CheckRange("common_spam_from_subject", odata.common_res.subjectstat.Spam());

    m_hSp->CheckRange("login_age_days", odata.login_age_days);
    m_hSp->CheckRange("serv_age_days", odata.so_reg_age_days);

    if(odata.soCustomUserRegdate) {
        m_hSp->CheckRange("days_since_regdate", (odata.soCustomUserRegdate - Now()).Days());
    }

    m_hSp->CheckRange("so_rcpt_count", static_cast<ui32>(odata.rcpt_count));
    m_hSp->AddStat("rcpt", IntToStroka(odata.rcpt_count));
    TZoneDetector::TStrokaHash::iterator rcpt_it = odata.mlist_rcpt.begin();
    while (rcpt_it != odata.mlist_rcpt.end()) {
        if (!(*rcpt_it).first.empty())
            SpCheckFieldTick(m_hSp, "so_rcpt_email", (*rcpt_it).first.c_str(), (int)(*rcpt_it).first.length(), "cp1251", true, false, DelayClass.GetRuleHash());

        ++rcpt_it;
    }

    SpCheckFieldTick(m_hSp, "_FORMNAME_", odata.form_name.c_str(), (int)odata.form_name.length(), "cp1251", true, false, DelayClass.GetRuleHash());
    SpCheckFieldTick(m_hSp, "so_login", odata.login.c_str(), (int)odata.login.length(), "cp1251", true, false, DelayClass.GetRuleHash());
    SpCheckFieldTick(m_hSp, "so_service", odata.so_service.c_str(), (int)odata.so_service.length(), "cp1251", true, false, DelayClass.GetRuleHash());
    SpCheckFieldTick(m_hSp, "so_ip", odata.ip.toStroka().c_str(), (int)odata.ip.toStroka().length(), "cp1251", true, false, DelayClass.GetRuleHash());
    SpCheckFieldTick(m_hSp, "so_form_name", odata.form_name.c_str(), (int)odata.form_name.length(), "cp1251", true, false, DelayClass.GetRuleHash());
    if (odata.so_service == "XMPP")
        SpCheckFieldTick(m_hSp, "xmpp_server_name", odata.xmpp_server_name.c_str(), (int)odata.xmpp_server_name.length(), "cp1251", true, false, DelayClass.GetRuleHash());

    if (odata.est_contlen)
        m_hSp->CheckRange("contentlen", odata.so_contlen);

    std::list<TString>::iterator lvit = odata.login_virt1.begin();
    while (lvit != odata.login_virt1.end()) {
        if (!(*lvit).empty())
            m_hSp->CheckMailFrom(*lvit); //roll mailfrom (��������� ������)

        ++lvit;
    }

    if (odata.soname_cmp_sosubject)
        SpSetRule(m_hSp, "CMP_NAME_SUBJECT");

    if (true) //***** rules by urls
    {
        TString urlbody = "";

        TZoneDetector::TKUrlHash::iterator hit = odata.urlhash_t.begin();
        while (hit != odata.urlhash_t.end()) {
            urlbody = (*hit).second.m_url;
            SpCheckFieldTick(m_hSp, "_url_", urlbody.c_str(), (int)urlbody.length(), "cp1251", true, false, DelayClass.GetRuleHash());
            m_hSp->AddStat("*url", urlbody);

            ++hit;
        }
    }

    if (true) //***** rules by hosts
    {
        bool banhost = false;
        TString host = "";
        int thishostcount = 0;
        ui64 host_shingle = 0;
        TCountersStatEx service_stor_data;
        TCountersStatEx common_stor_data;
        bool exists = false;

        TZoneDetector::TStrokaHash::iterator it = odata.hlist.begin();
        while (it != odata.hlist.end()) {
            host = (*it).first;
            thishostcount = (*it).second;
            //������������ ������ ������, �� ��������� � excludehosts (fieldcheck.txt)
            if (!FieldCheck->KeyExistsCompare("excludehosts", host)) {
                exists = false;
                service_stor_data = {};
                common_stor_data = {};

                host_shingle = stordata::TTxtData(host).GetShingle();

                DelayClass.AddHostChecked();

                if (!banhost) {
                    if (FieldCheck->KeyExistsCompare("banhosts", host)) {
                        banhost = true;
                        SpSetRule(m_hSp, "RBANHOST");
                    }
                }

                //find in store array
                if (host_shingle != 0) {
                    for (size_t k = 0; k < odata.stordata_hmp.host_list_size; k++) {
                        if (host_shingle == odata.stordata_hmp.host_list[k].GetShingle()) {
                            service_stor_data = odata.stordata_hmp.host_list[k].m_service_data;
                            common_stor_data = odata.stordata_hmp.host_list[k].m_common_data;
                            exists = true;
                            break;
                        }
                    }
                }

                m_hSp->CheckRange("count_host_in_mess", thishostcount);
                SpCheckFieldTick(m_hSp, "_host_", host.c_str(), (int)host.length(), "cp1251", true, false, DelayClass.GetRuleHash());
                if (exists) {
                    DelayClass.AddMongoHostChecked();
                    m_hSp->CheckRange("total_per_host", service_stor_data.All());
                    m_hSp->CheckRange("spam_per_host", service_stor_data.Spam());
                    m_hSp->CheckRange("common_total_per_host", common_stor_data.All());
                    m_hSp->CheckRange("common_spam_per_host", common_stor_data.Spam());
                    m_hSp->AddStat("*host", ("(l:" + UI32ToStroka(service_stor_data.Spam()) + "/" + UI32ToStroka(service_stor_data.All()) + ", c:" + UI32ToStroka(common_stor_data.Spam()) + "/" + UI32ToStroka(common_stor_data.All()) + ") " + host + " (" + IntToStroka(thishostcount) + ")"));

                } else {
                    m_hSp->AddStat("*host", ("(l:-/-, c:-/-) " + host + " (" + IntToStroka(thishostcount) + ")"));
                }
            }

            ++it;
        }
        m_hSp->CheckRange("host_count_in_field", odata.max_host_count_in_field);
    }

    if (true) //*****rules by mails
    {
        bool banemail = false;
        TZoneDetector::TStrokaHashIt it;
        int thisemailcount = 0;
        TString email = "";
        ui64 email_shingle = 0;
        TCountersStatEx service_stor_data;
        TCountersStatEx common_stor_data;
        bool exists = false;

        it = odata.mlist.begin();
        while (it != odata.mlist.end()) {
            email = (*it).first;
            thisemailcount = (*it).second;
            //������������ ������ ������, �� ��������� � excludeemail (fieldcheck.txt)
            if (!FieldCheck->KeyExistsCompare("excludeemail", email)) {
                exists = false;
                service_stor_data = {};
                common_stor_data = {};

                email_shingle = stordata::TTxtData(email).GetShingle();

                DelayClass.AddEmailChecked();

                //find in store array
                if (email_shingle != 0) {
                    for (size_t k = 0; k < odata.stordata_hmp.mail_list_size; k++) {
                        if (email_shingle == odata.stordata_hmp.mail_list[k].GetShingle()) {
                            service_stor_data = odata.stordata_hmp.mail_list[k].m_service_data;
                            common_stor_data = odata.stordata_hmp.mail_list[k].m_common_data;
                            exists = true;
                            break;
                        }
                    }
                }

                if (!banemail) {
                    if (FieldCheck->KeyExistsCompare("banemail", email)) {
                        banemail = true;
                        SpSetRule(m_hSp, "RBANEMAIL");
                    }
                }

                m_hSp->CheckRange("count_email_in_mess", thisemailcount);
                SpCheckFieldTick(m_hSp, "_email_", email.c_str(), (int)email.length(), "cp1251", true, false, DelayClass.GetRuleHash());
                if (exists) {
                    DelayClass.AddMongoEmailChecked();

                    m_hSp->CheckRange("total_per_email", service_stor_data.All());
                    m_hSp->CheckRange("spam_per_email", service_stor_data.Spam());
                    m_hSp->CheckRange("common_total_per_email", common_stor_data.All());
                    m_hSp->CheckRange("common_spam_per_email", common_stor_data.Spam());

                    m_hSp->AddStat("*email", ("(l:" + UI32ToStroka(service_stor_data.Spam()) + "/" + UI32ToStroka(service_stor_data.All()) + ", c:" + UI32ToStroka(common_stor_data.Spam()) + "/" + UI32ToStroka(common_stor_data.All()) + ") " + email + " (" + IntToStroka(thisemailcount) + ")"));

                } else {
                    m_hSp->AddStat("*email", ("(l:-/-, c:-/-) " + email + " (" + IntToStroka(thisemailcount) + ")"));
                }
            }

            ++it;
        }
        m_hSp->CheckRange("email_count_in_field", odata.max_email_count_in_field);
    }

    //***** rules by phones
    if (true) {
        bool banphone = false;
        TString phone = "";
        ui64 phone_shingle = 0;
        TCountersStatEx service_stor_data;
        TCountersStatEx common_stor_data;
        bool exists = false;

        common::TPhoneParser::TPhoneList::iterator pit = odata.plist.begin();
        while (pit != odata.plist.end()) {
            phone = (*pit);
            //������������ ������ ��������, �� ��������� � excludephone (fieldcheck.txt)
            if (!FieldCheck->KeyExistsCompare("excludephone", phone)) {
                exists = false;
                service_stor_data = {};
                common_stor_data = {};

                phone_shingle = stordata::TTxtData(phone).GetShingle();

                DelayClass.AddPhoneChecked();

                //find in store array
                if (phone_shingle != 0) {
                    for (size_t k = 0; k < odata.stordata_hmp.phone_list_size; k++) {
                        if (phone_shingle == odata.stordata_hmp.phone_list[k].GetShingle()) {
                            service_stor_data = odata.stordata_hmp.phone_list[k].m_service_data;
                            common_stor_data = odata.stordata_hmp.phone_list[k].m_common_data;
                            exists = true;
                            break;
                        }
                    }
                }

                if (!banphone) {
                    if (FieldCheck->KeyExistsCompare("banphone", phone)) {
                        banphone = true;
                        SpSetRule(m_hSp, "RBANPHONE");
                    }
                }

                SpCheckFieldTick(m_hSp, "_phone_", phone.c_str(), (int)phone.length(), "cp1251", true, false, DelayClass.GetRuleHash());
                if (exists) {
                    DelayClass.AddMongoPhoneChecked();

                    m_hSp->CheckRange("total_per_phone", service_stor_data.All());
                    m_hSp->CheckRange("spam_per_phone", service_stor_data.Spam());
                    m_hSp->CheckRange("common_total_per_phone", common_stor_data.All());
                    m_hSp->CheckRange("common_spam_per_phone", common_stor_data.Spam());
                    m_hSp->AddStat("*phone", ("(l:" + UI32ToStroka(service_stor_data.Spam()) + "/" + UI32ToStroka(service_stor_data.All()) + ", c:" + UI32ToStroka(common_stor_data.Spam()) + "/" + UI32ToStroka(common_stor_data.All()) + ") " + phone));

                } else {
                    m_hSp->AddStat("*phone", ("(l:-/-, c:-/-) " + phone));
                }
            }
            ++pit;
        }
        m_hSp->CheckRange("phone_count_in_field", odata.max_phone_count_in_field);
    }

    //rules by fields
    for (TReqParams::iterator i = pReqParams->begin(); i != pReqParams->end(); i++) {
        skey_field = i->first;
        svalue_field = GetValue(i);

        if (ExcludeKey(odata.so_service, skey_field))
            continue;

        nowritetodlvr = false;

        if (("so_comment" == skey_field) || ("so_subject" == skey_field))
            nowritetodlvr = true;

        if (!nowritetodlvr)
            m_hSp->AddStat(("*" + skey_field).c_str(), svalue_field); // Log field

        allfields += svalue_field + " ";

        formname_fieldname = odata.form_name + "_" + skey_field;

        // Check every field
        SpCheckFieldTick(m_hSp, skey_field.c_str(), svalue_field.c_str(), (int)svalue_field.length(), "cp1251", true, false, DelayClass.GetRuleHash());

        // Check field by form name
        SpCheckFieldTick(m_hSp, odata.form_name.c_str(), svalue_field.c_str(), (int)svalue_field.length(), "cp1251", true, false, DelayClass.GetRuleHash());

        // Check field by form name
        SpCheckFieldTick(m_hSp, formname_fieldname.c_str(), svalue_field.c_str(), (int)svalue_field.length(), "cp1251", true, false, DelayClass.GetRuleHash());

        // Check field for all forms
        SpCheckFieldTick(m_hSp, "_all_", svalue_field.c_str(), (int)svalue_field.length(), "cp1251", true, false, DelayClass.GetRuleHash());

        // Check url
        if ((FieldCheck != NULL) && (FieldCheck->KeyExists("url", skey_field))) {
            if (m_hSp->GetPattern("url", svalue_field, url))
                SpCheckFieldTick(m_hSp, "embedded_url", url.c_str(), (int)url.length(), "cp1251", true, false, DelayClass.GetRuleHash());
        }

        // Check host stat
        for (size_t j = 0; j < exclude_field_list.size(); j++) {
            if ((skey_field.length() >= exclude_field_list[j].length()) && (!memcmp(skey_field.c_str(), exclude_field_list[j].c_str(), exclude_field_list[j].length()))) {
                break;
            }
        }

        //��� ������ �� ����� ���� ����������� �������� ������ (������ ��� �������� �����)
        if (odata.so_service == "MSEARCH-PROXY") {
            bool is_include_signature = false;
            if ((FieldCheck != NULL) && (FieldCheck->KeyExists("msearchlines", skey_field))) {
                if (m_signhash->FindTextS(svalue_field))
                    is_include_signature = true;
            }
            if (is_include_signature)
                SpSetRule(m_hSp, "MS_FIND_SIGNATURE");
        }
    }

    if (odata.so_service != "MSEARCH-PROXY") {
        CProf prof;
        TString pureText;
        if (m_hSp->IsHTMLText(allfields))
            m_hSp->CheckBody(prof, allfields.c_str(), allfields.length(), spTextHtml, pureText, &pPureUtf8Text, &property);
        else
            m_hSp->CheckBody(prof, allfields.c_str(), allfields.length(), spTextPlain, pureText, &pPureUtf8Text, &property);
    }

    //���������� �� �������� ������� (by so_service)
    m_hSp->CheckRange("stat_all5min", odata.srvc_data.m_request.m_5min.m_all);
    m_hSp->CheckRange("stat_spam5min", odata.srvc_data.m_request.m_5min.m_spam);
    m_hSp->CheckRange("stat_error5min", odata.srvc_data.m_request.m_5min.m_error);
    m_hSp->CheckRange("stat_all_day", odata.srvc_data.m_request.m_day.m_all);
    m_hSp->CheckRange("stat_spam_day", odata.srvc_data.m_request.m_day.m_spam);
    m_hSp->CheckRange("stat_error_day", odata.srvc_data.m_request.m_day.m_error);
    m_hSp->CheckRange("stat_all_yesterday", odata.srvc_data.m_request.m_yesterday.m_all);
    m_hSp->CheckRange("stat_spam_yesterday", odata.srvc_data.m_request.m_yesterday.m_spam);
    m_hSp->CheckRange("stat_error_yesterday", odata.srvc_data.m_request.m_yesterday.m_error);

    //���������� �� ����������� ������� (by so_service; ��� ������� POSTCARD �������� ��������� ����������� � ����� ��������)
    m_hSp->CheckRange("stat_all5min_rcpt", odata.srvc_data.m_rcpt.m_5min.m_all);
    m_hSp->CheckRange("stat_spam5min_rcpt", odata.srvc_data.m_rcpt.m_5min.m_spam);
    m_hSp->CheckRange("stat_error5min_rcpt", odata.srvc_data.m_rcpt.m_5min.m_error);
    m_hSp->CheckRange("stat_all_day_rcpt", odata.srvc_data.m_rcpt.m_day.m_all);
    m_hSp->CheckRange("stat_spam_day_rcpt", odata.srvc_data.m_rcpt.m_day.m_spam);
    m_hSp->CheckRange("stat_error_day_rcpt", odata.srvc_data.m_rcpt.m_day.m_error);
    m_hSp->CheckRange("stat_all_yesterday_rcpt", odata.srvc_data.m_rcpt.m_yesterday.m_all);
    m_hSp->CheckRange("stat_spam_yesterday_rcpt", odata.srvc_data.m_rcpt.m_yesterday.m_spam);
    m_hSp->CheckRange("stat_error_yesterday_rcpt", odata.srvc_data.m_rcpt.m_yesterday.m_error);

    //���������� ����� (������� roll)

    m_hSp->CheckSuidRoll(odata.so_nick);

    //DelayClass.SetDelayFilterWork(ttt);

    if (odata.longipdataFuture.Wait(TDuration::MilliSeconds(100))) {
        const auto& longipdata = odata.longipdataFuture.GetValueSync();
        if (longipdata.exist_data) {
            if (odata.so_service == "MSEARCH-PROXY")
                m_hSp->AddStat("ipbs", ("days=" + IntToStroka(longipdata.days_elapsed) + ", ham=" + IntToStroka(longipdata.ham) + ", spam=" + IntToStroka(longipdata.spam) + ", spamperc=" + FloatToStr(longipdata.spam_percent) + ", found=" + IntToStroka(longipdata.found) + ", foundperc=" + FloatToStr(longipdata.found_percent)));
            else
                m_hSp->AddStat("ipbs", ("days=" + IntToStroka(longipdata.days_elapsed) + ", ham=" + IntToStroka(longipdata.ham) + ", spam=" + IntToStroka(longipdata.spam) + ", spamperc=" + FloatToStr(longipdata.spam_percent)));

            m_hSp->CheckRange("ipb_day", longipdata.days_elapsed);
            m_hSp->CheckRange("ipb_ham", longipdata.ham);
            m_hSp->CheckRange("ipb_spam", longipdata.spam);
            m_hSp->CheckRange("ipb_spamperc", longipdata.spam_percent);
            if (odata.so_service == "MSEARCH-PROXY") {
                m_hSp->CheckRange("ipb_found", longipdata.found);
                m_hSp->CheckRange("ipb_foundperc", longipdata.found_percent);
            }
        }
    }
    if (odata.longphonedataFuture.Wait(TDuration::MilliSeconds(100))) {
        const auto& longphonedata = odata.longphonedataFuture.GetValueSync();
        if (longphonedata.exist_data) {
            m_hSp->AddStat("phon", ("days=" + IntToStroka(longphonedata.days_elapsed) + ", ham=" + IntToStroka(longphonedata.ham) + ", spam=" + IntToStroka(longphonedata.spam) + ", spamperc=" + FloatToStr(longphonedata.spam_percent)));

            m_hSp->CheckRange("phone_day", longphonedata.days_elapsed);
            m_hSp->CheckRange("phone_ham", longphonedata.ham);
            m_hSp->CheckRange("phone_spam", longphonedata.spam);
            m_hSp->CheckRange("phone_spamperc", longphonedata.spam_percent);
        }
    }
    if (odata.curlFuture.Initialized() && odata.curlFuture.Wait(TDuration::MilliSeconds(100))) {
        const auto [bbInfo, cleanWebInfo, urlRepInfo] = odata.curlFuture.ExtractValueSync();
        if (bbInfo) {
            TStringStream log;
            log << bbInfo->Message() << ';';
            const NBlackbox2::TKarmaInfo karmaInfo(bbInfo.Get());
            m_hSp->CheckRange("karma", FromString<ui32>(karmaInfo.Karma()));
            m_hSp->CheckRange("karma_status", FromString<ui32>(karmaInfo.KarmaStatus()));
            log << karmaInfo.Karma() << ';' << karmaInfo.KarmaStatus() << ';';

            if (auto regDate = NFuncClient::CBB::GetRegDateFromBBResponse(*bbInfo)) {
                m_hSp->CheckRange("born_date", ui32((Now() - *regDate).Days()));
                log << *regDate << ';';
            }
            m_hSp->AddStat("bb  ", std::move(log.Str()));
        } else {
            m_hSp->AddStat("timeout", "bb");
            Cerr << "empty bbinfo for " << odata.login << Endl;
        }
        if(urlRepInfo) {
            for (const auto& item : *urlRepInfo) {
                const auto& info = item.isLink ? item.longUrl : item.checkUrl;
                m_hSp->AddStat("rpur", (TStringBuilder{}
                                                        << info.host << ' '
                                                        << info.FirstTimeElapsed() << ' '
                                                        << info.today.ham << ' '
                                                        << info.today.spam << ' '
                                                        << info.history.ham << ' '
                                                        << info.history.spam << ' '
                                                        << info.today.complaint_ham << ' '
                                                        << info.today.complaint_spam << ' '
                                                        << info.history.complaint_ham << ' '
                                                        << info.history.complaint_spam << ' '
                                                        ));
                int total_hist = (int)(info.history.ham + info.history.spam);
                {
                    m_hSp->CheckRange("repurl_age", (int)(info.FirstTimeElapsed()));

                    if (total_hist >= 1000) {
                        int perc = (info.history.spam * 1000) / total_hist;

                        if (total_hist < 100000)
                            m_hSp->CheckRange("repurl_hist_1000", perc, false);
                        else
                            m_hSp->CheckRange("repurl_hist_100000", perc, false);
                    }
                }

                if (info.today.spam > 10)
                    m_hSp->CheckRange("repurl_compl_ham_perc", (int)((100 * info.today.complaint_ham) / info.today.spam), false);

                if (info.today.ham > 50)
                    m_hSp->CheckRange("repurl_compl_spam_perc", (int)((100 * info.today.complaint_spam) / info.today.ham), false);

                if (info.history.spam > 100)
                    m_hSp->CheckRange("repurl_hist_compl_ham_perc", (int)((100 * info.history.complaint_ham) / info.history.spam), false);

                if (info.history.ham > 100)
                    m_hSp->CheckRange("repurl_hist_compl_spam_perc", (int)((100 * info.history.complaint_spam) / info.history.ham), false);
            }
        } else {
            m_hSp->AddStat("timeout", "url");
        }
        if (cleanWebInfo) {
            m_hSp->AddStat("cweb", *cleanWebInfo);
        }

    } else {
        m_hSp->AddStat("timeout", "bb, url, cweb");
    }

    // Check message

    NProf::Profiler prof;
    try {
        TCheckedMessage checkedMessage;
        m_hSp->CheckMessage(prof, nullptr, checkedMessage);
        res.Spam = IsSpam(checkedMessage.spClass);
    } catch (...) {
        Syslog(TLOG_ERR) << "CheckMessage:" << CurrentExceptionMessageWithBt();
    }
    res.dirtyword = m_hSp->ExistsbanInternal("DIRTYWORD");
    res.banrule = m_hSp->ExistsbanInternal("BLOCKRULE");

    ttt = CShingleTime::GetMs() - ttt;

    if (logoutClient && m_hSp->m_cur->rulesContext.IsRuleWorked("TO_FMS_HACK")) {
        NThreading::Async([this, login = odata.so_login] {
            const auto start = Now();
            try {
                const auto answer = logoutClient->Perform(login);
                Cerr << "lout" << (answer ? TStringBuf(" ok") : TStringBuf(" fail"))
                     << " for " << (Now() - start) << ' '
                     << answer << Endl;
            } catch (...) {
                Cerr << "lout" << TStringBuilder{} << "err for " << (Now() - start) << ' ' << CurrentExceptionMessageWithBt() << Endl;
            }
        }, threadPool);
    }

    {
        TCgiParameters cgiParams;
        for(const auto& [k, values]: *pReqParams) {
            for(const auto& v: values) {
                cgiParams.InsertUnescaped(k, v);
            }
        }

        try {
            auto localReceipt = TReceipt::MakeReceipt(cgiParams);
            Syslog(TLOG_INFO) << "receipt:" << localReceipt;
            m_hSp->AddStat("receipt", localReceipt.ToJson());
            if (m_hSp->m_cur->rulesContext.IsRuleWorked("SERVICE_WANTS_RECEIPT")) {
                receipt = std::move(localReceipt);
            }
        } catch (...) {
            const auto message = CurrentExceptionMessageWithBt();
            Syslog(TLOG_ERR) << "receipt:" << message;
            m_hSp->AddStat("receipt", message);
        }
    }

    filter_work_tick = CShingleTime::GetMs() - filter_work_tick;

    return res;
}

int CSoFilterCF::Chec2(TReqParams* pReqParams, const NJson::TJsonValue& js, bool& Spam, const TString& id, chkfrm::TDelayClass& DelayClass, TString& service, TString& formname_s, bool& skeep, bool& dirtyword, bool& banrule, TCheckformResponceType& format, TAntiDDOSCriterionInfo& antiddos, TMaybe<TReceipt>& receipt) {
    DelayClass.check_delay = CShingleTime::GetMs();

    TString NumbRequest = id;
    TString sid = "";
    TReqParams* pReqParamsNew = NULL;
    TReqParams::iterator par_it;
    //TString               host = "";
    int flag_ok = 0;
    TString control_str = "";
    TString format_s = "";
    ui32 ttt = 0;
    stordata::TStorData stordata;
    TObrabParams odata;

    skeep = false;
    Spam = false;
    dirtyword = false;
    banrule = false;
    antiddos.Clear();

    sid = id;
    pReqParamsNew = DecodeParams(id, pReqParams);

    const auto* payloadIdVec = MapFindPtr(*pReqParamsNew, "payload_id");
    const size_t payloadIdHash = (payloadIdVec && !payloadIdVec->empty()) ? ComputeHash(payloadIdVec->front()) : 0;
    if (payloadIdHash) {
        TResolution resolution;
        if (resCache.Get(payloadIdHash, resolution)) {
            service = resolution.service;
            Spam = resolution.Spam;
            formname_s = resolution.formname;
            skeep = resolution.Skeep;
            dirtyword = resolution.dirtyword;
            banrule = resolution.banrule;
            format = resolution.format;
            antiddos = resolution.antiddos;

            if (pReqParamsNew != NULL) {
                pReqParamsNew->clear();
                delete pReqParamsNew;
                pReqParamsNew = NULL;
            }

            return 0;
        }
    }

    //responce type
    format = RUNKNOWN;
    par_it = pReqParamsNew->find("format");
    if (par_it != pReqParamsNew->end()) {
        format_s = TString{Trim(GetValue(par_it))};
        format_s.to_lower(0, format_s.size());

        if (format_s == "json")
            format = RJSON;
        else if (format_s == "json2")
            format = RJSON2;
        else if (format_s == "json3")
            format = RJSON3;
        else if (format_s == "xml1")
            format = RXML1;
        else if (format_s == "xml2")
            format = RXML2;
        else if (format_s == "text")
            format = RTEXT;
        else
            format = RUNKNOWN;
    }

    //so_service
    par_it = pReqParamsNew->find("so_service");
    if (par_it != pReqParamsNew->end()) {
        odata.so_service = Trim(GetValue(par_it));
        to_upper_k(odata.so_service);
        service = odata.so_service;

    } else {
        WriteToLog(KERROR, id, "Bad request - no 'so_service' parameter");
        if (pReqParamsNew != NULL) {
            pReqParamsNew->clear();
            delete pReqParamsNew;
            pReqParamsNew = NULL;
        }
        return 1;
    }
    odata.so_service.to_upper(0, odata.so_service.length());
    DelayClass.SetService(odata.so_service);

    if (GroupFilter != NULL) {
        TFilterItem* fitem = NULL;

        fitem = GroupFilter->GetFilterItem(odata.so_service, id);
        if (fitem != NULL) {
            odata.so_service_from_fitem = fitem->GetIdent();

            fitem->AddWorkEvent();
            if (fitem->Skeep()) //����� ��������, �������
            {
                fitem->AddSkeepEvent();
                Spam = false;
                skeep = true;
                TRenginePoolCF* renginepool = fitem->GetFilterPointer();
                if (renginepool != NULL)
                    renginepool->AddSkeepFilter();

            } else {
                //fitem->Lock();

                bool is_max_count_from_login = false;
                TString controllog_dop = "";
                TString field_key = "";
                TString field_value = "";

                //anti DDOS
                if (pReqParamsNew != NULL) {
                    par_it = pReqParamsNew->begin();
                    while (par_it != pReqParamsNew->end()) {
                        field_key = (*par_it).first;
                        field_value = Trim(GetValue(par_it));

                        //top20
                        fitem->AddStatToTop20List(field_key, field_value);

                        //anti ddos
                        if (RunAntiDDOSActionOther(odata.so_service, field_key, field_value, antiddos)) {
                            antiddos.m_action = true;
                            antiddos.m_field_key = field_key;
                            Spam = true;
                            controllog_dop = antiddos.GetReportControlLog();
                            DelayClass.antiddos_action = true;
                            break;
                        }

                        ++par_it;
                    }
                }

                if (!antiddos.m_action) {
                    TString ips = "";
                    TString skey_field = "";
                    TString svalue_field = "";
                    TStoreBase *store = NULL, *commonstore = NULL;
                    TString ts = "";
                    const char* p = NULL;
                    TZoneDetector HostParser;
                    TRenginePoolCF* renginepool = NULL;
                    TSPSTData spst_data;
                    bool newgeos = false;
                    bool newhosts = false;
                    TIPInfoItemIPv6 ipinf;
                    TString step = "";
                    TString normtext = "";
                    TString srctext = "";
                    //bool          longip_storage_exists = false;
                    TKIPv6 blacknet = TKIPv6();
                    TKIPv6 whitenet = TKIPv6();

                    //�������� �����������
                    odata.max_mess_count_from_login = fitem->GetMaxMessCountFromLogin();

                    //�������� ��������� �� ��������
                    store = fitem->GetStore();
                    if (GroupFilter != NULL)
                        commonstore = GroupFilter->GetCommonStore();

                    //��������� ������ ����������

                    //��� ������� MSEARCH ���� ����� ����
                    if (odata.so_service == "MSEARCH-PROXY") {
                        par_it = pReqParamsNew->find("step");
                        if (par_it != pReqParamsNew->end()) {
                            step = Trim(GetValue(par_it));
                            if (step == "2")
                                odata.msearch_step2 = true;
                        }
                        if (odata.msearch_step2) {
                            par_it = pReqParamsNew->find("found");
                            if (par_it != pReqParamsNew->end())
                                odata.msearch_found = FromString(Trim(GetValue(par_it)));
                        }

                        par_it = pReqParamsNew->find("so_offset");
                        if (par_it != pReqParamsNew->end()) {
                            odata.msearch_offset = FromString(Trim(GetValue(par_it)));
                            if (odata.msearch_offset > 0)
                                odata.no_write_storage = true;
                        }
                    }

                    const bool UseSoLoginAsUid = [pReqParamsNew] {
                        auto i = MapFindPtr(*pReqParamsNew, "so_login_as_uid");
                        return i && !i->empty() && i->front() == "1";
                    }();

                    //so_ip
                    par_it = pReqParamsNew->find("so_ip");
                    if (par_it == pReqParamsNew->end()) {
                        WriteToLog(KERROR, id, "Bad request - no 'so_ip' parameter");
                        //fitem->AddStatError(1);
                        AddStatError(NumbRequest, store, odata, 1);
                        flag_ok = 2;
                        goto MEND;
                    }
                    ips = GetValue(par_it);
                    //if (RunAntiDDOSActionOther(so_service, "so_ip", ips, 0, criterion_number))
                    //{
                    //   fast_reject_text = "IP(" + IntToStroka(criterion_number) + ")";
                    //   goto MEND;
                    //}
                    odata.ip = TKIPv6(ips);
                    if (RunAntiDDOSActionIP(store, odata.so_service, odata.ip, antiddos)) {
                        antiddos.m_action = true;
                        antiddos.m_field_key = "~ip";
                        Spam = true;
                        controllog_dop = antiddos.GetReportControlLog();
                        DelayClass.antiddos_ip = true;
                        goto MEND;
                    }

                    if (BlackList != NULL)
                        odata.is_black_ip = BlackList->IsIncludeNet(odata.ip);

                    if (WhiteList != NULL)
                        odata.is_white_ip = WhiteList->IsIncludeNet(odata.ip);

                    //so_form_name
                    par_it = pReqParamsNew->find("so_form_name");
                    if (par_it == pReqParamsNew->end()) {
                        WriteToLog(KERROR, id, "Bad request - no 'so_form_name' parameter");
                        //fitem->AddStatError(1);
                        AddStatError(NumbRequest, store, odata, 1);
                        flag_ok = 3;
                        goto MEND;
                    }
                    odata.form_name = GetValue(par_it);
                    formname_s = odata.form_name;

                    //so_rctp
                    par_it = pReqParamsNew->find("so_rcpt");
                    if (par_it != pReqParamsNew->end())
                        odata.rcpt = GetValue(par_it);

                    if(par_it = pReqParamsNew->find("so_custom_user_regdate"); par_it != pReqParamsNew->end()) {
                        TInstant::TryParseIso8601(GetValue(par_it), odata.soCustomUserRegdate);
                    }

                    if ((odata.so_service == "POSTCARD") && (odata.rcpt.empty())) {
                        //so_q2
                        par_it = pReqParamsNew->find("so_q2");
                        if (par_it != pReqParamsNew->end())
                            odata.rcpt = GetValue(par_it);
                    }

                    odata.rcpt_count = 1;
                    if (HostParser.GetEmailsUrlsUniq(odata.rcpt.c_str(), odata.rcpt.length(), false, odata.hlist_rcpt, odata.mlist_rcpt) > 0) {
                        if (odata.mlist_rcpt.size() > 0)
                            odata.rcpt_count = odata.mlist_rcpt.size();
                    }

                    if (odata.so_service == "POSTCARD") {
                        //�������� �� ���� so_id ���������������� �������� (������� ���)
                        par_it = pReqParamsNew->find("so_id");
                        if (par_it != pReqParamsNew->end()) {
                            odata.id_postcard = GetValue(par_it);
                            if (odata.id_postcard.length() > 10) //������ ��� ���������������� �������� ������� POSTCARD
                                odata.so_idpostcard_shingle = GetShingle("idpstcard_" + odata.id_postcard);
                        }
                    }

                    //login
                    par_it = pReqParamsNew->find("so_name");
                    if (par_it != pReqParamsNew->end()) {
                        odata.login = GetValue(par_it);
                        odata.login.to_lower(0, odata.login.size());

                        if (!odata.login.empty()) {
                            p = strchr(odata.login.c_str(), '@');
                            if (p != NULL) {
                                odata.login_virt1.push_back(odata.login);
                                if (odata.so_service == "XMPP") {
                                    const char* pend = NULL;

                                    pend = strchr(p, '/');
                                    if (pend == NULL)
                                        pend = odata.login.c_str() + odata.login.length();
                                    if (pend != NULL) {
                                        int count = pend - p - 1;

                                        if (count > 0) {
                                            odata.xmpp_server_name = Trim({p + 1, size_t(count)});
                                            if (odata.xmpp_server_name.length() > 0) {
                                                odata.xmpp_server_name_shingle = GetShingle("xmpp_srv_" + odata.xmpp_server_name);
                                                if (RunAntiDDOSActionXMPPServerName(store, odata.so_service, odata.xmpp_server_name_shingle, antiddos)) {
                                                    antiddos.m_action = true;
                                                    antiddos.m_field_key = "~xmpp_srvname";
                                                    Spam = true;
                                                    controllog_dop = antiddos.GetReportControlLog();
                                                    DelayClass.antiddos_xmpp_srv = true;
                                                    goto MEND;
                                                }
                                            }
                                        }
                                    }
                                }
                            } else {
                                odata.login_virt1.push_back(odata.login + "@yandex.ru");
                                odata.login_virt1.push_back(odata.login + "@ya.ru");
                                odata.login_virt1.push_back(odata.login + "@narod.ru");
                                odata.login_virt1.push_back(odata.login + "@yandex.kz");
                                odata.login_virt1.push_back(odata.login + "@yandex.com");
                            }
                        }
                    } else {
                        par_it = pReqParamsNew->find("so_login");
                        if (par_it != pReqParamsNew->end()) {
                            odata.login = GetValue(par_it);
                            odata.login.to_lower(0, odata.login.size());
                            if (!odata.login.empty()) {
                                p = strchr(odata.login.c_str(), '@');
                                if (p != NULL) {
                                    odata.login_virt1.push_back(odata.login);

                                } else {
                                    odata.login_virt1.push_back(odata.login + "@yandex.ru");
                                    odata.login_virt1.push_back(odata.login + "@ya.ru");
                                    odata.login_virt1.push_back(odata.login + "@narod.ru");
                                    odata.login_virt1.push_back(odata.login + "@yandex.kz");
                                    odata.login_virt1.push_back(odata.login + "@yandex.com");
                                }
                            }
                        }
                    }
                    par_it = pReqParamsNew->find("so_login");
                    if (par_it != pReqParamsNew->end()) {
                        odata.so_login = GetValue(par_it);
                        odata.so_login.to_lower(0, odata.so_login.size());
                    }

                    if ((par_it = pReqParamsNew->find("form_id")) != pReqParamsNew->end()) {
                        odata.formId = GetValue(par_it);
                    }

                    if (RunAntiDDOSActionLogin(store, odata.so_service, odata.login, antiddos)) {
                        antiddos.m_action = true;
                        antiddos.m_field_key = "~login";
                        Spam = true;
                        controllog_dop = antiddos.GetReportControlLog();
                        DelayClass.antiddos_login = true;
                        goto MEND;
                    }

                    odata.is_black_login = false;
                    odata.is_white_login = false;
                    if (LoginCheck != NULL) {
                        //��������� � ������ "������ �������"
                        std::list<TString>::iterator lvit = odata.login_virt1.begin();
                        while (lvit != odata.login_virt1.end()) {
                            if (LoginCheck->KeyExists("black_login", (*lvit))) {
                                odata.is_black_login = true;
                                DelayClass.SetBlackLogin();
                                Spam = true;
                                break;
                            }

                            ++lvit;
                        }
                        if (!odata.is_black_login) {
                            //��������� � ������ "����� �������"
                            lvit = odata.login_virt1.begin();
                            while (lvit != odata.login_virt1.end()) {
                                if (LoginCheck->KeyExists("white_login", (*lvit))) {
                                    odata.is_white_login = true;
                                    DelayClass.SetWhiteLogin();
                                    Spam = false;
                                    break;
                                }

                                ++lvit;
                            }
                        }
                    }

                    if (odata.is_black_login) {
                        fitem->AddBlackLoginEvent();
                        GroupFilter->AddBlackLogin();
                        goto MEND;
                    }

                    if (odata.is_white_login) {
                        fitem->AddWhiteLoginEvent();
                        GroupFilter->AddWhiteLogin();
                        goto MEND;
                    }

                    //��������� �� ���-�� ��������� � ������ ������
                    if ((store != NULL) && (odata.max_mess_count_from_login > 0) && (!odata.login.empty()) && (odata.login != "-")) {
                        TCountersStatEx loginstat_predv = TCountersStatEx(); //GetLoginFastData(store, odata.login, odata.so_service, DelayClass);
                        if (loginstat_predv.All() > (ui32)odata.max_mess_count_from_login) {
                            fitem->AddMaxRequestFromLoginEvent();
                            GroupFilter->AddMaxRequestFromLogin();
                            is_max_count_from_login = true;
                            DelayClass.SetMaxMessFromLogin(odata.login);
                            Spam = true;
                            goto MEND;
                        }
                    }

                    //so_subject
                    par_it = pReqParamsNew->find("so_subject");
                    if (par_it != pReqParamsNew->end()) {
                        odata.so_subject = GetValue(par_it);
                        odata.so_subject.to_lower(0, odata.so_subject.size());
                        odata.so_subject_length = odata.so_subject.size();
                        odata.so_subject_words = GetCountWord(odata.so_subject);
                    }

                    if ((!odata.login.empty()) && (!odata.so_subject.empty()) && (odata.login != "-") && (odata.login == odata.so_subject))
                        odata.soname_cmp_sosubject = true;

                    if (odata.so_service == "MSEARCH-PROXY") {
                        //suid
                        par_it = pReqParamsNew->find("suid");
                        if (par_it != pReqParamsNew->end()) {
                            odata.so_nick = Trim(GetValue(par_it));
                            odata.so_nick_length = odata.so_nick.size();
                            odata.so_nick_words = 1;
                        }

                        //������/����� ������ ������ ��� MSearchProxy
                        odata.is_black_login = false;
                        odata.is_white_login = false;
                        if (LoginCheck != NULL) {
                            //��������� � ������ "������ �������"
                            if (LoginCheck->KeyExists("black_login", odata.so_nick)) {
                                odata.is_black_login = true;
                                DelayClass.SetBlackLogin();
                                Spam = true;
                            }

                            if (!odata.is_black_login) {
                                //��������� � ������ "����� �������"
                                if (LoginCheck->KeyExists("white_login", odata.so_nick)) {
                                    odata.is_white_login = true;
                                    DelayClass.SetWhiteLogin();
                                    Spam = false;
                                }
                            }
                        }

                        if (odata.is_black_login) {
                            fitem->AddBlackLoginEvent();
                            GroupFilter->AddBlackLogin();
                            goto MEND;
                        }

                        if (odata.is_white_login) {
                            fitem->AddWhiteLoginEvent();
                            GroupFilter->AddWhiteLogin();
                            goto MEND;
                        }

                    } else {
                        //so_nick
                        par_it = pReqParamsNew->find("so_nick");
                        if (par_it != pReqParamsNew->end()) {
                            odata.so_nick = GetValue(par_it);
                            odata.so_nick_length = odata.so_nick.size();
                            odata.so_nick_words = GetCountWord(odata.so_nick);
                        }
                    }
                    if (!odata.so_nick.empty()) {
                        odata.so_nick_shingle = GetShingle("nick_" + odata.so_nick);
                        if (RunAntiDDOSActionNick(store, odata.so_service, odata.so_nick_shingle, antiddos)) {
                            antiddos.m_action = true;
                            antiddos.m_field_key = "~nick";
                            Spam = true;
                            controllog_dop = antiddos.GetReportControlLog();
                            DelayClass.antiddos_nick = true;
                            goto MEND;
                        }
                    }

                    par_it = pReqParamsNew->find("so_phone");
                    if (par_it != pReqParamsNew->end()) {
                        odata.phone = to_lower(GetValue(par_it));
                    }

                    //so_comment
                    par_it = pReqParamsNew->find("so_comment");
                    if (par_it != pReqParamsNew->end()) {
                        odata.so_comment = GetValue(par_it);
                        odata.so_comment_length = odata.so_comment.size();
                        odata.so_comment_words = GetCountWord(odata.so_comment);
                    }

                    //so_serv_reg (sp_serv_reg) - ���� ����������� �� �������
                    par_it = pReqParamsNew->find("so_serv_reg");
                    if (par_it == pReqParamsNew->end())
                        par_it = pReqParamsNew->find("sp_serv_reg");
                    if (par_it != pReqParamsNew->end()) {
                        odata.so_reg_age = atol(GetValue(par_it).c_str());
                        odata.so_reg_age_days = odata.so_reg_age / 86400;
                        odata.so_reg_age_str = TimeToStr(odata.so_reg_age) + " (" + UI32ToStroka(odata.so_reg_age) + ", " + UI32ToStroka(odata.so_reg_age_days) + ")";
                    }

                    //so_content-length - ������ ���� content-length
                    par_it = pReqParamsNew->find("so_content-length");
                    if (par_it != pReqParamsNew->end()) {
                        odata.so_contlen = atol(GetValue(par_it).c_str());
                        odata.est_contlen = true;
                    }

                    //norm - ������������� ����� ��� MSEARCH-PROXY
                    if (odata.so_service == "MSEARCH-PROXY") {
                        par_it = pReqParamsNew->find("norm");
                        if (par_it != pReqParamsNew->end()) {
                            normtext = GetValue(par_it);
                            odata.so_comment = normtext;
                            odata.so_comment_length = odata.so_comment.size();
                            odata.so_comment_words = GetCountWord(odata.so_comment);
                        } else {
                            odata.so_comment = "";
                            odata.so_comment_length = 0;
                            odata.so_comment_words = 0;
                        }

                        par_it = pReqParamsNew->find("text");
                        if (par_it != pReqParamsNew->end()) {
                            srctext = GetValue(par_it);
                            odata.so_subject = srctext;
                            odata.so_subject.to_lower(0, odata.so_subject.size());
                            odata.so_subject_length = odata.so_subject.size();
                            odata.so_subject_words = GetCountWord(odata.so_subject);

                        } else {
                            odata.so_subject = "";
                            odata.so_subject_length = 0;
                            odata.so_subject_words = 0;
                        }
                    }

                    if (!odata.so_comment.empty()) {
                        odata.so_comment_shingle = GetShingle("comnt_" + odata.so_comment);
                        if (RunAntiDDOSActionComment(store, odata.so_service, odata.so_comment_shingle, antiddos)) {
                            antiddos.m_action = true;
                            antiddos.m_field_key = "~comment";
                            Spam = true;
                            controllog_dop = antiddos.GetReportControlLog();
                            DelayClass.antiddos_comment = true;
                            goto MEND;
                        }
                    }

                    if (odata.login)
                        odata.so_uniq_chats_by_login_shingle = GetShingle("uniq_chats_by_login_" + odata.login);

                    if (odata.login && odata.formId) {
                        odata.so_login_chatid_shingle = GetShingle(odata.login + odata.formId);
                    }

                    if (!odata.so_subject.empty()) {
                        odata.so_subject_shingle = GetShingle("subj_" + odata.so_subject);
                        if (RunAntiDDOSActionSubject(store, odata.so_service, odata.so_subject_shingle, antiddos)) {
                            antiddos.m_action = true;
                            antiddos.m_field_key = "~subject";
                            Spam = true;
                            controllog_dop = antiddos.GetReportControlLog();
                            DelayClass.antiddos_subj = true;
                            goto MEND;
                        }
                    }

                    //so_logn_age - ���� ����������� �� ��������
                    par_it = pReqParamsNew->find("so_login_age");
                    if (par_it != pReqParamsNew->end()) {
                        ts = GetValue(par_it);
                        if (ts.length() > 0) {
                            p = strstr(ts.c_str(), ":");
                            if (p != NULL)
                                odata.login_age = ConvertToTimestamp(ts);
                            else
                                odata.login_age = atol(ts.c_str());
                            if (odata.login_age > 0) {
                                if ((time(NULL) - odata.login_age) > 0)
                                    odata.login_age_days = (time(NULL) - odata.login_age) / 86400;
                                else
                                    odata.login_age_days = 0;
                                odata.login_age_str = TimeToStr(odata.login_age) + " (t:" + UI32ToStroka(odata.login_age) + ", days:" + UI32ToStroka(odata.login_age_days) + ")";
                            } else
                                odata.login_age_str = "unknown format ('" + ts + "')";
                        }
                    }

                    //������ �� ���� �� ip
                    if (ipcache.GetInfo(odata.ip, ipinf)) {
                        odata.geos = ipinf.m_geos;
                        odata.hosts = ipinf.m_hosts;
                    }
                    TString ipString = odata.ip.toStroka();
                    if (rblClient && ipString != "-") {
                        ui32 georbl_tick = CShingleTime::GetMs();
                        bool geo_okpr = false;

                        ttt = CShingleTime::GetMs();

                        auto responseOrError = rblClient->Perform(ipString, "checkform", NFuncClient::TRbl::GEO_ONLY, false);
                        Visit(responseOrError,
                              [&](const NFuncClient::TRbl::TResponse &response) mutable {
                                  geo_okpr = true;

                                  odata.geos = TStringBuilder{} << response.GetIsoName() << ' ' << response.GetCity();
                                  odata.geos_as = response.GetLastAsnItem().GetOrElse("");
                                  odata.geos_tor = response.IsTor() ? TStringBuf("tor") : TStringBuf("");
                                  geo_okpr = true;
                              },
                              [&](const NCurl::TError &error) mutable {
                                  Cerr << "rbl:" << error << Endl;
                              }
                        );

                        ttt = CShingleTime::GetMs() - ttt;


                        georbl_tick = CShingleTime::GetMs() - georbl_tick;
                        DelayClass.SetDelayGEO(georbl_tick, geo_okpr);
                    }

                    //���������
                    if (odata.hosts.empty()) {
                        if ((RBLHostObj != NULL) && (RBLHostObj->ResolvEnable())) {
                            ui32 resolv_tick = CShingleTime::GetMs();
                            bool resolv_okpr = false;

                            if (m_async_resolv) {
                                if (AsyncResolvCache != NULL) {
                                    TResolvRecordStructExt resolv_data = TResolvRecordStructExt();

                                    resolv_data = AsyncResolvCache->GetHost(NumbRequest, odata.ip);
                                    if (!resolv_data.m_notfound)
                                        odata.hosts = resolv_data.m_data.m_host;
                                }

                            } else {
                                if (!RBLHostObj->GetResolvData(NumbRequest, odata.ip, odata.hosts))
                                    odata.hosts = "";
                            }

                            if (!odata.hosts.empty())
                                resolv_okpr = true;

                            if (!odata.hosts.empty())
                                newhosts = true;

                            resolv_tick = CShingleTime::GetMs() - resolv_tick;
                            DelayClass.SetDelayResolv(resolv_tick, resolv_okpr);
                        }
                    }

                    if (newgeos || newhosts) {
                        TIPInfoItemIPv6 ipinft = TIPInfoItemIPv6(odata.ip, odata.geos, odata.hosts);
                        ipcache.AddInfo(odata.ip, ipinft);
                    }

                    //������� ��� ������
                    if (!odata.geos.empty()) {
                        odata.geo_shingle = CalcGeoShingle(odata.geos);
                        odata.geocountry_shingle = CalcGeoCountryShingle(odata.geos);
                    }

                    //��� ������� MSEARCH ��������� ������ �� ��������� �����������
                    if (odata.so_service == "MSEARCH-PROXY") {
                        TString ip_suid_total = "ip_suid_total_" + odata.ip.toStroka() + "-" + odata.so_nick;
                        TString ip_text_total = "ip_text_total_" + odata.ip.toStroka() + "-" + normtext;
                        TString ip_suid_text_total = "ip_suid_text_total_" + odata.ip.toStroka() + "-" + odata.so_nick + "-" + srctext;
                        TString ip_found = "ip_found_" + odata.ip.toStroka();
                        TString suid_found = "suid_found_" + odata.so_nick;
                        TString text_found = "text_found_" + normtext;
                        TString geos_found = "geos_found_" + odata.geos;
                        TString ip_suid_found = "ip_suid_found_" + odata.ip.toStroka() + "-" + odata.so_nick;
                        TString ip_text_found = "ip_text_found_" + odata.ip.toStroka() + "-" + normtext;

                        odata.ip_suid_total_hash = ShingleFromStroka(ip_suid_total);
                        odata.ip_text_total_hash = ShingleFromStroka(ip_text_total);
                        odata.ip_suid_text_total_hash = ShingleFromStroka(ip_suid_text_total);
                        odata.ip_found_hash = ShingleFromStroka(ip_found);
                        odata.suid_found_hash = ShingleFromStroka(suid_found);
                        odata.text_found_hash = ShingleFromStroka(text_found);
                        odata.geos_found_hash = ShingleFromStroka(geos_found);
                        odata.ip_suid_found_hash = ShingleFromStroka(ip_suid_found);
                        odata.ip_text_found_hash = ShingleFromStroka(ip_text_found);

                        if (odata.msearch_step2) //��� ������� ���� ������ ����������� *_found_hash ������ � �������
                        {
                            if (!odata.no_write_storage) {
                                if (store != NULL) {
                                    //����������� ���������
                                    if (odata.msearch_found > 0)
                                        DelayClass.updmsearchfound_tick = UpdateMSearchFound(NumbRequest, store, odata, false, odata.rcpt_count, DelayClass); //as HAM
                                    else
                                        DelayClass.updmsearchfound_tick = UpdateMSearchFound(NumbRequest, store, odata, true, odata.rcpt_count, DelayClass); //as SPAM
                                    DelayClass.updmsearchfound = true;

                                    //����������� �������������� ���������� �� ip
                                    if (odata.msearch_found > 0) {
                                        DelayClass.updlongipstat_tick = UpdateLongIPStat(NumbRequest, store, odata.so_service, odata.ip.toStroka2(), stordata::LIP_FOUND);
                                        DelayClass.updlongipstat = true;
                                    }
                                }
                            }

                            goto MEND;
                        }
                    }

                    //********************************** PARSE HOSTS, EMAILS AND PHONES FROM ALL FIELDS **********************************

                    DelayClass.parse_host_email_phone_tick = ParseHostsMailsPhones(pReqParamsNew, odata.so_service,
                                                                                   odata.urlhash_t, odata.urlmail_parse_max,
                                                                                   odata.hlist, odata.max_host_count_in_field,
                                                                                   odata.mlist, odata.max_email_count_in_field,
                                                                                   odata.plist, odata.max_phone_count_in_field,
                                                                                   odata.stordata_hmp);
                    DelayClass.parse_host_email_phone = true;


                    //********************************** GET DATA FROM LONG IP STORAGE **********************************

                    odata.longipdataFuture = NThreading::Async([this, &DelayClass, &NumbRequest, &odata, &store]() {
                        TLongIPData longIPData;
                        DelayClass.get_longip_stat_tick = GetLongIPStat(NumbRequest, store, odata.so_service, odata.ip.toStroka2(), longIPData, DelayClass.ipstat_collision_count);
                        DelayClass.get_longip_stat = true;
                        return longIPData;
                    }, threadPool);

                    //********************************** GET DATA FROM LONG PHONE STORAGE **********************************

                    odata.longphonedataFuture = NThreading::Async([this, &DelayClass, &NumbRequest, &odata, &store]() {
                        TLongIPData longIPData;
                        DelayClass.get_longphone_stat_tick = GetLongIPStat(NumbRequest, store, odata.so_service, odata.phone, longIPData, DelayClass.phonestat_collision_count);
                        DelayClass.get_longphone_stat = true;
                        return longIPData;
                    }, threadPool);

                    //********************************** GET DATA FROM STORAGE **********************************

                    DelayClass.get_from_storage_tick = GetServiceStorage(NumbRequest, store, odata, DelayClass.service_stat_collision_count, DelayClass.shingles_collision_count);
                    DelayClass.get_from_storage = true;

                    //********************************** GET DATA FROM BB **********************************






                    if ((odata.so_comment && cleanWebHost) || (odata.login && bbClient) || (odata.urlhash_t && urlReputationClient))
                        odata.curlFuture = NThreading::Async([this,
                                                              urlhash_t = odata.urlhash_t,
                                                              so_comment = odata.so_comment,
                                                              login = odata.so_login,
                                                              so_service = odata.so_service,
                                                              UseSoLoginAsUid]() {

                            TMaybe<TUrlStatisticVector> urlRepInfo;

                            if(urlhash_t && urlReputationClient) {
                                TUrlStatisticVector url_reputation_vec(Reserve(urlhash_t.size()));
                                for(const auto& [key, urlStat] : urlhash_t) {
                                    if (!FieldCheck->KeyExistsCompare("excludehosts", urlStat.m_host))
                                        url_reputation_vec.emplace_back(urlStat.m_url, CODES_WIN, false, false);
                                }

                                if (url_reputation_vec && urlReputationClient->Get(url_reputation_vec, SysLogger())) {
                                    urlRepInfo = std::move(url_reputation_vec);
                                }
                            }

                            TBBInfo bbInfo;
                            if (login && !login.EndsWith("@yandex-team.ru") && bbClient)
                                try {
                                    bbInfo = GetBB(((so_service == "CHECKMESSAGES") || UseSoLoginAsUid) ? NFuncClient::CBB::TBbKey{TUid{login}} : NFuncClient::CBB::TBbKey{TLogin{login}});
                                } catch (...) {
                                    Cerr << __FILE__ << ':' << __LINE__ << ' ' << CurrentExceptionMessageWithBt() << Endl;
                                }

                            TMaybe<TString> cwInfo;
                            if (so_comment && cleanWebHost)
                                try {
                                    cwInfo = CleanWebRequest(so_comment);
                                } catch (...) {
                                    Cerr << __FILE__ << ':' << __LINE__ << ' ' << CurrentExceptionMessageWithBt() << Endl;
                                }

                            return std::make_tuple(std::move(bbInfo), std::move(cwInfo), std::move(urlRepInfo));
                        }, threadPool);

                    //********************************************* CALL FILTER *********************************************
                    renginepool = fitem->GetFilterPointer();
                    if (renginepool != NULL) {
                        if (true) {
                            ui32 memoryindex = 0;
                            ui32 loop_count = 0;
                            ui32 filter_index = 0;

                            ttt = CShingleTime::GetMs();
                            TRengineElementCF* tre = renginepool->GetFilter(memoryindex, loop_count);
                            if (tre != NULL) {
                                ttt = CShingleTime::GetMs() - ttt;
                                DelayClass.get_filter_ok = true;
                                DelayClass.get_filter_ok_tick = ttt;

                                filter_index = tre->ObjectIndex();

                                if ((renginepool != NULL) && (renginepool->GetRulesMonStat() != NULL))
                                    renginepool->GetRulesMonStat()->AddFilterCall(tre->GetRulesCS());

                                DelayClass.SetLoopCount(loop_count);
                                DelayClass.SetFilterIndex(filter_index);

                                tre->InitAction();
                                Y_DEFER{
                                    tre->UnLock();
                                };

                                TRengine& m_hSp = tre->GetFilterHandle();
                                TString filter_index_log = tre->RengineTypeInfo() + "." + IntToStroka(filter_index) + "=" + tre->GetRulesCS();

                                TFilterResult fr = AppleFilter(sid,
                                                               js,
                                                               filter_index_log,
                                                               pReqParamsNew,
                                                               &m_hSp,
                                                               odata,
                                                               DelayClass,
                                                               DelayClass.filterwork_delay,
                                                               receipt,
                                                               LogsGroup,
                                                               service);

                                DelayClass.filterwork = true;

                                Spam = fr.Spam;
                                dirtyword = fr.dirtyword;
                                banrule = fr.banrule;

                                renginepool->ReturnFilter(tre, memoryindex, odata.so_service, Spam); //������ ��������, ���������� ��� � ��� ��������

                                if (true) {
                                    if (Spam)
                                        DelayClass.updlongipstat_tick = UpdateLongIPStat(NumbRequest, store, odata.so_service, odata.ip.toStroka2(), stordata::LIP_SPAM);
                                    else
                                        DelayClass.updlongipstat_tick = UpdateLongIPStat(NumbRequest, store, odata.so_service, odata.ip.toStroka2(), stordata::LIP_HAM);
                                    DelayClass.updlongipstat = true;
                                }
                                if (true) {
                                    if (Spam)
                                        DelayClass.updlongphonestat_tick = UpdateLongIPStat(NumbRequest, store, odata.so_service, odata.phone, stordata::LPHONE_SPAM);
                                    else
                                        DelayClass.updlongphonestat_tick = UpdateLongIPStat(NumbRequest, store, odata.so_service, odata.phone, stordata::LPHONE_HAM);
                                    DelayClass.updlongphonestat = true;
                                }
                                {
                                    // Store request
                                    DelayClass.upd_to_storage_tick = UpdateServiceStorage(NumbRequest, store, odata, normtext, Spam, odata.rcpt_count, DelayClass);
                                    DelayClass.upd_to_storage = true;

                                    ttt = CShingleTime::GetMs() - ttt;
                                }

                            } else {
                                ttt = CShingleTime::GetMs() - ttt;
                                DelayClass.get_filter_failed = true;
                                DelayClass.get_filter_failed_tick = ttt;

                                renginepool->AddSkeepFilter();
                            }
                        }
                    }
                }

            MEND:
                //fitem->UnLock();

                if (antiddos.m_action)
                    fitem->AddStatToTop20List("<ANTIDDOS>", antiddos.m_field_key);

                if (fitem->GetControlLog() != NULL) {
                    control_str = "";
                    if (Spam)
                        control_str += "********** MESSID: " + sid + ", SPAM: YES " + controllog_dop + "**********\n";
                    else
                        control_str += "********** MESSID: " + sid + ", SPAM: NO " + controllog_dop + "**********\n";

                    if (odata.is_black_login)
                        control_str += "IS_BLACK_LOGIN '" + odata.login + "'\n";
                    if (odata.is_white_login)
                        control_str += "IS_WHITE_LOGIN '" + odata.login + "'\n";
                    if (is_max_count_from_login)
                        control_str += "IS_MAX_MESS_COUNT_FROM_LOGIN '" + odata.login + "' (" + IntToStroka(odata.max_mess_count_from_login) + ")\n";
                    if (pReqParamsNew != NULL) {
                        par_it = pReqParamsNew->begin();
                        while (par_it != pReqParamsNew->end()) {
                            control_str = control_str + RemoveCR(par_it->first) + ": " + RemoveCR(par_it->second[0]) + "\n";
                            ++par_it;
                        }
                    }
                    control_str += "******************************** END MESS **************************************\n";
                    fitem->GetControlLog()->WriteMessageAndDataBUFF(control_str.c_str(), control_str.size());
                }
            }

        } else {
            flag_ok = 4;
        }

    } else {
        flag_ok = 5;
    }

    if (pReqParamsNew != NULL) {
        pReqParamsNew->clear();
        delete pReqParamsNew;
        pReqParamsNew = NULL;
    }

    DelayClass.check_delay = CShingleTime::GetMs() - DelayClass.check_delay;

    if (flag_ok == 0 && payloadIdHash) {
        TResolution resolution;
        resolution.service = service;
        resolution.Spam = Spam;
        resolution.formname = formname_s;
        resolution.Skeep = skeep;
        resolution.dirtyword = dirtyword;
        resolution.banrule = banrule;
        resolution.format = format;
        resolution.antiddos = antiddos;

        resCache.Add(payloadIdHash, std::move(resolution));
    }

    return flag_ok;
}

int CSoFilterCF::CreateServiceN(TString& ServiceName, TString& score, TString& rules) {
    int res = -1;

    if (GroupFilter != NULL)
        res = GroupFilter->CreateServiceF(ServiceName, score, rules);

    return res;
}

//
// Close
//
void CSoFilterCF::Close() {
    if (GroupFilter != NULL)
        GroupFilter->Close();
}

void CSoFilterCF::CleanStore() {
    if (GroupFilter != NULL)
        GroupFilter->CleanStore();
}

void CSoFilterCF::Midnight() {
    if (GroupFilter != NULL)
        GroupFilter->Midnight();
    ipcache.Midnight();
    ipcache.WriteDump();

    normtext_heavystat.Midnight();
    suid_heavystat.Midnight();
    ip_heavystat.Midnight();

    if (AsyncResolvCache != nullptr)
        AsyncResolvCache->Midnight();
}

void CSoFilterCF::Shutdown() {
}

void CSoFilterCF::EventTick() {
    bbCache.EventTick();
    resCache.EventTick();
    cleanWebCache.EventTick();
}

//
// Reset
//
bool CSoFilterCF::Reset() {
    bool res = false;

    if (GroupFilter != NULL)
        res = GroupFilter->Reset();
    ReloadBlackList();
    ReloadWhiteList();

    return res;
}

TResetResponceCF CSoFilterCF::ReloadRules() {
    TResetResponceCF res;

    if (GroupFilter != NULL)
        res = GroupFilter->ReloadRules();
    ReloadBlackList();
    ReloadWhiteList();

    return res;
}

void CSoFilterCF::WriteToLog(const TLogStatus LogLevel, const TString& NumbRequest, const char* msg, ...) {
    if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL)) {
        char tbuff[4096];
        TString rid = "";
        va_list args;

        rid = NumbRequest;

        memset(tbuff, 0, sizeof(tbuff));
        va_start(args, msg);
        vsnprintf(tbuff, sizeof(tbuff) - 1, msg, args);
        va_end(args);

        switch (LogLevel) {
            case KMESSAGE:
                LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "%17s\t%s", rid.c_str(), tbuff);
                break;
            case KWARNING:
                LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KWARNING, "%17s\t%s", rid.c_str(), tbuff);
                break;
            case KERROR:
                LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "%17s\t%s", rid.c_str(), tbuff);
                break;
            default:
                break;
        }
    }
}

TString CSoFilterCF::GetDomen2level(const TString& host) {
    TString res = host;
    int point_count = 0;
    bool est = false;

    if (!res.empty()) {
        ReverseInPlace(res);
        for (size_t i = 0; i < res.length(); i++) {
            if (res[i] == '.') {
                point_count++;
                if (point_count == 2) {
                    res = res.substr(0, i);
                    est = true;
                    break;
                }
            }
        }
        ReverseInPlace(res);
    }

    if (!est) //���� host ��� �������� ������� ������� ������, �� ������ �� ����������
        res = "";

    return res;
}

TString CSoFilterCF::GetDomen3level(const TString& host) {
    TString res = host;
    int point_count = 0;
    bool est = false;

    if (!res.empty()) {
        ReverseInPlace(res);
        for (size_t i = 0; i < res.length(); i++) {
            if (res[i] == '.') {
                point_count++;
                if (point_count == 3) {
                    res = res.substr(0, i);
                    est = true;
                    break;
                }
            }
        }
        ReverseInPlace(res);
    }
    if (!est) //���� host ��� �������� ������� �������� ������, �� ������ �� ����������
        res = "";

    return res;
}

ui32 CSoFilterCF::SpCheckFieldTick(TRengine* sph, const char* pFieldName, const char* pField, int FieldLen, const char* coding, bool fValid, bool fMime, chkfrm::TRuleInfoHash* ruleshash) {
    ui32 res = 0;

    res = CShingleTime::GetMs();

    SpCheckField(sph, pFieldName, {pField, size_t(FieldLen)}, coding, fValid, fMime);

    res = CShingleTime::GetMs() - res;

    if (ruleshash != NULL) {
        chkfrm::TRuleInfoHashIt it;
        TString fieldname = TString(pFieldName);

        it = ruleshash->find(fieldname);
        if (it != ruleshash->end())
            (*it).second.AppendTime(res);
        else
            (*ruleshash)[fieldname] = chkfrm::TRuleInfo(1, res);
    }

    return res;
}

void CSoFilterCF::ReloadBlackList(void) {
    if (BlackList != NULL)
        BlackList->ReloadFileList();
}

void CSoFilterCF::ReloadWhiteList(void) {
    if (WhiteList != NULL)
        WhiteList->ReloadFileList();
}

bool CSoFilterCF::RunAntiDDOSItem(const TString& job, const TString& field_value, ui32 storvalue, ui32& tick, bool skeep_cnt, bool skeep_in, bool skeep_re) {
    bool res = false;
    TString cnt_ident = "cnt:";
    TString input_ident = "in:";
    TString regular_ident = "re:";

    tick = CShingleTime::GetMs();
    if (!job.empty()) {
        if ((job.length() >= cnt_ident.length()) && (memcmp(job.c_str(), cnt_ident.c_str(), cnt_ident.length()) == 0)) {
            if (!skeep_cnt) {
                ui32 job_count = 0;

                job_count = atol(job.c_str() + cnt_ident.length());
                if (storvalue >= job_count)
                    res = true;
            }

        } else if ((job.length() >= input_ident.length()) && (memcmp(job.c_str(), input_ident.c_str(), input_ident.length()) == 0)) {
            if (!skeep_in) {
                const char* p = NULL;

                p = strstr(field_value.c_str(), job.c_str() + input_ident.length());
                if (p != NULL)
                    res = true;
            }

        } else if ((job.length() >= regular_ident.length()) && (memcmp(job.c_str(), regular_ident.c_str(), regular_ident.length()) == 0)) {
            if (!skeep_re) {
            }
        }
    }
    tick = CShingleTime::GetMs() - tick;

    return res;
}

bool CSoFilterCF::RunAntiDDOSActionOther(const TString& so_service, const TString& field_name, const TString& field_value, TAntiDDOSCriterionInfo& antiddos) {
    bool res = false;
    TString job_s = "";
    int item_counter = 0;

    antiddos.m_so_service = so_service;
    antiddos.m_criterion_number = -1;
    if (AntiDDOSList != NULL) {
        antiddos.m_all_tick = CShingleTime::GetMs();
        if (AntiDDOSList->SectionExists(so_service)) {
            job_s = AntiDDOSList->ReadStroka(so_service, field_name, "");
            if (!job_s.empty()) {
                res = RunAntiDDOSItem(job_s, field_value, 0, antiddos.m_criterion_tick, true, false, false);
                if (res)
                    antiddos.m_criterion_number = 0;
            }

            if (!res) {
                while (!job_s.empty()) {
                    item_counter++;
                    if (item_counter > 99)
                        break;
                    job_s = AntiDDOSList->ReadStroka(so_service, field_name + "_" + IntToStroka2(item_counter), "");
                    if (!job_s.empty()) {
                        res = RunAntiDDOSItem(job_s, field_value, 0, antiddos.m_criterion_tick, true, false, false);
                        if (res) {
                            antiddos.m_criterion_number = item_counter;
                            break;
                        }
                    }
                }
            }
        }
        antiddos.m_all_tick = CShingleTime::GetMs() - antiddos.m_all_tick;
    }

    return res;
}

bool CSoFilterCF::NeedAddCntAntiDDOSAction(const TString& so_service, const TString& field_name) {
    bool res = false;
    TString job_s = "";

    if (AntiDDOSList != NULL) {
        job_s = AntiDDOSList->ReadStroka(so_service, field_name, "");
        if (!job_s.empty()) {
            TString cnt_ident = "cnt:";
            const char* p = NULL;

            p = strstr(job_s.c_str(), cnt_ident.c_str());
            if (p != NULL)
                res = true;
        }
    }

    return res;
}

bool CSoFilterCF::RunAntiDDOSActionIP(TStoreBase* store, const TString& so_service, TKIPv6 ip, TAntiDDOSCriterionInfo& antiddos) {
    bool res = false;
    TString job_s = "";
    ui32 storvalue = 0;
    int remain_sec = -1;

    antiddos.m_so_service = so_service;
    antiddos.m_criterion_number = -1;
    if ((AntiDDOSList != NULL) && (store != NULL)) {
        antiddos.m_all_tick = CShingleTime::GetMs();
        if (AntiDDOSList->SectionExists(so_service)) {
            job_s = AntiDDOSList->ReadStroka(so_service, "~ip", "");
            if (!job_s.empty()) {
                if (store->GetAntiDDOSActionIP(so_service, ip, storvalue, remain_sec)) {
                    res = RunAntiDDOSItem(job_s, "", storvalue, antiddos.m_criterion_tick, false, true, true);
                    if (res) {
                        antiddos.m_criterion_number = storvalue;
                        antiddos.m_remain_sec = remain_sec;
                    }
                }
            }
        }
        antiddos.m_all_tick = CShingleTime::GetMs() - antiddos.m_all_tick;
    }

    return res;
}

bool CSoFilterCF::RunAntiDDOSActionLogin(TStoreBase* store, const TString& so_service, const TString& login, TAntiDDOSCriterionInfo& antiddos) {
    bool res = false;
    TString job_s = "";
    ui32 storvalue = 0;
    int remain_sec = -1;

    antiddos.m_so_service = so_service;
    antiddos.m_criterion_number = -1;
    if ((AntiDDOSList != NULL) && (store != NULL)) {
        antiddos.m_all_tick = CShingleTime::GetMs();
        if (AntiDDOSList->SectionExists(so_service)) {
            job_s = AntiDDOSList->ReadStroka(so_service, "~login", "");
            if (!job_s.empty()) {
                if (store->GetAntiDDOSActionLogin(so_service, login, storvalue, remain_sec)) {
                    res = RunAntiDDOSItem(job_s, "", storvalue, antiddos.m_criterion_tick, false, true, true);
                    if (res) {
                        antiddos.m_criterion_number = storvalue;
                        antiddos.m_remain_sec = remain_sec;
                    }
                }
            }
        }
        antiddos.m_all_tick = CShingleTime::GetMs() - antiddos.m_all_tick;
    }

    return res;
}

bool CSoFilterCF::RunAntiDDOSActionNick(TStoreBase* store, const TString& so_service, ui64 nick_shingle, TAntiDDOSCriterionInfo& antiddos) {
    bool res = false;
    TString job_s = "";
    ui32 storvalue = 0;
    int remain_sec = -1;

    antiddos.m_so_service = so_service;
    antiddos.m_criterion_number = -1;
    if ((AntiDDOSList != NULL) && (store != NULL)) {
        antiddos.m_all_tick = CShingleTime::GetMs();
        if (AntiDDOSList->SectionExists(so_service)) {
            job_s = AntiDDOSList->ReadStroka(so_service, "~nick", "");
            if (!job_s.empty()) {
                if (store->GetAntiDDOSActionNick(so_service, nick_shingle, storvalue, remain_sec)) {
                    res = RunAntiDDOSItem(job_s, "", storvalue, antiddos.m_criterion_tick, false, true, true);
                    if (res) {
                        antiddos.m_criterion_number = storvalue;
                        antiddos.m_remain_sec = remain_sec;
                    }
                }
            }
        }
        antiddos.m_all_tick = CShingleTime::GetMs() - antiddos.m_all_tick;
    }

    return res;
}

bool CSoFilterCF::RunAntiDDOSActionComment(TStoreBase* store, const TString& so_service, ui64 comment_shingle, TAntiDDOSCriterionInfo& antiddos) {
    bool res = false;
    TString job_s = "";
    ui32 storvalue = 0;
    int remain_sec = -1;

    antiddos.m_so_service = so_service;
    antiddos.m_criterion_number = -1;
    if ((AntiDDOSList != NULL) && (store != NULL)) {
        antiddos.m_all_tick = CShingleTime::GetMs();
        if (AntiDDOSList->SectionExists(so_service)) {
            job_s = AntiDDOSList->ReadStroka(so_service, "~comment", "");
            if (!job_s.empty()) {
                if (store->GetAntiDDOSActionComment(so_service, comment_shingle, storvalue, remain_sec)) {
                    res = RunAntiDDOSItem(job_s, "", storvalue, antiddos.m_criterion_tick, false, true, true);
                    if (res) {
                        antiddos.m_criterion_number = storvalue;
                        antiddos.m_remain_sec = remain_sec;
                    }
                }
            }
        }
        antiddos.m_all_tick = CShingleTime::GetMs() - antiddos.m_all_tick;
    }

    return res;
}

bool CSoFilterCF::RunAntiDDOSActionXMPPServerName(TStoreBase* store, const TString& so_service, ui64 xmpp_srvname_shingle, TAntiDDOSCriterionInfo& antiddos) {
    bool res = false;
    TString job_s = "";
    ui32 storvalue = 0;
    int remain_sec = -1;

    antiddos.m_so_service = so_service;
    antiddos.m_criterion_number = -1;
    if ((AntiDDOSList != NULL) && (store != NULL)) {
        antiddos.m_all_tick = CShingleTime::GetMs();
        if (AntiDDOSList->SectionExists(so_service)) {
            job_s = AntiDDOSList->ReadStroka(so_service, "~xmpp_srvname", "");
            if (!job_s.empty()) {
                if (store->GetAntiDDOSActionXMPPServerName(so_service, xmpp_srvname_shingle, storvalue, remain_sec)) {
                    res = RunAntiDDOSItem(job_s, "", storvalue, antiddos.m_criterion_tick, false, true, true);
                    if (res) {
                        antiddos.m_criterion_number = storvalue;
                        antiddos.m_remain_sec = remain_sec;
                    }
                }
            }
        }
        antiddos.m_all_tick = CShingleTime::GetMs() - antiddos.m_all_tick;
    }

    return res;
}

bool CSoFilterCF::RunAntiDDOSActionSubject(TStoreBase* store, const TString& so_service, ui64 subject_shingle, TAntiDDOSCriterionInfo& antiddos) {
    bool res = false;
    TString job_s = "";
    ui32 storvalue = 0;
    int remain_sec = -1;

    antiddos.m_so_service = so_service;
    antiddos.m_criterion_number = -1;
    if ((AntiDDOSList != NULL) && (store != NULL)) {
        antiddos.m_all_tick = CShingleTime::GetMs();
        if (AntiDDOSList->SectionExists(so_service)) {
            job_s = AntiDDOSList->ReadStroka(so_service, "~subject", "");
            if (!job_s.empty()) {
                if (store->GetAntiDDOSActionSubject(so_service, subject_shingle, storvalue, remain_sec)) {
                    res = RunAntiDDOSItem(job_s, "", storvalue, antiddos.m_criterion_tick, false, true, true);
                    if (res) {
                        antiddos.m_criterion_number = storvalue;
                        antiddos.m_remain_sec = remain_sec;
                    }
                }
            }
        }
        antiddos.m_all_tick = CShingleTime::GetMs() - antiddos.m_all_tick;
    }

    return res;
}

void CSoFilterCF::AddAntiDDOSActionIP(TStoreBase* store, const TString& so_service, TKIPv6 ip, TCountersStatEx value) {
    if ((store != NULL) && (NeedAddCntAntiDDOSAction(so_service, "~ip")))
        store->AddAntiDDOSActionIP(so_service, ip, value);
}

void CSoFilterCF::AddAntiDDOSActionLogin(TStoreBase* store, const TString& so_service, const TString& login, TCountersStat value) {
    if ((store != NULL) && (NeedAddCntAntiDDOSAction(so_service, "~login")))
        store->AddAntiDDOSActionLogin(so_service, login, value);
}

void CSoFilterCF::AddAntiDDOSActionLogin(TStoreBase* store, const TString& so_service, const TString& login, TCountersStatEx value) {
    if ((store != NULL) && (NeedAddCntAntiDDOSAction(so_service, "~login")))
        store->AddAntiDDOSActionLogin(so_service, login, value);
}

void CSoFilterCF::AddAntiDDOSActionNick(TStoreBase* store, const TString& so_service, ui64 nick_shingle, TCountersStatEx value) {
    if ((store != NULL) && (NeedAddCntAntiDDOSAction(so_service, "~nick")))
        store->AddAntiDDOSActionNick(so_service, nick_shingle, value);
}

void CSoFilterCF::AddAntiDDOSActionXMPPServerName(TStoreBase* store, const TString& so_service, ui64 xmpp_srvname_shingle, TCountersStat value) {
    if ((store != NULL) && (NeedAddCntAntiDDOSAction(so_service, "~xmpp_srvname")))
        store->AddAntiDDOSActionXMPPServerName(so_service, xmpp_srvname_shingle, value);
}

void CSoFilterCF::AddAntiDDOSActionXMPPServerName(TStoreBase* store, const TString& so_service, ui64 xmpp_srvname_shingle, TCountersStatEx value) {
    if ((store != NULL) && (NeedAddCntAntiDDOSAction(so_service, "~xmpp_srvname")))
        store->AddAntiDDOSActionXMPPServerName(so_service, xmpp_srvname_shingle, value);
}

void CSoFilterCF::AdddAntiDDOSActionComment(TStoreBase* store, const TString& so_service, ui64 comment_shingle, TCountersStatEx value) {
    if ((store != NULL) && (NeedAddCntAntiDDOSAction(so_service, "~comment")))
        store->AdddAntiDDOSActionComment(so_service, comment_shingle, value);
}

void CSoFilterCF::AdddAntiDDOSActionSubject(TStoreBase* store, const TString& so_service, ui64 subject_shingle, TCountersStat value) {
    if ((store != NULL) && (NeedAddCntAntiDDOSAction(so_service, "~subject")))
        store->AdddAntiDDOSActionSubject(so_service, subject_shingle, value);
}

void CSoFilterCF::AdddAntiDDOSActionSubject(TStoreBase* store, const TString& so_service, ui64 subject_shingle, TCountersStatEx value) {
    if ((store != NULL) && (NeedAddCntAntiDDOSAction(so_service, "~subject")))
        store->AdddAntiDDOSActionSubject(so_service, subject_shingle, value);
}

TString CSoFilterCF::ViewResolvWebStat(const TString& ipfromcache_form) {
    TString res = "";

    if ((RBLHostObj != NULL) && (RBLHostObj->ResolvEnable())) {
        if ((m_async_resolv) && (AsyncResolvCache != NULL))
            res = AsyncResolvCache->GetWebStat(ipfromcache_form);
        else
            res = "Async mode - disabled!";

    } else {
        res = "Resolv disabled!";
    }

    return res;
}

TString CSoFilterCF::GetRslvIPFromCache(TKIPv6 ip) {
    TString res = "";

    if ((RBLHostObj != NULL) && (RBLHostObj->ResolvEnable())) {
        if ((m_async_resolv) && (AsyncResolvCache != NULL))
            res = AsyncResolvCache->GetRslvIPFromCache(ip);
        else
            res = "Async mode - disabled!";

    } else {
        res = "Resolv disabled!";
    }

    return res;
}

void CSoFilterCF::GetUdnsServicesIPData(const TString& NumbRequest, TKIPv6 ip, TSummDataUdnsList& stat_list) {
    stat_list.clear();
    if (RBLHostObj != NULL)
        RBLHostObj->GetAllStat(NumbRequest, ip, stat_list);
}

TString IntToStrM(ui32 value) {
    TString res = "";
    char tbuff[25];

    if (value == 0)
        res = "-";
    else {
        tbuff[sizeof(tbuff) - 1] = 0x00;
        snprintf(tbuff, sizeof(tbuff) - 1, "%u", value);
        res = TString(tbuff);
    }

    return res;
}

TString CSoFilterCF::GetServicesStatistik() {
    TString res = "";
    const char* table_color = "'#ffffcc'";
    TServiceFullStatList srvc_list;
    TServiceFullStatListIt it;
    stordata::TShingleDataList shingles_list;
    TStoreBase* store = NULL;
    ui32 service_stat_collision_count = 0;
    ui32 shingles_collision_count = 0;

    if (GroupFilter != NULL) {
        GroupFilter->GetServicesList(srvc_list);
        store = GroupFilter->GetCommonStore();
    }

    shingles_list.clear();
    if (srvc_list.size() > 0) {
        it = srvc_list.begin();
        while (it != srvc_list.end()) {
            if (!it->m_so_service.empty()) {
                it->m_data.so_service = it->m_so_service;
                it->m_data.so_service_from_fitem = it->m_so_service;
                CreateStatShingleList(shingles_list, it->m_data);
            }

            ++it;
        }

        if (store != NULL) {
            store->GetDataFromStorage("", shingles_list, service_stat_collision_count, shingles_collision_count);

            //debug
            {
                TString tv0 = "";
                stordata::DATATYPE tv1;
                stordata::TShingleDataItem tv2;
                stordata::TShingleDataItem tv3;
                TSrvcStatMinItem tv4;

                auto it = shingles_list.begin();
                while (it != shingles_list.end()) {
                    tv0 = it->m_so_service;
                    tv1 = it->m_type;
                    tv2 = it->m_data_today;
                    tv3 = it->m_data_yesterday;
                    if (it->m_stat_p != NULL)
                        tv4 = *it->m_stat_p;

                    ++it;
                }
            }
        }
    }

    res += "<table border='1' width='100%' cellspacing='0' cellpadding='4' bgcolor=" + TString(table_color) + ">";

    res += "<tr align='center'>";
    res += "<td width='25%' rowspan='2'>&nbsp;</td>";
    res += "<td width='25%' colspan='3'><b>5 min</b></td>";
    res += "<td width='25%' colspan='3'><b>TODAY</b></td>";
    res += "<td width='25%' colspan='3'><b>YESTERDAY</b></td>";
    res += "</tr>";

    res += "<tr align='center'>";
    //res += "<td width='25%'>&nbsp;</td>";
    res += "<td width='9%'><b>ALL (rqst/rcpt)</b></td>";
    res += "<td width='8%'><b>SPAM (rqst/rcpt)</b></td>";
    res += "<td width='8%'><b>ERROR (rqst/rcpt)</b></td>";
    res += "<td width='9%'><b>ALL (rqst/rcpt)</b></td>";
    res += "<td width='8%'><b>SPAM (rqst/rcpt)</b></td>";
    res += "<td width='8%'><b>ERROR (rqst/rcpt)</b></td>";
    res += "<td width='9%'><b>ALL (rqst/rcpt)</b></td>";
    res += "<td width='8%'><b>SPAM (rqst/rcpt)</b></td>";
    res += "<td width='8%'><b>ERROR (rqst/rcpt)</b></td>";
    res += "</tr>";

    if (srvc_list.size() > 0) {
        TString tmp_s = "";
        TSrvcStatMin tmp_stat;

        it = srvc_list.begin();
        while (it != srvc_list.end()) {
            if (!it->m_so_service.empty()) {
                tmp_s = it->m_so_service;
                tmp_stat = it->m_data.srvc_data;

                res += "<tr align='right'>";
                res += "<td align='left'>" + it->m_so_service + "</td>";
                if (it->m_data.srvc_data.m_request.m_5min.m_collision || it->m_data.srvc_data.m_rcpt.m_5min.m_collision) {
                    res += "<td colspan='3'><b>COLLISION</b></td>";

                } else {
                    res += "<td>" + IntToStrM(it->m_data.srvc_data.m_request.m_5min.m_all) + " (" + IntToStrM(it->m_data.srvc_data.m_rcpt.m_5min.m_all) + ")</td>";
                    res += "<td>" + IntToStrM(it->m_data.srvc_data.m_request.m_5min.m_spam) + " (" + IntToStrM(it->m_data.srvc_data.m_rcpt.m_5min.m_spam) + ")</td>";
                    res += "<td>" + IntToStrM(it->m_data.srvc_data.m_request.m_5min.m_error) + " (" + IntToStrM(it->m_data.srvc_data.m_rcpt.m_5min.m_error) + ")</td>";
                }

                if (it->m_data.srvc_data.m_request.m_day.m_collision || it->m_data.srvc_data.m_rcpt.m_day.m_collision) {
                    res += "<td colspan='3'><b>COLLISION</b></td>";

                } else {
                    res += "<td>" + IntToStrM(it->m_data.srvc_data.m_request.m_day.m_all) + " (" + IntToStrM(it->m_data.srvc_data.m_rcpt.m_day.m_all) + ")</td>";
                    res += "<td>" + IntToStrM(it->m_data.srvc_data.m_request.m_day.m_spam) + " (" + IntToStrM(it->m_data.srvc_data.m_rcpt.m_day.m_spam) + ")</td>";
                    res += "<td>" + IntToStrM(it->m_data.srvc_data.m_request.m_day.m_error) + " (" + IntToStrM(it->m_data.srvc_data.m_rcpt.m_day.m_error) + ")</td>";
                }

                if (it->m_data.srvc_data.m_request.m_yesterday.m_collision || it->m_data.srvc_data.m_rcpt.m_yesterday.m_collision) {
                    res += "<td colspan='3'><b>COLLISION</b></td>";

                } else {
                    res += "<td>" + IntToStrM(it->m_data.srvc_data.m_request.m_yesterday.m_all) + " (" + IntToStrM(it->m_data.srvc_data.m_rcpt.m_yesterday.m_all) + ")</td>";
                    res += "<td>" + IntToStrM(it->m_data.srvc_data.m_request.m_yesterday.m_spam) + " (" + IntToStrM(it->m_data.srvc_data.m_rcpt.m_yesterday.m_spam) + ")</td>";
                    res += "<td>" + IntToStrM(it->m_data.srvc_data.m_request.m_yesterday.m_error) + " (" + IntToStrM(it->m_data.srvc_data.m_rcpt.m_yesterday.m_error) + ")</td>";
                }

                res += "</tr>";
            }

            ++it;
        }
    }
    res += "</table>";

    return res;
}
