#include <mail/so/spamstop/tools/so-common/shtime.h>
#include "tdatabasa_hash.h"

TDataBasaHash::TDataBasaHash() {
    m_data = NULL;
    m_longip_data = NULL;
    m_print_debug = false;
    m_basa_type_name = "LOCALHASH";
}

TDataBasaHash::~TDataBasaHash() {
    if (m_longip_data != NULL) {
        delete m_longip_data;
        m_longip_data = NULL;
    }

    if (m_data != NULL) {
        delete m_data;
        m_data = NULL;
    }
}

bool TDataBasaHash::InitBeforeFork(TServiceType servicetypeA, const TString& server_hostname, const TString& server_code, const TString& server_start_time, const TString& server_version, TLogsGroupCF* LogsGroupA, TKConfig* configobjA) {
    return InitBeforeForkNum(-1, servicetypeA, server_hostname, server_code, server_start_time, server_version, LogsGroupA, configobjA);
}

bool TDataBasaHash::InitBeforeForkNum(int index, TServiceType servicetypeA, const TString& server_hostname, const TString& server_code, const TString& server_start_time, const TString& server_version, TLogsGroupCF* LogsGroupA, TKConfig* configobjA) {
    InitBeforeFork_base(index, server_hostname, server_code, server_start_time, server_version, LogsGroupA, configobjA, servicetypeA);

    m_storage_name = TServiceTypeToStroka(servicetypeA);

    if (configobjA != NULL)
        m_print_debug = configobjA->ReadBool("server", "print_debug", false);

    if (m_data == NULL)
        m_data = new TDaysHash();

    if (m_longip_data == NULL)
        m_longip_data = new TDaysHashLI();

    return true;
}

void TDataBasaHash::Lock() {
    m_Mutex.Acquire();
}

void TDataBasaHash::UnLock() {
    m_Mutex.Release();
}

void TDataBasaHash::InitAfterFork() {
    InitAfterFork_base();
    m_storage_status = "OK";
    m_storage_connect = true;
}

bool TDataBasaHash::StorageUpdate(ui64 shingle, stor_bt::TShingleStorageType sstype, const TString& ext_collname, const nosql::HashMap& incrs, const nosql::HashMap& sets, bool& err) {
    Y_UNUSED(shingle, sstype, ext_collname, incrs, sets, err);
    return true;
}

bool TDataBasaHash::StorageFindOne(ui64 shingle, stor_bt::TShingleStorageType sstype, const TString& ext_collname, nosql::HashMap& hash, bool& err) {
    Y_UNUSED(shingle, sstype, ext_collname, hash, err);
    return true;
}

bool TDataBasaHash::StorageFind(stor_bt::TShingleStorageType sstype, const TString& ext_collname, TVector<nosql::HashMap>& hashes, bool& err) {
    Y_UNUSED(sstype, ext_collname, hashes, err);
    return true;
}

bool TDataBasaHash::StorageErase(ui64 shingle, stor_bt::TShingleStorageType sstype, const TString& ext_collname, bool& err) {
    Y_UNUSED(shingle, sstype, ext_collname, err);
    return true;
}

i64 TDataBasaHash::StorageSize(int type, stor_bt::TShingleStorageType sstype, const TString& ext_collname, bool& err) {
    Y_UNUSED(type, sstype, ext_collname, err);
    return 0;
}

bool TDataBasaHash::StorageTableDrop(int type, stor_bt::TShingleStorageType sstype, const TString& ext_collname, bool& err) {
    Y_UNUSED(type, sstype, ext_collname, err);
    return true;
}

void TDataBasaHash::GetOtherCollection(const TString& collection_name, ui64 shingle, stor_bt::TNoSqlItem* nosqldata) {
    TDaysHashIt dit;
    TShingleDataItemHashIt sit;
    stordata::TShingleDataItem data_v;

    dit = m_data->find(collection_name);
    if (dit != m_data->end()) {
        sit = dit->second.find(shingle);
        if (sit != dit->second.end()) {
            data_v = sit->second;
            stordata::TShingleDataItemSerialization::Serialize(data_v, nosqldata->hash);
        }
    }
}

void TDataBasaHash::UpdateOtherCollection(const TString& collection_name, ui64 shingle, stor_bt::TNoSqlItem* nosqldata) {
    TDaysHashIt dit;
    TShingleDataItemHashIt sit;
    stordata::TShingleDataItem data_v;
    bool first_record = false;

    dit = m_data->find(collection_name);
    if (dit == m_data->end()) {
        (*m_data)[collection_name] = TShingleDataItemHash();
        dit = m_data->find(collection_name);
    }

    if (dit != m_data->end()) {
        sit = dit->second.find(shingle);
        if (sit == dit->second.end()) {
            dit->second[shingle] = stordata::TShingleDataItem();
            sit = dit->second.find(shingle);
            first_record = true;
        }

        if (sit != dit->second.end()) {
            data_v.Clear();
            stordata::TShingleDataItemSerialization::Deserialize2(nosqldata->incrs, nosqldata->sets, data_v);

            if (first_record) {
                sit->second.m_so_service = data_v.m_so_service;
                sit->second.m_type = data_v.m_type;
                sit->second.m_ham = data_v.m_ham;
                sit->second.m_spam = data_v.m_spam;

            } else {
                if (sit->second.m_so_service == data_v.m_so_service) {
                    const char* p = strstr(data_v.m_so_service.c_str(), TString(BASE_STAT_SHINGLE_DOPSTR).c_str());

                    if ((p != NULL) && (p == data_v.m_so_service.c_str())) //so_service stat
                    {
                        sit->second.m_type = IncMax32(sit->second.m_type, data_v.m_type); //all
                        sit->second.m_ham = IncMax32(sit->second.m_ham, data_v.m_ham);    //error
                        sit->second.m_spam = IncMax32(sit->second.m_spam, data_v.m_spam); //spam

                    } else //other
                    {
                        if (sit->second.m_type == data_v.m_type) {
                            sit->second.m_ham = IncMax32(sit->second.m_ham, data_v.m_ham);
                            sit->second.m_spam = IncMax32(sit->second.m_spam, data_v.m_spam);

                        } else {
                            //!!!COLLISION!!!
                        }
                    }

                } else {
                    //!!!COLLISION!!!
                }
            }
        }
    }
}

void TDataBasaHash::GetLongIPCollection(const TString& collection_name, ui64 shingle, stor_bt::TNoSqlItem* nosqldata) {
    TDaysHashLIIt dit;
    TLongIPDataItemHashIt sit;
    stordata::TLongIPDataItem data_v;

    dit = m_longip_data->find(collection_name);
    if (dit != m_longip_data->end()) {
        sit = dit->second.find(shingle);
        if (sit != dit->second.end()) {
            data_v = sit->second;
            stordata::TLongIPDataItemSerialization::Serialize(data_v, nosqldata->hash);
        }
    }
}

void TDataBasaHash::UpdateLongIPCollection(const TString& collection_name, ui64 shingle, stor_bt::TNoSqlItem* nosqldata) {
    TDaysHashLIIt dit;
    TLongIPDataItemHashIt sit;
    stordata::TLongIPDataItem data_v;
    bool first_record = false;

    dit = m_longip_data->find(collection_name);
    if (dit == m_longip_data->end()) {
        (*m_longip_data)[collection_name] = TLongIPDataItemHash();
        dit = m_longip_data->find(collection_name);
    }

    if (dit != m_longip_data->end()) {
        sit = dit->second.find(shingle);
        if (sit == dit->second.end()) {
            dit->second[shingle] = stordata::TLongIPDataItem();
            sit = dit->second.find(shingle);
            first_record = true;
        }

        if (sit != dit->second.end()) {
            data_v.Clear();
            stordata::TLongIPDataItemSerialization::Deserialize2(nosqldata->incrs, nosqldata->sets, data_v);

            if (first_record) {
                sit->second.m_so_service = data_v.m_so_service;
                sit->second.m_lasttime = data_v.m_lasttime;
                sit->second.m_ham = data_v.m_ham;
                sit->second.m_spam = data_v.m_spam;
                sit->second.m_found = data_v.m_found;

            } else {
                if (sit->second.m_so_service == data_v.m_so_service) {
                    sit->second.m_so_service = data_v.m_so_service;
                    sit->second.m_lasttime = data_v.m_lasttime;
                    sit->second.m_ham = IncMax32(sit->second.m_ham, data_v.m_ham);
                    sit->second.m_spam = IncMax32(sit->second.m_spam, data_v.m_spam);
                    sit->second.m_found = IncMax32(sit->second.m_found, data_v.m_found);

                } else {
                    //!!!COLLISION!!!
                }
            }
        }
    }
}

bool TDataBasaHash::IsLongIPCollection(const TString& collection_name) {
    bool res = false;
    const char* p = NULL;
    TString lip_ident = TString(LONGIP_COLLECTION_ID);

    if (!collection_name.empty()) {
        p = strstr(collection_name.c_str(), lip_ident.c_str());
        if ((p != NULL) && (p != collection_name.c_str()))
            res = true;
    }

    return res;
}

void TDataBasaHash::StorageMultiAction(stor_bt::TStorageActionList& actlist, bool& err) {
    err = false;

    if ((m_data != NULL) && (m_longip_data)) {
        stor_bt::TStorageActionListIt it;
        TString collection_name = "";
        ui64 shingle = 0;
        stordata::TShingleDataItem data_v;
        ui32 tick = 0;
        TString debug_data = "";
        ui32 tick_stat = 0;

        Lock();

        if (m_print_debug)
            debug_data += "DEBUG:\n";

        it = actlist.begin();
        while (it != actlist.end()) {
            if (m_print_debug)
                debug_data += it->toLog() + "\n";

            tick = CShingleTime::GetMs();

            shingle = 0;
            if (it->m_nosqldata != NULL) {
                shingle = it->m_nosqldata->shingle;
                if (shingle != 0) {
                    if (it->m_storage_type == stor_bt::SST_USRCOLL_ASTEXT)
                        collection_name = it->m_storage_name;

                    if (!collection_name.empty()) {
                        if (it->m_action == stor_bt::SAT_FINDONE) //get
                        {
                            tick_stat = CShingleTime::GetMs();

                            if (IsLongIPCollection(collection_name))
                                GetLongIPCollection(collection_name, shingle, it->m_nosqldata);
                            else
                                GetOtherCollection(collection_name, shingle, it->m_nosqldata);

                            tick_stat = CShingleTime::GetMs() - tick_stat;

                            AddStorageStat(it->m_storage_type, it->m_storage_name, stor_bt::SAT_FINDONE, 1, NStorageStats::SRS_OK, tick_stat);

                        } else if (it->m_action == stor_bt::SAT_UPDATE) //update
                        {
                            tick_stat = CShingleTime::GetMs();

                            if (IsLongIPCollection(collection_name))
                                UpdateLongIPCollection(collection_name, shingle, it->m_nosqldata);
                            else
                                UpdateOtherCollection(collection_name, shingle, it->m_nosqldata);

                            tick_stat = CShingleTime::GetMs() - tick_stat;

                            AddStorageStat(it->m_storage_type, it->m_storage_name, stor_bt::SAT_UPDATE, 1, NStorageStats::SRS_OK, tick_stat);
                        }
                    }
                }
            }

            tick = CShingleTime::GetMs() - tick;

            if (it->m_err != NULL)
                *it->m_err = false;

            if (it->m_tick != NULL)
                *it->m_tick = tick;

            ++it;
        }

        UnLock();

        if (m_print_debug && (LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL))
            LogsGroup->GetStorageLog()->WriteMessageAndData("%s", debug_data.c_str());
    }
}

bool TDataBasaHash::CreateCollection(const TString& collection_name, TString& err_s) {
    bool res = false;

    err_s = "";
    if (m_data != NULL) {
        if (!collection_name.empty()) {
            Lock();

            if (m_data != NULL) {
                TDaysHashIt it;

                it = m_data->find(collection_name);
                if (it == m_data->end()) {
                    res = true;
                    (*m_data)[collection_name] = TShingleDataItemHash();

                } else {
                    res = false;
                    err_s = "collection '" + collection_name + "' already exist!";
                }
            }

            UnLock();

        } else {
            res = false;
            err_s = "collection name is empty!";
        }

    } else {
        res = false;
        err_s = "internal error!";
    }

    return res;
}

bool TDataBasaHash::DeleteCollection(const TString& collection_name, TString& err_s) {
    bool res = false;

    err_s = "";
    if (m_data != NULL) {
        if (!collection_name.empty()) {
            Lock();

            if (m_data != NULL) {
                TDaysHashIt it;

                it = m_data->find(collection_name);
                if (it == m_data->end()) {
                    res = false;
                    err_s = "collection '" + collection_name + "' not found!";

                } else {
                    res = true;
                    m_data->erase(it);
                }
            }

            UnLock();

        } else {
            res = false;
            err_s = "collection name is empty!";
        }

    } else {
        res = false;
        err_s = "internal error!";
    }

    return res;
}

TString CorrectPath(const TString& path) {
    TString corr_path = "";

    if (path.length() > 0) {
#ifdef _win_
        if (path[path.length() - 1] != '\\')
            corr_path = path + "\\";
        else
            corr_path = path;
#else  //_win_
        if (path[path.length() - 1] != '/')
            corr_path = path + "/";
        else
            corr_path = path;
#endif //_win_
    }

    return corr_path;
}

bool TDataBasaHash::WriteDumpOtherCollections(const TString& path, ui32& collection_count, ui32& record_count) {
    bool res = false;
    TString collection_name = "";
    TDaysHashIt dit;
    TShingleDataItemHashIt sit;
    TString filename = "";
    FILE* handle = NULL;
    ui32 tick = CShingleTime::GetMs();
    ui32 collection_count_local = 0;
    ui32 record_count_local = 0;

    if (m_data != NULL) {
        res = true;

        dit = m_data->begin();
        while (dit != m_data->end()) {
            collection_name = dit->first;
            if (!collection_name.empty()) {
                if (path.length() > 0) {
                    filename = CorrectPath(path) + collection_name + ".dmp";

                    handle = fopen(filename.c_str(), "wb");
                    if (handle != NULL) {
                        collection_count_local = IncMax32(collection_count_local, 1);

                        sit = dit->second.begin();
                        while (sit != dit->second.end()) {
                            record_count_local = IncMax32(record_count_local, 1);
                            fprintf(handle, "%s\n", (*sit).second.writeToDump((*sit).first).c_str());

                            ++sit;
                        }

                        fclose(handle);
                        handle = NULL;

                    } else {
                        if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                            LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: error write dump other collections, can't create file %s", filename.c_str());
                    }
                }
            }

            ++dit;
        }
    }

    tick = CShingleTime::GetMs() - tick;
    if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
        if (res)
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KMESSAGE, "STORHASH: write dump other collections - OK (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
        else
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: write dump other collections - FAILED (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
    }

    collection_count = IncMax32(collection_count, collection_count_local);
    record_count = IncMax32(record_count, record_count_local);

    return res;
}

bool TDataBasaHash::ReadDumpOtherCollections(const TString& path, ui32& collection_count, ui32& record_count) {
    bool res = true;
    std::list<TString> filelist;
    TString path_s = CorrectPath(path);
    TString filename_wo_path = "";
    ui32 tick = CShingleTime::GetMs();
    ui32 collection_count_local = 0;
    ui32 record_count_local = 0;

#ifdef _win_
    FindFiles ff;
    TString mask = path_s + "*.dmp";

    if (ff.FindFirst((char*)mask.c_str()) == CERR_OK) {
        do {
            filename_wo_path = TString(ff.ff.Name);
            if ((filename_wo_path != ".") && (filename_wo_path != ".."))
                filelist.push_back(filename_wo_path);

        } while (ff.FindNext() == CERR_OK);
    }
#else
    DIR* dir = NULL;
    if ((dir = opendir(path_s.c_str())) == NULL) {
        printf("Can\'t open folder: '%s'\n", path_s.c_str());
    } else {
        struct dirent* f_cur;
        while ((f_cur = readdir(dir)) != NULL) {
            filename_wo_path = TString(f_cur->d_name);
            if ((filename_wo_path != ".") && (filename_wo_path != ".."))
                filelist.push_back(filename_wo_path);
        }
        closedir(dir);
    }
#endif

    if (filelist.size() > 0) {
        FILE* handle = NULL;
        std::list<TString>::iterator it;
        char tbuff[2048];
        TString filename = "";
        TString collection_name = "";
        TDaysHashIt dit;
        TShingleDataItemHashIt sit;
        stordata::TShingleDataItem var_data;
        ui64 shingle = 0;
        const char* p = NULL;
        int cnt = 0;

        it = filelist.begin();
        while (it != filelist.end()) {
            if (!it->empty()) {
                p = strstr(it->c_str(), ".dmp");
                if (p != NULL) {
                    cnt = p - it->c_str();
                    if (cnt > 0)
                        collection_name = TString(it->c_str(), cnt);

                } else
                    collection_name = *it;

                collection_count_local = IncMax32(collection_count_local, 1);
                filename = path_s + (*it);
                handle = fopen(filename.c_str(), "rb");
                if (handle != NULL) {
                    memset(tbuff, 0, sizeof(tbuff));
                    while (fgets(tbuff, sizeof(tbuff) - 1, handle)) {
                        var_data.Clear();
                        shingle = 0;
                        if (var_data.LoadData(TString(tbuff), shingle)) {
                            record_count_local = IncMax32(record_count_local, 1);

                            dit = m_data->find(collection_name);
                            if (dit == m_data->end()) {
                                (*m_data)[collection_name] = TShingleDataItemHash();
                                dit = m_data->find(collection_name);
                            }

                            if (dit != m_data->end()) {
                                sit = dit->second.find(shingle);
                                if (sit == dit->second.end()) {
                                    dit->second[shingle] = stordata::TShingleDataItem();
                                    sit = dit->second.find(shingle);
                                }

                                if (sit != dit->second.end())
                                    sit->second = var_data;
                            }
                        }
                    }

                    fclose(handle);
                    handle = NULL;

                } else {
                    res = false;
                    if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL))
                        LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: error read dump other collections, can't open file %s", filename.c_str());
                }
            }

            ++it;
        }
    }

    tick = CShingleTime::GetMs() - tick;
    if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
        if (res)
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KMESSAGE, "STORHASH: read dump other collections - OK (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
        else
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: read dump other collections - FAILED (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
    }

    collection_count = IncMax32(collection_count, collection_count_local);
    record_count = IncMax32(record_count, record_count_local);

    return res;
}

bool TDataBasaHash::ListDumpOtherCollections(const TString& path, ui32& collection_count, ui32& record_count) {
    bool res = false;
    TString collection_name = "";
    TDaysHashIt dit;
    TShingleDataItemHashIt sit;
    TString filename = "";
    TString corr_path = path;
    FILE* handle = NULL;
    ui32 tick = CShingleTime::GetMs();
    ui32 collection_count_local = 0;
    ui32 record_count_local = 0;

    if (m_data != NULL) {
        res = true;

        dit = m_data->begin();
        while (dit != m_data->end()) {
            collection_name = dit->first;
            if (!collection_name.empty()) {
                if (path.length() > 0) {
                    filename = CorrectPath(path) + collection_name + ".lst";

                    handle = fopen(filename.c_str(), "wb");
                    if (handle != NULL) {
                        collection_count_local = IncMax32(collection_count_local, 1);

                        sit = dit->second.begin();
                        while (sit != dit->second.end()) {
                            record_count_local = IncMax32(record_count_local, 1);
                            fprintf(handle, "%s\n", (*sit).second.toList((*sit).first).c_str());

                            ++sit;
                        }

                        fclose(handle);
                        handle = NULL;

                    } else {
                        if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL))
                            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: error list dump other collections, can't create file %s", filename.c_str());
                    }
                }
            }

            ++dit;
        }
    }

    tick = CShingleTime::GetMs() - tick;
    if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
        if (res)
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KMESSAGE, "STORHASH: list dump other collections - OK (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
        else
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: list dump other collections - FAILED (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
    }

    collection_count = IncMax32(collection_count, collection_count_local);
    record_count = IncMax32(record_count, record_count_local);

    return res;
}

bool TDataBasaHash::WriteDumpLongIP(const TString& path, ui32& collection_count, ui32& record_count) {
    bool res = false;
    TString collection_name = "";
    TDaysHashLIIt dit;
    TLongIPDataItemHashIt sit;
    TString filename = "";
    FILE* handle = NULL;
    ui32 tick = CShingleTime::GetMs();
    ui32 collection_count_local = 0;
    ui32 record_count_local = 0;

    if (m_longip_data != NULL) {
        res = true;

        dit = m_longip_data->begin();
        while (dit != m_longip_data->end()) {
            collection_name = dit->first;
            if (!collection_name.empty()) {
                if (path.length() > 0) {
                    filename = CorrectPath(path) + collection_name + ".dli";

                    handle = fopen(filename.c_str(), "wb");
                    if (handle != NULL) {
                        collection_count_local = IncMax32(collection_count_local, 1);

                        sit = dit->second.begin();
                        while (sit != dit->second.end()) {
                            record_count_local = IncMax32(record_count_local, 1);
                            fprintf(handle, "%s\n", (*sit).second.writeToDump((*sit).first).c_str());

                            ++sit;
                        }

                        fclose(handle);
                        handle = NULL;

                    } else {
                        if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                            LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: error write dump longip, can't create file %s", filename.c_str());
                    }
                }
            }

            ++dit;
        }
    }

    tick = CShingleTime::GetMs() - tick;
    if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
        if (res)
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KMESSAGE, "STORHASH: write dump longip - OK (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
        else
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: write dump longip - FAILED (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
    }

    collection_count = IncMax32(collection_count, collection_count_local);
    record_count = IncMax32(record_count, record_count_local);

    return res;
}

bool TDataBasaHash::ReadDumpLongIP(const TString& path, ui32& collection_count, ui32& record_count) {
    bool res = true;
    std::list<TString> filelist;
    TString path_s = CorrectPath(path);
    TString filename_wo_path = "";
    ui32 tick = CShingleTime::GetMs();
    ui32 collection_count_local = 0;
    ui32 record_count_local = 0;

#ifdef _win_
    FindFiles ff;
    TString mask = path_s + "*.dli";

    if (ff.FindFirst((char*)mask.c_str()) == CERR_OK) {
        do {
            filename_wo_path = TString(ff.ff.Name);
            if ((filename_wo_path != ".") && (filename_wo_path != ".."))
                filelist.push_back(filename_wo_path);

        } while (ff.FindNext() == CERR_OK);
    }
#else
    DIR* dir = NULL;
    if ((dir = opendir(path_s.c_str())) == NULL) {
        printf("Can\'t open folder: '%s'\n", path_s.c_str());
    } else {
        struct dirent* f_cur;
        while ((f_cur = readdir(dir)) != NULL) {
            filename_wo_path = TString(f_cur->d_name);
            if ((filename_wo_path != ".") && (filename_wo_path != ".."))
                filelist.push_back(filename_wo_path);
        }
        closedir(dir);
    }
#endif

    if (filelist.size() > 0) {
        FILE* handle = NULL;
        std::list<TString>::iterator it;
        char tbuff[2048];
        TString filename = "";
        TString collection_name = "";
        TDaysHashLIIt dit;
        TLongIPDataItemHashIt sit;
        stordata::TLongIPDataItem var_data;
        ui64 shingle = 0;
        const char* p = NULL;
        int cnt = 0;

        it = filelist.begin();
        while (it != filelist.end()) {
            if (!it->empty()) {
                p = strstr(it->c_str(), ".dli");
                if (p != NULL) {
                    cnt = p - it->c_str();
                    if (cnt > 0)
                        collection_name = TString(it->c_str(), cnt);

                } else
                    collection_name = *it;

                collection_count_local = IncMax32(collection_count_local, 1);
                filename = path_s + (*it);
                handle = fopen(filename.c_str(), "rb");
                if (handle != NULL) {
                    memset(tbuff, 0, sizeof(tbuff));
                    while (fgets(tbuff, sizeof(tbuff) - 1, handle)) {
                        var_data.Clear();
                        shingle = 0;
                        if (var_data.LoadData(TString(tbuff), shingle)) {
                            record_count_local = IncMax32(record_count_local, 1);

                            dit = m_longip_data->find(collection_name);
                            if (dit == m_longip_data->end()) {
                                (*m_longip_data)[collection_name] = TLongIPDataItemHash();
                                dit = m_longip_data->find(collection_name);
                            }

                            if (dit != m_longip_data->end()) {
                                sit = dit->second.find(shingle);
                                if (sit == dit->second.end()) {
                                    dit->second[shingle] = stordata::TLongIPDataItem();
                                    sit = dit->second.find(shingle);
                                }

                                if (sit != dit->second.end())
                                    sit->second = var_data;
                            }
                        }
                    }

                    fclose(handle);
                    handle = NULL;

                } else {
                    res = false;
                    if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL))
                        LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: error read dump longip, can't open file %s", filename.c_str());
                }
            }

            ++it;
        }
    }

    tick = CShingleTime::GetMs() - tick;
    if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
        if (res)
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KMESSAGE, "STORHASH: read dump longip - OK (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
        else
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: read dump lomgip - FAILED (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
    }

    collection_count = IncMax32(collection_count, collection_count_local);
    record_count = IncMax32(record_count, record_count_local);

    return res;
}

bool TDataBasaHash::ListDumpLongIP(const TString& path, ui32& collection_count, ui32& record_count) {
    bool res = false;
    TString collection_name = "";
    TDaysHashLIIt dit;
    TLongIPDataItemHashIt sit;
    TString filename = "";
    TString corr_path = path;
    FILE* handle = NULL;
    ui32 tick = CShingleTime::GetMs();
    ui32 collection_count_local = 0;
    ui32 record_count_local = 0;

    if (m_longip_data != NULL) {
        res = true;

        dit = m_longip_data->begin();
        while (dit != m_longip_data->end()) {
            collection_name = dit->first;
            if (!collection_name.empty()) {
                if (path.length() > 0) {
                    filename = CorrectPath(path) + collection_name + ".lst";

                    handle = fopen(filename.c_str(), "wb");
                    if (handle != NULL) {
                        collection_count_local = IncMax32(collection_count_local, 1);

                        sit = dit->second.begin();
                        while (sit != dit->second.end()) {
                            record_count_local = IncMax32(record_count_local, 1);
                            fprintf(handle, "%s\n", (*sit).second.toList((*sit).first).c_str());

                            ++sit;
                        }

                        fclose(handle);
                        handle = NULL;

                    } else {
                        if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL))
                            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: error list dump longip, can't create file %s", filename.c_str());
                    }
                }
            }

            ++dit;
        }
    }

    tick = CShingleTime::GetMs() - tick;
    if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
        if (res)
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KMESSAGE, "STORHASH: list dump longip - OK (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
        else
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "STORHASH: list dump longip - FAILED (%u msec, %u collections, %u records)", tick, collection_count_local, record_count_local);
    }

    collection_count = IncMax32(collection_count, collection_count_local);
    record_count = IncMax32(record_count, record_count_local);

    return res;
}

bool TDataBasaHash::WriteDump(const TString& path, ui32& collection_count, ui32& record_count) {
    bool res = true;

    collection_count = 0;
    record_count = 0;

    if (res) {
        if (!WriteDumpOtherCollections(path, collection_count, record_count))
            res = false;
    }

    if (res) {
        if (!WriteDumpLongIP(path, collection_count, record_count))
            res = false;
    }

    return res;
}

bool TDataBasaHash::ReadDump(const TString& path, ui32& collection_count, ui32& record_count) {
    bool res = true;

    collection_count = 0;
    record_count = 0;

    if (res) {
        if (!ReadDumpOtherCollections(path, collection_count, record_count))
            res = false;
    }

    if (res) {
        if (!ReadDumpLongIP(path, collection_count, record_count))
            res = false;
    }

    return res;
}

bool TDataBasaHash::ListDump(const TString& path, ui32& collection_count, ui32& record_count) {
    bool res = true;

    collection_count = 0;
    record_count = 0;

    if (res) {
        if (!ListDumpOtherCollections(path, collection_count, record_count))
            res = false;
    }

    if (res) {
        if (!ListDumpLongIP(path, collection_count, record_count))
            res = false;
    }

    return res;
}
