#pragma once

#include "util/system/mutex.h"
#include "util/system/rwlock.h"
#include "util/generic/hash.h"
#include <mail/so/spamstop/tools/so-common/kfunc.h>
#include <mail/so/spamstop/tools/so-common/tlogclass.h>
#include <mail/so/spamstop/tools/so-common/tkipv6.h>
#include "tstorebase.h"
#include "tstorebase_data.h"

//*********************************************************************************************************************************
//                                                 THeavyStat
//*********************************************************************************************************************************

#define K_PERIOD_CALC_RPS_SEC 30

typedef THashMap<TKIPv6, ui8> TKIPHash;

typedef THashMap<ui64, ui8> TKSuidHash;
typedef TKSuidHash::iterator TKSuidHashIt;

struct THeavyStatStructItem {
    ui32 m_ham{};
    ui32 m_spam{};
    TKIPHash m_iphash;
    TKSuidHash m_suidhash;

    THeavyStatStructItem() = default;

    void Clear() {
        m_ham = 0;
        m_spam = 0;
        m_iphash.clear();
        m_suidhash.clear();
    }

    bool IsEmpty() {
        bool res = false;

        if ((m_ham == 0) && (m_spam == 0))
            res = true;

        return res;
    }
};

struct THeavyStatStructItemLittle {
    ui32 m_ham;
    ui32 m_spam;
    ui32 m_ip_uniq_count;
    ui32 m_suid_uniq_count;

    THeavyStatStructItemLittle(THeavyStatStructItem& value) {
        Clear();
        m_ham = value.m_ham;
        m_spam = value.m_spam;
        m_ip_uniq_count = value.m_iphash.size();
        m_suid_uniq_count = value.m_suidhash.size();
    }

    THeavyStatStructItemLittle() {
        Clear();
    }

    void Clear() {
        m_ham = 0;
        m_spam = 0;
        m_ip_uniq_count = 0;
        m_suid_uniq_count = 0;
    }

    bool IsEmpty() {
        bool res = false;

        if ((m_ham == 0) && (m_spam == 0))
            res = true;

        return res;
    }

    ui32 All() const {
        ui32 res = 0;

        res = IncMax32(m_ham, m_spam);

        return res;
    }
};

struct THeavyStatStruct {
    ui32 m_firsttime;
    ui32 m_lasttime;
    THeavyStatStructItem m_today;
    THeavyStatStructItem m_yesterday;
    TString m_text;
    ui32 m_last_calc_rps;
    ui32 m_rps_counter;
    float m_rps_value;
    TString m_geo;

    THeavyStatStruct() {
        Clear();
    }

    THeavyStatStruct(ui32 ttime, TCountersStatEx currentcounter, const TString& text, TKIPv6 ip, const TString& suid, const TString& geo) {
        Clear();
        m_geo = geo;
        m_firsttime = ttime;
        m_lasttime = ttime;

        m_today.m_spam = currentcounter.today_spam;
        m_today.m_ham = currentcounter.today_ham;
        m_yesterday.m_spam = currentcounter.yesterday_spam;
        m_yesterday.m_ham = currentcounter.yesterday_ham;

        m_text = text;

        if (!ip.Undefined())
            m_today.m_iphash.emplace(ip, 0);

        ui64 suid_shingle = ShingleFromStroka(suid);
        if (suid_shingle != 0)
            m_today.m_suidhash[suid_shingle] = 0;

        m_last_calc_rps = time(NULL);
        m_rps_counter = 1;
        m_rps_value = 0;
    }

    void Clear() {
        m_today.Clear();
        m_yesterday.Clear();
        m_last_calc_rps = 0;
        m_rps_counter = 0;
        m_rps_value = 0;
        m_geo = "";
    }

    ui32 Ham() const {
        ui32 res = 0;

        res = IncMax32(m_today.m_ham, m_yesterday.m_ham);

        return res;
    }

    ui32 Spam() const {
        ui32 res = 0;

        res = IncMax32(m_today.m_spam, m_yesterday.m_spam);

        return res;
    }

    void Midnight() {
        m_yesterday.Clear();
        m_yesterday = m_today;
        m_today.Clear();
    }

    bool IsEmpty() {
        return (m_today.IsEmpty() && m_yesterday.IsEmpty());
    }

    void CalcRPS(ui32 current_time, bool only_calc) {
        ui32 diff_time = 0;

        diff_time = current_time - m_last_calc_rps;
        if (diff_time >= (2 * K_PERIOD_CALC_RPS_SEC)) {
            m_rps_value = 0;
            m_rps_counter = (!only_calc) ? 1 : 0;
            m_last_calc_rps = current_time;

        } else if ((diff_time >= K_PERIOD_CALC_RPS_SEC) && (diff_time < (K_PERIOD_CALC_RPS_SEC * 2))) {
            m_rps_value = (float)m_rps_counter / (float)diff_time;
            m_rps_counter = (!only_calc) ? 1 : 0;
            m_last_calc_rps = current_time;

        } else {
            if (!only_calc)
                m_rps_counter = IncMax32(m_rps_counter, 1);
        }
    }
};

typedef THashMap<ui64, THeavyStatStruct> THeavyStatStructHash;
typedef THeavyStatStructHash::iterator THeavyStatStructHashIt;

struct THeavyStatStructLittle {
    ui32 m_firsttime;
    ui32 m_lasttime;
    THeavyStatStructItemLittle m_today;
    THeavyStatStructItemLittle m_yesterday;
    TString m_text;
    float m_rps;
    ui16 m_diff_calc_rps;
    TString m_geo;

    THeavyStatStructLittle() {
        Clear();
    }

    THeavyStatStructLittle(THeavyStatStruct& value) {
        time_t current_time = time(NULL);

        Clear();
        m_firsttime = value.m_firsttime;
        m_lasttime = value.m_lasttime;
        m_today = THeavyStatStructItemLittle(value.m_today);
        m_yesterday = THeavyStatStructItemLittle(value.m_yesterday);
        m_text = value.m_text;
        m_rps = value.m_rps_value;
        m_diff_calc_rps = current_time - value.m_last_calc_rps;
        m_geo = value.m_geo;
    }

    void Clear() {
        m_firsttime = 0;
        m_lasttime = 0;
        m_today.Clear();
        m_yesterday.Clear();
        m_text = "";
        m_rps = 0;
        m_diff_calc_rps = 0;
        m_geo = "";
    }

    ui32 Ham() const {
        ui32 res = 0;

        res = IncMax32(m_today.m_ham, m_yesterday.m_ham);

        return res;
    }

    ui32 Spam() const {
        ui32 res = 0;

        res = IncMax32(m_today.m_spam, m_yesterday.m_spam);

        return res;
    }
};

struct THeavyStatDisplay {
    enum TSortType { SRT_FIRSTTIME,
                     SRT_LASTTIME,
                     SRT_HAM,
                     SRT_SPAM,
                     SRT_TEXT,
                     SRT_HAM_TODAY,
                     SRT_SPAMTODAY,
                     SRT_ALL_TODAY,
                     SRT_ALL_YESTERDAY,
                     SRT_RPS };

    ui64 shingle;
    THeavyStatStructLittle data;
    TSortType sort_type;

    THeavyStatDisplay() {
        shingle = 0;
        data.Clear();
        sort_type = SRT_SPAM;
    }

    THeavyStatDisplay(ui64 shingleA, THeavyStatStruct& dataA, TSortType sort_typeA) {
        shingle = shingleA;
        data = THeavyStatStructLittle(dataA);
        sort_type = sort_typeA;
    }

    bool operator<(const THeavyStatDisplay& value) const {
        bool res = false;

        switch (sort_type) {
            case SRT_FIRSTTIME:
                res = data.m_firsttime < value.data.m_firsttime;
                break;
            case SRT_LASTTIME:
                res = data.m_lasttime > value.data.m_lasttime;
                break;
            case SRT_HAM:
                res = data.Ham() > value.data.Ham();
                break;
            case SRT_SPAM:
                res = data.Spam() > value.data.Spam();
                break;
            case SRT_TEXT:
                res = data.m_text < value.data.m_text;
                break;
            case SRT_HAM_TODAY:
                res = data.m_today.m_ham > value.data.m_today.m_ham;
                break;
            case SRT_SPAMTODAY:
                res = data.m_today.m_spam > value.data.m_today.m_spam;
                break;
            case SRT_ALL_TODAY:
                res = data.m_today.All() > value.data.m_today.All();
                break;
            case SRT_ALL_YESTERDAY:
                res = data.m_yesterday.All() > value.data.m_yesterday.All();
                break;
            case SRT_RPS:
                res = data.m_rps > value.data.m_rps;
                break;
        };

        return res;
    }
};

typedef std::list<THeavyStatDisplay> THeavyStatDisplayList;
typedef THeavyStatDisplayList::iterator THeavyStatDisplayListIt;

//*********************************************************************************************************************************
//                                                    THeavyStatItem
//*********************************************************************************************************************************

class THeavyStatItem {
private:
    static const int MAX_IPHASH_SIZE = 1000;
    static const int MAX_SUIDHASH_SIZE = 1000;

private:
    TString m_name;
    TLogClass* m_log;
    THeavyStatStructHash* data;
    TMutex m_Mutex;

    void Lock();
    void UnLock();

public:
    THeavyStatItem();
    ~THeavyStatItem();

    void Init(const TString& name, TLogClass* log);
    bool AddData(ui64 shingle, const TString& text, TCountersStatEx currentcounter, TKIPv6 ip, const TString& suid, const TString& geo);
    void Midnight();
    ui32 GetReportData(THeavyStatDisplayList& datalist, ui32 min_treshold, THeavyStatDisplay::TSortType sort_type);
    ui32 Count() {
        ui32 res = 0;

        Lock();

        if (data != NULL)
            res = data->size();

        UnLock();

        return res;
    }
};

//*********************************************************************************************************************************
//                                                    THeavyStat
//*********************************************************************************************************************************

class THeavyStat {
private:
    static const int DEFAULT_ADDTRESHOLD = 1000;
    static const ui32 DISPLAY_COUNT_DEFAULT = 1000;
    static const int ELEMENT_COUNT = 32;

private:
    THeavyStatItem arraydata[ELEMENT_COUNT];
    ui32 addtreshold;
    TString name;

    int GetIndex(ui64 shingle);
    TString CreateLink(const TString& label, const TString& prefix_url, ui32 max_display_record, ui32 min_treshold, THeavyStatDisplay::TSortType sort_type, const TString& text);

public:
    THeavyStat();
    ~THeavyStat();

    void Init(const TString& nameA, TLogClass* log, i32 addtresholdA);
    bool AddData(TCountersStatEx currentcounter, const TString& text, bool spam, TKIPv6 ip, const TString& suid, const TString& geo);
    void Midnight();
    TString GetReport(const TString& label, const TString& prefix_url, ui32 max_display_record, ui32 min_treshold, THeavyStatDisplay::TSortType sort_type, const TString& textlink);
    ui32 Count();
};

//*********************************************************************************************************************************
