#pragma once

#include "util/generic/hash.h"
#include <util/generic/string.h>
#include "util/system/mutex.h"
#include <mail/so/spamstop/tools/so-clients/ktime.h>
#include "tlogsgroup.h"
#include <mail/so/spamstop/tools/so-common/kfunc.h>
#include <mail/so/spamstop/tools/so-clients/shtime.h>
#include <list>

//**********************************************************************************
//                            TKKey - samples TKey
//**********************************************************************************

class TKKey {
private:
    ui32 m_key;

public:
    TKKey() {
        m_key = 0;
    }
    TKKey(ui32 key) {
        m_key = key;
    }
    ~TKKey(){};

    TString DataToText() {
        return IntToStroka(m_key);
    }
    bool operator==(const TKKey& value) const {
        return m_key == value.m_key;
    }
    operator size_t() const {
        return sizeof(m_key);
    }
    ui32 ident() {
        return m_key;
    }
    ui32 key() const {
        return m_key;
    }
};

//***********************************************************************************
//                           TKValue - samples TValue
//***********************************************************************************

class TKValue //  TValue (   -  )  TValueTr (  )
{
private:
    kday_t m_last_day;
    ui16 m_value;

public:
    TKValue() {
        m_last_day = (unsigned long)0;
        m_value = 0;
    }
    TKValue(const TKValue& value) {
        m_last_day = value.m_last_day;
        m_value = value.m_value;
    }
    TKValue(kday_t last_day, ui16 value) {
        m_last_day = last_day;
        m_value = value;
    }
    ~TKValue(){};

    operator kday_t() const {
        return m_last_day;
    }
    TKValue& operator+=(const TKValue& value) {
        m_last_day = value.m_last_day;
        m_value = ((m_value + value.m_value) > 0xFFFF) ? 0xFFFF : (m_value + value.m_value);
        return *this;
    }
    TString DataToText() {
        return IntToStroka((ui32)m_last_day) + "-" + IntToStroka(m_value);
    }
    kday_t last_day() const {
        return m_last_day;
    }
    ui16 value() const {
        return m_value;
    }
};

//***********************************************************************************
//                                  TKQueue
//***********************************************************************************

template <class TKey, class TValue>
class TKKeyValue {
public:
    TKey key;
    TValue value;

    TKKeyValue(TKey keyA, TValue valueA) {
        key = keyA;
        value = valueA;
    }
};

template <class TKey, class TValue>
class TKQueue {
private:
    ui32 m_QueueCapacity;
    ui32 m_lost_today;
    ui32 m_lost_all;
    TMutex m_Mutex;
    std::list<TKKeyValue<TKey, TValue>> data;

    void Lock();
    void UnLock();

public:
    TKQueue();
    ~TKQueue();

    void Init(ui32 QueueCapacity);
    bool Add(TKey key, TValue value);
    bool Get(TKey& key, TValue& value);
    bool GetWithoutErase(TKey& key, TValue& value);
    bool EraseFirst();
    ui32 Count();     //  
    ui32 LostToday(); // 
    ui32 LostAll();   // 
    void Midnight();
    TString GetStat();
};

template <class TKey, class TValue>
TKQueue<TKey, TValue>::TKQueue() {
    m_QueueCapacity = 1000;
    m_lost_today = 0;
    m_lost_all = 0;
}

template <class TKey, class TValue>
TKQueue<TKey, TValue>::~TKQueue() {
    data.clear();
}

template <class TKey, class TValue>
void TKQueue<TKey, TValue>::Lock() {
    m_Mutex.Acquire();
}

template <class TKey, class TValue>
void TKQueue<TKey, TValue>::UnLock() {
    m_Mutex.Release();
}

template <class TKey, class TValue>
void TKQueue<TKey, TValue>::Init(ui32 QueueCapacity) {
    m_QueueCapacity = QueueCapacity;
    data.clear();
}

template <class TKey, class TValue>
bool TKQueue<TKey, TValue>::Add(TKey key, TValue value) {
    bool res = false;

    Lock();

    if (data.size() < m_QueueCapacity) {
        data.push_back(TKKeyValue<TKey, TValue>(key, value));
        res = true;
    } else {
        m_lost_today = (m_lost_today < 0xFFFFFFFF) ? (m_lost_today + 1) : 0xFFFFFFFF;
        m_lost_all = (m_lost_all < 0xFFFFFFFF) ? (m_lost_all + 1) : 0xFFFFFFFF;
    }

    UnLock();

    return res;
}

template <class TKey, class TValue>
bool TKQueue<TKey, TValue>::Get(TKey& key, TValue& value) {
    bool res = false;
    typename std::list<TKKeyValue<TKey, TValue>>::iterator it;

    Lock();

    it = data.begin();
    if (it != data.end()) {
        key = (*it).key;
        value = (*it).value;
        data.erase(it);
        res = true;
    }

    UnLock();

    return res;
}

template <class TKey, class TValue>
bool TKQueue<TKey, TValue>::GetWithoutErase(TKey& key, TValue& value) {
    bool res = false;
    typename std::list<TKKeyValue<TKey, TValue>>::iterator it;

    Lock();

    it = data.begin();
    if (it != data.end()) {
        key = (*it).key;
        value = (*it).value;
        res = true;
    }

    UnLock();

    return res;
}

template <class TKey, class TValue>
bool TKQueue<TKey, TValue>::EraseFirst() {
    bool res = false;
    typename std::list<TKKeyValue<TKey, TValue>>::iterator it;

    Lock();

    it = data.begin();
    if (it != data.end()) {
        data.erase(it);
        res = true;
    }

    UnLock();

    return res;
}

template <class TKey, class TValue>
ui32 TKQueue<TKey, TValue>::Count() {
    return data.size();
}

template <class TKey, class TValue>
ui32 TKQueue<TKey, TValue>::LostToday() {
    return m_lost_today;
}

template <class TKey, class TValue>
ui32 TKQueue<TKey, TValue>::LostAll() {
    return m_lost_all;
}

template <class TKey, class TValue>
void TKQueue<TKey, TValue>::Midnight() {
    Lock();

    m_lost_today = 0;

    UnLock();
}

template <class TKey, class TValue>
TString TKQueue<TKey, TValue>::GetStat() {
    TString res = "";

    res = IntToStroka(data.size()) + "(" + IntToStroka(m_lost_today) + ")";

    return res;
}

//***********************************************************************************
//                                TKBasaItem
//***********************************************************************************

template <class TKey, class TValue, class TValueTr>
class TKBasaItem {
public:
    typedef yhash<TKey, TValue*> TDataHash;

protected:
    TMutex m_Mutex;
    TMutex m_MutexBusy;
    TDataHash* data;
    ui32 number;
    TString objectname;
    TLogClass* LogsClass;
    bool busy;
    static const ui32 RECORDS_MAX_COUNT = 10000000;
    static const ui32 KREADWRITEDUMPBUFFSIZE = 64001;
    //static const ui32 BUFFER_SIZE_KEYVALUE = 256;     //        
    ui32 records_max_count;
    ui32 liveday;
    bool overflow;
    bool printoverflow;

    typename yhash<TKey, TValue*>::iterator curr_it;

    void Lock();
    void UnLock();
    void SetBusy(bool busyA);
    bool Erase_element(TValue* elm, kday_t currenttime);
    ui32 GetBuffElement(TKey key, TValue* elm, char* buff, ui32 buffsize);
    TValue* AddBuffElement(TKey& key, char* buff, ui32 buffsize, ui32& elsize);
    TString GetInnerData(TKey key, TValue* elm);

public:
    TKBasaItem();
    ~TKBasaItem();
    //
    void Init(ui32 numberA, TLogClass* LogClassA, ui32 records_max_countA, TString objectnameA, ui32 LiveDayA);
    ///
    bool Add(TKey key, TValueTr value);
    int Get(TKey key, TValue& value);
    bool AddHard(TKey key, TValue value);
    // 
    void CorrectIP(TKey key, TValueTr value);
    //  (  ,    " ")
    void CleanUp();
    void CheckAndCleanUP();
    //  (- )
    ui32 Count();
    //
    bool Overflow() {
        return overflow;
    }
    // " "   
    void SetLiveDay(ui32 LiveDayA);
    ui32 GetLiveDay();
    ///  / 
    bool WriteDump(TString filename);
    bool ReadDump(TString filename);
    //  (   )
    bool PrintDataToFile(TString filename);
    // (,  )
    bool Busy();
    //   
    bool First(TKey& key, TValue& value);
    //   
    bool Next(TKey& key, TValue& value);
};

template <class TKey, class TValue, class TValueTr>
TKBasaItem<TKey, TValue, TValueTr>::TKBasaItem() {
    number = 0;
    LogsClass = NULL;
    //data              = new TDataHash(1000000);
    data = new TDataHash;
    busy = false;
    records_max_count = RECORDS_MAX_COUNT;
    objectname = "";
    liveday = 180;
    overflow = false;
    printoverflow = false;
}

template <class TKey, class TValue, class TValueTr>
TKBasaItem<TKey, TValue, TValueTr>::~TKBasaItem() {
    typename TDataHash::iterator it;

    if (data != NULL) {
        it = data->begin();
        while (it != data->end()) {
            if ((*it).second != NULL) {
                delete (*it).second;
                (*it).second = NULL;
            }
            ++it;
        }
        data->clear();
        delete data;
        data = NULL;
    }
}

template <class TKey, class TValue, class TValueTr>
void TKBasaItem<TKey, TValue, TValueTr>::Lock() {
    m_Mutex.Acquire();
}

template <class TKey, class TValue, class TValueTr>
void TKBasaItem<TKey, TValue, TValueTr>::UnLock() {
    m_Mutex.Release();
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaItem<TKey, TValue, TValueTr>::Busy() {
    return busy;
}

template <class TKey, class TValue, class TValueTr>
void TKBasaItem<TKey, TValue, TValueTr>::SetBusy(bool busyA) {
    m_MutexBusy.Acquire();

    busy = busyA;

    m_MutexBusy.Release();
}

template <class TKey, class TValue, class TValueTr>
void TKBasaItem<TKey, TValue, TValueTr>::Init(ui32 numberA, TLogClass* LogClassA, ui32 records_max_countA, TString objectnameA, ui32 LiveDayA) {
    number = numberA;
    LogsClass = LogClassA;
    records_max_count = records_max_countA;
    objectname = objectnameA;
    liveday = LiveDayA;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaItem<TKey, TValue, TValueTr>::Add(TKey key, TValueTr value) {
    bool res = false;
    typename TDataHash::iterator it;

    if (!Busy()) {
        Lock();

        if (data != NULL) {
            it = data->find(key);
            if (it != data->end()) {
                if ((*it).second != NULL)
                    (*(*it).second) += value;
            } else {
                TValue* pvalue = new TValue(value);
                (*data)[key] = pvalue;
            }
            res = true;
        }

        UnLock();
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
int TKBasaItem<TKey, TValue, TValueTr>::Get(TKey key, TValue& value) {
    int res = 0;
    // :
    // 0 -  
    // 1 - ,   
    // 2 - 
    typename TDataHash::iterator it;

    if (!Busy()) {
        Lock();

        if (data != NULL) {
            it = data->find(key);
            if (it != data->end()) {
                if ((*it).second != NULL) {
                    value = (*(*it).second);
                    res = 2;
                }
            }
        }

        UnLock();
    } else {
        res = 1;
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaItem<TKey, TValue, TValueTr>::AddHard(TKey key, TValue value) {
    bool res = false;
    typename TDataHash::iterator it;

    if (!Busy()) {
        Lock();

        if (data != NULL) {
            it = data->find(key);
            if (it != data->end()) {
                res = false;
            } else {
                TValue* pvalue = new TValue(value);
                (*data)[key] = pvalue;
                res = true;
            }
        }

        UnLock();
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
void TKBasaItem<TKey, TValue, TValueTr>::CorrectIP(TKey key, TValueTr value) {
    typename TDataHash::iterator it;

    if (!Busy()) {
        Lock();

        if (data != NULL) {
            it = data->find(key);
            if (it != data->end()) {
                if ((*it).second != NULL)
                    (*(*it).second) += value;
            }
        }

        UnLock();
    }
}

template <class TKey, class TValue, class TValueTr>
ui32 TKBasaItem<TKey, TValue, TValueTr>::Count() {
    ui32 res = 0;

    if (data != NULL)
        res = data->size();

    return res;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaItem<TKey, TValue, TValueTr>::Erase_element(TValue* elm, kday_t currenttime) {
    bool res = false;

    //if ((currenttime - *elm) > liveday)
    //   res = true;

    return res;
}

template <class TKey, class TValue, class TValueTr>
void TKBasaItem<TKey, TValue, TValueTr>::CleanUp() {
    ui32 erasecount = 0, remaincount = 0;
    ui32 tick = CShingleTime::GetMs();

    if (!Busy()) {
        SetBusy(true);

        if (data != NULL) {
            typename TDataHash::iterator it;
            TDataHash* datanew = NULL;
            kday_t current_time = kday_t(time(NULL));

            Lock();

            datanew = new TDataHash();
            if (datanew != NULL) {
                it = data->begin();
                while (it != data->end()) {
                    if (!Erase_element((*it).second, current_time)) {
                        (*datanew)[(*it).first] = (*it).second;
                        remaincount++;
                    } else {
                        if ((*it).second != NULL) {
                            delete (*it).second;
                            (*it).second = NULL;
                        }
                        erasecount++;
                    }

                    ++it;
                }
            }
            data->clear();
            delete data;
            data = datanew;

            if (data->size() < records_max_count)
                overflow = false;
            else
                overflow = true;

            printoverflow = false;

            UnLock();
        }

        SetBusy(false);
    }
    tick = CShingleTime::GetMs() - tick;
    if (LogsClass != NULL)
        LogsClass->WriteMessageAndDataStatus(KMESSAGE, "Cleanup (%s N%03u) compleate in %u msec (erase %u, remain %u)", objectname.c_str(), number, tick, erasecount, remaincount);
}

template <class TKey, class TValue, class TValueTr>
void TKBasaItem<TKey, TValue, TValueTr>::CheckAndCleanUP() {
    if ((data != NULL) && ((overflow) || (data->size() >= records_max_count)))
        CleanUp();
}

template <class TKey, class TValue, class TValueTr>
void TKBasaItem<TKey, TValue, TValueTr>::SetLiveDay(ui32 LiveDayA) {
    Lock();

    liveday = LiveDayA;

    UnLock();
}

template <class TKey, class TValue, class TValueTr>
ui32 TKBasaItem<TKey, TValue, TValueTr>::GetLiveDay() {
    return liveday;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaItem<TKey, TValue, TValueTr>::WriteDump(TString filename) {
    bool res = true;
    char* buff = NULL;
    ui32 buffsize = 0;
    ui32 inbuff = 0;
    TString newfilename = "";
    TString newfilename_tmp = "";
    FILE* handle = NULL;
    typename TDataHash::iterator it;
    ui32 esize = 0;
    const ui32 recsize = sizeof(TKey) + sizeof(TValue) + 1; //BUFFER_SIZE_KEYVALUE;
    char tbuff[recsize];
    ui32 bsize = 0;
    ui32 crc32 = 0;
    TString blockident = "DBID";
    ui32 shapsize = 0;
    unsigned char uc = 0;
    ui32 writecount = 0;

    Lock();

    if (data != NULL) {
        newfilename_tmp = filename + "." + IntToStroka3(number) + "_tmp";
        newfilename = filename + "." + IntToStroka3(number);
        handle = fopen(newfilename_tmp.c_str(), "w+b");
        if (handle != NULL) {
            buffsize = KREADWRITEDUMPBUFFSIZE;
            buff = new char[buffsize];
            try {
                shapsize = blockident.size() + sizeof(bsize) + sizeof(crc32);
                inbuff = shapsize;
                it = data->begin();
                while (it != data->end()) {
                    esize = GetBuffElement((*it).first, (*it).second, tbuff, sizeof(tbuff));
                    if (esize > 0) {
                        writecount++;
                        if ((esize + inbuff) >= buffsize) {
                            crc32 = 0;
                            for (int i = shapsize; i < inbuff; i++) {
                                memcpy(&uc, buff + i, sizeof(uc));
                                crc32 += uc;
                            }
                            memcpy(buff, blockident.c_str(), blockident.size());
                            memcpy(buff + blockident.size(), &inbuff, sizeof(inbuff));
                            memcpy(buff + blockident.size() + sizeof(inbuff), &crc32, sizeof(crc32));
                            fwrite(buff, 1, inbuff, handle);
                            inbuff = shapsize;
                        }
                        memcpy(buff + inbuff, tbuff, esize);
                        inbuff += esize;
                    }

                    ++it;
                }
                if (inbuff > 0) {
                    crc32 = 0;
                    for (int i = shapsize; i < inbuff; i++) {
                        memcpy(&uc, buff + i, sizeof(uc));
                        crc32 += uc;
                    }
                    memcpy(buff, blockident.c_str(), blockident.size());
                    memcpy(buff + blockident.size(), &inbuff, sizeof(inbuff));
                    memcpy(buff + blockident.size() + sizeof(inbuff), &crc32, sizeof(crc32));
                    fwrite(buff, 1, inbuff, handle);
                }

                delete[] buff;
                buff = NULL;
            } catch (...) {
                delete[] buff;
                buff = NULL;
            }
            fclose(handle);
        } else {
            res = false;
        }

        if (res) {
            remove(newfilename.c_str());
            if (rename(newfilename_tmp.c_str(), newfilename.c_str()))
                res = false;
        }

        if (res) {
            if (LogsClass != NULL)
                LogsClass->WriteMessageAndDataStatus(KMESSAGE, "Write dump (%s N%u) successfully, count = %u", objectname.c_str(), number, writecount);
            printf("Write dump (%s N%u) successfully, count = %u\n", objectname.c_str(), number, writecount);
        } else {
            if (LogsClass != NULL)
                LogsClass->WriteMessageAndDataStatus(KERROR, "Write dump (%s N%u) failed, file: %s", objectname.c_str(), number, newfilename.c_str());
            printf("Write dump (%s N%u) failed, file: %s\n", objectname.c_str(), number, newfilename.c_str());
        }
    }

    UnLock();

    return res;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaItem<TKey, TValue, TValueTr>::ReadDump(TString filename) {
    bool res = true;
    TString blockident = "DBID";
    FILE* handle = NULL;
    TString newfilename = "";
    char* buff = NULL;
    ui32 buffsize = 0;
    ui32 inbuff = 0;
    char* bufftemp = NULL;
    ui32 inbufftemp = 0;
    ui32 readbyte = 0;
    ui32 needread = 0;
    ui32 blocksize = 0;
    ui32 crc32 = 0;
    ui32 tcrc32 = 0;
    unsigned char uc = 0;
    ui32 shapsize = 0;
    ui32 readcount = 0;
    ui32 recsize = sizeof(TKey) + sizeof(TValue) + 1;
    ui32 timedelay = 0;
    ui32 td = 0;

    Lock();

    printf("Load %s N%u... ", objectname.c_str(), number);

    if (data != NULL) {
        newfilename = filename + "." + IntToStroka3(number);
        handle = fopen(newfilename.c_str(), "rb");
        if (handle != NULL) {
            buffsize = KREADWRITEDUMPBUFFSIZE;
            if (buffsize < recsize)
                buffsize = recsize;
            buff = new char[buffsize];
            bufftemp = new char[buffsize];
            try {
                inbuff = 0;
                int n = 0;
                while (true) {
                    n++;
                    if (n > 250)
                        n = n;

                    needread = buffsize - inbuff;
                    readbyte = fread(buff + inbuff, 1, needread, handle);
                    if ((readbyte > 0) || (needread == 0)) {
                        inbuff += readbyte;
                        if (memcmp(buff, blockident.c_str(), blockident.size()) == 0) {
                            memcpy(&blocksize, buff + blockident.size(), sizeof(blocksize));
                            if (inbuff >= blocksize) {
                                memcpy(&crc32, buff + blockident.size() + sizeof(blocksize), sizeof(crc32));
                                shapsize = blockident.size() + sizeof(blocksize) + sizeof(crc32);
                                tcrc32 = 0;
                                for (int i = shapsize; i < blocksize; i++) {
                                    memcpy(&uc, buff + i, sizeof(uc));
                                    tcrc32 += uc;
                                }
                                if (crc32 == tcrc32) {
                                    TKey key;
                                    TValue* mr = NULL;
                                    ui32 posiz = 0;
                                    ui32 tsize = 0;
                                    ui32 elsize = 0;

                                    posiz = 0;
                                    while (true) {
                                        tsize = blocksize - shapsize - posiz;
                                        if (tsize == 0)
                                            break;
                                        mr = AddBuffElement(key, buff + shapsize + posiz, tsize, elsize);
                                        if ((elsize > 0) && (mr != NULL)) {
                                            td = CShingleTime::GetMs();
                                            (*data)[key] = mr;
                                            //typename TDataHash::value_type vt(key, mr);
                                            //data->insert(vt);
                                            td = CShingleTime::GetMs() - td;
                                            timedelay += td;

                                            posiz += elsize;
                                            readcount++;
                                        } else {
                                            break;
                                        }
                                    }
                                } else {
                                    if (LogsClass != NULL)
                                        LogsClass->WriteMessageAndDataStatus(KERROR, "READDUMP (%s N%u): Bad crc32, file: %s", objectname.c_str(), number, newfilename.c_str());
                                    res = false;
                                    break;
                                }
                            } else {
                                if (LogsClass != NULL)
                                    LogsClass->WriteMessageAndDataStatus(KERROR, "READDUMP (%s N%u): Bad identificator, file: %s", objectname.c_str(), number, newfilename.c_str());
                                res = false;
                                break;
                            }

                            if ((inbuff - blocksize) > 0) {
                                memcpy(bufftemp, buff + blocksize, inbuff - blocksize);
                                memcpy(buff, bufftemp, inbuff - blocksize);
                                inbuff = inbuff - blocksize;
                            }

                        } else {
                            if (LogsClass != NULL)
                                LogsClass->WriteMessageAndDataStatus(KERROR, "READDUMP (%s N%u): Bad identificator, file: %s", objectname.c_str(), number, newfilename.c_str());
                            res = false;
                            break;
                        }

                    } else {
                        if (feof(handle)) {
                            break;
                        } else {
                            if (LogsClass != NULL)
                                LogsClass->WriteMessageAndDataStatus(KERROR, "READDUMP (%s N%u): Error read file: %s", objectname.c_str(), number, newfilename.c_str());
                            res = false;
                            break;
                        }
                    }
                }

                delete[] buff;
                buff = NULL;
                delete[] bufftemp;
                bufftemp = NULL;
            } catch (...) {
                delete[] buff;
                buff = NULL;
                delete[] bufftemp;
                bufftemp = NULL;
            }
            fclose(handle);
        }

        if ((res) && (LogsClass != NULL))
            LogsClass->WriteMessageAndDataStatus(KMESSAGE, "READDUMP (%s N%u): Read count = %u", objectname.c_str(), number, readcount);
    } else
        res = false;

    if (res)
        printf("OK (%u)\n", timedelay);
    else
        printf("FAILED (%u)\n", timedelay);

    UnLock();

    return res;
}

template <class TKey, class TValue, class TValueTr>
ui32 TKBasaItem<TKey, TValue, TValueTr>::GetBuffElement(TKey key, TValue* elm, char* buff, ui32 buffsize) {
    ui32 res = 0;

    if ((elm != NULL) && (buff != NULL) && ((sizeof(TKey) + sizeof(TValue)) <= buffsize)) {
        memcpy(buff, &key, sizeof(TKey));
        res += sizeof(TKey);
        memcpy(buff + res, elm, sizeof(TValue));
        res += sizeof(TValue);
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
TValue* TKBasaItem<TKey, TValue, TValueTr>::AddBuffElement(TKey& key, char* buff, ui32 buffsize, ui32& elsize) {
    TValue* res = NULL;

    elsize = 0;
    if ((sizeof(TKey) + sizeof(TValue)) <= buffsize) {
        memcpy(&key, buff, sizeof(TKey));
        res = new TValue();
        memcpy(res, buff + sizeof(TKey), sizeof(TValue));
        elsize = sizeof(TKey) + sizeof(TValue);
    }
    return res;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaItem<TKey, TValue, TValueTr>::PrintDataToFile(TString filename) {
    typedef std::list<TString> TTStringList;
    typedef TTStringList::iterator TTStringListIt;

    bool res = false;
    FILE* handle = NULL;
    TString newfilename = "";
    typename TDataHash::iterator it;
    TString s = "";
    TTStringList slist;
    TTStringListIt lit;

    Lock();

    newfilename = filename + "." + IntToStroka3(number);
    handle = fopen(newfilename.c_str(), "w+b");
    if (handle != NULL) {
        if (data != NULL) {
            res = true;

            slist.clear();
            it = data->begin();
            while (it != data->end()) {
                s = GetInnerData((*it).first, (*it).second);
                slist.push_back(s);
                ++it;
            }

            slist.sort();

            lit = slist.begin();
            while (lit != slist.end()) {
                s = *lit;
                fprintf(handle, "%s\n", s.c_str());
                ++lit;
            }

            slist.clear();
        }
        fclose(handle);
    }

    UnLock();

    return res;
}

template <class TKey, class TValue, class TValueTr>
TString TKBasaItem<TKey, TValue, TValueTr>::GetInnerData(TKey key, TValue* elm) {
    TString res = "";

    if (elm != NULL)
        //res = key.DataToText() + ": " + elm->DataToText();
        res = IntToStroka(key) + ": " + elm->DataToText();

    return res;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaItem<TKey, TValue, TValueTr>::First(TKey& key, TValue& value) {
    bool res = false;
    ui32 count = 0;

    if (data != NULL) {
        count = data->size();

        curr_it = data->begin();
        if (curr_it != data->end()) {
            key = (*curr_it).first;
            if ((*curr_it).second != NULL)
                value = *(*curr_it).second;
            res = true;
        }
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaItem<TKey, TValue, TValueTr>::Next(TKey& key, TValue& value) {
    bool res = false;
    ui32 count = 0;

    if (data != NULL) {
        count = data->size();

        ++curr_it;
        if (curr_it != data->end()) {
            key = (*curr_it).first;
            if ((*curr_it).second != NULL)
                value = *(*curr_it).second;
            res = true;
        }
    }

    return res;
}

//***********************************************************************************
//                                 TKBasaMain
//***********************************************************************************

template <class TKey, class TValue, class TValueTr>
class TKBasaMain {
private:
    static const int MAX_ELEMENT_COUNT = 32;
    TKBasaItem<TKey, TValue, TValueTr> ELEMENTARRAY[MAX_ELEMENT_COUNT];
    TKQueue<TKey, TValueTr> QUEUEARRAY[MAX_ELEMENT_COUNT];
    TLogClass* LogsClass;
    TString dumpfilename;
    ui32 records_max_count;

    int GetIndex(TKey key);

public:
    TKBasaMain();
    ~TKBasaMain();

    //
    void Init(TLogClass* LogsClassA, ui32 records_max_countA, ui32 LiveDayA, TString dumpfilenameA, ui32 QueueCapacity);
    ///
    bool Add(TKey key, TValueTr value);
    int Get(TKey key, TValue& value);
    bool AddHard(TKey key, TValue value); // ,  
    // 
    void CorrectIP(TKey key, TValueTr value);
    //    
    bool Transmit();
    //  (  ,    " ")
    void CleanUpAll();
    // " "   
    void SetLiveDay(ui32 LiveDayA);
    ui32 GetLiveDay();
    ///  / 
    bool WriteDump();
    bool ReadDump();
    //  (   )
    bool PrintDataToFile(TString filename);
    //
    bool Midnight();
    //   (   )
    bool Close();
    //
    ui32 Count();
    bool Overflow();
    TString GetQueueStat();
    TString GetHashStat();
    //  
    ui32 GetSectionCount() {
        return MAX_ELEMENT_COUNT;
    }
    TKBasaItem<TKey, TValue, TValueTr>* GetPSection(int index);
};

template <class TKey, class TValue, class TValueTr>
TKBasaMain<TKey, TValue, TValueTr>::TKBasaMain() {
    LogsClass = NULL;
    dumpfilename = "";
    records_max_count = 0;
}

template <class TKey, class TValue, class TValueTr>
TKBasaMain<TKey, TValue, TValueTr>::~TKBasaMain() {
}

template <class TKey, class TValue, class TValueTr>
int TKBasaMain<TKey, TValue, TValueTr>::GetIndex(TKey key) {
    int res = 0;

    //res = key.ident() % MAX_ELEMENT_COUNT;
    res = ((ui32)key) % MAX_ELEMENT_COUNT;

    return res;
}

template <class TKey, class TValue, class TValueTr>
void TKBasaMain<TKey, TValue, TValueTr>::Init(TLogClass* LogsClassA, ui32 records_max_countA, ui32 LiveDayA, TString dumpfilenameA, ui32 QueueCapacity) {
    records_max_count = records_max_countA;
    ui32 tailrecordcount = records_max_count / MAX_ELEMENT_COUNT;
    ui32 tailqueuecapacity = QueueCapacity / MAX_ELEMENT_COUNT;

    dumpfilename = dumpfilenameA;
    LogsClass = LogsClassA;
    for (int i = 0; i < MAX_ELEMENT_COUNT; i++) {
        ELEMENTARRAY[i].Init(i + 1, LogsClass, tailrecordcount, "ipbasa", LiveDayA);
        QUEUEARRAY[i].Init(tailqueuecapacity);
    }
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaMain<TKey, TValue, TValueTr>::Add(TKey key, TValueTr value) {
    int index = GetIndex(key);
    bool res = false;

    if ((index >= 0) && (index < MAX_ELEMENT_COUNT)) {
        if (!ELEMENTARRAY[index].Busy()) {
            res = ELEMENTARRAY[index].Add(key, value);
        } else {
            res = QUEUEARRAY[GetIndex(key)].Add(key, value);
        }
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
int TKBasaMain<TKey, TValue, TValueTr>::Get(TKey key, TValue& value) {
    return ELEMENTARRAY[GetIndex(key)].Get(key, value);
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaMain<TKey, TValue, TValueTr>::AddHard(TKey key, TValue value) {
    int index = GetIndex(key);
    bool res = false;

    if ((index >= 0) && (index < MAX_ELEMENT_COUNT)) {
        res = ELEMENTARRAY[index].AddHard(key, value);
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
void TKBasaMain<TKey, TValue, TValueTr>::CorrectIP(TKey key, TValueTr value) {
    ELEMENTARRAY[GetIndex(key)].CorrectIP(key, value);
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaMain<TKey, TValue, TValueTr>::Transmit() {
    bool res = false;
    TKey key;
    TValueTr value;

    for (int index = 0; index < MAX_ELEMENT_COUNT; index++) {
        if (QUEUEARRAY[index].GetWithoutErase(key, value)) {
            if (ELEMENTARRAY[index].Add(key, value)) {
                QUEUEARRAY[index].EraseFirst();
                res = true;
            }
        }
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
void TKBasaMain<TKey, TValue, TValueTr>::CleanUpAll() {
    for (int i = 0; i < MAX_ELEMENT_COUNT; i++)
        ELEMENTARRAY[i].CleanUp();
}

template <class TKey, class TValue, class TValueTr>
void TKBasaMain<TKey, TValue, TValueTr>::SetLiveDay(ui32 LiveDayA) {
    for (int i = 0; i < MAX_ELEMENT_COUNT; i++)
        ELEMENTARRAY[i].SetLiveDay(LiveDayA);
}

template <class TKey, class TValue, class TValueTr>
ui32 TKBasaMain<TKey, TValue, TValueTr>::GetLiveDay() {
    return ELEMENTARRAY[0].GetLiveDay();
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaMain<TKey, TValue, TValueTr>::WriteDump() {
    bool res = true;

    if (LogsClass != NULL)
        LogsClass->WriteMessageAndDataStatus(KMESSAGE, "IPBASA: Write dump...");
    printf("IPBASA: Write dump...\n");

    ui32 t = CShingleTime::GetMs();

    for (int i = 0; i < MAX_ELEMENT_COUNT; i++) {
        if (!ELEMENTARRAY[i].WriteDump(dumpfilename)) {
            res = false;
            break;
        }
    }

    t = CShingleTime::GetMs() - t;
    if (LogsClass != NULL)
        LogsClass->WriteMessageAndDataStatus(KMESSAGE, "IPBASA: Write dump compleate to %u msec", t);
    printf("IPBASA: Write dump compleate to %u msec", t);

    return res;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaMain<TKey, TValue, TValueTr>::ReadDump() {
    bool res = true;

    if (LogsClass != NULL)
        LogsClass->WriteMessageAndDataStatus(KMESSAGE, "IPBASA: Load at startup...");
    printf("IPBASA: Load at startup...\n");

    ui32 t = CShingleTime::GetMs();

    for (int i = 0; i < MAX_ELEMENT_COUNT; i++) {
        if (!ELEMENTARRAY[i].ReadDump(dumpfilename)) {
            res = false;
            break;
        }
    }

    t = CShingleTime::GetMs() - t;
    if (LogsClass != NULL)
        LogsClass->WriteMessageAndDataStatus(KMESSAGE, "IPBASA: Load compleate to %u msec", t);
    printf("IPBASA: Load compleate to %u msec\n", t);

    return res;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaMain<TKey, TValue, TValueTr>::PrintDataToFile(TString filename) {
    bool res = true;

    for (int i = 0; i < MAX_ELEMENT_COUNT; i++) {
        if (!ELEMENTARRAY[i].PrintDataToFile(filename)) {
            res = false;
            break;
        }
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaMain<TKey, TValue, TValueTr>::Midnight() {
    bool res = true;

    if (LogsClass != NULL)
        LogsClass->WriteMessageAndDataStatus(KMESSAGE, "IPBASA: Midnight processing...");

    ui32 t = CShingleTime::GetMs();

    CleanUpAll();

    for (int i = 0; i < MAX_ELEMENT_COUNT; i++)
        QUEUEARRAY[i].Midnight();

    t = CShingleTime::GetMs() - t;

    if (LogsClass != NULL) {
        if (res)
            LogsClass->WriteMessageAndDataStatus(KMESSAGE, "IPBASA: Midnight compleate to %u msec.", t);
        else
            LogsClass->WriteMessageAndDataStatus(KERROR, "IPBASA: Midnight failure to %u msec.", t);
    }

    WriteDump();

    return true;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaMain<TKey, TValue, TValueTr>::Close() {
    bool res = false;

    if (LogsClass != NULL)
        LogsClass->WriteMessageAndDataStatus(KMESSAGE, "IPBASA: Close processing...");

    ui32 t = CShingleTime::GetMs();

    res = WriteDump();

    t = CShingleTime::GetMs() - t;

    if (LogsClass != NULL) {
        if (res)
            LogsClass->WriteMessageAndDataStatus(KMESSAGE, "IPBASA: Close compleate to %u msec.", t);
        else
            LogsClass->WriteMessageAndDataStatus(KERROR, "IPBASA: Close failure to %u msec.", t);
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
ui32 TKBasaMain<TKey, TValue, TValueTr>::Count() {
    ui32 res = 0;

    for (int i = 0; i < MAX_ELEMENT_COUNT; i++) {
        res += ELEMENTARRAY[i].Count();
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
bool TKBasaMain<TKey, TValue, TValueTr>::Overflow() {
    bool res = false;

    for (int i = 0; i < MAX_ELEMENT_COUNT; i++) {
        if (ELEMENTARRAY[i].Overflow()) {
            res = true;
            break;
        }
    }

    return res;
}

template <class TKey, class TValue, class TValueTr>
TString TKBasaMain<TKey, TValue, TValueTr>::GetQueueStat() {
    TString res = "";
    ui32 incount = 0;
    ui32 lostcounttoday = 0;
    ui32 lostcountall = 0;

    for (int i = 0; i < MAX_ELEMENT_COUNT; i++) {
        incount += QUEUEARRAY[i].Count();
        lostcounttoday += QUEUEARRAY[i].LostToday();
        lostcountall += QUEUEARRAY[i].LostAll();
    }
    res = IntToStroka(incount) + " / " + IntToStroka(lostcounttoday) + " / " + IntToStroka(lostcountall);

    return res;
}

template <class TKey, class TValue, class TValueTr>
TString TKBasaMain<TKey, TValue, TValueTr>::GetHashStat() {
    TString res = "";
    ui32 count = 0;

    for (int i = 0; i < MAX_ELEMENT_COUNT; i++) {
        count += ELEMENTARRAY[i].Count();
    }
    res = IntToStroka(count);

    return res;
}

template <class TKey, class TValue, class TValueTr>
TKBasaItem<TKey, TValue, TValueTr>* TKBasaMain<TKey, TValue, TValueTr>::GetPSection(int index) {
    TKBasaItem<TKey, TValue, TValueTr>* res = NULL;

    if ((index >= 0) && (index < MAX_ELEMENT_COUNT))
        res = &ELEMENTARRAY[index];

    return res;
}

//*************************************************************************************
