#pragma once

#include <util/datetime/systime.h>
#include <util/string/ascii.h>
#include <util/generic/hash.h>
#include <util/system/mutex.h>
#include <mail/so/spamstop/tools/so-common/tkconfig.h>
#include <mail/so/spamstop/tools/so-common/tlogclass.h>
#include <mail/so/spamstop/tools/so-common/color_scheme.h>

//****************************************************************************************************************************************
//                                                          TStorageCacheBase
//****************************************************************************************************************************************

//#define USE_NCACHE

#ifdef USE_NCACHE
#include <mail/so/spamstop/tools/so-common/local_cache_template.h>
#endif

namespace st_cache {

    struct TInOutStat {
        ui32 m_add_all;
        ui32 m_add_new;
        ui32 m_get_all;
        ui32 m_get_exists;

        TInOutStat() {
            Clear();
        }

        void Clear() {
            m_add_all = 0;
            m_add_new = 0;
            m_get_all = 0;
            m_get_exists = 0;
        }
    };

    struct TInOutStatTY {
        TInOutStat m_today;
        TInOutStat m_yesterday;

        TInOutStatTY() {
            Clear();
        }

        void Clear() {
            m_today.Clear();
            m_yesterday.Clear();
        }

        void Midnight() {
            m_yesterday = m_today;
            m_today.Clear();
        }
    };

    enum TCacheMode {
        CM_ALL,
        CM_BYERR
    };

    TString TCacheModeToTString(TCacheMode mode);

    struct TStorageCacheBaseStat {
        bool enable;
        TCacheMode mode;
        ui32 live_period;
        ui32 max_record_in_cache;
        ui32 data_size;
        ui32 data_prev_size;
        ui32 last_swap_time;
        TInOutStatTY inoutstat;

        TStorageCacheBaseStat() {
            Clear();
        }

        void Clear() {
            enable = false;
            mode = CM_BYERR;
            live_period = 0;
            max_record_in_cache = 0;
            data_size = 0;
            data_prev_size = 0;
            last_swap_time = 0;
            inoutstat.Clear();
        }
    };

    template <typename TVALUE>
    class TStorageCacheBase {
    private:
        static const ui32 DEFAULT_LIVE_PERIOD_SEC = 60;
        static const ui32 DEFAULT_MAX_RECORD_IN_CACHE = 10000000;

    private:
        bool m_enable;
        TCacheMode m_mode;

#ifdef USE_NCACHE
        NCache::TCacheBase<ui64, TVALUE> data_cache;
#else //USE_NCACHE
        THashMap<ui64, TVALUE>* data;
        THashMap<ui64, TVALUE>* data_prev;
#endif
        TLogClass* m_Log;
        ui32 m_live_period;
        ui32 m_max_record_in_cache;
        ui32 m_last_swap_cache;
        TMutex m_Mutex;

        TInOutStatTY m_inout_stat;

    public:
        TStorageCacheBase();
        ~TStorageCacheBase();

        void Init(bool enable, TLogClass* LogsA, TKConfig* configA, const TString& config_ident);
        void EventTick();
        void Midnight();
        TStorageCacheBaseStat GetStat();
        TCacheMode GetCacheMode() {
            return m_mode;
        }

        void Add(ui64 shingle, const TVALUE& datacnt);
        TVALUE Get(ui64 shingle, bool& exists);

        TString GetCacheStatistic();
    };

    template <typename TVALUE>
    TStorageCacheBase<TVALUE>::TStorageCacheBase() {
        m_enable = true;
        m_mode = CM_BYERR;
        m_Log = NULL;
#ifndef USE_NCACHE
        data = new THashMap<ui64, TVALUE>();
        data_prev = new THashMap<ui64, TVALUE>();
#endif
        m_last_swap_cache = time(NULL);
    }

    template <typename TVALUE>
    TStorageCacheBase<TVALUE>::~TStorageCacheBase() {
#ifndef USE_NCACHE
        if (data != NULL) {
            delete data;
            data = NULL;
        }

        if (data_prev != NULL) {
            delete data_prev;
            data_prev = NULL;
        }
#endif
    }

    template <typename TVALUE>
    void TStorageCacheBase<TVALUE>::Init(bool enable, TLogClass* LogsA, TKConfig* configA, const TString& config_ident) {
        m_enable = enable;
        m_Log = LogsA;
        if (configA != NULL) {
            m_live_period = configA->ReadInteger(config_ident, "live_period", DEFAULT_LIVE_PERIOD_SEC);
            m_max_record_in_cache = configA->ReadInteger(config_ident, "max_record_in_cache", DEFAULT_MAX_RECORD_IN_CACHE);

            auto mode_s = Trim(configA->ReadStroka(config_ident, "mode", "by_err"));
            if (AsciiEqualsIgnoreCase(mode_s, "all"))
                m_mode = CM_ALL;
            else
                m_mode = CM_BYERR;
        }

#ifdef USE_NCACHE
        data_cache.Init(m_live_period);
#endif
    }

    template <typename TVALUE>
    void TStorageCacheBase<TVALUE>::EventTick() {
#ifdef USE_NCACHE
        data_cache.EventTick();
#else //USE_NCACHE
        THashMap<ui64, TVALUE>* tmp = NULL;
        ui32 current_time = time(NULL);

        if ((current_time > m_last_swap_cache) && ((current_time - m_last_swap_cache) >= m_live_period)) {
            m_Mutex.Acquire();

            tmp = data_prev;
            data_prev = data;
            data = new THashMap<ui64, TVALUE>();

            m_last_swap_cache = current_time;

            m_Mutex.Release();

            if (tmp != NULL) {
                delete tmp;
                tmp = NULL;
            }
        }
#endif
    }

    template <typename TVALUE>
    void TStorageCacheBase<TVALUE>::Midnight() {
        m_Mutex.Acquire();

        m_inout_stat.Midnight();

        m_Mutex.Release();
    }

    template <typename TVALUE>
    TStorageCacheBaseStat TStorageCacheBase<TVALUE>::GetStat() {
        TStorageCacheBaseStat res;

        m_Mutex.Acquire();

        res.enable = m_enable;
        res.mode = m_mode;
        res.live_period = m_live_period;
        res.max_record_in_cache = m_max_record_in_cache;

        if (data != NULL)
            res.data_size = data->size();

        if (data_prev != NULL)
            res.data_prev_size = data_prev->size();

        res.last_swap_time = m_last_swap_cache;

        res.inoutstat = m_inout_stat;

        m_Mutex.Release();

        return res;
    }

    template <typename TVALUE>
    void TStorageCacheBase<TVALUE>::Add(ui64 shingle, const TVALUE& datacnt) {
        if (m_enable) {
            ui32 current_time = time(NULL);

            if ((current_time > m_last_swap_cache) && ((current_time - m_last_swap_cache) > 2 * m_live_period))
                EventTick();

            //add
            //THashMap<ui64, TVALUE>::iterator it;

            m_Mutex.Acquire();

            m_inout_stat.m_today.m_add_all = IncMax32(m_inout_stat.m_today.m_add_all, 1);

            if (data != NULL) {
                auto it = data->find(shingle);
                if (it == data->end()) {
                    (*data)[shingle] = datacnt;
                    m_inout_stat.m_today.m_add_new = IncMax32(m_inout_stat.m_today.m_add_new, 1);

                } else {
                    (*it).second = datacnt;
                }
            }

            m_Mutex.Release();
        }
    }

    template <typename TVALUE>
    TVALUE TStorageCacheBase<TVALUE>::Get(ui64 shingle, bool& exists) {
        TVALUE res;

        exists = false;
        if (m_enable) {
            //THashMap<ui64, TVALUE>::iterator it;

            m_Mutex.Acquire();

            m_inout_stat.m_today.m_get_all = IncMax32(m_inout_stat.m_today.m_get_all, 1);

            if (!exists) {
                if (data != NULL) {
                    auto it = data->find(shingle);
                    if (it != data->end()) {
                        res = (*it).second;
                        exists = true;
                    }
                }
            }

            if (!exists) {
                if (data_prev != NULL) {
                    auto it = data_prev->find(shingle);
                    if (it != data_prev->end()) {
                        res = (*it).second;
                        exists = true;
                    }
                }
            }

            if (exists)
                m_inout_stat.m_today.m_get_exists = IncMax32(m_inout_stat.m_today.m_get_exists, 1);

            m_Mutex.Release();
        }

        return res;
    }

    template <typename TVALUE>
    TString TStorageCacheBase<TVALUE>::GetCacheStatistic() {
        TString res = "";
        TStorageCacheBaseStat stat;
        ui32 curr_time = time(NULL);

        stat = GetStat();

        res += "<table border='1' width='100%' cellspacing='0' cellpadding='4' bgcolor=" + TString(color_table_default) + ">";
        res += "<tr><td width='25%'>Enable</td><td colspan='2'>" + BoolToStroka2(stat.enable) + "</td></tr>";
        res += "<tr><td width='25%'>Mode</td><td colspan='2'>" + TCacheModeToTString(stat.mode) + "</td></tr>";
        res += "<tr><td width='25%'>Live period</td><td colspan='2'>" + IntToStroka(stat.live_period) + " sec</td></tr>";
        res += "<tr><td width='25%'>Max recod in cache</td><td colspan='2'>" + IntToStroka(stat.max_record_in_cache) + "</td></tr>";
        res += "<tr><td width='25%'>Data size</td><td colspan='2'>" + IntToStroka(stat.data_size) + "</td></tr>";
        res += "<tr><td width='25%'>Prev data size</td><td colspan='2'>" + IntToStroka(stat.data_prev_size) + "</td></tr>";
        res += "<tr><td width='25%'>Last swap time (elapsed)</td><td colspan='2'>" + TimeToStr(stat.last_swap_time) + " (" + IntToHourMinSec(curr_time - stat.last_swap_time) + ")</td></tr>";
        res += "<tr bgcolor='" + TString(color_table_shap) + "'><td width='25%'>&nbsp;</td><td width='37%'><b>TODAY</b></td><td width='38%'><b>YESTERDAY</b></td></tr>";
        res += "<tr><td width='25%'>Get, all</td><td>" + IntToStroka(stat.inoutstat.m_today.m_get_all) + "</td><td>" + IntToStroka(stat.inoutstat.m_yesterday.m_get_all) + "</td></tr>";
        res += "<tr><td width='25%'>Get, exists</td><td>" + IntToStroka(stat.inoutstat.m_today.m_get_exists) + "</td><td>" + IntToStroka(stat.inoutstat.m_yesterday.m_get_exists) + "</td></tr>";
        res += "<tr><td width='25%'>Add, all</td><td>" + IntToStroka(stat.inoutstat.m_today.m_add_all) + "</td><td>" + IntToStroka(stat.inoutstat.m_yesterday.m_add_all) + "</td></tr>";
        res += "<tr><td width='25%'>Add, new</td><td>" + IntToStroka(stat.inoutstat.m_today.m_add_new) + "</td><td>" + IntToStroka(stat.inoutstat.m_yesterday.m_add_new) + "</td></tr>";
        res += "</table>";

        return res;
    }
} // namespace st_cache

//****************************************************************************************************************************************
