#pragma once

#include "util/generic/hash.h"
#include <util/generic/string.h>
#include "util/system/mutex.h"
#include "util/string/util.h"
#include "util/generic/hash.h"

//*******************************************************************************************************************************
//                                                 TStorageInfo
//*******************************************************************************************************************************

enum TSQLTypeK { ST_UNKNOWN,
                 ST_SQL,
                 ST_NOSQL };
enum TShardType { SHT_UNKNOWN,
                  SHT_SKEEP,
                  SHT_NUMBER };

struct TStorageInfo {
    TSQLTypeK m_sql_type;
    TShardType m_shard_type;
    ui16 m_shar_number;
    TString m_name;

    TStorageInfo() {
        Clear();
    }

    TStorageInfo(TSQLTypeK sql_type, TShardType shard_type, ui16 shar_number, const TString& name) {
        m_sql_type = sql_type;
        m_shard_type = shard_type;
        m_shar_number = shar_number;
        m_name = name;
    }

    void Clear() {
        m_sql_type = ST_UNKNOWN;
        m_shard_type = SHT_UNKNOWN;
        m_shar_number = 0;
        m_name = "";
    }

    bool operator==(const TStorageInfo& value) const {
        bool res = false;

        if ((m_sql_type == value.m_sql_type) && (m_shard_type == value.m_shard_type) && (m_shar_number == value.m_shar_number))
            res = true;

        return res;
    }

    TString GetStorageIdent() {
        TString res = "";

        if (!m_name.empty()) {
            res = m_name;

        } else {
            switch (m_shard_type) {
                case SHT_UNKNOWN:
                    res = "UNKNOWN";
                    break;
                case SHT_SKEEP:
                    res = "SKEEP";
                    break;
                case SHT_NUMBER:
                    res = "NUMB" + IntToStroka(m_shar_number);
                    break;
            };
        }

        return res;
    }
};

//*******************************************************************************************************************************
//                                                 TStorageInfoExt
//*******************************************************************************************************************************

struct TStorageInfoExt {
    TStorageInfo info;
    void* storage;

    TStorageInfoExt() {
        info.Clear();
        storage = NULL;
    }

    TStorageInfoExt(TSQLTypeK sql_type, TShardType shard_type, ui16 shar_number, void* storA, const TString& name) {
        info = TStorageInfo(sql_type, shard_type, shar_number, name);
        storage = storA;
    }

    bool operator==(const TStorageInfoExt& value) const {
        bool res = false;

        res = (info == value.info);

        return res;
    }

    TString AsText() {
        TString res = "";

        res = res + "name='" + info.m_name + "' ";
        res = res + "sqltype=";
        switch (info.m_sql_type) {
            case ST_UNKNOWN:
                res = res + "UNKNOWN ";
                break;
            case ST_SQL:
                res = res + "SQL ";
                break;
            case ST_NOSQL:
                res = res + "NOSQL ";
                break;
        };
        res = res + "type=";
        switch (info.m_shard_type) {
            case SHT_UNKNOWN:
                res = res + "UNKNOWN ";
                break;
            case SHT_SKEEP:
                res = res + "SKEEP ";
                break;
            case SHT_NUMBER:
                res = res + "NUMB" + IntToStroka(info.m_shar_number) + " ";
                break;
        };
        if (storage != NULL)
            res = res + "address=" + UI64ToStroka((ui64)storage);
        else
            res = res + "address=NULL";

        return res;
    }
};

typedef std::vector<TStorageInfoExt> TStorageNoSqlVector;
typedef TStorageNoSqlVector::iterator TStorageNoSqlVectorIt;

//*******************************************************************************************************************************
//                                                 TStorageInfoFull
//*******************************************************************************************************************************

struct TStorageInfoFull {
    TStorageInfo info;
    //ui32          change_time_localcache;
    //bool          disable_localcache;
    //bool          localcache_add_empty_record;
    TString basatypename;
    TString storageident;
    //TString        other_data;

    TStorageInfoFull() {
        Clear();
    }

    void Clear() {
        info.Clear();
        //change_time_localcache      = 0;
        //disable_localcache          = false;
        //localcache_add_empty_record = false;
        basatypename = "";
        storageident = "";
        //other_data                  = "";
    }
};

//*******************************************************************************************************************************
