#pragma once

#include "util/generic/string.h"
#include <mail/so/spamstop/tools/so-common/tkipv6.h>
#include <mail/so/spamstop/tools/so-common/kfunc.h>
#include <mail/so/spamstop/tools/so-common/tbasestoragetypes.h>
#include <mail/so/spamstop/tools/so-common/serializehelper.h>
#include <mail/so/spamstop/tools/so-clients/ktime.h>

#define COMMON_SERVICE "COMMON"
#define LONGIP_COLLECTION_ID "_lip"
#define BASE_STAT_SHINGLE_RQST 0xFFFFFFFFFFFFFFFF
#define BASE_STAT_SHINGLE_RCPT 0xEEEEEEEEEEEEEEEE
#define BASE_STAT_SHINGLE_DOPSTR "***"

//******************************************************************************
//                                TLongIPData
//******************************************************************************

struct TLongIPData {
    bool exist_data;
    int err;
    ui32 days;
    ui32 days_elapsed;
    ui32 spam;
    ui32 ham;
    float spam_percent;
    ui32 found;
    float found_percent;

    TLongIPData() {
        Clear();
    }

    void Clear() {
        exist_data = false;
        err = 0;
        days = 0;
        days_elapsed = 0;
        spam = 0;
        ham = 0;
        spam_percent = 0;
        found = 0;
        found_percent = 0;
    }
};

//******************************************************************************
//                                TCountersStat
//******************************************************************************

struct TCountersStat {
    ui32 Ham{};
    ui32 Spam{};
    ui32 All{};
};

//******************************************************************************
//                                TCountersStatEx
//******************************************************************************

struct TCountersStatEx {
    ui32 today_ham{};
    ui32 yesterday_ham{};
    ui32 today_spam{};
    ui32 yesterday_spam{};

    ui32 TodayAll() {
        return IncMax32(today_ham, today_spam);
    }
    ui32 YesterdayAll() {
        return IncMax32(yesterday_ham, yesterday_spam);
    }
    ui32 Ham() {
        return IncMax32(today_ham, yesterday_ham);
    }
    ui32 Spam() {
        return IncMax32(today_spam, yesterday_spam);
    }
    ui32 All() {
        return IncMax32(TodayAll(), YesterdayAll());
    }
    void IncTodayHam() {
        today_ham = IncMax32(today_ham, 1);
    }
    void IncTodaySpam() {
        today_spam = IncMax32(today_spam, 1);
    }
};

//******************************************************************************
//                                TSrvcStatMin
//******************************************************************************

struct TASEItem {
    ui32 m_all;
    ui32 m_spam;
    ui32 m_error;
    bool m_collision;

    TASEItem() {
        Clear();
    }

    void Clear() {
        m_all = 0;
        m_spam = 0;
        m_error = 0;
        m_collision = false;
    }
};

struct TSrvcStatMinItem {
    TASEItem m_5min;
    TASEItem m_day;
    TASEItem m_yesterday;

    TSrvcStatMinItem() {
        Clear();
    }

    void Clear() {
        m_5min.Clear();
        m_day.Clear();
        m_yesterday.Clear();
    }
};

struct TSrvcStatMin {
    TSrvcStatMinItem m_request;
    TSrvcStatMinItem m_rcpt;

    TSrvcStatMin() {
        Clear();
    }

    void Clear() {
        m_request.Clear();
        m_rcpt.Clear();
    }
};

//******************************************************************************
//                                stordata
//******************************************************************************

namespace stordata {
    enum DATATYPE {
        UNKNOWN,
        IPSTAT,
        LOGINSTAT,
        HOSTSTAT,
        EMAILSTAT,
        PHONESTAT,
        NICKSTAT,
        COMMENTSTAT,
        SUBJECTSTAT,
        GEO,
        GEOCOUNTRY,
        IDPOSTCARD,
        SUIDTOTAL,
        TEXTTOTAL,
        IPSUIDTEXTTOTAL,
        IPFOUND,
        LOGINFOUND,
        TEXTFOUND,
        GEOSFOUND,
        IPSUIDFOUND,
        IPTEXTFOUND,
        SERVICE_STAT_RQST,
        SERVICE_STAT_RCPT,
        LOGIN_CHATID_STAT,
        UNIQ_CHATIDS_BY_LOGIN_STAT,
    };

    enum RESTYPE { RT_HAM,
                   RT_SPAM,
                   RT_ERROR };

    TString DATATYPEToStroka(DATATYPE type);

    struct TStorData {
        TKIPv6 ip;
        TString login;
        ui64 hostshingle;
        ui64 emailshingle;
        ui64 phoneshingle;
        ui64 nickshingle;
        ui64 commentshingle;
        ui64 geoshingle;
        ui64 geocountryshingle;
        //only service MSEARCH-PROXY
        ui64 suidtotal_shingle;
        ui64 texttotal_shingle;
        ui64 ip_suid_text_total_shingle;
        ui64 ipfound_shingle;
        ui64 loginfound_shingle;
        ui64 textfound_shingle;
        ui64 geosfound_shingle;
        ui64 ipsuidfound_shingle;
        ui64 iptextfound_shingle;

        TStorData() {
            Clear();
        }

        void Clear() {
            ip = TKIPv6();
            login = "";
            hostshingle = 0;
            emailshingle = 0;
            phoneshingle = 0;
            nickshingle = 0;
            commentshingle = 0;
            geoshingle = 0;
            geocountryshingle = 0;
            suidtotal_shingle = 0;
            texttotal_shingle = 0;
            ip_suid_text_total_shingle = 0;
            ipfound_shingle = 0;
            loginfound_shingle = 0;
            textfound_shingle = 0;
            geosfound_shingle = 0;
            ipsuidfound_shingle = 0;
            iptextfound_shingle = 0;
        }
    };

    ui64 CreateShingle(ui64 shingle, DATATYPE type);

    //****************************************************************************************************************
    //                                               TShingleDataItem
    //****************************************************************************************************************

    struct TShingleDataItem {
        ui32 m_type;
        TString m_so_service;
        ui32 m_ham;
        ui32 m_spam;

        TShingleDataItem() {
            Clear();
        }

        void Clear() {
            m_type = UNKNOWN;
            m_so_service = "";
            m_ham = 0;
            m_spam = 0;
        }

        TString toLog() {
            const char* p = strstr(m_so_service.c_str(), TString(BASE_STAT_SHINGLE_DOPSTR).c_str());

            if ((p != NULL) && (p == m_so_service.c_str()))
                return "srvc='" + m_so_service + "',type=" + IntToStroka(m_type) + ",h=" + IntToStroka(m_ham) + ",s=" + IntToStroka(m_spam);
            else
                return "srvc='" + m_so_service + "',type=" + DATATYPEToStroka(static_cast<DATATYPE>(m_type)) + ",h=" + IntToStroka(m_ham) + ",s=" + IntToStroka(m_spam);
        }

        TString toList(ui64 shingle) {
            const char* p = strstr(m_so_service.c_str(), TString(BASE_STAT_SHINGLE_DOPSTR).c_str());

            if ((p != NULL) && (p == m_so_service.c_str()))
                return "shingle=" + ShingleToStroka(shingle) + ",srvc='" + m_so_service + "',all=" + IntToStroka(m_type) + ",err=" + IntToStroka(m_ham) + ",spam=" + IntToStroka(m_spam);
            else
                return "shingle=" + ShingleToStroka(shingle) + ",srvc='" + m_so_service + "',type=" + DATATYPEToStroka(static_cast<DATATYPE>(m_type)) + "(" + IntToStroka(static_cast<int>(m_type)) + "),h=" + IntToStroka(m_ham) + ",s=" + IntToStroka(m_spam);
        }

        TString writeToDump(ui64 shingle) {
            return ShingleToStroka(shingle) + "," + m_so_service + "," + IntToStroka(m_type) + "," + IntToStroka(m_ham) + "," + IntToStroka(m_spam) + ",";
        }

        bool LoadData(const TString& s, ui64& shingle) {
            bool res = false;

            Clear();
            shingle = 0;
            if (!s.empty()) {
                const char *pb = NULL, *pe = NULL;
                std::vector<TString> field_list;
                int count = 0;
                bool ok_pr[5] = {false, false, false, false, false};

                pb = s.c_str();
                while ((pe = strchr(pb, ',')) != NULL) {
                    count = pe - pb;
                    if (count > 0)
                        field_list.push_back(TString(pb, count));

                    pb = pe + 1;
                }
                if (field_list.size() == 5) {
                    for (size_t i = 0; i < 5; i++) {
                        if (!field_list[i].empty()) {
                            if (i == 0) //shingle
                            {
                                if (sscanf(field_list[i].c_str(), "%lx", &shingle) != 1) {
                                    shingle = 0;
                                    res = false;
                                    break;

                                } else
                                    ok_pr[i] = true;

                            } else if (i == 1) //so_service
                            {
                                m_so_service = field_list[i];
                                ok_pr[i] = true;

                            } else if (i == 2) //type
                            {
                                ui32 type_var = 0;

                                if (sscanf(field_list[i].c_str(), "%u", &type_var) != 1) {
                                    res = false;
                                    break;

                                } else {
                                    m_type = type_var;
                                    ok_pr[i] = true;
                                }

                            } else if (i == 3) //ham
                            {
                                ui32 ham_var = 0;

                                if (sscanf(field_list[i].c_str(), "%u", &ham_var) != 1) {
                                    res = false;
                                    break;

                                } else {
                                    m_ham = ham_var;
                                    ok_pr[i] = true;
                                }

                            } else if (i == 4) //spam
                            {
                                ui32 spam_var = 0;

                                if (sscanf(field_list[i].c_str(), "%u", &spam_var) != 1) {
                                    res = false;
                                    break;

                                } else {
                                    m_spam = spam_var;
                                    ok_pr[i] = true;
                                }
                            }
                        }
                    }

                    res = ok_pr[0] && ok_pr[1] && ok_pr[2] && ok_pr[3] && ok_pr[4];
                }
            }

            return res;
        }
    };

    //****************************************************************************************************************
    //                                      TShingleDataItemSerialization
    //****************************************************************************************************************

#define FIELD_ID "id"
#define FIELD_SOSERVICE "srvc"
#define FIELD_TYPE "type"
#define FIELD_HAM "ham"
#define FIELD_SPAM "spam"

    class TShingleDataItemSerialization {
    public:
        static void Deserialize(nosql::HashMap& hash, TShingleDataItem& data);
        static void Deserialize2(nosql::HashMap& incrs, nosql::HashMap& sets, TShingleDataItem& data);
        static void Serialize(const TShingleDataItem& data, nosql::HashMap& sets);
    };

    //****************************************************************************************************************
    //                                               TLongIPDataItem
    //****************************************************************************************************************

    enum TLongIPCnt { LIP_HAM,
                      LIP_SPAM,
                      LIP_FOUND,
                      LPHONE_HAM,
                      LPHONE_SPAM };

    struct TLongIPDataItem {
        TString m_so_service;
        ui32 m_lasttime;
        ui32 m_ham;
        ui32 m_spam;
        ui32 m_found;

        TLongIPDataItem() {
            Clear();
        }

        void Clear() {
            m_so_service = "";
            m_lasttime = 0;
            m_ham = 0;
            m_spam = 0;
            m_found = 0;
        }

        TLongIPData ConverToTLongIPData() {
            TLongIPData res;
            ui32 currenttime = 0;

            res.exist_data = true;
            res.days = m_lasttime;
            res.ham = m_ham;
            res.spam = m_spam;
            ui32 divv = IncMax32(res.ham, res.spam);
            if (divv > 0)
                res.spam_percent = static_cast<float>(100) * static_cast<float>(res.spam) / static_cast<float>(divv);
            res.found = m_found;
            if (divv > 0)
                res.found_percent = (float)100 * (float)res.found / (float)divv;

            currenttime = (ui32)(kday_t(time(NULL)));
            if (currenttime >= res.days)
                res.days_elapsed = currenttime - res.days;

            return res;
        }

        TString toLog() {
            return "srvc='" + m_so_service + "',lt='" + IntToStroka(m_lasttime) + "',h=" + IntToStroka(m_ham) + ",s=" + IntToStroka(m_spam) + ",f=" + IntToStroka(m_found);
        }

        TString toList(ui64 shingle) {
            return "shingle=" + ShingleToStroka(shingle) + ",srvc='" + m_so_service + "',lt='" + IntToStroka(m_lasttime) + "',h=" + IntToStroka(m_ham) + ",s=" + IntToStroka(m_spam) + ",f=" + IntToStroka(m_found);
        }

        TString writeToDump(ui64 shingle) {
            return ShingleToStroka(shingle) + "," + m_so_service + "," + IntToStroka(m_lasttime) + "," + IntToStroka(m_ham) + "," + IntToStroka(m_spam) + "," + IntToStroka(m_found) + ",";
        }

        bool LoadData(const TString& s, ui64& shingle) {
            bool res = false;

            Clear();
            shingle = 0;
            if (!s.empty()) {
                const char *pb = NULL, *pe = NULL;
                std::vector<TString> field_list;
                int count = 0;
                bool ok_pr[6] = {false, false, false, false, false, false};

                pb = s.c_str();
                while ((pe = strchr(pb, ',')) != NULL) {
                    count = pe - pb;
                    if (count > 0)
                        field_list.push_back(TString(pb, count));

                    pb = pe + 1;
                }
                if (field_list.size() == 6) {
                    for (size_t i = 0; i < 6; i++) {
                        if (!field_list[i].empty()) {
                            if (i == 0) //shingle
                            {
                                if (sscanf(field_list[i].c_str(), "%lx", &shingle) != 1) {
                                    shingle = 0;
                                    res = false;
                                    break;

                                } else
                                    ok_pr[i] = true;

                            } else if (i == 1) //so_service
                            {
                                m_so_service = field_list[i];
                                ok_pr[i] = true;

                            } else if (i == 2) //lasttime
                            {
                                ui32 lt_var = 0;

                                if (sscanf(field_list[i].c_str(), "%u", &lt_var) != 1) {
                                    res = false;
                                    break;
                                } else {
                                    m_lasttime = lt_var;
                                    ok_pr[i] = true;
                                }

                            } else if (i == 3) //ham
                            {
                                ui32 ham_var = 0;

                                if (sscanf(field_list[i].c_str(), "%u", &ham_var) != 1) {
                                    res = false;
                                    break;
                                } else {
                                    m_ham = ham_var;
                                    ok_pr[i] = true;
                                }

                            } else if (i == 4) //spam
                            {
                                ui32 spam_var = 0;

                                if (sscanf(field_list[i].c_str(), "%u", &spam_var) != 1) {
                                    res = false;
                                    break;
                                } else {
                                    m_spam = spam_var;
                                    ok_pr[i] = true;
                                }

                            } else if (i == 5) //found
                            {
                                ui32 found_var = 0;

                                if (sscanf(field_list[i].c_str(), "%u", &found_var) != 1) {
                                    res = false;
                                    break;
                                } else {
                                    m_found = found_var;
                                    ok_pr[i] = true;
                                }
                            }
                        }
                    }

                    res = ok_pr[0] && ok_pr[1] && ok_pr[2] && ok_pr[3] && ok_pr[4] && ok_pr[5];
                }
            }

            return res;
        }
    };

    //****************************************************************************************************************
    //                                      TLongIPDataItemSerialization
    //****************************************************************************************************************

#define FIELD_LIP_ID "id"
#define FIELD_LIP_SOSERVICE "srvc"
#define FIELD_LIP_LASTTIME "lt"
#define FIELD_LIP_HAM "ham"
#define FIELD_LIP_SPAM "spam"
#define FIELD_LIP_FOUND "found"

    class TLongIPDataItemSerialization {
    public:
        static void Deserialize(nosql::HashMap& hash, TLongIPDataItem& data);
        static void Deserialize2(nosql::HashMap& incrs, nosql::HashMap& sets, TLongIPDataItem& data);
        static void Serialize(const TLongIPDataItem& data, nosql::HashMap& sets);
    };

    //****************************************************************************************************************
    //                                               TShingleData
    //****************************************************************************************************************

    struct TShingleData {
        ui64 m_shingle;
        TString m_so_service;
        DATATYPE m_type;
        TCountersStatEx* m_res_p;
        TSrvcStatMinItem* m_stat_p;
        TShingleDataItem m_data_today;
        TShingleDataItem m_data_yesterday;
        TLongIPDataItem m_longipdata;
        bool m_err;
        bool m_collision_error;
        ui32 m_tick;
        bool m_is_stat;
        stor_bt::TNoSqlItem m_nosqldata;

        TShingleData() {
            Clear();
            m_shingle = 0;
            m_so_service = "";
            m_type = UNKNOWN;
            m_res_p = NULL;
            m_stat_p = NULL;
            m_is_stat = false;
        }

        TShingleData(ui64 shingle, const TString& so_service, DATATYPE type, TCountersStatEx* res_p) {
            Clear();
            m_shingle = shingle;
            m_so_service = so_service;
            m_type = type;
            m_res_p = res_p;
            m_stat_p = NULL;
            m_is_stat = false;
        }

        TShingleData(ui64 shingle, const TString& so_service, DATATYPE type, TSrvcStatMinItem* stat_p, bool is_stat) {
            Clear();
            m_shingle = shingle;
            m_so_service = so_service;
            m_type = type;
            m_res_p = NULL;
            m_stat_p = stat_p;
            m_is_stat = is_stat;
        }

        TShingleData(ui64 long_ip_shingle, const TString& so_service) {
            Clear();
            m_shingle = long_ip_shingle;
            m_so_service = so_service;
            m_type = UNKNOWN;
            m_res_p = NULL;
            m_stat_p = NULL;
            m_is_stat = false;
        }

        void Clear() {
            m_data_today.Clear();
            m_data_yesterday.Clear();
            m_longipdata.Clear();
            m_err = false;
            m_collision_error = false;
            m_tick = 0;
            m_nosqldata.clear();
        }

        bool MoveTodayData(TString& err_s) {
            bool res = false;

            err_s = "";
            if (m_res_p != NULL) {
                if ((static_cast<DATATYPE>(m_data_today.m_type) == m_type) && (m_data_today.m_so_service == m_so_service)) {
                    m_res_p->today_ham = m_data_today.m_ham;
                    m_res_p->today_spam = m_data_today.m_spam;
                    res = true;

                } else {
                    m_collision_error = true;
                    err_s = "today collision other shingle=" + ShingleToStroka(m_shingle) + ", type=" + DATATYPEToStroka(static_cast<DATATYPE>(m_type)) + "(stor=" + DATATYPEToStroka(static_cast<DATATYPE>(m_data_today.m_type)) + ")" + ", so_service='" + m_so_service + "'(stor='" + m_data_today.m_so_service + "')";
                    res = false;
                }
            }

            return res;
        }

        bool MoveYesterdayData(TString& err_s) {
            bool res = false;

            err_s = "";
            if (m_res_p != NULL) {
                if ((static_cast<DATATYPE>(m_data_yesterday.m_type) == m_type) && (m_data_yesterday.m_so_service == m_so_service)) {
                    m_res_p->yesterday_ham = m_data_yesterday.m_ham;
                    m_res_p->yesterday_spam = m_data_yesterday.m_spam;
                    res = true;

                } else {
                    m_collision_error = true;
                    err_s = "yesterday collision other shingle=" + ShingleToStroka(m_shingle) + ", type=" + DATATYPEToStroka(static_cast<DATATYPE>(m_type)) + "(stor=" + DATATYPEToStroka(static_cast<DATATYPE>(m_data_yesterday.m_type)) + ")" + ", so_service='" + m_so_service + "'(stor='" + m_data_yesterday.m_so_service + "')";
                    res = false;
                }
            }

            return res;
        }

        TString GetStatSoService(const TString& text) {
            TString res = "";
            const char* p = NULL;
            TString ident = TString(BASE_STAT_SHINGLE_DOPSTR);

            p = strstr(text.c_str(), ident.c_str());
            if (p != NULL)
                res = TString(p + ident.length());

            return res;
        }

        bool MoveStatTodayData(TString& err_s) {
            bool res = false;
            //TString tmp_soservice = "";

            err_s = "";
            if (m_stat_p != NULL) {
                //tmp_soservice = GetStatSoService(m_data_today.m_so_service);
                //if (tmp_soservice == m_so_service)
                if (m_so_service == m_data_today.m_so_service)
                //if (false) //debug
                {
                    m_stat_p->m_day.m_all = m_data_today.m_type;
                    m_stat_p->m_day.m_spam = m_data_today.m_spam;
                    m_stat_p->m_day.m_error = m_data_today.m_ham;

                    res = true;

                } else {
                    m_stat_p->m_day.m_collision = true;
                    m_collision_error = true;
                    err_s = "today collision stat shingle=" + ShingleToStroka(m_shingle) + ", type=" + DATATYPEToStroka(static_cast<DATATYPE>(m_type)) + "(stor=" + DATATYPEToStroka(static_cast<DATATYPE>(m_data_today.m_type)) + ")" + ", so_service='" + m_so_service + "'(stor='" + m_data_today.m_so_service + "')";
                    res = false;
                }
            }

            return res;
        }

        bool MoveStatYesterdayData(TString& err_s) {
            bool res = false;
            //TString tmp_soservice = "";

            err_s = "";
            if (m_stat_p != NULL) {
                //tmp_soservice = GetStatSoService(m_data_yesterday.m_so_service);
                //if (tmp_soservice == m_so_service)
                if (m_so_service == m_data_yesterday.m_so_service) {
                    m_stat_p->m_yesterday.m_all = m_data_yesterday.m_type;
                    m_stat_p->m_yesterday.m_spam = m_data_yesterday.m_spam;
                    m_stat_p->m_yesterday.m_error = m_data_yesterday.m_ham;

                    res = true;

                } else {
                    m_stat_p->m_yesterday.m_collision = true;
                    m_collision_error = true;
                    err_s = "yesterday collision stat shingle=" + ShingleToStroka(m_shingle) + ", type=" + DATATYPEToStroka(static_cast<DATATYPE>(m_type)) + "(stor=" + DATATYPEToStroka(static_cast<DATATYPE>(m_data_yesterday.m_type)) + ")" + ", so_service='" + m_so_service + "'(stor='" + m_data_yesterday.m_so_service + "')";
                    res = false;
                }
            }

            return res;
        }
    };

    typedef std::list<TShingleData> TShingleDataList;
    typedef TShingleDataList::iterator TShingleDataListIt;

    //****************************************************************************************************************
    //                                               TServiceStoreResult
    //****************************************************************************************************************

    struct TServiceStoreResult {
        TCountersStatEx ipstat;
        TCountersStatEx loginstat;
        TCountersStatEx xmpp_server_stat;
        TCountersStatEx nickstat;
        TCountersStatEx commentstat;
        TCountersStatEx subjectstat;
        TCountersStatEx geostat;
        TCountersStatEx geocountrystat;
        TCountersStatEx idpostcardstat;
        TCountersStatEx ip_suid_total_stat;
        TCountersStatEx ip_text_total_stat;
        TCountersStatEx ip_suid_text_total_stat;
        TCountersStatEx ip_found_stat;
        TCountersStatEx suid_found_stat;
        TCountersStatEx text_found_stat;
        TCountersStatEx geos_found_stat;
        TCountersStatEx ip_suid_found_stat;
        TCountersStatEx ip_text_found_stat;
    };

    struct TCommonStoreResult {
        TCountersStatEx ipstat;
        TCountersStatEx loginstat;
        TCountersStatEx nickstat;
        TCountersStatEx commentstat;
        TCountersStatEx subjectstat;
        TCountersStatEx geostat;
        TCountersStatEx geocountrystat;
        TCountersStatEx loginchatidstat;
        TCountersStatEx uniquechatsbyloginstat;
    };

#define MAX_HOST_COUNT 30

    struct TTxtData {
        TString m_text;
        ui64 m_shingle{};
        TCountersStatEx m_service_data;
        TCountersStatEx m_common_data;

        TTxtData() = default;
        TTxtData(TString text) noexcept
            : m_text(std::move(text))
            , m_shingle(FnvHash<ui64>(m_text)) {
        }

        ui64 GetShingle() {
            return m_shingle;
        }
    };

    struct TStorDataByHMP {
        TTxtData host_list[MAX_HOST_COUNT]{};
        size_t host_list_size{};
        TTxtData mail_list[MAX_HOST_COUNT]{};
        size_t mail_list_size{};
        TTxtData phone_list[MAX_HOST_COUNT]{};
        size_t phone_list_size{};
    };

    struct TUpdateCacheStruct {
        bool m_is_longip;
        TString m_NumbRequest;
        stordata::TShingleDataList m_srvc_shingles_list;
        stordata::RESTYPE m_Spam;
        int m_rcpt_count;
        stordata::TLongIPCnt m_cnt_type;

        TUpdateCacheStruct() {
            Clear();
        }

        TUpdateCacheStruct(const TString& NumbRequest, const stordata::TShingleDataList& srvc_shingles_list, stordata::RESTYPE Spam, int rcpt_count) {
            Clear();

            m_is_longip = false;
            m_NumbRequest = NumbRequest;

            auto it = srvc_shingles_list.begin();
            while (it != srvc_shingles_list.end()) {
                m_srvc_shingles_list.push_back(*it);

                ++it;
            }

            m_Spam = Spam;
            m_rcpt_count = rcpt_count;
        }

        TUpdateCacheStruct(const TString& NumbRequest, const stordata::TShingleData& data, stordata::TLongIPCnt cnt_type) {
            Clear();

            m_is_longip = true;
            m_NumbRequest = NumbRequest;

            m_srvc_shingles_list.push_back(data);

            m_cnt_type = cnt_type;
        }

        void Clear() {
            m_is_longip = false;
            m_NumbRequest = "";
            m_srvc_shingles_list.clear();
            m_Spam = stordata::RT_HAM;
            m_rcpt_count = 0;
            m_cnt_type = stordata::LIP_HAM;
        }
    };

} // namespace stordata
