package ru.yandex.dkimpp.prepare;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Locale;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Pattern;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.parser.email.MailAliases;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.url.processor.CommonZoneExtractor;

public class DkimppPrepare {
    private static final Pattern FROM_PATTERN =
        Pattern.compile(
            "[\\p{Alnum}+=._-]+@[\\p{Alnum}._-]+[.][\\p{Alpha}][\\p{Alnum}-]+",
            Pattern.UNICODE_CHARACTER_CLASS);
    private static final Pattern HOST_PATTERN =
        Pattern.compile(
            "[\\p{Alnum}._-]+[.][\\p{Alpha}][\\p{Alnum}-]+",
            Pattern.UNICODE_CHARACTER_CLASS);

    private final Set<String> dkimDomains = new TreeSet<>();
    private final QueryConstructor query = new QueryConstructor("/");
    private final BasicGenericConsumer<JsonObject, ? extends JsonException>
        consumer;
    private final JsonParser parser;

    public DkimppPrepare() {
        consumer = new BasicGenericConsumer<>();
        parser = TypesafeValueContentHandler.prepareParser(consumer);
    }

    public void processLine(final String line) throws Exception {
        long timestamp;
        long count;
        String from;
        String cleanDkim;
        try {
            parser.parse(line);
            JsonMap map = consumer.get().asMap();
            timestamp = map.get("day").asLong();
            count = map.get("repeat").asLong();
            from = map.get("fromaddr").asString();
            cleanDkim =
                map.get("clean_dkim").asString().toLowerCase(Locale.ROOT);
        } catch (JsonException e) {
            System.err.println("Skipping malformed line: " + line);
            return;
        }

        int len = from.length();
        if (len == 0) {
            System.err.println("Skipping line with empty from: " + line);
            return;
        }
        if ((from.charAt(0) == '"' && from.charAt(len - 1) == '"')
            || (from.charAt(0) == '\'' && from.charAt(len - 1) == '\'')
            || (from.charAt(0) == '<' && from.charAt(len - 1) == '>'))
        {
            from = from.substring(1, len - 1);
        }
        if (!FROM_PATTERN.matcher(from).matches()) {
            System.err.println("Skipping line with malformed from: " + line);
            return;
        }
        from = MailAliases.INSTANCE.normalizeEmail(from);
        dkimDomains.clear();
        for (String domain: cleanDkim.split(" ")) {
            if (!domain.isEmpty()) {
                if (!HOST_PATTERN.matcher(domain).matches()) {
                    System.err.println(
                        "Skipping line with malformed dkim domains: " + line);
                    return;
                }
                dkimDomains.add(CommonZoneExtractor.INSTANCE.apply(domain));
            }
        }
        query.sb().setLength(0);
        query.sb().append("/api/async/so/update-dkim-stats?timestamp=");
        query.sb().append(timestamp);
        query.append("count", count);
        query.append("from", from);
        for (String domain: dkimDomains) {
            query.append("dkim-domain", domain);
        }
        System.out.println(query.sb().toString());
    }

    public static void main(final String... args) throws Exception {
        try (BufferedReader reader =
                new BufferedReader(
                    new InputStreamReader(
                        System.in,
                        StandardCharsets.UTF_8)))
        {
            long count = 0L;
            DkimppPrepare preparator = new DkimppPrepare();
            while (true) {
                String line = reader.readLine();
                if (line == null) {
                    break;
                }
                preparator.processLine(line);
                if ((++count % 100000L) == 0L) {
                    System.err.println(count + " lines processed");
                }
            }
        }
    }
}

