#include <util/network/init.h>
#include "tfcgiserverbase.h"

//**********************************************************************************
//                               TKBridge
//**********************************************************************************

TKBridge::TKBridge()
    : m_stop(0)
    , m_midnight_heavy(0)
    , m_StopShedulerHeavy(1)
    , m_StopSheduler(1)

{
    m_Sheduler = NULL;
    m_ShedulerHeavy = NULL;

    m_StreamCount = 0;
    m_ExitStream = false;
    m_srvcobj = NULL;
    m_server = NULL;
    m_pid = 0;

    m_stop_heavy = false;
    m_SleepTime = 1; // Seconds

    // Midnight
    memset(m_MidnightTimeStr, '\0', sizeof(m_MidnightTimeStr));
    struct tm* curtime = CShingleTime::GetTime();
    m_MidnightTime[DAY] = curtime->tm_yday;
    m_MidnightTime[HOUR] = 0;
    m_MidnightTime[MINUTE] = 0;
}

TKBridge::~TKBridge() {
}

bool TKBridge::Init(ui32 queuesize, TServiceObjectBase* srvcobjA, void* server) {
    bool res = true;

    m_srvcobj = srvcobjA;
    m_server = server;

    if ((m_srvcobj != NULL) && (m_srvcobj->GetConfigObject() != NULL)) {
        m_SleepTime = m_srvcobj->GetConfigObject()->ReadInteger("server", "sleep_time", 5);
        strncpy(m_MidnightTimeStr, m_srvcobj->GetConfigObject()->ReadStroka("server", "midnight_time", "00:00").c_str(), CShingleTime::MAX_TIME_SIZE);
    }

    if (!SetMidnightTime()) {
        WriteToLog(KERROR, "Can't parse midnight time.");
        res = false;
    }

    return res;
}

void ShedulerProcM(void* par) {
    ui32 writedumpdelay = 0;

    TKBridge* ss = (TKBridge*)par;

    if (ss != NULL) {
        ss->WriteThreadIDToLog();

        while (!ss->ShedulerShouldStop()) {
            ss->Sleep();

            ss->CheckMidnight();
            ss->EventTick();
        }

        ss->ShedulerStopped(true);
    }
}

void ShedulerHeavyProcM(void* par) {
    ui32 writedumpdelay = 0;

    TKBridge* ss = (TKBridge*)par;

    if (ss != NULL) {
        ss->WriteThreadIDHeavyToLog();

        while (!ss->ShedulerHeavyShouldStop()) {
            ss->Sleep();
            ss->CheckMidnightHeavy();
            ss->EventTickHeavy();
        }

        ss->ShedulerHeavyStopped(true);
    }
}

void TKBridge::Sleep() {
    ::sleep(m_SleepTime);
}

bool TKBridge::SetMidnightTime() {
    char *b, *e;
    b = m_MidnightTimeStr;
    e = strchr(m_MidnightTimeStr, ':');

    if (NULL == e)
        return false;
    m_MidnightTime[HOUR] = (ui8)strtoul(b, &e, 10);
    b = e + 1;
    m_MidnightTime[MINUTE] = (ui8)strtoul(b, &e, 10);

    return true;
}

void TKBridge::StartSheduler() {
    AtomicSwap(&m_StopSheduler, 0);
    m_Sheduler = new TThread((TThread::TThreadProc)&ShedulerProcM, this);
    m_Sheduler->Start();
}

void TKBridge::StartShedulerHeavy() {
    AtomicSwap(&m_StopShedulerHeavy, 0);
    m_ShedulerHeavy = new TThread((TThread::TThreadProc)&ShedulerHeavyProcM, this);
    m_ShedulerHeavy->Start();
}

void TKBridge::StopSheduler() {
    AtomicSwap(&m_StopSheduler, 1);
}

void TKBridge::StopShedulerHeavy() {
    AtomicSwap(&m_StopShedulerHeavy, 1);
}

void TKBridge::ShedulerStopped(bool Stopped) {
    AtomicSwap(&m_StopSheduler, Stopped ? 0 : 1);
}

void TKBridge::ShedulerHeavyStopped(bool Stopped) {
    AtomicSwap(&m_StopShedulerHeavy, Stopped ? 0 : 1);
}

bool TKBridge::ShedulerStopped() {
    return !m_StopSheduler;
}

bool TKBridge::ShedulerHeavyStopped() {
    return !m_StopShedulerHeavy;
}

ui32 TKBridge::IncrementStreamCount() {
    return AtomicIncrement(m_StreamCount);
}

void TKBridge::DecrementStreamCount() {
    AtomicDecrement(m_StreamCount);
}

ui32 TKBridge::GetStreamCount() {
    return m_StreamCount;
}

void TKBridge::SetStartStream() {
    AtomicSwap(&m_ExitStream, 0);
}

void TKBridge::SetExitStream() {
    AtomicSwap(&m_ExitStream, 1);
}

bool TKBridge::GetExitStream() {
    return m_ExitStream;
}

void TKBridge::Close() {
    StopSheduler();
    while (!ShedulerStopped())
        ::sleep(1);
    if (m_Sheduler != NULL) {
        delete m_Sheduler;
        m_Sheduler = NULL;
    }

    StopShedulerHeavy();

    while (!ShedulerHeavyStopped())
        ::sleep(1);
    if (m_ShedulerHeavy != NULL) {
        delete m_ShedulerHeavy;
        m_ShedulerHeavy = NULL;
    }
}

void TKBridge::CheckMidnight() {
    struct tm* curtime = CShingleTime::GetTime();

    if ((curtime->tm_yday != m_MidnightTime[DAY]) &&
        ((curtime->tm_hour > m_MidnightTime[HOUR]) ||
         ((curtime->tm_hour == m_MidnightTime[HOUR]) &&
          (curtime->tm_min >= m_MidnightTime[MINUTE])))) {
        m_MidnightTime[DAY] = curtime->tm_yday;

        AtomicSwap(&m_midnight_heavy, 1);

        if ((m_srvcobj != NULL) && (m_srvcobj->GetGeneralObject() != NULL))
            m_srvcobj->GetGeneralObject()->Midnight();
    }
}

void TKBridge::EventTick() {
    if ((m_srvcobj != NULL) && (m_srvcobj->GetGeneralObject() != NULL))
        m_srvcobj->GetGeneralObject()->EventTick();
}

void TKBridge::EventTickHeavy() {
    if ((m_srvcobj != NULL) && (m_srvcobj->GetGeneralObject() != NULL))
        m_srvcobj->GetGeneralObject()->EventTickHeavy();
}

void TKBridge::CheckMidnightHeavy() {
    if (AtomicCas(&m_midnight_heavy, 0, 1)) {
        if ((m_srvcobj != NULL) && (m_srvcobj->GetGeneralObject() != NULL))
            m_srvcobj->GetGeneralObject()->MidnightHeavy();
    }
}

void TKBridge::Shutdown() {
    AtomicSwap(&m_stop, 1);

    if (m_server != NULL)
        ((TFCGIServerBase*)m_server)->SendSIGUSR1();
}

bool TKBridge::WritePid() {
    bool res = false;
    FILE* fp = NULL;
    TString pidfilename = "";

    m_pid = getpid();
    if ((m_srvcobj != NULL) && (m_srvcobj->GetConfigObject()))
        pidfilename = m_srvcobj->GetConfigObject()->ReadStroka("server", "pid", "");

    if (!pidfilename.empty()) {
        fp = fopen(pidfilename.c_str(), "w");
        if (NULL == fp) {
            WriteToLog(KERROR, "Can't write process PID to file '%s' ", pidfilename.c_str());
        } else {
            //fprintf(fp, "%d", getpid());
            fprintf(fp, "%d", m_pid);
            fclose(fp);
            res = true;
        }
    } else {
        WriteToLog(KERROR, "Can't write process PID to file: filename no defined.");
    }

    return res;
}

void TKBridge::WriteToLog(const TLogStatus LogLevel, const char* msg, ...) {
    if ((m_srvcobj != NULL) && (m_srvcobj->GetLogsGroup() != NULL) && (m_srvcobj->GetLogsGroup()->GetServerLog() != NULL)) {
        char tbuff[4096];
        TString rid = "S";
        va_list args;

        memset(tbuff, 0, sizeof(tbuff));
        va_start(args, msg);
        vsnprintf(tbuff, sizeof(tbuff) - 1, msg, args);
        va_end(args);

        switch (LogLevel) {
            case KMESSAGE:
                m_srvcobj->GetLogsGroup()->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "%17s\t%s", rid.c_str(), tbuff);
                break;
            case KWARNING:
                m_srvcobj->GetLogsGroup()->GetServerLog()->WriteMessageAndDataStatus(KWARNING, "%17s\t%s", rid.c_str(), tbuff);
                break;
            case KERROR:
                m_srvcobj->GetLogsGroup()->GetServerLog()->WriteMessageAndDataStatus(KERROR, "%17s\t%s", rid.c_str(), tbuff);
                break;
            default:
                break;
        }
    }
}

void TKBridge::WriteThreadIDToLog() {
    int thread_id = -1;

#ifndef _win_
    thread_id = syscall(SYS_gettid);
#endif

    if ((m_srvcobj != NULL) && (m_srvcobj->GetLogsGroup() != NULL) && (m_srvcobj->GetLogsGroup()->GetDebugInfoLog() != NULL))
        m_srvcobj->GetLogsGroup()->GetDebugInfoLog()->WriteMessageAndDataStatus(KMESSAGE, "THREADID % 9d: MAIN_SHEDULER_THREAD", thread_id);
}

void TKBridge::WriteThreadIDHeavyToLog() {
    int thread_id = -1;

#ifndef _win_
    thread_id = syscall(SYS_gettid);
#endif

    if ((m_srvcobj != NULL) && (m_srvcobj->GetLogsGroup() != NULL) && (m_srvcobj->GetLogsGroup()->GetDebugInfoLog() != NULL))
        m_srvcobj->GetLogsGroup()->GetDebugInfoLog()->WriteMessageAndDataStatus(KMESSAGE, "THREADID % 9d: HEAVY_SHEDULER_THREAD", thread_id);
}

//**********************************************************************************
//                              TKThread
//**********************************************************************************

TString penv(const char* const* envp) {
    TString res = "";

    for (; *envp; ++envp)
        res = res + TString(*envp) + "<br>\n";

    return res;
}

ui32 penvh(const char* const* envp, THeaders& headers) {
    ui32 res = 0;

    headers.clear();
    for (; *envp; ++envp)
        headers.push_back(TString(*envp));

    res = headers.size();

    return res;
}

long gstdin(FCGX_Request* request, TVector<char>& content) {
    char* clenstr = FCGX_GetParam("CONTENT_LENGTH", request->envp);
    unsigned long clen = 0;

    if (clenstr) {
        clen = strtol(clenstr, &clenstr, 10);
        if (*clenstr) {
            FCGX_FPrintF(request->err, "can't parse \"CONTENT_LENGTH=%s\"\n", FCGX_GetParam("CONTENT_LENGTH", request->envp));
            clen = TFCGIServerBase::STDIN_MAX;
        }

        // *always* put a cap on the amount of data that will be read
        if (clen > TFCGIServerBase::STDIN_MAX)
            clen = TFCGIServerBase::STDIN_MAX;

        if (content.size() < clen + 1)
            content.resize(clen + 1);

        content[clen] = '\0';
        //memset(&content[0], 0, clen + 1);

        ui32 contlen = 0;
        if (clen > contlen) {
            const char* pstr = FCGX_GetLine(&content[contlen], clen - contlen + 1, request->in);
            while ((pstr != NULL) && (*pstr)) {
                contlen += strlen(pstr);
                if (clen > contlen)
                    pstr = FCGX_GetLine(&content[contlen], clen - contlen + 1, request->in);
                else
                    break;
            }
        }

        //request->in.read(*content, clen);
        //clen = request->in.gcount();
    } else {
        // *never* read stdin when CONTENT_LENGTH is missing or unparsable
        if (content.empty())
            content.resize(1);
        content[0] = '\0';
        clen = 0;
    }

    // Chew up any remaining stdin - this shouldn't be necessary
    // but is because mod_fastcgi doesn't handle it correctly.

    // ignore() doesn't set the eof bit in some versions of glibc++
    // so use gcount() instead of eof()...
    //do request->in.ignore(1024); while (request->in.gcount() == 1024);

    return clen;
}

void FillHeaderHash(FCGX_Request* request, TWStrokaHash& header_hash) {
    static const char* params[] = {
        "REQUEST_METHOD",
        "REMOTE_ADDR",
        "SERVER_PROTOCOL",
        "REQUEST_URI",
        "Authorization",
        "CONTENT_TYPE",
        "CONTENT_LENGTH",
        "HTTP_COOKIE",
        "HTTP_X_SO_RQSTIDENT",
        "HTTP_X_SO_MESSID"};

    static const size_t paramsNum = sizeof(params) / sizeof(params[0]);

    for (size_t i = 0; i < paramsNum; i++) {
        const char* s = params[i];
        const char* p = FCGX_GetParam(s, request->envp);
        if (p)
            header_hash[s] = p;
    }
}

#ifdef WIN32
void /*__stdcall*/ thread_body(void* obj)
#else
void thread_body(void* obj)
#endif
{
    TString text = "";
    THeaders Headers;
    TString testtxt = "";
    TWStrokaHash header_hash;
    int err = 0;
    TMakeRqstDelays work_delays;
    ui32 tick = 0;

    TKThread* thread_obj = (TKThread*)obj;
    if (!thread_obj)
        return;

    thread_obj->SetThreadHandle();

    TKBridge* bridge = thread_obj->Bridge();
    if (!bridge)
        return;

    TFCGIServerBase* server = (TFCGIServerBase*)bridge->GetFastServerLink();
    if (!server)
        return;

    const ui32 number_stream = bridge->IncrementStreamCount();
    server->WriteThreadIDToLog(number_stream);

    TMakeRequestBase* mr = server->CreateMakeRequest();

    FCGX_Request request;
    server->InitRequest(request);

    TVector<char> content;

    while (!bridge->IsStop()) {
        if (!server->GetRequest(err, bridge, number_stream, request)) {
            usleep(1000);
            continue;
        }

        work_delays.m_fulltick = CShingleTime::GetMs();
        text = "";
        Headers.clear();
        work_delays.m_gstdin = CShingleTime::GetMs();
        unsigned long clen = gstdin(&request, content);
        work_delays.m_gstdin = CShingleTime::GetMs() - work_delays.m_gstdin;

        header_hash.clear();
        work_delays.m_fillheaders = CShingleTime::GetMs();
        FillHeaderHash(&request, header_hash);
        work_delays.m_fillheaders = CShingleTime::GetMs() - work_delays.m_fillheaders;

        work_delays.m_reply = CShingleTime::GetMs();
        if (mr != NULL)
            mr->Reply(&request, &header_hash, &content[0], clen, number_stream, &work_delays);
        work_delays.m_reply = CShingleTime::GetMs() - work_delays.m_reply;

        FCGX_Finish_r(&request);

        work_delays.m_fulltick = CShingleTime::GetMs() - work_delays.m_fulltick;

        if (server->GetServiceObject().GetLogsGroup() && server->GetServiceObject().GetLogsGroup()->GetLongRequestLog()) {
            if (work_delays.m_fulltick >= server->GetServiceObject().GetLogsGroup()->GetLongrequestTreshold()) {
                switch (server->GetServiceObject().GetLogsGroup()->GetLongrequestMode()) {
                    case 0: //no write

                        break;
                    case 1: //short mode
                        server->GetServiceObject().GetLogsGroup()->GetLongRequestLog()->WriteMessageAndDataStatus(KMESSAGE, "%u %s %s srvc='%s' dop='%s'", work_delays.m_fulltick, work_delays.m_remote_ip.c_str(), work_delays.m_request_id.c_str(), work_delays.GetServiceTicks().c_str(), work_delays.m_dop_traccert.c_str());
                        break;
                    case 2: //full mode

                        break;
                };
            }
        }

        if (server->GetServiceObject().GetGeneralObject())
            server->GetServiceObject().GetGeneralObject()->AddINTick(work_delays);
    }

    //if (mr != NULL)
    //{
    //   delete mr;
    //   mr = NULL;
    //}
    server->DeleteMakeRequest(&mr);

    bridge->DecrementStreamCount();
}

TKThread::TKThread(TKBridge* bridge, int thread_number) {
    m_ThreadHandle = 0;
    m_bridge = bridge;
    m_thread_number = thread_number;
    m_thread = new TThread((TThread::TThreadProc)&thread_body, this);
}

TKThread::~TKThread() {
    if (m_thread != NULL) {
        delete m_thread;
        m_thread = NULL;
    }
}

void TKThread::Start() {
    if (m_thread != NULL)
        m_thread->Start();
}

void TKThread::SetThreadHandle() {
#ifndef _win_
    m_ThreadHandle = pthread_self();
#endif
}

//**********************************************************************************
//                              TStreamPool
//**********************************************************************************

TStreamPool::TStreamPool() {
    count = 1;
    bridge = NULL;
}

TStreamPool::~TStreamPool() {
    TStreamPoolCIt it;

    if (bridge != NULL) {
        bridge->SetExitStream();

        while (bridge->GetStreamCount() > 0)
            usleep(10000);

        it = Pool.begin();
        while (it != Pool.end()) {
            if ((*it) != NULL) {
                delete (*it);
                (*it) = NULL;
            }
            ++it;
        }
        Pool.clear();
    }
}

void TStreamPool::Wait() {
    if (bridge != NULL) {
        if (bridge->GetSrvcObj() != NULL) {
            ui32 streamcount = 0;

            streamcount = bridge->GetSrvcObj()->StreamCount();
            while (bridge->GetStreamCount() != streamcount)
                usleep(10000);
        }

        while (bridge->GetStreamCount() > 0) {
            usleep(10000);
        }
    }
}

void TStreamPool::Init(int countA, TKBridge* bridgeA) {
    TStreamPoolCIt it;

    count = countA;
    bridge = bridgeA;

    if (bridge != NULL) {
        bridge->SetStartStream();
        for (int i = 0; i < count; i++) {
            TKThread* tp = new TKThread(bridge, i);
            Pool.push_back(tp);
        }

        it = Pool.begin();
        while (it != Pool.end()) {
            if ((*it) != NULL)
                (*it)->Start();

            ++it;
        }
    }
}

void TStreamPool::SendSIGUSR1() {
#ifndef _win_
    TStreamPoolCIt it;
    pthread_t pth = NULL;

    it = Pool.begin();
    while (it != Pool.end()) {
        if ((*it) != NULL) {
            pth = (*it)->ThreadHandle();
            pthread_kill(pth, SIGUSR1);
        }

        ++it;
    }
#endif
}

void TStreamPool::Close() {
    TStreamPoolCIt it;

    if (bridge != NULL) {
        bridge->SetExitStream();

        SendSIGUSR1();

        while (bridge->GetStreamCount() > 0)
            usleep(10000);

        it = Pool.begin();
        while (it != Pool.end()) {
            if ((*it) != NULL) {
                delete (*it);
                (*it) = NULL;
            }
            ++it;
        }
        Pool.clear();
    }
}

ui32 TStreamPool::ThreadAMount() {
    return Pool.size();
}

//**********************************************************************************
//                               TFCGIServerBase
//**********************************************************************************

TFCGIServerBase::~TFCGIServerBase() {
    if (pool != NULL) {
        delete pool;
        pool = NULL;
    }
    if (m_listen_socket >= 0)
        closesocket(m_listen_socket);
}

bool TFCGIServerBase::InitBeforeFork() {
    bool res = false;
    TString port = "";

#ifndef _win_
    main_thread_id = pthread_self();
#endif

    //if (maino.Init(ServiceObject, this))
    //{
    if (ServiceObject.GetConfigObject() != NULL) {
        m_streamcount = ServiceObject.StreamCount();
        m_queuesize = ServiceObject.QueueSize();
        m_port = ServiceObject.Port();
        m_listenQueueBacklog = ServiceObject.ListenQueueBackLog();
        if (!m_port.empty()) {
            if (bridge.Init(m_queuesize, &ServiceObject, this)) {
                FCGX_Init();

                if (atoi(m_port.c_str()) > 0)
                    port = ":" + m_port;
                else
                    port = m_port;
                m_listen_socket = FCGX_OpenSocket(port.c_str(), m_listenQueueBacklog); //open new socket

                if (m_listen_socket >= 0) {
                    res = true;
                    WriteToLog(KMESSAGE, "FCGI start on port %s - OK.", port.c_str());
                    printf("FCGI start on port %s - OK.\n", port.c_str());
                } else {
                    WriteToLog(KERROR, "FCGI: error listen socket, port %s.", port.c_str());
                    printf("FCGI: error listen socket, port %s.\n", port.c_str());
                }

                //pool = new TStreamPool();
                //pool->Init(m_streamcount, &bridge);
            }

        } else {
            WriteToLog(KERROR, "FCGI: undefined port.");
            printf("FCGI: undefined port.\n");
        }
    }
    //}

    return res;
}

void TFCGIServerBase::StartFCGIServerThreads() {
    bridge.StartSheduler();
    bridge.StartShedulerHeavy();

    if (m_streamcount == 0)
        m_streamcount = 1;
    pool = new TStreamPool();
    pool->Init(m_streamcount, &bridge);
}

void TFCGIServerBase::WriteToLog(const TLogStatus LogLevel, const char* msg, ...) {
    if ((ServiceObject.GetLogsGroup() != NULL) && (ServiceObject.GetLogsGroup()->GetServerLog() != NULL)) {
        char tbuff[4096];
        TString rid = "S";
        va_list args;

        memset(tbuff, 0, sizeof(tbuff));
        va_start(args, msg);
        vsnprintf(tbuff, sizeof(tbuff) - 1, msg, args);
        va_end(args);

        switch (LogLevel) {
            case KMESSAGE:
                ServiceObject.GetLogsGroup()->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "%17s\t%s", rid.c_str(), tbuff);
                break;
            case KWARNING:
                ServiceObject.GetLogsGroup()->GetServerLog()->WriteMessageAndDataStatus(KWARNING, "%17s\t%s", rid.c_str(), tbuff);
                break;
            case KERROR:
                ServiceObject.GetLogsGroup()->GetServerLog()->WriteMessageAndDataStatus(KERROR, "%17s\t%s", rid.c_str(), tbuff);
                break;
            default:
                break;
        }
    }
}

bool TFCGIServerBase::GetRequest(int& err, TKBridge* bridge, int potok_number, FCGX_Request& request) {
    err = 0;
    if ((bridge != NULL) && (!bridge->IsStop())) {
        while (!bridge->IsStop()) {
            err = FCGX_Accept_r(&request);
            if (err == 0) {
                return true;
            } else {
                if (err < 0)
                    err = -1 * err;

                if (err == EAGAIN)
                    usleep(1000);
                else {
                    FCGX_Finish_r(&request);

                    if (err != 9999) {
                        WriteToLog(KMESSAGE, "FCGI wait error %d (%s).", err, strerror(err));
                        printf("FCGI wait error %d - %s.\n", err, strerror(err));
                    }

                    break;
                    ;
                }
            }
        }
    }

    return false;
}

void TFCGIServerBase::Wait() {
    if (pool != NULL)
        pool->Wait();
}

void TFCGIServerBase::Close() {
    bridge.Close();
    WriteToLog(KMESSAGE, "FCGI stop - OK.");
    printf("FCGI stop - OK.\n");
}

bool TFCGIServerBase::WritePid() {
    return bridge.WritePid();
}

void TFCGIServerBase::SendSIGUSR1() {
#ifndef _win_
    pthread_t thread_id = pthread_self();

    pool->SendSIGUSR1();
    pthread_kill(main_thread_id, SIGUSR1);

/*if ( (thread_id == main_thread_id) && (pool != NULL) )
      pool->SendSIGUSR1();
   else
   {
      TString filename = "/home/kerrik/sigtxt_" + IntToStroka(thread_id);
      FILE *handle = fopen(filename.c_str(), "a+b");
      if (handle != NULL)
      {
         fprintf(handle, "SIGUSR1-THREAD (%u)\n", thread_id);
         fclose(handle);
      }
   }*/
#endif
}

void TFCGIServerBase::Shutdown() {
    FCGX_ShutdownPending();

    if (m_listen_socket >= 0)
        closesocket(m_listen_socket);
    m_listen_socket = -1;

    bridge.Shutdown();
}

bool TFCGIServerBase::InitAfterFork() {
    bool res = false;

    res = ServiceObject.InitAfterFork();
    StartFCGIServerThreads();

    return res;
}

void TFCGIServerBase::WriteThreadIDToLog(int thread_index) {
    int thread_id = -1;

#ifndef _win_
    thread_id = syscall(SYS_gettid);
#endif

    if ((ServiceObject.GetLogsGroup() != NULL) && (ServiceObject.GetLogsGroup()->GetDebugInfoLog() != NULL))
        ServiceObject.GetLogsGroup()->GetDebugInfoLog()->WriteMessageAndDataStatus(KMESSAGE, "THREADID % 9d: ACCEPTTHREAD THREAD_INDEX_%02d", thread_id, thread_index);
}

void TFCGIServerBase::InitRequest(FCGX_Request& request) const {
    FCGX_InitRequest(&request, m_listen_socket, FCGI_FAIL_ACCEPT_ON_INTR);
}
