#include "tfcgiserverbase2.h"

#include <contrib/libs/fcgi/fcgiapp.h>

#include <util/datetime/base.h>

namespace NFCGI {
    struct TRequest: public FCGX_Request {
        ~TRequest() {
            FCGX_Finish_r(this);
            FCGX_Free(this, 1);
        }
    };

    struct ProcessTask : IObjectInQueue {
        void Process(void* /*ThreadSpecificResource*/) override {
            THolder<ProcessTask> t(this);
            server.Handle(*request);
        }

        ProcessTask(TServer& server, TSimpleSharedPtr<TRequest> request)
            : server(server)
            , request(request)
        {
        }

        TServer& server;
        TSimpleSharedPtr<TRequest> request;
    };

    void TServer::DoExecute() {
        while (!stop) {
            TSimpleSharedPtr<TRequest> request(new TRequest);

            int res = FCGX_InitRequest(request.Get(), socket, FCGI_FAIL_ACCEPT_ON_INTR);
            if (res != 0) {
                logger << "cannot init request: " << strerror(res) << '\n';
                continue;
            }
            if (!Accept(*request)) {
                continue;
            }

            if (processQueue.Add(new ProcessTask(*this, request))) {
            } else {
                FCGX_FPrintF(request->err, "queue is full");
                logger << "queue is full" << '\n';
            }
        }
    }

    void TServer::Listen(const TString& url) {
        socket = FCGX_OpenSocket(url.c_str(), settings.backLog);
        if (socket < 0)
            throw Error(socket) << "cannot connect to " << url;

        threads.clear();
        for (size_t i = 0; i < settings.listenThreads - 1; i++)
            threads.push_back(SystemThreadFactory()->Run(this));

        this->DoExecute();

        for (size_t i = 0; i < threads.size(); i++)
            threads[i]->Join();
    }

    void TServer::Stop() {
        AtomicSwap(&stop, 1);
    }

    void TServer::SetLogger(TAutoPtr<TLogBackend> back) {
        logger.ResetBackend(back);
    }

    bool TServer::Accept(FCGX_Request& request) {
        int err = FCGX_Accept_r(&request);
        while (err != 0) {
            if (err < 0)
                err = -1 * err;

            if (err == 11)
                usleep(1000);
            else {
                logger << strerror(err) << '\n';
                return false;
            }
            err = FCGX_Accept_r(&request);
        }
        return true;
    }

TServer::TServer(const TSettings & settings)
        :settings(settings)
        ,stop(false)
        ,socket(-1)
{
     FCGX_Init();
    processQueue.Start(settings.processThreads, settings.processQueueSize);
}

    TServer::~TServer() {
        Stop();
    }

} //namespace NFCGI
