#!/usr/bin/env python

import sys
import string
import itertools
import ConfigParser

# Monkey patching select method to use poll instead
import select
import pollselect
select.select = pollselect.select

from flup.server.fcgi import WSGIServer
from weightupdater import WeightUpdater
from fcgisender import FcgiSender
from nginxupdater import NginxUpdater
from logger import configurelog, action, error, TRACE_ID, ACCESS_ID, ERROR_ID, ACTION_ID


class Config:
    def __init__(self, filename):
        self.filename = filename

    def load(self):
        try:
            config = ConfigParser.SafeConfigParser()
            config.read(sys.argv[1])

            self.FASTCGI_PORT = config.getint("common", "port")
            self.HOSTS = filter(len, map(string.strip, config.get("common", "hosts").split(",")))
            self.SECONDARY_HOSTS = filter(len, map(string.strip, config.get("common", "secondary_hosts").split(",")))
            self.MAX_THREAD_COUNT = config.getint("common", "threads")

            self.ACTIONLOG = config.get("logs", "action")
            self.ACCESSLOG = config.get("logs", "access")
            self.TRACELOG = config.get("logs", "trace")
            self.ERRORLOG = config.get("logs", "error")

            self.SENDER_SOCKET_TIMEOUT = config.getfloat("sender", "socket_timeout")
            self.SENDER_QUEUE_SIZE = config.getint("sender", "queue_size")
            self.SENDER_APPEND_PARAM = config.get("sender", "append_param")

            self.WEIGHT_SOCKET_TIMEOUT = config.getfloat("weight", "socket_timeout")
            self.WEIGHT_CHECK_PERIOD = config.getfloat("weight", "check_period")
            self.WEIGHT_CHECK_QUERY = config.get("weight", "check_query")

            self.NGINX_CONFIG_TEMPLATE = config.get("nginx", "config_template")
            self.NGINX_CONFIG_DESTINATION = config.get("nginx", "config_destination")
            self.NGINX_RELOAD_CMD = config.get("nginx", "reload_cmd")
            return True
        except ConfigParser.Error, e:
            print >> sys.stderr, "Error: cannot parse config file - ", e
            return False


class FcgiHandler(object):
    def __init__(self, hosts, secondary_hosts, sender_queuesize, timeout, append_param):
        self.hosts_active = []
        self.new_hosts_active = []
        self.senders = {}
        self.append_param = append_param
        self.secondary_hosts = secondary_hosts

        for host in itertools.chain(hosts, secondary_hosts):
            self.senders[host] = FcgiSender(host, sender_queuesize, timeout)

    def handle_weight(self, prefered, others):
        self.new_hosts_active = others

    def __call__(self, environ, start_response):
        if self.new_hosts_active:
            self.hosts_active = self.new_hosts_active
            self.new_hosts_active = []

        if self.append_param:
            environ["REQUEST_URI"] += "&" + self.append_param if environ["REQUEST_URI"].find("?") >= 0 else "?"
            environ["QUERY_STRING"] += "&" + self.append_param if environ["QUERY_STRING"] else self.append_param

        for host in itertools.chain(self.hosts_active, self.secondary_hosts):
            self.senders[host].add(environ)

        start_response("200 OK", [("Content-type", "text/html")])
        return [""]


def main():
    cfg = Config(sys.argv[1])
    if not cfg.load():
        print "Cannot load config file"
        exit(1)

    configurelog(ACTION_ID, cfg.ACTIONLOG)
    configurelog(ACCESS_ID, cfg.ACCESSLOG)
    configurelog(TRACE_ID, cfg.TRACELOG)
    configurelog(ERROR_ID, cfg.ERRORLOG)

    print "nginx_balanced on port %d started" % cfg.FASTCGI_PORT

    fcgi_handler = FcgiHandler(cfg.HOSTS, cfg.SECONDARY_HOSTS, cfg.SENDER_QUEUE_SIZE, cfg.SENDER_SOCKET_TIMEOUT, cfg.SENDER_APPEND_PARAM)
    nginx_updater = NginxUpdater(cfg.FASTCGI_PORT, cfg.NGINX_CONFIG_TEMPLATE, cfg.NGINX_CONFIG_DESTINATION, cfg.NGINX_RELOAD_CMD)

    wu = WeightUpdater(cfg.HOSTS, cfg.WEIGHT_CHECK_QUERY, cfg.WEIGHT_CHECK_PERIOD, cfg.WEIGHT_SOCKET_TIMEOUT)
    wu.add_handler(fcgi_handler)
    wu.add_handler(nginx_updater)
    wu.start()

    wsgi_server = WSGIServer(fcgi_handler, bindAddress=("", cfg.FASTCGI_PORT), maxThreads=cfg.MAX_THREAD_COUNT)

    action("nginx_balanced on port %d started" % cfg.FASTCGI_PORT)

    while True:
        if not cfg.load():
            error("Cannot load config file during reload")
        configurelog(ACTION_ID, cfg.ACTIONLOG)
        configurelog(ACCESS_ID, cfg.ACCESSLOG)
        configurelog(TRACE_ID, cfg.TRACELOG)
        configurelog(ERROR_ID, cfg.ERRORLOG)

        action("nginx_balanced on port %d reloaded" % cfg.FASTCGI_PORT)
        wsgi_server.run()

if __name__ == "__main__":
    main()
