import os
import time
from threading import Thread
from fcgiapp import FCGIAppIpV6
from logger import logs, trace, error, TRACE_ID, ACTION_ID

null = open(os.devnull, "w+")


def send_fcgi_request(fcgi, host, port, url):
    env = {
        "REQUEST_METHOD": "GET",
        "GATEWAY_INTERFACE": "CGI/1.1",
        "SERVER_SOFTWARE": "ztc",
        "REDIRECT_STATUS": "200",
        "CONTENT_TYPE": "",
        "CONTENT_LENGTH": "0",
        "DOCUMENT_URI": url,
        "DOCUMENT_ROOT": "/",
        "REMOTE_ADDR": "",
        "REMOTE_PORT": "",
        "SERVER_ADDR": host,
        "SERVER_PORT": str(port),
        "SERVER_NAME": host,
        "SERVER_PROTOCOL": "HTTP/1.0"
    }
    parts = url.split("?")
    env["QUERY_STRING"] = parts[1] if len(parts) > 1 else ""
    env["SCRIPT_NAME"] = parts[0]
    env["SCRIPT_FILENAME"] = "/" + parts[0]
    env["REQUEST_URI"] = url
    env["PATH_INFO"] = parts[0]
    env["wsgi.input"] = null
    env["wsgi.errors"] = null

    data = {}

    def capture(newstatus, newheaders):
        data["status"] = newstatus
        data["headers"] = newheaders

    result = fcgi(env, capture)[0]
    return data["status"], data["headers"], result


def gethost(conn):
    if conn._connect:
        return conn._connect[0]
    return ""


def getport(conn):
    if conn._connect:
        return conn._connect[1]
    return -1


class WeightUpdater(Thread):
    def __init__(self, hosts, query_weight, period, timeout=5.0):
        Thread.__init__(self)
        self.daemon = True
        self.hosts = hosts
        self.query_weight = query_weight
        self.period = period
        self.timeout = timeout
        self.handlers = []

    def add_handler(self, handler):
        self.handlers.append(handler)

    def notify(self, preferred_host):
        for handler in self.handlers:
            handler.handle_weight(preferred_host, filter(lambda host: host != preferred_host, self.hosts))

    def run(self):
        if not self.hosts:
            return

        conns = {}
        for host in self.hosts:
            h, p = host.rsplit(":", 1)
            conns[host] = FCGIAppIpV6(host=h, port=int(p))

        current_upstream = ""

        while True:
            preferred = (0, self.hosts[0])
            for host in self.hosts:
                try:
                    conn = conns[host]
                    status = send_fcgi_request(conn, gethost(conn), getport(conn), self.query_weight)
                except Exception as e:
                    error("Error requesting weight from: %s. Exception: %s" % (host, str(e)))
                    continue

                code = status[0]
                answer = status[2].strip()

                if code and code.startswith("200"):
                    if unicode(answer).isnumeric():
                        preferred = max(preferred, (int(answer), host))
                        trace("Weight from: %s received = %s" % (host, answer))
                    else:
                        error("Error parsing weight from: %s. Weight is not a number: %s" % (host, answer))
                else:
                    error("Error requesting weight from: %s. Code: %s" % (host, str(code)))

            if current_upstream != preferred[1]:
                logs([TRACE_ID, ACTION_ID], "Preferred host changed: %s" % preferred[1])
                self.notify(preferred[1])

            current_upstream = preferred[1]

            time.sleep(self.period)
