#include <util/system/tls.h>

#include <iostream>
#include <iomanip>
#include <util/system/fs.h>
#include <library/cpp/digest/md5/md5.h>
#include <util/stream/buffer.h>
#include "tdatabasa_hash.h"

//*********************************************************************************************************************************
//                                                       TDataBasaHash
//*********************************************************************************************************************************

TDataBasaHash::TDataBasaHash() {
    m_basa_type_name = "HASH";
    m_storage_name = DATABASE_NAME;
}

TDataBasaHash::~TDataBasaHash() = default;

bool TDataBasaHash::InitBeforeForkNum(int index, const TString& server_hostname, const TString& server_code, const TString& server_start_time, const TString& server_version, TLogsGroup* LogsGroupA, TKConfig* configobjA) {
    bool res = InitBeforeFork_base(index, server_hostname, server_code, server_start_time, server_version, LogsGroupA, configobjA);
    const TString configSeed = index > 0 ? "storage_" + IntToStroka(index) : "postgre";

    LogsGroup = LogsGroupA;

    if (configobjA == NULL)
        return res;

    m_basa_type_name = "LONGSTORAGE-HASH";
    m_storage_name = "HASH";

    m_storage_status = "OK";
    m_storage_connect = true;

    if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL)) {
        LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "%s: inited - OK", m_basa_type_name.c_str());
        LogsGroup->GetServerLog()->FFlush();
    }

    return res;
}

bool TDataBasaHash::InitBeforeFork(const TString& server_hostname, const TString& server_code, const TString& server_start_time, const TString& server_version, TLogsGroup* LogsGroupA, TKConfig* configobjA) {
    return InitBeforeForkNum(-1, server_hostname, server_code, server_start_time, server_version, LogsGroupA, configobjA);
}

void TDataBasaHash::InitAfterFork() {
    InitAfterFork_base();
}

ui32 IncMax32m(const ui32 value, const i32 step) {
    ui32 res = 0;
    ui32 count = 0;
    ui32 stepn = 0;

    switch (step) {
        case 0:
            res = value;
            break;
        case 1:
            res = (value < 0xFFFFFFFF) ? (value + 1) : 0xFFFFFFFF;
            break;
        default:
            if (step > 0) {
                count = 0xFFFFFFFF - value;
                if (count > 0) {
                    stepn = (step > (i32)count) ? count : step;
                    res = value + stepn;
                } else
                    res = value;
            } else if (step < 0) {
                i32 step2 = -1 * step;

                if (step2 > (i32)value)
                    res = 0;
                else
                    res = value - step2;
            }
    };

    return res;
}

bool TDataBasaHash::StorageUpdate(ui64 shingle, int type, TShingleStorageType sstype, const nosql::HashMap& incrs, const nosql::HashMap& sets, bool& err) {
    bool res = false;
    TString collection_name = "";
    TString err_s = "";

    if ((!incrs.empty()) || (!sets.empty())) {
        collection_name = GetCollectionName(sstype, type);

        if (sstype == SST_LONGSTAT) {
            THashLongIPIt it;
            nosql::HashMap::const_iterator vit;
            nosql::HashMap::iterator vit2;
            frodo_st::TStorageDataItem frododata;
            frodo_st::TStorageDataItem fdtv;

            ui32 tick = CShingleTime::GetMs();
            {
                TWriteGuard g(m_Mutex);

                auto tp = sets.find(FIELD_TYPE);
                if (tp != sets.end()) {
                    it = m_data.find(shingle);
                    if (it != m_data.end()) {
                        //frododata = it->second;
                        if (static_cast<ui32>(it->second.m_stordata_type) == tp->second.Long()) {
                            if ((vit = sets.find(FIELD_LASTDAY)) != sets.end()) {
                                ui32 value = vit->second.Long();
                                it->second.m_lastday = kday_t(static_cast<unsigned long>(value));
                            }

                            if ((vit = sets.find(FIELD_HAM)) != sets.end()) {
                                ui32 value = vit->second.Long();
                                it->second.m_ham = value;
                            }

                            if ((vit = sets.find(FIELD_SPAM85)) != sets.end()) {
                                ui32 value = vit->second.Long();
                                it->second.m_spam85 = value;
                            }

                            if ((vit = sets.find(FIELD_SPAM100)) != sets.end()) {
                                ui32 value = vit->second.Long();
                                it->second.m_spam100 = value;
                            }

                            if ((vit = incrs.find(FIELD_HAM)) != incrs.end()) {
                                i32 value = vit->second.Long();
                                it->second.m_ham = IncMax32m(it->second.m_ham, value);
                            }

                            if ((vit = incrs.find(FIELD_SPAM85)) != incrs.end()) {
                                i32 value = vit->second.Long();
                                it->second.m_spam85 = IncMax32m(it->second.m_spam85, value);
                            }

                            if ((vit = incrs.find(FIELD_SPAM100)) != incrs.end()) {
                                i32 value = vit->second.Long();
                                it->second.m_spam100 = IncMax32m(it->second.m_spam100, value);
                            }

                            fdtv = it->second;

                            //FIELD_LASTDAY
                            //FIELD_TYPE
                            //FIELD_SOURCE
                            //FIELD_HAM
                            //FIELD_SPAM85
                            //FIELD_SPAM100
                            /*
                        TStorageDataType m_stordata_type;
                        char             m_data_sources[151];
                        kday_t           m_lastday;
                        ui32             m_ham;
                        ui32             m_spam85;
                        ui32             m_spam100
                        */

                        } else {
                            //collision
                        }

                        //frodo_st::TStorageDataItemSerialization::Serialize(frododata, hash);
                    } else {
                        nosql::HashMap both_hash;

                        vit = sets.begin();
                        while (vit != sets.end()) {
                            vit2 = both_hash.find(vit->first);
                            if (vit2 != both_hash.end())
                                vit2->second = vit->second;
                            else
                                both_hash[vit->first] = vit->second;

                            ++vit;
                        }

                        vit = incrs.begin();
                        while (vit != incrs.end()) {
                            vit2 = both_hash.find(vit->first);
                            if (vit2 != both_hash.end())
                                vit2->second = vit->second;
                            else
                                both_hash[vit->first] = vit->second;

                            ++vit;
                        }

                        frododata.Clear();
                        frodo_st::TStorageDataItemSerialization::Deserialize(both_hash, frododata);

                        m_data[shingle] = frododata;
                    }
                }
            }
            res = true;
            tick = CShingleTime::GetMs() - tick;
            AddStorageStat(sstype, SAT_UPDATE, 1, NStorageStats::SRS_OK, tick);

            IncrementStat(sstype, res);
        } else
            res = true;
    } else
        res = true;

    err = !res;

    return res;
}

bool TDataBasaHash::StorageFindOne(ui64 shingle, int type, TShingleStorageType sstype, nosql::HashMap& hash, bool& err) {
    bool res = true;
    TString table_name = GetCollectionName(sstype, type);

    if (sstype == SST_LONGSTAT) {
        THashLongIPIt it;
        frodo_st::TStorageDataItem frododata;

        hash.clear();
        ui32 tick = CShingleTime::GetMs();
        //sql::TResWithError<void> result = sql::PGFindOne(table_name.c_str(), tables[table_name], shingle, hash, FIELD_ID, ASYNC_MODE);
        {
            TReadGuard g(m_Mutex);
            it = m_data.find(shingle);
            if (it != m_data.end()) {
                frododata = it->second;
                frodo_st::TStorageDataItemSerialization::Serialize(frododata, hash);
            }
        }
        tick = CShingleTime::GetMs() - tick;
        AddStorageStat(sstype, SAT_FINDONE, 1, NStorageStats::SRS_OK, tick);
        res = true;
        IncrementStat(sstype, res);
    }

    err = !res;
    return res;
}

bool TDataBasaHash::StorageFind(TShingleStorageType /*sstype*/, TVector<nosql::HashMap>& hashes, bool& err) {
    auto it = hashes.begin();
    while (it != hashes.end())
        it->clear();
    err = false;
    return true;
}

bool TDataBasaHash::StorageErase(ui64 shingle, int type, TShingleStorageType sstype, bool& err) {
    bool res = true;
    TString collection_name = GetCollectionName(sstype, type);

    if (sstype == SST_LONGSTAT) {
        THashLongIPIt it;

        ui32 tick = CShingleTime::GetMs();
        {
            TWriteGuard g(m_Mutex);

            it = m_data.find(shingle);
            if (it != m_data.end()) {
                m_data.erase(it);
                res = true;
            }
        }
        tick = CShingleTime::GetMs() - tick;
        AddStorageStat(sstype, SAT_ERASE, 1, NStorageStats::SRS_OK, tick);

        IncrementStat(sstype, res);
    }

    err = !res;
    return res;
}

i64 TDataBasaHash::StorageSize(int type, TShingleStorageType sstype, bool& err) {
    i64 res = 0;
    TString collection_name = GetCollectionName(sstype, type);

    if (sstype == SST_LONGSTAT) {
        ui32 tick = CShingleTime::GetMs();

        {
            TReadGuard g(m_Mutex);
            res = static_cast<i64>(m_data.size());
        }

        tick = CShingleTime::GetMs() - tick;
        AddStorageStat(sstype, SAT_SIZE, 1, NStorageStats::SRS_OK, tick);

        IncrementStat(sstype, true);
    }

    err = false;
    return res;
}

void TDataBasaHash::StorageMultiAction(TStorageActionList& actlist, bool& err) {
    TStorageActionListIt it;
    ui32 tick = 0;

    it = actlist.begin();
    while (it != actlist.end()) {
        if ((*it).m_nosqldata != NULL) {
            tick = CShingleTime::GetMs();
            switch ((*it).m_action) {
                default:
                    break;
                case SAT_UPDATE:
                    StorageUpdate((*it).m_nosqldata->shingle, (*it).m_nosqldata->type, (*it).m_storage_type, (*it).m_nosqldata->incrs, (*it).m_nosqldata->sets, *(*it).m_err);
                    break;
                case SAT_FINDONE:
                    StorageFindOne((*it).m_nosqldata->shingle, (*it).m_nosqldata->type, (*it).m_storage_type, (*it).m_nosqldata->hash, *(*it).m_err);
                    break;
                case SAT_ERASE:
                    StorageErase((*it).m_nosqldata->shingle, (*it).m_nosqldata->type, (*it).m_storage_type, *(*it).m_err);
                    break;
            };
            tick = CShingleTime::GetMs() - tick;
            *(*it).m_tick = tick;

            if (*(*it).m_err)
                err = true;
        }

        ++it;
    }
}

TString TDataBasaHash::ReturnDriverProp() {
    TString res = "";

    //res += "<i><b>Driver properties:</b></i>&nbsp;&nbsp;";
    res += "<table border='0' width='100%' cellspacing='0' cellpadding='4'>";
    res += "<tr align='left'><td width='30%'>basa type</td><td>" + m_basa_type_name + "</td></tr>";
    res += "<tr align='left'><td width='30%'>basaname</td><td>" + m_storage_name + "</td></tr>";
    res += "<tr align='left'><td width='30%'>host</td><td>-</td></tr>";
    res += "<tr align='left'><td width='30%'>port</td><td>-</td></tr>";
    res += "<tr align='left'><td width='30%'>user</td><td>-</td></tr>";
    res += "<tr align='left'><td width='30%'>use password</td><td>-</td></tr>";
    res += "</table><br>";

    return res;
}

bool TDataBasaHash::CheckConnection(TString& status) {
    status = "OK";

    return true;
}

qustat::TDriverRqstPoolStat TDataBasaHash::ReturnLiveRqstStatMaster() {
    return qustat::TDriverRqstPoolStat();
}

qustat::TDriverRqstPoolStat TDataBasaHash::ReturnLiveRqstStatReplica() {
    return qustat::TDriverRqstPoolStat();
}
