
#include <util/system/tls.h>

#include <iostream>

#include "tdatabasa_postgre.h"

TAtomic TDataBasaPostgre::inited = 0;
bool TDataBasaPostgre::skipResponsesToBD = false;

//*********************************************************************************************************************************
//                                                       TDataBasaPostgre
//*********************************************************************************************************************************

TDataBasaPostgre::TDataBasaPostgre() {
    m_basa_type_name = "POSTGRE";
    m_storage_name = DATABASE_NAME;
    m_host = "127.0.0.1";
    m_user = "";
    m_pwd_define = false;
    m_port = 5432;
    skipResponsesToBD = false;
    m_pool_item_size_master = 32;
    m_wait_pool_item_ms_master = 50;
    m_pool_item_size_replica = 32;
    m_wait_pool_item_ms_replica = 50;
    m_rcv_timeout_ms = 100;
    if (m_wait_pool_item_ms_master > m_wait_pool_item_ms_replica)
        m_allow_full_time = m_wait_pool_item_ms_master + m_rcv_timeout_ms + 10;
    else
        m_allow_full_time = m_wait_pool_item_ms_replica + m_rcv_timeout_ms + 10;
    m_conn_string = "";
    m_conn_string_hidden = "";
}

TDataBasaPostgre::~TDataBasaPostgre() {
}

TString TDataBasaPostgre::CorrectInitPassword(const TString& init_str) {
    TString res = init_str;
    TString passwd_ident = "password=";

    if (!res.empty()) {
        const char* pb = nullptr;
        const char* pe = nullptr;
        TString passwd_ev_name = "";
        TString passwd_ev_res = "";
        size_t count = 0;

        pb = strstr(res.c_str(), passwd_ident.c_str());
        if (pb != nullptr) {
            pe = strchr(pb + 1, ' ');
            if (pe == nullptr)
                pe = res.c_str() + res.length();
            if (pe != nullptr) {
                count = pe - pb - passwd_ident.length();
                if (count > 0) {
                    passwd_ev_name = TString(pb + passwd_ident.length(), count);
                    if (!passwd_ev_name.empty()) {
                        if (passwd_ev_name[0] == '$') {
                            passwd_ev_res = GetEnv(TString(passwd_ev_name.c_str() + 1));
                            res.replace(pb + passwd_ident.length() - res.c_str(), count, passwd_ev_res.c_str());
                        }
                    }
                }
            }
        }
    }

    return res;
}

bool TDataBasaPostgre::InitBeforeForkNum(int index, const TString& server_hostname, const TString& server_code, const TString& server_start_time, const TString& server_version, TLogsGroup* LogsGroupA, TKConfig* configobjA) {
    bool res = InitBeforeFork_base(index, server_hostname, server_code, server_start_time, server_version, LogsGroupA, configobjA);

    const TString configSeed = index > 0 ? "storage_" + IntToStroka(index) : "postgre";
    LogsGroup = LogsGroupA;

    if (configobjA == NULL)
        return res;

    if (!AtomicSwap(&inited, 1)) {
        TString setup_log = "";

        m_basa_type_name = Trim(configobjA->ReadStroka(configSeed, "stortype", "POSTGRE"));
        m_basa_type_name.to_upper();

        m_storage_name = configobjA->ReadStroka(configSeed, "basaname", DATABASE_NAME);
        m_host = configobjA->ReadStroka(configSeed, "host", "127.0.0.1");
        m_user = configobjA->ReadStroka(configSeed, "user", "postgres");
        m_pwd_define = configobjA->KeyExists(configSeed, "password");
        TString pwd = configobjA->ReadStroka(configSeed, "password", "");
        m_port = configobjA->ReadUI64(configSeed, "port", 5432);
        skipResponsesToBD = configobjA->ReadBool(configSeed, "skipResponsesToBD", false);
        m_pool_item_size_master = configobjA->ReadInteger(configSeed, "pool_item_size_master", 32);
        m_wait_pool_item_ms_master = configobjA->ReadInteger(configSeed, "wait_pool_item_ms_master", 50);
        m_pool_item_size_replica = configobjA->ReadInteger(configSeed, "pool_item_size_replica", 32);
        m_wait_pool_item_ms_replica = configobjA->ReadInteger(configSeed, "wait_pool_item_ms_replica", 50);
        m_rcv_timeout_ms = configobjA->ReadInteger(configSeed, "rcv_timeout_ms", 100);
        m_conn_string = configobjA->ReadStroka(configSeed, "conn_string", "");

        if (m_wait_pool_item_ms_master > m_wait_pool_item_ms_replica)
            m_allow_full_time = m_wait_pool_item_ms_master + m_rcv_timeout_ms + 10;
        else
            m_allow_full_time = m_wait_pool_item_ms_replica + m_rcv_timeout_ms + 10;

        m_otherdata += "basatype='" + m_basa_type_name + "', ";
        if (m_conn_string.empty()) {
            m_otherdata += "basaname='" + m_storage_name + "', ";
            m_otherdata += "host='" + m_host + "', ";
            m_otherdata += "user='" + m_user + "', ";
            if (m_pwd_define)
                m_otherdata += "password='" + pwd + "', ";
            m_otherdata += "port=" + IntToStroka(m_port) + ", ";

        } else {
            TString ids1 = "password=";
            TString ids2 = " ";
            TString ids3 = "\"";
            TString ids4 = "\'";
            const char* p1 = NULL;
            const char* p2 = NULL;

            m_conn_string_hidden = m_conn_string;
            p1 = strstr(m_conn_string.c_str(), ids1.c_str());
            if (p1 != NULL) {
                p2 = strstr(p1, ids2.c_str());
                if (p2 == NULL)
                    p2 = strstr(p1, ids3.c_str());
                if (p2 == NULL)
                    p2 = strstr(p1, ids4.c_str());
                if (p2 == NULL)
                    p2 = m_conn_string.c_str() + m_conn_string.length();
                if (p2 != NULL) {
                    m_conn_string_hidden = TString(m_conn_string.c_str(), p1 - m_conn_string.c_str()) + ids1;
                    m_conn_string_hidden += "***";
                    m_conn_string_hidden += TString(p2);
                }
            }

            m_otherdata += "connstr='" + m_conn_string_hidden + "', ";
        }
        m_otherdata += "pool_item_size_master='" + IntToStroka(m_pool_item_size_master) + "', ";
        m_otherdata += "wait_pool_item_ms_master='" + IntToStroka(m_wait_pool_item_ms_master) + "', ";
        m_otherdata += "pool_item_size_replica='" + IntToStroka(m_pool_item_size_replica) + "', ";
        m_otherdata += "wait_pool_item_ms_replica='" + IntToStroka(m_wait_pool_item_ms_replica) + "', ";
        m_otherdata += "rcv_timeout_ms='" + IntToStroka(m_rcv_timeout_ms) + "', ";
        m_otherdata += "skipResponsesToBD=" + BoolToStroka2(skipResponsesToBD);

        loghandler.Init(LogsGroup);

        if (!skipResponsesToBD) {
            bool pg_init_ok = false;

            if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL))
                LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "%s: ASYNC_MODE - %s", m_basa_type_name.c_str(), BoolToStroka2(ASYNC_MODE).c_str());

            if (m_conn_string.empty()) {
                if (m_pwd_define)
                    pg_init_ok = sql::PGInit(m_storage_name.c_str(), m_user.c_str(), pwd.c_str(), m_host.c_str(), m_port, TDuration::MilliSeconds(m_rcv_timeout_ms));
                else
                    pg_init_ok = sql::PGInitWOPass(m_storage_name.c_str(), m_user.c_str(), m_host.c_str(), m_port, TDuration::MilliSeconds(m_rcv_timeout_ms));

            } else {
                TVector<TString> srv_hosts;
                const char* pb = NULL;
                const char* pe = NULL;
                size_t count = 0;
                TString init_host = "";

                pb = m_conn_string.c_str();
                pe = strchr(pb, ',');
                while (pe != NULL) {
                    count = pe - pb;
                    if ((count > 0) && (count < 1024)) {
                        init_host = Trim(TString(pb, count));
                        if (init_host.length() >= 2) {
                            if ((init_host[0] == '\'') || (init_host[0] == '\"'))
                                init_host = init_host.substr(1, init_host.length() - 1);
                            if ((init_host[init_host.length() - 1] == '\'') || (init_host[init_host.length() - 1] == '\"'))
                                init_host = init_host.substr(0, init_host.length() - 1);

                            init_host = CorrectInitPassword(init_host);

                            srv_hosts.push_back(init_host);
                        }
                    }

                    pb = pe + 1;
                    pe = strchr(pb, ',');
                }
                count = m_conn_string.c_str() + m_conn_string.length() - pb;
                if ((count > 0) && (count < 1024)) {
                    init_host = Trim(TString(pb, count));
                    if (init_host.length() >= 2) {
                        if ((init_host[0] == '\'') || (init_host[0] == '\"'))
                            init_host = init_host.substr(1, init_host.length() - 1);
                        if ((init_host[init_host.length() - 1] == '\'') || (init_host[init_host.length() - 1] == '\"'))
                            init_host = init_host.substr(0, init_host.length() - 1);

                        init_host = CorrectInitPassword(init_host);

                        srv_hosts.push_back(init_host);
                    }
                }

                if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL)) {
                    TVector<TString>::iterator sit;
                    ui32 num_str = 0;

                    sit = srv_hosts.begin();
                    while (sit != srv_hosts.end()) {
                        num_str++;
                        LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "%s: conn_str_%02u: '%s'", m_basa_type_name.c_str(), num_str, (*sit).c_str());

                        ++sit;
                    }

                    LogsGroup->GetServerLog()->FFlush();
                }

                pg_init_ok = sql::PGInit(srv_hosts, TDuration::MilliSeconds(m_rcv_timeout_ms));
            }
            sql::PGInitPool(TPoolParams(m_pool_item_size_master, TDuration::MilliSeconds(m_wait_pool_item_ms_master)), TPoolParams(m_pool_item_size_replica, TDuration::MilliSeconds(m_wait_pool_item_ms_replica)));

            TPoolParams pool_param_master = sql::PGMasterPoolParams();
            TPoolParams pool_param_replica = sql::PGReplicaPoolParams();

            if (m_conn_string.empty()) {
                setup_log += "basaname='" + m_storage_name + "', ";
                setup_log += "host='" + m_host + "', ";
                setup_log += "user='" + m_user + "', ";
                if (m_pwd_define)
                    setup_log += "password=yes, ";
                else
                    setup_log += "password=no, ";
                setup_log += "port=" + IntToStroka(m_port) + ", ";

            } else {
                setup_log += "connstr='" + m_conn_string_hidden + "', ";
            }
            setup_log += "pool_item_size_master='" + IntToStroka(pool_param_master.poolSize) + "', ";
            setup_log += "wait_pool_item_ms_master='" + IntToStroka(pool_param_master.timeout.MilliSeconds()) + "', ";
            setup_log += "pool_item_size_replica='" + IntToStroka(pool_param_replica.poolSize) + "', ";
            setup_log += "wait_pool_item_ms_replica='" + IntToStroka(pool_param_replica.timeout.MilliSeconds()) + "', ";
            setup_log += "rcv_timeout_ms='" + IntToStroka(m_rcv_timeout_ms) + "', ";
            setup_log += "skipResponsesToBD=" + BoolToStroka2(skipResponsesToBD);

            if (pg_init_ok) {
                if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL)) {
                    LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "%s: inited - OK (%s)", m_basa_type_name.c_str(), setup_log.c_str());
                    LogsGroup->GetServerLog()->FFlush();
                }

            } else {
                if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL)) {
                    LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "%s: inited - FAILED (%s)", m_basa_type_name.c_str(), setup_log.c_str());
                    LogsGroup->GetServerLog()->FFlush();
                }
            }

        } else {
            std::cout << "Warning! 'skipResponsesToBD' option is true\n";
            if ((LogsGroup != NULL) && (LogsGroup->GetServerLog() != NULL)) {
                LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KWARNING, "%s: 'skipResponsesToBD' option is true", m_basa_type_name.c_str());
                LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "%s: inited - OK", m_basa_type_name.c_str());
                LogsGroup->GetServerLog()->FFlush();
            }
        }

        ui32 checkconn_tick = CShingleTime::GetMs();
        sql::TResWithError<void> result = sql::PGCheckConnection();
        checkconn_tick = CShingleTime::GetMs() - checkconn_tick;
        m_storage_connect = result;
        if (!m_storage_connect) {
            m_storage_status = result.error.ToLog();
            if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL))
                LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "PGCHECKCONN: %u FAILED='%s'", checkconn_tick, m_storage_status.c_str());

        } else {
            if ((LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL))
                LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KMESSAGE, "PGCHECKCONN: %u OK", checkconn_tick, m_storage_status.c_str());
        }

        {
            prepareTables();

            if (!skipResponsesToBD) {
                bool ok = false;

                ok = sql::PGCreateTable(LONGSTAT_COLLECTION, tables[LONGSTAT_COLLECTION]);
                TVector<TString> fields;

                fields.push_back(FIELD_ID);
                ok = sql::PGCreateIndex("id_idx", LONGSTAT_COLLECTION, fields);
            }

            if (!skipResponsesToBD) {
                bool ok = false;

                ok = sql::PGCreateTable(STAT_COLLECTION, tables[STAT_COLLECTION]);
                TVector<TString> fields;

                fields.push_back(CST_ID);
                ok = sql::PGCreateIndex("id_idx", STAT_COLLECTION, fields);
            }
        }
    }

    return res;
}

bool TDataBasaPostgre::InitBeforeFork(const TString& server_hostname, const TString& server_code, const TString& server_start_time, const TString& server_version, TLogsGroup* LogsGroupA, TKConfig* configobjA) {
    return InitBeforeForkNum(-1, server_hostname, server_code, server_start_time, server_version, LogsGroupA, configobjA);
}

void TDataBasaPostgre::InitAfterFork() {
    InitAfterFork_base();
}

bool TDataBasaPostgre::StorageUpdate(ui64 shingle, int type, TShingleStorageType sstype, const nosql::HashMap& incrs, const nosql::HashMap& sets, bool& err) {
    if (skipResponsesToBD)
        return true;

    bool res = false;
    TString collection_name = "";
    TString err_s = "";

    if ((!incrs.empty()) || (!sets.empty())) {
        collection_name = GetCollectionName(sstype, type);

        ui32 tick = CShingleTime::GetMs();
        auto errobj = PGUpdate(collection_name.c_str(), tables[collection_name], shingle, incrs, sets);
        res = errobj;
        tick = CShingleTime::GetMs() - tick;
        AddStorageStat(sstype, SAT_UPDATE, 1, sql::ToTSRSErrType(errobj), tick);

        IncrementStat(sstype, res);

        if ((!res || (tick > m_allow_full_time)) && (LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
            TString to_dop = "";

            if (tick > m_allow_full_time)
                to_dop = "_to";

            if (!res)
                LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u StorageUpdate error%s='%s'", m_basa_type_name.c_str(), tick, to_dop.c_str(), errobj.error.ToLog().c_str());
            else
                LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u StorageUpdate ok%s='%s'", m_basa_type_name.c_str(), tick, to_dop.c_str(), errobj.error.ToLog().c_str());
        }

    } else
        res = true;

    err = !res;

    return res;
}

bool TDataBasaPostgre::StorageFindOne(ui64 shingle, int type, TShingleStorageType sstype, nosql::HashMap& hash, bool& err) {
    if (skipResponsesToBD)
        return true;

    TString table_name = GetCollectionName(sstype, type);

    ui32 tick = CShingleTime::GetMs();
    sql::TResWithError<void> result = sql::PGFindOne(table_name.c_str(), tables[table_name], shingle, hash, FIELD_ID, ASYNC_MODE);
    tick = CShingleTime::GetMs() - tick;
    AddStorageStat(sstype, SAT_FINDONE, 1, sql::ToTSRSErrType(result), tick);

    if ((!result || (tick > m_allow_full_time)) && (LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
        TString to_dop = "";

        if (tick > m_allow_full_time)
            to_dop = "_to";

        if (!result)
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u FindOne error%s: %s", m_basa_type_name.c_str(), tick, to_dop.c_str(), result.error.ToLog().c_str());
        else
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u FindOne ok%s: %s", m_basa_type_name.c_str(), tick, to_dop.c_str(), result.error.ToLog().c_str());
    }

    IncrementStat(sstype, result);

    err = !result;
    return result;
}

bool TDataBasaPostgre::StorageFind(TShingleStorageType sstype, TVector<nosql::HashMap>& hashes, bool& err) {
    if (skipResponsesToBD)
        return true;

    bool res = false;
    TString collection_name = GetCollectionName(sstype, 0);

    ui32 tick = CShingleTime::GetMs();

    auto result = sql::PGFind(collection_name.c_str(), tables[collection_name], hashes, ASYNC_MODE, 1024);
    res = result;

    tick = CShingleTime::GetMs() - tick;
    AddStorageStat(sstype, SAT_FIND, 1, sql::ToTSRSErrType(result), tick);

    if ((!result || (tick > m_allow_full_time)) && (LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
        TString to_dop = "";

        if (tick > m_allow_full_time)
            to_dop = "_to";

        if (!result)
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u StorageFind error%s: %s", m_basa_type_name.c_str(), tick, to_dop.c_str(), result.error.ToLog().c_str());
        else
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u StorageFind ok%s: %s", m_basa_type_name.c_str(), tick, to_dop.c_str(), result.error.ToLog().c_str());
    }

    IncrementStat(sstype, res);

    err = !res;
    return res;
}

bool TDataBasaPostgre::StorageErase(ui64 shingle, int type, TShingleStorageType sstype, bool& err) {
    if (skipResponsesToBD)
        return true;

    TString collection_name = GetCollectionName(sstype, type);

    char ns[1024];
    sprintf(ns, "%s.%s", m_storage_name.c_str(), collection_name.c_str());

    ui32 tick = CShingleTime::GetMs();
    sql::TResWithError<void> result = sql::PGErase(collection_name.c_str(), shingle);
    tick = CShingleTime::GetMs() - tick;
    AddStorageStat(sstype, SAT_ERASE, 1, sql::ToTSRSErrType(result), tick);

    if ((!result || (tick > m_allow_full_time)) && (LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
        TString to_dop = "";

        if (tick > m_allow_full_time)
            to_dop = "_to";

        if (!result)
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u Erase error%s: %s", m_basa_type_name.c_str(), tick, to_dop.c_str(), result.error.ToLog().c_str());
        else
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u Erase ok%s: %s", m_basa_type_name.c_str(), tick, to_dop.c_str(), result.error.ToLog().c_str());
    }

    IncrementStat(sstype, result);

    err = !result;
    return result;
}

i64 TDataBasaPostgre::StorageSize(int type, TShingleStorageType sstype, bool& err) {
    if (skipResponsesToBD)
        return 0;

    TString collection_name = GetCollectionName(sstype, type);

    ui32 tick = CShingleTime::GetMs();

    if ((m_basa_type_name == "PGAAS") || (m_basa_type_name == "pgaas"))
        collection_name.to_lower();
    sql::TResWithError<size_t> result = ((m_basa_type_name == "PGAAS") || (m_basa_type_name == "pgaas")) ? sql::PGTableSizeApprox(collection_name.c_str()) : sql::PGTableSize(collection_name.c_str());

    tick = CShingleTime::GetMs() - tick;
    AddStorageStat(sstype, SAT_SIZE, 1, sql::ToTSRSErrType(result), tick);

    if ((!result || (tick > m_allow_full_time)) && (LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
        TString to_dop = "";

        if (tick > m_allow_full_time)
            to_dop = "_to";

        if (!result)
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u CollectionSize error%s: %s", m_basa_type_name.c_str(), tick, to_dop.c_str(), result.error.ToLog().c_str());
        else
            LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u CollectionSize ok%s: %s", m_basa_type_name.c_str(), tick, to_dop.c_str(), result.error.ToLog().c_str());
    }

    IncrementStat(sstype, result.res);

    err = !result;
    return result.res;
}

void TDataBasaPostgre::StorageMultiAction(TStorageActionList& actlist, bool& err) {
    TStorageActionListIt it;
    ui32 tick = 0;
    TBulkReadDataList bulkreadlist;
    TBulkReadDataListIt bit;
    TMap<ui64, nosql::HashMap*>::iterator mit;
    bool exists = false;
    ui32 read_count = 0;
    TString value_ident = "";
    ui32 value = 0;

    //calc read item
    it = actlist.begin();
    while (it != actlist.end()) {
        if ((*it).m_nosqldata != NULL) {
            if ((*it).m_action == SAT_FINDONE)
                read_count++;
        }

        ++it;
    }

    //create request
    it = actlist.begin();
    while (it != actlist.end()) {
        if ((*it).m_nosqldata != NULL) {
            tick = CShingleTime::GetMs();
            switch ((*it).m_action) {
                default:
                    break;
                case SAT_UPDATE:
                    StorageUpdate((*it).m_nosqldata->shingle, (*it).m_nosqldata->type, (*it).m_storage_type, (*it).m_nosqldata->incrs, (*it).m_nosqldata->sets, *(*it).m_err);
                    break;
                case SAT_FINDONE:

                    if (read_count > 1) {
                        exists = false;
                        bit = bulkreadlist.begin();
                        while (bit != bulkreadlist.end()) {
                            if ((*bit).m_sstype == (*it).m_storage_type) {
                                mit = (*bit).m_bulk_read_hash.find((*it).m_nosqldata->shingle);
                                if (mit == (*bit).m_bulk_read_hash.end()) {
                                    (*bit).m_bulk_read_hash[(*it).m_nosqldata->shingle] = &(*it).m_nosqldata->hash;
                                    (*bit).m_bulk_read_error[(*it).m_nosqldata->shingle] = (*it).m_err;
                                }

                                exists = true;
                                break;
                            }

                            ++bit;
                        }
                        if (!exists)
                            bulkreadlist.push_back(TBulkReadData((*it).m_storage_type, (*it).m_nosqldata->shingle, &(*it).m_nosqldata->hash, (*it).m_err));

                    } else
                        StorageFindOne((*it).m_nosqldata->shingle, (*it).m_nosqldata->type, (*it).m_storage_type, (*it).m_nosqldata->hash, *(*it).m_err);

                    break;
                case SAT_ERASE:
                    StorageErase((*it).m_nosqldata->shingle, (*it).m_nosqldata->type, (*it).m_storage_type, *(*it).m_err);
                    break;
            };
            tick = CShingleTime::GetMs() - tick;
            *(*it).m_tick = tick;

            if (*(*it).m_err)
                err = true;
        }

        ++it;
    }

    if ((read_count > 1) && (bulkreadlist.size() > 0)) {
        bit = bulkreadlist.begin();
        while (bit != bulkreadlist.end()) {
            TString table_name = GetCollectionName((*bit).m_sstype, 0);

            ui32 tickr = CShingleTime::GetMs();
            sql::TResWithError<void> result = sql::PGFindByIDs(table_name.c_str(), tables[table_name], (*bit).m_bulk_read_hash, FIELD_ID, ASYNC_MODE);
            tickr = CShingleTime::GetMs() - tickr;
            AddStorageStat((*bit).m_sstype, SAT_MULTYACTION, 1, sql::ToTSRSErrType(result), tickr);

            try {
                value_ident = sql::ProfKeys::total;
                value = result.prof.GetItemDuration(value_ident).MilliSeconds();
                m_traccert_driver.AddTick(GetCollectionName((*bit).m_sstype, 0) + "::MULTYACTION::" + value_ident, value);
            } catch (...) {
            }

            try {
                value_ident = sql::ProfKeys::getConnect;
                value = result.prof.GetItemDuration(value_ident).MilliSeconds();
                m_traccert_driver.AddTick(GetCollectionName((*bit).m_sstype, 0) + "::MULTYACTION::" + value_ident, value);
            } catch (...) {
            }

            try {
                value_ident = sql::ProfKeys::createQuery;
                value = result.prof.GetItemDuration(value_ident).MilliSeconds();
                m_traccert_driver.AddTick(GetCollectionName((*bit).m_sstype, 0) + "::MULTYACTION::" + value_ident, value);
            } catch (...) {
            }

            try {
                value_ident = sql::ProfKeys::exec;
                value = result.prof.GetItemDuration(value_ident).MilliSeconds();
                m_traccert_driver.AddTick(GetCollectionName((*bit).m_sstype, 0) + "::MULTYACTION::" + value_ident, value);
            } catch (...) {
            }

            try {
                value_ident = sql::ProfKeys::parseRes;
                value = result.prof.GetItemDuration(value_ident).MilliSeconds();
                m_traccert_driver.AddTick(GetCollectionName((*bit).m_sstype, 0) + "::MULTYACTION::" + value_ident, value);
            } catch (...) {
            }

            try {
                value_ident = sql::ProfKeys::runQuery;
                value = result.prof.GetItemDuration(value_ident).MilliSeconds();
                m_traccert_driver.AddTick(GetCollectionName((*bit).m_sstype, 0) + "::MULTYACTION::" + value_ident, value);
            } catch (...) {
            }

            if (((!result) || (tickr > m_allow_full_time)) && (LogsGroup != NULL) && (LogsGroup->GetStorageLog() != NULL)) {
                TString to_dop = "";

                if (tickr > m_allow_full_time)
                    to_dop = "_to";

                if (!result)
                    LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u FindByIDs error%s: %s", m_basa_type_name.c_str(), tickr, to_dop.c_str(), result.error.ToLog().c_str());
                else
                    LogsGroup->GetStorageLog()->WriteMessageAndDataStatus(KERROR, "%s: %u FindByIDs ok%s: %s", m_basa_type_name.c_str(), tickr, to_dop.c_str(), result.error.ToLog().c_str());

                err = true;
            }

            IncrementStat((*bit).m_sstype, result);

            (*bit).m_err = !result;
            (*bit).SetErrors();

            ++bit;
        }
    }
}

TString TDataBasaPostgre::ReturnDriverProp() {
    TString res = "";
    TPoolParams pool_param_master = sql::PGMasterPoolParams();
    TPoolParams pool_param_replic = sql::PGReplicaPoolParams();

    //res += "<i><b>Driver properties:</b></i>&nbsp;&nbsp;";
    res += "<table border='0' width='100%' cellspacing='0' cellpadding='4'>";
    res += "<tr align='left'><td width='30%'>basa type</td><td>" + m_basa_type_name + "</td></tr>";
    res += "<tr align='left'><td width='30%'>basaname</td><td>" + m_storage_name + "</td></tr>";
    res += "<tr align='left'><td width='30%'>host</td><td>" + m_host + "</td></tr>";
    res += "<tr align='left'><td width='30%'>port</td><td>" + IntToStroka(m_port) + "</td></tr>";
    res += "<tr align='left'><td width='30%'>user</td><td>" + m_user + "</td></tr>";
    res += "<tr align='left'><td width='30%'>use password</td><td>" + BoolToStroka2(m_pwd_define) + "</td></tr>";
    res += "<tr align='left'><td width='30%'>rcv_timeout_ms</td><td>" + IntToStroka(m_rcv_timeout_ms) + "</td></tr>";
    res += "<tr align='left'><td width='30%'>skipResponsesToBD</td><td>" + BoolToStroka2(skipResponsesToBD) + "</td></tr>";
    res += "<tr align='left'><td width='30%'>pool_item_size</td><td>MASTER: " + IntToStroka(pool_param_master.poolSize) + "&nbsp; REPLICA: " + IntToStroka(pool_param_replic.poolSize) + "</td></tr>";
    res += "<tr align='left'><td width='30%'>wait_pool_item_ms</td><td>MASTER: " + IntToStroka(pool_param_master.timeout.MilliSeconds()) + "&nbsp; REPLICA: " + IntToStroka(pool_param_replic.timeout.MilliSeconds()) + "</td></tr>";
    res += "</table><br>";

    return res;
}

#define ADDKEY(table, name, type) table[name] = sql::Field(type, name, true)
#define ADDTOTABLE(table, name, type) table[name] = sql::Field(type, name)

bool TDataBasaPostgre::CheckConnection(TString& status) {
    sql::TResWithError<void> result = sql::PGCheckConnection();

    if (!result)
        status = result.error.ToLog();

    return result;
}

void TDataBasaPostgre::prepareTables() {
    {
        //frodooborona
        sql::table_t frodooborona_tbl;

        ADDKEY(frodooborona_tbl, FIELD_ID, sql::INT64);
        ADDTOTABLE(frodooborona_tbl, FIELD_LASTDAY, sql::INT32);
        ADDTOTABLE(frodooborona_tbl, FIELD_TYPE, sql::INT32);
        ADDTOTABLE(frodooborona_tbl, FIELD_SOURCE, sql::TEXT);
        ADDTOTABLE(frodooborona_tbl, FIELD_HAM, sql::INT32);
        ADDTOTABLE(frodooborona_tbl, FIELD_SPAM85, sql::INT32);
        ADDTOTABLE(frodooborona_tbl, FIELD_SPAM100, sql::INT32);

        tables[LONGSTAT_COLLECTION] = frodooborona_tbl;

        //stat
        sql::table_t stat_tbl;

        ADDKEY(stat_tbl, CST_ID, sql::INT64);
        ADDTOTABLE(stat_tbl, CST_LASTDATE, sql::INT32);
        ADDTOTABLE(stat_tbl, CST_HOSTNAME, sql::TEXT);
        ADDTOTABLE(stat_tbl, CST_CODE, sql::TEXT);
        ADDTOTABLE(stat_tbl, CST_STARTTIME, sql::TEXT);
        ADDTOTABLE(stat_tbl, CST_VERSION, sql::TEXT);
        ADDTOTABLE(stat_tbl, CST_RATEMODE, sql::TEXT);
        ADDTOTABLE(stat_tbl, CST_CHECK, sql::TEXT);
        ADDTOTABLE(stat_tbl, CST_RCPT, sql::TEXT);
        ADDTOTABLE(stat_tbl, CST_CORRLS, sql::TEXT);
        ADDTOTABLE(stat_tbl, CST_RESERV, sql::TEXT);

        tables[STAT_COLLECTION] = stat_tbl;
    }
}

qustat::TDriverRqstPoolStat TDataBasaPostgre::ReturnLiveRqstStatMaster() {
    qustat::TDriverRqstPoolStat res;

    return res;
}

qustat::TDriverRqstPoolStat TDataBasaPostgre::ReturnLiveRqstStatReplica() {
    qustat::TDriverRqstPoolStat res;

    return res;
}

//*********************************************************************************************************************************
