#include <mail/so/spamstop/sp/spamstop.h>
#include <mail/so/spamstop/sp/spk3.h>
#include "trenginepool.h"

//************************************************************************************
//                             TRengineElement
//************************************************************************************

TRengineElement::TRengineElement(int indexA) : m_pSpTop(true)
{
   m_index           = indexA;
   m_hSp             = nullptr;
   LogsGroup         = nullptr;
   m_lastcalctime    = time(NULL);
   m_cps             = 0;
   m_count           = 0;
   m_dnRules         = "";
   m_Score           = 1;
   RulePrintList     = nullptr;
   m_rules_cs        = "";
   m_loadrules_time  = 0;
}

void TRengineElement::Lock()
{
   m_Mutex.Acquire();
}

void TRengineElement::UnLock()
{
   m_Mutex.Release();
}

bool TRengineElement::Init(TLogsGroup *LogsGroupA, const TString &dnRulesA, float ScoreA, kipv6::TStringsLists *RulePrintListA, TTrueAtomicSharedPtr<TRulesHolder> pRulesHolder)
{
   bool        res   = false;

   m_dnRules               = dnRulesA;
   m_Score                 = ScoreA;
   LogsGroup               = LogsGroupA;
   RulePrintList           = RulePrintListA;

   m_pSpTop.dnRules = {m_dnRules};
   m_pSpTop.score = m_Score;

   Lock();

   if (pRulesHolder)
   {
       if (pRulesHolder->IsOK())
       {
           SpLoggers.GetSpLoggerPtr(SO_RULES_LOGGER)->SetSPKLog(LogsGroup->GetFilterLog());
           SpLoggers.GetSpLoggerPtr(SO_RULES_LOGGER)->SetOpen(true);

           if (!(m_hSp = SpOpenA(RulePrintList, pRulesHolder, &m_pSpTop, SpLoggers)))
           {
              m_hSp = nullptr;
              PrintActionLog(KERROR, "MESS:       Can't open filter");
           } else
              res = true;

       }
       else
       {
           PrintActionLog(KERROR, "MESS:       RulesHolder FAILED.");
       }
   }
   else
   {
       PrintActionLog(KERROR, "MESS:       RulesHolder is NULL.");
   }

   m_loadrules_time = time(NULL);

   UnLock();

   return res;
}

TString TRengineElement::Reset(bool &ok, TTrueAtomicSharedPtr<TRulesHolder> pRulesHolder)
{
  TString      res   = "";

  ok = false;

  m_pSpTop.dnRules = {m_dnRules};
  m_pSpTop.score = m_Score;

  Lock();

  if ( (LogsGroup != nullptr) && (LogsGroup->GetFilterLog() != nullptr) )
     LogsGroup->GetFilterLog()->SetDiffPoint();

  if (pRulesHolder)
  {
      if (pRulesHolder->IsOK())
      {
          SpLoggers.GetSpLoggerPtr(SO_RULES_LOGGER)->SetSPKLog(LogsGroup->GetFilterLog());
          SpLoggers.GetSpLoggerPtr(SO_RULES_LOGGER)->SetOpen(true);

          if (!(m_hSp = SpOpenA(RulePrintList, pRulesHolder, &m_pSpTop, SpLoggers)))
          {
             PrintActionLog(KERROR, "MESS:       Can't reset filter");
             m_hSp = nullptr;
             ok = false;
          } else
          {
             PrintActionLog(KMESSAGE, "MESS:       Filter reset - OK");

             if ( (LogsGroup != nullptr) && (LogsGroup->GetFilterLog() != nullptr) )
                res = LogsGroup->GetFilterLog()->GetDiff();
             ok = true;
          }
      }
      else
      {
          PrintActionLog(KERROR, "MESS:       RulesHolder FAILED");
      }
  }
  else
  {
      PrintActionLog(KERROR, "MESS:       RulesHolder is NULL");
  }


  m_loadrules_time = time(NULL);

  UnLock();

  return res;
}

ui32 TRengineElement::PrintActionLog(TLogStatus status, const char *msg, ...)
{
   ui32   tt    = 0;

   tt = CShingleTime::GetMs();
   if ( (LogsGroup != nullptr) && (LogsGroup->GetServerLog() != nullptr) )
   {
      va_list args;
      char    buf[2048];

      memset(buf, 0, sizeof(buf));
      va_start(args, msg);
      vsnprintf(buf, sizeof(buf) - 1, msg, args);
      va_end(args);

      LogsGroup->GetServerLog()->WriteMessageAndDataStatus(status, "%s", buf);
   }
   tt = CShingleTime::GetMs() - tt;

   return tt;
}

void TRengineElement::CalcCPS()
{
   if (m_count < 0xFFFFFFFF)
      m_count++;

   OnlyCalcCPS();
}

void TRengineElement::OnlyCalcCPS()
{
   time_t currenttime = time(NULL);

   if ((currenttime - m_lastcalctime) > 30)
   {
      m_cps = (float)m_count / (float)(currenttime - m_lastcalctime);
      m_count = 0;
      m_lastcalctime = currenttime;
   }
}

float TRengineElement::GetCPS()
{
   OnlyCalcCPS();

   return m_cps;
}

//************************************************************************************
//                               TRenginePool
//************************************************************************************

TRenginePool::TRenginePool()
{
   RulePrintList              = nullptr;
   elementcount               = 0;
   m_half_elementcount        = 0;
   m_quarter_elementcount     = 0;
   m_threefourth_elementcount = 0;
   m_last_calc_usedrengine    = time(NULL);
   m_max_usedrengine          = 0;
   m_exit = false;
   m_dnRules                  = "";
   m_Score                    = 1;
#ifdef KSEMA
   m_Sema         = nullptr;
#endif
   m_good_filter_count        = 0;
   m_bad_filter_count         = 0;
   m_rules_path               = "";
}

TRenginePool::~TRenginePool()
{
   SetExit();

   Lock();

   TRengineElementListIt it = m_AllElementList.begin();
   while (it != m_AllElementList.end())
   {
      if ((*it) != nullptr)
      {
         delete (*it);
         (*it) = nullptr;
      }

      ++it;
   }

   UnLock();

#ifdef KSEMA
   if (m_Sema != nullptr)
   {
      delete m_Sema;
      m_Sema = nullptr;
   }
#endif
}

void TRenginePool::Lock()
{
   m_Mutex.Acquire();
}

void TRenginePool::UnLock()
{
   m_Mutex.Release();
}

bool TRenginePool::Init(TLogsGroup *LogsGroupA, int elementcountA, const TString &dnRulesA, float ScoreA, kipv6::TStringsLists *RulePrintListA)
{
   bool res = false;

   TRengineElement       *elem = nullptr;
   TRengineElementListIt it;
   ui32                  begin_time = CShingleTime::GetMs();

   m_dnRules      = dnRulesA;
   m_Score        = ScoreA;
   RulePrintList  = RulePrintListA;
   LogsGroup      = LogsGroupA;
   elementcount   = elementcountA;

#ifdef KSEMA
   if (m_Sema == nullptr)
      m_Sema = new TUnnamedSemaphore(elementcount);
#endif

   m_half_elementcount        = elementcount / 2;
   m_quarter_elementcount     = elementcount / 4;
   m_threefourth_elementcount = 3 * elementcount / 4;

   m_MutexExit.Acquire();
   m_exit = false;
   m_MutexExit.Release();

   Lock();

   m_rules_path = dnRulesA;
   SpLoggersMain.GetSpLoggerPtr(SO_RULES_LOGGER)->SetSPKLog(LogsGroup->GetFilterLog());
   pRulesHolder = MakeTrueAtomicShared<TRulesHolder>(true, TVector<TFsPath>{m_rules_path}, SpLoggersMain, NRegexp::TSettings{});

   for (int i = 0; i < elementcount; i++)
   {
      elem = new TRengineElement(i + 1);
      m_AllElementList.push_back(elem);
      m_AccessibleList.push_back(elem);
   }

   m_good_filter_count = 0;
   m_bad_filter_count  = 0;

   it = m_AllElementList.begin();
   while (it != m_AllElementList.end())
   {
      if ((*it) != nullptr)
      {
         if ( (*it)->Init(LogsGroup, m_dnRules, m_Score, RulePrintList, pRulesHolder) )
            m_good_filter_count++;
         else
            m_bad_filter_count++;
      }

      ++it;
   }

   UnLock();

   begin_time = CShingleTime::GetMs() - begin_time;

   if ( (m_good_filter_count > 0) && (m_bad_filter_count == 0) )
      res = true;
   else
      res = false;

   if ( (LogsGroup != nullptr) && (LogsGroup->GetServerLog() != nullptr) )
   {
      if (m_bad_filter_count > 0)
         LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KERROR, "Load filters FAILED (%u good, %u bad, %u msec).", m_good_filter_count, m_bad_filter_count, begin_time);
      else
         LogsGroup->GetServerLog()->WriteMessageAndDataStatus(KMESSAGE, "Load filters OK (%u good, %u msec).", m_good_filter_count, begin_time);
   }

   return res;
}

TResetResponce TRenginePool::Reset(bool &ok)
{
   TResetResponce        res;
   TRengineElementListIt it;
   bool                  first_elm = true;
   bool                  ok_elm = false;
   int                   ok_count = 0;
   int                   failed_count = 0;
   TString                s = "";
   ui32                  begin_time = CShingleTime::GetMs();

   Lock();

   begin_time = CShingleTime::GetMs();

   m_good_filter_count = 0;
   m_bad_filter_count  = 0;

   SpLoggersMain.GetSpLoggerPtr(SO_RULES_LOGGER)->SetSPKLog(LogsGroup->GetFilterLog());

   pRulesHolder = MakeTrueAtomicShared<TRulesHolder>(true, TVector<TFsPath>{m_rules_path}, SpLoggersMain, NRegexp::TSettings{});

   UnLock();

   it = m_AllElementList.begin();
   while (it != m_AllElementList.end())
   {
      if ((*it) != nullptr)
      {
         s = (*it)->Reset(ok_elm, pRulesHolder);
         if (first_elm)
         {
            res.filterres = s;
            first_elm = false;
         }
         if (ok_elm)
            ok_count++;
         else
            failed_count++;
      }

      ++it;
   }

   begin_time = CShingleTime::GetMs() - begin_time;

   Lock();

   m_good_filter_count = ok_count;
   m_bad_filter_count  = failed_count;

   UnLock();

   res.filterres = "<b>Reload filters (" + IntToStroka(begin_time) + " msec): succesfull=" + IntToStroka(ok_count) + " failed=" + IntToStroka(failed_count) + "<b>\n" + res.filterres;
   res.text = "Reload filters (" + IntToStroka(begin_time) + " msec): succesfull=" + IntToStroka(ok_count) + " failed=" + IntToStroka(failed_count);

   if ( (ok_count > 0) && (failed_count == 0) )
      ok = true;
   else
      ok = false;

   res.flag_ok = ok;

   return res;
}

void TRenginePool::ReturnFilter(TRengineElement *elem)
{
   Lock();

   m_AccessibleList.push_back(elem);

   UnLock();

#ifdef KSEMA
   if (m_Sema != nullptr)
      m_Sema->Release();
#endif
}

TRengineElement *TRenginePool::GetFilter()
{
   TRengineElement       *res = nullptr;
   TRengineElementListIt it;
   int                   remain_size = 0;

   while (true)
   {
      if (m_exit)
         break;

#ifdef KSEMA
      if (m_Sema != nullptr)
         m_Sema->Acquire();
#endif

      if (m_exit)
         break;

      Lock();

      it = m_AccessibleList.begin();
      if (it != m_AccessibleList.end())
      {
         res = (*it);
         m_AccessibleList.erase(it);

         remain_size = m_AccessibleList.size();
         if ((elementcount - remain_size) > m_max_usedrengine)
            m_max_usedrengine = elementcount - remain_size;

         UnLock();
         break;
      }

      UnLock();
   }

   /*if (res != nullptr)
   {
      if (remain_size < m_quarter_elementcount)
      {
         if ( (LogsGroup != nullptr) && (LogsGroup->TraceIpLog() != nullptr) )
            LogsGroup->TraceIpLog()->WriteMessageAndData("Remain quarter element count.");
      } else if (remain_size < m_half_elementcount)
      {
         if ( (LogsGroup != nullptr) && (LogsGroup->TraceIpLog() != nullptr) )
            LogsGroup->TraceIpLog()->WriteMessageAndData("Remain half element count.");
      } else if (remain_size < m_threefourth_elementcount)
      {
         if ( (LogsGroup != nullptr) && (LogsGroup->TraceIpLog() != nullptr) )
            LogsGroup->TraceIpLog()->WriteMessageAndData("Remain three fourth element count.");
      }
   }*/

   return res;
}

void TRenginePool::SetExit()
{
   m_MutexExit.Acquire();

   m_exit = true;

   m_MutexExit.Release();

#ifdef KSEMA
   if ( (m_Sema != nullptr) && (elementcount > 0) )
   {
      for (int i = 0; i < elementcount; i++)
         m_Sema->Release();
   }
#endif
}

ui32 TRenginePool::GetAccessibleFilterCount()
{
   ui32 res = 0;

   Lock();

   res = m_AccessibleList.size();

   UnLock();

   return res;
}

int TRenginePool::CalcUsedRengine()
{
   time_t currenttime = time(NULL);
   int    res = 0;

   if (currenttime > m_last_calc_usedrengine)
   {
      Lock();

      res = m_max_usedrengine;
      m_last_calc_usedrengine = currenttime;
      m_max_usedrengine = 0;

      UnLock();
   }

   return res;
}

int TRenginePool::GetUsedRengine()
{
   return m_max_usedrengine;
}

TString TRenginePool::GetCPSList()
{
   TString res = "";
   TRengineElementListIt it;
   bool   first = true;

   it = m_AllElementList.begin();
   while (it != m_AllElementList.end())
   {
      if ((*it) != nullptr)
      {
         if (first)
         {
            res = res + FloatToStr((*it)->GetCPS());
            first = false;
         } else
            res = res + " - " + FloatToStr((*it)->GetCPS());
      }

      ++it;
   }

   return res;
}

TString TRenginePool::GetRulesCSList()
{
   TString res = "";
   TRengineElementListIt it;
   bool   first = true;

   it = m_AllElementList.begin();
   while (it != m_AllElementList.end())
   {
      if ((*it) != nullptr)
      {
         if (first)
         {
            res = res + (*it)->GetRulesCS() + "(" + (*it)->GetLoadRulesElapsed() + ")";
            first = false;
         } else
            res = res + " - " + (*it)->GetRulesCS() + "(" + (*it)->GetLoadRulesElapsed() + ")";
      }

      ++it;
   }

   return res;
}

TRenginePoolStat TRenginePool::GetStat()
{
   TRenginePoolStat res;

   res.filter_count      = elementcount;
   res.good_filter_count = m_good_filter_count;
   res.bad_filter_count  = m_bad_filter_count;
   res.rules             = m_dnRules;
   res.score             = m_Score;
   res.rps_str           = GetCPSList();
   res.rulescs_str       = GetRulesCSList();

   return res;
}

int TRenginePool::GetNumberSection(ui32 ip)
{
   int res = 0;

   if (elementcount > 0)
   {
      res = ip % elementcount;
      if ((res < 0) || (res >= elementcount))
         res = 0;
   } else
      res = 0;

   return res;
}

void TRenginePool::Close()
{

}


//************************************************************************************
