#pragma once

#include "util/generic/hash.h"
#include "util/system/mutex.h"
#include "util/string/util.h"
#include "util/generic/hash.h"
#include "util/string/cast.h"

//*******************************************************************************************************************************
//                                                 TStorageInfo
//*******************************************************************************************************************************

enum TSQLTypeK { ST_UNKNOWN,
                 ST_SQL,
                 ST_NOSQL };
enum TShardType { SHT_UNKNOWN,
                  SHT_SKEEP,
                  SHT_NUMBER };

struct TStorageInfo {
    TSQLTypeK m_sql_type;
    TShardType m_shard_type;
    ui16 m_shar_number{};
    TString m_name;

    TStorageInfo() {
        Clear();
    }

    TStorageInfo(TSQLTypeK sql_type, TShardType shard_type, ui16 shar_number, const TString& name) {
        m_sql_type = sql_type;
        m_shard_type = shard_type;
        m_shar_number = shar_number;
        m_name = name;
    }

    void Clear() {
        m_sql_type = ST_UNKNOWN;
        m_shard_type = SHT_UNKNOWN;
        m_shar_number = 0;
        m_name = "";
    }

    bool operator==(const TStorageInfo& value) const {
        bool res = false;

        if ((m_sql_type == value.m_sql_type) && (m_shard_type == value.m_shard_type) && (m_shar_number == value.m_shar_number))
            res = true;

        return res;
    }

    TString GetStorageIdent() {
        TString res = "";

        if (!m_name.empty()) {
            res = m_name;

        } else {
            switch (m_shard_type) {
                case SHT_UNKNOWN:
                    res = "UNKNOWN";
                    break;
                case SHT_SKEEP:
                    res = "SKEEP";
                    break;
                case SHT_NUMBER:
                    res = "NUMB" + IntToString<10>(m_shar_number);
                    break;
            }
        }

        return res;
    }
};

//*******************************************************************************************************************************
//                                                 TStorageInfoExt
//*******************************************************************************************************************************

class TStorageNoSql;

struct TStorageInfoExt {
    TStorageInfo info;
    TAtomicSharedPtr<TStorageNoSql> storage;

    TStorageInfoExt();

    TStorageInfoExt(TSQLTypeK sql_type, TShardType shard_type, ui16 shar_number, TAtomicSharedPtr <TStorageNoSql> storA, const TString& name);

    TStorageInfoExt(TStorageInfoExt&&) = default;
    TStorageInfoExt& operator=(TStorageInfoExt&&) = default;

    ~TStorageInfoExt();

    void Save(IOutputStream* s) const;
    void Load(IInputStream* s);

    bool operator==(const TStorageInfoExt& value) const {
        bool res;

        res = (info == value.info);

        return res;
    }

    TString AsText() const {
        TString res = "";

        res = res + "name='" + info.m_name + "' ";
        res = res + "sqltype=";
        switch (info.m_sql_type) {
            case ST_UNKNOWN:
                res = res + "UNKNOWN ";
                break;
            case ST_SQL:
                res = res + "SQL ";
                break;
            case ST_NOSQL:
                res = res + "NOSQL ";
                break;
        }
        res = res + "type=";
        switch (info.m_shard_type) {
            case SHT_UNKNOWN:
                res = res + "UNKNOWN ";
                break;
            case SHT_SKEEP:
                res = res + "SKEEP ";
                break;
            case SHT_NUMBER:
                res = res + "NUMB" + IntToString<10>(info.m_shar_number) + " ";
                break;
        }
        if (storage)
            res = res + "address=" + IntToString<10>((ui64)storage.Get());
        else
            res = res + "address=NULL";

        return res;
    }
};

//*******************************************************************************************************************************
//                                                 TStorageInfoFull
//*******************************************************************************************************************************

struct TStorageInfoFull {
    TStorageInfo info;
    //ui32          change_time_localcache;
    //bool          disable_localcache;
    //bool          localcache_add_empty_record;
    TString basatypename;
    TString storageident;
    //TString        other_data;

    TStorageInfoFull() {
        Clear();
    }

    void Clear() {
        info.Clear();
        //change_time_localcache      = 0;
        //disable_localcache          = false;
        //localcache_add_empty_record = false;
        basatypename = "";
        storageident = "";
        //other_data                  = "";
    }
};

//*******************************************************************************************************************************
