#include "cache.h"
#include <mail/so/spamstop/tools/so-common/local_cache_template.h>
#include <mail/so/spamstop/tools/so-common/sputil.h>
#include <mail/so/libs/talkative_config/config.h>
#include <numeric>
#include <tbb/tbb.h>
#include <util/folder/path.h>

namespace NGeneralShingler {

    class TStepCache : public ICache {
        using TInnerCache = NCache::TCacheBase<TKey, TValue>;
    public:
        void Cleanup() { inner.Cleanup(); }

        bool Add(TKey key, TValue && value) override { return inner.Add(key, std::move(value)); }

        bool Get(TKey key, TValue& val) const override { return inner.Get(key, val); }

        size_t Size() const override { return inner.Size(); }

        TStepCache() = default;

        explicit TStepCache(const TFsPath &src) : diskCache(src) {
            if(!diskCache.Defined() || !src.Exists())
                return;

            TMappedFileInput f(*diskCache);
            ::Load(&f, inner);
        }

        ~TStepCache() override {
            if(!diskCache.Defined())
                return;

            try{
                TFileOutput f(*diskCache);
                ::Save(&f, inner);
            } catch (const std::exception & e) {
                Cerr << "Error while saving cache to " << *diskCache << " " << e.what();
            }
        }
    private:
        TMaybe<TFsPath> diskCache;
        TInnerCache inner;
    };

    class TLRUCache : public ICache {
        struct TValueHolder{
            bool empty = true;
            TSimpleSharedPtr<TValue> value;

            TValueHolder() = default;

            explicit TValueHolder(TValue && value) : empty(false), value(MakeSimpleShared<TValue>(std::move(value))) {}
        };
        using TTBBCache = tbb::concurrent_lru_cache<TKey, TValueHolder>;
    public:
        bool Add(TKey key, TValue && value) override {
            auto handle = cache[key];

            handle.value() = TValueHolder{ std::move(value) };

            return true;
        }

        bool Get(TKey key, TValue& val) const override {
            auto handle = cache[key];

            if (handle.value().empty)
                return false;
            val = *handle.value().value;
            return true;
        }

        size_t Size() const override {
            return 0;
        }

        explicit TLRUCache(size_t size) : cache(Create, size) {}
    private:
        static inline TValueHolder Create(TKey) { return{}; }
    private:
        mutable TTBBCache cache;
    };

    class TLATCache : public ICache {
        using TConcurrentMap = tbb::concurrent_hash_map<TKey, TValue>;
        using TAccessor = typename TConcurrentMap::const_accessor;
    public:
        explicit TLATCache(size_t size)
            : map(size)
            , counter(0)
            , queue(size)
        {
            Y_ENSURE(size != 0);
        }

        bool Add(TKey key, TValue && value) override {
            if (!map.insert(std::make_pair(key, std::move(value))))
                return false;

            TKey old = queue[counter++ % queue.size()].exchange(key);
            map.erase(old);

            return true;
        }

        bool Get(TKey key, TValue& val) const override {
            TAccessor accessor;
            if (!map.find(accessor, key) || accessor.empty())
                return false;

            val = accessor->second;

            return true;
        }

        size_t Size() const override {
            return map.size();
        }

    private:
        TConcurrentMap map;
        std::atomic_size_t counter;
        TVector<std::atomic_size_t> queue;
    };

    TAtomicSharedPtr<ICache> CreateCache(const NConfig::TConfig & config, TSimpleScheduler* scheduler, const TString & scheduleTag) {
        const TCacheType type = config.Has("type") ? FromString<TCacheType>(config["type"].Get<TString>()) : TCacheType::Step;

        switch (type) {
            case TCacheType::Step: {
                TAtomicSharedPtr<TStepCache> cache;
                if(config.Has("disk_cache")) {
                    const TFsPath & diskCache = NTalkativeConfig::Get<TString>(config, "disk_cache");
                    cache = MakeAtomicShared<TStepCache>(diskCache);
                } else {
                    cache = MakeAtomicShared<TStepCache>();
                }

                if (scheduler != nullptr) {
                    const auto refreshTime = NTalkativeConfig::As<TDuration>(config, "refresh_time");
                    scheduler->Add([cache]() { cache->Cleanup(); }, refreshTime, scheduleTag);
                }
                return cache;
            }
            case TCacheType::LRU:{
                const auto size = config["size"].As<size_t>();
                return MakeAtomicShared<TLRUCache>(size);
            }
            case TCacheType::LAT: {
                const auto size = config["size"].As<size_t>();
                return MakeAtomicShared<TLATCache>(size);
            }
        }
    }

    size_t TCacheContext::CreateKey(const NJson::TJsonValue & value, const TFieldSet & set) const {
        return std::accumulate(fieldNames.cbegin(), fieldNames.cend(), 0ull, [&set, &value](size_t acc, const TString & fieldName){
            const auto & field = set.GetField(fieldName);
            if(!field)
                return acc;
            const NJson::TJsonValue * res = value.GetValueByPath(fieldName);

            return res != nullptr ? CombineHashes(acc, field->Hash(*res)) : acc;
        });
    }

    TCacheContext::TCacheContext(const NConfig::TConfig & config, TAtomicSharedPtr<ICache> cache) : cache(std::move(cache)) {
        if(!config.Has("fields"))
            ythrow TWithBackTrace<yexception>() << R"(cache must contain "fields" field: )" << config;

        const auto fieldsConfig = config["fields"];

        if(!fieldsConfig.IsA<NConfig::TArray>())
            ythrow TWithBackTrace<yexception>() << R"(cache's "fields" must be array: )" << fieldsConfig;

        for(const auto & field : fieldsConfig.Get<NConfig::TArray>()) {
            if(!field.IsA<TString>())
                ythrow TWithBackTrace<yexception>() << R"(cache's "fields" must contain strings: )" << fieldsConfig;
            fieldNames.emplace_back(field.Get<TString>());
        }
    }

    TCacheContext::~TCacheContext() = default;
}   //  namespace NGeneralShingler

#ifdef WIN32 //fix tbb compilation for VS
void __TBB_get_cpu_ctl_env(tbb::internal::cpu_ctl_env*) {}
void __TBB_set_cpu_ctl_env(const tbb::internal::cpu_ctl_env*) {}
#endif
