#include <library/cpp/config/config.h>
#include <library/cpp/json/json_reader.h>
#include <mail/so/spamstop/tools/general_shingler/shingler/schemes/base.h>
#include <mail/so/libs/scheduler/scheduler.h>
#include <mail/so/spamstop/tools/so-common/StorageBase.h>
#include <mail/so/libs/talkative_config/config.h>

#include "helper.h"
#include "cleanup.h"

namespace NGeneralShingler {
    class TCleanupInfo {
    public:
        TCleanupInfo(const NConfig::TConfig& config, const TTimeSet& timeSet, const TAtomicSharedPtr<TStorageBase>& db, const TString& collection);

        bool PutScheme(const TString& name, const TAtomicSharedPtr<TSchemeBase>& scheme);
        void SetSchedule(TSimpleScheduler& scheduler) const;
    private:
        TTimeSet timeSet;
        TDuration period;
        TCleanupType type = TCleanupType::Period;
        NJson::TJsonValue::TArray fields;
        THashMap<TString, TAtomicSharedPtr<TSchemeBase>> schemes;

        TString collection;
        TAtomicSharedPtr<TStorageBase> db;
    };

    TCleanupInfo::TCleanupInfo(const NConfig::TConfig& config, const TTimeSet& timeSet, const TAtomicSharedPtr<TStorageBase>& db, const TString& collection)
        : timeSet(timeSet)
        , collection(collection)
        , db(db)
    {
        const TString& schedule = NTalkativeConfig::Get<TString>(config, "schedule");
        if (schedule == "PeriodChange") {
            type = TCleanupType::Change;
            period = timeSet.GetResolution();
        }
        else
            period = TDuration::Parse(schedule);

        TStringStream s;
        config["fields"].DumpJson(s);
        fields.push_back(NJson::ReadJsonTree(&s, true));
    }

    bool TCleanupInfo::PutScheme(const TString& name, const TAtomicSharedPtr<TSchemeBase>& scheme) {
        Y_VERIFY(scheme);
        return schemes.emplace("cleanup::" + name, scheme).second;
    }

    static void CreateStatisticRecord(const TString& name, const TAtomicSharedPtr<TStorageBase>& db, const TString& collection)
    try {
        TUpdateAction action;
        action.upsert = true;
        action.query.equals["name"] = name;
        action.update.incrs["all_count"] = 0u;
        action.update.incrs["timestamp"] = 0u;
        action.update.incrs["last_count"] = 0u;
        db->Update(collection, action);
    } catch (const std::exception) { }

    static size_t Remove(const TAtomicSharedPtr<TSchemeBase>& scheme, const NJson::TJsonValue::TArray &fields, TString& error_message, std::exception_ptr& exception)
    try {
        return scheme->Remove(TShardedFields{ fields });
    } catch (const yexception& e) {
        exception = std::current_exception();
        error_message = e.what();
        return 0;
    }

    static std::function<void()> GetAction(const TString& name, const TAtomicSharedPtr<TSchemeBase>& scheme, const NJson::TJsonValue::TArray &fields, const TDuration& period, const TAtomicSharedPtr<TStorageBase>& db, const TString& collection) {
        if (!db || collection.empty())
            return [scheme, fields]() { scheme->Remove(TShardedFields{fields}); };

        CreateStatisticRecord(name, db, collection);
        return [name, scheme, fields, period, db, collection]() {
            TUpdateAction action;
            auto now = TInstant::Now();
            action.query.equals["name"] = name;
            action.query.lt["timestamp"] = now.Seconds();
            action.update.sets["timestamp"] = (now + period / 2).Seconds();
            if (db->Update(collection, action) == 1) {
                TString message("OK");
                std::exception_ptr exception;
                const size_t count = Remove(scheme, fields, message, exception);

                TUpdateAction update;
                update.query.equals["name"] = name;
                update.update.incrs["all_count"] = ui64(count);
                update.update.sets["last_count"] = ui64(count);
                update.update.sets["last_error"] = message;
                db->Update(collection, update);

                if (!!exception)
                    std::rethrow_exception(exception);
            }
        };
    }

    void TCleanupInfo::SetSchedule(TSimpleScheduler& scheduler) const {
        for (const auto& item : schemes) {
            auto action = GetAction(item.first, item.second, fields, period, db, collection);

            if (type == TCleanupType::Period)
                scheduler.Add(action, period, item.first);
            else
                scheduler.Add(action, timeSet.ToInstant(timeSet.GetCurrentPeriod() + 1), period, item.first);
        }
    }

    TVector<TCleanupInfo> Create(const NConfig::TConfig& config, const TAtomicSharedPtr<TStorageBase>& db, const TString& collection, const THashMap<TString, TAtomicSharedPtr<TSchemeBase>>& schemesByName, const THashMap<TString, TTimeSet>& timeSetsByName) {
        TVector<TCleanupInfo> result;
        for (const auto & data : NTalkativeConfig::Get<NConfig::TArray>(config)) {
            result.emplace_back(data, data.Has("time_set") ? NHelper::GetSafe(timeSetsByName, NTalkativeConfig::Get<TString>(data, "time_set")) : TTimeSet(), db, collection);

            const auto& item = data["scheme"];
            if (item.IsA<TString>()) {
                const TString name = item.As<TString>();
                result.back().PutScheme(name, NHelper::GetSafe(schemesByName, name));
            }
            else {
                for (const auto& v : NTalkativeConfig::Get<NConfig::TArray>(item)) {
                    const TString name = v.As<TString>();
                    result.back().PutScheme(name, NHelper::GetSafe(schemesByName, name));
                }
            }
        }
        return result;
    }

    void SetCleanupSchedules(TSimpleScheduler& scheduler,
        const NConfig::TConfig& config,
        const TAtomicSharedPtr<TStorageBase>& db,
        const TString& collection,
        const THashMap<TString, TAtomicSharedPtr<TSchemeBase>>& schemesByName,
        const THashMap<TString, TTimeSet>& timeSetsByName
    ) {
        for (auto& item : Create(config, db, collection, schemesByName, timeSetsByName)) {
            item.SetSchedule(scheduler);
        }
    }
}
