
#include <util/string/cast.h>
#include <util/generic/maybe.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/config/config.h>
#include <mail/so/spamstop/tools/so-common/AnyValue_v2.h>
#include <mail/so/libs/talkative_config/config.h>
#include "field.h"
#include "message.h"

namespace NGeneralShingler {

    template<typename T, typename Traits>
    class TFieldBase : public IField {
        size_t Hash(const NJson::TJsonValue &js) const override {
            return THash<T>()(static_cast<T>(Traits::Get(js)));
        }

        void JsonToAnyvalue(const NJson::TJsonValue &message, NAnyValue::TScalar &anyValue) const override {
            anyValue = static_cast<T>(Traits::Get(message));
        }

        void Incr(const NJson::TJsonValue &message, NAnyValue::TScalar &anyValue) const override {
            anyValue = static_cast<T>(Traits::Get(message) + anyValue.GetRobust<T>());
        }

        void Or(const NJson::TJsonValue &message, NAnyValue::TScalar &anyValue) const override {
            anyValue = static_cast<T>(Traits::Get(message) | anyValue.GetRobust<T>());
        }
    };

    struct UIntegerTraits {
        static ui64 Get(const NJson::TJsonValue &v) {

            if(v.IsString()) {
                const auto & s = v.GetString();
                if(s.StartsWith("0x")) {
                    return strtoull(s.c_str(), nullptr, 16);
                }
            }

            return v.GetUIntegerRobust();
        }
    };
    struct IntegerTraits {
        static i64 Get(const NJson::TJsonValue &v) {
            return static_cast<i64>(UIntegerTraits::Get(v));
        }
    };

    struct StringTraits {
        static TString Get(const NJson::TJsonValue &v) {
            return v.GetStringRobust();
        }
    };

    struct BoolTraits {
        static bool Get(const NJson::TJsonValue &v) {
            return v.GetBooleanRobust();
        }
    };

    struct DoubleTraits {
        static double Get(const NJson::TJsonValue &v) {
            return v.GetDoubleRobust();
        }
    };


    template<> void TFieldBase<TString, StringTraits>::Or(const NJson::TJsonValue &/*message*/, NAnyValue::TScalar &/*anyValue*/) const {
        ythrow TWithBackTrace<yexception>() << "'|' operation is not supported for string";
    };


    template<> void TFieldBase<double, DoubleTraits>::Or(const NJson::TJsonValue &/*message*/, NAnyValue::TScalar &/*anyValue*/) const {
        ythrow TWithBackTrace<yexception>() << "'|' operation is not supported for double";
    };

    template<typename T> using TUIntegerField = TFieldBase<T, UIntegerTraits>;
    template<typename T> using TIntegerField = TFieldBase<T, IntegerTraits>;
    using TStringField = TFieldBase<TString, StringTraits>;
    using TBoolField = TFieldBase<bool, BoolTraits>;
    using TDoubleField = TFieldBase<double, DoubleTraits>;

    THolder<IField> FieldFabric(const NConfig::TConfig &config) {
        const auto type = FromString<TSupportedType>(NTalkativeConfig::Get<TString>(config));

        switch(type) {
            case TSupportedType::Ui16:      return MakeHolder<TUIntegerField<ui16>>();
            case TSupportedType::Ui32:      return MakeHolder<TUIntegerField<ui32>>();
            case TSupportedType::Ui64:      return MakeHolder<TUIntegerField<ui64>>();
            case TSupportedType::I64:       return MakeHolder<TIntegerField<i64>>();
            case TSupportedType::String:    return MakeHolder<TStringField>();
            case TSupportedType::Bool:      return MakeHolder<TBoolField>();
            case TSupportedType::Double:    return MakeHolder<TDoubleField>();
        }
    }

    IFieldPtr TFieldSet::GetField(const TString & name) const {
        auto it = fields.find(name);

        return it != fields.cend() ? it->second : nullptr;
    }

    IFieldPtr TFieldSet::GetFieldSafe(const TString & name) const {
        auto field = GetField(name);
        if (!field)
            ythrow TWithBackTrace<yexception>() << "cannot find \"" << name << "\" field";
        return field;
    }

    TFieldSet::TFieldSet(const NConfig::TConfig &config) {
        for(const auto & p : NTalkativeConfig::Get<NConfig::TDict>(config)) {
            fields.emplace(p.first, FieldFabric(p.second));
        }
    }

    TVector<TFieldSet::TNamedField> TFieldSet::LinkFields(const NConfig::TArray &array) const {
        TVector<TFieldSet::TNamedField> fields;

        for (const auto & fieldConf : array) {
            const auto & fieldName = NTalkativeConfig::Get<TString>(fieldConf);
            fields.emplace_back(fieldName, GetFieldSafe(fieldName));
        }

        return fields;
    }

    TVector<TFieldSet::TNamedField> TFieldSet::LinkFields(const NConfig::TConfig &config) const {
        if (config.IsA<TString>()) {
            NConfig::TArray local;
            local.push_back(config);
            return LinkFields(local);
        }

        return LinkFields(NTalkativeConfig::Get<NConfig::TArray>(config));
    }

    const TFieldSet & TFieldSets::GetSetSafe(const NConfig::TConfig &config) const {
        const auto & setName = NTalkativeConfig::Get<TString>(config, "field_set");
        auto it = sets.find(setName);

        if(sets.cend() == it)
            ythrow TWithBackTrace<yexception>() << "cannot find fieldSet by name " << setName;

        return it->second;
    }

    TFieldSets::TFieldSets(const NConfig::TConfig &config) {
        if(!config.IsA<NConfig::TDict>())
            ythrow TWithBackTrace<yexception>() << "TFieldSets must initialized from dict: " << config;

        for(const auto & p : config.Get<NConfig::TDict>()) {
            sets.emplace(p.first, p.second);
        }
    }
}


