
#include <library/cpp/config/config.h>
#include <mail/so/spamstop/tools/so-common/StorageBase.h>
#include <mail/so/libs/talkative_config/config.h>
#include <mail/so/spamstop/tools/general_shingler/data/helper.h>
#include <mail/so/spamstop/tools/general_shingler/data/time_set.h>

#include "keys.h"

namespace NGeneralShingler {

    size_t TKeyScheme::Hash(const NJson::TJsonValue & messageFields) const {
        size_t h = 0;
        for(const auto& f : fields) {
            h = CombineHashes(h, THash<TString>()(f.first));
            h = CombineHashes(h, f.second->Hash(messageFields[f.first]));
        }
        return h;
    }

    TKeyScheme::TKeyScheme(TVector<TFieldSet::TNamedField> fields) : fields(std::move(fields)) {}

    template<TKeyType type> class TKeyImpl : public TKeyScheme{
    public:
        void Apply(TQueryData & query, const NJson::TJsonValue & messageFields) const override;
        using TKeyScheme::TKeyScheme;
    };

    template<TKeyType type> class TTimedKeyImpl : public TKeyScheme{
    public:
        void Apply(TQueryData & query, const NJson::TJsonValue & messageFields) const override;
        TTimedKeyImpl(TVector<TFieldSet::TNamedField> fields, const TTimeSet & timeSet)
                : TKeyScheme(fields), timeSet(timeSet) {}

    private:
        TTimeSet timeSet;
    };

    template<TKeyType type>
    static THolder<TKeyScheme> GetKeyScheme(const NConfig::TConfig & config, const TFieldSet & set) {
        auto fields = set.LinkFields(config);
        return MakeHolder<TKeyImpl<type>>(std::move(fields));
    }

    template<TKeyType type>
    static THolder<TKeyScheme> GetTimeKeyScheme(const NConfig::TConfig & config, const TFieldSet & set, const THashMap<TString, TTimeSet> & timeSetsByName) {
        auto fields = set.LinkFields(NTalkativeConfig::Get(config, "fields"));
        auto timeSetName = NTalkativeConfig::Get<TString>(config, "time_set");
        const auto & timeSet = NHelper::GetSafe(timeSetsByName, timeSetName);
        return MakeHolder<TTimedKeyImpl<type>>(std::move(fields), timeSet);
    }

    THolder<TKeyScheme> KeyFactory(const TString& typeName, const NConfig::TConfig & config, const TFieldSet & set, const THashMap<TString, TTimeSet> & timeSetsByName) {
        switch (FromString<TKeyType>(typeName)) {
            case TKeyType::Simple:
                return GetKeyScheme<TKeyType::Simple>(config, set);
            case TKeyType::Greater:
                return GetKeyScheme<TKeyType::Greater>(config, set);
            case TKeyType::Time:
                return GetTimeKeyScheme<TKeyType::Time>(config, set, timeSetsByName);
            case TKeyType::History:
                return GetTimeKeyScheme<TKeyType::History>(config, set, timeSetsByName);
            case TKeyType::Yesterday:
                return GetTimeKeyScheme<TKeyType::Yesterday>(config, set, timeSetsByName);
            case TKeyType::Older:
                return GetTimeKeyScheme<TKeyType::Older>(config, set, timeSetsByName);
            case TKeyType::In:
                return GetKeyScheme<TKeyType::In>(config, set);
        }
    }

    template<bool use_field>
    static void ApplyImpl(const TVector<TFieldSet::TNamedField>& fields, const NJson::TJsonValue & messageFields, NAnyValue::TScalarMap& action, const std::function<NJson::TJsonValue(const NJson::TJsonValue&)>& getValue) {
        for (const auto & field : fields) {
            const TString & fieldName = field.first;
            if (use_field) {
                if (messageFields.Has(fieldName)) {
                    const auto &messageField = messageFields[fieldName];
                    field.second->JsonToAnyvalue(getValue(messageField), action[fieldName]);
                }
            } else {
                NJson::TJsonValue empty;
                field.second->JsonToAnyvalue(getValue(empty), action[fieldName]);
            }
        }
    }

    template<> void TKeyImpl<TKeyType::Simple>::Apply(TQueryData & query, const NJson::TJsonValue & messageFields) const {
        ApplyImpl<true>(fields, messageFields, query.equals, [](const NJson::TJsonValue& value) {
            return value;
        });
    }

    template<> void TKeyImpl<TKeyType::Greater>::Apply(TQueryData & query, const NJson::TJsonValue & messageFields) const {
        ApplyImpl<true>(fields, messageFields, query.gt, [](const NJson::TJsonValue& value) {
            return value;
        });
    }

    template<> void TTimedKeyImpl<TKeyType::Time>::Apply(TQueryData & query, const NJson::TJsonValue & messageFields) const {
        const auto now = timeSet.GetCurrentPeriod();
        ApplyImpl<false>(fields, messageFields, query.equals, [&now](const NJson::TJsonValue&) {
            return now;
        });
    }

    template<> void TTimedKeyImpl<TKeyType::Yesterday>::Apply(TQueryData & query, const NJson::TJsonValue & messageFields) const {
        const auto now = timeSet.GetCurrentPeriod();
        ApplyImpl<false>(fields, messageFields, query.lt, [&now](const NJson::TJsonValue&) {
            return now;
        });
    }

    template<> void TTimedKeyImpl<TKeyType::History>::Apply(TQueryData & query, const NJson::TJsonValue & messageFields) const {
        const auto now = timeSet.GetCurrentPeriod();
        ApplyImpl<true>(fields, messageFields, query.gt, [&now](const NJson::TJsonValue& value) {
            return now - value.GetUIntegerRobust() - 1;
        });
    }

    template<> void TTimedKeyImpl<TKeyType::Older>::Apply(TQueryData & query, const NJson::TJsonValue & messageFields) const {
        const auto now = timeSet.GetCurrentPeriod();
        ApplyImpl<true>(fields, messageFields, query.lt, [&now](const NJson::TJsonValue& value) {
            return now - value.GetUIntegerRobust();
        });
    }

    template<> void TKeyImpl<TKeyType::In>::Apply(TQueryData & query, const NJson::TJsonValue & messageFields) const {
        for(const auto & field : fields) {
            const TString & fieldName = field.first;
            if(messageFields.Has(fieldName) && messageFields[fieldName].IsArray()) {
                for (const auto &v :  messageFields[fieldName].GetArray()) {
                    field.second->JsonToAnyvalue(v, query.in[fieldName].emplace_back());
                }
            }
        }
    }
}

