#include <numeric>
#include <util/generic/xrange.h>
#include <mail/so/libs/talkative_config/config.h>
#include "shard.h"


namespace NGeneralShingler {

    void TSharder::TSharded::AddSharded(size_t shard, NJson::TJsonValue::TArray && values) {
        auto it = groupedByShard.find(shard);
        if(it == groupedByShard.end())
            it = groupedByShard.emplace(shard, TShardedFields(shard)).first;

        for(auto & val : values)
            it->second.fields.emplace_back(std::move(std::move(val)));
    }

    static void PopulateByShard(size_t shard, NJson::TJsonValue && value, THashMap<size_t, NJson::TJsonValue::TArray> & target) {
        auto shardIt = target.find(shard);

        if(shardIt == target.end())
            target.emplace(shard, NJson::TJsonValue::TArray{std::move(value)});
        else
            shardIt->second.emplace_back(std::move(value));
    }

    TSharder::TSharded TSharder::Shardify(NJson::TJsonValue::TArray && messageFields) const {
        TSharder::TSharded sharded;
        for(auto & fields : messageFields) {
            auto & asMap = fields.GetMapSafe();
            auto it = asMap.find(shardBy);
            if(asMap.cend() == it) {
                sharded.withoutShard.fields.emplace_back(std::move(fields));
                continue;
            }

            NJson::TJsonValue & value = it->second;

            THashMap<size_t, NJson::TJsonValue::TArray> populatedByShard;

            switch(value.GetType()){

                case NJson::JSON_UNDEFINED:
                case NJson::JSON_NULL:
                case NJson::JSON_MAP:
                    continue;
                case NJson::JSON_BOOLEAN:
                case NJson::JSON_INTEGER:
                case NJson::JSON_DOUBLE:
                case NJson::JSON_STRING:
                case NJson::JSON_UINTEGER: {
                    const auto shard = CalcShard(value.GetUIntegerRobust());
                    PopulateByShard(shard, std::move(fields), populatedByShard);
                    break;
                }
                case NJson::JSON_ARRAY: {
                    const size_t maxSize = std::accumulate(asMap.cbegin(), asMap.cend(), 0ul, [](size_t s, const std::pair<TString, NJson::TJsonValue> & p){
                        if(p.second.IsArray())
                            return std::max(s, p.second.GetArraySafe().size());

                        return s;
                    });

                    for(size_t i : xrange(maxSize)) {
                        NJson::TJsonValue slice;

                        size_t shard = {};

                        for(auto & field : asMap) {
                            if(field.second.IsArray()) {
                                auto & val = field.second.GetArraySafe()[i];
                                if(field.first == shardBy)
                                    shard = CalcShard(val.GetUIntegerRobust());

                                slice[field.first].SetType(NJson::JSON_ARRAY).GetArraySafe().emplace_back(std::move(val));
                            } else {
                                if(field.first == shardBy)
                                    shard = CalcShard(field.second.GetUIntegerRobust());

                                slice[field.first] = field.second;
                            }
                        }

                        PopulateByShard(shard, std::move(slice), populatedByShard);
                    }

                    break;
                }
            }

            for(auto & shardedValue : populatedByShard) {
                sharded.AddSharded(shardedValue.first, std::move(shardedValue.second));
            }
        }

        return sharded;
    }

    TSharder::TSharder(const NConfig::TConfig & config) {
        shardBy = NTalkativeConfig::Get<TString>(config, "field");
    }
}
