#include <util/thread/pool.h>
#include <library/cpp/testing/unittest/registar.h>
#include "cache.h"

using namespace NGeneralShingler;

Y_UNIT_TEST_SUITE(Cache) {

    TAtomicSharedPtr<ICache> CreateCacheSimple(TCacheType type, size_t size) {
        TStringStream ss = R"({"type":")" + ToString(type) + R"(","size":)" + ToString(size) + R"(})";
        return CreateCache(NConfig::TConfig::FromJson(ss), nullptr, nullptr);
    }

    Y_UNIT_TEST(LAT) {
        auto cache = CreateCacheSimple(TCacheType::LAT, 5);

        NJson::TJsonValue::TArray array;
        for (size_t i = 1; i < 5; ++i) {
            array.emplace_back(i);
            auto copy = array;
            UNIT_ASSERT_EQUAL(cache->Add(i, std::move(copy)), true);
        }

        UNIT_ASSERT_EQUAL(cache->Size(), 4);

        for (size_t i = 1; i < 5; ++i) {
            NJson::TJsonValue::TArray local;
            UNIT_ASSERT_EQUAL(cache->Get(i, local), true);
            UNIT_ASSERT_EQUAL(local.size(), i);
        }

        UNIT_ASSERT_EQUAL(cache->Get(23, array), false);

        UNIT_ASSERT_EQUAL(cache->Add(1ul, std::move(array)), false);
        UNIT_ASSERT_EQUAL(cache->Size(), 4);
        UNIT_ASSERT_EQUAL(cache->Add(23, std::move(array)), true);
        UNIT_ASSERT_EQUAL(cache->Add(42, std::move(array)), true);

        UNIT_ASSERT_EQUAL(cache->Size(), 5);

        UNIT_ASSERT_EQUAL(cache->Get(1ul, array), false);

        for (size_t i = 2; i < 5; ++i) {
            NJson::TJsonValue::TArray local;
            UNIT_ASSERT_EQUAL(cache->Get(i, local), true);
            UNIT_ASSERT_EQUAL(local.size(), i);
        }
    }

    Y_UNIT_TEST(LRU) {
        auto cache = CreateCacheSimple(TCacheType::LRU, 5);

        NJson::TJsonValue::TArray array;
        for (size_t i = 0; i < 5; ++i) {
            auto copy = array;
            UNIT_ASSERT_EQUAL(cache->Add(i, std::move(copy)), true);
            array.emplace_back(i);
        }

        for (size_t i = 0; i < 4; ++i) {
            NJson::TJsonValue::TArray local;
            UNIT_ASSERT_EQUAL(cache->Get(i, local), true);
            UNIT_ASSERT_EQUAL(local.size(), i);
        }

        UNIT_ASSERT_EQUAL(cache->Get(23, array), false);
        UNIT_ASSERT_EQUAL(cache->Add(23, std::move(array)), true);
        UNIT_ASSERT_EQUAL(cache->Get(5ul, array), false);
        UNIT_ASSERT_EQUAL(cache->Size(), 0);
    }

    Y_UNIT_TEST(Step) {
        auto threadPool = MakeAtomicShared<TAdaptiveThreadPool>();
        threadPool->Start(0, 0);

        TStringStream ss = TString(R"({"refresh_time":"1m"})");
        TSimpleScheduler scheduler(*threadPool);
        auto cache = CreateCache(NConfig::TConfig::FromJson(ss), &scheduler, nullptr);
        UNIT_ASSERT_UNEQUAL(cache, nullptr);
    }

    void Check(TString input) {
        TStringStream ss = input;
        TCacheContext context(NConfig::TConfig::FromJson(ss), nullptr);
    }

    Y_UNIT_TEST(ContextException) {
        UNIT_ASSERT_EXCEPTION(Check(R"({})"), std::exception);
        UNIT_ASSERT_EXCEPTION(Check(R"({"fields": 1})"), std::exception);
        UNIT_ASSERT_EXCEPTION(Check(R"({"fields":[1]})"), std::exception);
        UNIT_ASSERT_NO_EXCEPTION(Check(R"({"fields":["shingle"]})"));
    }

    Y_UNIT_TEST(Context) {
        TStringStream cc = TString(R"({"fields":["shingle","date"]})");
        TCacheContext context(NConfig::TConfig::FromJson(cc), nullptr);

        TStringStream fs = TString(R"({"shingle":"Ui64","date":"Ui16"})");
        TFieldSet set(NConfig::TConfig::FromJson(fs));

        NJson::TJsonValue value;
        value["date"] = 42;
        value["shingle"] = ~42;

        UNIT_ASSERT_EQUAL(context.CreateKey(value, set), 12897285240902822183ull);
    }
}

