#include <library/cpp/testing/unittest/registar.h>

#include <library/cpp/json/json_value.h>
#include <library/cpp/config/config.h>
#include "field.h"

using namespace NGeneralShingler;

Y_UNIT_TEST_SUITE(Field) {
    Y_UNIT_TEST(Ops) {
        NConfig::TConfig config;
        {
            const char mock [] = "{\n"
                                 "        \"ui16\": \"Ui16\",\n"
                                 "        \"ui32\": \"Ui32\",\n"
                                 "        \"ui64\": \"Ui64\",\n"
                                 "        \"bool\": \"Bool\",\n"
                                 "        \"i64\": \"I64\",\n"
                                 "        \"string\": \"String\",\n"
                                 "        \"double\": \"Double\"\n"
                                 "      }";
            TMemoryInput s(mock, sizeof(mock));
            config = NConfig::TConfig::FromJson(s);
        }

        NJson::TJsonValue v;

        v["ui16"] = 1;
        v["ui32"] = 2;
        v["ui64"] = 3;
        v["i64"] = TString("0x4");
        v["bool"] = true;
        v["double"] = 6;
        v["string"] = "test";

        NAnyValue::TScalarMap anyvalueMap;

        anyvalueMap["ui16"] = 789u;
        anyvalueMap["ui32"] = 789u;
        anyvalueMap["ui64"] = 789u;
        anyvalueMap["i64"] = 789u;
        anyvalueMap["bool"] = false;
        anyvalueMap["double"] = 789u;
        anyvalueMap["string"] = TString("1234566");

        TFieldSet fields(config);

        {
            auto field = fields.GetFieldSafe("ui16");
            field->Incr(v, anyvalueMap, "ui16");
            UNIT_ASSERT_EQUAL(anyvalueMap["ui16"].AsUi16(), 790);
            field->Or(v, anyvalueMap, "ui16");
            UNIT_ASSERT_EQUAL(anyvalueMap["ui16"].AsUi16(), 790 | 1);
            field->JsonToAnyvalue(v, anyvalueMap, "ui16");
            UNIT_ASSERT_EQUAL(anyvalueMap["ui16"].AsUi16(), 1);

            UNIT_ASSERT_EQUAL(field->Hash(v["ui16"]), THash<ui16>()(1));
        }

        {
            auto field = fields.GetFieldSafe("ui32");
            field->Incr(v, anyvalueMap, "ui32");
            UNIT_ASSERT_EQUAL(anyvalueMap["ui32"].AsUi32(), 791);
            field->Or(v, anyvalueMap, "ui32");
            UNIT_ASSERT_EQUAL(anyvalueMap["ui32"].AsUi32(), 791 | 2);
            field->JsonToAnyvalue(v, anyvalueMap, "ui32");
            UNIT_ASSERT_EQUAL(anyvalueMap["ui32"].AsUi32(), 2);

            UNIT_ASSERT_EQUAL(field->Hash(v["ui32"]), THash<ui32>()(2));
        }

        {
            auto field = fields.GetFieldSafe("ui64");
            field->Incr(v, anyvalueMap, "ui64");
            UNIT_ASSERT_EQUAL(anyvalueMap["ui64"].AsUi64(), 792);
            field->Or(v, anyvalueMap, "ui64");
            UNIT_ASSERT_EQUAL(anyvalueMap["ui64"].AsUi64(), 792 | 3);
            field->JsonToAnyvalue(v, anyvalueMap, "ui64");
            UNIT_ASSERT_EQUAL(anyvalueMap["ui64"].AsUi64(), 3);

            UNIT_ASSERT_EQUAL(field->Hash(v["ui64"]), THash<ui64>()(3));
        }

        {
            auto field = fields.GetFieldSafe("i64");
            field->Incr(v, anyvalueMap, "i64");
            UNIT_ASSERT_EQUAL(anyvalueMap["i64"].AsI64(), 793);
            field->Or(v, anyvalueMap, "i64");
            UNIT_ASSERT_EQUAL(anyvalueMap["i64"].AsI64(), 793 | 4);
            field->JsonToAnyvalue(v, anyvalueMap, "i64");
            UNIT_ASSERT_EQUAL(anyvalueMap["i64"].AsI64(), 4);

            UNIT_ASSERT_EQUAL(field->Hash(v["i64"]), THash<i64>()(4));
        }

        {
            auto field = fields.GetFieldSafe("bool");
            field->Incr(v, anyvalueMap, "bool");
            UNIT_ASSERT_EQUAL(anyvalueMap["bool"].AsBool(), true);
            field->Or(v, anyvalueMap, "bool");
            UNIT_ASSERT_EQUAL(anyvalueMap["bool"].AsBool(), true);
            field->JsonToAnyvalue(v, anyvalueMap, "bool");
            UNIT_ASSERT_EQUAL(anyvalueMap["bool"].AsBool(), true);

            UNIT_ASSERT_EQUAL(field->Hash(v["bool"]), THash<bool>()(true));
        }

        {
            auto field = fields.GetFieldSafe("double");
            field->Incr(v, anyvalueMap, "double");
            UNIT_ASSERT_EQUAL_C(anyvalueMap["double"].AsDouble(), 795, anyvalueMap["double"]);

            UNIT_ASSERT_EXCEPTION(field->Or(v, anyvalueMap, "double"), yexception);
            UNIT_ASSERT_EQUAL(anyvalueMap["double"].AsDouble(), 795);

            field->JsonToAnyvalue(v, anyvalueMap, "double");
            UNIT_ASSERT_EQUAL(anyvalueMap["double"].AsDouble(), 6);

            UNIT_ASSERT_EQUAL(field->Hash(v["double"]), THash<double>()(6));
        }

        {
            auto field = fields.GetFieldSafe("string");
            field->Incr(v, anyvalueMap, "string");
            UNIT_ASSERT_EQUAL(anyvalueMap["string"].AsString(), "test1234566");

            UNIT_ASSERT_EXCEPTION(field->Or(v, anyvalueMap, "string"), yexception);
            UNIT_ASSERT_EQUAL(anyvalueMap["string"].AsString(), "test1234566");

            field->JsonToAnyvalue(v, anyvalueMap, "string");
            UNIT_ASSERT_EQUAL(anyvalueMap["string"].AsString(), "test");

            UNIT_ASSERT_EQUAL(field->Hash(v["string"]), THash<TString>()("test"));
        }
    }

    auto GetByName(TFieldSets& sets, TString name) {
        TStringStream ss = R"({"field_set":")" + name + R"("})";
        return sets.GetSetSafe(NConfig::TConfig::FromJson(ss));
    }

    Y_UNIT_TEST(Sets) {
        TStringStream ss = TString(R"({
            "first" : {
                "ui16": "Ui16",
                "ui32": "Ui32"
            },
            "second" : {
                "ui64": "Ui64"
            }
        })");

        TFieldSets sets(NConfig::TConfig::FromJson(ss));
        UNIT_ASSERT_NO_EXCEPTION(GetByName(sets, "first"));
        UNIT_ASSERT_NO_EXCEPTION(GetByName(sets, "second"));
        UNIT_ASSERT_EXCEPTION(GetByName(sets, "third"), std::exception);
    }
}

