#include <library/cpp/logger/all.h>
#include <library/cpp/http/server/response.h>
#include <library/cpp/svnversion/svnversion.h>
#include <util/stream/file.h>

#include <mail/so/libs/syslog/so_log.h>
#include <mail/so/libs/talkative_config/config.h>
#include <mail/so/spamstop/tools/general_shingler/data/cache.h>
#include <mail/so/spamstop/tools/simple_shingler/static_serve.h>
#include <mail/so/spamstop/tools/so-common/stats_consumer.h>
#include <util/generic/scope.h>

#include "processor.h"
#include "yasm_stater.h"

bool FormatFromCgi(const TCgiParameters& cgiParameters) {
    auto it = cgiParameters.find("format_output");

    if (cgiParameters.cend() == it)
        return false;

    return FromString<bool>(it->second);
}

class THandle : public THandlerFunctor {
public:
    void Reply(THandleContext&& handleContext, Y_DECLARE_UNUSED void* tsr) override {
        auto parsedMessages = ParseContext(handleContext);

        if (!parsedMessages.updateMessages.empty()) {
            processor->ProcessUpdateMessages(std::move(parsedMessages.updateMessages));
        }

        if (!parsedMessages.syncUpdateMessages.empty()) {
            processor->ProcessUpdateMessagesSync(std::move(parsedMessages.syncUpdateMessages));
        }

        if (!parsedMessages.getMessages.empty()) {
            const auto& res = processor->ProcessGetMessages(std::move(parsedMessages.getMessages));
            const bool formatOutput = FormatFromCgi(handleContext.cgiParameters);
            THttpResponse(HTTP_OK).SetContent(NJson::WriteJson(res, formatOutput), "application/json").OutTo(handleContext.output);
        } else {
            THttpResponse(HTTP_OK).OutTo(handleContext.output);
        }
    }

    explicit THandle(TAtomicSharedPtr<NGeneralShingler::TProcessor> processor, TAtomicSharedPtr<TLog> logger)
        : processor(std::move(processor))
        , logger(std::move(logger)) {
        Y_ENSURE(this->logger);
        Y_ENSURE(this->processor);
    }

private:
    struct TParsedMessages {
        TParsedMessages(
            TVector<NGeneralShingler::TMessage> updateMessages,
            TVector<NGeneralShingler::TMessage> getMessages,
            TVector<NGeneralShingler::TMessage> syncUpdateMessages) noexcept
            : updateMessages(std::move(updateMessages))
            , getMessages(std::move(getMessages))
            , syncUpdateMessages(std::move(syncUpdateMessages)) {
        }
        TVector<NGeneralShingler::TMessage> updateMessages, getMessages, syncUpdateMessages;
    };
    TParsedMessages ParseContext(const THandleContext& handleContext) {
        TVector<NGeneralShingler::TMessage> messages = processor->ParseMessages(handleContext);

        if (messages.empty())
            ythrow THttpError(HTTP_BAD_REQUEST) << "zero messages. " << handleContext;

        TVector<NGeneralShingler::TMessage>
            updateMessages(Reserve(messages.size())),
            getMessages(Reserve(messages.size())),
            syncUpdateMessages(Reserve(messages.size()));

        for (NGeneralShingler::TMessage& message : messages) {
            switch (message.GetType()) {
                case NGeneralShingler::TMessageType::Get:
                    getMessages.emplace_back(std::move(message));
                    break;
                case NGeneralShingler::TMessageType::Update:
                    updateMessages.emplace_back(std::move(message));
                    break;
                case NGeneralShingler::TMessageType::SyncUpdate:
                    syncUpdateMessages.emplace_back(std::move(message));
                    break;
            }
        }
        return {std::move(updateMessages), std::move(getMessages), std::move(syncUpdateMessages)};
    }

private:
    TAtomicSharedPtr<NGeneralShingler::TProcessor> processor;
    TAtomicSharedPtr<TLog> logger;
};

TAtomicSharedPtr<TLog> CreateLog(const NConfig::TConfig& config) {
    THolder<TLogBackend> back(new TStreamLogBackend(&Cerr));
    ELogPriority priority = LOG_MAX_PRIORITY;

    if (config.Has("logger")) {
        const auto& loggerConfig = config["logger"];

        if (loggerConfig.Has("path")) {
            back.Reset(new TFileLogBackend(loggerConfig["path"].Get<TString>()));
        }

        if (loggerConfig.Has("priority")) {
            priority = FromString<ELogPriority>(loggerConfig["priority"].Get<TString>());
        }
    }

    auto log = MakeAtomicShared<TLog>(MakeHolder<TFilteredLogBackend>(std::move(back), priority));

    log->SetFormatter([](ELogPriority priority, TStringBuf buf) -> TString {
        TStringStream s;
        s << Now() << ' ' << priority << ' ' << buf;
        return s.Str();
    });

    return log;
}

class TLogProgressInfo : public TSimpleScheduler::IProgressInfo {
public:
    explicit TLogProgressInfo(TAtomicSharedPtr<TLog> log)
        : m_log(std::move(log)) {
    }

    void OnStart(const TString& name) final {
        *m_log << TLOG_DEBUG << "Start task: " << name << Endl;
    }

    void OnFinish(const TString& name) final {
        *m_log << TLOG_DEBUG << "Finish task: " << name << Endl;
    }

    void OnException(const TString& name) final {
        *m_log << TLOG_ERR << "Throw exception while execute task: " << name << " - " << CurrentExceptionMessageWithBt() << Endl;
    }

private:
    TAtomicSharedPtr<TLog> m_log;
};

struct TPoolOptions {
    TPoolOptions() = default;
    explicit TPoolOptions(const NConfig::TConfig& config)
        : TPoolOptions() {
        if (config.Has("threads"))
            threads = NTalkativeConfig::As<size_t>(config, "threads");
        if (config.Has("queue_size"))
            queueSize = NTalkativeConfig::As<size_t>(config, "queue_size");
    }

    size_t threads = 128;
    size_t queueSize = 128;
};

int main(int argc, const char** argv) {
    Cout << "running " << argv[0] << '\n'
         << GetProgramSvnVersion() << '\n';

    argc--;
    argv++;

    if (argc < 1) {
        Cerr << "Usage: prog <conf.json>" << Endl;
        return 1;
    }

    NConfig::TConfig config;
    {
        TFileInput input(*argv);
        config = NConfig::TConfig::FromJson(input);
    }

    auto log = CreateLog(config);

    const auto& serverOpts = TServerOptions::ReadOptionsFromConfig(NTalkativeConfig::Get<NConfig::TDict>(config, "server"));

    const TPoolOptions poolOptions = [&config] {
        if (config.Has("thread_pool"))
            return TPoolOptions(config["thread_pool"]);
        return TPoolOptions{};
    }();

    auto threadPool = MakeAtomicShared<TAdaptiveThreadPool>();

    threadPool->Start(poolOptions.threads, poolOptions.queueSize);

    THashMap<TString, TAtomicSharedPtr<NGeneralShingler::ICache>> caches;
    TSimpleScheduler scheduler(*threadPool, MakeAtomicShared<TLogProgressInfo>(log));
    auto processor = NGeneralShingler::ProcessorFactory(config["shingler"], threadPool, log, caches, scheduler);
    auto handler = MakeAtomicShared<THandle>(processor, log);

    THandlerServer server(serverOpts, threadPool.Get());

    auto pingHandler = MakeAtomicShared<TPingHandler>();
    auto yasmStater = MakeAtomicShared<YasmStater>(TVector<TString>({"/api"}));
    server.On("/stop", MakeAtomicShared<TStopServerHandler>(&server, pingHandler))
        .On("/rotate_logs", [&server, log](THandleContext&& handleContext, void* /*tsrt*/){
            server.ReopenLog();
            log->ReopenLog();
            handleContext.output << THttpResponse().SetContent("Ok");
        })
        .On("/api", handler)
        .On("/ping", pingHandler)
        .On("/yasm", yasmStater)
        .OnEvent(yasmStater);

    if (config.Has("static")) {
        const auto& staticConf = NTalkativeConfig::Get(config, "static");
        const auto& path = NTalkativeConfig::As<TFsPath>(staticConf, "path");

        server.On(
            NTalkativeConfig::Get<TString>(staticConf, "on"),
            new TStaticServe(path));
    }

    Y_VERIFY(server.Start());

    server.Wait();

    return 0;
}
